/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include "qfle3i.h"

static vmk_IscsiTransIscsiParam qfle3i_sess_params[] = { QFLE3I_TRANSPORT_SESS_PARAMS };
static vmk_IscsiTransIscsiParam qfle3i_conn_params[] = { QFLE3I_TRANSPORT_CONN_PARAMS };
static vmk_IscsiTransHostParam  qfle3i_host_params[] = { QFLE3I_TRANSPORT_HOST_PARAMS };
static vmk_IscsiTransCapabilities qfle3i_supportedCaps[] = { QFLE3I_TRANSPORT_CAPS };

extern unsigned int qfle3i_dump_queue;
extern unsigned int qfle3i_esx_mtu_max;
extern unsigned int sq_size;
extern unsigned int rq_size;
extern unsigned int tcp_buf_size;
extern unsigned int en_tcp_dack;
extern unsigned int time_stamps;
extern unsigned int en_hba_poll;
extern unsigned int qfle3i_nopout_when_cmds_active;
extern vmk_DMAConstraints qfle3i_dmaConstraint;
extern vmk_Lock qfle3i_resc_lock;
extern vmk_LogComponent qfle3i_vmkLog;

static void *qfle3i_ep_pages[MAX_PAGES_PER_CTRL_STRUCT_POOL];
static vmk_ListLinks qfle3i_free_ep_list;
//static vmk_ListLinks qfle3i_unbound_ep;
static vmk_uint32 qfle3i_num_free_ep;
static vmk_uint32 qfle3i_max_free_ep;


void qfle3i_unbind_adapter_devices(struct qfle3i_hba *hba);
static void ep_tmo_poll_task(void *data);
static struct io_bdt *qfle3i_alloc_bd_table(struct qfle3i_sess *sess,
					   struct qfle3i_cmd *);

static int qfle3i_ep_tcp_conn_active(struct qfle3i_endpoint *ep);
static void qfle3i_conn_main_worker(unsigned long data);
int qfle3i_do_iscsi_sess_recovery(struct qfle3i_sess *sess, int reason, int signal);
static void qfle3i_conn_poll(unsigned long data);
static void qfle3i_xmit_work_send_cmd(struct qfle3i_conn *conn, struct qfle3i_cmd *cmd);
static void qfle3i_withdraw_sess_recovery(struct qfle3i_sess *sess);
static void conn_err_recovery_task(void *data);
static void qfle3i_handle_net_event(void *data);

int use_poll_timer = 1;

int qfle3i_adapter_ready(struct qfle3i_hba *hba)
{
	if (!hba || !vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_UP) ||
	    vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_FW_RECOVERY) ||
	    vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_GOING_DOWN) ||
	    vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_LINK_DOWN))
		return VMK_FAILURE;
	return VMK_OK;
}

/*
 * qfle3i_ep_destroy_list_del - add an entry to EP destroy list
 *
 * @hba: 		pointer to adapter instance
 * @ep: 		pointer to endpoint (transport indentifier) structure
 *
 * EP destroy queue manager
 */
static int qfle3i_ep_destroy_list_del(struct qfle3i_hba *hba,
				     struct qfle3i_endpoint *ep)
{
	vmk_SpinlockLock(hba->lock);
	vmk_ListRemove(&ep->link);
	vmk_ListInit(&ep->link);
	vmk_SpinlockUnlock(hba->lock);
	return 0;
}

/*
 * qfle3i_ep_destroy_list_add - add an entry to EP destroy list
 *
 * @hba: 		pointer to adapter instance
 * @ep: 		pointer to endpoint (transport indentifier) structure
 *
 * EP destroy queue manager
 */
static int qfle3i_ep_destroy_list_add(struct qfle3i_hba *hba,
				  struct qfle3i_endpoint *ep)
{
	vmk_SpinlockLock(hba->lock);
	vmk_ListInsert(&ep->link, vmk_ListAtRear(&hba->ep_destroy_list));
	vmk_SpinlockUnlock(hba->lock);
	return 0;
}

/*
 * qfle3i_find_ep_in_destroy_list - find iscsi_cid in destroy list
 *
 * @hba: 		pointer to adapter instance
 * @iscsi_cid:		iscsi context ID to find
 *
 */
struct qfle3i_endpoint *
qfle3i_find_ep_in_destroy_list(struct qfle3i_hba *hba, vmk_uint32 iscsi_cid)
{
	struct qfle3i_endpoint *ep = NULL;
	struct qfle3i_endpoint *tmp_ep;

	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(tmp_ep, &hba->ep_destroy_list, link, struct qfle3i_endpoint) {

		if (tmp_ep->ep_iscsi_cid == iscsi_cid) {
			ep = tmp_ep;
			break;
		}
	}
	vmk_SpinlockUnlock(hba->lock);

	if (!ep)
		PRINT(hba, "destroy_list - icid %d not found\n", iscsi_cid);

	return ep;
}

/*
 * qfle3i_ep_ofld_list_add - add an entry to ep offload pending list
 *
 * @hba: 		pointer to adapter instance
 * @ep: 		pointer to endpoint (transport indentifier) structure
 *
 * pending conn offload completion queue manager
 */
static int qfle3i_ep_ofld_list_add(struct qfle3i_hba *hba,
				  struct qfle3i_endpoint *ep)
{
	vmk_SpinlockLock(hba->lock);
	vmk_ListInsert(&ep->link, vmk_ListAtRear(&hba->ep_ofld_list));
	vmk_SpinlockUnlock(hba->lock);
	return 0;
}

/*
 * qfle3i_ep_ofld_list_del - add an entry to ep offload pending list
 *
 * @hba: 		pointer to adapter instance
 * @ep: 		pointer to endpoint (transport indentifier) structure
 *
 * pending conn offload completion queue manager
 */
static int qfle3i_ep_ofld_list_del(struct qfle3i_hba *hba,
				  struct qfle3i_endpoint *ep)
{
	vmk_SpinlockLock(hba->lock);
	vmk_ListRemove(&ep->link);
	vmk_ListInit(&ep->link);
	vmk_SpinlockUnlock(hba->lock);
	return 0;
}

/*
 * qfle3i_find_ep_in_ofld_list - find iscsi_cid in pending list of endpoints
 *
 * @hba: 		pointer to adapter instance
 * @iscsi_cid:		iscsi context ID to find
 *
 */
struct qfle3i_endpoint *
qfle3i_find_ep_in_ofld_list(struct qfle3i_hba *hba, vmk_uint32 iscsi_cid)
{
	struct qfle3i_endpoint *ep = NULL;
	struct qfle3i_endpoint *tmp_ep;

	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(tmp_ep, &hba->ep_ofld_list, link, struct qfle3i_endpoint) {
		if (tmp_ep->ep_iscsi_cid == iscsi_cid) {
			ep = tmp_ep;
			break;
		}
	}
	vmk_SpinlockUnlock(hba->lock);

	if (!ep)
		PRINT(hba, "ofld_list - icid %d not found\n", iscsi_cid);

	return ep;
}

/*
 * qfle3i_free_conn_cid_tbl_entry - returns tcp port to free list
 *
 * @hba: 		pointer to adapter instance
 * @iscsi_cid:		iscsi context ID to free
 */
static void qfle3i_free_conn_cid_tbl_entry(struct qfle3i_hba *hba,
										vmk_uint16 iscsi_cid)
{
	if (iscsi_cid == (vmk_uint16)ISCSI_RESERVED_TAG)
		return;

	vmk_SpinlockLock(hba->cid_que_lock);
	hba->cid_que.conn_cid_tbl[iscsi_cid] = NULL;
	vmk_SpinlockUnlock(hba->cid_que_lock);
}

/*
 * qfle3i_tear_down_ep - tear down endpoint and free resources
 *
 * @hba:                pointer to adapter instance
 * @ep:                 endpoint (transport indentifier) structure
 *
 * destroys cm_sock structure and on chip iscsi context
 */
static void qfle3i_tear_down_ep(struct qfle3i_hba *hba,
				 struct qfle3i_endpoint *ep)
{
	ep->state = EP_STATE_CLEANUP_START;

	vmk_TimerSchedule(qfle3i_driver_info.timer_queue,
			(void *)qfle3i_ep_ofld_timer,
			(void *) (unsigned long) ep,
			hba->conn_ctx_destroy_tmo * VMK_USEC_PER_SEC,
			VMK_TIMER_DEFAULT_TOLERANCE,
			VMK_TIMER_ATTR_NONE,
			VMK_LOCKDOMAIN_INVALID,
			VMK_SPINLOCK_UNRANKED,
			&ep->ofld_timer);

	qfle3i_ep_destroy_list_add(hba, ep);

	/* destroy iSCSI context, wait for it to complete */
	qfle3i_send_conn_destroy(hba, ep);

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "Before wait, ep->state:0x%x \n",
			  ep->state);

	ql_vmk_wait_for_completion(&ep->ofld_wait,
					(ep->state != EP_STATE_CLEANUP_START),
					100 * VMK_MSEC_PER_SEC);

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "After wait, ep->state:0x%x \n",
			  ep->state);
#if 0
	if (signal_pending(current))
		flush_signals(current);
#endif
	vmk_TimerCancel(ep->ofld_timer, VMK_TRUE);
	qfle3i_ep_destroy_list_del(hba, ep);

	if (ep->state != EP_STATE_CLEANUP_CMPL)
		/* should never happen */
		PRINT_ALERT(hba, "conn destroy failed\n");
}

/*
 * qfle3i_free_ep - returns endpoint struct and tcp port to free pool
 *
 * @endpoint:		pointer to endpoint structure
 *
 */
static void qfle3i_free_ep(struct qfle3i_endpoint *endpoint)
{

	vmk_SpinlockLock(qfle3i_resc_lock);
	if (endpoint->in_progress == 1)
		endpoint->hba->ofld_conns_active--;
	vmk_SpinlockUnlock(qfle3i_resc_lock);

	qfle3i_free_conn_cid_tbl_entry(endpoint->hba,
						endpoint->ep_iscsi_cid);

	vmk_SpinlockLock(qfle3i_resc_lock);
	endpoint->state = EP_STATE_IDLE;

	if (endpoint->ofld_wait.cmpl_lock)
		vmk_SpinlockDestroy(endpoint->ofld_wait.cmpl_lock);

	if (endpoint->conn) {
		endpoint->conn->ep = NULL;
		endpoint->conn = NULL;
	}
	endpoint->sess = NULL;
	endpoint->hba = NULL;
	endpoint->in_progress = 0;
	vmk_ListInsert(&endpoint->link, vmk_ListAtRear(&qfle3i_free_ep_list));
	qfle3i_num_free_ep++;
	vmk_SpinlockUnlock(qfle3i_resc_lock);
}

/*
 * qfle3i_alloc_ep - allocates ep structure from global pool
 *
 * @hba: 		pointer to adapter instance
 *
 * routine allocates a free endpoint structure from global pool and
 *	a tcp port to be used for this connection.  Global resource lock,
 *	'qfle3i_resc_lock' is held while accessing shared global data structures
 */
static struct qfle3i_endpoint *qfle3i_alloc_ep(struct qfle3i_hba *hba)
{
	struct qfle3i_endpoint *endpoint;
	vmk_uint16 tcp_port;
	VMK_ReturnStatus status = VMK_OK;

	vmk_SpinlockLock(qfle3i_resc_lock);

	if (hba->ofld_conns_active >= hba->max_active_conns) {
		vmk_SpinlockUnlock(qfle3i_resc_lock);
		PRINT_ERR(hba, "Max limit for ep reached.\n");
		return NULL;
	}
	tcp_port = 0;

	if (vmk_ListIsEmpty(&qfle3i_free_ep_list)) {
		vmk_SpinlockUnlock(qfle3i_resc_lock);
		PRINT_ERR(hba, "ep struct pool empty\n");
		return NULL;
	}

	endpoint = VMK_LIST_ENTRY(vmk_ListFirst(&qfle3i_free_ep_list),
								struct qfle3i_endpoint, link);
	vmk_ListRemove(&endpoint->link);
	vmk_ListInit(&endpoint->link);
	qfle3i_num_free_ep--;

	endpoint->state = EP_STATE_IDLE;
	endpoint->hba = hba;
	endpoint->hba_age = hba->age;
	hba->ofld_conns_active++;
	endpoint->tcp_port = tcp_port;
	endpoint->in_progress = 1;

	status = ql_vmk_init_completion(&endpoint->ofld_wait, "ep->ofld_wait");
	if (status != VMK_OK) {
		vmk_SpinlockUnlock(qfle3i_resc_lock);
		goto alloc_failed;
	}
	vmk_SpinlockUnlock(qfle3i_resc_lock);
	QFLE3I_DBG(DBG_CONN_SETUP, hba, "%s:: endpoint:%p allocated \n",
		  vmk_NameToString(&hba->vmnic_name), endpoint);
	return endpoint;

alloc_failed:
	QFLE3I_DBG(DBG_CONN_SETUP, hba, "%s:: endpoint:%p cmpl_var alloc failed\n",
		  vmk_NameToString(&hba->vmnic_name), endpoint);
	qfle3i_free_ep(endpoint);
	return NULL;
}

/*
 * qfle3i_release_ep_pool - releases memory resources held by endpoint structs
 */
void qfle3i_release_ep_pool(void)
{
	int index;
	void *mem_ptr;

	for (index = 0; index < MAX_PAGES_PER_CTRL_STRUCT_POOL; index++) {
		mem_ptr = qfle3i_ep_pages[index];
		if (mem_ptr != NULL)
			vmk_HeapFree(qfle3i_driver_info.heap_id, mem_ptr);
		qfle3i_ep_pages[index] = NULL;
	}
	qfle3i_num_free_ep = 0;

	if (qfle3i_resc_lock)
		vmk_SpinlockDestroy(qfle3i_resc_lock);

	return;
}

/* qfle3i_alloc_ep_pool - alloccates a pool of endpoint structures
 *
 * allocates free pool of endpoint structures, which is used to store
 *	QP related control & PT info and other option-2 information
 */
int qfle3i_alloc_ep_pool(void)
{
	struct qfle3i_endpoint *endpoint;
	int index;
	int total_endpoints;
	int page_count = 0;
	void *mem_ptr;
	int mem_size;

	ql_vmk_spin_lock_init(&qfle3i_resc_lock, LOCK_RANK_HIGHEST, "qfle3i_resc_lock");
	vmk_ListInit(&qfle3i_free_ep_list);

	for (index = 0; index < MAX_PAGES_PER_CTRL_STRUCT_POOL; index++) {
		qfle3i_ep_pages[index] = NULL;
	}

	total_endpoints = ISCSI_MAX_CONNS_PER_HBA * ISCSI_MAX_ADAPTERS;
	qfle3i_num_free_ep = 0;
	mem_size = total_endpoints * sizeof(struct qfle3i_endpoint);

	mem_ptr = vmk_HeapAlign(qfle3i_driver_info.heap_id, mem_size,
		VMK_L1_CACHELINE_SIZE);
	if (mem_ptr == NULL) {
		vmk_AlertMessage("ep_pool: mem alloc failed\n");
		goto mem_alloc_err;
	}
	vmk_Memset(mem_ptr, 0, mem_size);

	qfle3i_ep_pages[page_count++] = mem_ptr;
	endpoint = mem_ptr;

	for (index = 0; index < total_endpoints; index++) {
		vmk_ListInit(&endpoint->link);
		vmk_ListInsert(&endpoint->link, vmk_ListAtRear(&qfle3i_free_ep_list));
		endpoint++;
		qfle3i_num_free_ep++;
	}
mem_alloc_err:
	if (qfle3i_num_free_ep == 0) {
		vmk_SpinlockDestroy(qfle3i_resc_lock);
		return VMK_NO_MEMORY;
	}
	qfle3i_max_free_ep = qfle3i_num_free_ep;
	return VMK_OK;
}

void qfle3i_hba_poll_timer(unsigned long data)
{
	struct qfle3i_hba *hba = (struct qfle3i_hba *)data;
	struct qfle3i_sess *sess;
	int iscsi_cid;
	vmk_uint32 src_ip;
	vmk_uint32 dst_ip;
	vmk_uint16 tcp_port;

	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(sess, &hba->active_sess, link, struct qfle3i_sess) {

		if (sess->lead_conn && sess->lead_conn->ep) {
			iscsi_cid = sess->lead_conn->ep->ep_iscsi_cid;
			tcp_port = vmk_BE16ToCPU(sess->lead_conn->ep->cm_sk->src_port);
			src_ip = vmk_BE32ToCPU(sess->lead_conn->ep->cm_sk->src_ip[0]);
			dst_ip = vmk_BE32ToCPU(sess->lead_conn->ep->cm_sk->dst_ip[0]);
		} else {
			iscsi_cid = 0xFF;
			tcp_port = src_ip = dst_ip = 0;
		}
		vmk_LogMessage("qfle3i:hba_poll[%p] = %p, %lu, %lu, {%d, %x, %x, %x}, "
				 "{%u, %u, %u}, %u, {%u, %u}, %u, %u\n", hba,
				sess, vmk_GetTimerCycles() - sess->timestamp, sess->recovery_state,
				iscsi_cid, src_ip, dst_ip, tcp_port,
				sess->cmdsn, sess->exp_cmdsn, sess->max_cmdsn,
				(unsigned int)vmk_AtomicRead64(&sess->tmf_active),
				sess->active_cmd_count, sess->pend_cmd_count,
				sess->cmd_win_closed, sess->host_busy_cmd_win);
	}
	vmk_SpinlockUnlock(hba->lock);
}

/*
 * qfle3i_unbind_conn_from_iscsi_cid - unbinds conn structure and 'iscsi_cid'
 *
 * @conn: 		pointer to iscsi connection
 * @iscsi_cid:		iscsi context ID, range 0 - (MAX_CONN - 1)
 *
 * Remove connection pointer from iscsi cid table entry. This is required
 *	to stop invalid pointer access if there are spurious KCQE indications
 *	after iscsi logout is performed
 */
static int qfle3i_unbind_conn_from_iscsi_cid(struct qfle3i_conn *conn,
					    vmk_uint32 iscsi_cid)
{
	struct qfle3i_hba *hba;

	if (!conn || !conn->sess || !conn->sess->hba ||
	    (iscsi_cid >= conn->sess->hba->max_active_conns))
		return VMK_BAD_PARAM;

	hba = conn->sess->hba;

	if (hba && !hba->cid_que.conn_cid_tbl[iscsi_cid]) {
		PRINT_ERR(hba, "conn unbind - entry %d is already free\n",
				iscsi_cid);
		return VMK_NOT_FOUND;
	}

	hba->cid_que.conn_cid_tbl[iscsi_cid] = NULL;
	return 0;
}


/*
 * qfle3i_bind_conn_to_iscsi_cid - bind conn structure to 'iscsi_cid'
 *
 * @conn: 		pointer to iscsi connection
 * @iscsi_cid:		iscsi context ID, range 0 - (MAX_CONN - 1)
 *
 * update iscsi cid table entry with connection pointer. This enables
 *	driver to quickly get hold of connection structure pointer in
 *	completion/interrupt thread using iscsi context ID
 */
static int qfle3i_bind_conn_to_iscsi_cid(struct qfle3i_conn *conn,
					 vmk_uint32 iscsi_cid)
{
	struct qfle3i_hba *hba;

	if (!conn || !conn->sess)
		return VMK_BAD_PARAM;

	hba = conn->sess->hba;

	if (!hba)
		return VMK_BAD_PARAM;

	if (hba->cid_que.conn_cid_tbl[iscsi_cid]) {
		PRINT_ERR(hba, "conn bind - entry #%d not free\n",
				iscsi_cid);
		return VMK_BUSY;
	}

	hba->cid_que.conn_cid_tbl[iscsi_cid] = conn;
	return 0;
}

/*
 * qfle3i_get_conn_from_id - maps an iscsi cid to corresponding conn ptr
 *
 * @hba: 		pointer to adapter instance
 * @iscsi_cid:		iscsi context ID, range 0 - (MAX_CONN - 1)
 */
struct qfle3i_conn *qfle3i_get_conn_from_id(struct qfle3i_hba *hba,
						 vmk_uint16 iscsi_cid)
{
	if (!hba->cid_que.conn_cid_tbl) {
		PRINT_ERR(hba, "ERROR - missing conn<->cid table\n");
		return NULL;

	} else if (iscsi_cid >= hba->max_active_conns) {
		PRINT_ERR(hba, "wrong cid #%d\n", iscsi_cid);
		return NULL;
	}
	return hba->cid_que.conn_cid_tbl[iscsi_cid];
}


/*
 * qfle3i_alloc_iscsi_cid - allocates a iscsi_cid from free pool
 *
 * @hba: 		pointer to adapter instance
 */
static vmk_uint32 qfle3i_alloc_iscsi_cid(struct qfle3i_hba *hba)
{
	int idx;

	if (!hba->cid_que.cid_free_cnt)
		return ISCSI_RESERVED_TAG;

	vmk_SpinlockLock(hba->cid_que_lock);
	idx = hba->cid_que.cid_q_cons_idx;
	hba->cid_que.cid_q_cons_idx++;
	if (hba->cid_que.cid_q_cons_idx == hba->cid_que.cid_q_max_idx)
		hba->cid_que.cid_q_cons_idx = 0;

	hba->cid_que.cid_free_cnt--;
	vmk_SpinlockUnlock(hba->cid_que_lock);
	return hba->cid_que.cid_que[idx];
}


/*
 * qfle3i_setup_conn_cid_table - sets up free iscsi cid queue
 *
 * @hba: 		pointer to adapter instance
 *
 * allocates memory for iscsi cid queue & 'cid - conn ptr' mapping table,
 * 	and initialize table attributes
 */
static VMK_ReturnStatus qfle3i_setup_conn_cid_table(struct qfle3i_hba *hba)
{
	int mem_size;
	int i;
	VMK_ReturnStatus status = VMK_OK;

	mem_size = hba->max_active_conns * sizeof(struct qfle3i_conn *);

	/* sending mem_size directly without page alignment as,
		ql_vmk_alloc function will take care of it.
	*/
	hba->cid_que.conn_cid_tbl = vmk_HeapAlign(
						vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						mem_size, VMK_L1_CACHELINE_SIZE);
	if (!hba->cid_que.conn_cid_tbl) {
		return VMK_NO_MEMORY;
	}

	for (i = 0; i < hba->max_active_conns; i++) {
		hba->cid_que.conn_cid_tbl[i] = NULL;
	}
	status = ql_vmk_spin_lock_init(&hba->cid_que_lock,
						LOCK_RANK_HIGHEST, "hba->cid_que_lock");
	if (status != VMK_OK) {
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						hba->cid_que.conn_cid_tbl);
		hba->cid_que.conn_cid_tbl = NULL;
		return status;
	}
	return status;
}

/*
 * qfle3i_free_conn_cid_table - releases 'iscsi_cid' queue resources
 *
 * @hba:                pointer to adapter instance
 */

static void qfle3i_free_conn_cid_table(struct qfle3i_hba *hba)
{
	if (hba->cid_que_lock)
		vmk_SpinlockDestroy(hba->cid_que_lock);

	if (hba->cid_que.conn_cid_tbl)
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						hba->cid_que.conn_cid_tbl);
	hba->cid_que.conn_cid_tbl = NULL;
}

static void qfle3i_setup_cmd_wqe_template(struct qfle3i_cmd *cmd)
{
	vmk_Memset(&cmd->req, 0x00, sizeof(cmd->req));
	cmd->req.op_code = 0xFF;
	cmd->req.bd_list_addr_lo = (vmk_uint32) cmd->bd_tbl->bd_tbl_dma;
	cmd->req.bd_list_addr_hi =
		(vmk_uint32) ((vmk_uint64) cmd->bd_tbl->bd_tbl_dma >> 32);

}


/*
 * qfle3i_identify_device - identifies QLogic 1/10 Gigabit ESX
 *						Ethernet device type
 * @hba: 		Adapter structure pointer
 * @cnic:		Corresponding cnic device
 *
 * This function identifies the NX2 device type and sets appropriate
 *	queue mailbox register access method, 5709 requires driver to
 *	access MBOX regs using *bin* mode
 */
void qfle3i_identify_device(struct qfle3i_hba *hba, struct cnic_dev *dev)
{
	/* 1G is not supported, so this condition will never be true */
	if (vmk_BitVectorTest(dev->flags, CNIC_F_QFLE3_LEGACY_CLASS)) {
		if ((hba->pci_did == PCI_DEVICE_ID_NX2_5706) ||
		    (hba->pci_did == PCI_DEVICE_ID_NX2_5706S))
			vmk_BitVectorSet(hba->cnic_dev_type, QFLE3I_NX2_DEV_5706);
		else if ((hba->pci_did == PCI_DEVICE_ID_NX2_5708) ||
		    (hba->pci_did == PCI_DEVICE_ID_NX2_5708S))
			vmk_BitVectorSet(hba->cnic_dev_type, QFLE3I_NX2_DEV_5708);
		else if ((hba->pci_did == PCI_DEVICE_ID_NX2_5709) ||
		    (hba->pci_did == PCI_DEVICE_ID_NX2_5709S)) {
			vmk_BitVectorSet(hba->cnic_dev_type, QFLE3I_NX2_DEV_5709);
			hba->mail_queue_access = QFLE3I_MQ_BIN_MODE;
		}
	} else if (vmk_BitVectorTest(dev->flags, CNIC_F_QFLE3_CLASS))
		vmk_BitVectorSet(hba->cnic_dev_type, QFLE3I_NX2_DEV_57710);
	else
		PRINT_ERR(hba, "unknown device, 0x%x\n",
				  hba->pci_did);
}

static void qfle3i_setup_bd_tbl(struct qfle3i_hba *hba, struct qfle3i_dma *dma)
{
	struct iscsi_bd *mp_bdt;
	int pages = dma->size / VMK_PAGE_SIZE;
	vmk_uint64 addr;

	mp_bdt = (struct iscsi_bd *) dma->pgtbl;
	addr = (unsigned long) dma->mem;
	mp_bdt->flags = ISCSI_BD_FIRST_IN_BD_CHAIN;
	do {
		mp_bdt->buffer_addr_lo = addr & 0xffffffff;
		mp_bdt->buffer_addr_hi = addr >> 32;
		mp_bdt->buffer_length = VMK_PAGE_SIZE;

		pages--;
		if (!pages)
			break;

		addr += VMK_PAGE_SIZE;
		mp_bdt++;
		mp_bdt->flags = 0;
	} while (1);
	mp_bdt->flags |= ISCSI_BD_LAST_IN_BD_CHAIN;
}



void qfle3i_free_dma(struct qfle3i_hba *hba, struct qfle3i_dma *dma)
{
	if (dma->mem) {
		ql_vmk_dma_free(hba, dma->mem, dma->size);
		dma->mem = NULL;
	}
	if (dma->pgtbl && dma->pgtbl_type) {
		ql_vmk_dma_free(hba, dma->pgtbl, dma->pgtbl_size);
		dma->pgtbl = NULL;
	}
}

int qfle3i_alloc_dma(struct qfle3i_hba *hba, struct qfle3i_dma *dma,
		    int size, int pgtbl_type, int pgtbl_off)
{
	int pages = (size + VMK_PAGE_SIZE - 1) / VMK_PAGE_SIZE;

	dma->size = size;
	dma->pgtbl_type = pgtbl_type;

	dma->mem = ql_vmk_dma_alloc(hba, &dma->mapping, size);
	if (dma->mem == NULL)
		goto mem_err;

	if (!pgtbl_type)
		return 0;

	/* ql_vmk_dma_alloc does the page size alignment, avoid doing alignment over here,
	 * else it will waste one page memory.
	*/
	dma->pgtbl_size = ((pages * 8) + VMK_PAGE_SIZE - 1) & (~VMK_PAGE_MASK);
	//dma->pgtbl_size = (pages * 8);
	dma->pgtbl = ql_vmk_dma_alloc(hba, &dma->pgtbl_map, dma->pgtbl_size);
	if (dma->pgtbl == NULL)
		goto mem_err;

	if (pgtbl_type == QFLE3I_TBL_TYPE_PG)
		hba->setup_pgtbl(hba, dma, pgtbl_off);
	if (pgtbl_type == QFLE3I_TBL_TYPE_BD)
		qfle3i_setup_bd_tbl(hba, dma);

	return 0;

mem_err:
        qfle3i_free_dma(hba, dma);
        return VMK_NO_MEMORY;
}

/*
 * qfle3i_setup_mp_bdt - allocated BD table resources to be used as
 *			the dummy buffer for '0' payload length iscsi requests
 *
 * @hba: 		pointer to adapter structure
 *
 * allocate memory for dummy buffer and associated BD table to be used by
 *	middle path (MP) requests
 */
int qfle3i_setup_mp_bdt(struct qfle3i_hba *hba)
{
	int rc = 0;

	if (qfle3i_alloc_dma(hba, &hba->mp_dma_buf, VMK_PAGE_SIZE, QFLE3I_TBL_TYPE_BD, 0)) {
		PRINT_ERR(hba, "unable to allocate Middle Path BDT\n");
		rc = -1;
	}
	return rc;
}


/*
 * qfle3i_free_mp_bdt - releases ITT back to free pool
 *
 * @hba: 		pointer to adapter instance
 *
 * free MP dummy buffer and associated BD table
 */
void qfle3i_free_mp_bdt(struct qfle3i_hba *hba)
{
	qfle3i_free_dma(hba, &hba->mp_dma_buf);
}

/**
 * qfle3i_setup_5771x_pgtbl - iscsi QP page table setup function
 *
 * @ep: 		endpoint (transport indentifier) structure
 *
 * Sets up page tables for SQ/RQ/CQ, 1G/sec (5706/5708/5709) devices requires
 * 	64-bit address in big endian format. Whereas 10G/sec (57710) requires
 * 	PT in little endian format
 */
void qfle3i_setup_5771x_pgtbl(struct qfle3i_hba *hba, struct qfle3i_dma *dma, int pgtbl_off)
{
	int num_pages;
	vmk_uint32 *ptbl;
	vmk_IOA page;
	char *pgtbl_virt;

	/* SQ page table */
	pgtbl_virt = dma->pgtbl;
	vmk_Memset(pgtbl_virt, 0, dma->pgtbl_size);
	num_pages = dma->size / VMK_PAGE_SIZE;
	page = dma->mapping;

	ptbl = (vmk_uint32 *) ((vmk_uint8 *) dma->pgtbl + pgtbl_off);
	while (num_pages--) {
		/* PTE is written in little endian format for 57710 */
		*ptbl = (vmk_uint32) page;
		ptbl++;
		*ptbl = (vmk_uint32) ((vmk_uint64) page >> 32);
		ptbl++;
		page += VMK_PAGE_SIZE;
	}
}

/**
 * qfle3i_setup_570x_pgtbl - iscsi QP page table setup function
 *
 * @ep: 		endpoint (transport indentifier) structure
 *
 * Sets up page tables for SQ/RQ/CQ, 1G/sec (5706/5708/5709) devices requires
 * 	64-bit address in big endian format. Whereas 10G/sec (57710) requires
 * 	PT in little endian format
 */
void qfle3i_setup_570x_pgtbl(struct qfle3i_hba *hba, struct qfle3i_dma *dma, int pgtbl_off)
{
	int num_pages;
	vmk_uint32 *ptbl;
	vmk_IOA page;
	char *pgtbl_virt;

	/* SQ page table */
	pgtbl_virt = dma->pgtbl;
	vmk_Memset(pgtbl_virt, 0, dma->pgtbl_size);
	num_pages = dma->size / VMK_PAGE_SIZE;
	page = dma->mapping;

	ptbl = (vmk_uint32 *) ((vmk_uint8 *) dma->pgtbl + pgtbl_off);
	while (num_pages--) {
		/* PTE is written in big endian format for
		 * 5706/5708/5709 devices */
		*ptbl = (vmk_uint32) ((vmk_uint64) page >> 32);
		ptbl++;
		*ptbl = (vmk_uint32) page;
		ptbl++;
		page += VMK_PAGE_SIZE;
	}
}

int qfle3i_bind_adapter_devices(struct qfle3i_hba *hba)
{
	struct cnic_dev *cnic;
	int nx2_10g_dev = 0;
//	vmk_VA mapped_addr;
//	VMK_ReturnStatus status = VMK_OK;

	if (!hba->cnic)
		return VMK_INVALID_ADAPTER;

	cnic = hba->cnic;
	hba->pcidev = cnic->pdev;
	if (hba->pcidev) {
		vmk_PCIQueryDeviceAddr(hba->pcidev, &hba->sbdf);
		vmk_PCIQueryDeviceID(hba->pcidev, &hba->pcidev_id);
	}
	QFLE3I_DBG(DBG_INIT, hba, "PCI_DEV_ADDR:[%04x:%02x:%02x:%02x]\n",
			  hba->sbdf.seg, hba->sbdf.bus,
			  hba->sbdf.dev, hba->sbdf.fn);

	QFLE3I_DBG(DBG_INIT, hba, "PCI_DEV_ID: "
			  "V:0x%x D:0x%x SV:0x%x SD:0x%x\n",
			  hba->pcidev_id.vendorID, hba->pcidev_id.deviceID,
			  hba->pcidev_id.subVendorID,
			  hba->pcidev_id.subDeviceID);

	qfle3i_identify_device(hba, cnic);

#if 0
/* as these bars are already mapped by nic layer, avoid doing it over here
	find a way to get the populated through cnic
*/
	if (vmk_BitVectorTest(hba->cnic_dev_type, QFLE3I_NX2_DEV_5709)) {
		status = vmk_PCIMapIOResource(vmk_ModuleCurrentID, hba->pcidev,
					0, NULL, &mapped_addr);
		hba->regview = (void *)mapped_addr;
		if (!hba->regview)
			goto mem_err;
	} else if (vmk_BitVectorTest(hba->cnic_dev_type, QFLE3I_NX2_DEV_57710)) {
		nx2_10g_dev = 1;
		status = vmk_PCIMapIOResource(vmk_ModuleCurrentID, hba->pcidev,
					0, NULL, &mapped_addr);
		hba->regview = (void *)mapped_addr;
		if (!hba->regview)
			goto mem_err;
	}
#endif

	if (vmk_BitVectorTest(hba->cnic_dev_type, QFLE3I_NX2_DEV_57710))
		nx2_10g_dev = 1;

	if (nx2_10g_dev) {
		hba->conn_teardown_tmo = 15;
		hba->conn_ctx_destroy_tmo = 6;
		hba->hba_shutdown_tmo = 240;
	} else {
		hba->conn_teardown_tmo = 6;
		hba->conn_ctx_destroy_tmo = 2;
		hba->hba_shutdown_tmo = 30;
	}

	if (qfle3i_setup_mp_bdt(hba))
		goto mem_err;

	return 0;

mem_err:
	qfle3i_unbind_adapter_devices(hba);
	return VMK_NO_MEMORY;
}

/*
 * qfle3i_unbind_adapter_devices - removes qfle3i adapter to pcidev mapping
 *
 * @hba: 		Adapter instance
 *
 **/
void qfle3i_unbind_adapter_devices(struct qfle3i_hba *hba)
{
#if 0
/*
	VK: mapping is done by nic, so let nic free it.
*/
	if (hba->regview) {
		vmk_PCIUnmapIOResource(vmk_ModuleCurrentID, hba->pcidev, 0);
		hba->regview = NULL;
	}
#endif
	qfle3i_free_mp_bdt(hba);
#if 0
	if (hba->pcidev)
		pci_dev_put(hba->pcidev);
#endif
	hba->pcidev = NULL;
}

/*
 * qfle3i_free_hba- releases hba structure and resources held by the adapter
 *
 * @hba:                pointer to adapter instance
 *
 * free adapter structure and call various cleanup routines.
 */
void qfle3i_free_hba(struct qfle3i_hba *hba)
{

	qfle3i_unbind_adapter_devices(hba);
	qfle3i_remove_hba_from_adapter_list(hba);

	if (hba->name_string) {
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), hba->name_string);
		hba->name_string = NULL;
	}

	if (hba->vmkdev_name) {
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), hba->vmkdev_name);
		hba->vmkdev_name = NULL;
	}

	if (hba->stat_lock)
		vmk_SpinlockDestroy(hba->stat_lock);

	if (hba->sess_recov_list)
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					hba->sess_recov_list);

// find oppsite
//	ql_vmk_init_singlethread_workitem(&hba->err_rec_task, (void *)conn_err_recovery_task, hba);
//	ql_vmk_init_singlethread_workitem(&hba->ep_poll_task, (void *)ep_tmo_poll_task, hba);

	vmk_DMAEngineDestroy(hba->dmaEngine);
	vmk_SgDestroyOpsHandle(hba->SgOps_handle);

	if (en_hba_poll)
		vmk_TimerCancel(hba->hba_poll_timer, VMK_TRUE);

	if (hba->ep_tmo_wait.cmpl_lock)
		vmk_SpinlockDestroy(hba->ep_tmo_wait.cmpl_lock);

	if (hba->eh_wait.cmpl_lock)
		vmk_SpinlockDestroy(hba->eh_wait.cmpl_lock);

	if (hba->net_dev_lock)
		vmk_SemaDestroy(&hba->net_dev_lock);

	ql_vmk_tasklet_disable(&hba->scan_bh);

	if (hba->scsi_scan_list_lck)
		vmk_SpinlockDestroy(hba->scsi_scan_list_lck);
	if (hba->lock)
		vmk_SpinlockDestroy(hba->lock);

	vmk_ListInit(&hba->active_sess);
	vmk_ListInit(&hba->ep_ofld_list);
	vmk_ListInit(&hba->scsi_scan_list);
	vmk_ListInit(&hba->ep_destroy_list);
	vmk_ListInit(&hba->ep_stale_list);
 	vmk_ListInit(&hba->ep_tmo_list);
	vmk_ListInit(&hba->link);


	qfle3i_free_conn_cid_table(hba);
	vmk_BitVectorFree(qfle3i_driver_info.heap_id, hba->adapter_state);
	vmk_BitVectorFree(qfle3i_driver_info.heap_id, hba->reg_with_cnic);
	vmk_BitVectorFree(qfle3i_driver_info.heap_id, hba->cnic_dev_type);
	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), hba);
	hba = NULL;
}

struct qfle3i_hba *qfle3i_alloc_hba(struct cnic_dev *cnic, vmk_Device device)
{
	VMK_ReturnStatus status = VMK_OK;
	struct qfle3i_hba *hba;
	vmk_SgOpsHandle SgOps_handle;
	vmk_DMAEngineProps dmaProps;

	hba = vmk_HeapAlign(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
			sizeof(struct qfle3i_hba), VMK_L1_CACHELINE_SIZE);
	if (hba == NULL) {
		DRV_WARN("Failed to alloc qfle3i adapter\n", status);
		return NULL;
	}

	vmk_Memset(hba, 0, sizeof(struct qfle3i_hba));

	hba->ldev = device;
	hba->q_depth = 1024;

	/* Get PCI related information and update hba struct members */
	hba->cnic = cnic;
	//hba->netdev = cnic->netdev;
	hba->pcidev = cnic->pdev;

	vmk_ListInit(&hba->active_sess);
	vmk_ListInit(&hba->ep_ofld_list);
	vmk_ListInit(&hba->scsi_scan_list);
	vmk_ListInit(&hba->ep_destroy_list);
	vmk_ListInit(&hba->ep_stale_list);
 	vmk_ListInit(&hba->ep_tmo_list);
	vmk_ListInit(&hba->link);

	hba->cnic_dev_type = vmk_BitVectorAlloc(qfle3i_driver_info.heap_id,
							(sizeof(long) * VMK_BITS_PER_BYTE));
	if (VMK_UNLIKELY(!hba->cnic_dev_type))
		goto cnic_dev_type_bitvector_alloc_failed;
	vmk_BitVectorZap(hba->cnic_dev_type);


	hba->reg_with_cnic = vmk_BitVectorAlloc(qfle3i_driver_info.heap_id,
							(sizeof(long) * VMK_BITS_PER_BYTE));
	if (VMK_UNLIKELY(!hba->reg_with_cnic))
		goto reg_with_cnic_bitvector_alloc_failed;
	vmk_BitVectorZap(hba->reg_with_cnic);

	hba->adapter_state = vmk_BitVectorAlloc(qfle3i_driver_info.heap_id,
							(sizeof(long) * VMK_BITS_PER_BYTE));
	if (VMK_UNLIKELY(!hba->adapter_state))
		goto adapter_state_bitvector_alloc_failed;
	vmk_BitVectorZap(hba->adapter_state);


	if (qfle3i_esx_mtu_max && qfle3i_esx_mtu_max <= QFLE3I_MAX_MTU_SUPPORTED)
		hba->mtu_supported = qfle3i_esx_mtu_max;
	else
		hba->mtu_supported = QFLE3I_MAX_MTU_SUPPORTED;

	hba->max_active_conns = ISCSI_MAX_CONNS_PER_HBA;

	/* Get device type required to determine default SQ size */
	if (cnic->pdev) {
		qfle3i_identify_device(hba, cnic);
		hba->regview = cnic->regview;
		hba->doorbells = cnic->doorbells;
	}

	/* SQ/RQ/CQ size can be changed via sysfs interface */
	sq_size = QL_VMK_ROUNDUP_POW_OF_TWO(sq_size);
	if (vmk_BitVectorTest(hba->cnic_dev_type, QFLE3I_NX2_DEV_57710)) {
        	hba->setup_pgtbl = qfle3i_setup_5771x_pgtbl;
		if (sq_size <= QFLE3I_5770X_SQ_WQES_MAX &&
			sq_size >= QFLE3I_SQ_WQES_MIN)
			hba->max_sqes = sq_size;
		else {
			hba->max_sqes = QFLE3I_5770X_SQ_WQES_DEFAULT;
			PRINT_INFO(hba, "qfle3i: sq_size %d out of range, using %d\n",
							sq_size, hba->max_sqes);
		}
	} else {	/* 5706/5708/5709 */
        	hba->setup_pgtbl = qfle3i_setup_570x_pgtbl;
		if (sq_size <= QFLE3I_570X_SQ_WQES_MAX &&
			sq_size >= QFLE3I_SQ_WQES_MIN)
			hba->max_sqes = sq_size;
		else {
			if (vmk_BitVectorTest(hba->cnic_dev_type, QFLE3I_NX2_DEV_5709))
				hba->max_sqes = QFLE3I_5709_SQ_WQES_DEFAULT;
			else
				hba->max_sqes = QFLE3I_570X_SQ_WQES_DEFAULT;
			PRINT_INFO(hba, "qfle3i: sq_size %d out of range, using %d\n",
							sq_size, hba->max_sqes);
		}
	}

	rq_size = QL_VMK_ROUNDUP_POW_OF_TWO(rq_size);
        if (rq_size < QFLE3I_RQ_WQES_MIN || rq_size > QFLE3I_RQ_WQES_MAX) {

		PRINT_INFO(hba, "qfle3i: rq_size %d out of range, using %d\n",
						rq_size, QFLE3I_RQ_WQES_DEFAULT);
		rq_size = QFLE3I_RQ_WQES_DEFAULT;
	}
	hba->max_rqes = rq_size;
	hba->max_cqes = hba->max_sqes + rq_size;
	hba->num_ccell = hba->max_sqes / 2;
	QFLE3I_DBG(DBG_CONN_SETUP, hba, "QP CONF: SQ=%d, CQ=%d, RQ=%d\n",
		  hba->max_sqes, hba->max_cqes, hba->max_rqes);

	if (qfle3i_setup_conn_cid_table(hba) != VMK_OK)
		goto cid_que_err;

	status = ql_vmk_spin_lock_init(&hba->lock, LOCK_RANK_HIGHEST, "hba->lock");
	if (status != VMK_OK)
		goto hba_lock_fail;

	status = ql_vmk_spin_lock_init(&hba->scsi_scan_list_lck, LOCK_RANK_HIGHEST, "scsi_scan_lck");
	if (status != VMK_OK)
		goto hba_scan_list_lck_fail;

	status = ql_vmk_tasklet_init(&hba->scan_bh, &qfle3i_scsi_scan,
			(void *)hba);
	if (status != VMK_OK)
		goto hba_scan_bh_fail;

	status = vmk_BinarySemaCreate(&hba->net_dev_lock, qfle3i_driver_info.heap_id, "net_dev_lock");
	if (status != VMK_OK)
		goto hba_dev_lock_fail;

	status = ql_vmk_init_completion(&hba->eh_wait, "hba->eh_wait");
	if (status != VMK_OK)
		goto eh_wait_cmpl_fail;

	status = ql_vmk_init_completion(&hba->ep_tmo_wait, "hba->ep_tmo_wait");
	if (status != VMK_OK)
		goto ep_tmo_wait_cmpl_fail;

	if (en_hba_poll) {
		status = vmk_TimerSchedule(qfle3i_driver_info.timer_queue,
					(void *)qfle3i_hba_poll_timer,
					(void *) hba,
					12 * VMK_USEC_PER_SEC,
					VMK_TIMER_DEFAULT_TOLERANCE,
					VMK_TIMER_ATTR_PERIODIC,
					VMK_LOCKDOMAIN_INVALID,
					VMK_SPINLOCK_UNRANKED,
					&hba->hba_poll_timer);
	}
	if (status != VMK_OK)
		goto timer_schedule_fail;

	status = vmk_SgCreateOpsHandle(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					&SgOps_handle, NULL, NULL);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "Failed to create sg handle: %s",
							vmk_StatusToString(status));
		goto sq_handle_create_failed;
	}

	hba->SgOps_handle = SgOps_handle;

	/* Create a DMA engine for mapping DMA buffers. */
	vmk_uint8 tmp_name[64];
	vmk_Snprintf(tmp_name, 64, "%s_%lx", "qfle3i_DMAEngine", (unsigned long)hba);
	status = vmk_NameInitialize(&dmaProps.name, tmp_name);
	dmaProps.module = vmk_ModuleCurrentID;
	dmaProps.flags = VMK_DMA_ENGINE_FLAGS_NONE;
	dmaProps.device = cnic->ldev;
	dmaProps.constraints = &qfle3i_dmaConstraint;
	dmaProps.bounce = NULL;

	status = vmk_DMAEngineCreate(&dmaProps, &hba->dmaEngine);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "Failed to create DMA engine:%s\n",
						vmk_StatusToString(status));
		goto dmaEngine_create_failed;
	}

	ql_vmk_init_singlethread_workitem(&hba->err_rec_task, (void *)conn_err_recovery_task, hba);
	ql_vmk_init_singlethread_workitem(&hba->link_update, (void *)qfle3i_handle_net_event, hba);
	ql_vmk_init_singlethread_workitem(&hba->ep_poll_task, (void *)ep_tmo_poll_task, hba);

	hba->sess_recov_prod_idx = 0;
	hba->sess_recov_cons_idx = 0;
	hba->sess_recov_max_idx = 0;

	hba->sess_recov_list = vmk_HeapAlign(
						vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						VMK_PAGE_SIZE, VMK_L1_CACHELINE_SIZE);
	if (!hba->sess_recov_list)
		goto rec_que_err;

	hba->sess_recov_max_idx = VMK_PAGE_SIZE / sizeof (struct qfle3i_sess *) - 1;

	status = ql_vmk_spin_lock_init(&hba->stat_lock, LOCK_RANK_HIGHEST, "hba->stat_lock");
	if ( status != VMK_OK)
		goto hba_stat_lock_fail;
	vmk_Memset(&hba->login_stats, 0, sizeof(struct iscsi_login_stats_info));

	qfle3i_add_hba_to_adapter_list(hba);

	if (qfle3i_bind_adapter_devices(hba))
		goto pcidev_bind_err;

	return hba;

/* Failure Path */
pcidev_bind_err:
	qfle3i_remove_hba_from_adapter_list(hba);
	if (hba->stat_lock)
		vmk_SpinlockDestroy(hba->stat_lock);
hba_stat_lock_fail:
	if (hba->sess_recov_list)
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					hba->sess_recov_list);
rec_que_err:
	vmk_DMAEngineDestroy(hba->dmaEngine);
dmaEngine_create_failed:
	vmk_SgDestroyOpsHandle(hba->SgOps_handle);
sq_handle_create_failed:
	if (en_hba_poll)
		vmk_TimerCancel(hba->hba_poll_timer, VMK_TRUE);
timer_schedule_fail:
	if (hba->ep_tmo_wait.cmpl_lock)
		vmk_SpinlockDestroy(hba->ep_tmo_wait.cmpl_lock);
ep_tmo_wait_cmpl_fail:
	if (hba->eh_wait.cmpl_lock)
		vmk_SpinlockDestroy(hba->eh_wait.cmpl_lock);
eh_wait_cmpl_fail:
	if (hba->net_dev_lock)
		vmk_SemaDestroy(&hba->net_dev_lock);
hba_dev_lock_fail:
	ql_vmk_tasklet_disable(&hba->scan_bh);
hba_scan_bh_fail:
	if (hba->scsi_scan_list_lck)
		vmk_SpinlockDestroy(hba->scsi_scan_list_lck);
hba_scan_list_lck_fail:
	if (hba->lock)
		vmk_SpinlockDestroy(hba->lock);
hba_lock_fail:
	qfle3i_free_conn_cid_table(hba);
cid_que_err:
	vmk_BitVectorFree(qfle3i_driver_info.heap_id, hba->adapter_state);
adapter_state_bitvector_alloc_failed:
	vmk_BitVectorFree(qfle3i_driver_info.heap_id, hba->reg_with_cnic);
reg_with_cnic_bitvector_alloc_failed:
	vmk_BitVectorFree(qfle3i_driver_info.heap_id, hba->cnic_dev_type);
cnic_dev_type_bitvector_alloc_failed:
	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), hba);

	return NULL;
}

/*
 * qfle3i_check_route - checks if target IP route belongs to one of
 *			NX2 devices
 *
 * @dst_addr: 		target IP address
 *
 * check if route resolves to BNX2 device
 */
static VMK_ReturnStatus qfle3i_check_route(struct qfle3i_hba *hba,
										void *dst_addr,
										vmk_IscsiNetHandle iscsiNetHandle)
{
#if 0
	struct qfle3i_hba *chk_hba = NULL;
#endif
	vmk_SocketIPAddress *desti = (vmk_SocketIPAddress *)dst_addr;
	struct cnic_dev *cnic = NULL;

	if (qfle3i_adapter_ready(hba) != VMK_OK) {
		PRINT_ALERT(hba, "check route, hba not found\n");
		goto no_nx2_route;
	}

	if (hba && hba->cnic)
		cnic = hba->cnic->cm_select_dev(iscsiNetHandle, desti, CNIC_ULP_ISCSI);

	if (!cnic || !hba) {
		vmk_WarningMessage("check route, can't connect using cnic\n");
		goto no_nx2_route;
	}

	if (hba->cnic->mtu > hba->mtu_supported) {
		PRINT_ALERT(hba, "%s network i/f mtu is set to %d\n",
				  vmk_NameToString(&hba->vmnic_name), hba->cnic->mtu);
		PRINT_ALERT(hba, "iSCSI HBA can support mtu of %d\n",
				  hba->mtu_supported);
		goto no_nx2_route;
	}

	return VMK_OK;
no_nx2_route:
	return VMK_FAILURE;
}

/*
 * qfle3i_check_nx2_dev_busy - this routine unregister devices if
 *			there are no active conns
 */
void qfle3i_check_nx2_dev_busy(void)
{
	return;
}

void qfle3i_dump_queues( struct qfle3i_hba *hba,
		struct qfle3i_endpoint *ep)
{
	if (!qfle3i_dump_queue)
		return;

	if (!hba)
		return;

	if (ep) {
		qfle3i_print_sqe(ep);
		qfle3i_print_cqe(ep);
	}
	hba->cnic->dump_queues(hba->cnic);

	VMK_ASSERT(0);
}

/*
 * qfle3i_tear_down_conn - tear down iscsi/tcp connection and free resources
 *
 * @hba: 		pointer to adapter instance
 * @ep: 		endpoint (transport indentifier) structure
 *
 * destroys cm_sock structure and on chip iscsi context
 */
static int qfle3i_tear_down_conn(struct qfle3i_hba *hba,
				 struct qfle3i_endpoint *ep)
{

	if (vmk_BitVectorTest(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED) &&
		ep->state != EP_STATE_DISCONN_TIMEDOUT &&
		ep->cm_sk)
		hba->cnic->cm_destroy(ep->cm_sk);

	if (vmk_BitVectorTest(ep->hba->adapter_state, ADAPTER_STATE_GOING_DOWN))
		ep->state = EP_STATE_DISCONN_COMPL;

	if (vmk_BitVectorTest(hba->cnic_dev_type, QFLE3I_NX2_DEV_57710) &&
		ep->state == EP_STATE_DISCONN_TIMEDOUT) {
		PRINT_ALERT(hba, "###FW did not respond to Disconnect RAMROD Req; sess %p ep %p {0x%x, 0x%x}\n",
				ep->sess, ep, ep->ep_iscsi_cid, ep->ep_cid);

		qfle3i_dump_queues(hba, ep);

		ep->timestamp = vmk_GetTimerCycles();
		vmk_ListInsert(&ep->link, vmk_ListAtRear(&hba->ep_tmo_list));
		hba->ep_tmo_active_cnt++;
		if (!vmk_AtomicRead64(&hba->ep_tmo_poll_enabled)) {
			vmk_AtomicWrite64(&hba->ep_tmo_poll_enabled, 1);

			ql_vmk_singlethread_queue_work(qfle3i_driver_info.delayed_wq,
								&hba->ep_poll_task);
			PRINT_ALERT(hba, "ep_tmo_poll_task is activated.\n");
		}
		return 1;
	}
	qfle3i_tear_down_ep(hba, ep);
	return 0;
}

/*
 * qfle3i_alloc_bdt_resc_page - allocated a page to track BD table memory
 *
 * @sess:		iscsi session pointer
 *
 * allocated a page to track BD table memory
 */
struct bd_resc_page *qfle3i_alloc_bdt_resc_page(struct qfle3i_sess *sess)
{
	void *mem_ptr;
	struct bd_resc_page *resc_page;

	mem_ptr = vmk_HeapAlign(
					vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					VMK_PAGE_SIZE, VMK_L1_CACHELINE_SIZE);
	if (!mem_ptr)
		return NULL;

	resc_page = mem_ptr;
	vmk_ListInsert(&resc_page->link, vmk_ListAtRear(&sess->bd_resc_page));
	resc_page->max_ptrs = (VMK_PAGE_SIZE -
		((vmk_ByteCount) &((struct bd_resc_page *) 0)->page[0])) / sizeof(void *);
	resc_page->num_valid = 0;

	return resc_page;
}

/*
 * qfle3i_add_bdt_resc_page - add newly allocated memory page to list
 *
 * @sess:		iscsi session pointer
 * @bd_page:		pointer to page memory
 *
 * link newly allocated memory page to tracker list
 */
int qfle3i_add_bdt_resc_page(struct qfle3i_sess *sess, void *bd_page)
{
	struct bd_resc_page *resc_page;

#define is_resc_page_full(_resc_pg) (_resc_pg->num_valid == _resc_pg->max_ptrs)
#define active_resc_page(_resc_list) 	\
			(vmk_ListIsEmpty(_resc_list) ? NULL : (_resc_list)->prevPtr)
	if (vmk_ListIsEmpty(&sess->bd_resc_page)) {
		resc_page = qfle3i_alloc_bdt_resc_page(sess);
	} else {
		resc_page = (struct bd_resc_page *)
					active_resc_page(&sess->bd_resc_page);
	}

	if (!resc_page)
		return VMK_NO_MEMORY;

	resc_page->page[resc_page->num_valid++] = bd_page;
	if (is_resc_page_full(resc_page)) {
		qfle3i_alloc_bdt_resc_page(sess);
	}
	return 0;
}

/*
 * qfle3i_free_all_bdt_resc_pages - releases memory held by BD memory tracker tbl
 *
 * @sess:		iscsi session pointer
 *
 * Free up memory pages allocated held by BD resources
 */
static void qfle3i_free_all_bdt_resc_pages(struct qfle3i_sess *sess)
{
	int i;
	struct bd_resc_page *resc_page;

	vmk_SpinlockLock(sess->lock);
	while (!vmk_ListIsEmpty(&sess->bd_resc_page)) {
		resc_page = (struct bd_resc_page *)sess->bd_resc_page.prevPtr;
		vmk_ListRemove(sess->bd_resc_page.prevPtr);
		if(!resc_page)
			continue;
		for(i = 0; i < resc_page->num_valid; i++) {
			if(resc_page->page[i])
				vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
								resc_page->page[i]);
		}
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						resc_page);
	}
	vmk_SpinlockUnlock(sess->lock);
}

/*
 * qfle3i_alloc_bd_table - Alloc BD table to associate with this iscsi cmd 
 *
 * @sess:		iscsi session pointer
 * @cmd:		iscsi cmd pointer
 *
 * allocates a BD table and assigns it to given command structure. There is
 *	no synchronization issue as this code is executed in initialization
 *	thread
 */
static struct io_bdt *qfle3i_alloc_bd_table(struct qfle3i_sess *sess,
					   struct qfle3i_cmd *cmd)
{
	struct io_bdt *bd_tbl;

	if (vmk_ListIsEmpty(&sess->bd_tbl_list))
		return NULL;

	bd_tbl = VMK_LIST_ENTRY(vmk_ListFirst(&sess->bd_tbl_list),
								struct io_bdt, link);

	vmk_ListRemove(&bd_tbl->link);
	vmk_ListInsert(&bd_tbl->link, vmk_ListAtRear(&sess->bd_tbl_active));
	bd_tbl->bd_valid = 0;
	bd_tbl->cmdp = cmd;

	return bd_tbl;
}

/*
 * qfle3i_alloc_bd_table_pool - Allocates buffer descriptor (BD) pool
 *
 * @sess:		iscsi session pointer
 *
 * Allocate a pool of buffer descriptor tables and associated DMA'able memory
 *	to be used with the session.
 */
static int qfle3i_alloc_bd_table_pool(struct qfle3i_sess *sess)
{
	int index, count;
	int ret_val = 0;
	int num_elem_per_page;
	int num_pages;
	struct io_bdt *bdt_info;
	void *mem_ptr;
	int bd_tbl_size;
	vmk_uint32 mem_size;
	int total_bd_tbl;
	struct qfle3i_dma *dma;

	vmk_ListInit(&sess->bd_resc_page);
	vmk_ListInit(&sess->bd_tbl_list);
	vmk_ListInit(&sess->bd_tbl_active);

	total_bd_tbl = sess->sq_size;
	mem_size = total_bd_tbl * sizeof(struct io_bdt);
	num_elem_per_page = VMK_PAGE_SIZE / sizeof(struct io_bdt);

	for (index = 0; index < total_bd_tbl; index += num_elem_per_page) {
		if (((total_bd_tbl - index) * sizeof(struct io_bdt))
		    >= VMK_PAGE_SIZE) {
			mem_size = VMK_PAGE_SIZE;
			num_elem_per_page = VMK_PAGE_SIZE / sizeof(struct io_bdt);
		} else {
			mem_size =
				(total_bd_tbl - index) * sizeof(struct io_bdt);
			num_elem_per_page = (total_bd_tbl - index);
		}
		mem_ptr = vmk_HeapAlign(
					vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					mem_size, VMK_L1_CACHELINE_SIZE);
		if (mem_ptr == NULL) {
			PRINT_ERR(sess->hba, "alloc_bd_tbl: kmalloc failed\n");
			ret_val = VMK_NO_MEMORY;
			goto resc_alloc_failed;
		}
		qfle3i_add_bdt_resc_page(sess, mem_ptr);

		vmk_Memset(mem_ptr, 0, mem_size);
		bdt_info = mem_ptr;
		for (count = 0; count < num_elem_per_page; count++) {
			vmk_ListInit(&bdt_info->link);
			vmk_ListInsert(&bdt_info->link, vmk_ListAtRear(&sess->bd_tbl_list));
			bdt_info++;
		}
	}

	vmk_ListInit(&sess->bdt_dma_resc);

	bd_tbl_size = ISCSI_MAX_BDS_PER_CMD * sizeof(struct iscsi_bd);
	bdt_info = (struct io_bdt *)sess->bd_tbl_list.nextPtr;
	num_elem_per_page = VMK_PAGE_SIZE / bd_tbl_size;

	num_pages = ((sess->sq_size * bd_tbl_size) + VMK_PAGE_SIZE - 1) &
		    ~(VMK_PAGE_SIZE - 1);
	num_pages /= VMK_PAGE_SIZE;
	sess->bdt_dma_info = vmk_HeapAlign(
					vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					(sizeof(*dma) * num_pages), VMK_L1_CACHELINE_SIZE);
	if (sess->bdt_dma_info == NULL) {
		ret_val = VMK_NO_MEMORY;
		goto resc_alloc_failed;
	}
	vmk_Memset(sess->bdt_dma_info, 0, num_pages * sizeof(*dma));
	dma = (struct qfle3i_dma *)sess->bdt_dma_info;
	while (bdt_info && (bdt_info != (struct io_bdt *)&sess->bd_tbl_list)) {
		if (qfle3i_alloc_dma(sess->hba, dma, VMK_PAGE_SIZE, 0, 0)) {
			PRINT_ERR(sess->hba, "bd_tbl: DMA mem alloc failed\n");
			ret_val = VMK_NO_MEMORY;
			goto dma_alloc_failed;
		}
		vmk_ListInsert(&dma->link, vmk_ListAtRear(&sess->bdt_dma_resc));

		for (count = 0; count < num_elem_per_page; count++) {
			bdt_info->bd_tbl = (struct iscsi_bd *)(dma->mem +
						(count * bd_tbl_size));
			bdt_info->bd_tbl_dma = dma->mapping + count * bd_tbl_size;
			bdt_info->max_bd_cnt = ISCSI_MAX_BDS_PER_CMD;
			bdt_info->bd_valid = 0;
			bdt_info->cmdp = NULL;
			bdt_info = (struct io_bdt *)bdt_info->link.nextPtr;
			if (bdt_info == (struct io_bdt *)&sess->bd_tbl_list)
				break;
		}
		dma++;
	}
	return ret_val;

resc_alloc_failed:
dma_alloc_failed:
	return ret_val;
}

/*
 * qfle3i_free_bd_table_pool - releases resources held by BD table pool
 *
 * @sess:		iscsi session pointer
 *
 * releases BD table pool memory
 */
void qfle3i_free_bd_table_pool(struct qfle3i_sess *sess)
{
	struct qfle3i_dma *dma;

	ql_vmk_list_each_entry(dma, &sess->bdt_dma_resc, link, struct qfle3i_dma)
		qfle3i_free_dma(sess->hba, dma);

	if(sess->bdt_dma_info)
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						sess->bdt_dma_info);
}

struct qfle3i_cmd *qfle3i_alloc_cmd(struct qfle3i_sess *sess)
{
	struct qfle3i_cmd *cmd;

	if (VMK_UNLIKELY(!sess || (sess->num_free_cmds == 0))) {
		return NULL;
	}

	if (vmk_ListIsEmpty(&sess->free_cmds) && sess->num_free_cmds) {
		/* this is wrong */
		PRINT_ERR(sess->hba, "alloc %d, freed %d, num_free %d\n",
			  sess->total_cmds_allocated,
			  sess->total_cmds_freed, sess->num_free_cmds);
		return NULL;
	}

	cmd = VMK_LIST_ENTRY(vmk_ListFirst(&sess->free_cmds),
								struct qfle3i_cmd, link);
	vmk_ListRemove(&cmd->link);
	vmk_ListInit(&cmd->link);

	sess->num_free_cmds--;
	sess->total_cmds_allocated++;
	cmd->scsi_status_rcvd = 0;

	qfle3i_setup_cmd_wqe_template(cmd);
	cmd->req.itt = cmd->itt;
	return cmd;
}


/*
 * qfle3i_free_cmd - releases iscsi cmd struct & ITT to respective free pool
 *
 * @sess:		iscsi session pointer
 * @cmd:		iscsi cmd pointer
 *
 * return command structure and ITT back to free pool.
 */
void qfle3i_free_cmd(struct qfle3i_sess *sess, struct qfle3i_cmd *cmd)
{
	if (vmk_AtomicRead64(&cmd->cmd_state) == ISCSI_CMD_STATE_FREED) {
		PRINT_ALERT(sess->hba, "double freeing cmd %p\n", cmd);
		return;
	}

	if (&cmd->link == NULL) {
		PRINT_ALERT(sess->hba,
					"cmd->link is null.  cmd:%p\n", cmd);
		return;
	}
	if (&cmd->link == cmd->link.nextPtr) {
		PRINT_ALERT(sess->hba,
					"cmd not in a active_list. cmd:%p\n", cmd);
		goto skip;
	}
	vmk_ListRemove(&cmd->link);
	vmk_ListInit(&cmd->link);
skip:
	vmk_ListInsert(&cmd->link, vmk_ListAtRear(&sess->free_cmds));
	sess->num_free_cmds++;
	sess->total_cmds_freed++;
	vmk_AtomicWrite64(&cmd->cmd_state, ISCSI_CMD_STATE_FREED);
}


/*
 * qfle3i_scsi_cmd_in_active_list -
 *
 * @sess: 		session pointer
 * @sc: 		SCSI-ML command pointer
 *
 * Check to see if command to be aborted is in the pending list.
 * This routine is called with sess->lock held.
 */
struct qfle3i_cmd *qfle3i_scsi_cmd_in_active_list(struct qfle3i_sess *sess,
						vmk_ScsiCommandId cmdId, vmk_uint32 worldId, int lun_id)
{
	struct qfle3i_cmd *cmd, *tmp_cmd;

	ql_vmk_list_each_entry_safe(cmd, tmp_cmd, &sess->active_cmd_list,
							link, struct qfle3i_cmd) {
		if ((cmd->scsi_cmd->cmdId.serialNumber ==
							cmdId.serialNumber) &&
			(cmd->scsi_cmd->worldId == worldId) &&
			(cmd->ilun->lun_id == lun_id))
			return cmd;
	}
	return NULL;
}


/*
 * qfle3i_scsi_cmd_in_pend_list -
 *
 * @sess: 		session pointer
 * @sc: 		SCSI-ML command pointer
 *
 * Check to see if command to be aborted is in the pending list.
 * This routine is called with sess->lock held.
 */
struct qfle3i_scsi_task *qfle3i_scsi_cmd_in_pend_list(struct qfle3i_sess *sess,
						vmk_ScsiCommandId cmdId, vmk_uint32 worldId, int lun_id)
{
	struct qfle3i_scsi_task *scsi_task, *tmp_task;

	ql_vmk_list_each_entry_safe(scsi_task, tmp_task, &sess->pend_cmd_list,
							link, struct qfle3i_scsi_task) {
		if ((scsi_task->scsi_cmd->cmdId.serialNumber ==
							cmdId.serialNumber) &&
			(scsi_task->scsi_cmd->worldId == worldId) &&
			(scsi_task->ilun->lun_id == lun_id))
			return scsi_task;
	}
	return NULL;
}

/*
 * qfle3i_alloc_cmd_pool - allocates and initializes iscsi command pool
 *
 * @sess:		iscsi session pointer
 *
 * Allocate command structure pool for a given iSCSI session. Return 'ENOMEM'
 *	if memory allocation fails
 */
int qfle3i_alloc_cmd_pool(struct qfle3i_sess *sess)
{
	struct qfle3i_cmd *cmdp;
	int index, count;
	int ret_val = 0;
	int total_cmds;
	int num_cmds;
	int page_count;
	int num_cmds_per_page;
	void *mem_ptr;
	vmk_uint32 mem_size;
	int cmd_i;

	for (index = 0; index < MAX_PAGES_PER_CTRL_STRUCT_POOL; index++)
		sess->cmd_pages[index] = NULL;

	num_cmds_per_page = VMK_PAGE_SIZE / sizeof(struct qfle3i_cmd);

	total_cmds = sess->sq_size;
	mem_size = sess->sq_size * sizeof(struct qfle3i_cmd *);
	mem_size = (mem_size + (VMK_PAGE_SIZE - 1)) & (~VMK_PAGE_MASK);
	sess->itt_cmd = vmk_HeapAlign(
					vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					mem_size, VMK_L1_CACHELINE_SIZE);
	if (!sess->itt_cmd)
		return VMK_NO_MEMORY;

	vmk_Memset(sess->itt_cmd, 0x00, mem_size);

	cmd_i = 0;
	page_count = 0;
	for (index = 0; index < total_cmds;) {
		mem_ptr = vmk_HeapAlign(
					vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					VMK_PAGE_SIZE, VMK_L1_CACHELINE_SIZE);

		if (mem_ptr == NULL)
			break;

		sess->cmd_pages[page_count++] = mem_ptr;
		num_cmds = num_cmds_per_page;
		if ((total_cmds - index) < num_cmds_per_page)
			num_cmds = (total_cmds - index);

		vmk_Memset(mem_ptr, 0, VMK_PAGE_SIZE);
		cmdp = mem_ptr;
		for (count = 0; count < num_cmds; count++) {
			cmdp->req.itt = ITT_INVALID_SIGNATURE;
			vmk_ListInit(&cmdp->link);
			cmdp->itt = cmd_i;
			sess->itt_cmd[cmd_i] = cmdp;
			cmd_i++;

			/* Allocate BD table */
			cmdp->bd_tbl = qfle3i_alloc_bd_table(sess, cmdp);
			if (!cmdp->bd_tbl) {
				/* should never fail, as it's guaranteed to have
				 * (ISCSI_MAX_CMDS_PER_SESS + 1) BD tables
				 * allocated before calling this function.
				 */
				PRINT_ERR(sess->hba, "no BD table cmd %p\n",
					  cmdp);
				goto bd_table_failed;
			}
			vmk_ListInsert(&cmdp->link, vmk_ListAtRear(&sess->free_cmds));
			cmdp++;
		}

		sess->num_free_cmds += num_cmds;
		index += num_cmds;
	}
	sess->allocated_cmds = sess->num_free_cmds;

	if (sess->num_free_cmds == 0)
		ret_val = VMK_NO_MEMORY;
	return ret_val;

bd_table_failed:
	return VMK_NO_MEMORY;
}


/*
 * qfle3i_free_cmd_pool - releases memory held by free iscsi cmd pool
 *
 * @sess:		iscsi session pointer
 *
 * Release memory held by command struct pool.
 */
void qfle3i_free_cmd_pool(struct qfle3i_sess *sess)
{
	int index;
	void *mem_ptr;

	if (sess->num_free_cmds != sess->allocated_cmds) {
		/*
		 * WARN: either there is some command struct leak or
		 * still some SCSI commands are pending.
		 */
		PRINT_ERR(sess->hba, "missing cmd structs - %d, %d\n",
			  sess->num_free_cmds, sess->allocated_cmds);
	}
	for (index = 0; index < MAX_PAGES_PER_CTRL_STRUCT_POOL; index++) {
		mem_ptr = sess->cmd_pages[index];
		if(mem_ptr) {
			vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
							mem_ptr);
			sess->cmd_pages[index] = NULL;
		}
	}
	sess->num_free_cmds = sess->allocated_cmds = 0;

	if(sess->itt_cmd) {
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						sess->itt_cmd);
		sess->itt_cmd = NULL;
	}

	return;
}

struct qfle3i_scsi_task *qfle3i_alloc_scsi_task(struct qfle3i_sess *sess)
{
	struct qfle3i_scsi_task *scsi_task;

	if (vmk_ListIsEmpty(&sess->scsi_task_list)) {
		return NULL;
	}

	scsi_task = VMK_LIST_ENTRY(vmk_ListFirst(&sess->scsi_task_list),
								struct qfle3i_scsi_task, link);
	vmk_ListRemove(&scsi_task->link);
	vmk_ListInit(&scsi_task->link);

	return scsi_task;
}

void qfle3i_free_scsi_task(struct qfle3i_sess *sess,
				 struct qfle3i_scsi_task *scsi_task)
{
	vmk_ListRemove(&scsi_task->link);
	vmk_ListInit(&scsi_task->link);

	scsi_task->scsi_cmd = NULL;
	scsi_task->app_cmd = 0;
	scsi_task->ilun = NULL;
	vmk_ListInsert(&scsi_task->link, vmk_ListAtRear(&sess->scsi_task_list));
}


extern int qfle3i_max_task_pgs;
static int qfle3i_alloc_scsi_task_pool(struct qfle3i_sess *sess)
{
	struct qfle3i_scsi_task *scsi_task;
	int mem_size;
	int task_count;
	int i;

	if (qfle3i_max_task_pgs > 8)
		qfle3i_max_task_pgs = 8;
	else if (qfle3i_max_task_pgs < 2)
		qfle3i_max_task_pgs = 2;
	mem_size = qfle3i_max_task_pgs * VMK_PAGE_SIZE;
	vmk_ListInit(&sess->scsi_task_list);
	sess->task_list_mem = vmk_HeapAlign(
					vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					mem_size, VMK_L1_CACHELINE_SIZE);
	if (!sess->task_list_mem)
		return VMK_NO_MEMORY;

	scsi_task = (struct qfle3i_scsi_task *)sess->task_list_mem;
	task_count = mem_size / sizeof(struct qfle3i_scsi_task);
	sess->max_iscsi_tasks = task_count;
	for (i = 0; i < task_count; i++, scsi_task++) {
		vmk_ListInit(&scsi_task->link);
		scsi_task->scsi_cmd = NULL;
		scsi_task->ilun = NULL;
		vmk_ListInsert(&scsi_task->link, vmk_ListAtRear(&sess->scsi_task_list));
	}
	return 0;
}

static void qfle3i_free_scsi_task_pool(struct qfle3i_sess *sess)
{
	if(sess->task_list_mem) {
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						sess->task_list_mem);
		sess->task_list_mem = NULL;
	}
	vmk_ListInit(&sess->scsi_task_list);
/*TODO - clear pend list too */
}

void qfle3i_free_iscsilun(struct qfle3i_iscsilun *iscsilun)
{
	qfle3i_sess *sess = iscsilun->sess;

	PRINT_INFO(sess->hba, "Freeing LUN %p %d:%d:%d on sess %p\n",
		iscsilun, iscsilun->channel_id, iscsilun->target_id, iscsilun->lun_id, sess);
	vmk_SpinlockLock(sess->iscsilun_lock);
	vmk_ListRemove(&iscsilun->link);
	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
		iscsilun);
	iscsilun = NULL;
	vmk_SpinlockUnlock(sess->iscsilun_lock);
}

struct qfle3i_iscsilun *qfle3i_allocate_lun(qfle3i_sess *sess,
				int channel_id, int target_id, int lun_id)
{
	qfle3i_iscsilun *iscsilun = NULL;

	iscsilun = vmk_HeapAlign(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
				sizeof(qfle3i_iscsilun), VMK_L1_CACHELINE_SIZE);
	if (iscsilun) {
		iscsilun->sess = sess;
		iscsilun->channel_id = channel_id;
		iscsilun->target_id = target_id;
		iscsilun->lun_id = lun_id;
		iscsilun->q_depth = sess->q_depth;
		iscsilun->tmf_in_progress = 0;
		vmk_SpinlockLock(sess->iscsilun_lock);
		vmk_ListInsert(&iscsilun->link, vmk_ListAtRear(&sess->iscsilun_list));
		vmk_SpinlockUnlock(sess->iscsilun_lock);

		PRINT_INFO(sess->hba, "LUN %p Allocated %d:%d:%d on sess %p\n",
			iscsilun, channel_id, target_id, lun_id, sess);
	}

	return iscsilun;
}


struct qfle3i_iscsilun *qfle3i_get_iscsilun(
		struct qfle3i_sess *sess, int lun_id)
{
	struct qfle3i_iscsilun *iscsilun = NULL;

	vmk_SpinlockLock(sess->iscsilun_lock);
	ql_vmk_list_each_entry(iscsilun, &sess->iscsilun_list, link, qfle3i_iscsilun) {
		if (iscsilun) {
			if (lun_id == iscsilun->lun_id) {
				vmk_SpinlockUnlock(sess->iscsilun_lock);
				return iscsilun;
			}
		}
	}
	vmk_SpinlockUnlock(sess->iscsilun_lock);
	return NULL;
}

struct qfle3i_sess *qfle3i_get_session(
		struct qfle3i_hba *hba, int channel_id,
		int target_id)
{
	struct qfle3i_sess *sess = NULL;

	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(sess, &hba->active_sess, link, struct qfle3i_sess) {
		if (sess) {
			if ((target_id == sess->target_id) &&
				(channel_id == sess->channel_id)) {
				vmk_SpinlockUnlock(hba->lock);
				return sess;
			}
		}
	}
	vmk_SpinlockUnlock(hba->lock);
	return NULL;
}

/*
 * qfle3i_iscsi_sess_release - cleanup iscsi session & reclaim all resources
 *
 * @hba: 		pointer to adapter instance
 * @sess:		iscsi session pointer
 *
 * free up resources held by this session including ITT queue, cmd struct pool,
 *	BD table pool. HBA lock is held while manipulating active session list
 */
void qfle3i_iscsi_sess_release(struct qfle3i_sess *sess)
{
	struct qfle3i_hba *hba = sess->hba;

	if (sess->login_nopout_cmd) {
		/* set cmd state so that free_cmd() accepts it */
		vmk_AtomicWrite64(&sess->login_nopout_cmd->cmd_state,
			   ISCSI_CMD_STATE_COMPLETED);
		qfle3i_free_cmd(sess, sess->login_nopout_cmd);
	}
	if (sess->scsi_tmf_cmd) {
		vmk_AtomicWrite64(&sess->scsi_tmf_cmd->cmd_state,
			   ISCSI_CMD_STATE_COMPLETED);
		qfle3i_free_cmd(sess, sess->scsi_tmf_cmd);
	}
	if (sess->nopout_resp_cmd) {
		vmk_AtomicWrite64(&sess->nopout_resp_cmd->cmd_state,
			   ISCSI_CMD_STATE_COMPLETED);
		qfle3i_free_cmd(sess, sess->nopout_resp_cmd);
	}

	sess->login_nopout_cmd = NULL;
	sess->scsi_tmf_cmd = NULL;
	sess->nopout_resp_cmd = NULL;

	if (sess->er_wait.cmpl_lock)
		vmk_SpinlockDestroy(sess->er_wait.cmpl_lock);

	qfle3i_free_bd_table_pool(sess);
	qfle3i_free_all_bdt_resc_pages(sess);
	qfle3i_free_cmd_pool(sess);
	qfle3i_free_scsi_task_pool(sess);

	if (sess->device_lock)
		vmk_SpinlockDestroy(sess->device_lock);

	vmk_SemaDestroy(&sess->tmf_mutex);

	if (sess->iscsilun_lock)
		vmk_SpinlockDestroy(sess->iscsilun_lock);

	if (sess->lock)
		vmk_SpinlockDestroy(sess->lock);


	vmk_SpinlockLock(hba->lock);
	vmk_ListRemove(&sess->link);
	vmk_ListInit(&sess->link);
	vmk_SpinlockUnlock(hba->lock);

	if (sess->target_name) {
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), sess->target_name);
		sess->target_name = NULL;
	}

	vmk_IscsiTransportFreeSession(sess->trans_sess);

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "destroying sess %p completed\n",
					sess);
	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), sess);
}

/*
 * qfle3i_iscsi_sess_new - initialize newly allocated session structure
 *
 * @hba: 		pointer to adapter instance
 * @sess:		iscsi session pointer
 *
 * initialize session structure elements and allocate per sess resources.
 *	Some of the per session resources allocated are command struct pool,
 *	BD table pool and ITT queue region
 */
int qfle3i_iscsi_sess_new(struct qfle3i_hba *hba, struct qfle3i_sess *sess)
{
	VMK_ReturnStatus status = VMK_OK;

	sess->sq_size = hba->max_sqes;
	sess->tsih = 0;
	sess->lead_conn = NULL;
	sess->worker_time_slice = 1;

	sess->state = QFLE3I_SESS_INITIAL;
	sess->recovery_state = 0;
	vmk_AtomicWrite64(&sess->tmf_active, 0);
	sess->alloc_scsi_task_failed = 0;

	status = ql_vmk_spin_lock_init(&sess->lock, LOCK_RANK_HIGHEST, "sess->lock");
	if (status != VMK_OK)
		goto sess_lock_fail;

	status = ql_vmk_spin_lock_init(&sess->iscsilun_lock, LOCK_RANK_HIGHEST, "sess->iscsilun_lock");
	if (status != VMK_OK)
		goto iscsilun_lock_fail;

	status = ql_vmk_spin_lock_init(&sess->device_lock, LOCK_RANK_MEDIUM, "sess->device_lock");
	if (status != VMK_OK)
		goto sess_devicelock_fail;

	status = ql_vmk_init_completion(&sess->er_wait, "sess->er_wait");
	if (status != VMK_OK)
		goto err_wait_cmpl_fail;

	status = vmk_BinarySemaCreate(&sess->tmf_mutex, qfle3i_driver_info.heap_id, "sess->tmf_mutex");
	if (status != VMK_OK)
		goto tmf_mutex_fail;

	if (qfle3i_alloc_bd_table_pool(sess) != 0) {
		PRINT_ERR(hba, "sess_new: unable to alloc bd table pool\n");
		goto err_bd_pool;
	}

	vmk_ListInit(&sess->free_cmds);
	if (qfle3i_alloc_cmd_pool(sess) != 0) {
		PRINT_ERR(hba, "sess_new: alloc cmd pool failed\n");
		goto err_cmd_pool;
	}

	if (qfle3i_alloc_scsi_task_pool(sess) != 0) {
		PRINT_ERR(hba, "sess_new: alloc scsi_task pool failed\n");
		goto err_sc_pool;
	}

	/* initialize active connection list */
	vmk_ListInit(&sess->conn_list);
	vmk_ListInit(&sess->iscsilun_list);

	vmk_ListInit(&sess->pend_cmd_list);
	sess->pend_cmd_count = 0;
	vmk_ListInit(&sess->active_cmd_list);
	sess->active_cmd_count = 0;

	vmk_AtomicWrite64(&sess->login_noop_pending, 0);
	vmk_AtomicWrite64(&sess->logout_pending, 0);
	vmk_AtomicWrite64(&sess->tmf_pending, 0);

	sess->login_nopout_cmd = NULL;
	sess->scsi_tmf_cmd = NULL;
	sess->nopout_resp_cmd = NULL;

	sess->num_active_conn = 0;
	sess->max_conns = 1;
	sess->target_name = NULL;

	vmk_SpinlockLock(hba->lock);
	vmk_ListInsert(&sess->link, vmk_ListAtRear(&hba->active_sess));
	hba->num_active_sess++;
	vmk_SpinlockUnlock(hba->lock);

	return VMK_OK;

/* Failure Path */
err_sc_pool:
err_cmd_pool:
	qfle3i_free_cmd_pool(sess);
err_bd_pool:
	qfle3i_free_bd_table_pool(sess);
	qfle3i_free_all_bdt_resc_pages(sess);
	if (sess->tmf_mutex)
		vmk_SemaDestroy(&sess->tmf_mutex);
tmf_mutex_fail:
	if (sess->er_wait.cmpl_lock)
		vmk_SpinlockDestroy(sess->er_wait.cmpl_lock);
err_wait_cmpl_fail:
	if (sess->device_lock)
		vmk_SpinlockDestroy(sess->device_lock);
sess_devicelock_fail:
	if (sess->iscsilun_lock)
		vmk_SpinlockDestroy(sess->iscsilun_lock);
iscsilun_lock_fail:
	if (sess->lock)
		vmk_SpinlockDestroy(sess->lock);
sess_lock_fail:
	return status;
}

void qfle3i_conn_free_login_resources(struct qfle3i_hba *hba,
				     struct qfle3i_conn *conn)
{
	qfle3i_free_dma(hba, &conn->gen_pdu.login_req);
	qfle3i_free_dma(hba, &conn->gen_pdu.login_resp);
}


static int qfle3i_conn_alloc_login_resources(struct qfle3i_hba *hba,
					    struct qfle3i_conn *conn)
{
	/* Allocate memory for login request/response buffers */
	if (qfle3i_alloc_dma(hba, &conn->gen_pdu.login_req,
			    ISCSI_CONN_LOGIN_BUF_SIZE, QFLE3I_TBL_TYPE_BD, 0))
		goto error;

	conn->gen_pdu.req_buf_size = 0;
	conn->gen_pdu.req_wr_ptr = conn->gen_pdu.login_req.mem;

	if (qfle3i_alloc_dma(hba, &conn->gen_pdu.login_resp,
			    ISCSI_CONN_LOGIN_BUF_SIZE, QFLE3I_TBL_TYPE_BD, 0))
		goto error;

	conn->gen_pdu.resp_buf_size = ISCSI_CONN_LOGIN_BUF_SIZE;
	conn->gen_pdu.resp_wr_ptr = conn->gen_pdu.login_resp.mem;

	return 0;

error:
	PRINT_ERR(hba, "conn login resource alloc failed!!\n");
	qfle3i_conn_free_login_resources(hba, conn);
	return VMK_NO_MEMORY;

}

/*
 * qfle3i_iscsi_conn_new - initialize newly created connection structure
 *
 * @sess:		iscsi session pointer
 * @conn: 		iscsi connection pointer
 *
 * connection structure is initialized which mainly includes allocation of
 *	login resources and lock/time initialization
 */
int qfle3i_iscsi_conn_new(struct qfle3i_sess *sess, struct qfle3i_conn *conn)
{
	struct qfle3i_hba *hba = sess->hba;

	conn->sess = sess;
	conn->header_digest_en = 0;
	conn->data_digest_en = 0;

	vmk_ListInit(&conn->link);
	vmk_ListInitElement(&conn->scsi_scan_list);
	/* 'ep' ptr will be assigned in bind() call */
	conn->ep = NULL;

	if (vmk_BitVectorTest(hba->cnic_dev_type, QFLE3I_NX2_DEV_57710))
		conn->ring_doorbell = qfle3i_ring_sq_dbell_bnx2x;
	else
		conn->ring_doorbell = qfle3i_ring_sq_dbell_bnx2;

	if (qfle3i_conn_alloc_login_resources(hba, conn)) {
		PRINT_ALERT(hba, "conn_new: login resc alloc failed!!\n");
		return VMK_NO_MEMORY;
	}

	vmk_AtomicWrite64(&conn->stop_state, 0);
	vmk_AtomicWrite64(&conn->worker_running, 0);
	ql_vmk_tasklet_init(&conn->conn_tasklet, &qfle3i_conn_main_worker,
		     			(void *)conn);
	vmk_AtomicWrite64(&conn->worker_enabled, 0);

	/*VK: initialize and add a timer in conn_start function.
	init_timer(&conn->poll_timer);
	conn->poll_timer.expires = vmk_TimerCyclesPerSecond() + vmk_GetTimerCycles();*/	/* 200 msec */
/*	conn->poll_timer.function = qfle3i_conn_poll;
	conn->poll_timer.data = (unsigned long) conn;
*/

	return 0;
}

static void qfle3i_wait_for_tmf_completion(struct qfle3i_sess *sess)
{
	int lpcnt = 200;

	while (lpcnt-- && vmk_AtomicRead64(&sess->tmf_active))
		vmk_WorldSleep(100*VMK_USEC_PER_MSEC);
}

static VMK_ReturnStatus qfle3i_ep_connect(
				vmk_IscsiTransTransport *transport,
				void                    *destAddr,
				vmk_Bool                isNonBlocking,
				vmk_IscsiNetHandle      iscsiNetHandle,
				vmk_uint64              *ep_handle)
{
	VMK_ReturnStatus rc = VMK_OK;
	vmk_uint32 iscsi_cid = QFLE3I_CID_RESERVED;
	struct vmk_SocketIPAddress *desti;
	struct vmk_SocketIPv6Address *desti6;
	struct qfle3i_endpoint *endpoint;
	struct qfle3i_hba *hba = NULL;
	struct cnic_dev *cnic;
	struct cnic_sockaddr saddr;
	vmk_uint32 dstFamily;
	vmk_uint32 dstIPAddr[4];

	/* check if the given destination can be reached through NX2 device */
	hba = (qfle3i_hba *)transport->cookie;
	if (!hba) {
		vmk_LogMessage("qfle3i: %s:%d invalid cookie in transport struct\n",
						__func__, __LINE__);
		rc = VMK_BAD_PARAM;
		goto check_busy;
	}

	cnic = hba->cnic;
	qfle3i_get_link_state(hba);
	rc = qfle3i_check_route(hba, destAddr, iscsiNetHandle);
	if (rc != VMK_OK)
	{
		PRINT_ALERT(hba, "qfle3i_check_route failed.\n");
		rc = VMK_NO_MEMORY;
		goto check_busy;
	}

	endpoint = qfle3i_alloc_ep(hba);
	if (!endpoint) {
		PRINT_ALERT(hba, "qfle3i_alloc_ep failed, return ENOMEM\n");
		*ep_handle = (vmk_uint64)0;
		rc = VMK_NO_MEMORY;
		goto check_busy;
	}

	vmk_SemaLock(&hba->net_dev_lock);
	if (qfle3i_adapter_ready(hba) != VMK_OK) {
		rc = VMK_FAILURE;
		goto net_if_down;
	}
	vmk_AtomicWrite64(&endpoint->fp_kcqe_events, 1);
	endpoint->state = EP_STATE_IDLE;
	endpoint->teardown_mode = QFLE3I_ABORTIVE_SHUTDOWN;
	endpoint->ep_iscsi_cid = (vmk_uint16)ISCSI_RESERVED_TAG;
	/*
		New implementation for cid, no need to allocate it
		in driver now, cnic will provide a cid.
	*/
	rc = hba->cnic->cm_acquire(cnic, CNIC_ULP_ISCSI, &iscsi_cid);
	if (rc != VMK_OK) {
		PRINT_ALERT(hba, "cm_acquire: unable to acquire cid\n");
		rc = VMK_NO_MEMORY;
		goto cm_acquire_failed;
	}

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "%s: iscsi_cid:0x%x num active conns:%d\n",
				vmk_NameToString(&hba->vmnic_name), iscsi_cid,
				hba->ofld_conns_active);

	endpoint->hba_age = hba->age;
	endpoint->ep_iscsi_cid = iscsi_cid & 0xFFFF;

	rc = qfle3i_alloc_qp_resc(hba, endpoint);
	if (rc != 0) {
		PRINT_ALERT(hba, "ep_conn, alloc_qp_resc error\n");
		/* We cant add this call at the end of this function(under qp_resc_fail),
		   qfle3i_tear_down_conn internally calls this function and it will be
		   invoked twice.
		*/
		qfle3i_tear_down_ep(hba, endpoint);
		rc = VMK_NO_MEMORY;
		goto qp_resc_err;
	}

	endpoint->state = EP_STATE_OFLD_START;
	qfle3i_ep_ofld_list_add(hba, endpoint);
	vmk_TimerSchedule(qfle3i_driver_info.timer_queue,
			(void *)qfle3i_ep_ofld_timer,
			(void *)(unsigned long)endpoint,
			5 * VMK_USEC_PER_SEC,
			VMK_TIMER_DEFAULT_TOLERANCE,
			VMK_TIMER_ATTR_NONE,
			VMK_LOCKDOMAIN_INVALID,
			VMK_SPINLOCK_UNRANKED,
			&endpoint->ofld_timer);

	/* sending conn offload request */
	if (qfle3i_send_conn_ofld_req(hba, endpoint)) {
		if (endpoint->state == EP_STATE_OFLD_FAILED_CID_BUSY) {
			PRINT_ERR(hba, "%s: ep_iscsi_cid:0x%x is busy\n",
			       vmk_NameToString(&hba->vmnic_name),
			       endpoint->ep_iscsi_cid);
			rc = VMK_BUSY;
		} else {
			rc = VMK_NO_MEMORY;
		}
		qfle3i_tear_down_ep(hba, endpoint);
		PRINT_ERR(hba, "unable to send conn offld kwqe\n");
		goto offld_failed;
	}

	/* Wait for CNIC hardware to setup conn context and return 'cid' */
	ql_vmk_wait_for_completion(&endpoint->ofld_wait,
					(endpoint->state != EP_STATE_OFLD_START),
					100 * VMK_MSEC_PER_SEC);

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "After wait, endpoint->state:0x%x\n",
			  endpoint->state);
#if 0
	if (signal_pending(current))
		flush_signals(current);
#endif
	vmk_TimerCancel(endpoint->ofld_timer, VMK_TRUE);
	qfle3i_ep_ofld_list_del(hba, endpoint);

	if (endpoint->state != EP_STATE_OFLD_COMPL) {
		if (endpoint->state == EP_STATE_OFLD_FAILED_CID_BUSY) {
			PRINT_ERR(hba, " %s: ep_iscsi_cid:0x%x is busy\n",
						vmk_NameToString(&hba->vmnic_name),
						endpoint->ep_iscsi_cid);
			rc = VMK_BUSY;
		} else {
			rc = VMK_NO_MEMORY;
		}
		qfle3i_tear_down_ep(hba, endpoint);
		goto offld_failed;
	}

	if (!vmk_BitVectorTest(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED)) {
		PRINT_ERR(hba, "hba->reg_with_cnic is not QFLE3I_CNIC_REGISTERED.\n");
		rc = VMK_BAD_PARAM;
		qfle3i_tear_down_ep(hba, endpoint);
		goto conn_failed;
	} else
		rc = hba->cnic->cm_create(cnic, CNIC_ULP_ISCSI, endpoint->ep_cid,
				     iscsi_cid, &endpoint->cm_sk, endpoint);
	if (rc) {
		PRINT_INFO(hba, "cm_create failed, rc:0x%x\n", rc);
		rc = VMK_BAD_PARAM;
		goto release_ep;
	}


	if (!vmk_BitVectorTest(hba->cnic_dev_type, QFLE3I_NX2_DEV_57710) && time_stamps)
		endpoint->cm_sk->tcp_flags |= SK_TCP_TIMESTAMP;

	endpoint->cm_sk->rcv_buf = tcp_buf_size;
	endpoint->cm_sk->snd_buf = tcp_buf_size;

	rc = vmk_IscsiTransportGetDstIP(iscsiNetHandle,
		&dstFamily, (vmk_uint8 *)dstIPAddr);
	if (rc != VMK_OK) {
		PRINT_ERR(hba, "vmk_IscsiTransportGetDstIP failed with status = 0x%x\n", rc);
		goto release_ep;
	} else
		PRINT_INFO(hba, "vmk_IscsiTransportGetDstIP: dstFamily = 0x%x, dstIPAddr = %x:%x:%x:%x\n",
			dstFamily,
			 (int)dstIPAddr[0], (int)dstIPAddr[1],
			 (int)dstIPAddr[2], (int)dstIPAddr[3]);

	vmk_Memset(&saddr, 0, sizeof(saddr));
	if (dstFamily == VMK_SOCKET_AF_INET) {
		desti = (vmk_SocketIPAddress *)destAddr;
		saddr.remote.v4 = *desti;
		saddr.local.v4.sin_port = htons(endpoint->tcp_port);
		saddr.local.v4.sin_family = dstFamily;
		saddr.remote.v4.sin_family = dstFamily;
	} else if (dstFamily == AF_INET6) {
		desti6 = (vmk_SocketIPv6Address *)destAddr;
		saddr.remote.v6 = *desti6;
		saddr.local.v6.sin6_port = htons(endpoint->tcp_port);
		saddr.local.v6.sin6_family = dstFamily;
		saddr.remote.v6.sin6_family = dstFamily;
	} else {
		PRINT_ERR(hba, "wrong sa_family\n");
		rc = VMK_BAD_PARAM;
		goto release_ep;
	}

	if (!vmk_BitVectorTest(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED)) {
		rc = VMK_NOT_FOUND;
		PRINT_INFO(hba, "Not registered with qcnic, rc:0x%x \n", rc);
		goto release_ep;
	} else {
		endpoint->state = EP_STATE_CONNECT_START;
		endpoint->timestamp = vmk_GetTimerCycles();
		vmk_TimerSchedule(qfle3i_driver_info.timer_queue,
			(void *)qfle3i_ep_ofld_timer,
			(void *)(unsigned long)endpoint,
			2 * VMK_USEC_PER_SEC,
			VMK_TIMER_DEFAULT_TOLERANCE,
			VMK_TIMER_ATTR_NONE,
			VMK_LOCKDOMAIN_INVALID,
			VMK_SPINLOCK_UNRANKED,
			&endpoint->ofld_timer);
		rc = hba->cnic->cm_connect(endpoint->cm_sk, &saddr);
	}

	if (rc) {
		PRINT_INFO(hba, "cm_connect failed, rc:0x%x \n", rc);
		rc = VMK_BAD_PARAM;
		goto release_ep;
	}

	/* Wait for CNIC hardware to setup conn context and return 'cid' */
	ql_vmk_wait_for_completion(&endpoint->ofld_wait,
		(endpoint->state != EP_STATE_CONNECT_START), 100 * VMK_MSEC_PER_SEC);

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "After wait, endpoint->state:0x%x\n",
		endpoint->state);
	vmk_TimerCancel(endpoint->ofld_timer, VMK_TRUE);
	if (endpoint->state != EP_STATE_CONNECT_COMPL) {
		PRINT_ERR(hba, "%s: connect failed, iscsi cid:0x%x\n",
			vmk_NameToString(&hba->vmnic_name), endpoint->ep_iscsi_cid);
		rc = VMK_FAILURE;
		goto release_ep;
	}

	/* always returns VMK_OK */
	qfle3i_map_ep_dbell_regs(endpoint);

	PRINT(hba, "ep:%p, created successfully, ep_handle:%p,"
		"ep_iscsi_cid:0x%x ep_cid:0x%x \n",
		endpoint, ep_handle, endpoint->ep_iscsi_cid,
		endpoint->ep_cid);
	*ep_handle = (vmk_uint64)(unsigned long)endpoint;
	vmk_SemaUnlock(&hba->net_dev_lock);
	/*
	 * unregister idle devices, without this user can't uninstall
	 * unused bnx2/bnx2x driver because registration will increment
	 * the usage count
	 */
#if defined(_QFLE3I_SEL_DEV_UNREG__)
	if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_GOING_DOWN))
#endif
	qfle3i_check_nx2_dev_busy();
	return 0;

release_ep:
	qfle3i_tear_down_conn(hba, endpoint);
conn_failed:
offld_failed:
	qfle3i_free_qp_resc(hba, endpoint);
qp_resc_err:
cm_acquire_failed:
net_if_down:
	qfle3i_free_ep(endpoint);
	vmk_SemaUnlock(&hba->net_dev_lock);
check_busy:
	*ep_handle = (vmk_uint64)0;
	qfle3i_check_nx2_dev_busy();
	return rc;
}

static VMK_ReturnStatus qfle3i_ep_disconnect(
				vmk_IscsiTransTransport *transport,
				vmk_uint64 ep_handle)
{
	struct qfle3i_endpoint *ep;
	struct cnic_dev *cnic;
	struct qfle3i_hba *hba;
	struct qfle3i_sess *sess;
	int rc;

	hba = (qfle3i_hba *)transport->cookie;
	if (!hba) {
		vmk_LogMessage("qfle3i: %s invalid cookie in transport struct\n",
						__func__);
		return VMK_BAD_PARAM;
	}

	ep = (struct qfle3i_endpoint *) (vmk_uint64)ep_handle;
	if (!ep || (ep_handle == -1)) {
		PRINT_ALERT(hba, "wrong endpoint passed by transport.\n");
		return VMK_BAD_PARAM;
	}

	if (ep->in_progress == 0)
		return VMK_OK;

        while ((ep->state == EP_STATE_CONNECT_START) &&
				(vmk_GetTimerCycles() <
				(ep->timestamp + (12 * vmk_TimerCyclesPerSecond()))))
                	vmk_WorldSleep(250 * VMK_USEC_PER_MSEC);

	/** This could be a bug, we always disable here, but we enable in BIND
	 *  so we should ONLY ever disable if the ep was assigned a conn ( AKA BIND was run )
	 *  Another case is that conn_destroy got called, which sets the ep->conn to NULL 
	 *  which is ok since destroy stops the tasklet
	 */
	if (ep->conn) {
		vmk_AtomicWrite64(&ep->conn->worker_enabled, 0);
		//ql_vmk_tasklet_disable(&ep->conn->conn_tasklet);
	}

	hba = ep->hba;
	if (ep->state == EP_STATE_IDLE)
		goto return_ep;
	cnic = hba->cnic;

	vmk_uint8 *astate = vmk_BitVectorGetRef(hba->adapter_state, 0, 1);
	PRINT(hba, "%s: disconnecting ep %p {0x%x, 0x%x}, conn %p, sess %p, "
	      "hba-state:0x%x, num active conns:%d\n",
	      vmk_NameToString(&hba->vmnic_name), ep, ep->ep_iscsi_cid, ep->ep_cid,
	      ep->conn, ep->sess, *astate,
	      hba->ofld_conns_active);

	vmk_SemaLock(&hba->net_dev_lock);
	if (!vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_UP))
		goto free_resc;
	if (ep->hba_age != hba->age)
		goto dev_reset;

	if (!qfle3i_ep_tcp_conn_active(ep))
		goto destroy_conn;

	if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_FW_RECOVERY))
		goto destroy_conn;

	vmk_TimerSchedule(qfle3i_driver_info.timer_queue,
			(void *)qfle3i_ep_ofld_timer,
			(void *) (unsigned long) ep,
			hba->conn_teardown_tmo * VMK_USEC_PER_SEC,
			VMK_TIMER_DEFAULT_TOLERANCE,
			VMK_TIMER_ATTR_NONE,
			VMK_LOCKDOMAIN_INVALID,
			VMK_SPINLOCK_UNRANKED,
			&ep->ofld_timer);

	if (!vmk_BitVectorTest(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED)) {
		vmk_TimerCancel(ep->ofld_timer, VMK_TRUE);
		goto free_resc;
	}

	ep->state = EP_STATE_DISCONN_START;
	if (ep->teardown_mode == QFLE3I_GRACEFUL_SHUTDOWN)
		rc = cnic->cm_close(ep->cm_sk);
	else
		rc = cnic->cm_abort(ep->cm_sk);

	if (rc == VMK_EALREADY)
        ep->state = EP_STATE_DISCONN_COMPL;

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "Before wait, ep->state:0x%x\n",
			  ep->state);

	/* wait for option-2 conn teardown */
	ql_vmk_wait_for_completion(&ep->ofld_wait,
				((ep->state != EP_STATE_DISCONN_START) &&
				(ep->state != EP_STATE_TCP_FIN_RCVD)),
				100 * VMK_MSEC_PER_SEC);

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "After wait, ep->state:0x%x\n",
			  ep->state);
#if 0
	if (signal_pending(current))
		flush_signals(current);
#endif
	vmk_TimerCancel(ep->ofld_timer, VMK_TRUE);

destroy_conn:
	if (qfle3i_tear_down_conn(hba, ep)) {
		vmk_SemaUnlock(&hba->net_dev_lock);
		return VMK_FAILURE;
	}

dev_reset:
free_resc:
	/* in case of 3-way handshake failure, there won't be any binding
	 * between EP and SESS
	 */
	if (ep->sess) {
		int cmds_a = 0, cmds_p = 0;
		cmds_p = qfle3i_flush_pend_queue(ep->sess, NULL, VMK_SCSI_HOST_RESET);
		cmds_a = qfle3i_flush_cmd_queue(ep->sess, NULL, VMK_SCSI_HOST_RESET, 0);
		QFLE3I_DBG(DBG_ITT_CLEANUP, hba, "CMDS FLUSHED, pend=%d, active=%d\n",
			  cmds_p, cmds_a);
	}

	vmk_SemaUnlock(&hba->net_dev_lock);
	qfle3i_free_qp_resc(hba, ep);
return_ep:
	/* check if session recovery in progress */
	sess = ep->sess;

	qfle3i_free_ep(ep);
	if (sess) {
		sess->state = QFLE3I_SESS_INITIAL;
		ql_vmk_world_wakeup(&sess->er_wait);
	}
	ql_vmk_world_wakeup(&hba->eh_wait);

	if (!hba->ofld_conns_active)
		qfle3i_check_nx2_dev_busy();

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "qfle3i: ep %p, destroyed\n", ep);
	return VMK_OK;
}


static VMK_ReturnStatus qfle3i_ep_poll(
			vmk_IscsiTransTransport  *transport,
			vmk_uint64               ep_handle,
			vmk_int32                timeout_ms,
			vmk_IscsiTransPollStatus *eventReply)
{

	struct qfle3i_hba *hba;
	struct qfle3i_endpoint *ep;
	int rc = 0;

	hba = (qfle3i_hba *)transport->cookie;

	ep = (struct qfle3i_endpoint *) (unsigned long) ep_handle;
	if (!ep || (ep_handle == -1)) {
		PRINT_ALERT(hba, "wrong endpoint passed by transport. \n");
		*eventReply = VMK_ISCSI_TRANSPORT_POLL_STATUS_NO_DATA;
		return VMK_FAILURE;
	}

	if ((ep->state == EP_STATE_IDLE) ||
	    (ep->state == EP_STATE_CONNECT_FAILED) ||
	    (ep->state == EP_STATE_OFLD_FAILED)) {
		QFLE3I_DBG(DBG_CONN_SETUP, hba, "FAILURE, ep->state:0x%x", ep->state);
		*eventReply = VMK_ISCSI_TRANSPORT_POLL_STATUS_NO_DATA;
		return VMK_FAILURE;
	}

	if (ep->state == EP_STATE_CONNECT_COMPL) {
		if (vmk_BitVectorTest(ep->hba->adapter_state, ADAPTER_STATE_GOING_DOWN)) {
			/* Eventhough option2 connect is successful pretend
			 * as if TCP 3-way handshake failed, This will immediately
			 * trigger the connection context cleanup so that the adapter
			 * can be shutdown quickly.
			 */
			QFLE3I_DBG(DBG_CONN_SETUP, hba, "FAILURE, ep->state:0x%x\n", ep->state);
			ep->hba->stop_event_ifc_abort_poll++;
			*eventReply = VMK_ISCSI_TRANSPORT_POLL_STATUS_NO_DATA;
			return VMK_FAILURE;
		} else {
			QFLE3I_DBG(DBG_CONN_SETUP, hba, "SUCCESS, ep->state:0x%x\n", ep->state);
			*eventReply = VMK_ISCSI_TRANSPORT_POLL_STATUS_DATA_AVAILABLE;
			return VMK_OK;
		}
	}

	QFLE3I_DBG(DBG_INTERNAL, ep->hba, "Before wait, ep->state:0x%x\n",
			  ep->state);
	rc = ql_vmk_wait_for_completion(&ep->ofld_wait,
				((ep->state == EP_STATE_OFLD_FAILED) ||
				 (ep->state == EP_STATE_CONNECT_FAILED) ||
				 (ep->state == EP_STATE_CONNECT_COMPL)),
				 timeout_ms);
	QFLE3I_DBG(DBG_INTERNAL, ep->hba, "After wait, ep->state:0x%x\n",
			  ep->state);

	if (ep->state == EP_STATE_OFLD_FAILED) {
		*eventReply = VMK_ISCSI_TRANSPORT_POLL_STATUS_NO_DATA;
		rc = VMK_FAILURE;
	}

	if (rc == VMK_OK) {
		if (vmk_BitVectorTest(ep->hba->adapter_state, ADAPTER_STATE_GOING_DOWN)) {
			/* Eventhough option2 connect is successful pretend
			 * as if TCP 3-way handshake failed, This will immediately
			 * trigger the connection context cleanup so that the adapter
			 * can be shutdown quickly.
			 */
			QFLE3I_DBG(DBG_CONN_SETUP, hba, "FAILURE, ep->state:0x%x\n", ep->state);
			ep->hba->stop_event_ifc_abort_poll++;
			*eventReply = VMK_ISCSI_TRANSPORT_POLL_STATUS_NO_DATA;
			return VMK_FAILURE;
		} else {
			QFLE3I_DBG(DBG_CONN_SETUP, hba, "SUCCESS, ep->state:0x%x\n", ep->state);
			*eventReply = VMK_ISCSI_TRANSPORT_POLL_STATUS_DATA_AVAILABLE;
			return VMK_OK;
		}
	}
		*eventReply = VMK_ISCSI_TRANSPORT_POLL_STATUS_NO_DATA;
		return VMK_FAILURE;
}

static VMK_ReturnStatus qfle3i_create_sess(
	vmk_IscsiTransTransport     *transport,
	vmk_IscsiSessionCreateParam *sess_param,
	vmk_uint32                  *hostNum,
	vmk_IscsiTransSession       **out_sess)
{
	VMK_ReturnStatus status = VMK_OK;
	struct qfle3i_hba *hba;
	struct qfle3i_sess *sess;
	vmk_ScsiAdapter *scsiAdapter;
	vmk_IscsiTransSession *trans_sess;

	vmk_uint32 max_cmds = sess_param->maxCmds;
	vmk_uint32 q_depth = sess_param->qDepth;
	vmk_uint32 initial_cmdsn = sess_param->initialCmdSeqNum;
	vmk_uint32 target_id = sess_param->targetId;
	vmk_uint32 channel_id = sess_param->channelId;

	hba = (qfle3i_hba *)transport->cookie;
	if(!hba) {
		vmk_WarningMessage("Failed to find hba from transport cookie.\n");
		return VMK_FAILURE;
	}
	QFLE3I_DBG(DBG_CONN_SETUP, hba, "tgt id %d, ch id %d, cmds_max %d\n",
			  target_id, channel_id, max_cmds);

	if (qfle3i_adapter_ready(hba) != VMK_OK) {
		PRINT_ERR(hba, "adapter not ready \n");
		return VMK_FAILURE;
	}

	scsiAdapter = hba->scsiAdapter;
	if (!scsiAdapter){
		PRINT_ERR(hba, "no vmk_scsiAdapter associated with this hba\n");
		return VMK_FAILURE;
	}

	status = vmk_IscsiTransportAllocSession((void *)hba,
					transport, &trans_sess);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "vmk_IscsiTransportAllocSession failed.\n");
		goto tsess_alloc_failed;
	}

	status = vmk_IscsiTransportAddSession(trans_sess,
				hba->host_num, target_id, channel_id);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "vmk_IscsiTransportAddSession failed.\n");
		goto tsess_add_failed;
	}

	sess = vmk_HeapAlign(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
				sizeof(struct qfle3i_sess), VMK_L1_CACHELINE_SIZE);
	if (!sess) {
		PRINT_ERR(hba, "Failed to alloc qfle3i_sess"
					"channelId:%d, targetId:%d \n", channel_id, target_id);
		goto isess_alloc_failed;
	}
	vmk_Memset(sess, 0, sizeof(struct qfle3i_sess));

	*hostNum = hba->host_num;
	vmk_Memset(sess, 0, sizeof(struct qfle3i_sess));


	sess->hba = hba;
	sess->trans = transport;
	sess->trans_sess = trans_sess;
	sess->target_id = target_id;
	sess->channel_id = channel_id;
	sess->max_cmds = max_cmds;
	sess->q_depth = q_depth;
	sess->cmdsn = initial_cmdsn;
	sess->exp_cmdsn = initial_cmdsn + 1;
	sess->max_cmdsn = initial_cmdsn + 1;

	/*
	 * For Open-iSCSI, only normal sessions go through qfle3i.
	 * Discovery session goes through host stack TCP/IP stack.
	 */
	status = qfle3i_iscsi_sess_new(hba, sess);
	if (status != VMK_OK) {
		/* failed to allocate memory */
		PRINT_ALERT(hba, "qfle3i_sess_create: unable to alloc sess\n");
		goto isess_init_failed;
	}

	/* save qfle3i session object in transport class session structure */
	trans_sess->cookie = (vmk_AddrCookie *)sess;
	*out_sess = trans_sess;

	PRINT(hba, "sess %p created successfully\n", sess);
	return status;

isess_init_failed:
	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), sess);
isess_alloc_failed:
	vmk_IscsiTransportRemoveSession(trans_sess);
tsess_add_failed:
	vmk_IscsiTransportFreeSession(trans_sess);
tsess_alloc_failed:
	return VMK_FAILURE;
}

static VMK_ReturnStatus qfle3i_destroy_sess(
		vmk_IscsiTransSession *trans_sess)
{
	VMK_ReturnStatus status = VMK_OK;
	struct qfle3i_hba *hba;
	struct qfle3i_sess *sess = (qfle3i_sess *)trans_sess->cookie;
	hba = sess->hba;

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "destroying sess %p\n",
		  sess);

	qfle3i_withdraw_sess_recovery(sess);
	vmk_IscsiTransportRemoveSession(trans_sess);

	vmk_ScsiNotifyPathStateChangeAsync(hba->scsiAdapter,
						sess->channel_id, sess->target_id, -1);
	PRINT_NOTICE(hba, "Notified Path State Change: "
				"Target port ACTIVE: Channel:0x%x TGT ID:0x%x\n",
				sess->channel_id, sess->target_id);

	vmk_SpinlockLock(sess->iscsilun_lock);
	if (vmk_ListIsEmpty(&sess->iscsilun_list)) {
		vmk_SpinlockUnlock(sess->iscsilun_lock);
		qfle3i_iscsi_sess_release(sess);
	} else {
		sess->state = QFLE3I_SESS_DESTROYED;
		vmk_SpinlockUnlock(sess->iscsilun_lock);
	}

	vmk_SpinlockLock(hba->lock);
	hba->num_active_sess--;
	vmk_SpinlockUnlock(hba->lock);

	return status;
}

static VMK_ReturnStatus qfle3i_create_conn(
		vmk_IscsiTransSession    *trans_sess,
		vmk_uint32               connId,
		vmk_IscsiTransConnection **out_conn)
{
	VMK_ReturnStatus status = VMK_OK;
	struct qfle3i_conn *conn;
	vmk_IscsiTransConnection *trans_conn = NULL;
	struct qfle3i_hba *hba;

	struct qfle3i_sess *sess = (struct qfle3i_sess *)trans_sess->cookie;
	if(!sess){
		vmk_WarningMessage("Can find qfle3i Session from transport session.\n");
		return VMK_FAILURE;
	}

	hba = sess->hba;
	if(!hba){
		vmk_WarningMessage("no valid hba in qfle3i session.\n");
		return VMK_FAILURE;
	}

	/* Create iscsi transport connection */
	status = vmk_IscsiTransportCreateConnection(trans_sess, connId, &trans_conn);
	if (status != VMK_OK) {
		PRINT_ALERT(hba, "Iscsi transport failed to create connection:0x%x\n",
				connId);
		return VMK_FAILURE;
	}

	/* allocate memory for qfle3i driver connection */
	conn = vmk_HeapAlign(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
				sizeof(struct qfle3i_conn), VMK_L1_CACHELINE_SIZE);
	if(!conn) {
		PRINT_ALERT(hba, " Can't alloc memory for iscsi connection:0x%x\n", connId);
   		status = VMK_NO_MEMORY;
		goto conn_alloc_failed;
	}
	vmk_Memset(conn, 0, sizeof(struct qfle3i_conn));

	trans_conn->cookie = (vmk_AddrCookie *)conn;
	conn->trans_conn = trans_conn;
	*out_conn = trans_conn;
	conn->exp_statsn = STATSN_UPDATE_SIGNATURE;
	conn->state = CONN_STATE_IDLE;

	/* Initialize the connection structure */
	if (qfle3i_iscsi_conn_new(sess, conn))
		goto mem_err;
	conn->conn_cid = connId;
	PRINT(hba, "conn %p for cid:0x%x created successfully\n",
		  conn, connId);
   	return status;

mem_err:
	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), conn);
conn_alloc_failed:
	vmk_IscsiTransportDestroyConnection(trans_conn);
	return status;
}

static VMK_ReturnStatus qfle3i_destroy_conn(
   vmk_IscsiTransConnection   *trans_conn)
{
	VMK_ReturnStatus status = VMK_OK;
	struct qfle3i_conn *conn = (qfle3i_conn *)trans_conn->cookie;
	struct qfle3i_sess *sess = conn->sess;
	int count = 0;

/* pending_queue will be flushed before sending logout to FW.
 * lets not flush cmd_queue here, destroy_conn() means transport layer
 * will send ep_disconnect eventually.
 * Lets ep_disconnect terminate the connection, FW will drop all IOs in
 * response and then return all pending IOs to mid-layer
 */

/*
	if (sess) {
		int cmds_a = 0, cmds_p = 0;
		cmds_p = qfle3i_flush_pend_queue(sess, NULL, VMK_SCSI_HOST_RESET);
		cmds_a = qfle3i_flush_cmd_queue(sess, NULL, VMK_SCSI_HOST_RESET, 1);
		PRINT(sess->hba,
			  "CMDS FLUSHED, pend=%d, active=%d\n",
			  cmds_p, cmds_a);

		count = 10;
		while (count-- && sess->active_cmd_count)
			vmk_WorldSleep(100 * VMK_USEC_PER_MSEC);
	}
*/
	PRINT(sess->hba, "destroying conn %p\n", conn);
	qfle3i_conn_free_login_resources(conn->sess->hba, conn);

	/* disable will not remove already scheduled tasklet, it just disables it.
	 */
	ql_vmk_tasklet_disable(&conn->conn_tasklet);
	vmk_AtomicWrite64(&conn->worker_enabled, 0);

	/* Need to unbind 'conn' and 'iscsi_cid' to avoid any invalid
	 * pointer access due to spurious KCQE notifications.
	 */
	if (conn->ep)
		qfle3i_unbind_conn_from_iscsi_cid(conn, conn->ep->ep_iscsi_cid);


	vmk_SpinlockLock(sess->lock);

	if ((&conn->link == NULL) || (&conn->link == conn->link.nextPtr))
		goto skip;
	vmk_ListRemove(&conn->link);
	vmk_ListInit(&conn->link);
skip:
	if (sess->lead_conn == conn)
		sess->lead_conn = NULL;

	if (conn->ep) {
		conn->ep->conn = NULL;
		conn->ep = NULL;
	}
	vmk_SpinlockUnlock(sess->lock);

	if (conn->persist_address) {
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), conn->persist_address);
		conn->persist_address = NULL;
	}

	status = vmk_IscsiTransportDestroyConnection(trans_conn);
	if (status != VMK_OK) {
		PRINT_ERR(sess->hba, "failed destroy connection: %s",
						vmk_StatusToString(status));
	}

	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), conn);
	return status;
}

static VMK_ReturnStatus qfle3i_bind_conn(
	vmk_IscsiTransSession    *trans_sess,
	vmk_IscsiTransConnection *trans_conn,
	vmk_uint64               transportEPH,
	vmk_Bool                 isLeading)
{
	VMK_ReturnStatus status = VMK_OK;
	struct qfle3i_sess *sess;
	struct qfle3i_conn *tmp;
	struct qfle3i_conn *conn;
	struct qfle3i_endpoint *ep;
	struct qfle3i_hba *hba;

	sess = (struct qfle3i_sess *)trans_sess->cookie;
	conn = (struct qfle3i_conn *)trans_conn->cookie;
	hba = sess->hba;
	ep = (struct qfle3i_endpoint *) (vmk_uint64)transportEPH;

	PRINT(hba, "binding sess %p, conn %p, ep %p ep_iscsi_cid:0x%x\n",
		  sess, conn, ep, ep->ep_iscsi_cid);

	/* If adapter is going down (mtu change, vlan, selftest, etc'),
	 * fail this bind request so that connection context be cleaned
	 */
	if (vmk_BitVectorTest(ep->hba->adapter_state, ADAPTER_STATE_GOING_DOWN)) {
		ep->hba->stop_event_ifc_abort_bind++;
		return VMK_ACCESS_DENIED;
	}

	if (ep->ep_iscsi_cid >= hba->max_active_conns) {
		PRINT_ERR(hba, "wrong cid: 0x%x\n", ep->ep_iscsi_cid);
		return VMK_FAILURE;
	}

	if ((ep->state == EP_STATE_TCP_FIN_RCVD) ||
	    (ep->state == EP_STATE_TCP_RST_RCVD))
		/* Peer disconnect via' FIN or RST */
		return VMK_BAD_PARAM;

	if (ep->hba != sess->hba) {
		/* Error - TCP connection does not belong to this device
		 */
		PRINT_ALERT(sess->hba, "conn bind, ep=0x%p (%s) does not",
				  ep, vmk_NameToString(&hba->vmnic_name));
		PRINT_ALERT(sess->hba, "belong to hba (%s)\n",
				  vmk_NameToString(&hba->vmnic_name));
		return VMK_NOT_FOUND;
	}

	if (!sess->login_nopout_cmd)
		sess->login_nopout_cmd = qfle3i_alloc_cmd(sess);
	if (!sess->scsi_tmf_cmd)
		sess->scsi_tmf_cmd = qfle3i_alloc_cmd(sess);
	if (!sess->nopout_resp_cmd)
		sess->nopout_resp_cmd = qfle3i_alloc_cmd(sess);

	/* look-up for existing connection, MC/S is not currently supported */
	vmk_SpinlockLock(sess->lock);
	tmp = NULL;
	if (!vmk_ListIsEmpty(&sess->conn_list)) {
		ql_vmk_list_each_entry(tmp, &sess->conn_list, link, struct qfle3i_conn) {
			if (tmp == conn)
				break;
		}
	}
	if ((tmp != conn) && (conn->sess == sess)) {
		/* bind iSCSI connection to this session */
		vmk_ListInsert(&conn->link, vmk_ListAtFront(&sess->conn_list));
		if (isLeading)
			sess->lead_conn = conn;
	}

	if (conn->ep) {
		/* This happens when 'iscsid' is killed and restarted. Daemon
		 * has no clue of tranport handle, but knows active conn/sess
		 * and tried to rebind a new tranport (EP) to already active
		 * iSCSI session/connection
		 */

		PRINT_ALERT(sess->hba, "conn->ep already exists.\n");

		vmk_SpinlockUnlock(sess->lock);
		qfle3i_ep_disconnect(conn->sess->trans, (vmk_uint64)conn->ep);
		vmk_SpinlockLock(sess->lock);
	}

	conn->ep = (struct qfle3i_endpoint *) (unsigned long) transportEPH;
	conn->ep->conn = conn;
	conn->ep->sess = sess;
	conn->state = CONN_STATE_XPORT_READY;
	conn->iscsi_conn_cid = conn->ep->ep_iscsi_cid;
	conn->fw_cid = conn->ep->ep_cid;

	status = qfle3i_bind_conn_to_iscsi_cid(conn, ep->ep_iscsi_cid);
	vmk_SpinlockUnlock(sess->lock);

	sess->total_cmds_sent = 0;
	sess->total_cmds_queued = 0;
	sess->total_cmds_completed = 0;
	sess->total_cmds_failed = 0;
	sess->total_cmds_completed_by_chip = 0;
	sess->cmd_win_closed = 0;
	sess->host_busy_cmd_win = 0;

	/* 5706/5708/5709 FW takes RQ as full when initiated, but for 57710
	 * driver needs to explicitly replenish RQ index during setup.
  	 */
	if (vmk_BitVectorTest(ep->hba->cnic_dev_type, QFLE3I_NX2_DEV_57710))
		qfle3i_put_rq_buf(conn, 0);

	vmk_AtomicWrite64(&conn->worker_enabled, 1);
	qfle3i_arm_cq_event_coalescing(conn->ep, CNIC_ARM_CQE);
	return status;
}

static VMK_ReturnStatus qfle3i_setparam(
	vmk_IscsiTransConnection  *trans_conn,
	vmk_IscsiTransIscsiParam  param,
	vmk_uint8                 *buf,
	vmk_ByteCountSmall        buf_len)
{
	VMK_ReturnStatus ret_val = VMK_OK;
	struct qfle3i_conn *conn = (struct qfle3i_conn *)trans_conn->cookie;
	struct qfle3i_sess *sess = conn->sess;

	vmk_SpinlockLock(sess->lock);
	if (conn->state != CONN_STATE_IN_LOGIN) {
		PRINT_ERR(sess->hba, "can't change param [%d]\n", param);
		vmk_SpinlockUnlock(sess->lock);
		return VMK_FAILURE;
	}
	vmk_SpinlockUnlock(sess->lock);
	switch (param) {
	case VMK_ISCSI_CONN_PARAM_MAX_RECV_DLENGTH:
		vmk_Sscanf(buf, "%d", &conn->max_data_seg_len_recv);
		break;
	case VMK_ISCSI_CONN_PARAM_MAX_XMIT_DLENGTH:
		vmk_Sscanf(buf, "%d", &conn->max_data_seg_len_xmit);
		break;
	case VMK_ISCSI_CONN_PARAM_HDRDGST_EN:
		vmk_Sscanf(buf, "%d", &conn->header_digest_en);
		break;
	case VMK_ISCSI_CONN_PARAM_DATADGST_EN:
		vmk_Sscanf(buf, "%d", &conn->data_digest_en);
		break;
	case VMK_ISCSI_CONN_PARAM_EXP_STATSN:
		vmk_Sscanf(buf, "%u", &conn->exp_statsn);
		break;
	case VMK_ISCSI_CONN_PARAM_PERSISTENT_PORT:
		vmk_Sscanf(buf, "%d", &conn->persist_port);
		break;
	case VMK_ISCSI_CONN_PARAM_PERSISTENT_ADDRESS:
		if (conn->persist_address)
			break;
		ret_val = vmk_StringDup(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
							buf, TARGET_MAX_LEN, &conn->persist_address);
		if (ret_val != VMK_OK)
			PRINT_ERR(sess->hba, "Set TARGET NAME failed.\n");
		break;
    case VMK_ISCSI_CONN_PARAM_IFMARKER_EN:
		vmk_Sscanf(buf, "%d", &conn->ifmarker_enable);
		ret_val = VMK_BAD_PARAM;
		break;
	case VMK_ISCSI_CONN_PARAM_OFMARKER_EN:
		vmk_Sscanf(buf, "%d", &conn->ofmarker_enable);
		ret_val = VMK_BAD_PARAM;
		break;
	case VMK_ISCSI_SESS_PARAM_INITIAL_R2T_EN:
		if (conn == sess->lead_conn)
			vmk_Sscanf(buf, "%d", &sess->initial_r2t);
		break;
	case VMK_ISCSI_SESS_PARAM_MAX_R2T:
		if (conn == sess->lead_conn)
			vmk_Sscanf(buf, "%d", &sess->max_r2t);
		break;
	case VMK_ISCSI_SESS_PARAM_IMM_DATA_EN:
		if (conn == sess->lead_conn)
			vmk_Sscanf(buf, "%d", &sess->imm_data);
		break;
	case VMK_ISCSI_SESS_PARAM_FIRST_BURST:
		if (conn == sess->lead_conn)
			vmk_Sscanf(buf, "%d", &sess->first_burst_len);
		break;
	case VMK_ISCSI_SESS_PARAM_MAX_BURST:
		if (conn == sess->lead_conn)
			vmk_Sscanf(buf, "%d", &sess->max_burst_len);
		break;
	case VMK_ISCSI_SESS_PARAM_PDU_INORDER_EN:
		if (conn == sess->lead_conn)
			vmk_Sscanf(buf, "%d", &sess->pdu_inorder);
		break;
	case VMK_ISCSI_SESS_PARAM_DATASEQ_INORDER_EN:
		if (conn == sess->lead_conn)
			vmk_Sscanf(buf, "%d", &sess->dataseq_inorder);
		break;
	case VMK_ISCSI_SESS_PARAM_ERL:
		if (conn == sess->lead_conn)
			vmk_Sscanf(buf, "%d", &sess->erl);
		break;
	case VMK_ISCSI_SESS_PARAM_TARGET_NAME:
		if (sess->target_name)
			break;
		ret_val = vmk_StringDup(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
							    buf, TARGET_MAX_LEN, &sess->target_name);
		if (ret_val != VMK_OK)
			PRINT_ERR(sess->hba, "Set TARGET NAME failed.\n");
		break;
	case VMK_ISCSI_SESS_PARAM_TPGT:
		vmk_Sscanf(buf, "%d", &sess->tgt_prtl_grp);
		break;
	case VMK_ISCSI_SESS_PARAM_ISID:
		vmk_Snprintf(sess->isid, sizeof(sess->isid), "%s", buf);
		break;
	default:
		PRINT_ALERT(sess->hba, "PARAM_UNKNOWN: 0x%x\n", param);
		ret_val = VMK_BAD_PARAM;
		break;
	}

	return ret_val;
}

static VMK_ReturnStatus qfle3i_get_connparam(
			vmk_IscsiTransConnection *trans_conn,
			vmk_IscsiTransIscsiParam param,
			vmk_uint8                *buf,
			vmk_ByteCountSmall       *data_len)
{
	VMK_ReturnStatus ret_val = VMK_OK;
	struct qfle3i_conn *conn = (struct qfle3i_conn *)trans_conn->cookie;
	int len = 0;

	switch (param) {
	case VMK_ISCSI_CONN_PARAM_MAX_RECV_DLENGTH:
		len = vmk_Sprintf(buf, "%u\n", conn->max_data_seg_len_recv);
		break;
	case VMK_ISCSI_CONN_PARAM_MAX_XMIT_DLENGTH:
		len = vmk_Sprintf(buf, "%u\n", conn->max_data_seg_len_xmit);
		break;
	case VMK_ISCSI_CONN_PARAM_HDRDGST_EN:
		len = vmk_Sprintf(buf, "%d\n", conn->header_digest_en);
		break;
	case VMK_ISCSI_CONN_PARAM_DATADGST_EN:
		len = vmk_Sprintf(buf, "%d\n", conn->data_digest_en);
		break;
	case VMK_ISCSI_CONN_PARAM_EXP_STATSN:
		len = vmk_Sprintf(buf, "%u\n", conn->exp_statsn);
		break;
	case VMK_ISCSI_CONN_PARAM_PERSISTENT_PORT:
		len = vmk_Sprintf(buf, "%d\n", conn->persist_port);
		break;
	case VMK_ISCSI_CONN_PARAM_PERSISTENT_ADDRESS:
		if (conn->persist_address)
			len = vmk_Sprintf(buf, "%s\n", conn->persist_address);
		break;
	case VMK_ISCSI_CONN_PARAM_IFMARKER_EN:
		len = vmk_Sprintf(buf, "%u\n", conn->ifmarker_enable);
		break;
	case VMK_ISCSI_CONN_PARAM_OFMARKER_EN:
		len = vmk_Sprintf(buf, "%u\n", conn->ofmarker_enable);
		break;
	case VMK_ISCSI_CONN_PARAM_CONN_PORT:
		if (conn->ep)
			len = vmk_Sprintf(buf, "%u\n",
					(vmk_uint32)(vmk_BE16ToCPU((vmk_uint32)conn->ep->cm_sk->dst_port)));
		else
			len = vmk_Sprintf(buf, "0\n");
		break;
	case VMK_ISCSI_CONN_PARAM_CONN_ADDRESS:
		if (conn->ep)
			len = vmk_Sprintf(buf, NIPQUAD_FMT "\n",
				      NIPQUAD(conn->ep->cm_sk->dst_ip));
		else
			len = vmk_Sprintf(buf, "0.0.0.0\n");
		/* check if we have to bother about input length */
		break;
	default:
		PRINT_ALERT(conn->sess->hba,
			    "get_param: conn 0x%p param %d not found\n",
			    conn, (vmk_uint32)param);
		return VMK_BAD_PARAM;
	}
	if (len > 0)
		buf[len - 1] = '\0';

	*data_len = len;
	return ret_val;
}

static VMK_ReturnStatus qfle3i_get_sessparam(
			vmk_IscsiTransSession    *trans_sess,
			vmk_IscsiTransIscsiParam param,
			vmk_uint8                *buf,
			vmk_ByteCountSmall       *data_len)
{
	struct qfle3i_sess *sess =(struct qfle3i_sess *)trans_sess->cookie;
	int len = 0;

	switch (param) {
	case VMK_ISCSI_SESS_PARAM_INITIAL_R2T_EN:
		len = vmk_Sprintf(buf, "%d\n", sess->initial_r2t);
		break;
	case VMK_ISCSI_SESS_PARAM_MAX_R2T:
		len = vmk_Sprintf(buf, "%hu\n", sess->max_r2t);
		break;
	case VMK_ISCSI_SESS_PARAM_IMM_DATA_EN:
		len = vmk_Sprintf(buf, "%d\n", sess->imm_data);
		break;
	case VMK_ISCSI_SESS_PARAM_FIRST_BURST:
		len = vmk_Sprintf(buf, "%u\n", sess->first_burst_len);
		break;
	case VMK_ISCSI_SESS_PARAM_MAX_BURST:
		len = vmk_Sprintf(buf, "%u\n", sess->max_burst_len);
		break;
	case VMK_ISCSI_SESS_PARAM_PDU_INORDER_EN:
		len = vmk_Sprintf(buf, "%d\n", sess->pdu_inorder);
		break;
	case VMK_ISCSI_SESS_PARAM_DATASEQ_INORDER_EN:
		len = vmk_Sprintf(buf, "%d\n", sess->dataseq_inorder);
		break;
	case VMK_ISCSI_SESS_PARAM_ERL:
		len = vmk_Sprintf(buf, "%d\n", sess->erl);
		break;
	case VMK_ISCSI_SESS_PARAM_TARGET_NAME:
		if (sess->target_name)
			len = vmk_Sprintf(buf, "%s\n", sess->target_name);
		break;
	case VMK_ISCSI_SESS_PARAM_TPGT:
		len = vmk_Sprintf(buf, "%d\n", sess->tgt_prtl_grp);
		break;
	case VMK_ISCSI_SESS_PARAM_ISID:
		len = vmk_Sprintf(buf,"%s\n", sess->isid);
		break;
	default:
		PRINT_ALERT(sess->hba, "sess_get_param: sess 0x%p", sess);
		PRINT_ALERT(sess->hba, "param (0x%x) not found\n", (vmk_uint32) param);
		return VMK_BAD_PARAM;
	}

	if (len > 0)
		buf[len - 1] = '\0';
	*data_len = len;
	return VMK_OK;
}

static VMK_ReturnStatus qfle3i_start_conn
		(vmk_IscsiTransConnection *trans_conn)
{
	struct qfle3i_conn *conn = (struct qfle3i_conn *) trans_conn->cookie;
	struct qfle3i_sess *sess;

	QFLE3I_DBG(DBG_CONN_SETUP, conn->sess->hba, "conn %p iscsi_conn_cid:0x%x\n",
		  conn, conn->iscsi_conn_cid);

	if (conn->state != CONN_STATE_IN_LOGIN) {
		PRINT_ALERT(conn->sess->hba,
			"conn_start: conn 0x%p state 0x%x err!!\n",
			conn, conn->state);
		return VMK_BAD_PARAM;
	}
	sess = conn->sess;

	if ((sess->imm_data || !sess->initial_r2t) &&
		sess->first_burst_len > sess->max_burst_len) {
		PRINT_ALERT(conn->sess->hba, "invalid params, FBL > MBL\n");
			return VMK_BAD_PARAM;
	}


	conn->sess->hba->is_nic_up = 1;
	conn->ep->state = EP_STATE_ULP_UPDATE_START;

	vmk_TimerSchedule(qfle3i_driver_info.timer_queue,
			(void *)qfle3i_ep_ofld_timer,
			(void *) (unsigned long) conn->ep,
			10 * VMK_USEC_PER_SEC,
			VMK_TIMER_DEFAULT_TOLERANCE,
			VMK_TIMER_ATTR_NONE,
			VMK_LOCKDOMAIN_INVALID,
			VMK_SPINLOCK_UNRANKED,
			&conn->ep->ofld_timer);

	if (qfle3i_update_iscsi_conn(conn)) {
		PRINT_ERR(sess->hba, "unable to send conn update kwqe\n");
		return VMK_NO_SPACE;
	}

	QFLE3I_DBG(DBG_CONN_SETUP, conn->sess->hba, "Before wait, ep->state:0x%x\n",
			  conn->ep->state);

	/* update iSCSI context for this conn, wait for CNIC to complete */
	ql_vmk_wait_for_completion(&conn->ep->ofld_wait,
				(conn->ep->state != EP_STATE_ULP_UPDATE_START),
				100 * VMK_MSEC_PER_SEC);

	QFLE3I_DBG(DBG_CONN_SETUP, conn->sess->hba, "After wait, ep->state:0x%x\n",
			  conn->ep->state);

	if(conn->ep->state != EP_STATE_ULP_UPDATE_COMPL) {
		PRINT_ERR(conn->sess->hba, "Update iscsi conn "
				"failed ep_cid:0x%x ep_iscsi_cid:0x%x state = 0x%x\n",
				conn->ep->ep_cid, conn->ep->ep_iscsi_cid,
				conn->ep->state);
		return VMK_FAILURE;
	}

#if 0
	if (signal_pending(current))
		flush_signals(current);
#endif
	vmk_TimerCancel(conn->ep->ofld_timer, VMK_TRUE);

	/* cannot hold 'sess->lock' during iscsi_trans API calls. Doing so will
	 * cause PCPU lockup. That is the reason for taking a lock other than
	 * 'sess->lock'
	 */
	vmk_SpinlockLock(sess->device_lock);
	switch (vmk_AtomicRead64(&conn->stop_state)) {
	case VMK_ISCSI_STOP_CONN_RECOVER:
		QFLE3I_DBG(DBG_SESS_RECO, sess->hba, "sess %p recovery CMPL\n",
			  sess);
		break;
	case VMK_ISCSI_STOP_CONN_TERM:
		break;
	default:
		;
	}

	sess->recovery_state = 0;

	if (sess->lead_conn == conn) {
		sess->state = QFLE3I_SESS_IN_FFP;
		conn->state = CONN_STATE_FFP_STATE;
	}

	vmk_AtomicWrite64(&conn->stop_state, 0);
	vmk_IscsiTransportStopSessionRecoveryTimer(sess->trans_sess);
	vmk_AtomicWrite64(&sess->device_offline, 0);
	ql_vmk_world_wakeup(&sess->er_wait);

	vmk_SpinlockUnlock(sess->device_lock);

	if (use_poll_timer)
		vmk_TimerSchedule(qfle3i_driver_info.timer_queue,
			(void *)qfle3i_conn_poll,
			(void *) (unsigned long) conn,
			500 * VMK_USEC_PER_MSEC,
			VMK_TIMER_DEFAULT_TOLERANCE,
			VMK_TIMER_ATTR_PERIODIC,
			VMK_LOCKDOMAIN_INVALID,
			VMK_SPINLOCK_UNRANKED,
			&conn->poll_timer);

	sess->if_unblocked = VMK_TRUE;

	sess->scsi_scan_counter = 0;
	qfle3i_queue_scsi_scan(conn);
	ql_vmk_tasklet_schedule(&conn->sess->hba->scan_bh);
	QFLE3I_DBG(DBG_CONN_SETUP, sess->hba, "conn %p for iscsi_conn_cid:0x%x completed\n",
		  conn, conn->iscsi_conn_cid);
	return VMK_OK;
}

static VMK_ReturnStatus qfle3i_stop_conn(
		vmk_IscsiTransConnection *trans_conn,
		vmk_IscsiStopConnMode    flag)
{
	struct qfle3i_conn *conn = (struct qfle3i_conn *)trans_conn->cookie;
	int icid = 0xFFFF;

	if (conn->ep)
		icid = conn->ep->ep_iscsi_cid;

	struct qfle3i_sess *sess = conn->sess;
	PRINT(sess->hba, "%s - sess %p conn %p, ep_iscsi_cid:0x%x, "
	      "cmd stats={p=%d,a=%d,ts=%d,tc=%d}, flag:%d, ofld_conns:%d\n",
	      vmk_NameToString(&sess->hba->vmnic_name), sess, conn, icid,
	      sess->pend_cmd_count, sess->active_cmd_count,
	      sess->total_cmds_sent, sess->total_cmds_completed,
	      flag, sess->hba->ofld_conns_active);

	/*
	 * this stop_conn call may have come before conn is picked up
	 * by scsi_scan tasklet for processing, in that case better to
	 * remove it over here from the hba->scsi_scan_list.
	 */
	vmk_SpinlockLock(sess->hba->scsi_scan_list_lck);

	if (vmk_ListIsUnlinkedElement(&conn->scsi_scan_list) == VMK_FALSE)
		vmk_ListRemove(&conn->scsi_scan_list);

	vmk_SpinlockUnlock(sess->hba->scsi_scan_list_lck);

	vmk_AtomicWrite64(&conn->stop_state, flag);
	conn->state = CONN_STATE_XPORT_FREEZE;
	/*
	 * The strategy here is to take the SAME recovery path as if it was triggered
	 *  internally to the driver. This way we run the same code path in all recovery
	 *  cases
	 */
	qfle3i_do_iscsi_sess_recovery(conn->sess, VMK_SCSI_HOST_RESET, 0); /** Don't signal daemon again */

	/** Update state to */
	vmk_SpinlockLock(conn->sess->lock);
	conn->sess->recovery_state = ISCSI_SESS_RECOVERY_OPEN_ISCSI;
	vmk_SpinlockUnlock(conn->sess->lock);

	switch (flag) {
	case VMK_ISCSI_STOP_CONN_RECOVER:
		conn->sess->state = QFLE3I_SESS_IN_RECOVERY;
		if (!conn->sess->recovery_state) {	/* nopout timeout */

			vmk_SpinlockLock(conn->sess->lock);
			conn->sess->recovery_state =
				ISCSI_SESS_RECOVERY_OPEN_ISCSI;
			vmk_SpinlockUnlock(conn->sess->lock);
		}
		break;
	case VMK_ISCSI_STOP_CONN_TERM:
		if (conn->sess && (conn->sess->state & QFLE3I_SESS_IN_FFP)) {
			conn->sess->state = QFLE3I_SESS_IN_SHUTDOWN;
		}
		break;
	default:
		PRINT_ERR(conn->sess->hba, "invalid conn stop req %d\n", flag);
	}

	if (use_poll_timer)
		vmk_TimerCancel(conn->poll_timer, VMK_TRUE);

	/* Wait for TMF code to exit before returning to daemon */
	qfle3i_wait_for_tmf_completion(conn->sess);

	QFLE3I_DBG(DBG_CONN_SETUP, conn->sess->hba,
		  "conn %p for iscsi_conn_cid:0x%x request complete\n",
		  conn, conn->iscsi_conn_cid);

	if (conn->reject_recvd)
		qfle3i_ep_disconnect(conn->sess->trans, (vmk_uint64)conn->ep);
	conn->reject_recvd = 0;

	return VMK_OK;
}

static VMK_ReturnStatus qfle3i_get_hostparam(
		vmk_IscsiTransTransport *transport,
		void                    *hostPrivate,
		vmk_IscsiTransHostParam param,
		vmk_uint8               *buf,
		vmk_ByteCountSmall      *data_len)
{
	int len = 0;
	struct qfle3i_hba *hba = (qfle3i_hba *)transport->cookie;

	if (!hba) {
		vmk_LogMessage("qfle3i: %s Invalid transport cookie\n",
						__func__);
		return VMK_BAD_PARAM;
	}

	switch(param) {
	case VMK_ISCSI_HOST_PARAM_INITIATOR_NAME:
		if (hba->name_string)
			len = vmk_Sprintf(buf, "%s\n", hba->name_string);
		break;
	case VMK_ISCSI_HOST_PARAM_NETDEV_NAME:
			len = vmk_Sprintf(buf, "%s\n", hba->vmkdev_name);
		break;
	default:
		PRINT_ALERT(hba, "PARAM_UNKNOWN: 0x%x\n", param);
   		return VMK_BAD_PARAM;
	}

	if (len > 0)
		buf[len - 1] = '\0';

	*data_len = len;
	return VMK_OK;
}

static VMK_ReturnStatus qfle3i_set_hostparam(
		vmk_IscsiTransTransport *transport,
		void                    *hostPrivate,
		vmk_IscsiTransHostParam param,
		vmk_uint8                *buf,
		vmk_ByteCountSmall       buflen)
{
	VMK_ReturnStatus ret_val = VMK_OK;
	struct qfle3i_hba *hba = (qfle3i_hba *)transport->cookie;

	if (!hba) {
		vmk_LogMessage("qfle3i: %s Invalid transport cookie\n",
						 __func__);
		return VMK_BAD_PARAM;
	}

	switch(param) {
	case VMK_ISCSI_HOST_PARAM_INITIATOR_NAME:
		if (hba->name_string)
			break;
		ret_val = vmk_StringDup(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
							buf, INITIATOR_MAX_LEN, &hba->name_string);
		if (ret_val != VMK_OK)
			PRINT_ERR(hba, "Set INITIATOR NAME failed\n");
		break;
	case VMK_ISCSI_HOST_PARAM_NETDEV_NAME:
		if (hba->vmkdev_name)
			break;
		ret_val = vmk_StringDup(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
							buf, INITIATOR_MAX_LEN, &hba->vmkdev_name);
		if (ret_val != VMK_OK)
			PRINT_ERR(hba, "Set NETDEV NAME failed\n");
		break;
	default:
		PRINT_ALERT(hba, "PARAM_UNKNOWN: 0x%x\n", param);
		ret_val = VMK_BAD_PARAM;
		break;
	}

	return ret_val;
}

static VMK_ReturnStatus qfle3i_get_limits(
		vmk_IscsiTransTransport    *transport,
		vmk_IscsiTransIscsiParam   param,
		vmk_IscsiTransTransportParamLimits  *limit,
		vmk_int32                 maxListLen)
{
	if ((limit == NULL) || (maxListLen < 2))
		return VMK_FAILURE;

	limit->param = param;

	switch (param) {
/*
		case  ISCSI_PARAM_MAX_SESSIONS:
			limit->type = VMK_ISCSI_TRANPORT_LIMIT_TYPE_LIST;
			limit->hasPreferred = VMK_TRUE;
			limit->limit.list.count = 1;
			if (max_bnx2x_sessions > 128) {
				limit->preferred = 128;
				limit->limit.list.value[0] = 128;
			} else {
				limit->preferred = max_bnx2x_sessions;
				limit->limit.list.value[0] = max_bnx2x_sessions;
			}
			break;
*/
		case  VMK_ISCSI_SESS_PARAM_MAX_R2T:
			limit->type =VMK_ISCSI_TRANPORT_LIMIT_TYPE_MINMAX;
			limit->hasPreferred = VMK_TRUE;
			limit->preferred = 1;
			limit->limit.minMax.min = 1;
			limit->limit.minMax.max = 1;
			break;
		case  VMK_ISCSI_SESS_PARAM_MAX_BURST:
			limit->type = VMK_ISCSI_TRANPORT_LIMIT_TYPE_MINMAX;
			limit->hasPreferred = VMK_TRUE;
			limit->preferred = 262144;
			limit->limit.minMax.min = 8192;
			limit->limit.minMax.max = 16777215;
			break;
		case  VMK_ISCSI_SESS_PARAM_FIRST_BURST:
			limit->type = VMK_ISCSI_TRANPORT_LIMIT_TYPE_MINMAX;
			limit->hasPreferred = VMK_TRUE;
			limit->preferred = 262144;
			limit->limit.minMax.min = 8192;
			limit->limit.minMax.max = 16777215;
			break;
		case VMK_ISCSI_CONN_PARAM_MAX_RECV_DLENGTH:
			limit->type = VMK_ISCSI_TRANPORT_LIMIT_TYPE_MINMAX;
			limit->hasPreferred = VMK_TRUE;
			limit->preferred = 262144;
			limit->limit.minMax.min = 8192;
			limit->limit.minMax.max = 16777215;
			break;
		default:
			limit->type = VMK_ISCSI_TRANPORT_LIMIT_TYPE_UNSUPPORTED;
			break;
	}
   return VMK_OK;
}

static VMK_ReturnStatus qfle3i_get_connstats(
		vmk_IscsiTransConnection *transConn,
		vmk_IscsiTransIscsiStats *stats)
{
   VMK_ReturnStatus status = VMK_OK;

   return status;
}

static VMK_ReturnStatus qfle3i_conn_sendpdu(
		vmk_IscsiTransConnection *trans_conn,
		void                     *pduHdr,
		vmk_uint8                *data,
		vmk_ByteCountSmall       data_size)
{
	struct qfle3i_conn *conn;
	struct qfle3i_cmd *cmnd;
	vmk_uint32 payload_size;
	int count;
	struct iscsi_hdr *hdr = (struct iscsi_hdr *)pduHdr;

	if (!trans_conn) {
		vmk_LogMessage("qfle3i:qfle3i_conn_send_pdu: NULL conn ptr.\n");
		return VMK_IO_ERROR;
	}

	conn = (struct qfle3i_conn *)trans_conn->cookie;
	if (!conn->gen_pdu.req_buf) {
		PRINT_ALERT(conn->sess->hba,
			"send_pdu: login buf not allocated\n");
		goto error;
	}

	/* If adapter is going down (mtu change, vlan, selftest, etc'),
	 * fail this pdu send request so that connection cleanup process
	 * can start right away. If the connection proceeds past this stage
	 * before cnic calls 'qfle3i->ulp_stop', will let this connection
	 * complete FFP migration and then make qfle3i_stop() trigger the cleanup
	 * process.
	 */
	if (!conn->sess)
		return VMK_IO_ERROR;

	if (!conn->sess->hba)
		return VMK_IO_ERROR;

	if (vmk_BitVectorTest(conn->sess->hba->adapter_state, ADAPTER_STATE_GOING_DOWN)) {
		conn->sess->hba->stop_event_ifc_abort_login++;
		goto error;
	}

	if (conn->state != CONN_STATE_XPORT_READY &&
	    conn->state != CONN_STATE_IN_LOGIN &&
	    (hdr->opcode & ISCSI_OPCODE_MASK) == ISCSI_OP_LOGIN) {
		/* login pdu request is valid in transport ready state */
		PRINT_ALERT(conn->sess->hba, "send_pdu: %d != XPORT_READY\n",
				  conn->state);
		goto error;
	}

	if (conn->sess->login_nopout_cmd) {
		cmnd = conn->sess->login_nopout_cmd;
	} else 		/* should not happen ever... */
		goto error;

	vmk_Memset(conn->gen_pdu.req_buf, 0, ISCSI_CONN_LOGIN_BUF_SIZE);
	conn->gen_pdu.req_buf_size = data_size;

	cmnd->conn = conn;
	cmnd->scsi_cmd = NULL;

	ADD_STATS_64(conn->sess->hba, tx_pdus, 1);

	switch (hdr->opcode & ISCSI_OPCODE_MASK) {
	case ISCSI_OP_LOGIN:
			QFLE3I_DBG(DBG_CONN_SETUP, conn->sess->hba,
				  "iscsi login send request, conn %p, iscsi_conn_cid:0x%x, payload %d\n",
				  conn, conn->iscsi_conn_cid, data_size);
			/* Login request, copy hdr & data to buffer in conn struct */
			vmk_Memcpy(&conn->gen_pdu.pdu_hdr, (const void *) hdr,
				sizeof(struct iscsi_hdr));
			if (conn->state == CONN_STATE_XPORT_READY) {
				conn->state = CONN_STATE_IN_LOGIN;
			}
			payload_size = (hdr->dlength[0] << 16) |
						(hdr->dlength[1] << 8) | hdr->dlength[2];

			if (data_size) {
				vmk_Memcpy(conn->gen_pdu.login_req.mem, (const void *)data,
				       data_size);
				conn->gen_pdu.req_wr_ptr =
					conn->gen_pdu.req_buf + payload_size;
			}
			cmnd->iscsi_opcode = hdr->opcode;
			vmk_CPUMemFenceReadWrite();
			vmk_AtomicWrite64(&conn->sess->login_noop_pending, 1);
			ADD_STATS_64(conn->sess->hba, tx_bytes, payload_size);
			break;
		case ISCSI_OP_LOGOUT:
			QFLE3I_DBG(DBG_CONN_SETUP, conn->sess->hba,
				  "iscsi logout send request, conn %p, iscsi_conn_cid:0x%x\n",
				  conn, conn->iscsi_conn_cid);
			/* Logout request, copy header only */
			vmk_Memcpy(&conn->gen_pdu.pdu_hdr, (const void *) hdr,
				sizeof(struct iscsi_hdr));
			conn->gen_pdu.req_wr_ptr = conn->gen_pdu.req_buf;
			conn->state = CONN_STATE_IN_LOGOUT;
			conn->sess->state = QFLE3I_SESS_IN_LOGOUT;

			vmk_IscsiTransportStartSessionRecoveryTimer(conn->sess->trans_sess);
			conn->sess->if_unblocked = VMK_FALSE;
			if (vmk_AtomicRead64(&conn->sess->tmf_active)) {
				/* This should never happen because conn_stop()
				 * will force TMF to fail, also it wait for
				 * 'tmf_active' to clear before returning.
				 */
				qfle3i_wait_for_tmf_completion(conn->sess);
			}

			/* Wait for any outstanding IOs to complete */
			qfle3i_flush_pend_queue(conn->sess, NULL, VMK_SCSI_HOST_RESET);
			count = 10;
			while (count-- && conn->sess->active_cmd_count)
				vmk_WorldSleep(50 * VMK_USEC_PER_MSEC);

			/* Wait for any outstanding iscsi nopout to complete */
			count = 10;
			while (count-- && cmnd->iscsi_opcode)
				vmk_WorldSleep(100 * VMK_USEC_PER_MSEC);
			if (cmnd->iscsi_opcode)
				goto error;

			cmnd->iscsi_opcode = hdr->opcode;
			vmk_CPUMemFenceReadWrite();
			vmk_AtomicWrite64(&conn->sess->logout_pending, 1);
			break;
		case ISCSI_OP_NOOP_OUT:
			if(!conn->sess->hba->is_nic_up)
			{
				PRINT(conn->sess->hba, "NIC is down; not sending nop-out\n");
				return VMK_OK;
			}
			conn->sess->last_nooput_requested = vmk_GetTimerCycles();
			conn->sess->noopout_requested_count++;
			/* connection is being logged out, do not allow NOOP */
			if (conn->state == CONN_STATE_IN_LOGOUT)
				goto error;

			/* unsolicited iSCSI NOOP copy hdr into conn struct */
			vmk_Memcpy(&conn->gen_pdu.nopout_hdr, (const void *) hdr,
				sizeof(struct iscsi_hdr));
			cmnd->iscsi_opcode = hdr->opcode;
			cmnd->ttt = ISCSI_RESERVED_TAG;
			vmk_CPUMemFenceReadWrite();
			vmk_AtomicWrite64(&conn->sess->login_noop_pending, 1);
			break;
		default:
			;
		}

	if (vmk_AtomicRead64(&conn->worker_enabled)) {
		vmk_AtomicWrite64(&conn->lastSched,11);
		QFLE3I_DBG(DBG_CONN_SETUP, conn->sess->hba,
				  "scheduling conn tasklet\n");
		ql_vmk_tasklet_schedule(&conn->conn_tasklet);
	}
	return VMK_OK;
error:
	PRINT(conn->sess->hba, "failed to send pdu!!\n");
	return VMK_IO_ERROR;
}

static void qfle3i_sessrecov_timedout(
		vmk_IscsiTransSession   *trans_sess)
{
	struct qfle3i_sess *sess = (qfle3i_sess *)trans_sess->cookie;
	PRINT(sess->hba, "sess %p recovery timed out\n", sess);

	vmk_SpinlockLock(sess->device_lock);
	if (sess->recovery_state) {

		/* Inform mid-layer about path going down */
		vmk_ScsiNotifyPathStateChangeAsync(
					sess->hba->scsiAdapter,
					sess->channel_id,
					sess->target_id, -1);

		vmk_SpinlockLock(sess->lock);
		vmk_AtomicWrite64(&sess->device_offline, 1);
		sess->recovery_state |= ISCSI_SESS_RECOVERY_FAILED;
		vmk_SpinlockUnlock(sess->lock);
	}
	vmk_SpinlockUnlock(sess->device_lock);
}

VMK_ReturnStatus qfle3i_register_adapter(qfle3i_hba *hba)
{
	VMK_ReturnStatus status = VMK_OK;

	/* Bind scsi_adapter and iscsi transport */
	status = vmk_IscsiTransportRegisterAdapter(hba->scsiAdapter,
	                               hba->iscsiTransport);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "Failed to register adapter with transport, status:%s\n",
							vmk_StatusToString(status));
		return status;
	}
	QFLE3I_DBG(DBG_INIT, hba, "adapter registered, transport_name: %s "
							 "vmhba_name:%s\n",
						hba->iscsiTransport->name,
						vmk_NameToString(&hba->scsiAdapter->name));


	/* Copy vmhba name from scsi adapter */
	vmk_NameCopy(&hba->vmhba_name, &hba->scsiAdapter->name);

	return status;
}


VMK_ReturnStatus qfle3i_unreg_iscsi_adapter(qfle3i_hba *hba)
{
	VMK_ReturnStatus status = VMK_OK;
	status =
		vmk_IscsiTransportUnregisterAdapter(hba->scsiAdapter);
	if (status != VMK_OK) {
		vmk_WarningMessage("vmk_IscsiTransportUnregisterAdapter failed: (%s)",
					vmk_StatusToString(status));
	}


#if 0
	if (hba->scsiAdapter->mgmtAdapter.t.iscsi != NULL) {
		status =
			vmk_IscsiTransportUnregisterAdapter(hba->scsiAdapter);
		if (status != VMK_OK) {
			vmk_WarningMessage("vmk_IscsiTransportUnregisterAdapter failed: (%s)",
						vmk_StatusToString(status));
		}
	}
#endif
	return status;
}

VMK_ReturnStatus qfle3i_register_transport(qfle3i_hba *hba)
{
	vmk_int32   idx;
	VMK_ReturnStatus status = VMK_OK;
	vmk_IscsiTransIscsiParamProperty    iscsiParamProperty;
	vmk_IscsiTransHostParamProperty     hostParamProperty;
	vmk_IscsiTransCapabilitiesProperty  capsProperty;
	vmk_IscsiTransTransport *trans = NULL;

	QFLE3I_DBG(DBG_INIT, hba, "Allocating mem for iscsi trans, hba:0x%lx\n",
							(unsigned long) hba);

	trans = vmk_HeapAlign(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
							sizeof(vmk_IscsiTransTransport),
							VMK_L1_CACHELINE_SIZE);
	if (trans == NULL) {
		DRV_WARN("Failed to alloc transport", status);
		return VMK_NO_MEMORY;
	}

	status = vmk_IscsiTransportCreateIscsiParamProperty(&iscsiParamProperty);;
	if (status != VMK_OK) {
		goto iscsiParamCreatePropertyFailed;
	}

	/* add connection params */
	for (idx = 0; idx <
		 sizeof(qfle3i_conn_params) / sizeof(qfle3i_conn_params[0]);
		 idx++) {
		status = vmk_IscsiTransportAddIscsiParamProperty(iscsiParamProperty,
											qfle3i_conn_params[idx]);

		if (status != VMK_OK) {
			vmk_WarningMessage("Failed to add IscsiParam property %d, error 0x%x",
								qfle3i_conn_params[idx], status);
			goto iscsiParamAddPropertyItemFailed;
		}
	}

	/* add session params */
	for (idx = 0; idx <
		 sizeof(qfle3i_sess_params) / sizeof(qfle3i_sess_params[0]);
		 idx++) {
		status = vmk_IscsiTransportAddIscsiParamProperty(iscsiParamProperty,
										qfle3i_sess_params[idx]);

		if(status != VMK_OK) {
			vmk_WarningMessage("Failed to add IscsiParam property %d, error 0x%x",
								qfle3i_sess_params[idx], status);
			goto iscsiParamAddPropertyItemFailed;
		}
	}

	status = vmk_IscsiTransportCreateHostParamProperty(&hostParamProperty);
	if (status != VMK_OK) {
		goto hostParamCreatePropertyFailed;
	}

   /* add host params */
	for (idx = 0; idx <
		 sizeof(qfle3i_host_params) / sizeof (qfle3i_host_params[0]);
		 idx++) {
		status = vmk_IscsiTransportAddHostParamProperty(hostParamProperty,
									qfle3i_host_params[idx]);

		if(status != VMK_OK) {
			vmk_WarningMessage("Failed to add hostParam property %d, 0x%x",
								qfle3i_host_params[idx], status);
			goto hostParamAddPropertyItemFailed;
		}
	}

	status = vmk_IscsiTransportCreateCapabilitiesProperty(&capsProperty);
	if (status != VMK_OK) {
		goto capsCreatePropertyFailed;
	}

/* add individual caps */
	for (idx = 0; idx <
		 sizeof(qfle3i_supportedCaps) / sizeof (qfle3i_supportedCaps[0]);
		 idx++) {
		status = vmk_IscsiTransportAddCapabilitiesProperty(capsProperty,
								qfle3i_supportedCaps[idx]);

		if(status != VMK_OK) {
			vmk_WarningMessage("Failed to add hostParam property %d, 0x%x",
								qfle3i_supportedCaps[idx], status);
			goto capsAddPropertyItemFailed;
		}
	}

	trans->iscsiParamProperty = iscsiParamProperty;
	trans->hostParamProperty = hostParamProperty;
	trans->capsProperty = capsProperty;

	/* copy mac address from cnic_dev */
	vmk_Memcpy(hba->mac_addr, hba->cnic->nic_mac, VMK_ETH_ADDR_LENGTH);

	/* VK: we will have add polling here, if we dont get uplink_name,
		this is because cnic dont have uplink name yet
	*/
	status = hba->cnic->cm_get_uplinkName(hba->cnic, &hba->vmnic_name);
	if (status != VMK_OK)
		goto getuplink_failed;

	QFLE3I_DBG(DBG_INIT, hba, "nic_name:%s "
		"mac-addr: %02hhx:%02hhx:%02hhx:%02hhx:%02hhx:%02hhx\n",
		vmk_NameToString(&hba->vmnic_name),
		hba->mac_addr[0], hba->mac_addr[1], hba->mac_addr[2],
		hba->mac_addr[3], hba->mac_addr[4], hba->mac_addr[5]);


	/* set transport name as scsi adapter name */
	vmk_Snprintf(trans->name, VMK_ISCSI_MAX_TRANSPORT_NAME_SZ,
				"%s-%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx-%s",
				BNX2I_DRIVER_NAME,
				hba->mac_addr[0],
				hba->mac_addr[1],
				hba->mac_addr[2],
				hba->mac_addr[3],
				hba->mac_addr[4],
				hba->mac_addr[5],
				vmk_NameToString(&hba->vmnic_name));

	switch (hba->pcidev_id.deviceID) {
		case PCI_DEVICE_ID_NX2_5706:
		case PCI_DEVICE_ID_NX2_5706S:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 5706 1 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_5708:
		case PCI_DEVICE_ID_NX2_5708S:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 5708 1 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_5709:
		case PCI_DEVICE_ID_NX2_5709S:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 5709 1 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_5716:
		case PCI_DEVICE_ID_NX2_5716S:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 5716 1 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57710:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57710 10 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57711:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57711 10 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57712:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57712 10 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57800:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57800 10 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57810:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57810 10 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57811:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57811 10 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57811S:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57811S 10 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57840:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57840 10 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57840S:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57840 10 Gigabit Ethernet Adapter");
			break;
		case PCI_DEVICE_ID_NX2_57840_20G:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
							"%s", "QLogic 57840 20 Gigabit Ethernet Adapter");
			break;
		default:
			vmk_Snprintf(trans->description, VMK_ISCSI_MAX_TRANSPORT_DESC_SZ,
					"%s", "Qlogic 10 Gb Ethernet Adapter");
	}

	 /* some limits */
	trans->maxConn                   = ISCSI_MAX_CONNS_PER_HBA;

	/* callbacks */
	trans->createSessionPersistent   = qfle3i_create_sess;
	trans->destroySession            = qfle3i_destroy_sess;
	trans->createConnection          = qfle3i_create_conn;
	trans->bindConnection            = qfle3i_bind_conn;
	trans->destroyConnection         = qfle3i_destroy_conn;
	trans->setParam                  = qfle3i_setparam;
	trans->getConnectionParam        = qfle3i_get_connparam;
	trans->getSessionParam           = qfle3i_get_sessparam;
	trans->startConnection           = qfle3i_start_conn;
	trans->stopConnection            = qfle3i_stop_conn;
	trans->getHostParam              = qfle3i_get_hostparam;
	trans->setHostParam              = qfle3i_set_hostparam;
	trans->sendPDU                   = qfle3i_conn_sendpdu;
	trans->getStats                  = qfle3i_get_connstats;
	trans->getTransportLimits        = qfle3i_get_limits;
	trans->sessionRecoveryTimedout   = qfle3i_sessrecov_timedout;
	trans->connectExtended           = qfle3i_ep_connect;
	trans->poll                      = qfle3i_ep_poll;
	trans->disconnect                = qfle3i_ep_disconnect;

	/* keep hba saved in cookie */
	trans->cookie = (void *)hba;

	/* Register transport structure with transport layer */
	status = vmk_IscsiTransportRegisterTransport(trans);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "Failed to register transport, status: %s",
						vmk_StatusToString(status));
		goto registerTransport_Failed;
	}

	hba->iscsiTransport = trans;
	return VMK_OK;

registerTransport_Failed:
capsAddPropertyItemFailed:
getuplink_failed:
	vmk_IscsiTransportDestroyCapabilitiesProperty(capsProperty);

capsCreatePropertyFailed:
hostParamAddPropertyItemFailed:
	vmk_IscsiTransportDestroyHostParamProperty(hostParamProperty);

hostParamCreatePropertyFailed:
iscsiParamAddPropertyItemFailed:
	vmk_IscsiTransportDestroyIscsiParamProperty(iscsiParamProperty);

iscsiParamCreatePropertyFailed:
	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), trans);
	return VMK_FAILURE;
}

VMK_ReturnStatus qfle3i_unregister_transport(qfle3i_hba *hba)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_IscsiTransTransport *iscsiTransport;

	iscsiTransport = hba->iscsiTransport;
	if (!iscsiTransport) {
		vmk_WarningMessage("hba->iscsiTransport is null.\n");
		return VMK_FAILURE;
	}

	if (iscsiTransport->iscsiParamProperty) {
		status =
		vmk_IscsiTransportDestroyIscsiParamProperty(iscsiTransport->iscsiParamProperty);
		if (status != VMK_OK) {
			vmk_WarningMessage("vmk_IscsiTransportDestroyIscsiParamProperty "
								"failed: (%s)", vmk_StatusToString(status));
			return status;
		}
		else {
			iscsiTransport->iscsiParamProperty = NULL;
		}
	}

	if (iscsiTransport->hostParamProperty) {
		status =
		vmk_IscsiTransportDestroyHostParamProperty(iscsiTransport->hostParamProperty);
		if (status != VMK_OK) {
			vmk_WarningMessage("vmk_IscsiTransportDestroyHostParamProperty "
								"failed: (%s)", vmk_StatusToString(status));
			return status;
		}
		else {
			iscsiTransport->hostParamProperty = NULL;
		}
	}

	if (iscsiTransport->capsProperty) {
		status =
			vmk_IscsiTransportDestroyCapabilitiesProperty(iscsiTransport->capsProperty);
		if (status != VMK_OK) {
			vmk_WarningMessage("vmk_IscsiTransportDestroyCapabilitiesProperty "
								"failed: (%s)", vmk_StatusToString(status));
			return status;
		}
		else {
			iscsiTransport->capsProperty = NULL;
		}
	}

   /* Unregister from iscsi_transport */
	if (hba->iscsiTransport) {
		status =
			vmk_IscsiTransportUnregisterTransport(hba->iscsiTransport);
		if (status != VMK_OK) {
			vmk_WarningMessage("vmk_IscsiTransportUnregisterTransport failed: (%s)",
									vmk_StatusToString(status));
			return status;
		}
		else {
			vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						 hba->iscsiTransport);
			hba->iscsiTransport = NULL;
		}
	}
	return status;
}

static void qfle3i_handle_net_event(void *data)
{
	struct qfle3i_hba *hba = (qfle3i_hba *)data;
	int timeout = 0;
	int count = 0;

	if (!hba) {
		vmk_WarningMessage("%s:%d: Fix this, it should not happen.\n",
				__func__, __LINE__);
		return;
	}
	while(1) {
		if(!vmk_BitVectorTest(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED))
			return;

		if(!hba)
			return;

		vmk_WorldSleep(1000 * VMK_USEC_PER_MSEC);

		if(hba->is_nic_up)
			break;

		count++;

		if(count > 10) {
			timeout = 1;
			break;
		}
	}

	if(timeout) {
		PRINT_NOTICE(hba, "Initiating recovery on all sess; link is down for more 10 seconds");
		qfle3i_start_iscsi_hba_shutdown(hba);
	}
}
/*
 * conn_err_recovery_task - does recovery on all queued sessions
 *
 * @work:		pointer to work struct
 *
 * iSCSI Session recovery queue manager
 */
static void conn_err_recovery_task(void *data)
{
	struct qfle3i_hba *hba = (qfle3i_hba *)data;
	struct qfle3i_sess *sess;

	if (!hba) {
		vmk_WarningMessage("%s:%d: Fix this, it should not happen.\n",
							__func__, __LINE__);
		return;
	}
	int cons_idx = hba->sess_recov_cons_idx;

	vmk_SpinlockLock(hba->lock);
	while (hba->sess_recov_prod_idx != cons_idx) {
		sess = hba->sess_recov_list[cons_idx];
		if (cons_idx == hba->sess_recov_max_idx)
			cons_idx = 0;
		else
			cons_idx++;
		vmk_SpinlockUnlock(hba->lock);
		if (sess) {
			if (sess->state == QFLE3I_SESS_IN_LOGOUT)
				qfle3i_do_iscsi_sess_recovery(sess, VMK_SCSI_HOST_NO_CONNECT, 1);
			else
				qfle3i_do_iscsi_sess_recovery(sess, VMK_SCSI_HOST_RESET, 1);
		}
		vmk_SpinlockLock(hba->lock);
	}
	hba->sess_recov_cons_idx = cons_idx;
	vmk_SpinlockUnlock(hba->lock);
}

static void ep_tmo_cleanup(struct qfle3i_hba *hba, struct qfle3i_endpoint *ep)
{
	struct qfle3i_sess *sess;

	hba->ep_tmo_cmpl_cnt++;
	/* remove the recovered EP from the link list */
	vmk_ListRemove(&ep->link);
	vmk_ListInit(&ep->link);

	qfle3i_tear_down_ep(hba, ep);
	if (ep->sess) {
		int cmds_a = 0, cmds_p = 0;
		cmds_p = qfle3i_flush_pend_queue(ep->sess, NULL, VMK_SCSI_HOST_RESET);
		cmds_a = qfle3i_flush_cmd_queue(ep->sess, NULL, VMK_SCSI_HOST_RESET, 0);
		QFLE3I_DBG(DBG_ITT_CLEANUP, hba,
			"CMDS FLUSHED, pend=%d, active=%d\n",
			cmds_p, cmds_a);
	}

	QFLE3I_DBG(DBG_CONN_SETUP, hba, "qfle3i: ep %p, destroyed\n", ep);
	PRINT_ALERT(hba, "####CID recovered sess %p ep %p {0x%x, 0x%x}\n",
		ep->sess, ep, ep->ep_iscsi_cid, ep->ep_cid);
	qfle3i_free_qp_resc(hba, ep);
	/* check if session recovery in progress */
	sess = ep->sess;
	qfle3i_free_ep(ep);
	if (sess) {
		sess->state = QFLE3I_SESS_INITIAL;
		ql_vmk_world_wakeup(&sess->er_wait);
	}
	ql_vmk_world_wakeup(&hba->eh_wait);

}
/*
 * ep_tmo_poll_task - check endpoints that are in disconnect timeout state and
 *                    cleanup the endpoint if the chip has acknowledged the
 *                    disconnect.
 *
 * @work:		pointer to work struct
 *
 */
static void ep_tmo_poll_task(void *data)
{
	qfle3i_hba *hba = (qfle3i_hba *)data;
	struct qfle3i_endpoint *ep, *ep_n;

	if (!hba) {
		vmk_WarningMessage("Fix this, it should not happen.\n");
		return;
	}

	vmk_SemaLock(&hba->net_dev_lock);
	ql_vmk_list_each_entry_safe(ep, ep_n, &hba->ep_stale_list, link, struct qfle3i_endpoint) {
		if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_GOING_DOWN))
			ep->state = EP_STATE_DISCONN_COMPL;
		if (ep->state != EP_STATE_DISCONN_TIMEDOUT) {
			ep_tmo_cleanup(hba, ep);
			break;
		}
	}
	ql_vmk_list_each_entry_safe(ep, ep_n, &hba->ep_tmo_list, link, struct qfle3i_endpoint) {
		if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_GOING_DOWN))
			ep->state = EP_STATE_DISCONN_COMPL;
		if (ep->state != EP_STATE_DISCONN_TIMEDOUT) {
			ep_tmo_cleanup(hba, ep);
			break;
		} else if (vmk_GetTimerCycles() > (ep->timestamp + (60 * vmk_TimerCyclesPerSecond()))) {
			PRINT_ALERT(hba,
				"please submit GRC Dump (ep %p {0x%x, 0x%x}),"
				" NW/PCIe trace, driver msgs to developers"
				" for analysis\n",
				ep, ep->ep_iscsi_cid, ep->ep_cid);

			vmk_ListRemove(&ep->link);
			vmk_ListInit(&ep->link);
			vmk_ListInsert(&ep->link, vmk_ListAtRear(&hba->ep_stale_list));

			vmk_SpinlockLock(qfle3i_resc_lock);
			ep->hba->ofld_conns_active--;
			ep->in_progress = 0;
			vmk_SpinlockUnlock(qfle3i_resc_lock);
		}
	}

	if (vmk_ListIsEmpty(&hba->ep_tmo_list) && vmk_ListIsEmpty(&hba->ep_stale_list)) {
		vmk_AtomicWrite64(&hba->ep_tmo_poll_enabled, 0);
		PRINT_ALERT(hba, "ep_tmo_list and ep_stale_list is now empty and"
			" ep_poll_task is terminated!\n");
	}
	vmk_SemaUnlock(&hba->net_dev_lock);

	if (vmk_AtomicRead64(&hba->ep_tmo_poll_enabled)) {
		ql_vmk_wait_for_completion(&hba->ep_tmo_wait,
				vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_GOING_DOWN),
				1000);

		ql_vmk_singlethread_queue_work(qfle3i_driver_info.delayed_wq,
								&hba->ep_poll_task);
	}
}

/*
 * qfle3i_ep_tcp_conn_active - check EP state transition to check
 *		if underlying TCP connection is active
 *
 * @ep: 		endpoint pointer
 *
 */
static int qfle3i_ep_tcp_conn_active(struct qfle3i_endpoint *ep)
{
	int ret;

	switch (ep->state) {
	case EP_STATE_CLEANUP_FAILED:
	case EP_STATE_OFLD_FAILED:
	case EP_STATE_DISCONN_TIMEDOUT:
		ret = 0;
		break;
	case EP_STATE_CONNECT_COMPL:
	case EP_STATE_ULP_UPDATE_START:
	case EP_STATE_ULP_UPDATE_COMPL:
	case EP_STATE_TCP_FIN_RCVD:
	case EP_STATE_ULP_UPDATE_FAILED:
	case EP_STATE_CONNECT_FAILED:
	case EP_STATE_TCP_RST_RCVD:
	case EP_STATE_ULP_UPDATE_TIMEOUT:
		/* cnic need to upload PG for 570x chipsets and there is
		 * an understanding it is safe to call cm_abort() even if
		 * cm_connect() failed for all chip types
		 */
		ret = 1;
		break;
	case EP_STATE_CONNECT_START:
		/* qfle3i will not know whether PG needs to be uploaded or not.
		 * qfle3i will call cm_abort() and let cnic decide the clean-up
		 * action that needs to be taken
		 */
		ret = 1;
		break;
	default:
		ret = 0;
	}

	return ret;
}

/*
 * qfle3i_start_iscsi_hba_shutdown - start hba shutdown by cleaning up
 *			all active sessions
 *
 * @hba: 		pointer to adapter instance
 *
 *  interface is being brought down by the user, fail all active iSCSI sessions
 *	belonging to this adapter
 */
void qfle3i_start_iscsi_hba_shutdown(struct qfle3i_hba *hba)
{
	struct qfle3i_sess *sess;
	int lpcnt;
	int rc;

	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(sess, &hba->active_sess, link, struct qfle3i_sess) {
		vmk_SpinlockUnlock(hba->lock);
		lpcnt = 4;
		rc = qfle3i_do_iscsi_sess_recovery(sess, VMK_SCSI_HOST_NO_CONNECT, 1);

		while ((rc != VMK_OK) && lpcnt--) {
			vmk_WorldSleep(1000 * VMK_USEC_PER_MSEC);
			rc = qfle3i_do_iscsi_sess_recovery(sess, VMK_SCSI_HOST_NO_CONNECT, 1);
		}
		vmk_SpinlockLock(hba->lock);
	}
	vmk_SpinlockUnlock(hba->lock);
}

/*
 * qfle3i_iscsi_hba_cleanup - System is shutting down, cleanup all offloaded connections
 *
 * @hba: 		pointer to adapter instance
 *
 */
void qfle3i_iscsi_hba_cleanup(struct qfle3i_hba *hba)
{
	struct qfle3i_sess *sess;
	struct qfle3i_endpoint *ep;

	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(sess, &hba->active_sess, link, struct qfle3i_sess) {
		vmk_SpinlockUnlock(hba->lock);
		if (sess)
			if (sess->lead_conn && sess->lead_conn->ep) {
				ep = sess->lead_conn->ep;
				qfle3i_stop_conn(sess->lead_conn->trans_conn,
								 VMK_ISCSI_STOP_CONN_RECOVER);
				qfle3i_ep_disconnect(hba->iscsiTransport, (vmk_uint64)ep);
			}
		vmk_SpinlockLock(hba->lock);
	}
	vmk_SpinlockUnlock(hba->lock);
}

static void qfle3i_withdraw_sess_recovery(struct qfle3i_sess *sess)
{
	struct qfle3i_hba *hba = sess->hba;
	int cons_idx = hba->sess_recov_cons_idx;

	vmk_SpinlockLock(hba->lock);
	while (hba->sess_recov_prod_idx != cons_idx) {
		if (sess == hba->sess_recov_list[cons_idx]) {
			hba->sess_recov_list[cons_idx] = NULL;
			break;
		}
		if (cons_idx == hba->sess_recov_max_idx)
			cons_idx = 0;
		else
			cons_idx++;
	}
	vmk_SpinlockUnlock(hba->lock);
}

/*
 * qfle3i_do_iscsi_sess_recovery - implements session recovery code
 *
 * @sess:		iscsi session pointer
 * @reason: 		SCSI ML error code, VMK_SCSI_HOST_BUS_BUSY, VMK_SCSI_HOST_NO_CONNECT,
 *			VMK_SCSI_HOST_RESET
 *
 * SCSI host reset handler, which is translates to iSCSI session
 *	recovery. This routine starts internal driver session recovery,
 *	indicates connection error to 'iscsid' which does session reinstatement
 *	This is an synchronous call which waits for completion and returns
 *	the ultimate result of session recovery process to caller
 */
int qfle3i_do_iscsi_sess_recovery(struct qfle3i_sess *sess, int reason, int signal)
{
	struct qfle3i_hba *hba;
	struct qfle3i_conn *conn = sess->lead_conn;
	int cmds_1 = 0;

	if (!conn)
		return VMK_FAILURE;

	QFLE3I_DBG(DBG_SESS_RECO, sess->hba, "sess %p conn %p\n",
		  sess, conn);

	vmk_SpinlockLock(sess->lock);
	if (sess->recovery_state) {
		QFLE3I_DBG(DBG_SESS_RECO, sess->hba,
			  "RECOVERY ALREADY ACTIVE sess %p\n", sess);
		vmk_SpinlockUnlock(sess->lock);
		ql_vmk_wait_for_completion(&sess->er_wait,
			!vmk_AtomicRead64(&sess->do_recovery_inprogess),
			128 * VMK_MSEC_PER_SEC);
		return VMK_OK;
	}
	sess->lead_conn->state = CONN_STATE_XPORT_FREEZE;
	sess->recovery_state = ISCSI_SESS_RECOVERY_START;
	vmk_SpinlockUnlock(sess->lock);
	vmk_AtomicWrite64(&sess->do_recovery_inprogess, 1);
	if (sess->state != QFLE3I_SESS_IN_LOGOUT) {
		/* logout blocks session to prevent accepting more cmds */
		vmk_IscsiTransportStartSessionRecoveryTimer(sess->trans_sess);
		sess->if_unblocked = VMK_FALSE;
	}
	hba = sess->hba;

	/*
	 *  DO THE CLEANUP HERE
	 */
	if (vmk_AtomicRead64(&sess->tmf_active)) {
		QFLE3I_DBG(DBG_TMF, hba, "WAKE UP TMF WAITER sess %p\n", sess);
		vmk_AtomicWrite64(&sess->scsi_tmf_cmd->cmd_state, ISCSI_CMD_STATE_TMF_TIMEOUT);
		ql_vmk_world_wakeup(&sess->er_wait);
	} else {
		QFLE3I_DBG(DBG_TMF, hba, "TMF NOT ACTIVE sess %p\n", sess);
	}
	/** 
	 *  Aborts have been woken, and should be cleared out 
	 *  We can wait for the tmf_active flag to drop if need be here
	 *  but it should not matter
	 */
//	sess->cmd_cleanup_req = 0;
//	sess->cmd_cleanup_cmpl = 0;

	QFLE3I_DBG(DBG_SESS_RECO, hba, "FLUSH PENDING/ACTIVE sess %p\n", sess);
	cmds_1 = qfle3i_flush_pend_queue(sess, NULL, VMK_SCSI_HOST_RESET);
	/* We can't cleanup here because link may be down and CMD_CLEANUP may not
	 * be processed by the firmware because of lack on CCELL. CMD_CLEANUP
	 * works well when TCP layer is function well and the fault lies in
	 * in iSCSI or SCSI backend. Because at this stage we can't guarantee
	 * any of the active commands are setup in F/W and TCP ACK'ed by target,
	 * we can't flush active queue here.
	 qfle3i_flush_cmd_queue(sess, NULL, VMK_SCSI_HOST_RESET, 1);
	 */
	QFLE3I_DBG(DBG_SESS_RECO, hba, "qfle3i: FLUSH DONE sess %p\n", sess);
	/** SIGNAL CONNECTION FAULT */
	if (signal) {
		QFLE3I_DBG(DBG_SESS_RECO, hba, "Notify daemon to start"
			  " reconnecting\n");
		vmk_IscsiTransportReportConnectionError(conn->trans_conn,
												VMK_ISCSI_ERR_CONN_FAILED);
	}

#if 0
	VK: find alternate in native.
	if (signal_pending(current))
		flush_signals(current);
#endif

	QFLE3I_DBG(DBG_SESS_RECO, hba,
		  "sess %p cmds stats: PD=%d, Q=%d, S=%d, D=%d, F=%d, CC=%d\n",
		  sess, cmds_1, sess->total_cmds_queued, sess->total_cmds_sent,
		  sess->total_cmds_completed, sess->total_cmds_failed,
		  sess->total_cmds_completed_by_chip);

	vmk_AtomicWrite64(&sess->do_recovery_inprogess, 0);
	vmk_CPUMemFenceReadWrite();
	ql_vmk_world_wakeup(&sess->er_wait);
	return VMK_OK;

}

/*
 * qfle3i_login_resp_update_cmdsn - extracts SN & MAX_SN from login response header &
 *			updates driver 'cmdsn' with 
 *
 * @conn: 		iscsi connection pointer
 *
 * extract & update SN counters from login response
 */
static int qfle3i_login_resp_update_cmdsn(struct qfle3i_conn *conn)
{
	vmk_uint32 max_cmdsn;
	vmk_uint32 exp_cmdsn;
	vmk_uint32 stat_sn;
	struct qfle3i_sess *sess = conn->sess;
	struct iscsi_nopin *hdr;

	hdr = (struct iscsi_nopin *) &conn->gen_pdu.resp_hdr;

	max_cmdsn = ntohl(hdr->max_cmdsn);
	exp_cmdsn = ntohl(hdr->exp_cmdsn);
	stat_sn = ntohl(hdr->statsn);
#define SN_DELTA_ISLAND		0xffff
	if (max_cmdsn < exp_cmdsn -1 &&
	    max_cmdsn > exp_cmdsn - SN_DELTA_ISLAND)
		return VMK_BAD_PARAM;

	if (max_cmdsn > sess->max_cmdsn ||
	    max_cmdsn < sess->max_cmdsn - SN_DELTA_ISLAND)
		sess->max_cmdsn = max_cmdsn;

	if (exp_cmdsn > sess->exp_cmdsn ||
	    exp_cmdsn < sess->exp_cmdsn - SN_DELTA_ISLAND)
		sess->exp_cmdsn = exp_cmdsn;

	if (stat_sn == conn->exp_statsn)
		conn->exp_statsn++;

	return 0;
}


/*
 * qfle3i_update_cmd_sequence - update session sequencing parameter
 *
 * @sess:		iscsi session pointer
 * @exp_sn: 		iscsi expected command seq num
 * @max_sn: 		iscsi max command seq num
 *
 * update iSCSI SN counters for the given session
 */
void qfle3i_update_cmd_sequence(struct qfle3i_sess *sess,
			       vmk_uint32 exp_sn, vmk_uint32 max_sn)
{
	vmk_uint32 exp_cmdsn = exp_sn;
	vmk_uint32 max_cmdsn = max_sn;

	if (max_cmdsn < exp_cmdsn -1 &&
	    max_cmdsn > exp_cmdsn - SN_DELTA_ISLAND) {
		PRINT_ALERT(sess->hba,
				"cmd_sequence: error, exp 0x%x, max 0x%x\n",
				exp_cmdsn, max_cmdsn);
	}
	if (max_cmdsn > sess->max_cmdsn ||
	    max_cmdsn < sess->max_cmdsn - SN_DELTA_ISLAND)
		sess->max_cmdsn = max_cmdsn;
	if (exp_cmdsn > sess->exp_cmdsn ||
	    exp_cmdsn < sess->exp_cmdsn - SN_DELTA_ISLAND)
		sess->exp_cmdsn = exp_cmdsn;
}

/*
 * qfle3i_indicate_login_resp - process iscsi login response
 *
 * @conn: 		iscsi connection pointer
 *
 * pushes login response PDU to application daemon, 'iscsid' by
 *		calling iscsi_recv_pdu()
 */
int qfle3i_indicate_login_resp(struct qfle3i_conn *conn)
{
	int data_len;
	struct iscsi_login_rsp *login_resp =
		(struct iscsi_login_rsp *) &conn->gen_pdu.resp_hdr;

	/* check if this is the first login response for this connection.
	 * If yes, we need to copy initial StatSN to connection structure.
	 */
	if (conn->exp_statsn == STATSN_UPDATE_SIGNATURE) {
		conn->exp_statsn = ntohl(login_resp->statsn) + 1;
	}

	if (qfle3i_login_resp_update_cmdsn(conn))
		return VMK_BAD_PARAM;

	data_len = conn->gen_pdu.resp_wr_ptr - conn->gen_pdu.resp_buf;

	QFLE3I_DBG(DBG_CONN_EVENT, conn->sess->hba,
		  "indicate login resp, conn %p, ep_iscsi_cid:0x%x, trans_conn %p\n",
		  conn, conn->ep->ep_iscsi_cid, conn->trans_conn);

	vmk_IscsiTransportForwardPDU(conn->trans_conn, (void *)login_resp,
		       (vmk_uint8 *)conn->gen_pdu.resp_buf, data_len);

	return 0;
}


/*
 * qfle3i_indicate_logout_resp - process iscsi logout response
 *
 * @conn: 		iscsi connection pointer
 *
 * pushes logout response PDU to application daemon, 'iscsid' by
 *		calling iscsi_recv_pdu()
 */
int qfle3i_indicate_logout_resp(struct qfle3i_conn *conn)
{
	struct iscsi_logout_rsp *logout_resp =
		(struct iscsi_logout_rsp *) &conn->gen_pdu.resp_hdr;

	QFLE3I_DBG(DBG_CONN_EVENT, conn->sess->hba,
		  "indicate logout resp, ep_iscsi_cid:0x%x, trans_conn %p, logout_hdr %p\n",
		  conn->ep->ep_iscsi_cid, conn->trans_conn, logout_resp);

	vmk_IscsiTransportForwardPDU(conn->trans_conn, (void *)logout_resp,
		       (vmk_uint8 *) NULL, 0);
	return 0;
}


/*
 * qfle3i_indicate_async_mesg - process iscsi ASYNC message indication
 *
 * @conn: 		iscsi connection pointer
 *
 * pushes iSCSI async PDU to application daemon, 'iscsid' by calling
 *	iscsi_recv_pdu()
 */
int qfle3i_indicate_async_mesg(struct qfle3i_conn *conn)
{
	struct iscsi_async *async_msg =
		(struct iscsi_async *) &conn->gen_pdu.async_hdr;

	PRINT(conn->sess->hba,
			"indicating async message, opcode:0x%x, async_event:0x%x,"
			"async_vcode:0x%x  params=0x%x:0x%x:0x%x, ep_iscsi_cid:0x%x\n",
			async_msg->opcode, async_msg->async_event, async_msg->async_vcode,
			async_msg->param1, async_msg->param2, async_msg->param3,
			conn->ep->ep_iscsi_cid);

	vmk_IscsiTransportForwardPDU(conn->trans_conn, (void *)async_msg,
		       (vmk_uint8 *) NULL, 0);
	return 0;
}

/*
 * qfle3i_process_nopin - process iscsi nopin pdu
 *
 * @conn: 		iscsi connection pointer
 * @cmd:		iscsi cmd pointer
 * @data_buf:		payload buffer pointer
 * @data_len:		payload length
 *
 * pushes nopin pdu to application daemon, 'iscsid' by calling iscsi_recv_pdu
 */
int qfle3i_process_nopin(struct qfle3i_conn *conn, struct qfle3i_cmd *cmd,
			char *data_buf, int data_len)
{
	struct iscsi_nopin *nopin_msg =
		(struct iscsi_nopin *) &conn->gen_pdu.nopin_hdr;

	vmk_IscsiTransportForwardPDU(conn->trans_conn, (void *)nopin_msg,
		       (vmk_uint8 *) data_buf, data_len);

	conn->sess->last_noopin_indicated = vmk_GetTimerCycles();
	conn->sess->noopin_indicated_count++;

	cmd->iscsi_opcode = 0;
	return 0;
}

/*
 * qfle3i_nopout_check_active_cmds - checks if iscsi link is idle
 *
 * @hba: 		pointer to adapter instance
 *
 * called to check if iscsi connection is idle or not. Pro-active nopout
 *	 is sent only if the link is idle
 */
static int qfle3i_nopout_check_active_cmds(struct qfle3i_conn *conn,
					  struct qfle3i_cmd *cmnd)
{
	struct iscsi_nopin *nopin_msg =
		(struct iscsi_nopin *) &conn->gen_pdu.resp_hdr;

	if ((conn->nopout_num_scsi_cmds == conn->num_scsi_cmd_pdus) &&
	    !conn->sess->active_cmd_count) {
		return -1;
	}

	vmk_Memset(nopin_msg, 0x00, sizeof(struct iscsi_nopin));
	nopin_msg->opcode = ISCSI_OP_NOOP_IN;
	nopin_msg->flags = ISCSI_FLAG_CMD_FINAL;
	vmk_Memcpy(nopin_msg->lun, conn->gen_pdu.nopout_hdr.lun, 8);
	nopin_msg->itt = conn->gen_pdu.nopout_hdr.itt;
	nopin_msg->ttt = ISCSI_RESERVED_TAG;
	nopin_msg->statsn = conn->gen_pdu.nopout_hdr.exp_statsn;;
	nopin_msg->exp_cmdsn = htonl(conn->sess->exp_cmdsn);
	nopin_msg->max_cmdsn = htonl(conn->sess->max_cmdsn);

	vmk_IscsiTransportForwardPDU(conn->trans_conn,
					(void *) nopin_msg,
					(vmk_uint8 *) NULL, 0);

	conn->nopout_num_scsi_cmds = conn->num_scsi_cmd_pdus;
	return 0;
}

/*
 * qfle3i_process_scsi_resp - complete SCSI command processing by calling
 *			'scsi_done', free iscsi cmd structure to free list
 *
 * @cmd:		iscsi cmd pointer
 * @resp_cqe:		scsi response cqe pointer
 *
 * validates scsi response indication for normal completion, sense data if any
 *	underflow/overflow condition and propogates SCSI response to SCSI-ML by
 *	calling scsi_done() and also returns command struct back to free pool
 */
void qfle3i_process_scsi_resp(struct qfle3i_cmd *cmd,
			    struct iscsi_cmd_response *resp_cqe)
{
	vmk_ScsiCommand *sc = cmd->scsi_cmd;
	vmk_uint16 sense_data[128];
	int data_len = 0;
	vmk_uint16 sense_len = 0;
	int res_count = 0;
	vmk_uint8 flags;
	vmk_ByteCountSmall bytes_xferred = 0;

	sc->status.host = VMK_SCSI_HOST_OK;
	sc->status.device = resp_cqe->status;
	sc->status.plugin = VMK_SCSI_PLUGIN_GOOD;

	if (resp_cqe->response != ISCSI_STATUS_CMD_COMPLETED) {
		sc->status.host = VMK_SCSI_HOST_ERROR;
		goto out;
	}

	if (resp_cqe->status) {
		data_len = resp_cqe->data_length;
		if (data_len < 2) {
			if (resp_cqe->status != VMK_SCSI_DEVICE_CHECK_CONDITION) {
				if (data_len) {
					/* treat as if no sense data was
					 * received, because it's optional
					 * in any case*/
					qfle3i_get_rq_buf(cmd->conn,
							 (char *)sense_data,
							 data_len);
					qfle3i_put_rq_buf(cmd->conn, 1);
				}
				goto out;
			}
invalid_len:
			PRINT_ERR(cmd->conn->sess->hba,
				"CHK_CONDITION - invalid data length %d\n",
				data_len);
			/*
			 * PR 570447: Returning BAD_TARGET breaks
			 * VMware Pluggable Storage Architecture (PSA)
			 * assumptions. Returning DID_OK with CC to
			 * indicate specific error.
			 */
			sc->status.host = VMK_SCSI_HOST_OK;
			sc->status.device = resp_cqe->status;

			if ((((vmk_uint8 *)&sc->senseBuf)[0] & 0x7f) == 0x70 ||
				(((vmk_uint8 *)&sc->senseBuf)[0] & 0x7f) == 0x71) {
				/* Sense buffer already set, no action required; */
			} else {
				/* Sense buffer not set, setting to CC w/ HARDWARE ERROR */
				vmk_ScsiCmdClearSenseData(sc);
				((vmk_uint8 *)&sc->senseBuf)[0] = 0x70;		// Fixed format sense data;
				((vmk_uint8 *)&sc->senseBuf)[2] = 0x4; 		// SENSE KEY = HARDWARE ERROR;
			}
			goto out;
		}

		if (data_len > QFLE3I_RQ_WQE_SIZE) {
			PRINT_ALERT(cmd->conn->sess->hba,
					"sense data len %d > RQ sz\n",
					data_len);
			data_len = QFLE3I_RQ_WQE_SIZE;
		}
		if (data_len) {
			vmk_ScsiCmdClearSenseData(sc);
			qfle3i_get_rq_buf(cmd->conn, (char *)sense_data, data_len);
			qfle3i_put_rq_buf(cmd->conn, 1);
			cmd->conn->total_data_octets_rcvd += data_len;
			sense_len = vmk_BE16ToCPU(*((/*__be16*/vmk_uint16 *) sense_data));

			if (data_len < sense_len)
				goto invalid_len;

			if (sense_len > SCSI_SENSE_BUFFERSIZE)
				sense_len = SCSI_SENSE_BUFFERSIZE;

			vmk_ScsiCmdClearSenseData(sc);
			vmk_ScsiCmdSetSenseData(((struct vmk_ScsiSenseData *)(sense_data + 1)),
								sc, (int)sense_len);
		}
	}
	/*
		Do not bother checking underrun/overflow condition when device_status
		is non-zero. Let mid-layer take an action based on device_status and
		sense_data
	*/
	else {
		flags = resp_cqe->response_flags;
		if (flags & (ISCSI_CMD_RESPONSE_RESIDUAL_UNDERFLOW |
					 ISCSI_CMD_RESPONSE_RESIDUAL_OVERFLOW))
			res_count = resp_cqe->residual_count;

		if (res_count) {
			if (res_count <= vmk_SgGetDataLen(sc->sgArray)) {
				bytes_xferred = vmk_SgGetDataLen(sc->sgArray) - res_count;
				if (bytes_xferred < sc->requiredDataLen) {
					PRINT_ERR(cmd->conn->sess->hba,
							  "Data returned 0x%x less then Required data length 0x%x\n",
							  bytes_xferred, sc->requiredDataLen);
					/* This is weird, sending error in this case, causing a problem,
					   we will investigate further with vmware folks. native does not
					   have res_id bit in scsi_cmd too
					*/
					sc->status.host = VMK_SCSI_HOST_ERROR;
				}
			}
			else {
				PRINT_ERR(cmd->conn->sess->hba,
						"iSCSI residual > data len, %x %lx\n",
						res_count, vmk_SgGetDataLen(sc->sgArray));
				/* This is weird, sending error in this case, causing a problem,
				   we will investigate further with vmware folks. native does not
				   have res_id bit in scsi_cmd too
				*/
				//sc->status.host = VMK_SCSI_HOST_ERROR;
			}
			cmd->conn->total_data_octets_rcvd -= res_count;
		}
		else
			bytes_xferred = vmk_SgGetDataLen(sc->sgArray);
	}
out:
	sc->bytesXferred = bytes_xferred;
	return;
}

/*
 * qfle3i_cpy_scsi_cdb - copies LUN & CDB fields in required format to sq wqe
 *
 * @sc: 		SCSI-ML command pointer
 * @cmd:		iscsi cmd pointer
 *
 */
static void qfle3i_cpy_scsi_cdb(vmk_ScsiCommand *sc,
				      struct qfle3i_cmd *cmd)
{
	vmk_uint32 dword;
	int lpcnt;
	vmk_uint8 *srcp;
	vmk_uint32 *dstp;
	vmk_uint32 scsi_lun[2];

	if(!cmd->app_cmd)
		qfle3i_int_to_scsilun_with_sec_lun_id(
				(vmk_uint16) cmd->ilun->lun_id,
				(vmk_uint8 *) scsi_lun,
				vmk_ScsiCmdGetSecondLevelLunId(sc));
	else
		int_to_scsilun((vmk_uint16) cmd->ilun->lun_id,
				(vmk_uint8 *) scsi_lun);

	cmd->req.lun[0] = ntohl(scsi_lun[0]);
	cmd->req.lun[1] = ntohl(scsi_lun[1]);

	lpcnt = cmd->scsi_cmd->cdbLen / sizeof(dword);
	srcp = (vmk_uint8 *) sc->cdb;
	dstp = (vmk_uint32 *) cmd->req.cdb;
	while (lpcnt--) {
		vmk_Memcpy(&dword, (const void *) srcp, 4);
		*dstp = vmk_CPUToBE32(dword);
		srcp += 4;
		dstp++;
	}
	if (sc->cdbLen & 0x3) {
		dword = (vmk_uint32) srcp[0] | ((vmk_uint32) srcp[1] << 8);
		*dstp = vmk_CPUToBE32(dword);
	}
}

/*
 * qfle3i_get_write_cmd_bd_idx - identifies various BD bookmarks for a
 *			scsi write command
 *
 * @cmd:		iscsi cmd struct pointer
 * @buf_off:		absolute buffer offset
 * @start_bd_off:	vmk_uint32 pointer to return the offset within the BD
 *			indicated by 'start_bd_idx' on which 'buf_off' falls
 * @start_bd_idx:	index of the BD on which 'buf_off' falls
 *
 * identifies & marks various bd info for imm data, unsolicited data
 *	and the first solicited data seq.
 */
static void qfle3i_get_write_cmd_bd_idx(struct qfle3i_cmd *cmd, vmk_uint32 buf_off,
				       vmk_uint32 *start_bd_off, vmk_uint32 *start_bd_idx)
{
	vmk_uint32 cur_offset = 0;
	vmk_uint32 cur_bd_idx = 0;
	struct iscsi_bd *bd_tbl;

	if (!cmd->bd_tbl || !cmd->bd_tbl->bd_tbl)
		return;

	bd_tbl = cmd->bd_tbl->bd_tbl;
	if (buf_off) {
		while (buf_off >= (cur_offset + bd_tbl->buffer_length)) {
			cur_offset += bd_tbl->buffer_length;
			cur_bd_idx++;
			bd_tbl++;
		}
	}

	*start_bd_off = buf_off - cur_offset;
	*start_bd_idx = cur_bd_idx;
}

/*
 * qfle3i_setup_write_cmd_bd_info - sets up BD various information for
 *			scsi write command
 *
 * @cmd:		iscsi cmd struct pointer
 *
 * identifies & marks various bd info for immediate data, unsolicited data
 *	and first solicited data seq which includes BD start index & BD buf off
 *	This function takes into account iscsi parameter such as immediate data
 *	and unsolicited data is support on this connection
 *
 */
static void qfle3i_setup_write_cmd_bd_info(struct qfle3i_cmd *cmd)
{
	struct qfle3i_sess *sess;
	vmk_uint32 start_bd_offset = 0;
	vmk_uint32 start_bd_idx = 0;
	vmk_uint32 buffer_offset = 0;
	vmk_uint32 seq_len = 0;
	vmk_uint32 fbl, mrdsl;
	vmk_uint32 cmd_len = cmd->req.total_data_transfer_length;

	sess = cmd->conn->sess;

	/* if ImmediateData is turned off & IntialR2T is turned on,
	 * there will be no immediate or unsolicited data, just return.
	 */
	if (sess->initial_r2t && !sess->imm_data)
		return;

	fbl = sess->first_burst_len;
	mrdsl = cmd->conn->max_data_seg_len_xmit;

	/* Immediate data */
	if (sess->imm_data) {
		seq_len = min(mrdsl, fbl);
		seq_len = min(cmd_len, seq_len);
		buffer_offset += seq_len;
	}

	if (seq_len == cmd_len)
		return;

	if (!sess->initial_r2t) {
		if (seq_len >= fbl)
			goto r2t_data;
		seq_len = min(fbl, cmd_len) - seq_len;
		qfle3i_get_write_cmd_bd_idx(cmd, buffer_offset,
					   &start_bd_offset, &start_bd_idx);
		cmd->req.ud_buffer_offset = start_bd_offset;
		cmd->req.ud_start_bd_index = start_bd_idx;
		buffer_offset += seq_len;
	}
r2t_data:
	if (buffer_offset != cmd_len) {
		qfle3i_get_write_cmd_bd_idx(cmd, buffer_offset,
					   &start_bd_offset, &start_bd_idx);
		if (start_bd_offset > fbl) {
			int i = 0;

			PRINT_ERR(sess->hba, "error, buf offset 0x%x "
				  "bd_valid %d use_sg %d\n", buffer_offset,
				  cmd->bd_tbl->bd_valid,
				  cmd->scsi_cmd->sgArray->numElems);
			for (i = 0; i < cmd->bd_tbl->bd_valid; i++)
				PRINT_ERR(sess->hba, "err, bd[%d]: len %x\n", i,
					  cmd->bd_tbl->bd_tbl[i].buffer_length);
		}
		cmd->req.sd_buffer_offset = start_bd_offset;
		cmd->req.sd_start_bd_index = start_bd_idx;
	}
}

/*
 * qfle3i_split_bd - splits buffer > 64KB into 32KB chunks
 *
 * @cmd:		iscsi cmd struct pointer
 * @addr: 		base address of the buffer
 * @sg_len: 		buffer length
 * @bd_index: 		starting index into BD table
 *
 * This is not required as driver limits max buffer size of less than 64K by
 *	advertising 'max_sectors' within this limit. 5706/5708 hardware limits
 *	BD length to less than or equal to 0xFFFF
 **/
static int qfle3i_split_bd(struct qfle3i_cmd *cmd, vmk_uint64 addr, int sg_len,
			  int bd_index)
{
	struct iscsi_bd *bd = cmd->bd_tbl->bd_tbl;
	int frag_size, sg_frags;

	sg_frags = 0;
	while (sg_len) {
		if (sg_len > BD_SPLIT_SIZE)
			frag_size = BD_SPLIT_SIZE;
		else
			frag_size = sg_len;
		bd[bd_index + sg_frags].buffer_addr_lo = (vmk_uint32)addr & 0xffffffff;
		bd[bd_index + sg_frags].buffer_addr_hi = addr >> 32;
		bd[bd_index + sg_frags].buffer_length = frag_size;
		bd[bd_index + sg_frags].flags = 0;
		if ((bd_index + sg_frags) == 0)
			bd[0].flags = ISCSI_BD_FIRST_IN_BD_CHAIN;
		addr += (vmk_IOA)frag_size;
		sg_frags++;
		sg_len -= frag_size;
	}
	return sg_frags;
}

/*
 * qfle3i_map_sg - maps IO buffer and prepares the BD table
 *
 * @hba: 		adapter instance
 * @cmd:		iscsi cmd struct pointer
 *
 * map SG list
 */
static int qfle3i_map_sg(struct qfle3i_hba *hba, struct qfle3i_cmd *cmd)
{
	vmk_ScsiCommand *sc = cmd->scsi_cmd;
	struct iscsi_bd *bd = cmd->bd_tbl->bd_tbl;
	vmk_SgElem *cur_seg = NULL;
	int byte_count = 0;
	int sg_frags;
	int bd_count = 0;
	int sg_count;
	int sg_len;
	vmk_IOA addr;
	int i;

	sg_count = sc->sgArray->numElems;
	if (sg_count > ISCSI_MAX_SG_PER_CMD) {
		vmk_WarningMessage("sg_count is over ISCSI_MAX_SG_PER_CMD\n");
		return VMK_FAILURE;
	}

//	scsi_for_each_sg(sc, sg, sg_count, i) {
	for (i = 0; i < sg_count; i++) {
		cur_seg = &sc->sgArray->elem[i];
		sg_len = cur_seg->length;
		addr = cur_seg->ioAddr;

		if (sg_len > MAX_BD_LENGTH)
			sg_frags = qfle3i_split_bd(cmd, addr, sg_len,
								bd_count);
		else {
			sg_frags = 1;
			bd[bd_count].buffer_addr_lo = (vmk_uint32)addr & 0xffffffff;
			bd[bd_count].buffer_addr_hi = addr >> 32;
			bd[bd_count].buffer_length = sg_len;
			bd[bd_count].flags = 0;
			if ((bd_count) == 0)
				bd[bd_count].flags = ISCSI_BD_FIRST_IN_BD_CHAIN;
		}
		byte_count += sg_len;
		bd_count += sg_frags;
	}
	/* do reset - for SG_VMK type */
	bd[bd_count - 1].flags |= ISCSI_BD_LAST_IN_BD_CHAIN;

	if (byte_count != vmk_SgGetDataLen(sc->sgArray)) {
		vmk_WarningMessage("byte_count != vmk_SgGetDataLen(sc->sgArray. \n");
	}
	return bd_count;
}



/*
 * qfle3i_iscsi_map_sg_list - maps SG list
 *
 * @cmd:		iscsi cmd struct pointer
 *
 * creates BD list table for the command
 */
static void qfle3i_iscsi_map_sg_list(struct qfle3i_hba *hba, struct qfle3i_cmd *cmd)
{
	vmk_ScsiCommand *sc = cmd->scsi_cmd;
	int bd_count;

	if (sc->sgArray->numElems)
		bd_count = qfle3i_map_sg(hba, cmd);
	else {
		struct iscsi_bd *bd = cmd->bd_tbl->bd_tbl;
		bd_count  = 0;
		bd[0].buffer_addr_lo = bd[0].buffer_addr_hi = 0;
		bd[0].buffer_length = bd[0].flags = 0;
	}
	cmd->bd_tbl->bd_valid = bd_count;
}

/*
 * qfle3i_iscsi_unmap_sg_list - unmaps SG list
 *
 * @cmd:		iscsi cmd struct pointer
 *
 * unmap IO buffers and invalidate the BD table
 */
void qfle3i_iscsi_unmap_sg_list(struct qfle3i_hba *hba, struct qfle3i_cmd *cmd)
{
	vmk_ScsiCommand *sc = cmd->scsi_cmd;

	if (cmd->bd_tbl->bd_valid && sc) {
		cmd->bd_tbl->bd_valid = 0;
	}
}

/*
 * qfle3i_iscsi_prep_generic_pdu_bd - prepares BD table to be used with
 *			generic iscsi pdus
 *
 * @conn: 		iscsi connection pointer
 *
 * Allocates buffers and BD tables before shipping requests to cnic
 *	for PDUs prepared by 'iscsid' daemon
 */
static void qfle3i_iscsi_prep_generic_pdu_bd(struct qfle3i_conn *conn)
{
	struct iscsi_bd *bd_tbl;

	bd_tbl = (struct iscsi_bd *) conn->gen_pdu.login_req.pgtbl;

	bd_tbl->buffer_addr_hi =
		(vmk_uint32) ((vmk_uint64) conn->gen_pdu.login_req.mapping >> 32);
	bd_tbl->buffer_addr_lo = (vmk_uint32) conn->gen_pdu.login_req.mapping;
	bd_tbl->buffer_length = conn->gen_pdu.req_wr_ptr -
				conn->gen_pdu.req_buf;
	bd_tbl->reserved0 = 0;
	bd_tbl->flags = ISCSI_BD_LAST_IN_BD_CHAIN |
			ISCSI_BD_FIRST_IN_BD_CHAIN;

	bd_tbl = (struct iscsi_bd  *) conn->gen_pdu.login_resp.pgtbl;
	bd_tbl->buffer_addr_hi = (vmk_uint64) conn->gen_pdu.login_resp.mapping >> 32;
	bd_tbl->buffer_addr_lo = (vmk_uint32) conn->gen_pdu.login_resp.mapping;
	bd_tbl->buffer_length = ISCSI_CONN_LOGIN_BUF_SIZE;
	bd_tbl->reserved0 = 0;
	bd_tbl->flags = ISCSI_BD_LAST_IN_BD_CHAIN |
			ISCSI_BD_FIRST_IN_BD_CHAIN;
}


/*
 * qfle3i_iscsi_send_generic_request - called to send iscsi login/nopout/logout
 *			pdus
 *
 * @hba: 		pointer to adapter instance
 *
 * called to transmit PDUs prepared by the 'iscsid' daemon. iSCSI login,
 *	Nop-out and Logout requests flow through this path.
 */
static int qfle3i_iscsi_send_generic_request(struct qfle3i_cmd *cmnd)
{
	int rc = 0;
	struct qfle3i_conn *conn = cmnd->conn;

	qfle3i_iscsi_prep_generic_pdu_bd(conn);
	switch (cmnd->iscsi_opcode & ISCSI_OPCODE_MASK) {
	case ISCSI_OP_LOGIN:
		qfle3i_send_iscsi_login(conn, cmnd);
		break;

	case ISCSI_OP_NOOP_OUT:
		if (!qfle3i_nopout_when_cmds_active)
			if (!qfle3i_nopout_check_active_cmds(conn, cmnd)) {
				return 0;
			}

		conn->nopout_num_scsi_cmds = conn->num_scsi_cmd_pdus;
		rc = qfle3i_send_iscsi_nopout(conn, cmnd, NULL, 0);
		break;

	case ISCSI_OP_LOGOUT:
		rc = qfle3i_send_iscsi_logout(conn, cmnd);
		break;

	default:
		PRINT_ALERT(conn->sess->hba, "send_gen: unsupported op 0x%x\n",
				   cmnd->iscsi_opcode);
	}
	return rc;
}

int qfle3i_cqe_work_pending(struct qfle3i_conn *conn)
{
	struct qp_info *qp;
	volatile struct iscsi_nop_in_msg *nopin;
	int exp_seq_no;

	qp = &conn->ep->qp;
	nopin = (struct iscsi_nop_in_msg *)qp->cq_cons_qe;

	exp_seq_no = conn->ep->qp.cqe_exp_seq_sn;
	if (exp_seq_no > qp->cqe_size * 2)
		exp_seq_no -= qp->cqe_size * 2;

	if (nopin->cq_req_sn ==  exp_seq_no) {
		return 1;
	} else
		return 0;
}

static void qfle3i_process_control_pdu(struct qfle3i_sess *sess)
{
	qfle3i_iscsilun *ilun;
	int num_cmds;
	vmk_uint8 tmf_func;

	vmk_SpinlockLock(sess->lock);

	if (vmk_AtomicRead64(&sess->tmf_pending)) {
		tmf_func = sess->scsi_tmf_cmd->tmf_func;

		QFLE3I_DBG(DBG_TMF, sess->hba,
				"TMF pending. tmf_func=%d \n",
				tmf_func);
		if (tmf_func == ISCSI_TM_FUNC_LOGICAL_UNIT_RESET) {
			ilun = sess->scsi_tmf_cmd->ilun;
			vmk_SpinlockUnlock(sess->lock);
			num_cmds = qfle3i_flush_pend_queue(sess, ilun,
						  VMK_SCSI_HOST_RESET);
			vmk_SpinlockLock(sess->lock);
		} else if (tmf_func == ISCSI_TM_FUNC_TARGET_WARM_RESET) {
			vmk_SpinlockUnlock(sess->lock);
			num_cmds = qfle3i_flush_pend_queue(sess, NULL,
						  VMK_SCSI_HOST_RESET);
			vmk_SpinlockLock(sess->lock);
		}
		qfle3i_send_iscsi_tmf(sess->lead_conn, sess->scsi_tmf_cmd);
		vmk_AtomicWrite64(&sess->tmf_pending, 0);
	}

	if (vmk_AtomicRead64(&sess->nop_resp_pending)) {
		qfle3i_iscsi_send_generic_request(sess->nopout_resp_cmd);
		vmk_AtomicWrite64(&sess->nop_resp_pending, 0);
	}

	if (vmk_AtomicRead64(&sess->login_noop_pending)) {
		QFLE3I_DBG(DBG_CONN_SETUP, sess->hba,
					"login noop pending.\n");
		qfle3i_iscsi_send_generic_request(sess->login_nopout_cmd);
		vmk_AtomicWrite64(&sess->login_noop_pending, 0);
	}

	/* flush pending SCSI cmds before transmitting logout request */
        if (vmk_AtomicRead64(&sess->logout_pending) &&
	    vmk_ListIsEmpty(&sess->pend_cmd_list)) {
		PRINT(sess->hba, "logout pending on ep_iscsi_cid:0x%x\n",
			sess->lead_conn->ep->ep_iscsi_cid);
		qfle3i_iscsi_send_generic_request(sess->login_nopout_cmd);
		vmk_AtomicWrite64(&sess->logout_pending, 0);
	}
	vmk_SpinlockUnlock(sess->lock);
}

static int qfle3i_conn_transmits_pending(struct qfle3i_conn *conn)
{
	struct qfle3i_sess *sess = conn->sess;
	extern int qfle3i_chip_cmd_max;

	/* If TCP connection is not active or in FFP (connection parameters updated)
	 * then do not transmit anything
	 */
	if (conn->ep && !(conn->ep->state & (EP_STATE_ULP_UPDATE_COMPL |
	    EP_STATE_CONNECT_COMPL)))
		return 0;

	if (sess->recovery_state ||
	   vmk_BitVectorTest(sess->hba->adapter_state, ADAPTER_STATE_LINK_DOWN) ||
		vmk_ListIsEmpty(&sess->pend_cmd_list))
		return 0;

	if (vmk_BitVectorTest(sess->hba->cnic_dev_type, QFLE3I_NX2_DEV_57710))
		return 8;

	if (sess->active_cmd_count < qfle3i_chip_cmd_max)
		return (qfle3i_chip_cmd_max - sess->active_cmd_count);
	else
		return 0;
}

int qfle3i_flush_pend_queue(struct qfle3i_sess *sess,
				   qfle3i_iscsilun *ilun, int reason)
{

	int num_pend_cmds_returned = 0;
	struct qfle3i_scsi_task *scsi_task, *scsi_task_n;

	vmk_SpinlockLock(sess->lock);
	ql_vmk_list_each_entry_safe(scsi_task, scsi_task_n, &sess->pend_cmd_list,
								link, struct qfle3i_scsi_task) {
		/* cmd queue flush request could be due to LUN RESET or
		 * the session recovery. In former case just fail only the
		 * command belonging that particular LUN.
		 */
		if (ilun) {
			if (scsi_task->ilun->lun_id
				   != ilun->lun_id)
				continue;
		}

		num_pend_cmds_returned++;
		qfle3i_return_failed_command(sess, scsi_task->scsi_cmd,
			vmk_SgGetDataLen(scsi_task->scsi_cmd->sgArray),
			reason);
		qfle3i_free_scsi_task(sess, scsi_task);
	}
	QFLE3I_DBG(DBG_ITT_CLEANUP, sess->hba, "sess %p, cleaned %d out "
		  "of %d commands from the pend queue\n",
		  sess, num_pend_cmds_returned, sess->pend_cmd_count);

	sess->pend_cmd_count -= num_pend_cmds_returned;
	vmk_SpinlockUnlock(sess->lock);
	return num_pend_cmds_returned;
}

void qfle3i_cleanup_task_context(struct qfle3i_sess *sess,
					struct qfle3i_cmd *cmd, int reason)
{
	if (!cmd->scsi_cmd)
		return;

	/* cleanup on chip task context for command affected by
	 * ABORT_TASK/LUN_RESET
	 */
	cmd->failed_reason = reason;
	vmk_AtomicWrite64(&cmd->cmd_state, ISCSI_CMD_STATE_CLEANUP_PEND);
	sess->cmd_cleanup_req++;
	qfle3i_send_cmd_cleanup_req(sess->hba, cmd);
}

/*
 * qfle3i_flush_cmd_queue - flush active command queue
 *
 * @sess:		iscsi session pointer
 * @reason: 		SCSI ML error code, VMK_SCSI_HOST_BUS_BUSY
 *
 * return all commands in active queue which should already have been
 * 	cleaned up by the cnic device.
 */
int qfle3i_flush_cmd_queue(struct qfle3i_sess *sess,
				  qfle3i_iscsilun *ilun,
				  int reason, int clear_ctx)
{
	vmk_ListLinks failed_cmds;
	int cmd_cnt = 0;
	int cmd_diff_lun = 0;
	int total_sess_active_cmds = 0;
	int iscsi_cid = 0xFFFF;
	struct qfle3i_cmd *cmd, *cmd_n;

	if (sess->lead_conn && sess->lead_conn->ep)
		iscsi_cid = sess->lead_conn->ep->ep_iscsi_cid;

	vmk_ListInit(&failed_cmds);
	vmk_SpinlockLock(sess->lock);

	ql_vmk_list_each_entry_safe(cmd, cmd_n, &sess->active_cmd_list,
								link, struct qfle3i_cmd) {
		total_sess_active_cmds++;

		if (!cmd->scsi_cmd) {
			QFLE3I_DBG(DBG_ITT_CLEANUP, sess->hba, "cid %d, flush q,"
					  " cmd %p is not associated with any"
					  " scsi cmd\n", iscsi_cid, cmd);
			continue;
		}
		/* cmd queue flush request could be due to LUN RESET or
		 * the session recovery. In former case just fail only the
		 * command belonging that particular LUN.
		 */

		if (ilun) {
			if (cmd->ilun->lun_id !=
				   ilun->lun_id) {
				cmd_diff_lun++;
				continue;
			}
		}

		if (vmk_AtomicRead64(&cmd->cmd_state) == ISCSI_CMD_STATE_CMPL_RCVD){
			/* completion pdu is being processed and we will let
			 * it run to completion, fail the request here
			 */
			QFLE3I_DBG(DBG_ITT_CLEANUP, sess->hba,
				  "iscsi cid %d, completion & TMF cleanup "
				  "are running in parallel, cmd %p\n",
				  iscsi_cid, cmd);
			continue;
		}

		cmd->scsi_cmd->status.host = reason;
		/* Now that qfle3i_cleanup_task_context() does not sleep waiting
		 * for completion it is safe to hold sess lock and this will
		 * avoid race between LUN/TARGET RESET TMF completion followed
		 * by command completion with check condition
		 */

		if (clear_ctx) {
			vmk_AtomicWrite64(&cmd->cmd_state, ISCSI_CMD_STATE_CLEANUP_START);
			qfle3i_cleanup_task_context(sess, cmd, reason);
		} else {
			vmk_ListRemove(&cmd->link);
			vmk_ListInit(&cmd->link);
			vmk_ListInsert(&cmd->link, vmk_ListAtRear(&failed_cmds));
		}
		cmd_cnt++;
	}
	vmk_SpinlockUnlock(sess->lock);

	ql_vmk_list_each_entry_safe(cmd, cmd_n, &failed_cmds,
								link, struct qfle3i_cmd) {
		cmd->failed_reason = reason;
		qfle3i_fail_cmd(sess, cmd);
	}

	if (ilun)
		QFLE3I_DBG(DBG_ITT_CLEANUP, sess->hba, "sess %p, "
			  "total active %d, total cleaned %d, skipped %d\n",
			  sess, total_sess_active_cmds, cmd_cnt, cmd_diff_lun);
	else
		QFLE3I_DBG(DBG_ITT_CLEANUP, sess->hba, "qfle3i: sess %p,"
			" total active %d, total cleaned %d\n",
			sess, total_sess_active_cmds, cmd_cnt);

	QFLE3I_DBG(DBG_ITT_CLEANUP, sess->hba, "sess %p active %d, pend %d\n",
			sess, sess->pend_cmd_count, sess->active_cmd_count);

	return cmd_cnt;
}

#define QFLE3I_SERIAL_32 2147483648UL

static int iscsi_cmd_win_closed(struct qfle3i_sess *sess)
{
	vmk_uint32 cmdsn = sess->cmdsn;
	vmk_uint32 maxsn = sess->max_cmdsn;

	return ((cmdsn < maxsn && (maxsn - cmdsn > QFLE3I_SERIAL_32)) ||
		 (cmdsn > maxsn && (cmdsn - maxsn < QFLE3I_SERIAL_32)));
}

static int qfle3i_process_pend_queue(struct qfle3i_sess *sess)
{
	struct qfle3i_cmd *cmd;
	struct qfle3i_conn *conn;
	struct qfle3i_scsi_task *scsi_task;
	struct qfle3i_scsi_task *tmp;
	int xmits_per_work;
	int cmds_sent = 0;
	int rc = 0;

	xmits_per_work = qfle3i_conn_transmits_pending(sess->lead_conn);
	if (!xmits_per_work)
		return VMK_STATUS_PENDING;

	conn = sess->lead_conn;
	vmk_SpinlockLock(sess->lock);

	ql_vmk_list_each_entry_safe(scsi_task, tmp, &sess->pend_cmd_list,
								link, struct qfle3i_scsi_task) {
		/* do not post any SCSI CMDS while TMF is active */
		if (iscsi_cmd_win_closed(sess)) {
			sess->cmd_win_closed++;
			rc = VMK_STATUS_PENDING;
			break;
		}

		if (conn->ep && ((conn->ep->state == EP_STATE_TCP_FIN_RCVD) ||
		   (conn->ep->state == EP_STATE_TCP_RST_RCVD))) {
			rc = VMK_STATUS_PENDING;
			break;
		}

		cmd = qfle3i_alloc_cmd(sess);
		if (cmd == NULL) {
			rc = VMK_STATUS_PENDING;
			break;
		}

		cmd->scsi_cmd = scsi_task->scsi_cmd;
		cmd->ilun = scsi_task->ilun;
		cmd->app_cmd = scsi_task->app_cmd;
		sess->pend_cmd_count--;
		qfle3i_free_scsi_task(sess, scsi_task);

		cmd->conn = sess->lead_conn;
		qfle3i_xmit_work_send_cmd(sess->lead_conn, cmd);
		cmds_sent++;
		sess->total_cmds_sent++;
		if (cmds_sent >= xmits_per_work)
			break;
	}
	vmk_SpinlockUnlock(sess->lock);

	return rc;
}

static void qfle3i_xmit_work_send_cmd(struct qfle3i_conn *conn, struct qfle3i_cmd *cmd)
{
	vmk_ScsiCommand *sc = cmd->scsi_cmd;
	struct qfle3i_hba *hba  = conn->ep->hba;
	struct qfle3i_sess *sess  = conn->sess;
	struct iscsi_cmd_request *req = &cmd->req;

	cmd->req.total_data_transfer_length = vmk_SgGetDataLen(sc->sgArray);
	cmd->iscsi_opcode = cmd->req.op_code = ISCSI_OP_SCSI_CMD;
	cmd->req.cmd_sn = sess->cmdsn++;

	qfle3i_iscsi_map_sg_list(hba, cmd);
	qfle3i_cpy_scsi_cdb(sc, cmd);

	req->op_attr = ISCSI_ATTR_SIMPLE;
	if (sc->dataDirection == VMK_SCSI_COMMAND_DIRECTION_WRITE) {
		req->op_attr |= ISCSI_CMD_REQUEST_WRITE;
		req->itt |= (ISCSI_TASK_TYPE_WRITE <<
				 ISCSI_CMD_REQUEST_TYPE_SHIFT);
		qfle3i_setup_write_cmd_bd_info(cmd);
	} else {
		if (vmk_SgGetDataLen(sc->sgArray))
			req->op_attr |= ISCSI_CMD_REQUEST_READ;
		req->itt |= (ISCSI_TASK_TYPE_READ <<
				 ISCSI_CMD_REQUEST_TYPE_SHIFT);
	}
	req->num_bds = cmd->bd_tbl->bd_valid;
	if (!cmd->bd_tbl->bd_valid) {
		req->bd_list_addr_lo =
			(vmk_uint32) sess->hba->mp_dma_buf.pgtbl_map;
		req->bd_list_addr_hi =
			(vmk_uint32) ((vmk_uint64) sess->hba->mp_dma_buf.pgtbl_map >> 32);
		req->num_bds = 1;
	}
	if(cmd->bd_tbl->bd_valid >= ISCSI_MAX_BDS_PER_CMD)
		PRINT_ALERT(hba, "Number of BDs = 0x%x\n", cmd->bd_tbl->bd_valid);

	vmk_AtomicWrite64(&cmd->cmd_state, ISCSI_CMD_STATE_INITIATED);
// VK: find out how to deal with this.
//	sc->SCp.ptr = (char *) cmd;

	if (req->itt != ITT_INVALID_SIGNATURE) {
		vmk_ListInsert(&cmd->link, vmk_ListAtRear(&sess->active_cmd_list));
		sess->active_cmd_count++;

		QFLE3I_DBG(DBG_IO, sess->hba, "IO REQ, iscsi_conn_cid:0x%x scsi_cmd:%p"
				" lun:%d cmdID:0x%lx opcode:0x%x op_attr:0x%x itt:0x%x"
				" num_bds:%d xer_len:0x%x activecc:%d\n",
				conn->iscsi_conn_cid, cmd->scsi_cmd, cmd->ilun->lun_id,
				cmd->scsi_cmd->cmdId.serialNumber, cmd->scsi_cmd->cdb[0],
				req->op_attr, req->itt, req->num_bds,
				req->total_data_transfer_length, sess->active_cmd_count);
		qfle3i_send_iscsi_scsicmd(conn, cmd);
	}
}

static void qfle3i_conn_main_worker(unsigned long data)
{
	struct qfle3i_sess *sess;
	int cqe_pending;
	int defer_pendq = 0;
	struct qfle3i_conn *conn = (struct qfle3i_conn *)data;

	if (!vmk_AtomicRead64(&conn->worker_enabled)) {
		PRINT_ERR(conn->sess->hba,
			  "working scheduled while disabled\n");
		return;
	}

	vmk_AtomicWrite64(&conn->conn_tasklet.is_running, 1);
	conn->tasklet_entry++;
	sess = conn->sess;
	sess->timestamp = vmk_GetTimerCycles();
	conn->tasklet_loop = 0;

	do {
		qfle3i_process_control_pdu(sess);
		defer_pendq = qfle3i_process_pend_queue(sess);

		if (use_poll_timer)
			cqe_pending = qfle3i_process_new_cqes(conn, 0,
						conn->ep->qp.cqe_size);
		else
			cqe_pending = qfle3i_process_new_cqes(conn, 0,
						cmd_cmpl_per_work);

		defer_pendq = qfle3i_process_pend_queue(sess);

		if (vmk_GetTimerCycles() < sess->timestamp){
			QFLE3I_DBG(DBG_IO, conn->sess->hba,
				  "jiffies wrap-around, iscsi_conn_cid:0x%x\n",
				  conn->iscsi_conn_cid);
			break;
		}

		/* VK: check if we need to multiply time_slice here */
		if (vmk_GetTimerCycles() >
		   (sess->timestamp +
		   (sess->worker_time_slice * vmk_TimerCyclesPerSecond()))) {
			QFLE3I_DBG(DBG_IO, conn->sess->hba,
				  "busy loop more than %ld second, iscsi_conn_cid:0x%x\n",
				  sess->worker_time_slice, conn->iscsi_conn_cid);

			conn->tasklet_timeslice_exit++;
			break;
		}
	} while (cqe_pending && !defer_pendq);


#if 0
	if (defer_pendq  == VMK_STATUS_PENDING) {
		QFLE3I_DBG(DBG_IO, conn->sess->hba,
				  "going to tasklet_exit, iscsi_conn_cid:0x%x\n",
				  conn->iscsi_conn_cid);
		goto tasklet_exit;
	}

	/*
		We dont need this code, scheduling/force wakeup of world
		from already running world is a problem.
		This is a fix for NMI issue.
	*/

	if (qfle3i_cqe_work_pending(conn) ||
	   !vmk_ListIsEmpty(&sess->pend_cmd_list)) {
        	if (vmk_AtomicRead64(&conn->worker_enabled)) {
				vmk_AtomicWrite64(&conn->lastSched,10);
				QFLE3I_DBG(DBG_IO, conn->sess->hba,
				  "scheduling conn tasklet thru tasklet, "
				  "iscsi_conn_cid:0x%x \n", conn->iscsi_conn_cid);
				ql_vmk_tasklet_schedule(&conn->conn_tasklet);
		}
	}

tasklet_exit:
#endif
	QFLE3I_DBG(DBG_IO, conn->sess->hba,
				  "tasklet_exit, iscsi_conn_cid:0x%x\n",
				  conn->iscsi_conn_cid);

	vmk_SpinlockLock(qfle3i_resc_lock);
	if (conn->ep)
		qfle3i_arm_cq_event_coalescing(conn->ep, CNIC_ARM_CQE);
	vmk_SpinlockUnlock(qfle3i_resc_lock);

	vmk_AtomicWrite64(&conn->conn_tasklet.is_running, 0);
}

static void qfle3i_conn_poll(unsigned long data)
{
	struct qfle3i_conn *conn = (struct qfle3i_conn *) data;

	if (!vmk_AtomicRead64(&conn->worker_enabled))
		return;

	if (qfle3i_cqe_work_pending(conn) ||
	    !vmk_ListIsEmpty(&conn->sess->pend_cmd_list) ||
		vmk_AtomicRead64(&conn->sess->login_noop_pending) ||
		vmk_AtomicRead64(&conn->sess->tmf_pending) ||
		vmk_AtomicRead64(&conn->sess->logout_pending) ||
		vmk_AtomicRead64(&conn->sess->nop_resp_pending)) {

		QFLE3I_DBG(DBG_INTERNAL, conn->sess->hba,
				"scheduling conn tasklet, iscsi_conn_cid:0x%x\n",
				conn->iscsi_conn_cid);
		ql_vmk_tasklet_schedule(&conn->conn_tasklet);
	}
}

