/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include <vmkapi.h>
#include "qfle3i.h"
#include "qfle3i_vmk.h"

extern struct qfle3i_driver_info_t qfle3i_driver_info;
/*
 *****************************************************************************
 *
 *  qfle3i_map_pci_mem --
 *
 *     Setup mapping of specified BAR and memory regions
 *
 *  Parameters:
 *     adapter: Pointer to driver qfle3i_hba structure
 *     bar: PCI bar value
 *
 *  Results:
 *     VMK_ReturnStatus
 *
 *  Side effects:
 *     None
 *
 *****************************************************************************
 */

VMK_ReturnStatus
qfle3i_map_pci_mem(vmk_PCIDevice pcidev,
                  int bar, vmk_VA *vaddr)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_VA vaAddr;

	status = vmk_PCIMapIOResource(vmk_ModuleCurrentID, pcidev,
                                 bar, NULL, &vaAddr);
	if (status != VMK_OK) {
		vmk_WarningMessage("failed to map PCI resource (%s)\n",
			 vmk_StatusToString(status));
		return status;
	}

	*vaddr = vaAddr;

	return VMK_OK;
}

/*
 *******************************************************************************
 *
 *   qfle3i_unmap_pci_mem --
 *
 *      Cleanup the mapping of specified BAR and memory regions
 *
 *   Parameters:
 *      adapter: Pointer to driver qfle3i_hba structure
 *      bar: PCI bar value
 *
 *   Results:
 *      VMK_ReturnStatus
 *
 *   Side effects:
 *      None
 *
 *******************************************************************************
 */

VMK_ReturnStatus
qfle3i_unmap_pci_mem(struct qfle3i_hba *adapter,
                    int bar)
{
	VMK_ReturnStatus status = VMK_OK;

	status = vmk_PCIUnmapIOResource(vmk_ModuleCurrentID, adapter->pcidev,
				   bar);
	if (status != VMK_OK) {
		vmk_WarningMessage("failed to unmap PCI resource (%s)\n",
			vmk_StatusToString(status));
		return status;
	}

	return VMK_OK;
}

void *ql_vmk_alloc(vmk_uint32 size)
{
	void *data;

	data = vmk_HeapAlign(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
			size, VMK_PAGE_SIZE);

	if (data != NULL)
		vmk_Memset(data, 0, size);

	return data;
}

inline void ql_vmk_free(void *buffer)
{
	if (buffer)
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), buffer);
}

VMK_ReturnStatus ql_vmk_lock_domain_create(vmk_LockDomainID *domain)
{
	VMK_ReturnStatus vmk_stat;
	vmk_Name module_name;

	vmk_stat = vmk_NameInitialize(&module_name, QFLE3I_DRIVER_NAME);
	if (vmk_stat != VMK_OK) {
		ql_vmk_pr_err("%s failed", __func__);
		goto lock_dom_fail;
	}

	vmk_stat = vmk_LockDomainCreate(vmk_ModuleCurrentID,
		vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
		&module_name,
		domain);
	if (vmk_stat != VMK_OK) {
		ql_vmk_pr_err("%s failed", __func__);
		goto lock_dom_fail;
	}

lock_dom_fail:
	return vmk_stat;
}

VMK_ReturnStatus ql_vmk_spin_lock_init(
				vmk_Lock *lock,
				ql_vmk_lock_rank lock_rank,
				const char *name)
{
	vmk_SpinlockCreateProps spinlock_props;
	VMK_ReturnStatus vmk_stat;
	vmk_Name module_name;

	vmk_Memset(&spinlock_props, 0, sizeof(spinlock_props));
	spinlock_props.moduleID = vmk_ModuleCurrentID;
	spinlock_props.heapID = vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	spinlock_props.type = VMK_SPINLOCK;
	spinlock_props.domain = qfle3i_driver_info.lock_domain;
	spinlock_props.rank = lock_rank;

	vmk_stat = vmk_NameInitialize(&module_name, name);
	if (vmk_stat != VMK_OK) {
		ql_vmk_pr_err("%s failed", __func__);
		goto spin_lock_fail;
	}
	spinlock_props.name = module_name;

	vmk_stat = vmk_SpinlockCreate(&spinlock_props, lock);

	if (vmk_stat != VMK_OK) {
		ql_vmk_pr_err("%s failed to create lock", __func__);
		goto spin_lock_fail;
	}

spin_lock_fail:
	return vmk_stat;
}

int
ql_vmk_dma_map(struct qfle3i_hba *hba, void *virtaddr, vmk_IOA *physaddr,
	vmk_uint32 size)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_DMAMapErrorInfo err;
	struct ql_vmk_dma_meta *dma_meta = NULL;
	vmk_SgArray *MachSg;

	if (virtaddr == NULL) {
		goto sg_create;
	}

	/* Allocate and initialize a Input sg array with machine-addresses */
	status = vmk_SgAllocWithInit(hba->SgOps_handle,
		&MachSg, virtaddr, size);

	if (status != VMK_OK)
		goto sg_create;

	/* Map the sg array with the DMA Engine of the device */
	status = vmk_DMAMapSg(hba->dmaEngine,
		VMK_DMA_DIRECTION_BIDIRECTIONAL,
		hba->SgOps_handle,
		MachSg, &MachSg, &err);

	if (status != VMK_OK) {
		ql_vmk_pr_err("failed to map %p size %d to IO address, %s",
			virtaddr, size, vmk_DMAMapErrorReasonToString(err.reason));
		goto sg_alloc;
	}

	/* Point to metatdata area and populate */
	dma_meta = virtaddr + size - sizeof(struct ql_vmk_dma_meta);
	dma_meta->MachSg = MachSg;

	/* Return physaddr with buffer area to be used by caller */
	*physaddr = MachSg->elem[0].ioAddr;

	return status;

sg_alloc:
	vmk_SgFree(hba->SgOps_handle, MachSg);
sg_create:
	ql_vmk_pr_err("Unable to map DMA buffer, %s", vmk_StatusToString(status));

	return status;
}


void
ql_vmk_dma_unmap(struct qfle3i_hba *hba, void *virtaddr, vmk_uint32 size)
{
	struct ql_vmk_dma_meta *dma_meta = NULL;
	vmk_uint32 internal_size;

	if (virtaddr == NULL)
		return;

	internal_size = size + sizeof(struct ql_vmk_dma_meta);
	internal_size = (internal_size + (VMK_PAGE_SIZE-1)) & (~VMK_PAGE_MASK);

	/* Point to metadata area */
	dma_meta = virtaddr + internal_size - sizeof(struct ql_vmk_dma_meta);

	vmk_DMAUnmapSg(hba->dmaEngine, VMK_DMA_DIRECTION_BIDIRECTIONAL,
	    hba->SgOps_handle, dma_meta->MachSg);
	vmk_SgFree(hba->SgOps_handle, dma_meta->MachSg);
}

void *
ql_vmk_dma_alloc(void *addr, vmk_IOA *physaddr, vmk_uint32 size)
{
	struct qfle3i_hba *hba = (struct qfle3i_hba *)addr;
	void *virtaddr = NULL;
	vmk_uint32 internal_size;

	/* Include metadata region in alloc size */
	internal_size = size + sizeof(struct ql_vmk_dma_meta);

	/* Page size aligned */
	internal_size = (internal_size + (VMK_PAGE_SIZE-1)) & (~VMK_PAGE_MASK);

	/*SV:*/
	//virtaddr = ql_vmk_alloc(internal_size);
	if (internal_size > VMK_PAGE_SIZE)
		virtaddr = vmk_HeapAlign(qfle3i_driver_info.heap_id_dma,
				internal_size, VMK_PAGE_SIZE);
	else
		virtaddr = vmk_HeapAlloc(qfle3i_driver_info.heap_id_dma, internal_size);
	/*SV:*/

	if (virtaddr == NULL)
		return NULL;
	vmk_Memset(virtaddr, 0, internal_size);

	if (ql_vmk_dma_map(hba, virtaddr, physaddr, internal_size) != 0) {
		vmk_HeapFree(qfle3i_driver_info.heap_id_dma, virtaddr);
		virtaddr = NULL;
	}

	return virtaddr;
}

void
ql_vmk_dma_free(void *addr, void *virtaddr, vmk_uint32 size)
{
	struct qfle3i_hba *hba = (struct qfle3i_hba *)addr;
	if (virtaddr == NULL)
		return;

	ql_vmk_dma_unmap(hba, virtaddr, size);
	//ql_vmk_free(virtaddr);
	//SV:
	if (virtaddr)
		vmk_HeapFree(qfle3i_driver_info.heap_id_dma, virtaddr);
}

void *
ql_vmk_dma_slab_alloc(struct qfle3i_hba *hba, vmk_SlabID slab_id, vmk_IOA *buffer, vmk_uint32 size)
{
	void *map = NULL;

	map = vmk_SlabAlloc(slab_id);
	if (map == NULL) {
		return NULL;
	}

	if (ql_vmk_dma_map(hba, map, buffer, size) != 0) {
		vmk_SlabFree(slab_id, map);
		map = NULL;
	}

	return map;
}

void
ql_vmk_dma_slab_free(struct qfle3i_hba *hba, vmk_SlabID slab_id, void *virtaddr, vmk_uint32 size)
{
	ql_vmk_dma_unmap(hba, virtaddr, size);
	vmk_SlabFree(slab_id, virtaddr);
}

/* Workqueue functions */
VMK_ReturnStatus
ql_vmk_do_singlethread_work(void *data)
{
	ql_vmk_singlethread_workqueue_t *wq = (ql_vmk_singlethread_workqueue_t *)data;

	while (vmk_WorldWait(VMK_EVENT_NONE, VMK_LOCK_INVALID,
			VMK_TIMEOUT_UNLIMITED_MS, "wq thread sleeping") == VMK_OK ) {
		ql_vmk_singlethread_workitem_t *item;

		vmk_SpinlockLock(wq->lock);
		/* there could be spurious wakeups */
		while (!vmk_ListIsEmpty(&wq->work_items)) {
			item = VMK_LIST_ENTRY(vmk_ListFirst(&wq->work_items),
				ql_vmk_singlethread_workitem_t, links);
			vmk_ListRemove(&item->links);
			vmk_ListInitElement(&item->links);
			item->workitem_state = INACTIVE;
			vmk_SpinlockUnlock(wq->lock);
			item->handler(item->data);
			vmk_SpinlockLock(wq->lock);
		}
		vmk_SpinlockUnlock(wq->lock);
	}
	return VMK_OK;
}

void
ql_vmk_init_singlethread_workitem(ql_vmk_singlethread_workitem_t *item,
	ql_vmk_singlethread_workhandler_t handler, void *data)
{
	vmk_ListInitElement(&item->links);
	item->handler = handler;
	item->data = data;
	item->workitem_state = INACTIVE;
}

void ql_vmk_singlethread_queue_work(ql_vmk_singlethread_workqueue_t *wq,
	ql_vmk_singlethread_workitem_t *item)
{
	if (!wq || !item)
		return;

	vmk_SpinlockLock(wq->lock);
	if (item->workitem_state == INACTIVE ||
	    item->workitem_state == DELAY_ACTIVE ) {
		if (vmk_ListIsUnlinkedElement(&item->links))
			vmk_ListInsert(&item->links, vmk_ListAtRear(&wq->work_items));
		else
			ql_vmk_pr_err("Work item was on list but not active!!\n");
		item->workitem_state = ACTIVE;
	}
	vmk_SpinlockUnlock(wq->lock);

	if (vmk_WorldForceWakeup(wq->world_id) != VMK_OK)
		ql_vmk_pr_err("Could not wake up %s", wq->name);
}

ql_vmk_singlethread_workqueue_t *
ql_vmk_create_singlethread_workqueue(const char *name, vmk_WorldSchedClass prio)
{
	ql_vmk_singlethread_workqueue_t *wq = NULL;
	VMK_ReturnStatus status;
	vmk_WorldProps wq_props;

	wq = vmk_HeapAlign(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
						sizeof(*wq), VMK_L1_CACHELINE_SIZE);
	if (wq == NULL) {
		ql_vmk_pr_err("Unable to allocate WQ");
		return wq;
	}

	vmk_ListInit(&wq->work_items);

	vmk_StringCopy(wq->name, name, sizeof(wq->name));

	status = ql_vmk_spin_lock_init(&wq->lock, LOCK_RANK_HIGHEST, name);

	if (status != VMK_OK) {
		ql_vmk_pr_err("Unable to create sp lock for WQ thread!");
		goto wq_failed;
	}

	vmk_Memset(&wq_props, 0, sizeof(vmk_WorldProps));
	wq_props.heapID = qfle3i_driver_info.heap_id;
	wq_props.moduleID = vmk_ModuleCurrentID;
	wq_props.startFunction = (void *)ql_vmk_do_singlethread_work;
	wq_props.data = (void *)wq;
	wq_props.schedClass = prio;
	wq_props.name = wq->name;
	status = vmk_WorldCreate(&wq_props, &wq->world_id);
	if (status != VMK_OK) {
		ql_vmk_pr_err("Unable to start WQ thread!");
		goto spinlock_failed;
	}

spinlock_failed:
	if (status != VMK_OK) {
		vmk_SpinlockDestroy(wq->lock);
	}

wq_failed:
	if (status != VMK_OK) {
		vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), wq);
		wq = NULL;
	}

	return wq;
}

void
ql_vmk_destroy_singlethread_workqueue(ql_vmk_singlethread_workqueue_t *wq)
{
	vmk_WorldDestroy(wq->world_id);
	vmk_WorldWaitForDeath(wq->world_id);
	vmk_SpinlockDestroy(wq->lock);

	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), wq);
}

VMK_ReturnStatus
ql_vmk_do_tasklet(void *data)
{
	ql_vmk_tasklet_t *tasklet = (ql_vmk_tasklet_t *)data;

	while (vmk_WorldWait(VMK_EVENT_NONE, VMK_LOCK_INVALID,
			VMK_TIMEOUT_UNLIMITED_MS, "tasklet thread sleeping") == VMK_OK ) {
			tasklet->func(tasklet->data);
	}

	return VMK_OK;
}


VMK_ReturnStatus ql_vmk_tasklet_init(ql_vmk_tasklet_t *tasklet,
	void (*func)(long unsigned int), void *data)
{
	VMK_ReturnStatus status;
	vmk_WorldProps props;
	vmk_Name name;

	status = vmk_NameInitialize(&name, QFLE3I_DRIVER_NAME);
	if (status != VMK_OK) {
		ql_vmk_pr_err("%s failed", __func__);
		goto done;
	}

	tasklet->func = (void (*)(void*))func;
	tasklet->data = data;
	vmk_AtomicWrite64(&tasklet->is_running, 0);
	vmk_StringCopy(tasklet->name, (const char *)&name, sizeof(tasklet->name));

	vmk_Memset(&props, 0, sizeof(vmk_WorldProps));
	props.heapID = qfle3i_driver_info.heap_id;
	props.moduleID = vmk_ModuleCurrentID;
	props.startFunction = (void *)ql_vmk_do_tasklet;
	props.data = (void *)tasklet;
	props.schedClass = VMK_WORLD_SCHED_CLASS_QUICK;
	props.name = tasklet->name;
	status = vmk_WorldCreate(&props, &tasklet->world_id);
	if (status != VMK_OK) {
		ql_vmk_pr_err("Unable to start WQ thread!");
	}
done:
	return status;
}

void ql_vmk_tasklet_schedule(ql_vmk_tasklet_t *tasklet)
{
	VMK_ReturnStatus status;

	if (vmk_AtomicRead64(&tasklet->is_running)) {
//		vmk_LogMessage("[%s:%d] tasklet running, no wake up:%d\n",
//					 __func__, __LINE__, tasklet->world_id);
		return;
	}

	status = vmk_WorldForceWakeup(tasklet->world_id);
	if (status != VMK_OK) {
		ql_vmk_pr_err("world wake up failed for world-id=%d. status=%s",
						tasklet->world_id,
						vmk_StatusToString(status));
		return;
	}

//	vmk_LogMessage("[%s:%d] woke up world:%d\n",
//				 __func__, __LINE__, tasklet->world_id);
}

void ql_vmk_tasklet_disable(ql_vmk_tasklet_t *tasklet)
{
	VMK_ReturnStatus status;

	vmk_AtomicWrite64(&tasklet->is_running, 0);
	status = vmk_WorldDestroy(tasklet->world_id);
	if(status != VMK_OK) {
		ql_vmk_pr_err("world destroy failed for world-id=%d. status=%s",
						tasklet->world_id,
						vmk_StatusToString(status));
	}

	status = vmk_WorldWaitForDeath(tasklet->world_id);
	if(status == VMK_OK)
		ql_vmk_pr("wait for world %d to die successful.", tasklet->world_id);
	else
        ql_vmk_pr_err("wait for death of world-id=%d failed. status=%s",
				tasklet->world_id, vmk_StatusToString(status));
}

vmk_uint32 ql_vmk_bar_size(struct qfle3i_hba *hba, vmk_uint8 bar_id)
{
	vmk_PCIResource resources[VMK_PCI_NUM_BARS];

	vmk_PCIQueryIOResources(hba->pcidev, VMK_PCI_NUM_BARS, resources);

	return resources[bar_id].size;
}


/* Sets the requested power state */
int cnic_set_power_state(struct qfle3i_hba *hba,
				 pci_power_t state)
{
	//vmk_PCIDevice pdev = hba->pcidev;
	//vmk_uint16 pmcsr;

	/* FIXME - emulation currently does not support PM (13_06_04) */
	//DP_VERBOSE(hba, NETIF_MSG_DRV, "Omitting Power state change\n");
	return 0;

	//SV: XXX:
#if 0
	vmk_PCIReadConfig(vmk_ModuleCurrentID, pdev, VMK_PCI_CONFIG_ACCESS_16,
			hba->pci_params.pm_cap + PCI_PM_CTRL, (vmk_uint32 *)&pmcsr);

	switch (state) {
	case PCI_D0:
		vmk_PCIWriteConfig(vmk_ModuleCurrentID, pdev, VMK_PCI_CONFIG_ACCESS_16,
				hba->pci_params.pm_cap + PCI_PM_CTRL,
				(vmk_uint32)((pmcsr & ~PCI_PM_CTRL_STATE_MASK) |
					PCI_PM_CTRL_PME_STATUS));

		if (pmcsr & PCI_PM_CTRL_STATE_MASK)
			/* delay required during transition out of D3hot */
			vmk_WorldSleep(20*VMK_USEC_PER_MSEC);
		break;

	case PCI_D3hot:
		/* If there are other clients (e.g. Diag) above don't shut down
		 * the power.
		 */
		//if (vmk_AtomicRead64(&hba->pdev->enable_cnt) != 1)
		//	return 0;
		pmcsr &= ~PCI_PM_CTRL_STATE_MASK;
		pmcsr |= 3;

		if (hba->wol)
			pmcsr |= PCI_PM_CTRL_PME_ENABLE;

		vmk_PCIWriteConfig(vmk_ModuleCurrentID, pdev, VMK_PCI_CONFIG_ACCESS_16,
				hba->pci_params.pm_cap + PCI_PM_CTRL, (vmk_uint32)pmcsr);

		/* No more memory access after this point until
		* device is brought back to D0.
		*/
		break;

	default:
		//DP_ERR(hba, "Can't support state = %d\n", state);
		return VMK_BAD_PARAM;
	}
	return 0;
#endif
}

void ql_vmk_pci_disable_msix(vmk_PCIDevice pdev)
{
	VMK_ReturnStatus status;

	status = vmk_PCIFreeIntrCookie(vmk_ModuleCurrentID, pdev);
	//DP_VERBOSE(hba, CNIC_MSG_INTR, "PCI Free Intr Cookie : %s",
	//		vmk_StatusToString(status));
}

void ql_vmk_pci_disable_msi(vmk_PCIDevice pdev)
{
}

int ql_vmk_pci_enable_msix(vmk_PCIDevice pdev, struct ql_msix_entry *entries,
	int minvec, int maxvec, int *vectors)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_IntrCookie *intr_array;
	vmk_uint32 num_vectors;
	int i;

	if (maxvec < minvec)
		return VMK_BAD_PARAM;

	ql_vmk_pr("%s: Allocating interrupt cookies for %d vectors\n", __func__, maxvec);

	if ((intr_array = ql_vmk_alloc(sizeof(vmk_IntrCookie) * maxvec))
			== NULL) {
		status = VMK_NO_MEMORY;
		goto intr_alloc_failed;
	}

	status = vmk_PCIAllocIntrCookie(vmk_ModuleCurrentID,
	    pdev, VMK_PCI_INTERRUPT_TYPE_MSIX,
	    maxvec, minvec, NULL, intr_array, &num_vectors);

	if (status != VMK_OK) {
		ql_vmk_pr_err("Failed to allocate interrupt cookies\n");
		goto cookie_alloc_failed;
	}

	if (num_vectors < minvec) {
		vmk_PCIFreeIntrCookie(vmk_ModuleCurrentID, pdev);
		status = VMK_NO_RESOURCES;
		goto cookie_alloc_failed;
	}

	for (i = 0; i < num_vectors; i++) {
		entries[i].intr_cookie = intr_array[i];
		entries[i].vector = intr_array[i];
	}
	*vectors = num_vectors;
	ql_vmk_pr("%s: num vectors allocated = %x\n", __func__, num_vectors);

	ql_vmk_free(intr_array);

	return status;

cookie_alloc_failed:
	ql_vmk_free(intr_array);
intr_alloc_failed:
	return status;
}

int ql_vmk_pci_enable_single(vmk_PCIDevice pdev, vmk_PCIInterruptType type,
	vmk_IntrCookie *intr_cookie)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_uint32 num_intrs_alloced;

	status = vmk_PCIAllocIntrCookie(vmk_ModuleCurrentID, pdev, type, 1,
		1, NULL, intr_cookie, &num_intrs_alloced);
	if (status != VMK_OK) {
		ql_vmk_pr("Failed allocating MSI interrupt: 0x%x", status);
		return 1;
	}

	return 0;
}

void ql_vmk_free_irq(struct qfle3i_hba *hba, vmk_IntrCookie intr_cookie,
	void *handlerData)
{
	VMK_ReturnStatus status;

	status = vmk_IntrDisable(intr_cookie);
	ql_vmk_pr("Disabling interrupt %s\n", vmk_StatusToString(status));
	status = vmk_IntrUnregister(vmk_ModuleCurrentID, intr_cookie, handlerData);
	ql_vmk_pr("Unregistering interrupt %s\n", vmk_StatusToString(status));
}


VMK_ReturnStatus
ql_vmk_ack_int(void *client_data, vmk_IntrCookie intr_cookie)
{
	/* TODO: Placeholder for now */
	return VMK_OK;
}


int ql_vmk_request_irq(struct qfle3i_hba *hba, vmk_IntrCookie intr_cookie,
	vmk_IntrHandler handler, const char *name, void *dev,
	vmk_IntrAcknowledge ack)
{
	vmk_IntrProps props;
	VMK_ReturnStatus status = VMK_OK;

	vmk_Memset(&props, 0, sizeof(vmk_IntrProps));

	props.device = hba->ldev;
	props.handlerData = dev;
	props.attrs = VMK_INTR_ATTRS_ENTROPY_SOURCE;
	props.acknowledgeInterrupt = ack;
	vmk_NameInitialize(&(props.deviceName), name);
	props.handler = handler;

	ql_vmk_pr("Attempting to register MSI-X interrupt %x\n", intr_cookie);
	status = vmk_IntrRegister(vmk_ModuleCurrentID, intr_cookie, &props);
	if (status != VMK_OK) {
		ql_vmk_pr_err("Interrupt registration failed:%s",
			vmk_StatusToString(status));
		goto fail;
	}

	status = vmk_IntrEnable(intr_cookie);
	if (status != VMK_OK) {
		ql_vmk_pr_err("Enabling the Interrupt failed:%s",
			vmk_StatusToString(status));
		vmk_IntrUnregister(vmk_ModuleCurrentID, intr_cookie, dev);
		goto fail;
	}

fail:
	return status;
}

int ql_vmk_pci_find_ext_capability(struct qfle3i_hba *hba, int cap)
{
	VMK_ReturnStatus status;
	vmk_uint16 cap_offset = 0;

	status = vmk_PCIFindCapability(hba->pcidev, cap, &cap_offset);

	if (status == VMK_OK)
		return cap_offset;
	else
		return 0;
}

void ql_vmk_ref_init(vmk_atomic64 *refcount)
{
	vmk_AtomicWrite64(refcount, 1);
	vmk_AtomicEpilogue();
}

VMK_ReturnStatus ql_vmk_ref_get(vmk_atomic64 *refcount)
{
	if (!vmk_AtomicRead64(refcount)) {
		vmk_WarningMessage("%s: refcount has already reached 0.", __func__);
		return VMK_FAILURE;
	}

	vmk_AtomicInc64(refcount);
	vmk_AtomicEpilogue();
	return VMK_OK;
}

VMK_ReturnStatus ql_vmk_ref_put(vmk_atomic64 *refcount,
	VMK_ReturnStatus (*release)(void *arg), void *arg)
{
	if (vmk_AtomicRead64(refcount) == 0) {
		vmk_AlertMessage("%s: Reference already zero!\n", __func__);
		return VMK_FAILURE;
	}

	if (vmk_AtomicReadDec64(refcount) == 1) {
		return release(arg);
	}
	return VMK_OK;
}

void ql_vmk_world_wakeup(struct ql_vmk_cmpl *cmpl)
{
	cmpl->cmpl_done = 1;
	vmk_WorldWakeup((vmk_WorldEventID)&cmpl->cmpl_event);
}

/* timeout value should in miliseconds */

VMK_ReturnStatus ql_vmk_wait_for_completion(struct ql_vmk_cmpl *cmpl,
		vmk_Bool cond, vmk_uint32 wait_time)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_uint32 start = 0, current = 0;

	/* reset completion variable */
	cmpl->cmpl_done = 0;

	/* check if condition is true */
	if (cond)
		return VMK_OK;

	vmk_SpinlockLock(cmpl->cmpl_lock);

	start = vmk_GetTimerCycles() / vmk_TimerCyclesPerSecond();
	/* converting time from sec to msec */
	start = start * VMK_MSEC_PER_SEC;


wait_again:
	status = vmk_WorldWait((vmk_WorldEventID)&cmpl->cmpl_event,
							cmpl->cmpl_lock, wait_time,
							"waiting for completion");

	/* If VMK_BAD_PARAM is returned, still holding the lock. */
	if(status != VMK_BAD_PARAM) {
		vmk_SpinlockLock(cmpl->cmpl_lock);
	}

	current = vmk_GetTimerCycles() / vmk_TimerCyclesPerSecond();
	current = current * VMK_MSEC_PER_SEC;

	if (!cmpl->cmpl_done) {
		if ((current - start) < wait_time) {
			wait_time -= (current - start);
			start = current;
			goto wait_again;
		}
		else {
			vmk_SpinlockUnlock(cmpl->cmpl_lock);
			return VMK_TIMEOUT;
		}
	}
	vmk_SpinlockUnlock(cmpl->cmpl_lock);

	return VMK_OK;
}

VMK_ReturnStatus ql_vmk_init_completion(struct ql_vmk_cmpl *cmpl, const char *name)
{
	return ql_vmk_spin_lock_init(&cmpl->cmpl_lock, LOCK_RANK_HIGHEST, name);
}

void ql_vmk_dump_buffer(vmk_uint8 *b, vmk_uint32 size)
{
	vmk_uint32 cnt;

	vmk_LogMessage(" 0   1   2   3   4   5   6   7   8   9  "
	    "Ah  Bh  Ch  Dh  Eh  Fh");

	vmk_LogMessage("----------------------------------------"
	    "----------------------");

	for (cnt = 0; cnt < size; cnt++) {
		if (!(cnt % 16) && (cnt != 0))
			vmk_LogNoLevel(VMK_LOG_URGENCY_NORMAL, "\n");
		vmk_LogNoLevel(VMK_LOG_URGENCY_NORMAL, "%02x   ", *b++);
	}
	vmk_LogNoLevel(VMK_LOG_URGENCY_NORMAL, "\n");
}

void cnic_msix_sp_int(void *tasklet, vmk_IntrCookie irq)
{
	ql_vmk_tasklet_schedule((ql_vmk_tasklet_t *)tasklet);
}

void ql_vmk_list_splice(vmk_ListLinks *list, vmk_ListLinks *prev,
	vmk_ListLinks *next)
{
	vmk_ListLinks *first = list->nextPtr;
	vmk_ListLinks *last = list->prevPtr;

	first->prevPtr = prev;
	prev->nextPtr = first;

	last->nextPtr = next;
	next->prevPtr = last;
}

void ql_vmk_list_splice_tail_init(vmk_ListLinks *list, vmk_ListLinks *head)
{
	if (!vmk_ListIsEmpty(list)) {
		ql_vmk_list_splice(list, head->prevPtr, head);
		vmk_ListInit(list);
	}
}



/* SV: From qed_helper.c */

void ql_vmk_mod_timer(vmk_Timer *tmr, void *tmo, void *data,
	unsigned long delay, vmk_Bool cancel)
{
	VMK_ReturnStatus rc;

	/* SV: XXX: Verify whether it can replace mod_timer. */
	if ((signed long)delay <= 0) {
		delay = 1;
		//vmk_LogMessage("%s: negative timer, setting to 1 timer tick\n",
		//	__func__);
	}
	if (cancel == VMK_TRUE)
		vmk_TimerCancel(*tmr, VMK_FALSE);
	rc = vmk_TimerSchedule(qfle3i_driver_info.timer_queue,
		(void *)tmo,
		(void *)data,
		vmk_TimerTCToMS(delay) * VMK_USEC_PER_MSEC,
		VMK_TIMER_DEFAULT_TOLERANCE,
		VMK_TIMER_ATTR_NONE,
		VMK_LOCKDOMAIN_INVALID,
		VMK_SPINLOCK_UNRANKED,
		tmr);
	if (rc != VMK_OK) {
		//vmk_WarningMessage("Error: Could not re-schedule timer.\n");
	}
}

void ql_vmk_init_singlethread_delayed_workitem(
		ql_vmk_singlethread_delayed_workitem_t *d_work)
{
	ql_vmk_init_singlethread_workitem(&d_work->work, NULL, NULL);
	d_work->wq = NULL;
}

VMK_ReturnStatus ql_vmk_queue_delayed_work(vmk_TimerQueue tq,
    ql_vmk_singlethread_workqueue_t *wq,
    ql_vmk_singlethread_delayed_workitem_t *d_work, void *handler,
    void *data, unsigned long delay_msec)
{
	VMK_ReturnStatus rc;

	vmk_SpinlockLock(wq->lock);
	if (d_work->work.workitem_state == ACTIVE) {
		vmk_SpinlockUnlock(wq->lock);
		return VMK_FAILURE;
	}

	if (d_work->work.workitem_state == DELAY_ACTIVE)  {
		vmk_LogMessage("cnic: delayed work %p timer already active\n", d_work);
		vmk_TimerCancel(d_work->timer, VMK_FALSE);
	}

	ql_vmk_init_singlethread_workitem(&d_work->work, handler, data);
	d_work->wq = wq;

	d_work->work.workitem_state = DELAY_ACTIVE;
	rc = vmk_TimerSchedule(tq,
		(void *)ql_vmk_handle_tmo,
		(void *)d_work,
		delay_msec * VMK_USEC_PER_MSEC,
		VMK_TIMER_DEFAULT_TOLERANCE,
		VMK_TIMER_ATTR_NONE,
		VMK_LOCKDOMAIN_INVALID,
		VMK_SPINLOCK_UNRANKED,
		&d_work->timer);

	vmk_SpinlockUnlock(wq->lock);
	if (rc != VMK_OK) {
		vmk_LogMessage("cnic: could not sched timer.\n");
	}
	return rc;
}

void ql_vmk_handle_tmo(ql_vmk_singlethread_delayed_workitem_t *d_work)
{
	if (!d_work) {
		vmk_LogMessage("d_work NULL!\n");
		return;
	}

	ql_vmk_singlethread_queue_work(d_work->wq, &d_work->work);
}


/* SV: From qed_helper.c */

/* SV: XXX: alternative for is_valid_ether_addr() needed. */
VMK_ReturnStatus ql_vmk_is_valid_ether_addr(vmk_uint8 *addr)
{
	return VMK_TRUE;
}

/*SV: XXX: verify code to reset L2 and L3 headers. */
	//skb_reset_mac_header(pkt);
	//skb_reset_network_header(pkt);
void ql_vmk_reset_l2_l3_hdr(vmk_PktHandle *pkt)
{
	vmk_PktHeaderEntry *l2_hdr = NULL, *l3_hdr = NULL;
	vmk_uint16 l2_hdr_idx = 0, l3_hdr_idx = 0;

	vmk_PktHeaderL2Find(pkt, &l2_hdr, &l2_hdr_idx);
	vmk_PktHeaderL3Find(pkt, &l3_hdr, &l3_hdr_idx);
	vmk_PktHeaderEntryRemove(pkt, l2_hdr_idx);
	vmk_PktHeaderEntryRemove(pkt, l3_hdr_idx);
}

/* SV: XXX: Verify L2 header manipulation. */
vmk_EthHdr *ql_vmk_get_l2_hdr(vmk_PktHandle *pkt)
{
	vmk_EthHdr *eh = NULL;
	vmk_PktHeaderEntry *hdr_entry = NULL;
	vmk_uint16 hdr_idx = 0;
	vmk_uint32 eth_hdr_offset = 0;
	char *mapped_data = NULL;

	if (vmk_PktHeaderL2Find(pkt, &hdr_entry, &hdr_idx) == VMK_OK) {
		mapped_data = (char *)vmk_PktFrameMappedPointerGet(pkt);
		eth_hdr_offset = hdr_entry->offset;
		eh = (vmk_EthHdr *)((vmk_uint8 *)mapped_data + eth_hdr_offset);
	}

	return eh;
}

VMK_ReturnStatus ql_vmk_flush_singlethread_workqueue(ql_vmk_singlethread_workqueue_t *wq)
{
	//SV: XXX: while (vmk_WorldWait(VMK_EVENT_NONE, VMK_LOCK_INVALID,
	//		VMK_TIMEOUT_UNLIMITED_MS, "wq thread sleeping") == VMK_OK ) {
		ql_vmk_singlethread_workitem_t *item;

		vmk_SpinlockLock(wq->lock);
		/* there could be spurious wakeups */
		while (!vmk_ListIsEmpty(&wq->work_items)) {
			item = VMK_LIST_ENTRY(vmk_ListFirst(&wq->work_items),
					ql_vmk_singlethread_workitem_t, links);
			vmk_ListRemove(&item->links);
			vmk_ListInitElement(&item->links);
		}
		vmk_SpinlockUnlock(wq->lock);
	//}

	return VMK_OK;
}

VMK_ReturnStatus ql_vmk_cancel_delayed_work(
	ql_vmk_singlethread_delayed_workitem_t *d_work)
{
	/* SV: XXX: do we need to cleanup anything for the delayed-work? */
	return (vmk_TimerCancel(d_work->timer, VMK_FALSE));
}

VMK_ReturnStatus ql_vmk_cancel_delayed_work_sync(
	ql_vmk_singlethread_delayed_workitem_t *d_work)
{
	/* SV: XXX: do we need to cleanup anything for the delayed-work? */
	return (vmk_TimerCancel(d_work->timer, VMK_TRUE));
}

VMK_ReturnStatus ql_vmk_is_list_looped(vmk_ListLinks *head, const char *func)
{
	vmk_ListLinks *itr1 = NULL, *itr2 = NULL;

	itr1 = head;
	itr2 = head->nextPtr->nextPtr;
	while (itr1->nextPtr != head) {
		if ((itr1 != head) && (itr1 == itr2)) {
			vmk_LogMessage("Warning: %s: loop found at %p for list:%p.",
				func, itr1, head);
			return 1;
		}

		if (itr1->nextPtr == head || itr2->nextPtr == head ||
				itr2->nextPtr->nextPtr == head)
			break;

		itr1 = itr1->nextPtr;
		itr2 = itr2->nextPtr->nextPtr;
	}
	return 0;
}

