/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.internal.storage.dfs;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.concurrent.locks.ReentrantLock;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.internal.storage.dfs.DfsBlock;
import org.eclipse.jgit.internal.storage.dfs.DfsBlockCacheConfig;
import org.eclipse.jgit.internal.storage.dfs.DfsPackDescription;
import org.eclipse.jgit.internal.storage.dfs.DfsPackFile;
import org.eclipse.jgit.internal.storage.dfs.DfsPackKey;
import org.eclipse.jgit.internal.storage.dfs.DfsReader;

public final class DfsBlockCache {
    private static volatile DfsBlockCache cache;
    private final int tableSize;
    private final AtomicReferenceArray<HashEntry> table;
    private final ReentrantLock[] loadLocks;
    private final long maxBytes;
    private final long maxStreamThroughCache;
    private final int blockSize;
    private final int blockSizeShift;
    private final Map<DfsPackDescription, DfsPackFile> packCache;
    private final Collection<DfsPackFile> packFiles;
    private final AtomicLong statHit;
    private final AtomicLong statMiss;
    private volatile long statEvict;
    private final ReentrantLock clockLock;
    private Ref clockHand;
    private volatile long liveBytes;

    public static void reconfigure(DfsBlockCacheConfig cfg) {
        DfsBlockCache nc = new DfsBlockCache(cfg);
        DfsBlockCache oc = cache;
        cache = nc;
        if (oc != null) {
            for (DfsPackFile pack : oc.getPackFiles()) {
                pack.key.cachedSize.set(0L);
            }
        }
    }

    public static DfsBlockCache getInstance() {
        return cache;
    }

    private DfsBlockCache(DfsBlockCacheConfig cfg) {
        this.tableSize = DfsBlockCache.tableSize(cfg);
        if (this.tableSize < 1) {
            throw new IllegalArgumentException(JGitText.get().tSizeMustBeGreaterOrEqual1);
        }
        this.table = new AtomicReferenceArray(this.tableSize);
        this.loadLocks = new ReentrantLock[32];
        for (int i = 0; i < this.loadLocks.length; ++i) {
            this.loadLocks[i] = new ReentrantLock(true);
        }
        this.maxBytes = cfg.getBlockLimit();
        this.maxStreamThroughCache = (long)((double)this.maxBytes * cfg.getStreamRatio());
        this.blockSize = cfg.getBlockSize();
        this.blockSizeShift = Integer.numberOfTrailingZeros(this.blockSize);
        this.clockLock = new ReentrantLock(true);
        this.clockHand = new Ref<Object>(new DfsPackKey(), -1L, 0, null);
        this.clockHand.next = this.clockHand;
        this.packCache = new ConcurrentHashMap<DfsPackDescription, DfsPackFile>(16, 0.75f, 1);
        this.packFiles = Collections.unmodifiableCollection(this.packCache.values());
        this.statHit = new AtomicLong();
        this.statMiss = new AtomicLong();
    }

    boolean shouldCopyThroughCache(long length) {
        return length <= this.maxStreamThroughCache;
    }

    public long getCurrentSize() {
        return this.liveBytes;
    }

    public long getFillPercentage() {
        return this.getCurrentSize() * 100L / this.maxBytes;
    }

    public long getHitCount() {
        return this.statHit.get();
    }

    public long getMissCount() {
        return this.statMiss.get();
    }

    public long getTotalRequestCount() {
        return this.getHitCount() + this.getMissCount();
    }

    public long getHitRatio() {
        long miss;
        long hits = this.statHit.get();
        long total = hits + (miss = this.statMiss.get());
        if (total == 0L) {
            return 0L;
        }
        return hits * 100L / total;
    }

    public long getEvictions() {
        return this.statEvict;
    }

    public Collection<DfsPackFile> getPackFiles() {
        return this.packFiles;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    DfsPackFile getOrCreate(DfsPackDescription dsc, DfsPackKey key) {
        Map<DfsPackDescription, DfsPackFile> map = this.packCache;
        synchronized (map) {
            DfsPackFile pack = this.packCache.get(dsc);
            if (pack != null && pack.invalid()) {
                this.packCache.remove(dsc);
                pack = null;
            }
            if (pack == null) {
                if (key == null) {
                    key = new DfsPackKey();
                }
                pack = new DfsPackFile(this, dsc, key);
                this.packCache.put(dsc, pack);
            }
            return pack;
        }
    }

    private int hash(int packHash, long off) {
        return packHash + (int)(off >>> this.blockSizeShift);
    }

    int getBlockSize() {
        return this.blockSize;
    }

    private static int tableSize(DfsBlockCacheConfig cfg) {
        int wsz = cfg.getBlockSize();
        long limit = cfg.getBlockLimit();
        if (wsz <= 0) {
            throw new IllegalArgumentException(JGitText.get().invalidWindowSize);
        }
        if (limit < (long)wsz) {
            throw new IllegalArgumentException(JGitText.get().windowSizeMustBeLesserThanLimit);
        }
        return (int)Math.min(5L * (limit / (long)wsz) / 2L, Integer.MAX_VALUE);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    DfsBlock getOrLoad(DfsPackFile pack, long position, DfsReader ctx) throws IOException {
        long requestedPosition = position;
        DfsPackKey key = pack.key;
        int slot = this.slot(key, position = pack.alignToBlock(position));
        HashEntry e1 = this.table.get(slot);
        DfsBlock v = (DfsBlock)this.scan(e1, key, position);
        if (v != null) {
            this.statHit.incrementAndGet();
            return v;
        }
        this.reserveSpace(this.blockSize);
        ReentrantLock regionLock = this.lockFor(key, position);
        regionLock.lock();
        try {
            HashEntry n;
            HashEntry e2 = this.table.get(slot);
            if (e2 != e1 && (v = (DfsBlock)this.scan(e2, key, position)) != null) {
                this.statHit.incrementAndGet();
                this.creditSpace(this.blockSize);
                DfsBlock dfsBlock = v;
                return dfsBlock;
            }
            this.statMiss.incrementAndGet();
            boolean credit = true;
            try {
                v = pack.readOneBlock(position, ctx);
                credit = false;
            }
            finally {
                if (credit) {
                    this.creditSpace(this.blockSize);
                }
            }
            if (position != v.start) {
                position = v.start;
                slot = this.slot(key, position);
                e2 = this.table.get(slot);
            }
            key.cachedSize.addAndGet(v.size());
            Ref<DfsBlock> ref = new Ref<DfsBlock>(key, position, v.size(), v);
            ref.hot = true;
            while (!this.table.compareAndSet(slot, e2, n = new HashEntry(DfsBlockCache.clean(e2), ref))) {
                e2 = this.table.get(slot);
            }
            this.addToClock(ref, this.blockSize - v.size());
        }
        finally {
            regionLock.unlock();
        }
        if (v.contains(pack.key, requestedPosition)) {
            return v;
        }
        return this.getOrLoad(pack, requestedPosition, ctx);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void reserveSpace(int reserve) {
        this.clockLock.lock();
        try {
            long live = this.liveBytes + (long)reserve;
            if (this.maxBytes < live) {
                Ref prev = this.clockHand;
                Ref hand = this.clockHand.next;
                do {
                    if (hand.hot) {
                        hand.hot = false;
                        prev = hand;
                        hand = hand.next;
                        continue;
                    }
                    if (prev == hand) break;
                    Ref dead = hand;
                    prev.next = hand = hand.next;
                    dead.next = null;
                    dead.value = null;
                    live -= (long)dead.size;
                    dead.pack.cachedSize.addAndGet(-dead.size);
                    ++this.statEvict;
                } while (this.maxBytes < live);
                this.clockHand = prev;
            }
            this.liveBytes = live;
        }
        finally {
            this.clockLock.unlock();
        }
    }

    private void creditSpace(int credit) {
        this.clockLock.lock();
        this.liveBytes -= (long)credit;
        this.clockLock.unlock();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addToClock(Ref ref, int credit) {
        this.clockLock.lock();
        try {
            if (credit != 0) {
                this.liveBytes -= (long)credit;
            }
            Ref ptr = this.clockHand;
            ref.next = ptr.next;
            ptr.next = ref;
            this.clockHand = ref;
        }
        finally {
            this.clockLock.unlock();
        }
    }

    void put(DfsBlock v) {
        this.put(v.pack, v.start, v.size(), v);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    <T> Ref<T> put(DfsPackKey key, long pos, int size2, T v) {
        int slot = this.slot(key, pos);
        HashEntry e1 = this.table.get(slot);
        Ref<T> ref = this.scanRef(e1, key, pos);
        if (ref != null) {
            return ref;
        }
        this.reserveSpace(size2);
        ReentrantLock regionLock = this.lockFor(key, pos);
        regionLock.lock();
        try {
            HashEntry n;
            HashEntry e2 = this.table.get(slot);
            if (e2 != e1 && (ref = this.scanRef(e2, key, pos)) != null) {
                this.creditSpace(size2);
                Ref<T> ref2 = ref;
                return ref2;
            }
            key.cachedSize.addAndGet(size2);
            ref = new Ref<T>(key, pos, size2, v);
            ref.hot = true;
            while (!this.table.compareAndSet(slot, e2, n = new HashEntry(DfsBlockCache.clean(e2), ref))) {
                e2 = this.table.get(slot);
            }
            this.addToClock(ref, 0);
        }
        finally {
            regionLock.unlock();
        }
        return ref;
    }

    boolean contains(DfsPackKey key, long position) {
        return this.scan(this.table.get(this.slot(key, position)), key, position) != null;
    }

    <T> T get(DfsPackKey key, long position) {
        T val = this.scan(this.table.get(this.slot(key, position)), key, position);
        if (val == null) {
            this.statMiss.incrementAndGet();
        } else {
            this.statHit.incrementAndGet();
        }
        return val;
    }

    private <T> T scan(HashEntry n, DfsPackKey pack, long position) {
        Ref<T> r = this.scanRef(n, pack, position);
        return r != null ? (T)r.get() : null;
    }

    private <T> Ref<T> scanRef(HashEntry n, DfsPackKey pack, long position) {
        while (n != null) {
            Ref r = n.ref;
            if (r.pack == pack && r.position == position) {
                return r.get() != null ? r : null;
            }
            n = n.next;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void remove(DfsPackFile pack) {
        Map<DfsPackDescription, DfsPackFile> map = this.packCache;
        synchronized (map) {
            this.packCache.remove(pack.getPackDescription());
        }
    }

    private int slot(DfsPackKey pack, long position) {
        return (this.hash(pack.hash, position) >>> 1) % this.tableSize;
    }

    private ReentrantLock lockFor(DfsPackKey pack, long position) {
        return this.loadLocks[(this.hash(pack.hash, position) >>> 1) % this.loadLocks.length];
    }

    private static HashEntry clean(HashEntry top) {
        while (top != null && top.ref.next == null) {
            top = top.next;
        }
        if (top == null) {
            return null;
        }
        HashEntry n = DfsBlockCache.clean(top.next);
        return n == top.next ? top : new HashEntry(n, top.ref);
    }

    static {
        DfsBlockCache.reconfigure(new DfsBlockCacheConfig());
    }

    private static final class HashEntry {
        final HashEntry next;
        final Ref ref;

        HashEntry(HashEntry n, Ref r) {
            this.next = n;
            this.ref = r;
        }
    }

    static final class Ref<T> {
        final DfsPackKey pack;
        final long position;
        final int size;
        volatile T value;
        Ref next;
        volatile boolean hot;

        Ref(DfsPackKey pack, long position, int size2, T v) {
            this.pack = pack;
            this.position = position;
            this.size = size2;
            this.value = v;
        }

        T get() {
            T v = this.value;
            if (v != null) {
                this.hot = true;
            }
            return v;
        }

        boolean has() {
            return this.value != null;
        }
    }
}

