/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.servlet;

import ccl.util.Util;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Vector;
import javax.servlet.http.HttpServletResponse;

/**
 * This class encapsulates printing of html code in a very
 * basic way without any fancy output support.
 *
 * @author    Chr. Clemens Lee (clemens@kclee.com)
 * @version   $Id: HTML.java,v 1.17 2003/10/18 07:53:12 clemens Exp clemens $
 */
public class HTML 
{
    private PrintWriter _out = null;

    /**
     * All following output goes to the given
     * HttpServletResponse object. The content type is
     * set immediately to text/html and a html and head
     * section is opened as well.
     * Use the close method on this object to finish the
     * html page. To 'close' the already open head section
     * make sure you printed a title with 'printTitle'.
     *
     * @exception   IOException   in case no print writer 
     *                            could be created from the
     *                            response object.
     */
    public HTML( HttpServletResponse response_ ) 
        throws IOException
    {
        super();

        response_.setContentType( "text/html" );

        _initialize( response_.getWriter() );
    }

    /**
     * All following output goes to the given
     * writer object.
     * Use the close method on this object to finish the
     * html page. To 'close' the already open head section
     * make sure you printed a title with 'printTitle'.
     */
    public HTML( Writer writer ) 
    {
        super();

        _initialize( new PrintWriter( writer ) );
    }

    /**
     * All following output goes to the given
     * writer object.
     * Use the close method on this object to finish the
     * html page. To 'close' the already open head section
     * make sure you printed a title with 'printTitle'.
     */
    public HTML( PrintWriter writer ) 
    {
        super();

        _initialize( writer );
    }

    /**
     * Create a dummy html object useful for testing.
     */
    static public HTML getDummyHTML()
    {
        StringWriter writer = new StringWriter();
        HTML html = new HTML( writer );

        return html;
    }

    /**
     * Sets up the internal print writer and starts to print
     * already the first to opening standarg HTML tags.
     */
    private void _initialize( PrintWriter writer )
    {
        _out = writer;

        _out.println("<html>");
        _out.println("<head>");
    }

    /**
     * This method prints a title, closes the html head
     * section, starts the body section, and prints an
     * h1 section with the same text as the title.
     */
    public void printTitle( String sTitle_ ) 
{
        _out.println( "<title>" + sTitle_ + "</title>" );
        _out.println( "</head>" );
        _out.println( "<body>" );
        _out.println( "<center><h1>" + sTitle_ + "</h1></center>" );
    }

    /**
     * This method prints a title, closes the html head
     * section, starts the body section, and prints an
     * h1 section with the same text as the title.
     */
    public void printTitle( String sTitle_
                            , String bgcolor )
    {
        _out.println( "<title>" + sTitle_ + "</title>" );
        _out.println( "</head>" );
        _out.println( "<body bgcolor=\"" + bgcolor + "\">" );
        _out.println( "<center><h1>" + sTitle_ + "</h1></center>" );
    }

    /**
     * This method prints a title, closes the html head
     * section, starts the body section, and prints an
     * h1 section with the same text as the title.
     *
     * @param   title       the title string to print.
     * @param   bgcolor     the background color of the web page.
     *                      Use standard HTML syntax like "#ffffff".
     * @param   textcolor   the foreground color of the web page.
     *                      Use standard HTML syntax like "#000000".
     */
    public void printTitle( String title
                            , String bgcolor
                            , String textcolor )
    {
        _out.println( "<title>" + title + "</title>" );
        _out.println( "</head>" );
        _out.println( "<body text=\"" + textcolor + "\" bgcolor=\"" + bgcolor + "\">" );
        _out.println( "<center><h1>" + title + "</h1></center>" );
    }


    /**
     * This method prints a title, closes the html head
     * section, starts the body section, and prints an
     * h1 section with the same text as the title.
     *
     * @param   title        the title string to print.
     * @param   bgcolor      the background color of the web page.
     *                       Use standard HTML syntax like "#ffffff".
     * @param   textcolor    the foreground color of the web page.
     *                       Use standard HTML syntax like "#000000".
     * @param   linkcolor    the color used for new links.
     * @param   vlinkcolor   the color used for visited links.
     */
    public void printTitle( String title
                            , String bgcolor
                            , String textcolor 
                            , String linkcolor
                            , String vlinkcolor )
    {
        _out.println( "<title>" + title + "</title>" );
        _out.println( "</head>" );
        _out.println( "<body text=\"" + textcolor + "\" bgcolor=\"" + bgcolor + "\" link=\"" + linkcolor + "\" vlink=\"" + vlinkcolor + "\">" );
        _out.println( "<center><h1>" + title + "</h1></center>" );
    }

    /**
     * This method prints the given string unfiltered as html
     * without any html formatting.
     */
    public void print( String sHTML_ ) 
    {
        _out.print( sHTML_ );
    }

    /**
     * This method prints the given string unfiltered as html
     * without any html formatting. But an '\n' character is
     * added to the text for better readability in the html
     * source code.
     */
    public void println( String sLine_ ) 
{
        _out.println( sLine_ );
    }

    /**
     * Prints a <br> tag.
     */
    public void printBR() 
{
        println( "<br>" );
    }

    /**
     * Prints the text and adds a <br> tag.
     */
    public void printBR( String sText ) 
{
        print  ( sText );
        printBR();
    }

    /**
     * Prints a <p> tag.
     */
    public void printP() 
{
        println( "<p>" );
    }

    /**
     * Prints the text and adds a <p> tag.
     */
    public void printP( String sText ) 
{
        print  ( sText );
        println( "<p>" );
    }

    /**
     * This method prints the start tag of an html formular.
     * All following text input fields etc. will be aligned using an html table
     * under the hood.
     */
    public void printFormularStart( String sMethod_
                                    , String sAction_ ) 
    {
        _out.println( "<form method=\"" 
                      + sMethod_ 
                      + "\" action=\""
                      + sAction_
                      + "\">" );
        _out.println( "<table>" );
    }

    /**
     * This method prints an input tag with hidden data.
     */
    public void printFormularHidden( String sName_
                                     , String sValue_ )
    {
        _out.println( "<input type=\"hidden\" name=\""
                      + sName_
                      + "\" value=\""
                      + sValue_
                      + "\">" );
    }

    /**
     * This method prints a text input tag.
     */
    public void printFormularTextInput( String sMessage_
                                        , String sName_  )
    {
        printFormularTextInput( sMessage_, sName_, "" );
    }

    /**
     * Adds a text area to the formular.
     */
    public void printFormularTextArea( String sMessage_
                                       , String sName_
                                       , String sDefault_ )
    {
        _out.print( "<tr><td>" + sMessage_ + ": </td>"                                 );
        _out.print( "<td rowspan=2><textarea name=\"" + sName_ + "\" cols=80 rows=12>" );
        _out.print( sDefault_                                                          );
        _out.print( "</textarea></td></tr>"                                            );
        _out.print( "<tr><td></td></tr>"                                                    );
    }

    /**
     * This method prints a text input tag.
     */
    public void printFormularTextInput( String sMessage_
                                        , String sName_
                                        , String sDefault_ 
                                        , int size )
    {
        _out.println( "<tr><td>" + sMessage_ + ": </td>" );
        _out.println( "<td><input type=\"text\" name=\""
                      + sName_
                      + "\" value=\""
                      + sDefault_
                      + "\" size=\""
                      + size
                      + "\"></td></tr>" );
    }

    /**
     * This method prints a text input tag.
     */
    public void printFormularTextInput( String sMessage_
                                        , String sName_
                                        , String sDefault_ )
    {
        _out.println( "<tr><td>" + sMessage_ + ": </td>" );
        _out.println( "<td><input type=\"text\" name=\""
                      + sName_
                      + "\" value=\""
                      + sDefault_
                      + "\"></td></tr>" );
    }

    /**
     * This method prints a formular select tag with
     * the given values.
     */
    public void printFormularSelection( String sMessage_
                                        , String sName_
                                        , String[] asValue_
                                        , String sDefault_ )
    {
        printFormularSelection( sMessage_
                                , sName_
                                , Util.objectsToVector( asValue_ )
                                , Util.objectsToVector( asValue_ )
                                , sDefault_ );
    }

    /**
     * This method prints a formular select tag with
     * the given values.
     */
    public void printFormularSelection( String sMessage_
                                        , String sName_
                                        , String[] asValue_
                                        , String[] asRepresentation_
                                        , String sDefaultValue_      )
    {
        printFormularSelection( sMessage_
                                , sName_
                                , Util.objectsToVector( asValue_ )
                                , Util.objectsToVector( asRepresentation_ )
                                , sDefaultValue_ );
    }

    /**
     * This method prints a formular select tag with
     * the given values.
     */
    public void printFormularSelection( String sMessage_
                                        , String sName_
                                        , String[] asValue_
                                        , String[] asRepresentation_ )
    {
        printFormularSelection( sMessage_
                                , sName_
                                , Util.objectsToVector( asValue_ )
                                , Util.objectsToVector( asRepresentation_ )
                                , asValue_[ 0 ] );
    }

    /**
     * This method prints a formular select tag with
     * the given values.
     */
    public void printFormularSelection( String sMessage_
                                        , String sName_
                                        , Vector vValues_
                                        , Vector vRepresentations_
                                        , String sDefaultValue_   )
    {
        _out.println( "<tr><td>" + sMessage_ + ": </td>" );
        _out.println( "<td><select name=\""
                      + sName_
                      + "\">" );
        for( int option = 0; option < vValues_.size(); option++ ) 
{
            _out.print( "  <option value=\""
                        + vValues_.elementAt( option )
                        + "\"" );
            if ( ((String)vValues_.elementAt( option )).equals( sDefaultValue_ ) ) 
{
                _out.print( " selected" );
            }                  
            _out.println( ">"
                          + vRepresentations_.elementAt( option )
                          + "</option>" );
        }
        _out.println( "</select></td></tr>" );
    }

    /**
     * This method prints a formular select tag.
     * SQL query must return a result set with 'id' and 'name' columns.
     *
     * @exception   SQLException   if the db query fails.
     */
    public void printFormularSelection( String sMessage_
                                        , String sName_
                                        , String sSqlQuery_
                                        , String sDefaultValue_    
                                        , Statement statement_     )
        throws SQLException
    {
        Vector vIds   = new Vector();
        Vector vNames = new Vector();
        ResultSet result = statement_.executeQuery( sSqlQuery_ );
        while( result.next() ) 
{
            vIds  .addElement( result.getString( "id"   ) );
            vNames.addElement( result.getString( "name" ) );
        }
        result.close();
        printFormularSelection( sMessage_
                                , sName_
                                , vIds
                                , vNames
                                , sDefaultValue_ );
    }

    /**
     * Let the user select if the formular entry should be
     * use in "add", "change", or "delete" mode.
     * The name of the field is "operation_mode".
     * Three different submit buttons are used.
     */
    public void printFormularOperationMode()
    {
        /*
        printFormularSelection( "Operation mode"
                                , "operation_mode"
                                , new String[] { "add", "change", "delete" }
                                , new String[] { "Add", "Change", "Delete" } );
        */
        _out.println( "<tr><td>Operation mode:&nbsp</td>"
                      + "<td>"
                      + "<input type=\"submit\" name=\"operation_mode\" value=\"   Add   \" length=\"10\">"
                      + "<input type=\"submit\" name=\"operation_mode\" value=\"Change\" length=\"10\">"
                      + "<input type=\"submit\" name=\"operation_mode\" value=\" Delete \" length=\"10\">"
                      + "</td></tr>" );
        _out.println( "</table>" );
        _out.println( "</form>" );
    }

    /**
     * This method prints a submit button for use in an html
     * formular. The form gets closed as well.
     */
    public void printFormularSubmit()
    {
        printFormularSubmit( "Submit" );
    }

    /**
     * This method prints a submit button for use in an html
     * formular. The form gets closed as well.
     *
     * @param   label   a name or label which appears on
     *                  the submit button.
     */
    public void printFormularSubmit( String label )
    {
        _out.println( "<tr><td></td><td><input type=\"submit\" name=\"" + label + "\"></td></tr>" );
        _out.println( "</table>" );
        _out.println( "</form>" );
    }

    /**
     * Prints an html link to the page.
     */
    public void printLink( String sText, String sURL ) 
    {
        _out.print( "<a href=\"" + sURL + "\">" + sText + "</a>" );
    }

    private boolean _bBeforeTR = true;

    /**
     * Table methods provide a service where the user can
     * just add new value entries to the table, specify when
     * a line is finished etc. without having to deal with
     * table tags.
     */
    public void printTableStart()
    {
        _out.println( "<table>" );
    }

    /**
     * Table methods provide a service where the user can
     * just add new value entries to the table, specify when
     * a line is finished etc. without having to deal with
     * table tags.
     */
    public void printTableValue( String value )
    {
        if ( _bBeforeTR )
        {
            _out.println( "  <tr>" );
            _bBeforeTR = false;
        }
        _out.println( "    <td>" + value + "</td>" );
    }

    /**
     * Table methods provide a service where the user can
     * just add new value entries to the table, specify when
     * a line is finished etc. without having to deal with
     * table tags.
     */
    public void printTableValue( String value, String align )
    {
        if ( _bBeforeTR )
        {
            _out.println( "  <tr>" );
            _bBeforeTR = false;
        }
        _out.println( "    <td align=" + align + ">" + value + "</td>" );
    }

    /**
     * Table methods provide a service where the user can
     * just add new value entries to the table, specify when
     * a line is finished etc. without having to deal with
     * table tags.
     */
    public void endTableLine()
    {
        _out.println( "  </tr>" );
        _bBeforeTR = true;
    }

    /**
     * Table methods provide a service where the user can
     * just add new value entries to the table, specify when
     * a line is finished etc. without having to deal with
     * table tags.
     */
    public void printTableEnd()
    {
        _out.println( "</table>" );
    }

    private String _headerBGColor = null;
    private String _secondBGColor = null;
    private boolean _bSecondTableColor = false;

    /**
     * Table methods provide a service where the user can
     * just add new value entries to the table, specify when
     * a line is finished etc. without having to deal with
     * table tags. The difference to the simple printTableStart
     * and accompanying methods is that this one uses a 
     * different color for every second row and has also a
     * background color for the headers.
     */
    public void printDataTableStart( String headerBGColor
                                     , String secondBGColor )
    {
        _out.println( "<table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">" );
        _headerBGColor     = headerBGColor;
        _secondBGColor     = secondBGColor;
        _bBeforeTR         = true;
        _bSecondTableColor = false;
    }

    /**
     * Add header entry to table.
     */
    public void printDataTableHeader( String header )
    {
        printDataTableHeader( header, "left" );
    }

    /**
     * Add header entry to table.
     */
    public void printDataTableHeader( String header
                                      , String align )
    {
        printDataTableHeader( header, align, 1 );
    }

    /**
     * Add header entry to table.
     */
    public void printDataTableHeader( String header
                                      , String align
                                      , int colspan )
    {
        if ( _bBeforeTR )
        {
            _out.println( "  <tr bgcolor=\"" 
                          + _headerBGColor 
                          + "\">" );
            _bBeforeTR = false;
        }
        _out.print( "    <td>&nbsp;&nbsp;</td><td" );
        if ( colspan > 1 )
        {
            _out.print( " colspan=" + colspan);
        }
        _out.print( " valign=top align=" + align +"><b>" + header + "</b></td>" );
    }

    /**
     * Add value entry to table.
     */
    public void printDataTableValue( String value )
    {
        printDataTableValue( value, "left" );
    }

    /**
     * Add value entry to table.
     */
    public void printDataTableValue( String value
                                     , String align )
    {
        printDataTableValue( value, align, 1 );
    }

    /**
     * Add value entry to table.
     */
    public void printDataTableValue( String value
                                     , String align
                                     , int colspan )
    {
        if ( _bBeforeTR )
        {
            _out.print( "  <tr" );
            if ( _bSecondTableColor )
            {
                _out.print( " bgcolor=\"" 
                          + _secondBGColor
                          + "\"" );
            }
            _out.println( ">" );

            _bSecondTableColor = !_bSecondTableColor;
            _bBeforeTR         = false;
        }
        _out.print( "    <td>&nbsp;</td><td" );
        if ( colspan > 1 )
        {
            _out.print( " colspan=" + colspan);
        }
        _out.println( " valign=top align=" + align +">" + value + "</td>" );
    }

    /**
     * Table methods provide a service where the user can
     * just add new value entries to the table, specify when
     * a line is finished etc. without having to deal with
     * table tags.
     */
    public void endDataTableLine()
    {
        _out.println( "<td>&nbsp;&nbsp;</td>\n  </tr>" );
        _bBeforeTR = true;
    }

    /**
     * Prints the content of a jdbc result set via a data
     * table.
     *
     * @param       headerBGColor   HTML color scheme for background
     *                              for table header backround, 
     *                              for example "#ffcc00".
     * @param       secondBGColor   HTML color scheme for background
     *                              of every second data row,
     *                              for example "#c8c8c8".
     *
     * @exception   SQLException    if accessing db meta data
     *                              fails this exception is 
     *                              thrown.
     */
    public void printResultSet( String headerBGColor
                                , String secondBGColor
                                , ResultSet result     )
        throws SQLException
    {
        ResultSetMetaData metaData = result.getMetaData();
        int numberOfColumns = metaData.getColumnCount();

        printDataTableStart( headerBGColor, secondBGColor );

        // header line
        for( int column = 1; column <= numberOfColumns; column++ ) 
        {
            String sColumnName = metaData.getColumnName( column );
            printDataTableHeader( sColumnName );
        }
        endDataTableLine();

        while( result.next() )
        {
            for( int column = 1; column <= numberOfColumns; column++ ) 
            {
                Object oValue = result.getObject( column );
                if ( oValue == null )
                {
                    oValue = (Object)"";
                }
                String sValue = String.valueOf( oValue ).trim();
                if ( Util.isEmpty( sValue ) )
                {
                    sValue = "&nbsp;";
                }
                printDataTableValue( sValue );
            }
            endDataTableLine();
        }

        printTableEnd();
    }


    /**
     * This method prints a closing body and html tag.
     * The should be no following invokation on this object.
     */
    public void close() 
    {
        _out.println("</body>");
        _out.println("</html>");
        _out.close();
    }

    /* does not seem to work *
     * Forces to flush out the current content of the output 
     * stream to be sent to the client before the whole
     * page is finished.
     *
    public void flush()
    {
        _out.flush();
    }*/

    /**
     * Return "&nbsp;" in case the given value is null or an empty string,
     * otherwise the given value is returned.
     */
    static public String spacify( String value )
    {
        if ( Util.isEmpty( value ) )
        {
            return "&nbsp;";
        }

        return value;
    }

    /**
     * Convert all tabs to space (1).
     */
    static private String untabify( String s )
    {
        return Util.replace( s, "\t", " " );
    }

    /**
     * This method does not print html output but instead
     * creates a csv file which can be used to import into
     * a spread sheet application like Excel or StarCalc.
     * The data is taken from the given result set.
     *
     * @exception   IOException    in case anything goes wrong
     *                             writing output to the http
     *                             servlet response.
     * @exception   SQLException   in case anything goes wrong
     *                             with the database.
     */
    static public void exportCSV( HttpServletResponse response
                                  , ResultSet result )
        throws IOException
               , SQLException
    {
        response.setContentType( "text/plain" );
        PrintWriter out = response.getWriter();

        ResultSetMetaData metaData = result.getMetaData();
        int numberOfColumns = metaData.getColumnCount();

        // header line
        for( int column = 1; column <= numberOfColumns; column++ ) 
        {
            String sColumnName = metaData.getColumnName( column );
            if ( column > 1 )
            {
                out.print( "\t" );
            }
            out.print( untabify( sColumnName ) );
        }
        out.println();

        while( result.next() )
        {
            for( int column = 1; column <= numberOfColumns; column++ ) 
            {
                Object oValue = result.getObject( column );
                if ( oValue == null )
                {
                    oValue = (Object)"";
                }
                String sValue = untabify( String.valueOf( oValue ).trim() );
                if ( column > 1 )
                {
                    out.print( "\t" );
                }
                out.print( sValue );
            }
            out.println();
        }
    }
}
