/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.swing;

import ccl.util.Util;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.FocusManager;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JWindow;

/**
 * A panel with OK and Cancel button for use in dialogs.
 * You can overwrite the button texts. You can also
 * specify the alignement, either horizontal (default)
 * or vertical. Buttons have mnemonics working.
 *
 * Also a FocusManager gets installed that takes care
 * of the focus cycle and default button (OK).
 *
 * For now, use it only in modal dialogs.
 *
 * Internationalisation is planned.
 *
 * Warning!!!: because this panel installs it's on local
 *             focus manager, be sure to close a dialog
 *             only interactively. Don't setVisible( false )
 *             in you dialog code yourself, or the old focus
 *             manager won't be reinstalled.
 *             If this is needed, an method for that purpose
 *             has to be added.
 *
 * @author  Chr. Clemens Lahme
 * @version $Id: OKCancelPanel.java,v 1.11 2003/05/01 16:44:27 clemens Exp clemens $
 */
public class OKCancelPanel extends JPanel
                           implements ActionListener
{
    private JButton _btnOK = null;
    private JButton _btnCancel = null;
    private boolean _bOK = false;
    private Window _winParent = null;
    private final static int DEFAULT_GAP = 2;
    private int _defaultGap = DEFAULT_GAP;

    private FocusManager _fmOriginal = null;
    private FocusManager _fmLast = null;

    private WindowAdapter _pWindowAdapter = null;

    public OKCancelPanel( Window winParent_ ) 
    {
        this( winParent_, DEFAULT_GAP );
    }

    public OKCancelPanel( Window winParent_, int gap_ ) 
    {
        this( winParent_, gap_, false );
    }
        
    public OKCancelPanel( Window winParent_, boolean bVertical_ ) 
    {
        this( winParent_, DEFAULT_GAP, bVertical_ );
    }
        
    public OKCancelPanel( Window winParent_, int gap_, boolean bVertical_ ) 
    {
        super();
        
        _bOK = false;
        _winParent = winParent_;
        _defaultGap = gap_;

        setName( "OKCancelPanel" );

        Color clrBackground = _winParent.getBackground();
        if ( _winParent instanceof JWindow ) 
        {
            clrBackground = ((JWindow)_winParent).getContentPane().
                   getBackground();
        }
        setBackground( clrBackground );

        int rows = 0;
        int columns = 2;
        if ( bVertical_ ) 
        {
            rows = 2;
            columns = 0;
        }
        GridLayout pGridLayout = new GridLayout( rows, columns );
        if ( bVertical_ ) 
        {
            pGridLayout.setVgap( _defaultGap );
        }
        else 
        {
            pGridLayout.setHgap( _defaultGap );
        }
        setLayout( pGridLayout );

        _btnOK = new JButton( "OK" );
        _btnCancel = new JButton( "Cancel" );

        _btnOK.setName( "PanelOKButton" );
        _btnCancel.setName( "PanelCancelButton" );

        // cancel button consumes ENTER key, we have to prevent this for
        // default button handling
        SwingUtil.unregisterEnterAction( _btnCancel );

        _btnOK.setMnemonic( 'O' );
        _btnCancel.setMnemonic( 'C' );

        _btnOK.addActionListener( this );
        _btnCancel.addActionListener( this );
        
        add( _btnOK );
        add( _btnCancel );

        // focus and default button stuff
        JRootPane pRootPane = null;
        if ( _winParent instanceof JDialog ) 
        {
            pRootPane = ((JDialog)_winParent).getRootPane();
        }
        if ( _winParent instanceof JFrame ) 
        {
            pRootPane = ((JFrame)_winParent).getRootPane();
        }
        if ( pRootPane != null ) 
        {
            pRootPane.setDefaultButton( _btnOK );
        }

        _pWindowAdapter = new WindowAdapter() {
                public void windowOpened( WindowEvent pWindowEvent_ ) 
                {
                    Util.debug( this, "windowOpened(..)._winParent: " +
                                _winParent );
                    _fmOriginal = FocusManager.getCurrentManager();
                    _fmLast = _fmOriginal;
                    FocusManager.setCurrentManager
                           ( new OKCancelFocusManager( _btnOK,
                                                       _btnCancel ) );
                }

                public void windowClosed( WindowEvent pWindowEvent_ ) 
                {
                    FocusManager.setCurrentManager( _fmOriginal );
                    Util.debug( this, "windowClosed(..)._winParent: " +
                                _winParent );
                }
            };
        _winParent.addWindowListener( _pWindowAdapter );
    }

    public WindowAdapter getWindowAdapter() 
    {
        return _pWindowAdapter;
    }

    public boolean isOK() 
    {
        return _bOK;
    }

    /**
     * @deprecated   Use continueOperation() instead.
     */
    public void setFalse() 
    {
        _bOK = false;
    }

    /**
     * If you pop up another dialog inside the ok cancel
     * dialog, you might want to switch back temporarily
     * to the default focus manager. You can do this with
     * this method. So be sure, you use this method always
     * an even number of times.
     */
    public void switchFocusManager() 
    {
        FocusManager fmTemp = FocusManager.getCurrentManager();
        FocusManager.setCurrentManager( _fmLast );
        _fmLast = fmTemp;
    }

    /**
     * You don't wan't to close your dialog.
     * setDefaultCloseOperation( DO_NOTHING_ON_CLOSE ) on your
     * dialog and invoke this method (and of course, prevent
     * setVisible( false ) from being taken place.
     */
    public void continueOperation() 
    {
        _bOK = false;

        FocusManager.setCurrentManager
               ( new OKCancelFocusManager( _btnOK, _btnCancel ) );
    }

    public void setPreferredButtonSize( Dimension dimButton_ ) 
    {
        _btnOK.setPreferredSize( dimButton_ );
        _btnCancel.setPreferredSize( dimButton_ );
    }

    public void setMinimumButtonSize( Dimension dimButton_ ) 
    {
        _btnOK.setMinimumSize( dimButton_ );
        _btnCancel.setMinimumSize( dimButton_ );
    }

    public void setInitialFocusOnOKButton() 
    {
        SwingUtil.setInitialFocus( _winParent, _btnOK );
    }

    public JButton getOKButton() 
    {
        return _btnOK;
    }

    public JButton getCancelButton() 
    {
        return _btnCancel;
    }

    /*public void setInitialFocusOnCancelButton() {
      }*/

    /**
     * If you use these, just make sure, OK and Cancel don't
     * start with the same character (for mnemonics handling).
     */
    public void setOKText( String sOKText_ ) 
    {
        _btnOK.setText( sOKText_ );
        _btnOK.setMnemonic( sOKText_.charAt( 0 ) );
    }

    /**
     * If you use these, just make sure, OK and Cancel don't
     * start with the same character (for mnemonics handling).
     */
    public void setCancelText( String sCancelText_ ) 
    {
        _btnCancel.setText( sCancelText_ );
        _btnCancel.setMnemonic( sCancelText_.charAt( 0 ) );
    }

    public void actionPerformed( ActionEvent pActionEvent_ ) 
    {
        Object oSource = pActionEvent_.getSource();

        //WindowEvent pWindowEvent = new WindowEvent
        //       ( _winParent, WindowEvent.WINDOW_CLOSING );
        if ( oSource == _btnOK ) 
        {
            _bOK = true;
            _winParent.setVisible( false );
            //_winParent.dispatchEvent( pWindowEvent );
        }
        if ( oSource == _btnCancel ) 
        {
            _bOK = false;
            _winParent.setVisible( false );
            //_winParent.dispatchEvent( pWindowEvent );
        }
    }

    /*public void stopExit( String sMessage ) {
      runAction( new MissingData( S_NO_FIELD_SPECIFIED ) );
      _pOKCancelPanel.setFalse();
      }*/
}
