/*
 * Decompiled with CFR 0.152.
 */
package ghidra.file.formats.ext4;

import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.MemoryByteProvider;
import ghidra.app.util.importer.MessageLog;
import ghidra.file.analyzers.FileFormatAnalyzer;
import ghidra.file.formats.ext4.Ext4GroupDescriptor;
import ghidra.file.formats.ext4.Ext4Inode;
import ghidra.file.formats.ext4.Ext4SuperBlock;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.data.DataType;
import ghidra.program.model.listing.Program;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;

public class Ext4Analyzer
extends FileFormatAnalyzer {
    private int blockSize;

    public String getName() {
        return "Ext4 Analyzer";
    }

    public boolean getDefaultEnablement(Program program) {
        return true;
    }

    public String getDescription() {
        return "Annotates Ext4 file systems.";
    }

    public boolean canAnalyze(Program program) {
        MemoryByteProvider provider = MemoryByteProvider.createDefaultAddressSpaceByteProvider((Program)program, (boolean)false);
        BinaryReader reader = new BinaryReader((ByteProvider)provider, true);
        int start = this.getSuperBlockStart(reader);
        if (start == -1) {
            return false;
        }
        reader.setPointerIndex(start + 56);
        int magic = -1;
        try {
            magic = reader.readNextShort();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return magic == -4269;
    }

    public boolean isPrototype() {
        return false;
    }

    @Override
    public boolean analyze(Program program, AddressSetView set, TaskMonitor monitor, MessageLog log) throws Exception {
        MemoryByteProvider provider = MemoryByteProvider.createDefaultAddressSpaceByteProvider((Program)program, (boolean)false);
        BinaryReader reader = new BinaryReader((ByteProvider)provider, true);
        int start = this.getSuperBlockStart(reader);
        int groupStart = 0;
        reader.setPointerIndex(start);
        Ext4SuperBlock superBlock = new Ext4SuperBlock(reader);
        this.createData(program, this.toAddr(program, start), superBlock.toDataType());
        boolean is64Bit = superBlock.getS_desc_size() > 32 && (superBlock.getS_feature_incompat() & 0x80) > 0;
        long numBytes = program.getMaxAddress().getOffset() - program.getMinAddress().getOffset() + 1L;
        int groupSize = this.calculateGroupSize(superBlock);
        int numGroups = (int)numBytes / groupSize;
        if (numBytes % (long)groupSize != 0L) {
            ++numGroups;
        }
        long groupDescOffset = groupStart + this.blockSize;
        Address groupDescAddress = this.toAddr(program, groupDescOffset);
        reader.setPointerIndex(groupDescOffset);
        Ext4GroupDescriptor[] groupDescriptors = new Ext4GroupDescriptor[numGroups];
        monitor.setMessage("Creating group descriptors...");
        monitor.setMaximum((long)numGroups);
        for (int i = 0; i < numGroups; ++i) {
            monitor.checkCancelled();
            groupDescriptors[i] = new Ext4GroupDescriptor(reader, is64Bit);
            DataType groupDescDataType = groupDescriptors[i].toDataType();
            this.createData(program, groupDescAddress, groupDescDataType);
            groupDescAddress = groupDescAddress.add((long)groupDescDataType.getLength());
            monitor.incrementProgress(1L);
        }
        boolean isSparseSuper = (superBlock.getS_feature_ro_compat() & 1) != 0;
        this.createSuperBlockCopies(program, reader, groupSize, numGroups, is64Bit, isSparseSuper, monitor);
        this.createInodeTables(program, reader, superBlock, groupDescriptors, is64Bit, monitor);
        return true;
    }

    private void createInodeTables(Program program, BinaryReader reader, Ext4SuperBlock superBlock, Ext4GroupDescriptor[] groupDescriptors, boolean is64Bit, TaskMonitor monitor) throws DuplicateNameException, Exception {
        int inodeCount = superBlock.getS_inodes_count();
        Ext4Inode[] inodes = new Ext4Inode[inodeCount];
        for (int i = 0; i < groupDescriptors.length; ++i) {
            monitor.checkCancelled();
            long inodeTableBlockOffset = (long)groupDescriptors[i].getBg_inode_table_lo() & 0xFFFFFFFFL;
            if (is64Bit) {
                inodeTableBlockOffset = (long)groupDescriptors[i].getBg_inode_table_hi() << 32 | inodeTableBlockOffset;
            }
            long offset = inodeTableBlockOffset * (long)this.blockSize;
            reader.setPointerIndex(offset);
            Address address = null;
            try {
                address = this.toAddr(program, offset);
            }
            catch (Exception e) {
                throw new IOException("offset " + offset + " not in program.");
            }
            int inodesPerGroup = superBlock.getS_inodes_per_group();
            monitor.setMessage("Creating inode table " + i + " of " + (groupDescriptors.length - 1) + "...");
            monitor.setMaximum((long)inodesPerGroup);
            monitor.setProgress(0L);
            for (int j = 0; j < inodesPerGroup; ++j) {
                if (i == 0 && j == 0) continue;
                monitor.checkCancelled();
                Ext4Inode inode = new Ext4Inode(reader);
                DataType dataType = inode.toDataType();
                this.createData(program, address, dataType);
                program.getListing().setComment(address, 0, "0x" + Integer.toHexString(inodesPerGroup * i + j));
                address = address.add((long)superBlock.getS_inode_size());
                reader.setPointerIndex(address.getOffset());
                monitor.incrementProgress(1L);
                inodes[inodesPerGroup * i + j] = inode;
            }
        }
        this.processInodes(program, reader, superBlock, inodes, monitor);
    }

    private void processInodes(Program program, BinaryReader reader, Ext4SuperBlock superBlock, Ext4Inode[] inodes, TaskMonitor monitor) throws Exception {
        for (int i = 1; i < inodes.length; ++i) {
            monitor.checkCancelled();
            Ext4Inode inode = inodes[i];
            short mode = inode.getI_mode();
            if ((mode & 0x4000) != 0) {
                this.processDirectory(program, reader, superBlock, inode, monitor);
                continue;
            }
            if ((mode & 0x8000) == 0) continue;
            this.processFile(program, reader, superBlock, inode, monitor);
        }
    }

    private void processFile(Program program, BinaryReader reader, Ext4SuperBlock superBlock, Ext4Inode inode, TaskMonitor monitor) {
    }

    private void processDirectory(Program program, BinaryReader reader, Ext4SuperBlock superBlock, Ext4Inode inode, TaskMonitor monitor) throws Exception {
        boolean isDirEntry2;
        if ((inode.getI_flags() & 0x1000) != 0) {
            this.processHashTreeDirectory(program, reader, superBlock, inode, monitor);
        }
        boolean bl = isDirEntry2 = (superBlock.getS_feature_incompat() & 2) != 0;
        if ((inode.getI_flags() & 0x80000) != 0) {
            // empty if block
        }
    }

    private void processHashTreeDirectory(Program program, BinaryReader reader, Ext4SuperBlock superBlock, Ext4Inode inode, TaskMonitor monitor) {
    }

    private void createSuperBlockCopies(Program program, BinaryReader reader, int groupSize, int numGroups, boolean is64Bit, boolean isSparseSuper, TaskMonitor monitor) throws DuplicateNameException, IOException, Exception {
        monitor.setMessage("Creating super block and group descriptor copies...");
        monitor.setMaximum((long)numGroups);
        for (int i = 1; i < numGroups; ++i) {
            monitor.checkCancelled();
            if (isSparseSuper && !this.isXpowerOfY(i, 3) && !this.isXpowerOfY(i, 5) && !this.isXpowerOfY(i, 7)) continue;
            int offset = groupSize * i;
            Address address = this.toAddr(program, offset);
            reader.setPointerIndex(offset);
            Ext4SuperBlock superBlock = new Ext4SuperBlock(reader);
            this.createData(program, address, superBlock.toDataType());
            long groupDescOffset = offset + this.blockSize;
            Address groupDescAddress = this.toAddr(program, groupDescOffset);
            reader.setPointerIndex(groupDescOffset);
            for (int j = 0; j < numGroups; ++j) {
                Ext4GroupDescriptor groupDesc = new Ext4GroupDescriptor(reader, is64Bit);
                DataType groupDescDataType = groupDesc.toDataType();
                this.createData(program, groupDescAddress, groupDescDataType);
                groupDescAddress = groupDescAddress.add((long)groupDescDataType.getLength());
            }
            monitor.incrementProgress(1L);
        }
    }

    private boolean isXpowerOfY(int x, int y) {
        if (x == 0) {
            return false;
        }
        while (x % y == 0) {
            x /= y;
        }
        return x == 1;
    }

    private int calculateGroupSize(Ext4SuperBlock superBlock) {
        int logBlockSize = superBlock.getS_log_block_size();
        this.blockSize = (int)Math.pow(2.0, 10 + logBlockSize);
        int groupSize = this.blockSize * superBlock.getS_blocks_per_group();
        return groupSize;
    }

    private int getSuperBlockStart(BinaryReader reader) {
        try {
            int padding = -1;
            int padStart = 0;
            boolean isPadding = false;
            while (padStart < 1024) {
                if (!isPadding) {
                    padStart = (int)reader.getPointerIndex();
                }
                if ((padding = reader.readNextInt()) == 0) {
                    if (isPadding) {
                        return padStart + 1024;
                    }
                    isPadding = true;
                    continue;
                }
                isPadding = false;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return -1;
    }
}

