/* api.c API unit tests
 *
 * Copyright (C) 2006-2023 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/* For AES-CBC, input lengths can optionally be validated to be a
 * multiple of the block size, by defining WOLFSSL_AES_CBC_LENGTH_CHECKS,
 * also available via the configure option --enable-aescbc-length-checks.
 */


/*----------------------------------------------------------------------------*
 | Includes
 *----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#include <wolfssl/wolfcrypt/settings.h>
#undef TEST_OPENSSL_COEXIST /* can't use this option with this example */

#ifndef FOURK_BUF
    #define FOURK_BUF 4096
#endif
#ifndef TWOK_BUF
    #define TWOK_BUF 2048
#endif
#ifndef ONEK_BUF
    #define ONEK_BUF 1024
#endif
#if defined(WOLFSSL_STATIC_MEMORY)
    #include <wolfssl/wolfcrypt/memory.h>

#if defined(WOLFSSL_STATIC_MEMORY) && !defined(WOLFCRYPT_ONLY)
    #if (defined(HAVE_ECC) && !defined(ALT_ECC_SIZE)) || \
         defined(SESSION_CERTS)
        #ifdef OPENSSL_EXTRA
            #define TEST_TLS_STATIC_MEMSZ (400000)
        #else
            #define TEST_TLS_STATIC_MEMSZ (320000)
        #endif
    #else
            #define TEST_TLS_STATIC_MEMSZ (80000)
    #endif
#endif

#endif /* WOLFSSL_STATIC_MEMORY */
#ifndef HEAP_HINT
    #define HEAP_HINT NULL
#endif /* WOLFSSL_STAIC_MEMORY */
#ifdef WOLFSSL_ASNC_CRYPT
    #include <wolfssl/wolfcrypt/async.h>
#endif
#ifdef HAVE_ECC
    #include <wolfssl/wolfcrypt/ecc.h>   /* wc_ecc_fp_free */
    #ifndef ECC_ASN963_MAX_BUF_SZ
        #define ECC_ASN963_MAX_BUF_SZ 133
    #endif
    #ifndef ECC_PRIV_KEY_BUF
        #define ECC_PRIV_KEY_BUF 66  /* For non user defined curves. */
    #endif
    /* ecc key sizes: 14, 16, 20, 24, 28, 30, 32, 40, 48, 64 */
    /* logic to choose right key ECC size */
    #if (defined(HAVE_ECC112) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 112
        #define KEY14 14
    #else
        #define KEY14 32
    #endif
    #if (defined(HAVE_ECC128) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 128
        #define KEY16 16
    #else
        #define KEY16 32
    #endif
    #if (defined(HAVE_ECC160) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 160
        #define KEY20 20
    #else
        #define KEY20 32
    #endif
    #if (defined(HAVE_ECC192) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 192
        #define KEY24 24
    #else
        #define KEY24 32
    #endif
    #if defined(HAVE_ECC224) || defined(HAVE_ALL_CURVES)
        #define KEY28 28
    #else
        #define KEY28 32
    #endif
    #if defined(HAVE_ECC239) || defined(HAVE_ALL_CURVES)
        #define KEY30 30
    #else
        #define KEY30 32
    #endif
    #define KEY32 32
    #if defined(HAVE_ECC320) || defined(HAVE_ALL_CURVES)
        #define KEY40 40
    #else
        #define KEY40 32
    #endif
    #if defined(HAVE_ECC384) || defined(HAVE_ALL_CURVES)
        #define KEY48 48
    #else
        #define KEY48 32
    #endif
    #if defined(HAVE_ECC512) || defined(HAVE_ALL_CURVES)
        #define KEY64 64
    #else
        #define KEY64 32
    #endif

    #if !defined(HAVE_COMP_KEY)
        #if !defined(NOCOMP)
            #define NOCOMP 0
        #endif
    #else
        #if !defined(COMP)
            #define COMP 1
        #endif
    #endif
    #if !defined(DER_SZ)
        #define DER_SZ(ks) ((ks) * 2 + 1)
    #endif
    #ifdef WOLFSSL_SM2
        #include <wolfssl/wolfcrypt/sm2.h>
    #endif
#endif
#ifndef NO_ASN
    #include <wolfssl/wolfcrypt/asn_public.h>
#endif
#include <wolfssl/error-ssl.h>

#include <stdlib.h>
#include <wolfssl/ssl.h>  /* compatibility layer */
#include <wolfssl/test.h>
#include <tests/unit.h>
#include "examples/server/server.h"
     /* for testing compatibility layer callbacks */

#ifndef NO_MD5
    #include <wolfssl/wolfcrypt/md5.h>
#endif
#ifndef NO_SHA
    #include <wolfssl/wolfcrypt/sha.h>
#endif
#ifndef NO_SHA256
    #include <wolfssl/wolfcrypt/sha256.h>
#endif
#ifdef WOLFSSL_SHA512
    #include <wolfssl/wolfcrypt/sha512.h>
#endif
#ifdef WOLFSSL_SHA384
    #include <wolfssl/wolfcrypt/sha512.h>
#endif

#ifdef WOLFSSL_SHA3
    #include <wolfssl/wolfcrypt/sha3.h>
    #ifndef HEAP_HINT
        #define HEAP_HINT   NULL
    #endif
#endif

#ifdef WOLFSSL_SM3
    #include <wolfssl/wolfcrypt/sm3.h>
#endif

#ifndef NO_AES
    #include <wolfssl/wolfcrypt/aes.h>
    #ifdef HAVE_AES_DECRYPT
        #include <wolfssl/wolfcrypt/wc_encrypt.h>
    #endif
#endif
#ifdef WOLFSSL_SM4
    #include <wolfssl/wolfcrypt/sm4.h>
#endif
#ifdef WOLFSSL_RIPEMD
    #include <wolfssl/wolfcrypt/ripemd.h>
#endif
#ifndef NO_DES3
    #include <wolfssl/wolfcrypt/des3.h>
    #include <wolfssl/wolfcrypt/wc_encrypt.h>
#endif
#ifdef WC_RC2
    #include <wolfssl/wolfcrypt/rc2.h>
#endif

#ifndef NO_HMAC
    #include <wolfssl/wolfcrypt/hmac.h>
#endif

#ifdef HAVE_CHACHA
    #include <wolfssl/wolfcrypt/chacha.h>
#endif

#ifdef HAVE_POLY1305
    #include <wolfssl/wolfcrypt/poly1305.h>
#endif

#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    #include <wolfssl/wolfcrypt/chacha20_poly1305.h>
#endif

#ifdef HAVE_CAMELLIA
    #include <wolfssl/wolfcrypt/camellia.h>
#endif

#ifndef NO_RC4
    #include <wolfssl/wolfcrypt/arc4.h>
#endif

#ifdef HAVE_BLAKE2
    #include <wolfssl/wolfcrypt/blake2.h>
#endif

#include <wolfssl/wolfcrypt/hash.h>
#ifndef NO_RSA
    #include <wolfssl/wolfcrypt/rsa.h>

    #define FOURK_BUF 4096
    #define GEN_BUF  294

    #ifndef USER_CRYPTO_ERROR
        #define USER_CRYPTO_ERROR (-101) /* error returned by IPP lib. */
    #endif
#endif

#ifndef NO_SIG_WRAPPER
    #include <wolfssl/wolfcrypt/signature.h>
#endif


#ifdef HAVE_AESCCM
    #include <wolfssl/wolfcrypt/aes.h>
#endif

#ifdef HAVE_PKCS7
    #include <wolfssl/wolfcrypt/pkcs7.h>
    #include <wolfssl/wolfcrypt/asn.h>
    #ifdef HAVE_LIBZ
    #include <wolfssl/wolfcrypt/compress.h>
    #endif
#endif

#ifdef WOLFSSL_SMALL_CERT_VERIFY
    #include <wolfssl/wolfcrypt/asn.h>
#endif

#ifndef NO_DSA
    #include <wolfssl/wolfcrypt/dsa.h>
    #ifndef ONEK_BUF
        #define ONEK_BUF 1024
    #endif
    #ifndef TWOK_BUF
        #define TWOK_BUF 2048
    #endif
    #ifndef FOURK_BUF
        #define FOURK_BUF 4096
    #endif
    #ifndef DSA_SIG_SIZE
        #define DSA_SIG_SIZE 40
    #endif
    #ifndef MAX_DSA_PARAM_SIZE
        #define MAX_DSA_PARAM_SIZE 256
    #endif
#endif

#ifdef WOLFSSL_CMAC
    #include <wolfssl/wolfcrypt/cmac.h>
#endif

#ifdef HAVE_ED25519
    #include <wolfssl/wolfcrypt/ed25519.h>
#endif
#ifdef HAVE_CURVE25519
    #include <wolfssl/wolfcrypt/curve25519.h>
#endif
#ifdef HAVE_ED448
    #include <wolfssl/wolfcrypt/ed448.h>
#endif
#ifdef HAVE_CURVE448
    #include <wolfssl/wolfcrypt/curve448.h>
#endif

#ifdef HAVE_PKCS12
    #include <wolfssl/wolfcrypt/pkcs12.h>
#endif

#include <wolfssl/wolfcrypt/logging.h>

#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL) || defined(OPENSSL_ALL))
    #include <wolfssl/openssl/ssl.h>
    #ifndef NO_ASN
        /* for ASN_COMMON_NAME DN_tags enum */
        #include <wolfssl/wolfcrypt/asn.h>
    #endif
    #ifdef HAVE_OCSP
        #include <wolfssl/openssl/ocsp.h>
    #endif
#endif
#ifdef OPENSSL_EXTRA
    #include <wolfssl/openssl/cmac.h>
    #include <wolfssl/openssl/x509v3.h>
    #include <wolfssl/openssl/asn1.h>
    #include <wolfssl/openssl/crypto.h>
    #include <wolfssl/openssl/pkcs12.h>
    #include <wolfssl/openssl/evp.h>
    #include <wolfssl/openssl/dh.h>
    #include <wolfssl/openssl/bn.h>
    #include <wolfssl/openssl/buffer.h>
    #include <wolfssl/openssl/pem.h>
    #include <wolfssl/openssl/ec.h>
    #include <wolfssl/openssl/ecdh.h>
    #include <wolfssl/openssl/engine.h>
    #include <wolfssl/openssl/hmac.h>
    #include <wolfssl/openssl/objects.h>
    #include <wolfssl/openssl/rand.h>
    #include <wolfssl/openssl/modes.h>
    #include <wolfssl/openssl/fips_rand.h>
    #include <wolfssl/openssl/kdf.h>
#ifdef OPENSSL_ALL
    #include <wolfssl/openssl/txt_db.h>
    #include <wolfssl/openssl/lhash.h>
#endif
#ifndef NO_AES
    #include <wolfssl/openssl/aes.h>
#endif
#ifndef NO_DES3
    #include <wolfssl/openssl/des.h>
#endif
#ifdef HAVE_ECC
    #include <wolfssl/openssl/ecdsa.h>
#endif
#ifdef HAVE_PKCS7
    #include <wolfssl/openssl/pkcs7.h>
#endif
#ifdef HAVE_ED25519
    #include <wolfssl/openssl/ed25519.h>
#endif
#ifdef HAVE_ED448
    #include <wolfssl/openssl/ed448.h>
#endif
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) \
    && !defined(NO_SHA256) && !defined(RC_NO_RNG)
        #include <wolfssl/wolfcrypt/srp.h>
#endif

#if (defined(SESSION_CERTS) && defined(TEST_PEER_CERT_CHAIN)) || \
    defined(HAVE_SESSION_TICKET) || (defined(OPENSSL_EXTRA) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)) || \
    defined(WOLFSSL_TEST_STATIC_BUILD) || defined(WOLFSSL_DTLS) || \
    defined(HAVE_ECH) || defined(HAVE_EX_DATA) || !defined(NO_SESSION_CACHE) \
    || !defined(WOLFSSL_NO_TLS12)
    /* for testing SSL_get_peer_cert_chain, or SESSION_TICKET_HINT_DEFAULT,
     * for setting authKeyIdSrc in WOLFSSL_X509, or testing DTLS sequence
     * number tracking */
#include "wolfssl/internal.h"
#endif

/* force enable test buffers */
#ifndef USE_CERT_BUFFERS_2048
    #define USE_CERT_BUFFERS_2048
#endif
#ifndef USE_CERT_BUFFERS_256
    #define USE_CERT_BUFFERS_256
#endif
#include <wolfssl/certs_test.h>

#include "tests/utils.h"

#ifndef WOLFSSL_HAVE_ECC_KEY_GET_PRIV
    /* FIPS build has replaced ecc.h. */
    #define wc_ecc_key_get_priv(key) (&((key)->k))
    #define WOLFSSL_HAVE_ECC_KEY_GET_PRIV
#endif

typedef struct testVector {
    const char* input;
    const char* output;
    size_t inLen;
    size_t outLen;

} testVector;

#if defined(HAVE_PKCS7)
    typedef struct {
        const byte* content;
        word32      contentSz;
        int         contentOID;
        int         encryptOID;
        int         keyWrapOID;
        int         keyAgreeOID;
        byte*       cert;
        size_t      certSz;
        byte*       privateKey;
        word32      privateKeySz;
    } pkcs7EnvelopedVector;

    #ifndef NO_PKCS7_ENCRYPTED_DATA
        typedef struct {
            const byte*     content;
            word32          contentSz;
            int             contentOID;
            int             encryptOID;
            byte*           encryptionKey;
            word32          encryptionKeySz;
        } pkcs7EncryptedVector;
    #endif
#endif /* HAVE_PKCS7 */

typedef int (*ctx_cb)(WOLFSSL_CTX* ctx);
typedef int (*ssl_cb)(WOLFSSL* ssl);
typedef int (*test_cbType)(WOLFSSL_CTX *ctx, WOLFSSL *ssl);

typedef struct test_ssl_cbf {
    method_provider method;
    ctx_cb ctx_ready;
    ssl_cb ssl_ready;
    ssl_cb on_result;
    ssl_cb on_cleanup;
    WOLFSSL_CTX* ctx;
    const char* caPemFile;
    const char* certPemFile;
    const char* keyPemFile;
    const char* crlPemFile;
#ifdef WOLFSSL_STATIC_MEMORY
    byte*               mem;
    word32              memSz;
    wolfSSL_method_func method_ex;
#endif
    int devId;
    int return_code;
    int last_err;
    unsigned char isSharedCtx:1;
    unsigned char loadToSSL:1;
    unsigned char ticNoInit:1;
    unsigned char doUdp:1;
} test_ssl_cbf;

#define TEST_SSL_MEMIO_BUF_SZ   (64 * 1024)
typedef struct test_ssl_memio_ctx {
    WOLFSSL_CTX* s_ctx;
    WOLFSSL_CTX* c_ctx;
    WOLFSSL* s_ssl;
    WOLFSSL* c_ssl;

    const char* c_ciphers;
    const char* s_ciphers;

    char* c_msg;
    int c_msglen;
    char* s_msg;
    int s_msglen;

    test_ssl_cbf s_cb;
    test_ssl_cbf c_cb;

    byte c_buff[TEST_SSL_MEMIO_BUF_SZ];
    int c_len;
    byte s_buff[TEST_SSL_MEMIO_BUF_SZ];
    int s_len;
} test_ssl_memio_ctx;

int test_wolfSSL_client_server_nofail_memio(test_ssl_cbf* client_cb,
    test_ssl_cbf* server_cb, test_cbType client_on_handshake);

#ifdef WOLFSSL_DUMP_MEMIO_STREAM
const char* currentTestName;
char tmpDirName[16];
int tmpDirNameSet = 0;
#endif

/*----------------------------------------------------------------------------*
 | Constants
 *----------------------------------------------------------------------------*/

/* Test result constants and macros. */

/* Test succeeded. */
#define TEST_SUCCESS    (1)
/* Test failed. */
#define TEST_FAIL       (0)
/* Test skipped - not run. */
#define TEST_SKIPPED    (-7777)

/* Returns the result based on whether check is true.
 *
 * @param [in] check  Condition for success.
 * @return  When condition is true: TEST_SUCCESS.
 * @return  When condition is false: TEST_FAIL.
 */
#ifdef DEBUG_WOLFSSL_VERBOSE
#define XSTRINGIFY(s) STRINGIFY(s)
#define STRINGIFY(s)  #s
#define TEST_RES_CHECK(check) ({ \
    int _ret = (check) ? TEST_SUCCESS : TEST_FAIL; \
    if (_ret == TEST_FAIL) { \
        fprintf(stderr, " check \"%s\" at %d ", \
            XSTRINGIFY(check), __LINE__); \
    } \
    _ret; })
#else
#define TEST_RES_CHECK(check) \
    ((check) ? TEST_SUCCESS : TEST_FAIL)
#endif /* DEBUG_WOLFSSL_VERBOSE */

#define TEST_STRING    "Everyone gets Friday off."
#define TEST_STRING_SZ 25

#if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
#define TEST_RSA_BITS 1024
#else
#define TEST_RSA_BITS 2048
#endif
#define TEST_RSA_BYTES (TEST_RSA_BITS/8)

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    (!defined(NO_WOLFSSL_SERVER) || !defined(NO_WOLFSSL_CLIENT))
    static const char* bogusFile  =
    #ifdef _WIN32
        "NUL"
    #else
        "/dev/null"
    #endif
    ;
#endif /* !NO_FILESYSTEM && !NO_CERTS && (!NO_WOLFSSL_SERVER || !NO_WOLFSSL_CLIENT) */

enum {
    TESTING_RSA = 1,
    TESTING_ECC = 2
};

#ifdef WOLFSSL_QNX_CAAM
#include <wolfssl/wolfcrypt/port/caam/wolfcaam.h>
static int testDevId = WOLFSSL_CAAM_DEVID;
#else
static int testDevId = INVALID_DEVID;
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_RSA)        && !defined(SINGLE_THREADED) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_WOLFSSL_CLIENT)
#define HAVE_IO_TESTS_DEPENDENCIES
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_WOLFSSL_CLIENT)
#define HAVE_SSL_MEMIO_TESTS_DEPENDENCIES
#endif

/*----------------------------------------------------------------------------*
 | BIO with fixed read/write size
 *----------------------------------------------------------------------------*/

#if defined(OPENSSL_EXTRA) && !defined(NO_BIO)

static int wolfssl_bio_s_fixed_mem_write(WOLFSSL_BIO* bio, const char* data,
    int len)
{
    if ((bio == NULL) || (bio->ptr == NULL) || (data == NULL)) {
        len = 0;
    }
    else {
        if (bio->wrSz - bio->wrIdx < len) {
            len = bio->wrSz - bio->wrIdx;
        }
        XMEMCPY((char*)bio->ptr + bio->wrIdx, data, len);
        bio->wrIdx += len;
    }

    return len;
}

static int wolfssl_bio_s_fixed_mem_read(WOLFSSL_BIO* bio, char* data, int len)
{
    if ((bio == NULL) || (bio->ptr == NULL) || (data == NULL)) {
        len = 0;
    }
    else {
        if (bio->wrSz - bio->rdIdx < len) {
            len = bio->wrSz - bio->rdIdx;
        }
        XMEMCPY(data, (char*)bio->ptr + bio->rdIdx, len);
        bio->rdIdx += len;
    }

    return len;
}

static WOLFSSL_BIO_METHOD* wolfSSL_BIO_s_fixed_mem(void)
{
    static WOLFSSL_BIO_METHOD meth;

    meth.type = WOLFSSL_BIO_BIO;
    XMEMCPY(meth.name, "Fixed Memory Size", 18);
    meth.writeCb = wolfssl_bio_s_fixed_mem_write;
    meth.readCb = wolfssl_bio_s_fixed_mem_read;

    return &meth;
}

#endif

/*----------------------------------------------------------------------------*
 | Setup
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_Init(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfSSL_Init(), WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}


static int test_wolfSSL_Cleanup(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfSSL_Cleanup(), WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}


/*  Initialize the wolfCrypt state.
 *  POST: 0 success.
 */
static int test_wolfCrypt_Init(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfCrypt_Init(), 0);
    return EXPECT_RESULT();

} /* END test_wolfCrypt_Init */

static int test_wolfCrypt_Cleanup(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfCrypt_Cleanup(), 0);
    return EXPECT_RESULT();
}

/*----------------------------------------------------------------------------*
 | Platform dependent function test
 *----------------------------------------------------------------------------*/
static int test_fileAccess(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_TEST_PLATFORMDEPEND) && !defined(NO_FILESYSTEM)
    const char *fname[] = {
        svrCertFile, svrKeyFile, caCertFile,
        eccCertFile, eccKeyFile, eccRsaCertFile,
        cliCertFile, cliCertDerFile, cliKeyFile,
        dhParamFile,
        cliEccKeyFile, cliEccCertFile, caEccCertFile, edCertFile, edKeyFile,
        cliEdCertFile, cliEdKeyFile, caEdCertFile,
        NULL
    };
    const char derfile[] = "./certs/server-cert.der";
    XFILE f = XBADFILE;
    size_t sz;
    byte *buff = NULL;
    int i;

    ExpectTrue(XFOPEN("badfilename", "rb") == XBADFILE);
    for (i=0; EXPECT_SUCCESS() && fname[i] != NULL ; i++) {
        ExpectTrue((f = XFOPEN(fname[i], "rb")) != XBADFILE);
        XFCLOSE(f);
    }

    ExpectTrue((f = XFOPEN(derfile, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(f, 0, XSEEK_END) == 0);
    ExpectIntGE(sz = (size_t) XFTELL(f), sizeof_server_cert_der_2048);
    ExpectTrue(XFSEEK(f, 0, XSEEK_SET) == 0);
    ExpectTrue((buff = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE)) != NULL);
    ExpectTrue(XFREAD(buff, 1, sz, f) == sz);
    ExpectIntEQ(XMEMCMP(server_cert_der_2048, buff, sz), 0);
    XFREE(buff, NULL, DYNAMIC_TYPE_FILE);
    XFCLOSE(f);
#endif
    return EXPECT_RESULT();
}

/*----------------------------------------------------------------------------*
 | Method Allocators
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_Method_Allocators(void)
{
    EXPECT_DECLS;

    #define TEST_METHOD_ALLOCATOR(allocator, condition) \
        do {                                            \
            WOLFSSL_METHOD *method = NULL;              \
            condition(method = allocator());            \
            XFREE(method, 0, DYNAMIC_TYPE_METHOD);      \
        } while (0)

    #define TEST_VALID_METHOD_ALLOCATOR(a) \
            TEST_METHOD_ALLOCATOR(a, ExpectNotNull)

    #define TEST_INVALID_METHOD_ALLOCATOR(a) \
            TEST_METHOD_ALLOCATOR(a, ExpectNull)

#ifndef NO_OLD_TLS
    #ifdef WOLFSSL_ALLOW_SSLV3
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfSSLv3_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfSSLv3_client_method);
        #endif
    #endif
    #ifdef WOLFSSL_ALLOW_TLSV10
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_client_method);
        #endif
    #endif
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_client_method);
    #endif
#endif /* !NO_OLD_TLS */

#ifndef WOLFSSL_NO_TLS12
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_client_method);
    #endif
#endif /* !WOLFSSL_NO_TLS12 */

#ifdef WOLFSSL_TLS13
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_client_method);
    #endif
#endif /* WOLFSSL_TLS13 */

#ifndef NO_WOLFSSL_SERVER
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_server_method);
#endif
#ifndef NO_WOLFSSL_CLIENT
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_client_method);
#endif

#ifdef WOLFSSL_DTLS
    #ifndef NO_OLD_TLS
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_client_method);
        #endif
    #endif
    #ifndef WOLFSSL_NO_TLS12
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_client_method);
        #endif
    #endif
#endif /* WOLFSSL_DTLS */

#if !defined(NO_OLD_TLS) && defined(OPENSSL_EXTRA)
    /* Stubs */
    #ifndef NO_WOLFSSL_SERVER
        TEST_INVALID_METHOD_ALLOCATOR(wolfSSLv2_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_INVALID_METHOD_ALLOCATOR(wolfSSLv2_client_method);
    #endif
#endif

    /* Test Either Method (client or server) */
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_method);
    #ifndef NO_OLD_TLS
        #ifdef WOLFSSL_ALLOW_TLSV10
            TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_method);
        #endif
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_method);
    #endif /* !NO_OLD_TLS */
    #ifndef WOLFSSL_NO_TLS12
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_method);
    #endif /* !WOLFSSL_NO_TLS12 */
    #ifdef WOLFSSL_TLS13
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_method);
    #endif /* WOLFSSL_TLS13 */
    #ifdef WOLFSSL_DTLS
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLS_method);
        #ifndef NO_OLD_TLS
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_method);
        #endif /* !NO_OLD_TLS */
        #ifndef WOLFSSL_NO_TLS12
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_method);
        #endif /* !WOLFSSL_NO_TLS12 */
    #endif /* WOLFSSL_DTLS */
#endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */

    return EXPECT_RESULT();
}


/*----------------------------------------------------------------------------*
 | Context
 *----------------------------------------------------------------------------*/
#ifndef NO_WOLFSSL_SERVER
static int test_wolfSSL_CTX_new(void)
{
    EXPECT_DECLS;
    WOLFSSL_CTX *ctx;
    WOLFSSL_METHOD* method;

    ExpectNull(ctx = wolfSSL_CTX_new(NULL));
    ExpectNotNull(method = wolfSSLv23_server_method());
    ExpectNotNull(ctx = wolfSSL_CTX_new(method));

    wolfSSL_CTX_free(ctx);

    return EXPECT_RESULT();
}
#endif

#if (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_FILESYSTEM)
static int test_for_double_Free(void)
{
    EXPECT_DECLS;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;
    int skipTest = 0;
    const char* testCertFile;
    const char* testKeyFile;
    char optionsCiphers[] = "RC4-SHA:RC4-MD5:DES-CBC3-SHA:AES128-SHA:AES256-SHA"
":NULL-SHA:NULL-SHA256:DHE-RSA-AES128-SHA:DHE-RSA-AES256-SHA:DHE-PSK-AES256-GCM"
"-SHA384:DHE-PSK-AES128-GCM-SHA256:PSK-AES256-GCM-SHA384:PSK-AES128-GCM-SHA256:"
"DHE-PSK-AES256-CBC-SHA384:DHE-PSK-AES128-CBC-SHA256:PSK-AES256-CBC-SHA384:PSK-"
"AES128-CBC-SHA256:PSK-AES128-CBC-SHA:PSK-AES256-CBC-SHA:DHE-PSK-AES128-CCM:DHE"
"-PSK-AES256-CCM:PSK-AES128-CCM:PSK-AES256-CCM:PSK-AES128-CCM-8:PSK-AES256-CCM-"
"8:DHE-PSK-NULL-SHA384:DHE-PSK-NULL-SHA256:PSK-NULL-SHA384:PSK-NULL-SHA256:PSK-"
"NULL-SHA:AES128-CCM-8:AES256-CCM-8:ECDHE-ECDSA-"
"AES128-CCM:ECDHE-ECDSA-AES128-CCM-8:ECDHE-ECDSA-AES256-CCM-8:ECDHE-RSA-AES128-"
"SHA:ECDHE-RSA-AES256-SHA:ECDHE-ECDSA-AES128-SHA:ECDHE-ECDSA-AES256-SHA:ECDHE-R"
"SA-RC4-SHA:ECDHE-RSA-DES-CBC3-SHA:ECDHE-ECDSA-RC4-SHA:ECDHE-ECDSA-DES-CBC3-SHA"
":AES128-SHA256:AES256-SHA256:DHE-RSA-AES128-SHA256:DHE-RSA-AES256-SHA256:ECDH-"
"RSA-AES128-SHA:ECDH-RSA-AES256-SHA:ECDH-ECDSA-AES128-SHA:ECDH-ECDSA-AES256-SHA"
":ECDH-RSA-RC4-SHA:ECDH-RSA-DES-CBC3-SHA:ECDH-ECDSA-RC4-SHA:ECDH-ECDSA-DES-CBC3"
"-SHA:AES128-GCM-SHA256:AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES"
"256-GCM-SHA384:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-E"
"CDSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDH-RSA-AES128-GCM-SHA25"
"6:ECDH-RSA-AES256-GCM-SHA384:ECDH-ECDSA-AES128-GCM-SHA256:ECDH-ECDSA-AES256-GC"
"M-SHA384:CAMELLIA128-SHA:DHE-RSA-CAMELLIA128-SHA:CAMELLIA256-SHA:DHE-RSA-CAMEL"
"LIA256-SHA:CAMELLIA128-SHA256:DHE-RSA-CAMELLIA128-SHA256:CAMELLIA256-SHA256:DH"
"E-RSA-CAMELLIA256-SHA256:ECDHE-RSA-AES128-SHA256:ECDHE-ECDSA-AES128-SHA256:ECD"
"H-RSA-AES128-SHA256:ECDH-ECDSA-AES128-SHA256:ECDHE-RSA-AES256-SHA384:ECDHE-ECD"
"SA-AES256-SHA384:ECDH-RSA-AES256-SHA384:ECDH-ECDSA-AES256-SHA384:ECDHE-RSA-CHA"
"CHA20-POLY1305:ECDHE-ECDSA-CHACHA20-POLY1305:DHE-RSA-CHACHA20-POLY1305:ECDHE-R"
"SA-CHACHA20-POLY1305-OLD:ECDHE-ECDSA-CHACHA20-POLY1305-OLD:DHE-RSA-CHACHA20-PO"
"LY1305-OLD:ECDHE-ECDSA-NULL-SHA:ECDHE-PSK-NULL-SHA256:ECDHE-PSK-A"
"ES128-CBC-SHA256:PSK-CHACHA20-POLY1305:ECDHE-PSK-CHACHA20-POLY1305:DHE-PSK-CHA"
"CHA20-POLY1305:EDH-RSA-DES-CBC3-SHA:TLS13-AES128-GCM-SHA256:TLS13-AES256-GCM-S"
"HA384:TLS13-CHACHA20-POLY1305-SHA256:TLS13-AES128-CCM-SHA256:TLS13-AES128-CCM-"
"8-SHA256:TLS13-SHA256-SHA256:TLS13-SHA384-SHA384";
    /* OpenVPN uses a "blacklist" method to specify which ciphers NOT to use */
#ifdef OPENSSL_EXTRA
    char openvpnCiphers[] = "DEFAULT:!EXP:!LOW:!MEDIUM:!kDH:!kECDH:!DSS:!PSK:"
                            "!SRP:!kRSA:!aNULL:!eNULL";
#endif

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#else
    skipTest = 1;
#endif

    if (skipTest != 1) {
#ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectNotNull(ssl = wolfSSL_new(ctx));

        /* First test freeing SSL, then CTX */
        wolfSSL_free(ssl);
        ssl = NULL;
        wolfSSL_CTX_free(ctx);
        ctx = NULL;

#ifndef NO_WOLFSSL_CLIENT
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectNotNull(ssl = wolfSSL_new(ctx));

        /* Next test freeing CTX then SSL */
        wolfSSL_CTX_free(ctx);
        ctx = NULL;
        wolfSSL_free(ssl);
        ssl = NULL;

#ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
        /* Test setting ciphers at ctx level */
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, optionsCiphers));
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_TLS13) && defined(HAVE_AESGCM) && \
        defined(WOLFSSL_SHA384) && defined(WOLFSSL_AES_256)
        /* only update TLSv13 suites */
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "TLS13-AES256-GCM-SHA384"));
#endif
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(HAVE_AESGCM) && \
    !defined(NO_SHA256) && !defined(WOLFSSL_NO_TLS12) && \
    defined(WOLFSSL_AES_128) && !defined(NO_RSA)
        /* only update pre-TLSv13 suites */
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx,
            "ECDHE-RSA-AES128-GCM-SHA256"));
#endif
#ifdef OPENSSL_EXTRA
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, openvpnCiphers));
#endif
        ExpectNotNull(ssl = wolfSSL_new(ctx));
        wolfSSL_CTX_free(ctx);
        ctx = NULL;
        wolfSSL_free(ssl);
        ssl = NULL;

#ifndef NO_WOLFSSL_CLIENT
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectNotNull(ssl = wolfSSL_new(ctx));
        /* test setting ciphers at SSL level */
        ExpectTrue(wolfSSL_set_cipher_list(ssl, optionsCiphers));
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_TLS13) && defined(HAVE_AESGCM) && \
        defined(WOLFSSL_SHA384) && defined(WOLFSSL_AES_256)
        /* only update TLSv13 suites */
        ExpectTrue(wolfSSL_set_cipher_list(ssl, "TLS13-AES256-GCM-SHA384"));
#endif
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(HAVE_AESGCM) && \
    !defined(NO_SHA256) && !defined(WOLFSSL_NO_TLS12) && \
    defined(WOLFSSL_AES_128) && !defined(NO_RSA)
        /* only update pre-TLSv13 suites */
        ExpectTrue(wolfSSL_set_cipher_list(ssl, "ECDHE-RSA-AES128-GCM-SHA256"));
#endif
        wolfSSL_CTX_free(ctx);
        ctx = NULL;
        wolfSSL_free(ssl);
        ssl = NULL;
    }

    return EXPECT_RESULT();
}
#endif


static int test_wolfSSL_CTX_set_cipher_list_bytes(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(WOLFSSL_SET_CIPHER_BYTES)) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_FILESYSTEM)
    const char* testCertFile;
    const char* testKeyFile;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;

    const byte cipherList[] =
    {
        /* TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA */ 0xC0, 0x16,
        /* TLS_DHE_RSA_WITH_AES_256_CBC_SHA  */ 0xC0, 0x39,
        /* TLS_DHE_RSA_WITH_AES_128_CBC_SHA  */ 0xC0, 0x33,
        /* TLS_DH_anon_WITH_AES_128_CBC_SHA  */ 0xC0, 0x34,
        /* TLS_RSA_WITH_AES_256_CBC_SHA      */ 0xC0, 0x35,
        /* TLS_RSA_WITH_AES_128_CBC_SHA      */ 0xC0, 0x2F,
        /* TLS_RSA_WITH_NULL_MD5             */ 0xC0, 0x01,
        /* TLS_RSA_WITH_NULL_SHA             */ 0xC0, 0x02,
        /* TLS_PSK_WITH_AES_256_CBC_SHA      */ 0xC0, 0x8d,
        /* TLS_PSK_WITH_AES_128_CBC_SHA256   */ 0xC0, 0xae,
        /* TLS_PSK_WITH_AES_256_CBC_SHA384   */ 0xC0, 0xaf,
        /* TLS_PSK_WITH_AES_128_CBC_SHA      */ 0xC0, 0x8c,
        /* TLS_PSK_WITH_NULL_SHA256          */ 0xC0, 0xb0,
        /* TLS_PSK_WITH_NULL_SHA384          */ 0xC0, 0xb1,
        /* TLS_PSK_WITH_NULL_SHA             */ 0xC0, 0x2c,
        /* SSL_RSA_WITH_RC4_128_SHA          */ 0xC0, 0x05,
        /* SSL_RSA_WITH_RC4_128_MD5          */ 0xC0, 0x04,
        /* SSL_RSA_WITH_3DES_EDE_CBC_SHA     */ 0xC0, 0x0A,

        /* ECC suites, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA     */ 0xC0, 0x14,
        /* TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA     */ 0xC0, 0x13,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA   */ 0xC0, 0x0A,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA   */ 0xC0, 0x09,
        /* TLS_ECDHE_RSA_WITH_RC4_128_SHA         */ 0xC0, 0x11,
        /* TLS_ECDHE_ECDSA_WITH_RC4_128_SHA       */ 0xC0, 0x07,
        /* TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA    */ 0xC0, 0x12,
        /* TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA  */ 0xC0, 0x08,
        /* TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256  */ 0xC0, 0x27,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256*/ 0xC0, 0x23,
        /* TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384  */ 0xC0, 0x28,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384*/ 0xC0, 0x24,
        /* TLS_ECDHE_ECDSA_WITH_NULL_SHA          */ 0xC0, 0x06,
        /* TLS_ECDHE_PSK_WITH_NULL_SHA256         */ 0xC0, 0x3a,
        /* TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256  */ 0xC0, 0x37,

        /* static ECDH, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDH_RSA_WITH_AES_256_CBC_SHA      */ 0xC0, 0x0F,
        /* TLS_ECDH_RSA_WITH_AES_128_CBC_SHA      */ 0xC0, 0x0E,
        /* TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA    */ 0xC0, 0x05,
        /* TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA    */ 0xC0, 0x04,
        /* TLS_ECDH_RSA_WITH_RC4_128_SHA          */ 0xC0, 0x0C,
        /* TLS_ECDH_ECDSA_WITH_RC4_128_SHA        */ 0xC0, 0x02,
        /* TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA     */ 0xC0, 0x0D,
        /* TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA   */ 0xC0, 0x03,
        /* TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256   */ 0xC0, 0x29,
        /* TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256 */ 0xC0, 0x25,
        /* TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384   */ 0xC0, 0x2A,
        /* TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384 */ 0xC0, 0x26,

        /* WDM_WITH_NULL_SHA256 */ 0x00, 0xFE, /* wolfSSL DTLS Multicast */

        /* SHA256 */
        /* TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 */ 0x00, 0x6b,
        /* TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 */ 0x00, 0x67,
        /* TLS_RSA_WITH_AES_256_CBC_SHA256     */ 0x00, 0x3d,
        /* TLS_RSA_WITH_AES_128_CBC_SHA256     */ 0x00, 0x3c,
        /* TLS_RSA_WITH_NULL_SHA256            */ 0x00, 0x3b,
        /* TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 */ 0x00, 0xb2,
        /* TLS_DHE_PSK_WITH_NULL_SHA256        */ 0x00, 0xb4,

        /* SHA384 */
        /* TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 */ 0x00, 0xb3,
        /* TLS_DHE_PSK_WITH_NULL_SHA384        */ 0x00, 0xb5,

        /* AES-GCM */
        /* TLS_RSA_WITH_AES_128_GCM_SHA256      */ 0x00, 0x9c,
        /* TLS_RSA_WITH_AES_256_GCM_SHA384      */ 0x00, 0x9d,
        /* TLS_DHE_RSA_WITH_AES_128_GCM_SHA256  */ 0x00, 0x9e,
        /* TLS_DHE_RSA_WITH_AES_256_GCM_SHA384  */ 0x00, 0x9f,
        /* TLS_DH_anon_WITH_AES_256_GCM_SHA384  */ 0x00, 0xa7,
        /* TLS_PSK_WITH_AES_128_GCM_SHA256      */ 0x00, 0xa8,
        /* TLS_PSK_WITH_AES_256_GCM_SHA384      */ 0x00, 0xa9,
        /* TLS_DHE_PSK_WITH_AES_128_GCM_SHA256  */ 0x00, 0xaa,
        /* TLS_DHE_PSK_WITH_AES_256_GCM_SHA384  */ 0x00, 0xab,

        /* ECC AES-GCM, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 */ 0xC0, 0x2b,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384 */ 0xC0, 0x2c,
        /* TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256  */ 0xC0, 0x2d,
        /* TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384  */ 0xC0, 0x2e,
        /* TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256   */ 0xC0, 0x2f,
        /* TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384   */ 0xC0, 0x30,
        /* TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256    */ 0xC0, 0x31,
        /* TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384    */ 0xC0, 0x32,

        /* AES-CCM, first byte is 0xC0 but isn't ECC,
         * also, in some of the other AES-CCM suites
         * there will be second byte number conflicts
         * with non-ECC AES-GCM */
        /* TLS_RSA_WITH_AES_128_CCM_8         */ 0xC0, 0xa0,
        /* TLS_RSA_WITH_AES_256_CCM_8         */ 0xC0, 0xa1,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CCM   */ 0xC0, 0xac,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8 */ 0xC0, 0xae,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8 */ 0xC0, 0xaf,
        /* TLS_PSK_WITH_AES_128_CCM           */ 0xC0, 0xa4,
        /* TLS_PSK_WITH_AES_256_CCM           */ 0xC0, 0xa5,
        /* TLS_PSK_WITH_AES_128_CCM_8         */ 0xC0, 0xa8,
        /* TLS_PSK_WITH_AES_256_CCM_8         */ 0xC0, 0xa9,
        /* TLS_DHE_PSK_WITH_AES_128_CCM       */ 0xC0, 0xa6,
        /* TLS_DHE_PSK_WITH_AES_256_CCM       */ 0xC0, 0xa7,

        /* Camellia */
        /* TLS_RSA_WITH_CAMELLIA_128_CBC_SHA        */ 0x00, 0x41,
        /* TLS_RSA_WITH_CAMELLIA_256_CBC_SHA        */ 0x00, 0x84,
        /* TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256     */ 0x00, 0xba,
        /* TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256     */ 0x00, 0xc0,
        /* TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA    */ 0x00, 0x45,
        /* TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA    */ 0x00, 0x88,
        /* TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 */ 0x00, 0xbe,
        /* TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 */ 0x00, 0xc4,

        /* chacha20-poly1305 suites first byte is 0xCC (CHACHA_BYTE) */
        /* TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256   */ 0xCC, 0xa8,
        /* TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 */ 0xCC, 0xa9,
        /* TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256     */ 0xCC, 0xaa,
        /* TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256   */ 0xCC, 0xac,
        /* TLS_PSK_WITH_CHACHA20_POLY1305_SHA256         */ 0xCC, 0xab,
        /* TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256     */ 0xCC, 0xad,

        /* chacha20-poly1305 earlier version of nonce and padding (CHACHA_BYTE) */
        /* TLS_ECDHE_RSA_WITH_CHACHA20_OLD_POLY1305_SHA256   */ 0xCC, 0x13,
        /* TLS_ECDHE_ECDSA_WITH_CHACHA20_OLD_POLY1305_SHA256 */ 0xCC, 0x14,
        /* TLS_DHE_RSA_WITH_CHACHA20_OLD_POLY1305_SHA256     */ 0xCC, 0x15,

        /* ECDHE_PSK RFC8442, first byte is 0xD0 (ECDHE_PSK_BYTE) */
        /* TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256 */ 0xD0, 0x01,

        /* TLS v1.3 cipher suites */
        /* TLS_AES_128_GCM_SHA256       */ 0x13, 0x01,
        /* TLS_AES_256_GCM_SHA384       */ 0x13, 0x02,
        /* TLS_CHACHA20_POLY1305_SHA256 */ 0x13, 0x03,
        /* TLS_AES_128_CCM_SHA256       */ 0x13, 0x04,
        /* TLS_AES_128_CCM_8_SHA256     */ 0x13, 0x05,

        /* TLS v1.3 Integrity only cipher suites - 0xC0 (ECC) first byte */
        /* TLS_SHA256_SHA256 */ 0xC0, 0xB4,
        /* TLS_SHA384_SHA384 */ 0xC0, 0xB5
    };

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#endif

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectTrue(wolfSSL_CTX_set_cipher_list_bytes(ctx, &cipherList[0U],
                                                           sizeof(cipherList)));

    wolfSSL_CTX_free(ctx);
    ctx = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));

    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectTrue(wolfSSL_set_cipher_list_bytes(ssl, &cipherList[0U],
                                                           sizeof(cipherList)));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* (OPENSSL_EXTRA || WOLFSSL_SET_CIPHER_BYTES) &&
    (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) && (!NO_RSA || HAVE_ECC) */

    return EXPECT_RESULT();
}


static int test_wolfSSL_CTX_use_certificate_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(NULL, svrCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    /* invalid cert file */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(ctx, bogusFile,
                                                         WOLFSSL_FILETYPE_PEM));
    /* invalid cert type */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, 9999));

#ifdef NO_RSA
    /* rsa needed */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
#else
    /* success */
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO)) && !defined(NO_RSA)
static int test_wolfSSL_CTX_use_certificate_ASN1(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_WOLFSSL_SERVER) && !defined(NO_ASN)
    WOLFSSL_CTX* ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(ctx, sizeof_server_cert_der_2048,
        server_cert_der_2048), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}
#endif /* (OPENSSL_ALL || WOLFSSL_ASIO) && !NO_RSA */

/*  Test function for wolfSSL_CTX_use_certificate_buffer. Load cert into
 *  context using buffer.
 *  PRE: NO_CERTS not defined; USE_CERT_BUFFERS_2048 defined; compile with
 *  --enable-testcert flag.
 */
static int test_wolfSSL_CTX_use_certificate_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && defined(USE_CERT_BUFFERS_2048) && \
        !defined(NO_RSA) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX* ctx = NULL;
    int          ret;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectIntEQ(ret = wolfSSL_CTX_use_certificate_buffer(ctx,
        server_cert_der_2048, sizeof_server_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();

} /* END test_wolfSSL_CTX_use_certificate_buffer */

static int test_wolfSSL_CTX_use_PrivateKey_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(NULL, svrKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    /* invalid key file */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, bogusFile,
                                                         WOLFSSL_FILETYPE_PEM));
    /* invalid key type */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, 9999));

    /* success */
#ifdef NO_RSA
    /* rsa needed */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
#else
    /* success */
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}


/* test both file and buffer versions along with unloading trusted peer certs */
static int test_wolfSSL_CTX_trust_peer_cert(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && defined(WOLFSSL_TRUST_PEER_CERT) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL* ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

#if !defined(NO_FILESYSTEM)
    /* invalid file */
    ExpectIntNE(wolfSSL_CTX_trust_peer_cert(ctx, NULL,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CTX_trust_peer_cert(ctx, bogusFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CTX_trust_peer_cert(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
    ExpectIntEQ(wolfSSL_CTX_trust_peer_cert(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* unload cert */
    ExpectIntNE(wolfSSL_CTX_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_Unload_trust_peers(ctx), WOLFSSL_SUCCESS);

    /* invalid file */
    ExpectIntNE(wolfSSL_trust_peer_cert(ssl, NULL,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_trust_peer_cert(ssl, bogusFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_trust_peer_cert(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
    ExpectIntEQ(wolfSSL_trust_peer_cert(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    #ifdef WOLFSSL_LOCAL_X509_STORE
    /* unload cert */
    ExpectIntNE(wolfSSL_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_Unload_trust_peers(ssl), WOLFSSL_SUCCESS);
    #endif
#endif

    /* Test of loading certs from buffers */

    /* invalid buffer */
    ExpectIntNE(wolfSSL_CTX_trust_peer_buffer(ctx, NULL, -1,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
#ifdef USE_CERT_BUFFERS_1024
    ExpectIntEQ(wolfSSL_CTX_trust_peer_buffer(ctx, client_cert_der_1024,
          sizeof_client_cert_der_1024, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
#ifdef USE_CERT_BUFFERS_2048
    ExpectIntEQ(wolfSSL_CTX_trust_peer_buffer(ctx, client_cert_der_2048,
          sizeof_client_cert_der_2048, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* unload cert */
    ExpectIntNE(wolfSSL_CTX_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_Unload_trust_peers(ctx), WOLFSSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_locations(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = NULL;
#ifndef NO_RSA
    WOLFSSL_CERT_MANAGER* cm = NULL;
#ifdef PERSIST_CERT_CACHE
    int cacheSz = 0;
    unsigned char* cache = NULL;
    int used = 0;
#ifndef NO_FILESYSTEM
    const char* cacheFile = "./tests/cert_cache.tmp";
#endif
    int i;
    int t;
    int* p;
#endif
#endif
#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS)
    const char* load_certs_path = "./certs/external";
    const char* load_no_certs_path = "./examples";
    const char* load_expired_path = "./certs/test/expired";
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    /* invalid arguments */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(NULL, caCertFile, NULL),
        WOLFSSL_FAILURE);
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, NULL, NULL),
        WOLFSSL_FAILURE);

    /* invalid ca file */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, bogusFile, NULL),
        WS_RETURN_CODE(WOLFSSL_BAD_FILE,WOLFSSL_FAILURE));


#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS) && \
  ((defined(WOLFSSL_QT) || defined(WOLFSSL_IGNORE_BAD_CERT_PATH)) && \
  !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_IGNORE_BAD_PATH_ERR))
    /* invalid path */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, NULL, bogusFile),
        WS_RETURN_CODE(BAD_PATH_ERROR,WOLFSSL_FAILURE));
#endif
#if defined(WOLFSSL_QT) || defined(WOLFSSL_IGNORE_BAD_CERT_PATH)
    /* test ignoring the invalid path */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, bogusFile,
        WOLFSSL_LOAD_FLAG_IGNORE_BAD_PATH_ERR), WOLFSSL_SUCCESS);
#endif

    /* load ca cert */
#ifdef NO_RSA
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
        WS_RETURN_CODE(ASN_UNKNOWN_OID_E,WOLFSSL_FAILURE));
#else /* Skip the following test without RSA certs. */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
        WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Get cert cache size */
    ExpectIntGT(cacheSz = wolfSSL_CTX_get_cert_cache_memsize(ctx), 0);

    ExpectNotNull(cache = (byte*)XMALLOC(cacheSz, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER));

    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, NULL, -1, NULL),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, NULL, -1, NULL),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, cache, -1, NULL),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, NULL, cacheSz, NULL),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, NULL, -1, &used),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, cache, cacheSz, &used),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, NULL, cacheSz, &used),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, -1, &used),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, cacheSz, NULL),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, cacheSz - 10, &used),
        BUFFER_E);
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, cacheSz, &used), 1);
    ExpectIntEQ(cacheSz, used);

    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, NULL, -1),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, NULL, -1),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, cache, -1),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, NULL, cacheSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, cache, cacheSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, NULL, cacheSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, -1),
        BAD_FUNC_ARG);
    /* Smaller than header. */
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, 1), BUFFER_E);
    for (i = 1; i < cacheSz; i++) {
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz - i),
            BUFFER_E);
    }
    if (EXPECT_SUCCESS()) {
        /* Modify header for bad results! */
        p = (int*)cache;
        /* version */
        t = p[0]; p[0] = 0xff;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            CACHE_MATCH_ERROR);
        p[0] = t; p++;
        /* rows */
        t = p[0]; p[0] = 0xff;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            CACHE_MATCH_ERROR);
        p[0] = t; p++;
        /* columns[0] */
        t = p[0]; p[0] = -1;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            PARSE_ERROR);
        p[0] = t; p += CA_TABLE_SIZE;
        /* signerSz*/
        t = p[0]; p[0] = 0xff;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            CACHE_MATCH_ERROR);
        p[0] = t;
    }

    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz), 1);
    ExpectIntEQ(cacheSz = wolfSSL_CTX_get_cert_cache_memsize(ctx), used);

#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(ctx, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(NULL, cacheFile), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(ctx, cacheFile), 1);

    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(NULL, cacheFile), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, "no-file"),
        WOLFSSL_BAD_FILE);
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, cacheFile), 1);
    /* File contents is not a cache. */
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, "./certs/ca-cert.pem"),
        CACHE_MATCH_ERROR);
#endif

    XFREE(cache, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    /* Test unloading CA's */
    ExpectIntEQ(wolfSSL_CTX_UnloadCAs(ctx), WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Verify no certs (result is less than cacheSz) */
    ExpectIntGT(cacheSz, wolfSSL_CTX_get_cert_cache_memsize(ctx));
#endif

    /* load ca cert again */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
        WOLFSSL_SUCCESS);

    /* Test getting CERT_MANAGER */
    ExpectNotNull(cm = wolfSSL_CTX_GetCertManager(ctx));

    /* Test unloading CA's using CM */
    ExpectIntEQ(wolfSSL_CertManagerUnloadCAs(cm), WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Verify no certs (result is less than cacheSz) */
    ExpectIntGT(cacheSz, wolfSSL_CTX_get_cert_cache_memsize(ctx));
#endif
#endif

#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS)
    /* Test loading CA certificates using a path */
    #ifdef NO_RSA
    /* failure here okay since certs in external directory are RSA */
    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_SUCCESS);
    #else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_SUCCESS);
    #endif

    /* Test loading path with no files */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL,
        load_no_certs_path, WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_FAILURE);

    /* Test loading expired CA certificates */
    #ifdef NO_RSA
    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL,
        load_expired_path,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY | WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WOLFSSL_SUCCESS);
    #else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL,
        load_expired_path,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY | WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WOLFSSL_SUCCESS);
    #endif

    /* Test loading CA certificates and ignoring all errors */
    #ifdef NO_RSA
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_IGNORE_ERR), WOLFSSL_FAILURE);
    #else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_IGNORE_ERR), WOLFSSL_SUCCESS);
    #endif
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_system_CA_certs(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SYS_CA_CERTS) && !defined(NO_WOLFSSL_CLIENT) && \
    (!defined(NO_RSA) || defined(HAVE_ECC))
    WOLFSSL_CTX* ctx;
    byte dirValid = 0;
    int ret = 0;

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    if (ctx == NULL) {
        fprintf(stderr, "wolfSSL_CTX_new failed.\n");
        ret = -1;
    }
    if (ret == 0) {
    #if defined(USE_WINDOWS_API) || defined(__APPLE__)
        dirValid = 1;
    #else
        word32 numDirs;
        const char** caDirs = wolfSSL_get_system_CA_dirs(&numDirs);

        if (caDirs == NULL || numDirs == 0) {
            fprintf(stderr, "wolfSSL_get_system_CA_dirs failed.\n");
            ret = -1;
        }
        else {
            ReadDirCtx dirCtx;
            word32 i;

            for (i = 0; i < numDirs; ++i) {
                if (wc_ReadDirFirst(&dirCtx, caDirs[i], NULL) == 0) {
                    /* Directory isn't empty. */
                    dirValid = 1;
                    wc_ReadDirClose(&dirCtx);
                    break;
                }
            }
        }
    #endif
    }
    /*
     * If the directory isn't empty, we should be able to load CA
     * certs from it. On Windows/Mac, we assume the CA cert stores are
     * usable.
     */
    if (ret == 0 && dirValid && wolfSSL_CTX_load_system_CA_certs(ctx) !=
        WOLFSSL_SUCCESS) {
        fprintf(stderr, "wolfSSL_CTX_load_system_CA_certs failed.\n");
        ret = -1;
    }
#ifdef OPENSSL_EXTRA
    if (ret == 0 &&
        wolfSSL_CTX_set_default_verify_paths(ctx) != WOLFSSL_SUCCESS) {
        fprintf(stderr, "wolfSSL_CTX_set_default_verify_paths failed.\n");
        ret = -1;
    }
#endif /* OPENSSL_EXTRA */

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(ret == 0);
#endif /* WOLFSSL_SYS_CA_CERTS && !NO_WOLFSSL_CLIENT */

    return res;
}

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
static int test_cm_load_ca_buffer(const byte* cert_buf, size_t cert_sz,
    int file_type)
{
    int ret;
    WOLFSSL_CERT_MANAGER* cm;

    cm = wolfSSL_CertManagerNew();
    if (cm == NULL) {
        fprintf(stderr, "test_cm_load_ca failed\n");
        return -1;
    }

    ret = wolfSSL_CertManagerLoadCABuffer(cm, cert_buf, cert_sz, file_type);

    wolfSSL_CertManagerFree(cm);

    return ret;
}

static int test_cm_load_ca_file(const char* ca_cert_file)
{
    int ret = 0;
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
#if defined(WOLFSSL_PEM_TO_DER)
    DerBuffer* pDer = NULL;
#endif

    ret = load_file(ca_cert_file, &cert_buf, &cert_sz);
    if (ret == 0) {
        /* normal test */
        ret = test_cm_load_ca_buffer(cert_buf, cert_sz, WOLFSSL_FILETYPE_PEM);

        if (ret == WOLFSSL_SUCCESS) {
            /* test including null terminator in length */
            byte* tmp = (byte*)realloc(cert_buf, cert_sz+1);
            if (tmp == NULL) {
                ret = MEMORY_E;
            }
            else {
                cert_buf = tmp;
                cert_buf[cert_sz] = '\0';
                ret = test_cm_load_ca_buffer(cert_buf, cert_sz+1,
                        WOLFSSL_FILETYPE_PEM);
            }

        }

    #if defined(WOLFSSL_PEM_TO_DER)
        if (ret == WOLFSSL_SUCCESS) {
            /* test loading DER */
            ret = wc_PemToDer(cert_buf, cert_sz, CA_TYPE, &pDer, NULL, NULL, NULL);
            if (ret == 0 && pDer != NULL) {
                ret = test_cm_load_ca_buffer(pDer->buffer, pDer->length,
                    WOLFSSL_FILETYPE_ASN1);

                wc_FreeDer(&pDer);
            }
        }
    #endif

    }
    free(cert_buf);

    return ret;
}

static int test_cm_load_ca_buffer_ex(const byte* cert_buf, size_t cert_sz,
                                     int file_type, word32 flags)
{
    int ret;
    WOLFSSL_CERT_MANAGER* cm;

    cm = wolfSSL_CertManagerNew();
    if (cm == NULL) {
        fprintf(stderr, "test_cm_load_ca failed\n");
        return -1;
    }

    ret = wolfSSL_CertManagerLoadCABuffer_ex(cm, cert_buf, cert_sz, file_type,
                                             0, flags);

    wolfSSL_CertManagerFree(cm);

    return ret;
}

static int test_cm_load_ca_file_ex(const char* ca_cert_file, word32 flags)
{
    int ret = 0;
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
#if defined(WOLFSSL_PEM_TO_DER)
    DerBuffer* pDer = NULL;
#endif

    ret = load_file(ca_cert_file, &cert_buf, &cert_sz);
    if (ret == 0) {
        /* normal test */
        ret = test_cm_load_ca_buffer_ex(cert_buf, cert_sz,
                                        WOLFSSL_FILETYPE_PEM, flags);

        if (ret == WOLFSSL_SUCCESS) {
            /* test including null terminator in length */
            byte* tmp = (byte*)realloc(cert_buf, cert_sz+1);
            if (tmp == NULL) {
                ret = MEMORY_E;
            }
            else {
                cert_buf = tmp;
                cert_buf[cert_sz] = '\0';
                ret = test_cm_load_ca_buffer_ex(cert_buf, cert_sz+1,
                        WOLFSSL_FILETYPE_PEM, flags);
            }

        }

    #if defined(WOLFSSL_PEM_TO_DER)
        if (ret == WOLFSSL_SUCCESS) {
            /* test loading DER */
            ret = wc_PemToDer(cert_buf, cert_sz, CA_TYPE, &pDer, NULL, NULL, NULL);
            if (ret == 0 && pDer != NULL) {
                ret = test_cm_load_ca_buffer_ex(pDer->buffer, pDer->length,
                    WOLFSSL_FILETYPE_ASN1, flags);

                wc_FreeDer(&pDer);
            }
        }
    #endif

    }
    free(cert_buf);

    return ret;
}

#endif /* !NO_FILESYSTEM && !NO_CERTS */

static int test_wolfSSL_CertManagerAPI(void)
{
    EXPECT_DECLS;
#ifndef NO_CERTS
    WOLFSSL_CERT_MANAGER* cm = NULL;
    unsigned char c;

    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));

    wolfSSL_CertManagerFree(NULL);
    ExpectIntEQ(wolfSSL_CertManager_up_ref(NULL), 0);
    ExpectIntEQ(wolfSSL_CertManagerUnloadCAs(NULL), BAD_FUNC_ARG);
#ifdef WOLFSSL_TRUST_PEER_CERT
    ExpectIntEQ(wolfSSL_CertManagerUnload_trust_peers(NULL), BAD_FUNC_ARG);
#endif

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer_ex(NULL, &c, 1,
        WOLFSSL_FILETYPE_ASN1, 0, 0), WOLFSSL_FATAL_ERROR);

#if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, &c, -1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, &c, 1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, &c, -1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, &c, 1, -1),
        WOLFSSL_BAD_FILETYPE);
#endif

#if !defined(NO_FILESYSTEM)
    {
        const char* ca_cert = "./certs/ca-cert.pem";
        const char* ca_cert_der = "./certs/ca-cert.der";
        const char* ca_path = "./certs";

    #if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
        ExpectIntEQ(wolfSSL_CertManagerVerify(NULL, NULL, -1),
            BAD_FUNC_ARG);
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, NULL, WOLFSSL_FILETYPE_ASN1),
            BAD_FUNC_ARG);
        ExpectIntEQ(wolfSSL_CertManagerVerify(NULL, ca_cert,
            WOLFSSL_FILETYPE_PEM), BAD_FUNC_ARG);
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, ca_cert, -1),
            WOLFSSL_BAD_FILETYPE);
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, "no-file",
            WOLFSSL_FILETYPE_ASN1), WOLFSSL_BAD_FILE);
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, ca_cert_der,
            WOLFSSL_FILETYPE_PEM), ASN_NO_PEM_HEADER);
    #endif

        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, NULL, NULL),
            WOLFSSL_FATAL_ERROR);
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, ca_cert, NULL),
            WOLFSSL_FATAL_ERROR);
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, NULL, ca_path),
            WOLFSSL_FATAL_ERROR);
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, ca_cert, ca_path),
            WOLFSSL_FATAL_ERROR);
    }
#endif

#ifdef OPENSSL_COMPATIBLE_DEFAULTS
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, 0), 1);
#elif !defined(HAVE_CRL)
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, 0), NOT_COMPILED_IN);
#endif

    ExpectIntEQ(wolfSSL_CertManagerDisableCRL(NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerDisableCRL(cm), 1);
#ifdef HAVE_CRL
    /* Test APIs when CRL is disabled. */
#ifdef HAVE_CRL_IO
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_IOCb(cm, NULL), 1);
#endif
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048), 1);
    ExpectIntEQ(wolfSSL_CertManagerFreeCRL(cm), 1);
#endif

    /* OCSP */
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSP(NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPStapling(NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPMustStaple(NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPMustStaple(NULL), BAD_FUNC_ARG);
#if !defined(HAVE_CERTIFICATE_STATUS_REQUEST) && \
    !defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPStapling(cm), NOT_COMPILED_IN);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPMustStaple(cm), NOT_COMPILED_IN);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPMustStaple(cm), NOT_COMPILED_IN);
#endif

#ifdef HAVE_OCSP
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, NULL, -1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, NULL, -1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, &c, -1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, &c, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, &c, -1), BAD_FUNC_ARG);

    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(NULL, NULL, 0,
        NULL, NULL, NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, NULL, 1,
        NULL, NULL, NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(NULL, &c, 1,
        NULL, NULL, NULL, NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(NULL, NULL),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(NULL, ""),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(cm, NULL), 1);

    ExpectIntEQ(wolfSSL_CertManagerSetOCSP_Cb(NULL, NULL, NULL, NULL),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerSetOCSP_Cb(cm, NULL, NULL, NULL), 1);

    ExpectIntEQ(wolfSSL_CertManagerDisableOCSP(cm), 1);
    /* Test APIs when OCSP is disabled. */
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, &c, 1,
        NULL, NULL, NULL, NULL), 1);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, &c, 1), 1);

#endif

    ExpectIntEQ(wolfSSL_CertManager_up_ref(cm), 1);
    wolfSSL_CertManagerFree(cm);
    wolfSSL_CertManagerFree(cm);
    cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));

#ifdef HAVE_OCSP
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(cm, WOLFSSL_OCSP_URL_OVERRIDE |
         WOLFSSL_OCSP_CHECKALL), 1);
#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
    defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPMustStaple(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPMustStaple(cm), 1);
#endif

    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(cm, ""), 1);
    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(cm, ""), 1);
#endif

#ifdef WOLFSSL_TRUST_PEER_CERT
    ExpectIntEQ(wolfSSL_CertManagerUnload_trust_peers(cm), 1);
#endif
    wolfSSL_CertManagerFree(cm);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerLoadCABuffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
    int ret;

    ExpectIntLE(ret = test_cm_load_ca_file(ca_cert), 1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WOLFSSL_FATAL_ERROR);
#elif defined(NO_RSA)
    ExpectIntEQ(ret, ASN_UNKNOWN_OID_E);
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif

    ExpectIntLE(ret = test_cm_load_ca_file(ca_expired_cert), 1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WOLFSSL_FATAL_ERROR);
#elif defined(NO_RSA)
    ExpectIntEQ(ret, ASN_UNKNOWN_OID_E);
#elif !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY) && \
      !defined(NO_ASN_TIME)
    ExpectIntEQ(ret, ASN_AFTER_DATE_E);
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerLoadCABuffer_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
    int ret;

    ExpectIntLE(ret = test_cm_load_ca_file_ex(ca_cert, WOLFSSL_LOAD_FLAG_NONE),
        1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WOLFSSL_FATAL_ERROR);
#elif defined(NO_RSA)
    ExpectIntEQ(ret, ASN_UNKNOWN_OID_E);
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif

    ExpectIntLE(ret = test_cm_load_ca_file_ex(ca_expired_cert,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), 1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WOLFSSL_FATAL_ERROR);
#elif defined(NO_RSA)
    ExpectIntEQ(ret, ASN_UNKNOWN_OID_E);
#elif !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY) && \
      !defined(NO_ASN_TIME) && defined(WOLFSSL_TRUST_PEER_CERT) && \
      defined(OPENSSL_COMPATIBLE_DEFAULTS)
    ExpectIntEQ(ret, ASN_AFTER_DATE_E);
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif

#endif

    return EXPECT_RESULT();
}


static int test_wolfSSL_CertManagerGetCerts(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
     defined(WOLFSSL_SIGNER_DER_CERT)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_STACK* sk = NULL;
    X509* x509 = NULL;
    X509* cert1 = NULL;
    FILE* file1 = NULL;
#ifdef DEBUG_WOLFSSL_VERBOSE
    WOLFSSL_BIO* bio = NULL;
#endif
    int i = 0;
    int ret = 0;
    const byte* der;
    int derSz = 0;

    ExpectNotNull(file1 = fopen("./certs/ca-cert.pem", "rb"));

    ExpectNotNull(cert1 = wolfSSL_PEM_read_X509(file1, NULL, NULL, NULL));
    if (file1 != NULL) {
        fclose(file1);
    }

    ExpectNull(sk = wolfSSL_CertManagerGetCerts(NULL));
    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
    ExpectNull(sk = wolfSSL_CertManagerGetCerts(cm));

    ExpectNotNull(der = wolfSSL_X509_get_der(cert1, &derSz));
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
    /* Check that ASN_SELF_SIGNED_E is returned for a self-signed cert for QT
     * and full OpenSSL compatibility */
    ExpectIntEQ(ret = wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
        WOLFSSL_FILETYPE_ASN1), ASN_SELF_SIGNED_E);
#else
    ExpectIntEQ(ret = wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
        WOLFSSL_FILETYPE_ASN1), ASN_NO_SIGNER_E);
#endif

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CertManagerLoadCA(cm,
                "./certs/ca-cert.pem", NULL));

    ExpectNotNull(sk = wolfSSL_CertManagerGetCerts(cm));

    for (i = 0; EXPECT_SUCCESS() && i < sk_X509_num(sk); i++) {
        ExpectNotNull(x509 = sk_X509_value(sk, i));
        ExpectIntEQ(0, wolfSSL_X509_cmp(x509, cert1));

#ifdef DEBUG_WOLFSSL_VERBOSE
        bio = BIO_new(wolfSSL_BIO_s_file());
        if (bio != NULL) {
            BIO_set_fp(bio, stderr, BIO_NOCLOSE);
            X509_print(bio, x509);
            BIO_free(bio);
        }
#endif /* DEBUG_WOLFSSL_VERBOSE */
    }
    wolfSSL_X509_free(cert1);
    sk_X509_pop_free(sk, NULL);
    wolfSSL_CertManagerFree(cm);
#endif /* defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
          !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
          defined(WOLFSSL_SIGNER_DER_CERT) */

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerSetVerify(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH))
    WOLFSSL_CERT_MANAGER* cm = NULL;
    int tmp = myVerifyAction;
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* expiredCert = "./certs/test/expired/expired-cert.pem";

    wolfSSL_CertManagerSetVerify(NULL, NULL);
    wolfSSL_CertManagerSetVerify(NULL, myVerify);

    ExpectNotNull(cm = wolfSSL_CertManagerNew());

    wolfSSL_CertManagerSetVerify(cm, myVerify);

#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL), -1);
#else
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL),
                WOLFSSL_SUCCESS);
#endif
    /* Use the test CB that always accepts certs */
    myVerifyAction = VERIFY_OVERRIDE_ERROR;

    ExpectIntEQ(wolfSSL_CertManagerVerify(cm, expiredCert,
        WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

#ifdef WOLFSSL_ALWAYS_VERIFY_CB
    {
        const char* verifyCert = "./certs/server-cert.der";
        /* Use the test CB that always fails certs */
        myVerifyAction = VERIFY_FORCE_FAIL;

        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, verifyCert,
            WOLFSSL_FILETYPE_ASN1), VERIFY_CERT_ERROR);
    }
#endif

    wolfSSL_CertManagerFree(cm);
    myVerifyAction = tmp;
#endif

    return EXPECT_RESULT();
}

#if !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_UNIT_TEST_CERTS)
/* Used when debugging name constraint tests. Not static to allow use in
 * multiple locations with complex define guards. */
void DEBUG_WRITE_CERT_X509(WOLFSSL_X509* x509, const char* fileName)
{
    BIO* out = BIO_new_file(fileName, "wb");
    if (out != NULL) {
        PEM_write_bio_X509(out, x509);
        BIO_free(out);
    }
}
void DEBUG_WRITE_DER(const byte* der, int derSz, const char* fileName)
{
    BIO* out = BIO_new_file(fileName, "wb");
    if (out != NULL) {
        BIO_write(out, der, derSz);
        BIO_free(out);
    }
}
#else
#define DEBUG_WRITE_CERT_X509(x509, fileName) WC_DO_NOTHING
#define DEBUG_WRITE_DER(der, derSz, fileName) WC_DO_NOTHING
#endif


static int test_wolfSSL_CertManagerNameConstraint(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-nc.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";
    int i = 0;
    static const byte extNameConsOid[] = {85, 29, 30};

    RsaKey  key;
    WC_RNG  rng;
    byte    *der = NULL;
    int     derSz = 0;
    word32  idx = 0;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    wc_InitRng(&rng);

    /* load in CA private key for signing */
    ExpectIntEQ(wc_InitRsaKey_ex(&key, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_RsaPrivateKeyDecode(server_key_der_2048, &idx, &key,
                sizeof_server_key_der_2048), 0);

    /* get ca certificate then alter it */
    ExpectNotNull(der =
            (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull(pt = (byte*)wolfSSL_X509_get_tbs(x509, &derSz));
    if (EXPECT_SUCCESS() && (der != NULL)) {
        XMEMCPY(der, pt, derSz);

        /* find the name constraint extension and alter it */
        pt = der;
        for (i = 0; i < derSz - 3; i++) {
            if (XMEMCMP(pt, extNameConsOid, 3) == 0) {
                pt += 3;
                break;
            }
            pt++;
        }
        ExpectIntNE(i, derSz - 3); /* did not find OID if this case is hit */

        /* go to the length value and set it to 0 */
        while (i < derSz && *pt != 0x81) {
            pt++;
            i++;
        }
        ExpectIntNE(i, derSz); /* did not place to alter */
        pt++;
        *pt = 0x00;
    }

    /* resign the altered certificate */
    ExpectIntGT((derSz = wc_SignCert(derSz, CTC_SHA256wRSA, der,
                             FOURK_BUF, &key, NULL, &rng)), 0);

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_PARSE_E);
    wolfSSL_CertManagerFree(cm);

    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_X509_free(x509);
    wc_FreeRsaKey(&key);
    wc_FreeRng(&rng);

    /* add email alt name to satisfy constraint */
    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* Good cert test with proper alt email name */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;


    /* Cert with bad alt name list */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.com", ASN_RFC822_TYPE);
    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}


static int test_wolfSSL_CertManagerNameConstraint2(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES)
    const char* ca_cert  = "./certs/test/cert-ext-ndir.der";
    const char* ca_cert2 = "./certs/test/cert-ext-ndir-exc.der";
    const char* server_cert = "./certs/server-cert.pem";
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    const unsigned char *der = NULL;
    const unsigned char *pt;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    int   derSz = 0;

    /* C=US*/
    char altName[] = {
        0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53
    };

    /* C=ID */
    char altNameFail[] = {
        0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x44
    };

    /* C=US ST=California*/
    char altNameExc[] = {
        0x30, 0x22,
        0x31, 0x0B,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53,
        0x31, 0x13,
        0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x0A,
        0x43, 0x61, 0x6c, 0x69, 0x66, 0x6f, 0x72, 0x6e, 0x69, 0x61
    };
    /* load in CA private key for signing */
    pt = ca_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL, &pt,
                                                sizeof_ca_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = wolfSSL_X509_get_der(ca, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* add in matching DIR alt name and resign */
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif

    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check verify fail */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    /* add in miss matching DIR alt name and resign */
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* check that it still fails if one bad altname and one good altname is in
     * the certificate */
    wolfSSL_X509_free(x509);
    x509 = NULL;
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* check it fails with switching position of bad altname */
    wolfSSL_X509_free(x509);
    x509 = NULL;
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
    wolfSSL_CertManagerFree(cm);

    wolfSSL_X509_free(x509);
    x509 = NULL;
    wolfSSL_X509_free(ca);
    ca = NULL;

    /* now test with excluded name constraint */
    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert2,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = wolfSSL_X509_get_der(ca, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    wolfSSL_X509_add_altname_ex(x509, altNameExc, sizeof(altNameExc),
            ASN_DIR_TYPE);
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerNameConstraint3(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-mnc.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der = NULL;
    int     derSz = 0;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying .wolfssl.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-1st-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check satisfying .random.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.example.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "wolfssl@info.example.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-2nd-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check fail case when neither constraint is matched */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                     (byte*)"support@info.com", 16, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerNameConstraint4(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-ncdns.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der = NULL;
    int     derSz;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying wolfssl.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-1st-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check satisfying example.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"example.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.example.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-2nd-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check satisfying wolfssl.com constraint passes with list of DNS's */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.info.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "extra.wolfssl.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-multiple-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check fail when one DNS in the list is bad */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.nomatch.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.info.wolfssl.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-multiple-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check fail case when neither constraint is matched */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"common", 6, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "www.random.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerNameConstraint5(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-ncmixed.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der = NULL;
    int     derSz;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying wolfssl.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"example", 7, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "good.example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "facts@into.wolfssl.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* fail with DNS check because of common name */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "facts@wolfssl.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cn-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* fail on permitted DNS name constraint */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.wolfssl", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "info@wolfssl.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-1st-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* fail on permitted email name constraint */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "info@wolfssl.com", ASN_RFC822_TYPE);
    wolfSSL_X509_add_altname(x509, "info@example.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-2nd-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* success with empty email name */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-missing-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerCRL(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && defined(HAVE_CRL) && \
    !defined(NO_RSA)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* crl1     = "./certs/crl/crl.pem";
    const char* crl2     = "./certs/crl/crl2.pem";
    const unsigned char crl_buff[] = {
        0x30, 0x82, 0x02, 0x04, 0x30, 0x81, 0xed, 0x02,
        0x01, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
        0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05,
        0x00, 0x30, 0x81, 0x94, 0x31, 0x0b, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
        0x53, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55,
        0x04, 0x08, 0x0c, 0x07, 0x4d, 0x6f, 0x6e, 0x74,
        0x61, 0x6e, 0x61, 0x31, 0x10, 0x30, 0x0e, 0x06,
        0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x42, 0x6f,
        0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x31, 0x11, 0x30,
        0x0f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x08,
        0x53, 0x61, 0x77, 0x74, 0x6f, 0x6f, 0x74, 0x68,
        0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
        0x0b, 0x0c, 0x0a, 0x43, 0x6f, 0x6e, 0x73, 0x75,
        0x6c, 0x74, 0x69, 0x6e, 0x67, 0x31, 0x18, 0x30,
        0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f,
        0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66,
        0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x31,
        0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48,
        0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10,
        0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c,
        0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d,
        0x17, 0x0d, 0x32, 0x32, 0x31, 0x32, 0x31, 0x36,
        0x32, 0x31, 0x31, 0x37, 0x35, 0x30, 0x5a, 0x17,
        0x0d, 0x32, 0x35, 0x30, 0x39, 0x31, 0x31, 0x32,
        0x31, 0x31, 0x37, 0x35, 0x30, 0x5a, 0x30, 0x14,
        0x30, 0x12, 0x02, 0x01, 0x02, 0x17, 0x0d, 0x32,
        0x32, 0x31, 0x32, 0x31, 0x36, 0x32, 0x31, 0x31,
        0x37, 0x35, 0x30, 0x5a, 0xa0, 0x0e, 0x30, 0x0c,
        0x30, 0x0a, 0x06, 0x03, 0x55, 0x1d, 0x14, 0x04,
        0x03, 0x02, 0x01, 0x02, 0x30, 0x0d, 0x06, 0x09,
        0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
        0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00,
        0x39, 0x44, 0xff, 0x39, 0xf4, 0x04, 0x45, 0x79,
        0x7e, 0x73, 0xe2, 0x42, 0x48, 0xdb, 0x85, 0x66,
        0xfd, 0x99, 0x76, 0x94, 0x7c, 0xb5, 0x79, 0x5d,
        0x15, 0x71, 0x36, 0xa9, 0x87, 0xf0, 0x73, 0x05,
        0x50, 0x08, 0x6b, 0x1c, 0x6e, 0xde, 0x96, 0x45,
        0x31, 0xc3, 0xc0, 0xba, 0xba, 0xf5, 0x08, 0x1d,
        0x05, 0x4a, 0x52, 0x39, 0xe9, 0x03, 0xef, 0x59,
        0xc8, 0x1d, 0x4a, 0xf2, 0x86, 0x05, 0x99, 0x7b,
        0x4b, 0x74, 0xf6, 0xd3, 0x75, 0x8d, 0xb2, 0x57,
        0xba, 0xac, 0xa7, 0x11, 0x14, 0xd6, 0x6c, 0x71,
        0xc4, 0x4c, 0x1c, 0x68, 0xbc, 0x49, 0x78, 0xf0,
        0xc9, 0x52, 0x8a, 0xe7, 0x8b, 0x54, 0xe6, 0x20,
        0x58, 0x20, 0x60, 0x66, 0xf5, 0x14, 0xd8, 0xcb,
        0xff, 0xe0, 0xa0, 0x45, 0xbc, 0xb4, 0x81, 0xad,
        0x1d, 0xbc, 0xcf, 0xf8, 0x8e, 0xa8, 0x87, 0x24,
        0x55, 0x99, 0xd9, 0xce, 0x47, 0xf7, 0x5b, 0x4a,
        0x33, 0x6d, 0xdb, 0xbf, 0x93, 0x64, 0x1a, 0xa6,
        0x46, 0x5f, 0x27, 0xdc, 0xd8, 0xd4, 0xf9, 0xc2,
        0x42, 0x2a, 0x7e, 0xb2, 0x7c, 0xdd, 0x98, 0x77,
        0xf5, 0x88, 0x7d, 0x15, 0x25, 0x08, 0xbc, 0xe0,
        0xd0, 0x8d, 0xf4, 0xc3, 0xc3, 0x04, 0x41, 0xa4,
        0xd1, 0xb1, 0x39, 0x4a, 0x6b, 0x2c, 0xb5, 0x2e,
        0x9a, 0x65, 0x43, 0x0d, 0x0e, 0x73, 0xf4, 0x06,
        0xe1, 0xb3, 0x49, 0x34, 0x94, 0xb0, 0xb7, 0xff,
        0xc0, 0x27, 0xc1, 0xb5, 0xea, 0x06, 0xf7, 0x71,
        0x71, 0x97, 0xbb, 0xbc, 0xc7, 0x1a, 0x9f, 0xeb,
        0xf6, 0x3d, 0xa5, 0x7b, 0x55, 0xa7, 0xbf, 0xdd,
        0xd7, 0xee, 0x97, 0xb8, 0x9d, 0xdc, 0xcd, 0xe3,
        0x06, 0xdb, 0x9a, 0x2c, 0x60, 0xbf, 0x70, 0x84,
        0xfa, 0x6b, 0x8d, 0x70, 0x7d, 0xde, 0xe8, 0xb7,
        0xab, 0xb0, 0x38, 0x68, 0x6c, 0xc0, 0xb1, 0xe1,
        0xba, 0x45, 0xe0, 0xd7, 0x12, 0x3d, 0x71, 0x5b
    };

    WOLFSSL_CERT_MANAGER* cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew());

    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, WOLFSSL_CRL_CHECKALL), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, WOLFSSL_CRL_CHECK), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm,
        WOLFSSL_CRL_CHECK | WOLFSSL_CRL_CHECKALL), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, 16), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, WOLFSSL_CRL_CHECKALL), 1);

    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, NULL, -1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, NULL, -1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, server_cert_der_2048, -1),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, server_cert_der_2048, 1),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048, -1),
        BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048), ASN_NO_SIGNER_E);

    ExpectIntEQ(wolfSSL_CertManagerSetCRL_Cb(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_Cb(cm, NULL), 1);
#ifdef HAVE_CRL_IO
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_IOCb(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_IOCb(cm, NULL), 1);
#endif

#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CertManagerLoadCRL(NULL, NULL, WOLFSSL_FILETYPE_ASN1,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerLoadCRL(cm, NULL, WOLFSSL_FILETYPE_ASN1,
        0), BAD_FUNC_ARG);
    /* -1 seen as !WOLFSSL_FILETYPE_PEM */
    ExpectIntEQ(wolfSSL_CertManagerLoadCRL(cm, "./certs/crl", -1, 0), 1);

    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(NULL, NULL,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, NULL, WOLFSSL_FILETYPE_ASN1),
        BAD_FUNC_ARG);
    /* -1 seen as !WOLFSSL_FILETYPE_PEM */
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, "./certs/crl/crl.pem", -1),
        ASN_PARSE_E);
#endif

    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, crl_buff, -1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, crl_buff, 1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, crl_buff, -1,
        WOLFSSL_FILETYPE_ASN1), BAD_FUNC_ARG);

    ExpectIntEQ(wolfSSL_CertManagerFreeCRL(NULL), BAD_FUNC_ARG);
    DoExpectIntEQ(wolfSSL_CertManagerFreeCRL(cm), 1);

    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl1, WOLFSSL_FILETYPE_PEM, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl2, WOLFSSL_FILETYPE_PEM, 0));
    wolfSSL_CertManagerFreeCRL(cm);

    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl1, WOLFSSL_FILETYPE_PEM, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048), CRL_MISSING);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1), CRL_MISSING);

    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, crl_buff, sizeof(crl_buff),
        WOLFSSL_FILETYPE_ASN1), 1);

    wolfSSL_CertManagerFree(cm);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerCheckOCSPResponse(void)
{
    EXPECT_DECLS;
#if defined(HAVE_OCSP) && !defined(NO_RSA)
/* Need one of these for wolfSSL_OCSP_REQUEST_new. */
#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_APACHE_HTTPD) || \
    defined(HAVE_LIGHTY)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    /* Raw OCSP response bytes captured using the following setup:
     * - Run responder with
     *      openssl ocsp -port 9999 -ndays 9999
     *      -index certs/ocsp/index-intermediate1-ca-issued-certs.txt
     *      -rsigner certs/ocsp/ocsp-responder-cert.pem
     *      -rkey certs/ocsp/ocsp-responder-key.pem
     *      -CA certs/ocsp/intermediate1-ca-cert.pem
     * - Run client with
     *      openssl ocsp -host 127.0.0.1:9999 -respout resp.out
     *      -issuer certs/ocsp/intermediate1-ca-cert.pem
     *      -cert certs/ocsp/server1-cert.pem
     *      -CAfile certs/ocsp/root-ca-cert.pem -noverify
     * - Copy raw response from Wireshark.
     */
    byte response[] = {
        0x30, 0x82, 0x07, 0x40, 0x0a, 0x01, 0x00, 0xa0, 0x82, 0x07, 0x39, 0x30, 0x82, 0x07, 0x35, 0x06,
        0x09, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82, 0x07, 0x26, 0x30, 0x82,
        0x07, 0x22, 0x30, 0x82, 0x01, 0x40, 0xa1, 0x81, 0xa1, 0x30, 0x81, 0x9e, 0x31, 0x0b, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
        0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10,
        0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, 0x65,
        0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07, 0x77, 0x6f, 0x6c, 0x66, 0x53,
        0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45, 0x6e, 0x67,
        0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x04,
        0x03, 0x0c, 0x16, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x4f, 0x43, 0x53, 0x50, 0x20,
        0x52, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a,
        0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77,
        0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31,
        0x30, 0x35, 0x30, 0x33, 0x32, 0x31, 0x34, 0x37, 0x31, 0x30, 0x5a, 0x30, 0x64, 0x30, 0x62, 0x30,
        0x3a, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x71, 0x4d,
        0x82, 0x23, 0x40, 0x59, 0xc0, 0x96, 0xa1, 0x37, 0x43, 0xfa, 0x31, 0xdb, 0xba, 0xb1, 0x43, 0x18,
        0xda, 0x04, 0x04, 0x14, 0x83, 0xc6, 0x3a, 0x89, 0x2c, 0x81, 0xf4, 0x02, 0xd7, 0x9d, 0x4c, 0xe2,
        0x2a, 0xc0, 0x71, 0x82, 0x64, 0x44, 0xda, 0x0e, 0x02, 0x01, 0x05, 0x80, 0x00, 0x18, 0x0f, 0x32,
        0x30, 0x32, 0x31, 0x30, 0x35, 0x30, 0x33, 0x32, 0x31, 0x34, 0x37, 0x31, 0x30, 0x5a, 0xa0, 0x11,
        0x18, 0x0f, 0x32, 0x30, 0x34, 0x38, 0x30, 0x39, 0x31, 0x37, 0x32, 0x31, 0x34, 0x37, 0x31, 0x30,
        0x5a, 0xa1, 0x23, 0x30, 0x21, 0x30, 0x1f, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30,
        0x01, 0x02, 0x04, 0x12, 0x04, 0x10, 0x38, 0x31, 0x60, 0x99, 0xc8, 0x05, 0x09, 0x68, 0x1c, 0x33,
        0x49, 0xea, 0x45, 0x26, 0x2f, 0x6d, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
        0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x4d, 0x58, 0xcc, 0x69, 0x42, 0xe2,
        0x9e, 0x64, 0xf6, 0x57, 0xce, 0xcb, 0x5f, 0x14, 0xaf, 0x08, 0x6c, 0xc1, 0x52, 0x7a, 0x40, 0x0a,
        0xfd, 0xb6, 0xce, 0xbb, 0x40, 0xf4, 0xb9, 0xa5, 0x88, 0xc7, 0xf3, 0x42, 0x9f, 0xa9, 0x94, 0xbe,
        0x6e, 0x7e, 0x09, 0x30, 0x9d, 0x0e, 0x10, 0x6f, 0x9c, 0xd9, 0x4c, 0x71, 0x81, 0x41, 0x64, 0x95,
        0xf5, 0x85, 0x77, 0x94, 0x81, 0x61, 0x88, 0xc8, 0x0b, 0x50, 0xbb, 0x37, 0xc8, 0x86, 0x76, 0xd8,
        0xa2, 0xed, 0x66, 0x34, 0xfb, 0xe4, 0xe7, 0x09, 0x8c, 0xf5, 0xb5, 0x85, 0xd0, 0x4b, 0xb5, 0xe6,
        0x23, 0x62, 0xc3, 0xd0, 0xef, 0xf7, 0x42, 0x89, 0x02, 0x80, 0x64, 0xc9, 0xed, 0xdd, 0x7c, 0x8f,
        0x0d, 0xe7, 0x43, 0x9b, 0x88, 0x1f, 0xb0, 0xfd, 0x24, 0x01, 0xc7, 0x55, 0xc3, 0x73, 0x12, 0x84,
        0x09, 0x7c, 0x57, 0xa8, 0x5d, 0xab, 0x75, 0x29, 0x5c, 0x36, 0x97, 0x64, 0x40, 0x0b, 0x55, 0x34,
        0x0a, 0x5d, 0xb1, 0x1b, 0x61, 0x1b, 0xdc, 0xe5, 0x89, 0xdd, 0x92, 0x62, 0x57, 0xa7, 0x52, 0xb4,
        0x38, 0x9a, 0x48, 0xc8, 0x3a, 0x14, 0xde, 0x69, 0x42, 0xe9, 0x37, 0xa4, 0xe7, 0x2d, 0x00, 0xa7,
        0x0b, 0x29, 0x18, 0xd5, 0xce, 0xd9, 0x0d, 0xdd, 0xfe, 0xae, 0x86, 0xb3, 0x32, 0x1c, 0xc9, 0x33,
        0xb0, 0x2b, 0xb7, 0x3c, 0x0d, 0x43, 0xd8, 0x6c, 0xf2, 0xb7, 0xcd, 0x7b, 0xd5, 0x7d, 0xf0, 0xde,
        0x34, 0x9f, 0x6d, 0x83, 0xb9, 0xd5, 0xed, 0xe3, 0xda, 0x96, 0x40, 0x9e, 0xd6, 0xa6, 0xfd, 0x70,
        0x80, 0x70, 0x87, 0x61, 0x0f, 0xc5, 0x9f, 0x75, 0xfe, 0x11, 0x78, 0x34, 0xc9, 0x42, 0x16, 0x73,
        0x46, 0x7b, 0x05, 0x53, 0x28, 0x43, 0xbe, 0xee, 0x88, 0x67, 0x1d, 0xcc, 0x74, 0xa7, 0xb6, 0x58,
        0x7b, 0x29, 0x68, 0x40, 0xcf, 0xce, 0x7b, 0x19, 0x33, 0x68, 0xa0, 0x82, 0x04, 0xc6, 0x30, 0x82,
        0x04, 0xc2, 0x30, 0x82, 0x04, 0xbe, 0x30, 0x82, 0x03, 0xa6, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02,
        0x01, 0x04, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05,
        0x00, 0x30, 0x81, 0x97, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
        0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68,
        0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c,
        0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04,
        0x0a, 0x0c, 0x07, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03,
        0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67,
        0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x6f, 0x6c, 0x66, 0x53,
        0x53, 0x4c, 0x20, 0x72, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09,
        0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f, 0x40,
        0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x32,
        0x31, 0x30, 0x32, 0x31, 0x30, 0x31, 0x39, 0x34, 0x39, 0x35, 0x34, 0x5a, 0x17, 0x0d, 0x32, 0x33,
        0x31, 0x31, 0x30, 0x37, 0x31, 0x39, 0x34, 0x39, 0x35, 0x34, 0x5a, 0x30, 0x81, 0x9e, 0x31, 0x0b,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06,
        0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e,
        0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74,
        0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07, 0x77, 0x6f, 0x6c,
        0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45,
        0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03,
        0x55, 0x04, 0x03, 0x0c, 0x16, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x4f, 0x43, 0x53,
        0x50, 0x20, 0x52, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x31, 0x1f, 0x30, 0x1d, 0x06,
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f,
        0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82, 0x01, 0x22,
        0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03,
        0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xb8, 0xba, 0x23,
        0xb4, 0xf6, 0xc3, 0x7b, 0x14, 0xc3, 0xa4, 0xf5, 0x1d, 0x61, 0xa1, 0xf5, 0x1e, 0x63, 0xb9, 0x85,
        0x23, 0x34, 0x50, 0x6d, 0xf8, 0x7c, 0xa2, 0x8a, 0x04, 0x8b, 0xd5, 0x75, 0x5c, 0x2d, 0xf7, 0x63,
        0x88, 0xd1, 0x07, 0x7a, 0xea, 0x0b, 0x45, 0x35, 0x2b, 0xeb, 0x1f, 0xb1, 0x22, 0xb4, 0x94, 0x41,
        0x38, 0xe2, 0x9d, 0x74, 0xd6, 0x8b, 0x30, 0x22, 0x10, 0x51, 0xc5, 0xdb, 0xca, 0x3f, 0x46, 0x2b,
        0xfe, 0xe5, 0x5a, 0x3f, 0x41, 0x74, 0x67, 0x75, 0x95, 0xa9, 0x94, 0xd5, 0xc3, 0xee, 0x42, 0xf8,
        0x8d, 0xeb, 0x92, 0x95, 0xe1, 0xd9, 0x65, 0xb7, 0x43, 0xc4, 0x18, 0xde, 0x16, 0x80, 0x90, 0xce,
        0x24, 0x35, 0x21, 0xc4, 0x55, 0xac, 0x5a, 0x51, 0xe0, 0x2e, 0x2d, 0xb3, 0x0a, 0x5a, 0x4f, 0x4a,
        0x73, 0x31, 0x50, 0xee, 0x4a, 0x16, 0xbd, 0x39, 0x8b, 0xad, 0x05, 0x48, 0x87, 0xb1, 0x99, 0xe2,
        0x10, 0xa7, 0x06, 0x72, 0x67, 0xca, 0x5c, 0xd1, 0x97, 0xbd, 0xc8, 0xf1, 0x76, 0xf8, 0xe0, 0x4a,
        0xec, 0xbc, 0x93, 0xf4, 0x66, 0x4c, 0x28, 0x71, 0xd1, 0xd8, 0x66, 0x03, 0xb4, 0x90, 0x30, 0xbb,
        0x17, 0xb0, 0xfe, 0x97, 0xf5, 0x1e, 0xe8, 0xc7, 0x5d, 0x9b, 0x8b, 0x11, 0x19, 0x12, 0x3c, 0xab,
        0x82, 0x71, 0x78, 0xff, 0xae, 0x3f, 0x32, 0xb2, 0x08, 0x71, 0xb2, 0x1b, 0x8c, 0x27, 0xac, 0x11,
        0xb8, 0xd8, 0x43, 0x49, 0xcf, 0xb0, 0x70, 0xb1, 0xf0, 0x8c, 0xae, 0xda, 0x24, 0x87, 0x17, 0x3b,
        0xd8, 0x04, 0x65, 0x6c, 0x00, 0x76, 0x50, 0xef, 0x15, 0x08, 0xd7, 0xb4, 0x73, 0x68, 0x26, 0x14,
        0x87, 0x95, 0xc3, 0x5f, 0x6e, 0x61, 0xb8, 0x87, 0x84, 0xfa, 0x80, 0x1a, 0x0a, 0x8b, 0x98, 0xf3,
        0xe3, 0xff, 0x4e, 0x44, 0x1c, 0x65, 0x74, 0x7c, 0x71, 0x54, 0x65, 0xe5, 0x39, 0x02, 0x03, 0x01,
        0x00, 0x01, 0xa3, 0x82, 0x01, 0x0a, 0x30, 0x82, 0x01, 0x06, 0x30, 0x09, 0x06, 0x03, 0x55, 0x1d,
        0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14,
        0x32, 0x67, 0xe1, 0xb1, 0x79, 0xd2, 0x81, 0xfc, 0x9f, 0x23, 0x0c, 0x70, 0x40, 0x50, 0xb5, 0x46,
        0x56, 0xb8, 0x30, 0x36, 0x30, 0x81, 0xc4, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x81, 0xbc, 0x30,
        0x81, 0xb9, 0x80, 0x14, 0x73, 0xb0, 0x1c, 0xa4, 0x2f, 0x82, 0xcb, 0xcf, 0x47, 0xa5, 0x38, 0xd7,
        0xb0, 0x04, 0x82, 0x3a, 0x7e, 0x72, 0x15, 0x21, 0xa1, 0x81, 0x9d, 0xa4, 0x81, 0x9a, 0x30, 0x81,
        0x97, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13,
        0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67,
        0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65,
        0x61, 0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07,
        0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b,
        0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x18, 0x30,
        0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20,
        0x72, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48,
        0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c,
        0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x01, 0x63, 0x30, 0x13, 0x06, 0x03, 0x55,
        0x1d, 0x25, 0x04, 0x0c, 0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x09,
        0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03,
        0x82, 0x01, 0x01, 0x00, 0x07, 0xca, 0xa6, 0xa1, 0x9f, 0xbf, 0xaf, 0x92, 0x41, 0x35, 0x66, 0x51,
        0xac, 0xbc, 0x2c, 0xec, 0xe7, 0x8d, 0x65, 0x7e, 0xe9, 0x40, 0xfe, 0x5a, 0xab, 0x8a, 0x1d, 0x3d,
        0x13, 0xdb, 0xb4, 0x43, 0x2c, 0x9a, 0x36, 0x98, 0x21, 0xa5, 0xe8, 0xca, 0xa9, 0x4d, 0xfc, 0xe3,
        0xf7, 0x45, 0x88, 0xcd, 0x33, 0xbf, 0x8a, 0x62, 0x10, 0x2f, 0xb2, 0xb7, 0x04, 0xef, 0x26, 0x43,
        0x51, 0x1d, 0x43, 0x62, 0x7d, 0x1e, 0x50, 0xc8, 0xd5, 0x98, 0x94, 0x71, 0x8f, 0x3b, 0x23, 0x26,
        0xf1, 0x71, 0x8e, 0x1e, 0x3d, 0x3f, 0x21, 0xfd, 0xb7, 0x2d, 0x65, 0xe4, 0x07, 0x65, 0xac, 0x3c,
        0xfc, 0xc0, 0x47, 0xa9, 0x32, 0xf6, 0xda, 0x26, 0x93, 0x10, 0xb2, 0xd1, 0x6d, 0xc8, 0x81, 0x31,
        0x7c, 0xb0, 0x6b, 0xc5, 0x22, 0x8d, 0xb3, 0xfa, 0xbe, 0x82, 0xea, 0x41, 0x42, 0xc4, 0xc0, 0xef,
        0xe3, 0x84, 0x0f, 0x6f, 0x9a, 0x03, 0x63, 0xb3, 0x30, 0xe0, 0x31, 0x81, 0x2a, 0x16, 0xb3, 0x47,
        0xd9, 0x5b, 0x38, 0x93, 0x07, 0xd0, 0x6e, 0x79, 0x52, 0x2c, 0xe5, 0x50, 0x84, 0x79, 0x10, 0xe7,
        0xf6, 0x31, 0x7a, 0x3e, 0x48, 0xa2, 0x38, 0x21, 0x90, 0x7a, 0xf2, 0x5f, 0x48, 0xa4, 0x46, 0x93,
        0x87, 0xdd, 0x5c, 0x83, 0x64, 0xea, 0xb5, 0x99, 0xa2, 0xe9, 0x01, 0x40, 0xfe, 0xf0, 0x48, 0x66,
        0x4f, 0x96, 0xf7, 0x83, 0x52, 0xf8, 0x6d, 0xf8, 0x5f, 0xed, 0x0c, 0xbb, 0xbe, 0xd0, 0x69, 0x10,
        0x4b, 0x99, 0x8f, 0xf8, 0x61, 0x53, 0x9d, 0x12, 0xca, 0x86, 0xaa, 0xb1, 0x80, 0xb4, 0xa6, 0xc1,
        0xcb, 0xb7, 0x48, 0xf7, 0x9f, 0x55, 0xb4, 0x6e, 0xab, 0xd3, 0xa1, 0xaa, 0x4b, 0xa7, 0x21, 0x6e,
        0x16, 0x7f, 0xad, 0xbb, 0xea, 0x0f, 0x41, 0x80, 0x9b, 0x7f, 0xd6, 0x46, 0xa2, 0xc0, 0x61, 0x72,
        0x59, 0x59, 0xa0, 0x07
    };
    OcspEntry entry[1];
    CertStatus status[1];
    OcspRequest* request = NULL;
#ifndef NO_FILESYSTEM
    const char* ca_cert = "./certs/ca-cert.pem";
#endif

    byte serial[] = {0x05};
    byte issuerHash[] = {0x71, 0x4d, 0x82, 0x23, 0x40, 0x59, 0xc0, 0x96, 0xa1, 0x37, 0x43, 0xfa, 0x31, 0xdb, 0xba, 0xb1, 0x43, 0x18, 0xda, 0x04};
    byte issuerKeyHash[] = {0x83, 0xc6, 0x3a, 0x89, 0x2c, 0x81, 0xf4, 0x02, 0xd7, 0x9d, 0x4c, 0xe2, 0x2a, 0xc0, 0x71, 0x82, 0x64, 0x44, 0xda, 0x0e};


    XMEMSET(entry, 0, sizeof(OcspEntry));
    XMEMSET(status, 0, sizeof(CertStatus));

    ExpectNotNull(request = wolfSSL_OCSP_REQUEST_new());
    ExpectNotNull(request->serial = (byte*)XMALLOC(sizeof(serial), NULL,
                                                   DYNAMIC_TYPE_OCSP_REQUEST));

    if ((request != NULL) && (request->serial != NULL)) {
        request->serialSz = sizeof(serial);
        XMEMCPY(request->serial, serial, sizeof(serial));
        XMEMCPY(request->issuerHash, issuerHash, sizeof(issuerHash));
        XMEMCPY(request->issuerKeyHash, issuerKeyHash, sizeof(issuerKeyHash));
    }

    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(cm, 0), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm,
        "./certs/ocsp/intermediate1-ca-cert.pem", NULL), WOLFSSL_SUCCESS);

    /* Response should be valid. */
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, response,
        sizeof(response), NULL, status, entry, request), WOLFSSL_SUCCESS);

    /* Flip a byte in the request serial number, response should be invalid
     * now. */
    if ((request != NULL) && (request->serial != NULL))
        request->serial[0] ^= request->serial[0];
    ExpectIntNE(wolfSSL_CertManagerCheckOCSPResponse(cm, response,
        sizeof(response), NULL, status, entry, request), WOLFSSL_SUCCESS);

#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, server_cert_der_2048,
        sizeof(server_cert_der_2048)), ASN_NO_SIGNER_E);
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, server_cert_der_2048,
        sizeof(server_cert_der_2048)), 1);
#endif

    wolfSSL_OCSP_REQUEST_free(request);
    wolfSSL_CertManagerFree(cm);
#endif /* OPENSSL_ALL || WOLFSSL_NGINX ||  WOLFSSL_HAPROXY ||
        * WOLFSSL_APACHE_HTTPD || HAVE_LIGHTY */
#endif /* HAVE_OCSP */
    return EXPECT_RESULT();
}

static int test_wolfSSL_CheckOCSPResponse(void)
{
    EXPECT_DECLS;
#if defined(HAVE_OCSP) && !defined(NO_RSA) && defined(OPENSSL_ALL)
    const char* responseFile = "./certs/ocsp/test-response.der";
    const char* responseMultiFile = "./certs/ocsp/test-multi-response.der";
    const char* responseNoInternFile =
        "./certs/ocsp/test-response-nointern.der";
    const char* caFile = "./certs/ocsp/root-ca-cert.pem";
    OcspResponse* res = NULL;
    byte data[4096];
    const unsigned char* pt;
    int  dataSz = 0; /* initialize to mitigate spurious maybe-uninitialized from
                      * gcc sanitizer with --enable-heapmath.
                      */
    XFILE f = XBADFILE;
    WOLFSSL_OCSP_BASICRESP* bs = NULL;
    WOLFSSL_X509_STORE* st = NULL;
    WOLFSSL_X509* issuer = NULL;


    ExpectTrue((f = XFOPEN(responseFile, "rb")) != XBADFILE);
    ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    pt = data;
    ExpectNotNull(res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz));
    ExpectNotNull(issuer = wolfSSL_X509_load_certificate_file(caFile,
        SSL_FILETYPE_PEM));
    ExpectNotNull(st = wolfSSL_X509_STORE_new());
    ExpectIntEQ(wolfSSL_X509_STORE_add_cert(st, issuer), WOLFSSL_SUCCESS);
    ExpectNotNull(bs = wolfSSL_OCSP_response_get1_basic(res));
    ExpectIntEQ(wolfSSL_OCSP_basic_verify(bs, NULL, st, 0), WOLFSSL_SUCCESS);
    wolfSSL_OCSP_BASICRESP_free(bs);
    bs = NULL;
    wolfSSL_OCSP_RESPONSE_free(res);
    res = NULL;
    wolfSSL_X509_STORE_free(st);
    st = NULL;
    wolfSSL_X509_free(issuer);
    issuer = NULL;

    /* check loading a response with optional certs */
    ExpectTrue((f = XFOPEN(responseNoInternFile, "rb")) != XBADFILE);
    ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    f = XBADFILE;

    pt = data;
    ExpectNotNull(res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz));
    wolfSSL_OCSP_RESPONSE_free(res);
    res = NULL;

    /* check loading a response with multiple certs */
    {
        WOLFSSL_CERT_MANAGER* cm = NULL;
        OcspEntry *entry = NULL;
        CertStatus* status = NULL;
        OcspRequest* request = NULL;

        byte serial1[] = {0x01};
        byte serial[] = {0x02};

        byte issuerHash[] = {
            0x44, 0xA8, 0xDB, 0xD1, 0xBC, 0x97, 0x0A, 0x83,
            0x3B, 0x5B, 0x31, 0x9A, 0x4C, 0xB8, 0xD2, 0x52,
            0x37, 0x15, 0x8A, 0x88
        };
        byte issuerKeyHash[] = {
            0x73, 0xB0, 0x1C, 0xA4, 0x2F, 0x82, 0xCB, 0xCF,
            0x47, 0xA5, 0x38, 0xD7, 0xB0, 0x04, 0x82, 0x3A,
            0x7E, 0x72, 0x15, 0x21
        };

        ExpectNotNull(entry = (OcspEntry*)XMALLOC(sizeof(OcspEntry), NULL,
             DYNAMIC_TYPE_OPENSSL));

        ExpectNotNull(status = (CertStatus*)XMALLOC(sizeof(CertStatus), NULL,
             DYNAMIC_TYPE_OPENSSL));

        if (entry != NULL)
            XMEMSET(entry, 0, sizeof(OcspEntry));
        if (status != NULL)
            XMEMSET(status, 0, sizeof(CertStatus));

        ExpectNotNull(request = wolfSSL_OCSP_REQUEST_new());
        ExpectNotNull(request->serial = (byte*)XMALLOC(sizeof(serial), NULL,
                                                    DYNAMIC_TYPE_OCSP_REQUEST));

        if (request != NULL && request->serial != NULL) {
             request->serialSz = sizeof(serial);
             XMEMCPY(request->serial, serial, sizeof(serial));
             XMEMCPY(request->issuerHash, issuerHash, sizeof(issuerHash));
             XMEMCPY(request->issuerKeyHash, issuerKeyHash,
                 sizeof(issuerKeyHash));
        }

        ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
        ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(cm, 0), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm, caFile, NULL),
            WOLFSSL_SUCCESS);

        ExpectTrue((f = XFOPEN(responseMultiFile, "rb")) != XBADFILE);
        ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
        if (f != XBADFILE)
            XFCLOSE(f);
        f = XBADFILE;

        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, status, entry, request), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, entry->status, entry, request), WOLFSSL_SUCCESS);
        ExpectNotNull(entry->status);

        if (request != NULL && request->serial != NULL)
            XMEMCPY(request->serial, serial1, sizeof(serial1));
        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, status, entry, request), WOLFSSL_SUCCESS);

        /* store both status's in the entry to check that "next" is not
         * overwritten */
        if (EXPECT_SUCCESS() && status != NULL && entry != NULL) {
            status->next  = entry->status;
            entry->status = status;
        }

        if (request != NULL && request->serial != NULL)
            XMEMCPY(request->serial, serial, sizeof(serial));
        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, entry->status, entry, request), WOLFSSL_SUCCESS);
        ExpectNotNull(entry->status->next);

        /* compare the status found */
        ExpectIntEQ(status->serialSz, entry->status->serialSz);
        ExpectIntEQ(XMEMCMP(status->serial, entry->status->serial,
            status->serialSz), 0);

        if (status != NULL && entry != NULL && entry->status != status) {
            XFREE(status, NULL, DYNAMIC_TYPE_OPENSSL);
        }
        wolfSSL_OCSP_CERTID_free(entry);
        wolfSSL_OCSP_REQUEST_free(request);
        wolfSSL_CertManagerFree(cm);
    }

#if defined(WC_RSA_PSS)
    {
        const char* responsePssFile = "./certs/ocsp/test-response-rsapss.der";

        /* check loading a response with RSA-PSS signature */
        ExpectTrue((f = XFOPEN(responsePssFile, "rb")) != XBADFILE);
        ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
        if (f != XBADFILE)
            XFCLOSE(f);

        pt = data;
        ExpectNotNull(res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz));

        /* try to verify the response */
        ExpectNotNull(issuer = wolfSSL_X509_load_certificate_file(caFile,
            SSL_FILETYPE_PEM));
        ExpectNotNull(st = wolfSSL_X509_STORE_new());
        ExpectIntEQ(wolfSSL_X509_STORE_add_cert(st, issuer), WOLFSSL_SUCCESS);
        ExpectNotNull(bs = wolfSSL_OCSP_response_get1_basic(res));
        ExpectIntEQ(wolfSSL_OCSP_basic_verify(bs, NULL, st, 0),
            WOLFSSL_SUCCESS);
        wolfSSL_OCSP_BASICRESP_free(bs);
        wolfSSL_OCSP_RESPONSE_free(res);
        wolfSSL_X509_STORE_free(st);
        wolfSSL_X509_free(issuer);
    }
#endif
#endif /* HAVE_OCSP */
    return EXPECT_RESULT();
}

static int test_wolfSSL_FPKI(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_FPKI) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE f = XBADFILE;
    const char* fpkiCert = "./certs/fpki-cert.der";
    DecodedCert cert;
    byte buf[4096];
    byte* uuid = NULL;
    byte* fascn = NULL;
    word32 fascnSz;
    word32 uuidSz;
    int bytes = 0;

    ExpectTrue((f = XFOPEN(fpkiCert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, NULL, &fascnSz), LENGTH_ONLY_E) ;
    ExpectNotNull(fascn = (byte*)XMALLOC(fascnSz, NULL,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, fascn, &fascnSz), 0);
    XFREE(fascn, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    ExpectIntEQ(wc_GetUUIDFromCert(&cert, NULL, &uuidSz), LENGTH_ONLY_E);
    ExpectNotNull(uuid = (byte*)XMALLOC(uuidSz, NULL, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetUUIDFromCert(&cert, uuid, &uuidSz), 0);
    XFREE(uuid, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    wc_FreeDecodedCert(&cert);
#endif

    return EXPECT_RESULT();
}

/* use RID in confuncture with other names to test parsing of unknown other
 * names */
static int test_wolfSSL_OtherName(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE f = XBADFILE;
    const char* ridCert = "./certs/rid-cert.der";
    DecodedCert cert;
    byte buf[4096];
    int bytes = 0;

    ExpectTrue((f = XFOPEN(ridCert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    wc_FreeDecodedCert(&cert);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertRsaPss(void)
{
    EXPECT_DECLS;
/* FIPS v2 and below don't support long salts. */
#if !defined(NO_RSA) && defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM) && \
    (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
     (HAVE_FIPS_VERSION > 2))) && (!defined(HAVE_SELFTEST) || \
     (defined(HAVE_SELFTEST_VERSION) && (HAVE_SELFTEST_VERSION > 2)))
    XFILE f = XBADFILE;
    const char* rsaPssSha256Cert = "./certs/rsapss/ca-rsapss.der";
    const char* rsaPssRootSha256Cert = "./certs/rsapss/root-rsapss.pem";
#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_PSS_LONG_SALT) && \
    RSA_MAX_SIZE >= 3072
    const char* rsaPssSha384Cert = "./certs/rsapss/ca-3072-rsapss.der";
#endif
#if defined(WOLFSSL_SHA384) && RSA_MAX_SIZE >= 3072
    const char* rsaPssRootSha384Cert = "./certs/rsapss/root-3072-rsapss.pem";
#endif
    DecodedCert cert;
    byte buf[4096];
    int bytes = 0;
    WOLFSSL_CERT_MANAGER* cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, rsaPssRootSha256Cert, NULL));
#if defined(WOLFSSL_SHA384) && RSA_MAX_SIZE >= 3072
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, rsaPssRootSha384Cert, NULL));
#endif

    ExpectTrue((f = XFOPEN(rsaPssSha256Cert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    wc_InitDecodedCert(&cert, buf, bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, VERIFY, cm), 0);
    wc_FreeDecodedCert(&cert);

#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_PSS_LONG_SALT) && \
    RSA_MAX_SIZE >= 3072
    ExpectTrue((f = XFOPEN(rsaPssSha384Cert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    wc_InitDecodedCert(&cert, buf, bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, VERIFY, cm), 0);
    wc_FreeDecodedCert(&cert);
#endif

    wolfSSL_CertManagerFree(cm);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_locations_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX* ctx = NULL;
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    /* test good CA */
    ExpectTrue(WOLFSSL_SUCCESS ==
        wolfSSL_CTX_load_verify_locations_ex(ctx, ca_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE));

    /* test expired CA */
#if !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_ASN_TIME)
    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#endif
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_buffer_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    defined(USE_CERT_BUFFERS_2048)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX* ctx;
    const char* ca_expired_cert_file = "./certs/test/expired/expired-ca.der";
    byte ca_expired_cert[TWOK_BUF];
    word32 sizeof_ca_expired_cert = 0;
    XFILE fp = XBADFILE;

#ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
#else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
#endif
    ExpectNotNull(ctx);

    /* test good CA */
    ExpectTrue(WOLFSSL_SUCCESS ==
        wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_cert_der_2048,
            sizeof_ca_cert_der_2048, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE));

    /* load expired CA */
    XMEMSET(ca_expired_cert, 0, sizeof(ca_expired_cert));
    ExpectTrue((fp = XFOPEN(ca_expired_cert_file, "rb")) != XBADFILE);
    ExpectIntGT(sizeof_ca_expired_cert = (word32)XFREAD(ca_expired_cert, 1,
        sizeof(ca_expired_cert), fp), 0);
    if (fp != XBADFILE)
        XFCLOSE(fp);

    /* test expired CA failure */


#if !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_ASN_TIME)
    ExpectIntNE(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#endif
    /* test expired CA success */
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_chain_buffer_format(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && defined(OPENSSL_EXTRA) && \
    defined(WOLFSSL_CERT_GEN) && defined(USE_CERT_BUFFERS_2048) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    ExpectTrue(WOLFSSL_SUCCESS == wolfSSL_CTX_load_verify_chain_buffer_format(
        ctx, ca_cert_chain_der, sizeof_ca_cert_chain_der,
        WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_add1_chain_cert(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && defined(OPENSSL_EXTRA) && \
    defined(KEEP_OUR_CERT) && !defined(NO_RSA) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*        ctx;
    WOLFSSL*            ssl = NULL;
    const char *certChain[] = {
            "./certs/intermediate/client-int-cert.pem",
            "./certs/intermediate/ca-int2-cert.pem",
            "./certs/intermediate/ca-int-cert.pem",
            "./certs/ca-cert.pem",
            NULL
    };
    const char** cert;
    WOLFSSL_X509* x509 = NULL;
    WOLF_STACK_OF(X509)* chain = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    for (cert = certChain; EXPECT_SUCCESS() && *cert != NULL; cert++) {
        ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(*cert,
            WOLFSSL_FILETYPE_PEM));
        ExpectIntEQ(SSL_CTX_add1_chain_cert(ctx, x509), 1);
        X509_free(x509);
        x509 = NULL;
    }
    for (cert = certChain; EXPECT_SUCCESS() && *cert != NULL; cert++) {
        ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(*cert,
            WOLFSSL_FILETYPE_PEM));
        ExpectIntEQ(SSL_add1_chain_cert(ssl, x509), 1);
        X509_free(x509);
        x509 = NULL;
    }

    ExpectIntEQ(SSL_CTX_get0_chain_certs(ctx, &chain), 1);
    ExpectIntEQ(sk_X509_num(chain), 3);
    ExpectIntEQ(SSL_get0_chain_certs(ssl, &chain), 1);
    ExpectIntEQ(sk_X509_num(chain), 3);

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_certificate_chain_file_format(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    const char* server_chain_der = "./certs/server-cert-chain.der";
    const char* client_single_pem = "./certs/client-cert.pem";
    WOLFSSL_CTX* ctx;

    (void)server_chain_der;
    (void)client_single_pem;
    (void)ctx;

  #ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #endif
    ExpectNotNull(ctx);

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file_format(ctx,
        server_chain_der, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file_format(ctx,
        client_single_pem, WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_SetTmpDH_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx = NULL;

    (void)ctx;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* invalid context */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(NULL,
                dhParamFile, WOLFSSL_FILETYPE_PEM));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx,
                NULL, WOLFSSL_FILETYPE_PEM));

    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx,
                bogusFile, WOLFSSL_FILETYPE_PEM));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx, dhParamFile,
                WOLFSSL_FILETYPE_PEM));

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_SetTmpDH_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx = NULL;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* invalid context */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(NULL,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(NULL, NULL,
                0, WOLFSSL_FILETYPE_ASN1));

    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dsa_key_der_2048, sizeof_dsa_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_SetMinMaxDhKey_Sz(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx;

    (void)ctx;

  #ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #endif
    ExpectNotNull(ctx);

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 3072));

    ExpectIntEQ(DH_KEY_SIZE_E, wolfSSL_CTX_SetTmpDH_buffer(ctx, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 2048));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 1024));

    ExpectIntEQ(DH_KEY_SIZE_E, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 2048));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_der_load_verify_locations(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && defined(WOLFSSL_DER_LOAD) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
    const char* derCert = "./certs/server-cert.der";
    const char* nullPath = NULL;
    const char* invalidPath = "./certs/this-cert-does-not-exist.der";
    const char* emptyPath = "";

    /* der load Case 1 ctx NULL */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_FAILURE);

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* Case 2 filePath NULL */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, nullPath,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_FAILURE);
    /* Case 3 invalid format */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_FAILURE);
    /* Case 4 filePath not valid */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, invalidPath,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_FAILURE);
    /* Case 5 filePath empty */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, emptyPath,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_FAILURE);
#ifndef NO_RSA
    /* Case 6 success case */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_enable_disable(void)
{
    EXPECT_DECLS;
#ifndef NO_CERTS
    WOLFSSL_CTX* ctx = NULL;

  #ifdef HAVE_CRL
    ExpectIntEQ(wolfSSL_CTX_DisableCRL(ctx), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_EnableCRL(ctx, 0), BAD_FUNC_ARG);
  #endif

  #ifdef HAVE_OCSP
    ExpectIntEQ(wolfSSL_CTX_DisableOCSP(ctx), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, 0), BAD_FUNC_ARG);
  #endif

  #if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
      defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    ExpectIntEQ(wolfSSL_CTX_DisableOCSPStapling(ctx), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_EnableOCSPStapling(ctx), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_CTX_EnableOCSPMustStaple(ctx), BAD_FUNC_ARG);
  #endif

  #ifndef NO_WOLFSSL_CLIENT

    #ifdef HAVE_EXTENDED_MASTER
    ExpectIntEQ(wolfSSL_CTX_DisableExtendedMasterSecret(ctx), BAD_FUNC_ARG);
    #endif
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    #ifdef HAVE_EXTENDED_MASTER
    ExpectIntEQ(wolfSSL_CTX_DisableExtendedMasterSecret(ctx), WOLFSSL_SUCCESS);
    #endif

  #elif !defined(NO_WOLFSSL_SERVER)
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

  #if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)

    #ifdef HAVE_CRL
        ExpectIntEQ(wolfSSL_CTX_DisableCRL(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableCRL(ctx, 0), WOLFSSL_SUCCESS);
    #endif

    #ifdef HAVE_OCSP
        ExpectIntEQ(wolfSSL_CTX_DisableOCSP(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_URL_OVERRIDE),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_NO_NONCE),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_CHECKALL),
                    WOLFSSL_SUCCESS);
    #endif

    #if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
        defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
        ExpectIntEQ(wolfSSL_CTX_DisableOCSPStapling(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSPStapling(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WOLFSSL_SUCCESS);
    #endif
        wolfSSL_CTX_free(ctx);
  #endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* NO_CERTS */

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_ticket_API(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SESSION_TICKET) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX* ctx = NULL;
    void *userCtx = (void*)"this is my ctx";

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_set_TicketEncCtx(ctx, userCtx));
    ExpectTrue(userCtx == wolfSSL_CTX_get_TicketEncCtx(ctx));

    wolfSSL_CTX_free(ctx);

    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_set_TicketEncCtx(NULL, userCtx));
    ExpectNull(wolfSSL_CTX_get_TicketEncCtx(NULL));
#endif /* HAVE_SESSION_TICKET && !NO_WOLFSSL_SERVER */
    return EXPECT_RESULT();
}

static int test_wolfSSL_set_minmax_proto_version(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL *ssl = NULL;

    (void)ssl;

#ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(NULL, 0), SSL_FAILURE);
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(NULL, 0), SSL_FAILURE);
    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(ctx, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, 0), SSL_SUCCESS);

    ExpectIntEQ(wolfSSL_set_min_proto_version(NULL, 0), SSL_FAILURE);
    ExpectIntEQ(wolfSSL_set_min_proto_version(ssl, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_set_max_proto_version(NULL, 0), SSL_FAILURE);
    ExpectIntEQ(wolfSSL_set_max_proto_version(ssl, 0), SSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    ctx = NULL;
#endif
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(NULL, 0), SSL_FAILURE);
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(NULL, 0), SSL_FAILURE);
    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(ctx, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, 0), SSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
#endif

    return EXPECT_RESULT();
}

#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_TLS12) && \
    defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_set_max_proto_version_on_result(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    ExpectStrEQ(wolfSSL_get_version(ssl), "TLSv1.2");
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_max_proto_version_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* Set TLS 1.2 */
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, TLS1_2_VERSION),
            WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}

/* Test using wolfSSL_CTX_set_max_proto_version to limit the version below
 * what was set at ctx creation. */
static int test_wolfSSL_CTX_set_max_proto_version(void)
{
    EXPECT_DECLS;
    test_ssl_cbf client_cbs;
    test_ssl_cbf server_cbs;

    XMEMSET(&client_cbs, 0, sizeof(client_cbs));
    XMEMSET(&server_cbs, 0, sizeof(server_cbs));

    client_cbs.method = wolfTLS_client_method;
    server_cbs.method = wolfTLS_server_method;

    server_cbs.ctx_ready = test_wolfSSL_CTX_set_max_proto_version_ctx_ready;

    client_cbs.on_result = test_wolfSSL_CTX_set_max_proto_version_on_result;
    server_cbs.on_result = test_wolfSSL_CTX_set_max_proto_version_on_result;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbs,
        &server_cbs, NULL), TEST_SUCCESS);

    return EXPECT_RESULT();
}
#else
static int test_wolfSSL_CTX_set_max_proto_version(void)
{
    return TEST_SKIPPED;
}
#endif

/*----------------------------------------------------------------------------*
 | SSL
 *----------------------------------------------------------------------------*/

static int test_server_wolfSSL_new(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
        !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_CTX *ctx_nocert = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx_nocert = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectNotNull(ctx        = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM));

    /* invalid context */
    ExpectNull(ssl = wolfSSL_new(NULL));
#if !defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_QT) && \
        !defined(OPENSSL_EXTRA) && !defined(WOLFSSL_NO_INIT_CTX_KEY)
    ExpectNull(ssl = wolfSSL_new(ctx_nocert));
#endif

    /* success */
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    wolfSSL_CTX_free(ctx_nocert);
#endif

    return EXPECT_RESULT();
}


static int test_client_wolfSSL_new(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
        !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_CTX *ctx_nocert = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx_nocert = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ctx        = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    ExpectTrue(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));

    /* invalid context */
    ExpectNull(ssl = wolfSSL_new(NULL));

    /* success */
    ExpectNotNull(ssl = wolfSSL_new(ctx_nocert));
    wolfSSL_free(ssl);
    ssl = NULL;

    /* success */
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    wolfSSL_free(ssl);

    wolfSSL_CTX_free(ctx);
    wolfSSL_CTX_free(ctx_nocert);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_SetTmpDH_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_DH) && \
        !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#ifndef NO_RSA
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
               WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
               WOLFSSL_FILETYPE_PEM));
#elif defined(HAVE_ECC)
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, eccCertFile,
               WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, eccKeyFile,
               WOLFSSL_FILETYPE_PEM));
#elif defined(HAVE_ED25519)
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, edCertFile,
               WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, edKeyFile,
               WOLFSSL_FILETYPE_PEM));
#elif defined(HAVE_ED448)
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, ed448CertFile,
               WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, ed448KeyFile,
               WOLFSSL_FILETYPE_PEM));
#endif
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* invalid ssl */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(NULL,
                dhParamFile, WOLFSSL_FILETYPE_PEM));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl,
                NULL, WOLFSSL_FILETYPE_PEM));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl,
                bogusFile, WOLFSSL_FILETYPE_PEM));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl, dhParamFile,
                WOLFSSL_FILETYPE_PEM));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_SetTmpDH_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_DH) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048,
               sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx, server_key_der_2048,
               sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* invalid ssl */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(NULL, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(NULL, NULL,
                0, WOLFSSL_FILETYPE_ASN1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dsa_key_der_2048,
                sizeof_dsa_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_SetMinMaxDhKey_Sz(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_DH) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_CTX *ctx2 = NULL;
    WOLFSSL *ssl = NULL;
    WOLFSSL *ssl2 = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx, server_key_der_2048,
                sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 3072));
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    ExpectNotNull(ctx2 = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_buffer(ctx2, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx2, server_key_der_2048,
                sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 1024));
    ExpectNotNull(ssl2 = wolfSSL_new(ctx2));

    ExpectIntEQ(DH_KEY_SIZE_E, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMinDhKey_Sz(ssl, 2048));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMinDhKey_Sz(ssl, 3072));
    ExpectIntEQ(DH_KEY_SIZE_E, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl2, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMaxDhKey_Sz(ssl2, 2048));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl2, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMaxDhKey_Sz(ssl2, 1024));
    ExpectIntEQ(DH_KEY_SIZE_E, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_free(ssl2);
    wolfSSL_CTX_free(ctx2);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}


/* Test function for wolfSSL_SetMinVersion. Sets the minimum downgrade version
 * allowed.
 * POST: return 1 on success.
 */
static int test_wolfSSL_SetMinVersion(void)
{
    int                 res = TEST_SKIPPED;
#ifndef NO_WOLFSSL_CLIENT
    int                 failFlag = WOLFSSL_SUCCESS;
    WOLFSSL_CTX*        ctx = NULL;
    WOLFSSL*            ssl = NULL;
    int                 itr;

    #ifndef NO_OLD_TLS
        const int versions[]  =  {
                            #ifdef WOLFSSL_ALLOW_TLSV10
                                   WOLFSSL_TLSV1,
                            #endif
                                   WOLFSSL_TLSV1_1,
                                   WOLFSSL_TLSV1_2};
    #elif !defined(WOLFSSL_NO_TLS12)
        const int versions[]  =  { WOLFSSL_TLSV1_2 };
    #else
        const int versions[]  =  { WOLFSSL_TLSV1_3 };
    #endif

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    ssl = wolfSSL_new(ctx);

    for (itr = 0; itr < (int)(sizeof(versions)/sizeof(int)); itr++) {
       if (wolfSSL_SetMinVersion(ssl, *(versions + itr)) != WOLFSSL_SUCCESS) {
            failFlag = WOLFSSL_FAILURE;
        }
    }

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(failFlag == WOLFSSL_SUCCESS);
#endif
    return res;

} /* END test_wolfSSL_SetMinVersion */


#ifdef OPENSSL_EXTRA
static int test_ED25519(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT) && \
    defined(WOLFSSL_KEY_GEN)
    byte         priv[ED25519_PRV_KEY_SIZE];
    unsigned int privSz = (unsigned int)sizeof(priv);
    byte         pub[ED25519_PUB_KEY_SIZE];
    unsigned int pubSz = (unsigned int)sizeof(pub);
#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_IMPORT)
    const char*  msg = TEST_STRING;
    unsigned int msglen = (unsigned int)TEST_STRING_SZ;
    byte         sig[ED25519_SIG_SIZE];
    unsigned int sigSz = (unsigned int)sizeof(sig);
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_IMPORT */

    ExpectIntEQ(wolfSSL_ED25519_generate_key(priv, &privSz, pub, &pubSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(privSz, ED25519_PRV_KEY_SIZE);
    ExpectIntEQ(pubSz, ED25519_PUB_KEY_SIZE);

#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_IMPORT)
    ExpectIntEQ(wolfSSL_ED25519_sign((byte*)msg, msglen, priv, privSz, sig,
        &sigSz), WOLFSSL_SUCCESS);
    ExpectIntEQ(sigSz, ED25519_SIG_SIZE);

#ifdef HAVE_ED25519_VERIFY
    ExpectIntEQ(wolfSSL_ED25519_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), WOLFSSL_SUCCESS);
#endif /* HAVE_ED25519_VERIFY */
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_IMPORT */
#endif /* HAVE_ED25519 && HAVE_ED25519_KEY_EXPORT && WOLFSSL_KEY_GEN */
    return EXPECT_RESULT();
}

static int test_ED448(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT) && \
    defined(WOLFSSL_KEY_GEN)
    byte         priv[ED448_PRV_KEY_SIZE];
    unsigned int privSz = (unsigned int)sizeof(priv);
    byte         pub[ED448_PUB_KEY_SIZE];
    unsigned int pubSz = (unsigned int)sizeof(pub);
#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_IMPORT)
    const char*  msg = TEST_STRING;
    unsigned int msglen = (unsigned int)TEST_STRING_SZ;
    byte         sig[ED448_SIG_SIZE];
    unsigned int sigSz = (unsigned int)sizeof(sig);
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_IMPORT */

    ExpectIntEQ(wolfSSL_ED448_generate_key(priv, &privSz, pub, &pubSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(privSz, ED448_PRV_KEY_SIZE);
    ExpectIntEQ(pubSz, ED448_PUB_KEY_SIZE);

#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_IMPORT)
    ExpectIntEQ(wolfSSL_ED448_sign((byte*)msg, msglen, priv, privSz, sig,
        &sigSz), WOLFSSL_SUCCESS);
    ExpectIntEQ(sigSz, ED448_SIG_SIZE);

#ifdef HAVE_ED448_VERIFY
    ExpectIntEQ(wolfSSL_ED448_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), WOLFSSL_SUCCESS);
#endif /* HAVE_ED448_VERIFY */
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_IMPORT */
#endif /* HAVE_ED448 && HAVE_ED448_KEY_EXPORT && WOLFSSL_KEY_GEN */
    return EXPECT_RESULT();
}
#endif /* OPENSSL_EXTRA */

#include <wolfssl/openssl/pem.h>
/*----------------------------------------------------------------------------*
 | EVP
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_EVP_PKEY_print_public(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_BIO)
    WOLFSSL_BIO* rbio = NULL;
    WOLFSSL_BIO* wbio = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;
    char line[256] = { 0 };
    char line1[256] = { 0 };
    int i;

    /* test error cases */
    ExpectIntEQ( EVP_PKEY_print_public(NULL,NULL,0,NULL),0L);

    /*
     *  test RSA public key print
     *  in this test, pass '3' for indent
     */
#if !defined(NO_RSA) && defined(USE_CERT_BUFFERS_1024)

    ExpectNotNull(rbio = BIO_new_mem_buf( client_keypub_der_1024,
        sizeof_client_keypub_der_1024));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,3,NULL),1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "   RSA Public-Key: (1024 bit)\n");
    ExpectIntEQ(XSTRNCMP(line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "   Modulus:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "       00:bc:73:0e:a8:49:f3:74:a2:a9:ef:18:a5:da:55:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* skip to the end of modulus element*/
    for (i = 0; i < 8 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "   Exponent: 65537 (0x010001)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif  /* !NO_RSA && USE_CERT_BUFFERS_1024*/

    /*
     *  test DSA public key print
     */
#if !defined(NO_DSA) && defined(USE_CERT_BUFFERS_2048)
    ExpectNotNull(rbio = BIO_new_mem_buf( dsa_pub_key_der_2048,
        sizeof_dsa_pub_key_der_2048));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL),1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "DSA Public-Key: (2048 bit)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "pub:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
        "    00:C2:35:2D:EC:83:83:6C:73:13:9E:52:7C:74:C8:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of pub element*/
    for (i = 0; i < 17 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "P:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of P element*/
    for (i = 0; i < 18 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "Q:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of Q element*/
    for (i = 0; i < 3 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }
    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "G:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of G element*/
    for (i = 0; i < 18 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }
    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* !NO_DSA && USE_CERT_BUFFERS_2048 */

    /*
     *  test ECC public key print
     */
#if defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)

    ExpectNotNull(rbio = BIO_new_mem_buf( ecc_clikeypub_der_256,
        sizeof_ecc_clikeypub_der_256));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL),1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "Public-Key: (256 bit)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "pub:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
            "    04:55:BF:F4:0F:44:50:9A:3D:CE:9B:B7:F0:C5:4D:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of pub element*/
    for (i = 0; i < 4 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "ASN1 OID: prime256v1\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "NIST CURVE: P-256\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* HAVE_ECC && USE_CERT_BUFFERS_256 */

    /*
     *  test DH public key print
     */
#if defined(WOLFSSL_DH_EXTRA) && defined(USE_CERT_BUFFERS_2048)

    ExpectNotNull(rbio = BIO_new_mem_buf( dh_pub_key_der_2048,
        sizeof_dh_pub_key_der_2048));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL), 1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "DH Public-Key: (2048 bit)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "public-key:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
        "    34:41:BF:E9:F2:11:BF:05:DB:B2:72:A8:29:CC:BD:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of public-key element*/
    for (i = 0; i < 17 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "prime:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
        "    00:D3:B2:99:84:5C:0A:4C:E7:37:CC:FC:18:37:01:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of prime element*/
    for (i = 0; i < 17 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "generator: 2 (0x02)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* WOLFSSL_DH_EXTRA && USE_CERT_BUFFERS_2048 */

    /* to prevent "unused variable" warning */
    (void)pkey;
    (void)wbio;
    (void)rbio;
    (void)line;
    (void)line1;
    (void)i;
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}
/* Test functions for base64 encode/decode */
static int test_wolfSSL_EVP_ENCODE_CTX_new(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && \
( defined(WOLFSSL_BASE64_ENCODE) || defined(WOLFSSL_BASE64_DECODE))
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());
    ExpectIntEQ(ctx->remaining,0);
    ExpectIntEQ(ctx->data[0],0);
    ExpectIntEQ(ctx->data[sizeof(ctx->data) -1],0);
    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && (WOLFSSL_BASE64_ENCODE || WOLFSSL_BASE64_DECODE) */
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_ENCODE_CTX_free(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && \
( defined(WOLFSSL_BASE64_ENCODE) || defined(WOLFSSL_BASE64_DECODE))
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());
    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && (WOLFSSL_BASE64_ENCODE || WOLFSSL_BASE64_DECODE) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_EncodeInit(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());
    ExpectIntEQ(ctx->remaining, 0);
    ExpectIntEQ(ctx->data[0], 0);
    ExpectIntEQ(ctx->data[sizeof(ctx->data) -1], 0);

    if (ctx != NULL) {
        /* make ctx dirty */
        ctx->remaining = 10;
        XMEMSET(ctx->data, 0x77, sizeof(ctx->data));
    }

    EVP_EncodeInit(ctx);

    ExpectIntEQ(ctx->remaining, 0);
    ExpectIntEQ(ctx->data[0], 0);
    ExpectIntEQ(ctx->data[sizeof(ctx->data) -1], 0);

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_EncodeUpdate(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    int outl;
    int total;

    const unsigned char plain0[] = {"Th"};
    const unsigned char plain1[] = {"This is a base64 encodeing test."};
    const unsigned char plain2[] = {"This is additional data."};

    const unsigned char enc0[]   = {"VGg=\n"};
    /* expected encoded result for the first output 64 chars plus trailing LF*/
    const unsigned char enc1[]   = {"VGhpcyBpcyBhIGJhc2U2NCBlbmNvZGVpbmcgdGVzdC5UaGlzIGlzIGFkZGl0aW9u\n"};

    const unsigned char enc2[]   =
    {"VGhpcyBpcyBhIGJhc2U2NCBlbmNvZGVpbmcgdGVzdC5UaGlzIGlzIGFkZGl0aW9u\nYWwgZGF0YS4=\n"};

    unsigned char encOutBuff[300];

    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());

    EVP_EncodeInit(ctx);

    /* illegal parameter test */
    ExpectIntEQ(
        EVP_EncodeUpdate(
            NULL,            /* pass NULL as ctx */
            encOutBuff,
            &outl,
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            NULL,           /* pass NULL as out buff */
            &outl,
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            NULL,            /* pass NULL as outl */
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            NULL,            /* pass NULL as in */
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(EVP_EncodeBlock(NULL, NULL, 0), -1);

    /* meaningless parameter test */

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            plain1,
            0),              /* pass zero input */
        1                    /* expected result code 1: success */
    );

    /* very small data encoding test */

    EVP_EncodeInit(ctx);

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            plain0,
            sizeof(plain0)-1),
        1                    /* expected result code 1: success */
    );
    ExpectIntEQ(outl,0);

    if (EXPECT_SUCCESS()) {
        EVP_EncodeFinal(
            ctx,
            encOutBuff + outl,
            &outl);
    }

    ExpectIntEQ( outl, sizeof(enc0)-1);
    ExpectIntEQ(
        XSTRNCMP(
            (const char*)encOutBuff,
            (const char*)enc0,sizeof(enc0) ),
    0);

    XMEMSET( encOutBuff,0, sizeof(encOutBuff));
    ExpectIntEQ(EVP_EncodeBlock(encOutBuff, plain0, sizeof(plain0)-1),
                sizeof(enc0)-1);
    ExpectIntEQ(
        XSTRNCMP(
            (const char*)encOutBuff,
            (const char*)enc0,sizeof(enc0) ),
    0);

    /* pass small size( < 48bytes ) input, then make sure they are not
     * encoded  and just stored in ctx
     */

    EVP_EncodeInit(ctx);

    total = 0;
    outl = 0;
    XMEMSET( encOutBuff,0, sizeof(encOutBuff));

    ExpectIntEQ(
    EVP_EncodeUpdate(
        ctx,
        encOutBuff,         /* buffer for output */
        &outl,              /* size of output */
        plain1,             /* input */
        sizeof(plain1)-1),  /* size of input */
        1);                 /* expected result code 1:success */

    total += outl;

    ExpectIntEQ(outl, 0);  /* no output expected */
    ExpectIntEQ(ctx->remaining, sizeof(plain1) -1);
    ExpectTrue(
        XSTRNCMP((const char*)(ctx->data),
                 (const char*)plain1,
                 ctx->remaining) ==0 );
    ExpectTrue(encOutBuff[0] == 0);

    /* call wolfSSL_EVP_EncodeUpdate again to make it encode
     * the stored data and the new input together
     */
    ExpectIntEQ(
    EVP_EncodeUpdate(
        ctx,
        encOutBuff + outl,  /* buffer for output */
        &outl,              /* size of output */
        plain2,             /* additional input */
        sizeof(plain2) -1), /* size of additional input */
        1);                 /* expected result code 1:success */

    total += outl;

    ExpectIntNE(outl, 0);   /* some output is expected this time*/
    ExpectIntEQ(outl, BASE64_ENCODE_RESULT_BLOCK_SIZE +1); /* 64 bytes and LF */
    ExpectIntEQ(
        XSTRNCMP((const char*)encOutBuff,(const char*)enc1,sizeof(enc1) ),0);

    /* call wolfSSL_EVP_EncodeFinal to flush all the unprocessed input */
    EVP_EncodeFinal(
        ctx,
        encOutBuff + outl,
        &outl);

    total += outl;

    ExpectIntNE(total,0);
    ExpectIntNE(outl,0);
    ExpectIntEQ(XSTRNCMP(
        (const char*)encOutBuff,(const char*)enc2,sizeof(enc2) ),0);

    /* test with illeagal parameters */
    outl = 1;
    EVP_EncodeFinal(NULL, encOutBuff + outl, &outl);
    ExpectIntEQ(outl, 0);
    outl = 1;
    EVP_EncodeFinal(ctx, NULL, &outl);
    ExpectIntEQ(outl, 0);
    EVP_EncodeFinal(ctx, encOutBuff + outl, NULL);
    EVP_EncodeFinal(NULL, NULL, NULL);

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_EncodeFinal(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    /* tests for wolfSSL_EVP_EncodeFinal are included in
     * test_wolfSSL_EVP_EncodeUpdate
     */
    res = TEST_SUCCESS;
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return res;
}


static int test_wolfSSL_EVP_DecodeInit(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull( ctx = EVP_ENCODE_CTX_new());
    ExpectIntEQ( ctx->remaining,0);
    ExpectIntEQ( ctx->data[0],0);
    ExpectIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    if (ctx != NULL) {
        /* make ctx dirty */
        ctx->remaining = 10;
        XMEMSET( ctx->data, 0x77, sizeof(ctx->data));
    }

    EVP_DecodeInit(ctx);

    ExpectIntEQ( ctx->remaining,0);
    ExpectIntEQ( ctx->data[0],0);
    ExpectIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_DecodeUpdate(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    int outl;
    unsigned char decOutBuff[300];

    EVP_ENCODE_CTX* ctx = NULL;

    static const unsigned char enc1[]   =
            {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg==\n"};
/*    const unsigned char plain1[] =
    {"This is a base64 decoding test."} */

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());

    EVP_DecodeInit(ctx);

    /* illegal parameter tests */

    /* pass NULL as ctx */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            NULL,            /* pass NULL as ctx */
            decOutBuff,
            &outl,
            enc1,
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    ExpectIntEQ( outl, 0);

    /* pass NULL as output */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            NULL,           /* pass NULL as out buff */
            &outl,
            enc1,
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    ExpectIntEQ( outl, 0);

    /* pass NULL as outl */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            NULL,            /* pass NULL as outl */
            enc1,
            sizeof(enc1)-1),
        -1                   /* expected result code -1: fail */
    );

    /* pass NULL as input */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            &outl,
            NULL,            /* pass NULL as in */
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    ExpectIntEQ( outl, 0);

    ExpectIntEQ(EVP_DecodeBlock(NULL, NULL, 0), -1);

    /* pass zero length input */

    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            &outl,
            enc1,
            0),              /* pass zero as input len */
        1                    /* expected result code 1: success */
    );

    /* decode correct base64 string */

    {
        static const unsigned char enc2[]   =
        {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg==\n"};
        static const unsigned char plain2[] =
        {"This is a base64 decoding test."};

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc2,
                sizeof(enc2)-1),
            0                    /* expected result code 0: success */
            );

        ExpectIntEQ(outl,sizeof(plain2) -1);

        ExpectIntEQ(
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl),
            1                    /* expected result code 1: success */
            );
        ExpectIntEQ(outl, 0);   /* expected DecodeFinal output no data */

        ExpectIntEQ(XSTRNCMP( (const char*)plain2,(const char*)decOutBuff,
                              sizeof(plain2) -1 ),0);
        ExpectIntEQ(EVP_DecodeBlock(decOutBuff, enc2, sizeof(enc2)),
                    sizeof(plain2)-1);
        ExpectIntEQ(XSTRNCMP( (const char*)plain2,(const char*)decOutBuff,
                              sizeof(plain2) -1 ),0);
    }

    /* decode correct base64 string which does not have '\n' in its last*/

    {
        static const unsigned char enc3[]   =
        {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg=="}; /* 44 chars */
        static const unsigned char plain3[] =
        {"This is a base64 decoding test."}; /* 31 chars */

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc3,
                sizeof(enc3)-1),
            0                    /* expected result code 0: success */
            );

        ExpectIntEQ(outl,sizeof(plain3)-1);   /* 31 chars should be output */

        ExpectIntEQ(XSTRNCMP( (const char*)plain3,(const char*)decOutBuff,
                              sizeof(plain3) -1 ),0);

        ExpectIntEQ(
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl),
            1                    /* expected result code 1: success */
            );

        ExpectIntEQ(outl,0 );

        ExpectIntEQ(EVP_DecodeBlock(decOutBuff, enc3, sizeof(enc3)-1),
                    sizeof(plain3)-1);
        ExpectIntEQ(XSTRNCMP( (const char*)plain3,(const char*)decOutBuff,
                              sizeof(plain3) -1 ),0);
    }

    /* decode string which has a padding char ('=') in the illegal position*/

    {
        static const unsigned char enc4[]   =
            {"VGhpcyBpcyBhIGJhc2U2N=CBkZWNvZGluZyB0ZXN0Lg==\n"};

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc4,
                sizeof(enc4)-1),
            -1                    /* expected result code -1: error */
            );
        ExpectIntEQ(outl,0);
        ExpectIntEQ(EVP_DecodeBlock(decOutBuff, enc4, sizeof(enc4)-1), -1);
    }

    /* small data decode test */

    {
        static const unsigned char enc00[]   = {"VG"};
        static const unsigned char enc01[]   = {"g=\n"};
        static const unsigned char plain4[]  = {"Th"};

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc00,
                sizeof(enc00)-1),
            1                    /* expected result code 1: success */
            );
        ExpectIntEQ(outl,0);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff + outl,
                &outl,
                enc01,
                sizeof(enc01)-1),
            0                    /* expected result code 0: success */
            );

        ExpectIntEQ(outl,sizeof(plain4)-1);

        /* test with illegal parameters */
        ExpectIntEQ(EVP_DecodeFinal(NULL,decOutBuff + outl,&outl), -1);
        ExpectIntEQ(EVP_DecodeFinal(ctx,NULL,&outl), -1);
        ExpectIntEQ(EVP_DecodeFinal(ctx,decOutBuff + outl, NULL), -1);
        ExpectIntEQ(EVP_DecodeFinal(NULL,NULL, NULL), -1);

        if (EXPECT_SUCCESS()) {
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl);
        }

        ExpectIntEQ( outl, 0);
        ExpectIntEQ(
            XSTRNCMP(
                (const char*)decOutBuff,
                (const char*)plain4,sizeof(plain4)-1 ),
            0);
    }

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_DecodeFinal(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    /* tests for wolfSSL_EVP_DecodeFinal are included in
     * test_wolfSSL_EVP_DecodeUpdate
     */
    res = TEST_SUCCESS;
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return res;
}

/* Test function for wolfSSL_EVP_get_cipherbynid.
 */

#ifdef OPENSSL_EXTRA
static int test_wolfSSL_EVP_get_cipherbynid(void)
{
    EXPECT_DECLS;
#ifndef NO_AES
    const WOLFSSL_EVP_CIPHER* c;

    c = wolfSSL_EVP_get_cipherbynid(419);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_128)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_128_CBC", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(423);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_192)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_192_CBC", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(427);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_256)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_256_CBC", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(904);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_128)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_128_CTR", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(905);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_192)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_192_CTR", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(906);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_256)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_256_CTR", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(418);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_128)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_128_ECB", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(422);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_192)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_192_ECB", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(426);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_256)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_256_ECB", c));
    #else
        ExpectNull(c);
    #endif
#endif /* !NO_AES */

#ifndef NO_DES3
    ExpectNotNull(XSTRCMP("EVP_DES_CBC", wolfSSL_EVP_get_cipherbynid(31)));
#ifdef WOLFSSL_DES_ECB
    ExpectNotNull(XSTRCMP("EVP_DES_ECB", wolfSSL_EVP_get_cipherbynid(29)));
#endif
    ExpectNotNull(XSTRCMP("EVP_DES_EDE3_CBC", wolfSSL_EVP_get_cipherbynid(44)));
#ifdef WOLFSSL_DES_ECB
    ExpectNotNull(XSTRCMP("EVP_DES_EDE3_ECB", wolfSSL_EVP_get_cipherbynid(33)));
#endif
#endif /* !NO_DES3 */

#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    ExpectNotNull(XSTRCMP("EVP_CHACHA20_POLY13O5", EVP_get_cipherbynid(1018)));
#endif

    /* test for nid is out of range */
    ExpectNull(wolfSSL_EVP_get_cipherbynid(1));

    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_CIPHER_CTX(void)
{
    EXPECT_DECLS;
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    EVP_CIPHER_CTX *ctx = EVP_CIPHER_CTX_new();
    const EVP_CIPHER *init = EVP_aes_128_cbc();
    const EVP_CIPHER *test;
    byte key[AES_BLOCK_SIZE] = {0};
    byte iv[AES_BLOCK_SIZE] = {0};

    ExpectNotNull(ctx);
    wolfSSL_EVP_CIPHER_CTX_init(ctx);
    ExpectIntEQ(EVP_CipherInit(ctx, init, key, iv, 1), WOLFSSL_SUCCESS);
    test = EVP_CIPHER_CTX_cipher(ctx);
    ExpectTrue(init == test);
    ExpectIntEQ(EVP_CIPHER_nid(test), NID_aes_128_cbc);

    ExpectIntEQ(EVP_CIPHER_CTX_reset(ctx), WOLFSSL_SUCCESS);
    ExpectIntEQ(EVP_CIPHER_CTX_reset(NULL), WOLFSSL_FAILURE);

    EVP_CIPHER_CTX_free(ctx);
    /* test EVP_CIPHER_CTX_cleanup with NULL */
    ExpectIntEQ(EVP_CIPHER_CTX_cleanup(NULL), WOLFSSL_SUCCESS);
#endif /* !NO_AES && HAVE_AES_CBC && WOLFSSL_AES_128 */
    return EXPECT_RESULT();
}
#endif /* OPENSSL_EXTRA */

/*----------------------------------------------------------------------------*
 | IO
 *----------------------------------------------------------------------------*/

#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) || \
    defined(HAVE_IO_TESTS_DEPENDENCIES)
#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    #ifdef WC_SHA512_DIGEST_SIZE
        #define MD_MAX_SIZE WC_SHA512_DIGEST_SIZE
    #else
        #define MD_MAX_SIZE WC_SHA256_DIGEST_SIZE
    #endif
    byte server_side_msg1[MD_MAX_SIZE] = {0};/* msg sent by server */
    byte server_side_msg2[MD_MAX_SIZE] = {0};/* msg received from client */
    byte client_side_msg1[MD_MAX_SIZE] = {0};/* msg sent by client */
    byte client_side_msg2[MD_MAX_SIZE] = {0};/* msg received from server */
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

/* TODO: Expand and enable this when EVP_chacha20_poly1305 is supported */
#if defined(HAVE_SESSION_TICKET) && defined(OPENSSL_EXTRA) && \
    defined(HAVE_AES_CBC)

    typedef struct openssl_key_ctx {
        byte name[WOLFSSL_TICKET_NAME_SZ]; /* server name */
        byte key[WOLFSSL_TICKET_KEY_SZ]; /* cipher key */
        byte hmacKey[WOLFSSL_TICKET_NAME_SZ]; /* hmac key */
        byte iv[WOLFSSL_TICKET_IV_SZ]; /* cipher iv */
    } openssl_key_ctx;

    static THREAD_LS_T openssl_key_ctx myOpenSSLKey_ctx;
    static THREAD_LS_T WC_RNG myOpenSSLKey_rng;

    static WC_INLINE int OpenSSLTicketInit(void)
    {
        int ret = wc_InitRng(&myOpenSSLKey_rng);
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.name,
                sizeof(myOpenSSLKey_ctx.name));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.key,
                sizeof(myOpenSSLKey_ctx.key));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.hmacKey,
                sizeof(myOpenSSLKey_ctx.hmacKey));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.iv,
                sizeof(myOpenSSLKey_ctx.iv));
        if (ret != 0) return ret;

        return 0;
    }

    static WC_INLINE int myTicketEncCbOpenSSL(WOLFSSL* ssl,
                             byte name[WOLFSSL_TICKET_NAME_SZ],
                             byte iv[WOLFSSL_TICKET_IV_SZ],
                             WOLFSSL_EVP_CIPHER_CTX *ectx,
                             WOLFSSL_HMAC_CTX *hctx, int enc) {
        (void)ssl;
        if (enc) {
            XMEMCPY(name, myOpenSSLKey_ctx.name, sizeof(myOpenSSLKey_ctx.name));
            XMEMCPY(iv, myOpenSSLKey_ctx.iv, sizeof(myOpenSSLKey_ctx.iv));
        }
        else if (XMEMCMP(name, myOpenSSLKey_ctx.name,
                            sizeof(myOpenSSLKey_ctx.name)) != 0 ||
                 XMEMCMP(iv, myOpenSSLKey_ctx.iv,
                            sizeof(myOpenSSLKey_ctx.iv)) != 0) {
            return 0;
        }
        HMAC_Init_ex(hctx, myOpenSSLKey_ctx.hmacKey, WOLFSSL_TICKET_NAME_SZ, EVP_sha256(), NULL);
        if (enc)
            EVP_EncryptInit_ex(ectx, EVP_aes_256_cbc(), NULL, myOpenSSLKey_ctx.key, iv);
        else
            EVP_DecryptInit_ex(ectx, EVP_aes_256_cbc(), NULL, myOpenSSLKey_ctx.key, iv);
        return 1;
    }

    static WC_INLINE void OpenSSLTicketCleanup(void)
    {
        wc_FreeRng(&myOpenSSLKey_rng);
    }
#endif
#endif

/* helper functions */
#ifdef HAVE_SSL_MEMIO_TESTS_DEPENDENCIES
static WC_INLINE int test_ssl_memio_write_cb(WOLFSSL *ssl, char *data, int sz,
    void *ctx)
{
    struct test_ssl_memio_ctx *test_ctx;
    byte *buf;
    int *len;

    test_ctx = (struct test_ssl_memio_ctx*)ctx;

    if (wolfSSL_GetSide(ssl) == WOLFSSL_SERVER_END) {
        buf = test_ctx->c_buff;
        len = &test_ctx->c_len;
    }
    else {
        buf = test_ctx->s_buff;
        len = &test_ctx->s_len;
    }

    if ((unsigned)(*len + sz) > TEST_SSL_MEMIO_BUF_SZ)
        return WOLFSSL_CBIO_ERR_WANT_WRITE;

    XMEMCPY(buf + *len, data, sz);
    *len += sz;

#ifdef WOLFSSL_DUMP_MEMIO_STREAM
    {
        /* This can be imported into Wireshark by transforming the file with
         *   od -Ax -tx1 -v test_output.dump > test_output.dump.hex
         * And then loading test_output.dump.hex into Wireshark using the
         * "Import from Hex Dump..." option ion and selecting the TCP
         * encapsulation option. */
        char dump_file_name[64];
        WOLFSSL_BIO *dump_file;
        sprintf(dump_file_name, "%s/%s.dump", tmpDirName, currentTestName);
        dump_file = wolfSSL_BIO_new_file(dump_file_name, "a");
        if (dump_file != NULL) {
            (void)wolfSSL_BIO_write(dump_file, data, sz);
            wolfSSL_BIO_free(dump_file);
        }
    }
#endif

    return sz;
}

static WC_INLINE int test_ssl_memio_read_cb(WOLFSSL *ssl, char *data, int sz,
    void *ctx)
{
    struct test_ssl_memio_ctx *test_ctx;
    int read_sz;
    byte *buf;
    int *len;

    test_ctx = (struct test_ssl_memio_ctx*)ctx;

    if (wolfSSL_GetSide(ssl) == WOLFSSL_SERVER_END) {
        buf = test_ctx->s_buff;
        len = &test_ctx->s_len;
    }
    else {
        buf = test_ctx->c_buff;
        len = &test_ctx->c_len;
    }

    if (*len == 0)
        return WOLFSSL_CBIO_ERR_WANT_READ;

    read_sz = sz < *len ? sz : *len;

    XMEMCPY(data, buf, read_sz);
    XMEMMOVE(buf, buf + read_sz, *len - read_sz);

    *len -= read_sz;

    return read_sz;
}

static WC_INLINE int test_ssl_memio_setup(test_ssl_memio_ctx *ctx)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    int c_sharedCtx = 0;
    int s_sharedCtx = 0;
#endif
    const char* certFile = svrCertFile;
    const char* keyFile = svrKeyFile;

    /********************************
     * Create WOLFSSL_CTX for client.
     ********************************/
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (ctx->c_ctx != NULL) {
        c_sharedCtx = ctx->c_cb.isSharedCtx;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (ctx->c_cb.method != NULL) {
            method = ctx->c_cb.method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ExpectNotNull(ctx->c_ctx = wolfSSL_CTX_new(method));
    }
    wolfSSL_SetIORecv(ctx->c_ctx, test_ssl_memio_read_cb);
    wolfSSL_SetIOSend(ctx->c_ctx, test_ssl_memio_write_cb);
#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx->c_ctx, PasswordCallBack);
#endif
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->c_ctx, caCertFile, 0),
        WOLFSSL_SUCCESS);
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!c_sharedCtx)
#endif
    {
        ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file(ctx->c_ctx,
            cliCertFile), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_file(ctx->c_ctx, cliKeyFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
#ifdef HAVE_CRL
    if (ctx->c_cb.crlPemFile != NULL) {
        ExpectIntEQ(wolfSSL_CTX_EnableCRL(ctx->c_ctx, WOLFSSL_CRL_CHECKALL),
            WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_LoadCRLFile(ctx->c_ctx, ctx->c_cb.crlPemFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
#endif
    if (ctx->c_ciphers != NULL) {
        ExpectIntEQ(wolfSSL_CTX_set_cipher_list(ctx->c_ctx, ctx->c_ciphers),
            WOLFSSL_SUCCESS);
    }
    if (ctx->c_cb.ctx_ready != NULL) {
        ExpectIntEQ(ctx->c_cb.ctx_ready(ctx->c_ctx), TEST_SUCCESS);
    }


    /********************************
     * Create WOLFSSL_CTX for server.
     ********************************/
    if (ctx->s_ctx != NULL) {
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
        s_sharedCtx = 1;
#endif
        ctx->s_cb.isSharedCtx = 1;
    }
    else
    {
        WOLFSSL_METHOD* method  = NULL;
        if (ctx->s_cb.method != NULL) {
            method = ctx->s_cb.method();
        }
        else {
            method = wolfSSLv23_server_method();
        }
        ExpectNotNull(ctx->s_ctx = wolfSSL_CTX_new(method));
        ctx->s_cb.isSharedCtx = 0;
    }
    if (!ctx->s_cb.ticNoInit && (ctx->s_ctx != NULL)) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketInit();
        wolfSSL_CTX_set_tlsext_ticket_key_cb(ctx->s_ctx, myTicketEncCbOpenSSL);
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketInit();
        wolfSSL_CTX_set_TicketEncCb(ctx->s_ctx, myTicketEncCb);
#endif
#endif
    }
    wolfSSL_SetIORecv(ctx->s_ctx, test_ssl_memio_read_cb);
    wolfSSL_SetIOSend(ctx->s_ctx, test_ssl_memio_write_cb);
    wolfSSL_CTX_set_verify(ctx->s_ctx, WOLFSSL_VERIFY_PEER |
        WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->s_ctx, cliCertFile, 0),
        WOLFSSL_SUCCESS);
#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx->s_ctx, PasswordCallBack);
#endif
    if (ctx->s_cb.certPemFile != NULL) {
        certFile = ctx->s_cb.certPemFile;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!s_sharedCtx)
#endif
    {
        ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file(ctx->s_ctx,
            certFile), WOLFSSL_SUCCESS);
    }
    if (ctx->s_cb.keyPemFile != NULL) {
        keyFile = ctx->s_cb.keyPemFile;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!s_sharedCtx)
#endif
    {
        ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_file(ctx->s_ctx, keyFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
    if (ctx->s_ciphers != NULL) {
        ExpectIntEQ(wolfSSL_CTX_set_cipher_list(ctx->s_ctx, ctx->s_ciphers),
            WOLFSSL_SUCCESS);
    }
    if (ctx->s_cb.ctx_ready != NULL) {
        ExpectIntEQ(ctx->s_cb.ctx_ready(ctx->s_ctx), TEST_SUCCESS);
    }


    /****************************
     * Create WOLFSSL for client.
     ****************************/
    ExpectNotNull(ctx->c_ssl = wolfSSL_new(ctx->c_ctx));
    wolfSSL_SetIOWriteCtx(ctx->c_ssl, ctx);
    wolfSSL_SetIOReadCtx(ctx->c_ssl, ctx);
    if (0
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
     || c_sharedCtx
#endif
        )
    {
        ExpectIntEQ(wolfSSL_use_certificate_chain_file(ctx->c_ssl, cliCertFile),
            WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_use_PrivateKey_file(ctx->c_ssl, cliKeyFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
    if (ctx->c_cb.ssl_ready != NULL) {
        ExpectIntEQ(ctx->c_cb.ssl_ready(ctx->c_ssl), TEST_SUCCESS);
    }

    /****************************
     * Create WOLFSSL for server.
     ****************************/
    ExpectNotNull(ctx->s_ssl = wolfSSL_new(ctx->s_ctx));
    wolfSSL_SetIOWriteCtx(ctx->s_ssl, ctx);
    wolfSSL_SetIOReadCtx(ctx->s_ssl, ctx);
    if (0
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
     || s_sharedCtx
#endif
        )
    {
        ExpectIntEQ(wolfSSL_use_certificate_chain_file(ctx->s_ssl, certFile),
            WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_use_PrivateKey_file(ctx->s_ssl, keyFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
    wolfSSL_SetTmpDH_file(ctx->s_ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
#elif !defined(NO_DH)
    /* will repick suites with DHE, higher priority than PSK */
    SetDH(ctx->s_ssl);
#endif
    if (ctx->s_cb.ssl_ready != NULL) {
        ExpectIntEQ(ctx->s_cb.ssl_ready(ctx->s_ssl), TEST_SUCCESS);
    }

    return EXPECT_RESULT();
}

static int test_ssl_memio_do_handshake(test_ssl_memio_ctx* ctx, int max_rounds,
    int* rounds)
{
    int handshake_complete = 0;
    int hs_c = 0;
    int hs_s = 0;
    int failing_s = 0;
    int failing_c = 0;
    int ret;
    int err;

    if (rounds != NULL) {
        *rounds = 0;
    }
    while ((!handshake_complete) && (max_rounds > 0)) {
        if (!hs_c) {
            wolfSSL_SetLoggingPrefix("client");
            ret = wolfSSL_connect(ctx->c_ssl);
            wolfSSL_SetLoggingPrefix(NULL);
            if (ret == WOLFSSL_SUCCESS) {
                hs_c = 1;
            }
            else {
                err = wolfSSL_get_error(ctx->c_ssl, ret);
                if (err != WOLFSSL_ERROR_WANT_READ &&
                    err != WOLFSSL_ERROR_WANT_WRITE) {
                    failing_c = 1;
                    hs_c = 1;
                    if (failing_c && failing_s) {
                        break;
                    }
                }
            }
        }
        if (!hs_s) {
            wolfSSL_SetLoggingPrefix("server");
            ret = wolfSSL_accept(ctx->s_ssl);
            wolfSSL_SetLoggingPrefix(NULL);
            if (ret == WOLFSSL_SUCCESS) {
                hs_s = 1;
            }
            else {
                err = wolfSSL_get_error(ctx->s_ssl, ret);
                if (err != WOLFSSL_ERROR_WANT_READ &&
                    err != WOLFSSL_ERROR_WANT_WRITE) {
                    failing_s = 1;
                    hs_s = 1;
                    if (failing_c && failing_s) {
                        break;
                    }
                }
            }
        }
        handshake_complete = hs_c && hs_s;
        max_rounds--;
        if (rounds != NULL) {
            *rounds += 1;
        }
    }

    if (!handshake_complete) {
        return TEST_FAIL;
    }

    return TEST_SUCCESS;
}

static int test_ssl_memio_read_write(test_ssl_memio_ctx* ctx)
{
    EXPECT_DECLS;
    char input[1024];
    int idx = 0;
    const char* msg_c = "hello wolfssl!";
    int msglen_c = (int)XSTRLEN(msg_c);
    const char* msg_s = "I hear you fa shizzle!";
    int msglen_s = (int)XSTRLEN(msg_s);

    if (ctx->c_msg != NULL) {
        msg_c = ctx->c_msg;
        msglen_c = ctx->c_msglen;
    }
    if (ctx->s_msg != NULL) {
        msg_s = ctx->s_msg;
        msglen_s = ctx->s_msglen;
    }

    wolfSSL_SetLoggingPrefix("client");
    ExpectIntEQ(wolfSSL_write(ctx->c_ssl, msg_c, msglen_c), msglen_c);
    wolfSSL_SetLoggingPrefix("server");
    ExpectIntGT(idx = wolfSSL_read(ctx->s_ssl, input, sizeof(input) - 1), 0);
    if (idx >= 0) {
        input[idx] = '\0';
    }
    ExpectIntGT(fprintf(stderr, "Client message: %s\n", input), 0);
    ExpectIntEQ(wolfSSL_write(ctx->s_ssl, msg_s, msglen_s), msglen_s);
    ctx->s_cb.return_code = EXPECT_RESULT();
    wolfSSL_SetLoggingPrefix("client");
    ExpectIntGT(idx = wolfSSL_read(ctx->c_ssl, input, sizeof(input) - 1), 0);
    wolfSSL_SetLoggingPrefix(NULL);
    if (idx >= 0) {
        input[idx] = '\0';
    }
    ExpectIntGT(fprintf(stderr, "Server response: %s\n", input), 0);
    ctx->c_cb.return_code = EXPECT_RESULT();
    if (ctx->c_cb.on_result != NULL) {
        ExpectIntEQ(ctx->c_cb.on_result(ctx->c_ssl), TEST_SUCCESS);
    }
    if (ctx->s_cb.on_result != NULL) {
        ExpectIntEQ(ctx->s_cb.on_result(ctx->s_ssl), TEST_SUCCESS);
    }

    return EXPECT_RESULT();
}

static void test_ssl_memio_cleanup(test_ssl_memio_ctx* ctx)
{
    ctx->c_cb.last_err = wolfSSL_get_error(ctx->c_ssl, 0);
    ctx->s_cb.last_err = wolfSSL_get_error(ctx->s_ssl, 0);
    if (ctx->c_cb.on_cleanup != NULL) {
        ctx->c_cb.on_cleanup(ctx->c_ssl);
    }
    if (ctx->s_cb.on_cleanup != NULL) {
        ctx->s_cb.on_cleanup(ctx->s_ssl);
    }
    wolfSSL_shutdown(ctx->s_ssl);
    wolfSSL_shutdown(ctx->c_ssl);
    wolfSSL_free(ctx->s_ssl);
    wolfSSL_free(ctx->c_ssl);
    if (!ctx->s_cb.isSharedCtx) {
        wolfSSL_CTX_free(ctx->s_ctx);
        ctx->s_ctx = NULL;
    }
    if (!ctx->c_cb.isSharedCtx) {
        wolfSSL_CTX_free(ctx->c_ctx);
        ctx->c_ctx = NULL;
    }

    if (!ctx->s_cb.ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketCleanup();
#endif
#endif
    }
}

int test_wolfSSL_client_server_nofail_memio(test_ssl_cbf* client_cb,
    test_ssl_cbf* server_cb, cbType client_on_handshake)
{
    EXPECT_DECLS;
    struct test_ssl_memio_ctx test_ctx;
#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    size_t msg_len;
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    XMEMCPY(&test_ctx.c_cb, client_cb, sizeof(test_ssl_cbf));
    XMEMCPY(&test_ctx.s_cb, server_cb, sizeof(test_ssl_cbf));

    test_ctx.c_ctx = client_cb->ctx;
    test_ctx.s_ctx = server_cb->ctx;
    test_ctx.c_cb.return_code = TEST_FAIL;
    test_ctx.s_cb.return_code = TEST_FAIL;

    ExpectIntEQ(test_ssl_memio_setup(&test_ctx), TEST_SUCCESS);
    ExpectIntEQ(test_ssl_memio_do_handshake(&test_ctx, 10, NULL), TEST_SUCCESS);

    if (client_on_handshake != NULL) {
        ExpectIntEQ(client_on_handshake(test_ctx.c_ctx, test_ctx.c_ssl),
            TEST_SUCCESS);
    }
#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    XMEMSET(server_side_msg2, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_peer_finished(test_ctx.s_ssl, server_side_msg2,
        MD_MAX_SIZE);
    ExpectIntGE(msg_len, 0);

    XMEMSET(server_side_msg1, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_finished(test_ctx.s_ssl, server_side_msg1,
        MD_MAX_SIZE);
    ExpectIntGE(msg_len, 0);
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    ExpectIntEQ(test_ssl_memio_read_write(&test_ctx), TEST_SUCCESS);
    test_ssl_memio_cleanup(&test_ctx);

    client_cb->return_code = test_ctx.c_cb.return_code;
    client_cb->last_err = test_ctx.c_cb.last_err;
    server_cb->return_code = test_ctx.s_cb.return_code;
    server_cb->last_err = test_ctx.s_cb.last_err;

    return EXPECT_RESULT();
}
#endif

#ifdef HAVE_IO_TESTS_DEPENDENCIES

#ifdef WOLFSSL_SESSION_EXPORT
#ifdef WOLFSSL_DTLS
/* set up function for sending session information */
static int test_export(WOLFSSL* inSsl, byte* buf, word32 sz, void* userCtx)
{
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;

    AssertNotNull(inSsl);
    AssertNotNull(buf);
    AssertIntNE(0, sz);

    /* Set ctx to DTLS 1.2 */
    ctx = wolfSSL_CTX_new(wolfDTLSv1_2_server_method());
    AssertNotNull(ctx);

    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);

    AssertIntGE(wolfSSL_dtls_import(ssl, buf, sz), 0);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    (void)userCtx;
    return 0;
}
#endif

/* returns negative value on fail and positive (including 0) on success */
static int nonblocking_accept_read(void* args, WOLFSSL* ssl, SOCKET_T* sockfd)
{
    int ret, err, loop_count, count, timeout = 10;
    char msg[] = "I hear you fa shizzle!";
    char input[1024];

    loop_count = ((func_args*)args)->argc;

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_accept(ssl);
        err = wolfSSL_get_error(ssl, 0);

        if (err == WOLFSSL_ERROR_WANT_READ ||
            err == WOLFSSL_ERROR_WANT_WRITE) {
            int select_ret;

            err = WC_PENDING_E;
            select_ret = tcp_select(*sockfd, timeout);
            if (select_ret == TEST_TIMEOUT) {
                return WOLFSSL_FATAL_ERROR;
            }
        }
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        return ret;
    }

    for (count = 0; count < loop_count; count++) {
        int select_ret;

        select_ret = tcp_select(*sockfd, timeout);
        if (select_ret == TEST_TIMEOUT) {
            ret = WOLFSSL_FATAL_ERROR;
            break;
        }

        do {
            ret = wolfSSL_read(ssl, input, sizeof(input)-1);
            if (ret > 0) {
                input[ret] = '\0';
                fprintf(stderr, "Client message: %s\n", input);
            }
        } while (err == WOLFSSL_ERROR_WANT_READ && ret != WOLFSSL_SUCCESS);

        do {
            if ((ret = wolfSSL_write(ssl, msg, sizeof(msg))) != sizeof(msg)) {
                return WOLFSSL_FATAL_ERROR;
            }
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WOLFSSL_ERROR_WANT_READ && ret != WOLFSSL_SUCCESS);
    }
    return ret;
}
#endif /* WOLFSSL_SESSION_EXPORT */

static THREAD_RETURN WOLFSSL_THREAD test_server_nofail(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    func_args* opts = (func_args*)args;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;
    int ret, err = 0;
    int sharedCtx = 0;
    int doUdp = 0;
    SOCKADDR_IN_T cliAddr;
    socklen_t     cliLen;
    const char* certFile = svrCertFile;
    const char* keyFile = svrKeyFile;

#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    size_t msg_len = 0;
#endif

    wolfSSL_SetLoggingPrefix("server");

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    opts->return_code = TEST_FAIL;
    cbf = opts->callbacks;

    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();

        }
        else {
            method = wolfSSLv23_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }
    if (ctx == NULL) {
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }

    if (cbf == NULL || !cbf->ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketInit();
        wolfSSL_CTX_set_tlsext_ticket_key_cb(ctx, myTicketEncCbOpenSSL);
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketInit();
        wolfSSL_CTX_set_TicketEncCb(ctx, myTicketEncCb);
#endif
#endif
    }

#if defined(USE_WINDOWS_API)
    port = opts->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    if (cbf != NULL)
        doUdp = cbf->doUdp;

    /* do it here to detect failure */
    tcp_accept(
        &sockfd, &clientfd, opts, port, 0, doUdp, 0, 0, 1, 0, 0);

    if (doUdp) {
        cliLen = sizeof(cliAddr);

        idx = (int)recvfrom(sockfd, input, sizeof(input), MSG_PEEK,
                  (struct sockaddr*)&cliAddr, &cliLen);

        AssertIntGT(idx, 0);
    }
    else {
        CloseSocket(sockfd);
    }

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0)
                                                           != WOLFSSL_SUCCESS) {
        /*err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }

    if (cbf != NULL && cbf->certPemFile != NULL)
        certFile = cbf->certPemFile;
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_certificate_file(ctx, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (cbf != NULL && cbf->keyPemFile != NULL)
        keyFile = cbf->keyPemFile;
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef HAVE_CRL
    if (cbf != NULL && cbf->crlPemFile != NULL) {
        if (wolfSSL_CTX_EnableCRL(ctx, WOLFSSL_CRL_CHECKALL) != WOLFSSL_SUCCESS)
            goto done;
        if (wolfSSL_CTX_LoadCRLFile(ctx, cbf->crlPemFile, WOLFSSL_FILETYPE_PEM)
                != WOLFSSL_SUCCESS)
            goto done;
    }
#endif

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }

    if (doUdp) {
        err = wolfSSL_dtls_set_peer(ssl, &cliAddr, cliLen);
        if (err != WOLFSSL_SUCCESS)
            goto done;
    }

#ifdef WOLFSSL_SESSION_EXPORT
    /* only add in more complex nonblocking case with session export tests */
    if (args && opts->argc > 0) {
        /* set as nonblock and time out for waiting on read/write */
        tcp_set_nonblocking(&clientfd);
        wolfSSL_dtls_set_using_nonblock(ssl, 1);
    }
#endif
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_certificate_file(ssl, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_PrivateKey_file(ssl, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (wolfSSL_set_fd(ssl, clientfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
    wolfSSL_SetTmpDH_file(ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
#elif !defined(NO_DH)
    SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
#endif

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

#ifdef WOLFSSL_SESSION_EXPORT
    /* only add in more complex nonblocking case with session export tests */
    if (opts->argc > 0) {
        ret = nonblocking_accept_read(args, ssl, &clientfd);
        if (ret >= 0) {
            opts->return_code = TEST_SUCCESS;
        }
    #ifdef WOLFSSL_TIRTOS
        Task_yield();
    #endif
        goto done;
    }
#endif

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_negotiate(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_accept failed");*/
        goto done;
    }

#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    XMEMSET(server_side_msg2, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_peer_finished(ssl, server_side_msg2, MD_MAX_SIZE);
    AssertIntGE(msg_len, 0);

    XMEMSET(server_side_msg1, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_finished(ssl, server_side_msg1, MD_MAX_SIZE);
    AssertIntGE(msg_len, 0);
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    idx = wolfSSL_read(ssl, input, sizeof(input)-1);
    if (idx > 0) {
        input[idx] = '\0';
        fprintf(stderr, "Client message: %s\n", input);
    }
    else if (idx < 0) {
        goto done;
    }

    if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    if (cbf != NULL && cbf->on_result != NULL)
        cbf->on_result(ssl);

#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    opts->return_code = TEST_SUCCESS;

done:
    if (cbf != NULL)
        cbf->last_err = err;
    if (cbf != NULL && cbf->on_cleanup != NULL)
        cbf->on_cleanup(ssl);

    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    if (cbf == NULL || !cbf->ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketCleanup();
#endif
#endif
    }

    wolfSSL_SetLoggingPrefix(NULL);

    WOLFSSL_RETURN_FROM_THREAD(0);
}

#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
    !defined(WOLFSSL_NO_TLS12)
static THREAD_RETURN WOLFSSL_THREAD test_server_loop(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = 0;
    WOLFSSL* ssl = 0;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;
    int ret, err = 0;
    int sharedCtx = 0;
    func_args* opts = (func_args*)args;
    int loop_count = opts->argc;
    int count = 0;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    opts->return_code = TEST_FAIL;
    cbf = opts->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

#if defined(USE_WINDOWS_API)
    port = opts->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0)
                                                           != WOLFSSL_SUCCESS) {
        /*err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }
    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    while (count != loop_count) {
        ssl = wolfSSL_new(ctx);
        if (ssl == NULL) {
            signal_ready(opts->signal);
            goto done;
        }
        if (sharedCtx && wolfSSL_use_certificate_file(ssl, svrCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            /*err_sys("can't load server cert chain file, "
                    "Please run from wolfSSL home dir");*/
            /* Release the wait for TCP ready. */
            signal_ready(opts->signal);
            goto done;
        }
        if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, svrKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            /*err_sys("can't load server key file, "
                    "Please run from wolfSSL home dir");*/
            /* Release the wait for TCP ready. */
            signal_ready(opts->signal);
            goto done;
        }

#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
        wolfSSL_SetTmpDH_file(ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
#elif !defined(NO_DH)
        SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
#endif
        /* call ssl setup callback */
        if (cbf != NULL && cbf->ssl_ready != NULL) {
            cbf->ssl_ready(ssl);
        }
        /* do it here to detect failure */
        tcp_accept(&sockfd, &clientfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0,
            0);
        CloseSocket(sockfd);
        if (wolfSSL_set_fd(ssl, clientfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }

        #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
        #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_accept(ssl);
            err = wolfSSL_get_error(ssl, 0);
        } while (err == WC_PENDING_E);
        if (ret != WOLFSSL_SUCCESS) {
            char buff[WOLFSSL_MAX_ERROR_SZ];
            fprintf(stderr, "error = %d, %s\n", err,
                wolfSSL_ERR_error_string(err, buff));
            /*err_sys("SSL_accept failed");*/
            goto done;
        }

        idx = wolfSSL_read(ssl, input, sizeof(input)-1);
        if (idx > 0) {
            input[idx] = '\0';
            fprintf(stderr, "Client message: %s\n", input);
        }

        if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
            /*err_sys("SSL_write failed");*/
            goto done;
        }
        /* free ssl for this connection */
        wolfSSL_shutdown(ssl);
        wolfSSL_free(ssl); ssl = NULL;
        CloseSocket(clientfd);

        count++;
    }
#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    opts->return_code = TEST_SUCCESS;

done:
    if (ssl != NULL) {
        wolfSSL_shutdown(ssl);
        wolfSSL_free(ssl);
    }
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && !defined(WOLFSSL_TLS13) */

static int test_client_nofail(void* args, cbType cb)
{
#if !defined(NO_WOLFSSL_CLIENT)
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    WOLFSSL_CIPHER*  cipher;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  input;
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    int  cipherSuite;
    int  sharedCtx = 0;
    int  doUdp = 0;
    const char* cipherName1, *cipherName2;

    wolfSSL_SetLoggingPrefix("client");

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = cbf->isSharedCtx;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

    if (cbf != NULL)
        doUdp = cbf->doUdp;

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                doUdp, 0, NULL);
    /* Connect the socket so that we don't have to set the peer later on */
    if (doUdp)
        udp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0) != WOLFSSL_SUCCESS)
    {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif

        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef HAVE_CRL
    if (cbf != NULL && cbf->crlPemFile != NULL) {
        if (wolfSSL_CTX_EnableCRL(ctx, WOLFSSL_CRL_CHECKALL) != WOLFSSL_SUCCESS)
            goto done;
        if (wolfSSL_CTX_LoadCRLFile(ctx, cbf->crlPemFile, WOLFSSL_FILETYPE_PEM)
                != WOLFSSL_SUCCESS)
            goto done;
    }
#endif

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (!doUdp) {
        if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }
    }
    else {
#ifdef WOLFSSL_DTLS
        if (wolfSSL_set_dtls_fd_connected(ssl, sockfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }
#else
        goto done;
#endif
    }

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_negotiate(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }

    /* test the various get cipher methods */
    /* Internal cipher suite names */
    cipherSuite = wolfSSL_get_current_cipher_suite(ssl);
    cipherName1 = wolfSSL_get_cipher_name(ssl);
    cipherName2 = wolfSSL_get_cipher_name_from_suite(
        (cipherSuite >> 8), cipherSuite & 0xFF);
    AssertStrEQ(cipherName1, cipherName2);

    /* IANA Cipher Suites Names */
    /* Unless WOLFSSL_CIPHER_INTERNALNAME or NO_ERROR_STRINGS,
        then it's the internal cipher suite name */
    cipher = wolfSSL_get_current_cipher(ssl);
    cipherName1 = wolfSSL_CIPHER_get_name(cipher);
    cipherName2 = wolfSSL_get_cipher(ssl);
    AssertStrEQ(cipherName1, cipherName2);
#if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS) && \
    !defined(WOLFSSL_QT)
    cipherName1 = wolfSSL_get_cipher_name_iana_from_suite(
            (cipherSuite >> 8), cipherSuite & 0xFF);
    AssertStrEQ(cipherName1, cipherName2);
#endif

    if (cb != NULL)
        (cb)(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    if (cbf != NULL && cbf->on_result != NULL)
        cbf->on_result(ssl);

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    if (cbf != NULL)
        cbf->last_err = err;
    if (cbf != NULL && cbf->on_cleanup != NULL)
        cbf->on_cleanup(ssl);

    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#else
    (void)args;
    (void)cb;
#endif /* !NO_WOLFSSL_CLIENT */

    wolfSSL_SetLoggingPrefix(NULL);

    return 0;
}

void test_wolfSSL_client_server_nofail_ex(callback_functions* client_cb,
    callback_functions* server_cb, cbType client_on_handshake)
{
    func_args client_args;
    func_args server_args;
    tcp_ready ready;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    server_args.callbacks = server_cb;
    client_args.signal = &ready;
    client_args.callbacks = client_cb;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, client_on_handshake);
    join_thread(serverThread);

    client_cb->return_code = client_args.return_code;
    server_cb->return_code = server_args.return_code;

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
}

void test_wolfSSL_client_server_nofail(callback_functions* client_cb,
    callback_functions* server_cb)
{
    test_wolfSSL_client_server_nofail_ex(client_cb, server_cb, NULL);
}


#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
   !defined(WOLFSSL_NO_TLS12) && !defined(NO_WOLFSSL_CLIENT)
static void test_client_reuse_WOLFSSLobj(void* args, cbType cb,
                                         void* server_args)
{
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    WOLFSSL_SESSION* session = NULL;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  input;
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    int  sharedCtx = 0;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0) !=
                                                              WOLFSSL_SUCCESS) {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
    /* keep handshake resources for re-using WOLFSSL obj */
    wolfSSL_KeepArrays(ssl);
    if (wolfSSL_KeepHandshakeResources(ssl)) {
        /* err_sys("SSL_KeepHandshakeResources failed"); */
        goto done;
    }
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }
    /* Build first session */
    if (cb != NULL)
        cb(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    /* Session Resumption by re-using WOLFSSL object */
    wolfSSL_set_quiet_shutdown(ssl, 1);
    if (wolfSSL_shutdown(ssl) != WOLFSSL_SUCCESS) {
        /* err_sys ("SSL shutdown failed"); */
        goto done;
    }
    session = wolfSSL_get1_session(ssl);
    if (wolfSSL_clear(ssl) != WOLFSSL_SUCCESS) {
        wolfSSL_SESSION_free(session);
        /* err_sys ("SSL_clear failed"); */
        goto done;
    }
    wolfSSL_set_session(ssl, session);
    wolfSSL_SESSION_free(session);
    session = NULL;
    /* close socket once */
    CloseSocket(sockfd);
    sockfd = 0;
    /* wait until server ready */
    wait_tcp_ready((func_args*)server_args);
    fprintf(stderr, "session resumption\n");
    /* Do re-connect  */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);
    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }
    /* Build first session */
    if (cb != NULL)
        cb(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    return;
}
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) &&
         !defined(WOLFSSL_TLS13) && !defined(NO_WOLFSSL_CLIENT) */

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
     defined(WOLFSSL_HAPROXY) || defined(HAVE_LIGHTY)) && \
    defined(HAVE_ALPN) && defined(HAVE_SNI) && \
    defined(HAVE_IO_TESTS_DEPENDENCIES) && !defined(NO_BIO)
    #define HAVE_ALPN_PROTOS_SUPPORT
#endif

/* Generic TLS client / server with callbacks for API unit tests
 * Used by SNI / ALPN / crypto callback helper functions */
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && \
    (defined(HAVE_SNI) || defined(HAVE_ALPN) || defined(WOLF_CRYPTO_CB) || \
     defined(HAVE_ALPN_PROTOS_SUPPORT)) || defined(WOLFSSL_STATIC_MEMORY)
    #define ENABLE_TLS_CALLBACK_TEST
#endif

#if defined(ENABLE_TLS_CALLBACK_TEST) || \
    (defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT))
/* TLS server for API unit testing - generic */
static THREAD_RETURN WOLFSSL_THREAD run_wolfssl_server(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;

    WOLFSSL_CTX*    ctx = NULL;
    WOLFSSL*        ssl = NULL;
    SOCKET_T        sfd = 0;
    SOCKET_T        cfd = 0;
    word16          port;

    char msg[] = "I hear you fa shizzle!";
    int  len   = (int) XSTRLEN(msg);
    char input[1024];
    int  idx;
    int  ret, err = 0;

    ((func_args*)args)->return_code = TEST_FAIL;

#if defined(USE_WINDOWS_API)
    port = ((func_args*)args)->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

#ifdef WOLFSSL_DTLS
    if (callbacks->method == wolfDTLS_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLS_server_method_ex
#endif
#ifndef NO_OLD_TLS
     || callbacks->method == wolfDTLSv1_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLSv1_server_method_ex
#endif
#endif
#ifndef WOLFSSL_NO_TLS12
     || callbacks->method == wolfDTLSv1_2_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLSv1_2_server_method_ex
#endif
#endif
#ifdef WOLFSSL_DTLS13
     || callbacks->method == wolfDTLSv1_3_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLSv1_3_server_method_ex
#endif
#endif
        ) {
        tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 1, 0, 0, 0, 0, 0);
    }
    else
#endif
    {
        tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0, 0);
    }

#ifdef WOLFSSL_STATIC_MEMORY
    if (callbacks->method_ex != NULL && callbacks->mem != NULL &&
        callbacks->memSz > 0) {
        ret = wolfSSL_CTX_load_static_memory(&ctx, callbacks->method_ex,
            callbacks->mem, callbacks->memSz, 0, 1);
        if (ret != WOLFSSL_SUCCESS) {
            fprintf(stderr, "CTX static new failed %d\n", ret);
            goto cleanup;
        }
    }
#else
    ctx = wolfSSL_CTX_new(callbacks->method());
#endif
    if (ctx == NULL) {
        fprintf(stderr, "CTX new failed\n");
        goto cleanup;
    }

    /* set defaults */
    if (callbacks->caPemFile == NULL)
        callbacks->caPemFile = cliCertFile;
    if (callbacks->certPemFile == NULL)
        callbacks->certPemFile = svrCertFile;
    if (callbacks->keyPemFile == NULL)
        callbacks->keyPemFile = svrKeyFile;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    wolfSSL_CTX_SetDevId(ctx, callbacks->devId);

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif
#if defined(WOLFSSL_SESSION_EXPORT) && defined(WOLFSSL_DTLS)
    if (callbacks->method == wolfDTLSv1_2_server_method) {
        if (wolfSSL_CTX_dtls_set_export(ctx, test_export) != WOLFSSL_SUCCESS)
            goto cleanup;
    }
#endif


    if (wolfSSL_CTX_load_verify_locations(ctx, callbacks->caPemFile, 0) !=
            WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (wolfSSL_CTX_use_certificate_file(ctx, callbacks->certPemFile,
            WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (wolfSSL_CTX_use_PrivateKey_file(ctx, callbacks->keyPemFile,
            WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

#ifdef HAVE_CRL
    if (callbacks->crlPemFile != NULL) {
        if (wolfSSL_CTX_LoadCRLFile(ctx, callbacks->crlPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }
#endif

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        fprintf(stderr, "SSL new failed\n");
        goto cleanup;
    }
    if (wolfSSL_dtls(ssl)) {
        SOCKADDR_IN_T cliAddr;
        socklen_t     cliLen;

        cliLen = sizeof(cliAddr);
        idx = (int)recvfrom(sfd, input, sizeof(input), MSG_PEEK,
                (struct sockaddr*)&cliAddr, &cliLen);
        if (idx <= 0) {
            goto cleanup;
        }
        wolfSSL_dtls_set_peer(ssl, &cliAddr, cliLen);
    }
    else {
        CloseSocket(sfd);
    }

    if (wolfSSL_set_fd(ssl, cfd) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (callbacks->loadToSSL) {
        wolfSSL_SetDevId(ssl, callbacks->devId);

        if (wolfSSL_use_certificate_file(ssl, callbacks->certPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }

        if (wolfSSL_use_PrivateKey_file(ssl, callbacks->keyPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }

#ifdef NO_PSK
    #if !defined(NO_FILESYSTEM) && !defined(NO_DH)
        wolfSSL_SetTmpDH_file(ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
    #elif !defined(NO_DH)
        SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
    #endif
#endif

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

#ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
#endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_accept(ssl);
        err = wolfSSL_get_error(ssl, ret);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "accept error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_accept failed");*/
    }
    else {
    #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
    #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            idx = wolfSSL_read(ssl, input, sizeof(input)-1);
            err = wolfSSL_get_error(ssl, idx);
        } while (err == WC_PENDING_E);
        if (idx > 0) {
            input[idx] = 0;
            fprintf(stderr, "Client message: %s\n", input);
        }

    #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
    #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_write(ssl, msg, len);
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WC_PENDING_E);
        if (len != ret) {
            goto cleanup;
        }

#if defined(WOLFSSL_SESSION_EXPORT) && !defined(HAVE_IO_POOL) && \
        defined(WOLFSSL_DTLS)
        if (wolfSSL_dtls(ssl)) {
            byte*  import;
            word32 sz;

            wolfSSL_dtls_export(ssl, NULL, &sz);
            import = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            if (import == NULL) {
                goto cleanup;
            }
            idx = wolfSSL_dtls_export(ssl, import, &sz);
            if (idx < 0) {
                goto cleanup;
            }
            if (wolfSSL_dtls_import(ssl, import, idx) < 0) {
                goto cleanup;
            }
            XFREE(import, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        }
#endif
#ifdef WOLFSSL_TIRTOS
        Task_yield();
#endif
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

    wolfSSL_shutdown(ssl);

cleanup:
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(cfd);


#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}

/* TLS Client for API unit testing - generic */
static void run_wolfssl_client(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;

    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;
    SOCKET_T    sfd = 0;

    char msg[] = "hello wolfssl server!";
    int  len   = (int) XSTRLEN(msg);
    char input[1024];
    int  ret, err = 0;

    ((func_args*)args)->return_code = TEST_FAIL;

    /* set defaults */
    if (callbacks->caPemFile == NULL)
        callbacks->caPemFile = caCertFile;
    if (callbacks->certPemFile == NULL)
        callbacks->certPemFile = cliCertFile;
    if (callbacks->keyPemFile == NULL)
        callbacks->keyPemFile = cliKeyFile;

#ifdef WOLFSSL_STATIC_MEMORY
    if (callbacks->method_ex != NULL && callbacks->mem != NULL &&
        callbacks->memSz > 0) {
        ret = wolfSSL_CTX_load_static_memory(&ctx, callbacks->method_ex,
            callbacks->mem, callbacks->memSz, 0, 1);
        if (ret != WOLFSSL_SUCCESS) {
            fprintf(stderr, "CTX static new failed %d\n", ret);
            goto cleanup;
        }
    }
#else
    ctx = wolfSSL_CTX_new(callbacks->method());
#endif
    if (ctx == NULL) {
        fprintf(stderr, "CTX new failed\n");
        goto cleanup;
    }

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    if (!callbacks->loadToSSL) {
        wolfSSL_CTX_SetDevId(ctx, callbacks->devId);
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, callbacks->caPemFile, 0) !=
            WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (!callbacks->loadToSSL) {
        if (wolfSSL_CTX_use_certificate_file(ctx, callbacks->certPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }

        if (wolfSSL_CTX_use_PrivateKey_file(ctx, callbacks->keyPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }

#ifdef HAVE_CRL
    if (callbacks->crlPemFile != NULL) {
        if (wolfSSL_CTX_LoadCRLFile(ctx, callbacks->crlPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }
#endif

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);
    if (wolfSSL_dtls(ssl)) {
        tcp_connect(&sfd, wolfSSLIP, ((func_args*)args)->signal->port,
                    1, 0, ssl);
    }
    else {
        tcp_connect(&sfd, wolfSSLIP, ((func_args*)args)->signal->port,
                    0, 0, ssl);
    }
    if (wolfSSL_set_fd(ssl, sfd) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (callbacks->loadToSSL) {
        wolfSSL_SetDevId(ssl, callbacks->devId);

        if (wolfSSL_use_certificate_file(ssl, callbacks->certPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }

        if (wolfSSL_use_PrivateKey_file(ssl, callbacks->keyPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, ret);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
    }
    else {
        #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
        #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_write(ssl, msg, len);
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WC_PENDING_E);
        if (len != ret)
            goto cleanup;

        #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
        #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_read(ssl, input, sizeof(input)-1);
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WC_PENDING_E);
        if (ret > 0) {
            input[ret] = '\0'; /* null term */
            fprintf(stderr, "Server response: %s\n", input);
        }
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

cleanup:
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(sfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif
}

#endif /* ENABLE_TLS_CALLBACK_TEST */


static int test_wolfSSL_read_write(void)
{
    /* The unit testing for read and write shall happen simultaneously, since
     * one can't do anything with one without the other. (Except for a failure
     * test case.) This function will call all the others that will set up,
     * execute, and report their test findings.
     *
     * Set up the success case first. This function will become the template
     * for the other tests. This should eventually be renamed
     *
     * The success case isn't interesting, how can this fail?
     * - Do not give the client context a CA certificate. The connect should
     *   fail. Do not need server for this?
     * - Using NULL for the ssl object on server. Do not need client for this.
     * - Using NULL for the ssl object on client. Do not need server for this.
     * - Good ssl objects for client and server. Client write() without server
     *   read().
     * - Good ssl objects for client and server. Server write() without client
     *   read().
     * - Forgetting the password callback?
    */
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    EXPECT_DECLS;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, NULL);
    join_thread(serverThread);

    ExpectTrue(client_args.return_code);
    ExpectTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_reuse_WOLFSSLobj(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
    !defined(WOLFSSL_NO_TLS12)
    /* The unit test for session resumption by re-using WOLFSSL object.
     * WOLFSSL object is not cleared after first session. It reuse the object
     * for second connection.
    */
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions client_cbf;
    callback_functions server_cbf;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&client_cbf, 0, sizeof(callback_functions));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    client_cbf.method = wolfTLSv1_2_client_method;
    server_cbf.method = wolfTLSv1_2_server_method;
    client_args.callbacks = &client_cbf;
    server_args.callbacks = &server_cbf;

    server_args.signal = &ready;
    client_args.signal = &ready;
    /* the var is used for loop number */
    server_args.argc = 2;

    start_thread(test_server_loop, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_reuse_WOLFSSLobj(&client_args, NULL, &server_args);
    join_thread(serverThread);

    ExpectTrue(client_args.return_code);
    ExpectTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) &&
        * !defined(WOLFSSL_TLS13) */
    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_TIRTOS) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_verifyDepth_ServerClient_1_ctx_ready(
    WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);
    myVerifyAction = VERIFY_USE_PREVERFIY;
    wolfSSL_CTX_set_verify_depth(ctx, 2);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_CTX_verifyDepth_ServerClient_1(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_TIRTOS) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */
    client_cbf.ctx_ready =
        test_wolfSSL_CTX_verifyDepth_ServerClient_1_ctx_ready;

    /* test case 1 verify depth is equal to peer chain */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    ExpectIntEQ(client_cbf.return_code, TEST_SUCCESS);
    ExpectIntEQ(server_cbf.return_code, TEST_SUCCESS);
#endif /* OPENSSL_EXTRA && !WOLFSSL_TIRTOS &&
        * HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_TIRTOS) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_verifyDepth_ServerClient_2_ctx_ready(
    WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);
    myVerifyAction = VERIFY_OVERRIDE_ERROR;
    wolfSSL_CTX_set_verify_depth(ctx, 0);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_CTX_verifyDepth_ServerClient_2(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_TIRTOS) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */
    client_cbf.ctx_ready =
        test_wolfSSL_CTX_verifyDepth_ServerClient_2_ctx_ready;

    /* test case 2
     * verify depth is zero, number of peer's chain is 2.
     * verify result becomes MAX_CHAIN_ERROR, but it is overridden in
     * callback.
     */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    ExpectIntEQ(client_cbf.return_code, TEST_SUCCESS);
    ExpectIntEQ(server_cbf.return_code, TEST_SUCCESS);
#endif /* OPENSSL_EXTRA && !WOLFSSL_TIRTOS &&
        * HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_TIRTOS) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_verifyDepth_ServerClient_3_ctx_ready(
    WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);
    myVerifyAction = VERIFY_USE_PREVERFIY;
    wolfSSL_CTX_set_verify_depth(ctx, 0);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_CTX_verifyDepth_ServerClient_3(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_TIRTOS) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */
    client_cbf.ctx_ready =
        test_wolfSSL_CTX_verifyDepth_ServerClient_3_ctx_ready;

    /* test case 3
     * verify depth is zero, number of peer's chain is 2
     * verify result becomes MAX_CHAIN_ERRO. call-back returns failure.
     * therefore, handshake becomes failure.
     */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_FAIL);

    ExpectIntEQ(client_cbf.return_code, TEST_FAIL);
    ExpectIntEQ(server_cbf.return_code, TEST_FAIL);
    ExpectIntEQ(client_cbf.last_err, MAX_CHAIN_ERROR);
    ExpectIntEQ(server_cbf.last_err, FATAL_ERROR);
#endif /* OPENSSL_EXTRA && !WOLFSSL_TIRTOS &&
        * HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_ALL) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    !defined(WOLFSSL_TIRTOS) && !defined(NO_AES) && !defined(WOLFSSL_NO_TLS12) \
    && !defined(NO_SHA256) && defined(HAVE_ECC)
static int test_wolfSSL_CTX_set_cipher_list_server_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "DEFAULT:!NULL"));
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_cipher_list_client_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256"));
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_CTX_set_cipher_list(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    !defined(WOLFSSL_TIRTOS) && !defined(NO_AES) && !defined(WOLFSSL_NO_TLS12) \
    && !defined(NO_SHA256) && defined(HAVE_ECC)
    WOLFSSL_CTX* ctxClient = NULL;
    WOLFSSL*     sslClient = NULL;
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

    server_cbf.method = wolfTLSv1_2_server_method;
    server_cbf.ctx_ready = test_wolfSSL_CTX_set_cipher_list_server_ctx_ready;
    client_cbf.method = wolfTLSv1_2_client_method;
    client_cbf.ctx_ready = test_wolfSSL_CTX_set_cipher_list_client_ctx_ready;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    ExpectIntEQ(client_cbf.return_code, TEST_SUCCESS);
    ExpectIntEQ(server_cbf.return_code, TEST_SUCCESS);

    /* check with cipher string that has '+' */
    ExpectNotNull((ctxClient = wolfSSL_CTX_new(wolfTLSv1_2_client_method())));
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctxClient, "ECDHE+AESGCM"));
    ExpectNotNull((sslClient = wolfSSL_new(ctxClient)));

    /* check for the existence of an ECDHE ECDSA cipher suite */
    if (EXPECT_SUCCESS()) {
        int i = 0;
        int found = 0;
        const char* suite;

        WOLF_STACK_OF(WOLFSSL_CIPHER)* sk = NULL;
        WOLFSSL_CIPHER* current;

        ExpectNotNull((sk = wolfSSL_get_ciphers_compat(sslClient)));
        do {
            current = wolfSSL_sk_SSL_CIPHER_value(sk, i++);
            if (current) {
                suite = wolfSSL_CIPHER_get_name(current);
                if (suite && XSTRSTR(suite, "ECDSA")) {
                    found = 1;
                    break;
                }
            }
        } while (current);
        ExpectIntEQ(found, 1);
    }

    wolfSSL_free(sslClient);
    wolfSSL_CTX_free(ctxClient);
#endif
    return EXPECT_RESULT();
}

#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    defined(WOLFSSL_HAVE_TLS_UNIQUE)
static int test_wolfSSL_get_finished_client_on_handshake(WOLFSSL_CTX* ctx,
    WOLFSSL* ssl)
{
    EXPECT_DECLS;
    size_t msg_len;

    (void)ctx;

    /* get_finished test */
    /* 1. get own sent message */
    XMEMSET(client_side_msg1, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_finished(ssl, client_side_msg1, MD_MAX_SIZE);
    ExpectIntGE(msg_len, 0);
    /* 2. get peer message */
    XMEMSET(client_side_msg2, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_peer_finished(ssl, client_side_msg2, MD_MAX_SIZE);
    ExpectIntGE(msg_len, 0);

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_get_finished(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    defined(WOLFSSL_HAVE_TLS_UNIQUE)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, test_wolfSSL_get_finished_client_on_handshake),
        TEST_SUCCESS);

    /* test received msg vs sent msg */
    ExpectIntEQ(0, XMEMCMP(client_side_msg1, server_side_msg2, MD_MAX_SIZE));
    ExpectIntEQ(0, XMEMCMP(client_side_msg2, server_side_msg1, MD_MAX_SIZE));
#endif /* HAVE_SSL_MEMIO_TESTS_DEPENDENCIES && WOLFSSL_HAVE_TLS_UNIQUE */

    return EXPECT_RESULT();
}

#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
    !defined(SINGLE_THREADED) && defined(WOLFSSL_TLS13) && \
    !defined(NO_SESSION_CACHE)

/* Sessions to restore/store */
static WOLFSSL_SESSION* test_wolfSSL_CTX_add_session_client_sess;
static WOLFSSL_SESSION* test_wolfSSL_CTX_add_session_server_sess;
static WOLFSSL_CTX*     test_wolfSSL_CTX_add_session_server_ctx;

static void test_wolfSSL_CTX_add_session_ctx_ready(WOLFSSL_CTX* ctx)
{
    /* Don't store sessions. Lookup is still enabled. */
    AssertIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    AssertIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
}

static void test_wolfSSL_CTX_add_session_on_result(WOLFSSL* ssl)
{
    WOLFSSL_SESSION** sess;
    if (wolfSSL_is_server(ssl))
        sess = &test_wolfSSL_CTX_add_session_server_sess;
    else
        sess = &test_wolfSSL_CTX_add_session_client_sess;
    if (*sess == NULL) {
#ifdef NO_SESSION_CACHE_REF
        AssertNotNull(*sess = wolfSSL_get1_session(ssl));
#else
        /* Test for backwards compatibility */
        if (wolfSSL_is_server(ssl)) {
            AssertNotNull(*sess = wolfSSL_get1_session(ssl));
        }
        else {
            AssertNotNull(*sess = wolfSSL_get_session(ssl));
        }
#endif
        /* Now save the session in the internal store to make it available
         * for lookup. For TLS 1.3, we can't save the session without
         * WOLFSSL_TICKET_HAVE_ID because there is no way to retrieve the
         * session from cache. */
        if (wolfSSL_is_server(ssl)
#ifndef WOLFSSL_TICKET_HAVE_ID
                && wolfSSL_version(ssl) != TLS1_3_VERSION
#endif
                )
            AssertIntEQ(wolfSSL_CTX_add_session(wolfSSL_get_SSL_CTX(ssl),
                    *sess), WOLFSSL_SUCCESS);
    }
    else {
        /* If we have a session retrieved then remaining connections should be
         * resuming on that session */
        AssertIntEQ(wolfSSL_session_reused(ssl), 1);
    }
    /* Save CTX to be able to decrypt tickets */
    if (wolfSSL_is_server(ssl) &&
            test_wolfSSL_CTX_add_session_server_ctx == NULL) {
        AssertNotNull(test_wolfSSL_CTX_add_session_server_ctx
                = wolfSSL_get_SSL_CTX(ssl));
        AssertIntEQ(wolfSSL_CTX_up_ref(wolfSSL_get_SSL_CTX(ssl)),
                WOLFSSL_SUCCESS);
    }
#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {
        /* With WOLFSSL_TICKET_HAVE_ID the peer certs should be available
         * for all connections. TLS 1.3 only has tickets so if we don't
         * include the session id in the ticket then the certificates
         * will not be available on resumption. */
        WOLFSSL_X509* peer = wolfSSL_get_peer_certificate(ssl);
        AssertNotNull(peer);
        wolfSSL_X509_free(peer);
        AssertNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
    #ifdef OPENSSL_EXTRA
        AssertNotNull(SSL_SESSION_get0_peer(*sess));
    #endif
    }
#endif /* SESSION_CERTS */
}

static void test_wolfSSL_CTX_add_session_ssl_ready(WOLFSSL* ssl)
{
    /* Set the session to reuse for the client */
    AssertIntEQ(wolfSSL_set_session(ssl,
            test_wolfSSL_CTX_add_session_client_sess), WOLFSSL_SUCCESS);
}
#endif

static int test_wolfSSL_CTX_add_session(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
    !defined(SINGLE_THREADED) && defined(WOLFSSL_TLS13) && \
    !defined(NO_SESSION_CACHE)
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions client_cb;
    callback_functions server_cb;
    method_provider methods[][2] = {
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
        /* Without AES there are almost no ciphersuites available. This leads
         * to no ciphersuites being available and an error. */
        { wolfTLSv1_1_client_method, wolfTLSv1_1_server_method },
#endif
#ifndef WOLFSSL_NO_TLS12
        { wolfTLSv1_2_client_method, wolfTLSv1_2_server_method },
#endif
        /* Needs the default ticket callback since it is tied to the
         * connection context and this makes it easy to carry over the ticket
         * crypto context between connections */
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET)
        { wolfTLSv1_3_client_method, wolfTLSv1_3_server_method },
#endif
    };
    const size_t methodsLen = sizeof(methods)/sizeof(*methods);
    size_t i, j;

    for (i = 0; i < methodsLen; i++) {
        /* First run creates a connection while the second+ run will attempt
         * to resume the connection. The trick is that the internal cache
         * is turned off. wolfSSL_CTX_add_session should put the session in
         * the cache anyway. */
        test_wolfSSL_CTX_add_session_client_sess = NULL;
        test_wolfSSL_CTX_add_session_server_sess = NULL;
        test_wolfSSL_CTX_add_session_server_ctx = NULL;

#ifdef NO_SESSION_CACHE_REF
        for (j = 0; j < 4; j++) {
#else
        /* The session may be overwritten in this case. Do only one resumption
         * to stop this test from failing intermittently. */
        for (j = 0; j < 2; j++) {
#endif
#ifdef WOLFSSL_TIRTOS
            fdOpenSession(Task_self());
#endif

            StartTCP();
            InitTcpReady(&ready);

            XMEMSET(&client_args, 0, sizeof(func_args));
            XMEMSET(&server_args, 0, sizeof(func_args));

            XMEMSET(&client_cb, 0, sizeof(callback_functions));
            XMEMSET(&server_cb, 0, sizeof(callback_functions));
            client_cb.method  = methods[i][0];
            server_cb.method  = methods[i][1];

            server_args.signal    = &ready;
            server_args.callbacks = &server_cb;
            client_args.signal    = &ready;
            client_args.callbacks = &client_cb;

            if (test_wolfSSL_CTX_add_session_server_ctx != NULL) {
                server_cb.ctx = test_wolfSSL_CTX_add_session_server_ctx;
                server_cb.isSharedCtx = 1;
            }
            server_cb.ctx_ready = test_wolfSSL_CTX_add_session_ctx_ready;
            client_cb.ctx_ready = test_wolfSSL_CTX_add_session_ctx_ready;
            if (j != 0)
                client_cb.ssl_ready = test_wolfSSL_CTX_add_session_ssl_ready;
            server_cb.on_result = test_wolfSSL_CTX_add_session_on_result;
            client_cb.on_result = test_wolfSSL_CTX_add_session_on_result;
            server_cb.ticNoInit = 1; /* Use default builtin */

            start_thread(test_server_nofail, &server_args, &serverThread);
            wait_tcp_ready(&server_args);
            test_client_nofail(&client_args, NULL);
            join_thread(serverThread);

            ExpectTrue(client_args.return_code);
            ExpectTrue(server_args.return_code);

            FreeTcpReady(&ready);

            if (EXPECT_FAIL())
                break;
        }
        wolfSSL_SESSION_free(test_wolfSSL_CTX_add_session_client_sess);
        wolfSSL_SESSION_free(test_wolfSSL_CTX_add_session_server_sess);
        wolfSSL_CTX_free(test_wolfSSL_CTX_add_session_server_ctx);

        if (EXPECT_FAIL())
            break;
    }
#endif

    return EXPECT_RESULT();
}
#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)

/* twcase - prefix for test_wolfSSL_CTX_add_session_ext */
/* Sessions to restore/store */
static WOLFSSL_SESSION* twcase_server_first_session_ptr;
static WOLFSSL_SESSION* twcase_client_first_session_ptr;
static WOLFSSL_CTX*     twcase_server_current_ctx_ptr;
static int twcase_new_session_called    = 0;
static int twcase_remove_session_called = 0;
static int twcase_get_session_called    = 0;

/* Test default, SESSIONS_PER_ROW*SESSION_ROWS = 3*11, see ssl.c */
#define SESSION_CACHE_SIZE 33

typedef struct {
    const byte* key;  /* key, altSessionID, session ID, NULL if empty */
    WOLFSSL_SESSION* value;
} hashTable_entry;

typedef struct {
    hashTable_entry entries[SESSION_CACHE_SIZE];  /* hash slots */
    size_t capacity;                     /* size of entries */
    size_t length;                       /* number of items in the hash table */
    wolfSSL_Mutex htLock;                /* lock */
}hashTable;

static hashTable server_sessionCache;

static int twcase_new_sessionCb(WOLFSSL *ssl, WOLFSSL_SESSION *sess)
{
    int i;
    unsigned int len;
    (void)ssl;

    /*
     * This example uses a hash table.
     * Steps you should take for a non-demo code:
     * - acquire a lock for the file named according to the session id
     * - open the file
     * - encrypt and write the SSL_SESSION object to the file
     * - release the lock
     *
     * Return:
     *  0: The callback does not wish to hold a reference of the sess
     *  1: The callback wants to hold a reference of the sess. The callback is
     *     now also responsible for calling wolfSSL_SESSION_free() on sess.
     */
    if (sess == NULL)
        return 0;

    if (wc_LockMutex(&server_sessionCache.htLock) != 0) {
        return 0;
    }
    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].value == NULL) {
            server_sessionCache.entries[i].key = SSL_SESSION_get_id(sess, &len);
            server_sessionCache.entries[i].value = sess;
            server_sessionCache.length++;
            break;
        }
    }
    ++twcase_new_session_called;
    wc_UnLockMutex(&server_sessionCache.htLock);
    fprintf(stderr, "\t\ttwcase_new_session_called %d\n",
            twcase_new_session_called);
    return 1;
}

static void twcase_remove_sessionCb(WOLFSSL_CTX *ctx, WOLFSSL_SESSION *sess)
{
    int i;
    (void)ctx;
    (void)sess;

    if (sess == NULL)
        return;
    /*
     * This example uses a hash table.
     * Steps you should take for a non-demo code:
     * - acquire a lock for the file named according to the session id
     * - remove the file
     * - release the lock
     */
    if (wc_LockMutex(&server_sessionCache.htLock) != 0) {
        return;
    }
    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].key != NULL &&
           XMEMCMP(server_sessionCache.entries[i].key,
                   sess->sessionID, SSL_MAX_SSL_SESSION_ID_LENGTH) == 0) {
            wolfSSL_SESSION_free(server_sessionCache.entries[i].value);
            server_sessionCache.entries[i].value = NULL;
            server_sessionCache.entries[i].key = NULL;
            server_sessionCache.length--;
            break;
        }
    }
    ++twcase_remove_session_called;
    wc_UnLockMutex(&server_sessionCache.htLock);
    fprintf(stderr, "\t\ttwcase_remove_session_called %d\n",
            twcase_remove_session_called);
}

static WOLFSSL_SESSION *twcase_get_sessionCb(WOLFSSL *ssl,
                                  const unsigned char *id, int len, int *ref)
{
    int i;
    (void)ssl;
    (void)id;
    (void)len;

    /*
     * This example uses a hash table.
     * Steps you should take for a non-demo code:
     * - acquire a lock for the file named according to the session id in the
     *   2nd arg
     * - read and decrypt contents of file and create a new SSL_SESSION
     * - object release the lock
     * - return the new session object
     */
    fprintf(stderr, "\t\ttwcase_get_session_called %d\n",
            ++twcase_get_session_called);
    /* This callback want to retain a copy of the object. If we want wolfSSL to
     * be responsible for the pointer then set to 0. */
    *ref = 1;

    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].key != NULL &&
           XMEMCMP(server_sessionCache.entries[i].key, id,
                   SSL_MAX_SSL_SESSION_ID_LENGTH) == 0) {
           return server_sessionCache.entries[i].value;
        }
    }
    return NULL;
}
static int twcase_get_sessionCb_cleanup(void)
{
    int i;
    int cnt = 0;

    /* If  twcase_get_sessionCb sets *ref = 1, the application is responsible
     * for freeing sessions */

    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].value != NULL) {
            wolfSSL_SESSION_free(server_sessionCache.entries[i].value);
            cnt++;
        }
    }

    fprintf(stderr, "\t\ttwcase_get_sessionCb_cleanup freed %d sessions\n",
            cnt);

    return TEST_SUCCESS;
}

static int twcase_cache_intOff_extOff(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* off - Disable internal cache */
    ExpectIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    ExpectIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* off - Do not setup external cache */

    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return EXPECT_RESULT();
}

static int twcase_cache_intOn_extOff(WOLFSSL_CTX* ctx)
{
    /* on - internal cache is on by default */
    /* off - Do not setup external cache */
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return TEST_SUCCESS;
}

static int twcase_cache_intOff_extOn(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* off - Disable internal cache */
    ExpectIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    ExpectIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* on - Enable external cache */
    wolfSSL_CTX_sess_set_new_cb(ctx, twcase_new_sessionCb);
    wolfSSL_CTX_sess_set_remove_cb(ctx, twcase_remove_sessionCb);
    wolfSSL_CTX_sess_set_get_cb(ctx, twcase_get_sessionCb);

    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return EXPECT_RESULT();
}

static int twcase_cache_intOn_extOn(WOLFSSL_CTX* ctx)
{
    /* on - internal cache is on by default */
    /* on - Enable external cache */
    wolfSSL_CTX_sess_set_new_cb(ctx, twcase_new_sessionCb);
    wolfSSL_CTX_sess_set_remove_cb(ctx, twcase_remove_sessionCb);
    wolfSSL_CTX_sess_set_get_cb(ctx, twcase_get_sessionCb);

    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return TEST_SUCCESS;
}
static int twcase_cache_intOn_extOn_noTicket(WOLFSSL_CTX* ctx)
{
    /* on - internal cache is on by default */
    /* on - Enable external cache */
    wolfSSL_CTX_sess_set_new_cb(ctx, twcase_new_sessionCb);
    wolfSSL_CTX_sess_set_remove_cb(ctx, twcase_remove_sessionCb);
    wolfSSL_CTX_sess_set_get_cb(ctx, twcase_get_sessionCb);

    wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TICKET);
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return TEST_SUCCESS;
}
static int twcase_server_sess_ctx_pre_shutdown(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    WOLFSSL_SESSION** sess;
    if (wolfSSL_is_server(ssl))
        sess = &twcase_server_first_session_ptr;
    else
        return TEST_SUCCESS;

    if (*sess == NULL) {
        ExpectNotNull(*sess = wolfSSL_get1_session(ssl));
        /* Now save the session in the internal store to make it available
         * for lookup. For TLS 1.3, we can't save the session without
         * WOLFSSL_TICKET_HAVE_ID because there is no way to retrieve the
         * session from cache. */
        if (wolfSSL_is_server(ssl)
#ifndef WOLFSSL_TICKET_HAVE_ID
                && wolfSSL_version(ssl) != TLS1_3_VERSION
                && wolfSSL_version(ssl) != DTLS1_3_VERSION
#endif
                ) {
            ExpectIntEQ(wolfSSL_CTX_add_session(wolfSSL_get_SSL_CTX(ssl),
                    *sess), WOLFSSL_SUCCESS);
        }
    }
    /* Save CTX to be able to decrypt tickets */
    if (twcase_server_current_ctx_ptr == NULL) {
        ExpectNotNull(twcase_server_current_ctx_ptr = wolfSSL_get_SSL_CTX(ssl));
        ExpectIntEQ(wolfSSL_CTX_up_ref(wolfSSL_get_SSL_CTX(ssl)),
                    WOLFSSL_SUCCESS);
    }
#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {
        /* With WOLFSSL_TICKET_HAVE_ID the peer certs should be available
         * for all connections. TLS 1.3 only has tickets so if we don't
         * include the session id in the ticket then the certificates
         * will not be available on resumption. */
        WOLFSSL_X509* peer = NULL;
        ExpectNotNull(peer = wolfSSL_get_peer_certificate(ssl));
        wolfSSL_X509_free(peer);
        ExpectNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
    }
#endif
    return EXPECT_RESULT();
}

static int twcase_client_sess_ctx_pre_shutdown(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    WOLFSSL_SESSION** sess;
    sess = &twcase_client_first_session_ptr;
    if (*sess == NULL) {
        ExpectNotNull(*sess = wolfSSL_get1_session(ssl));
    }
    else {
        /* If we have a session retrieved then remaining connections should be
         * resuming on that session */
        ExpectIntEQ(wolfSSL_session_reused(ssl), 1);
    }

#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {

        WOLFSSL_X509* peer = wolfSSL_get_peer_certificate(ssl);
        ExpectNotNull(peer);
        wolfSSL_X509_free(peer);
        ExpectNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
#ifdef OPENSSL_EXTRA
        ExpectNotNull(wolfSSL_SESSION_get0_peer(*sess));
#endif
    }
#endif
    return EXPECT_RESULT();
}
static int twcase_client_set_sess_ssl_ready(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    /* Set the session to reuse for the client */
    ExpectNotNull(ssl);
    ExpectNotNull(twcase_client_first_session_ptr);
    ExpectIntEQ(wolfSSL_set_session(ssl,twcase_client_first_session_ptr),
                WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}

struct test_add_session_ext_params {
    method_provider client_meth;
    method_provider server_meth;
    const char* tls_version;
};

static int test_wolfSSL_CTX_add_session_ext(
    struct test_add_session_ext_params* param)
{
    EXPECT_DECLS;
    /* Test the default 33 sessions */
    int j;

    /* Clear cache before starting */
    wolfSSL_CTX_flush_sessions(NULL, -1);

    XMEMSET(&server_sessionCache, 0, sizeof(hashTable));
    if (wc_InitMutex(&server_sessionCache.htLock) != 0)
        return BAD_MUTEX_E;
    server_sessionCache.capacity = SESSION_CACHE_SIZE;

    fprintf(stderr, "\tBegin %s\n", param->tls_version);
    for (j = 0; j < 5; j++) {
        int tls13 = XSTRSTR(param->tls_version, "TLSv1_3") != NULL;
        int dtls = XSTRSTR(param->tls_version, "DTLS") != NULL;
        test_ssl_cbf client_cb;
        test_ssl_cbf server_cb;

        (void)dtls;

        /* Test five cache configurations */
        twcase_client_first_session_ptr = NULL;
        twcase_server_first_session_ptr = NULL;
        twcase_server_current_ctx_ptr = NULL;
        twcase_new_session_called    = 0;
        twcase_remove_session_called = 0;
        twcase_get_session_called    = 0;

        /* connection 1 - first connection */
        fprintf(stderr, "\tconnect: %s: j=%d\n", param->tls_version, j);

        XMEMSET(&client_cb, 0, sizeof(callback_functions));
        XMEMSET(&server_cb, 0, sizeof(callback_functions));
        client_cb.method  = param->client_meth;
        server_cb.method  = param->server_meth;

        if (dtls)
            client_cb.doUdp = server_cb.doUdp = 1;

        /* Setup internal and external cache */
        switch (j) {
            case 0:
                /* SSL_OP_NO_TICKET stateful ticket case */
                server_cb.ctx_ready = twcase_cache_intOn_extOn_noTicket;
                break;
            case 1:
                server_cb.ctx_ready = twcase_cache_intOn_extOn;
                break;
            case 2:
                server_cb.ctx_ready = twcase_cache_intOff_extOn;
                break;
            case 3:
                server_cb.ctx_ready = twcase_cache_intOn_extOff;
                break;
            case 4:
                server_cb.ctx_ready = twcase_cache_intOff_extOff;
                break;
        }
        client_cb.ctx_ready = twcase_cache_intOff_extOff;

        /* Add session to internal cache and save SSL session for testing */
        server_cb.on_result = twcase_server_sess_ctx_pre_shutdown;
        /* Save client SSL session for testing */
        client_cb.on_result = twcase_client_sess_ctx_pre_shutdown;
        server_cb.ticNoInit = 1; /* Use default builtin */
        /* Don't free/release ctx */
        server_cb.ctx = twcase_server_current_ctx_ptr;
        server_cb.isSharedCtx = 1;

        ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
            &server_cb, NULL), TEST_SUCCESS);

        ExpectIntEQ(twcase_get_session_called, 0);
        if (EXPECT_FAIL()) {
            wolfSSL_SESSION_free(twcase_client_first_session_ptr);
            wolfSSL_SESSION_free(twcase_server_first_session_ptr);
            wolfSSL_CTX_free(twcase_server_current_ctx_ptr);
            break;
        }

        switch (j) {
            case 0:
            case 1:
            case 2:
                /* cache cannot be searched with out a connection */
                /* Add a new session */
                ExpectIntEQ(twcase_new_session_called, 1);
                /* In twcase_server_sess_ctx_pre_shutdown
                 * wolfSSL_CTX_add_session which evicts the existing session
                 * in cache and adds it back in */
                ExpectIntLE(twcase_remove_session_called, 1);
                break;
            case 3:
            case 4:
                /* no external cache  */
                ExpectIntEQ(twcase_new_session_called, 0);
                ExpectIntEQ(twcase_remove_session_called, 0);
                break;
        }

        /* connection 2 - session resume */
        fprintf(stderr, "\tresume: %s: j=%d\n", param->tls_version, j);
        twcase_new_session_called    = 0;
        twcase_remove_session_called = 0;
        twcase_get_session_called    = 0;
        server_cb.on_result = 0;
        client_cb.on_result = 0;
        server_cb.ticNoInit = 1; /* Use default builtin */

        server_cb.ctx = twcase_server_current_ctx_ptr;

        /* try session resumption */
        client_cb.ssl_ready = twcase_client_set_sess_ssl_ready;

        ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
            &server_cb, NULL), TEST_SUCCESS);

        /* Clear cache before checking */
        wolfSSL_CTX_flush_sessions(NULL, -1);

        switch (j) {
            case 0:
                if (tls13) {
                    /* (D)TLSv1.3 stateful case */
                    /* cache hit */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);

                    /* (D)TLSv1.3 creates a new ticket,
                     * updates both internal and external cache */
                    ExpectIntEQ(twcase_new_session_called, 1);
                    /* A new session ID is created for a new ticket */
                    ExpectIntEQ(twcase_remove_session_called, 2);

                }
                else {
                    /* non (D)TLSv1.3 case, no update */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
#ifdef WOLFSSL_DTLS_NO_HVR_ON_RESUME
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);
#else
                    ExpectIntEQ(twcase_get_session_called, 1);
#endif
                    ExpectIntEQ(twcase_new_session_called, 0);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                break;
            case 1:
                if (tls13) {
                    /* (D)TLSv1.3 case */
                    /* cache hit */
                    ExpectIntEQ(twcase_get_session_called, 1);
                    /* (D)TLSv1.3 creates a new ticket,
                     * updates both internal and external cache */
                    ExpectIntEQ(twcase_new_session_called, 1);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown and by wolfSSL */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                else {
                    /* non (D)TLSv1.3 case */
                    /* cache hit */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
#ifdef WOLFSSL_DTLS_NO_HVR_ON_RESUME
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);
#else
                    ExpectIntEQ(twcase_get_session_called, 1);
#endif
                    ExpectIntEQ(twcase_new_session_called, 0);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                break;
            case 2:
                if (tls13) {
                    /* (D)TLSv1.3 case */
                    /* cache hit */
                    ExpectIntEQ(twcase_get_session_called, 1);
                    /* (D)TLSv1.3 creates a new ticket,
                     * updates both internal and external cache */
                    ExpectIntEQ(twcase_new_session_called, 1);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown and by wolfSSL */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                else {
                    /* non (D)TLSv1.3 case */
                    /* cache hit */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
#ifdef WOLFSSL_DTLS_NO_HVR_ON_RESUME
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);
#else
                    ExpectIntEQ(twcase_get_session_called, 1);
#endif
                    ExpectIntEQ(twcase_new_session_called, 0);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                break;
            case 3:
            case 4:
                /* no external cache */
                ExpectIntEQ(twcase_get_session_called, 0);
                ExpectIntEQ(twcase_new_session_called, 0);
                ExpectIntEQ(twcase_remove_session_called, 0);
                break;
        }
        wolfSSL_SESSION_free(twcase_client_first_session_ptr);
        wolfSSL_SESSION_free(twcase_server_first_session_ptr);
        wolfSSL_CTX_free(twcase_server_current_ctx_ptr);

        if (EXPECT_FAIL())
            break;
    }
    twcase_get_sessionCb_cleanup();
    XMEMSET(&server_sessionCache.entries, 0,
            sizeof(server_sessionCache.entries));
    fprintf(stderr, "\tEnd %s\n", param->tls_version);

    wc_FreeMutex(&server_sessionCache.htLock);

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_CTX_add_session_ext_tls13(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET) && defined(WOLFSSL_TICKET_HAVE_ID)
    struct test_add_session_ext_params param[1] =  {
        { wolfTLSv1_3_client_method, wolfTLSv1_3_server_method, "TLSv1_3" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_dtls13(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET) && defined(WOLFSSL_TICKET_HAVE_ID)
#ifdef WOLFSSL_DTLS13
    struct test_add_session_ext_params param[1] =  {
        { wolfDTLSv1_3_client_method, wolfDTLSv1_3_server_method, "DTLSv1_3" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_tls12(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#ifndef WOLFSSL_NO_TLS12
    struct test_add_session_ext_params param[1] =  {
        { wolfTLSv1_2_client_method, wolfTLSv1_2_server_method, "TLSv1_2" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_dtls12(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#ifndef WOLFSSL_NO_TLS12
#ifdef WOLFSSL_DTLS
    struct test_add_session_ext_params param[1] =  {
        { wolfDTLSv1_2_client_method, wolfDTLSv1_2_server_method, "DTLSv1_2" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_tls11(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
    struct test_add_session_ext_params param[1] =  {
        { wolfTLSv1_1_client_method, wolfTLSv1_1_server_method, "TLSv1_1" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_dtls1(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
#ifdef WOLFSSL_DTLS
    struct test_add_session_ext_params param[1] =  {
        { wolfDTLSv1_client_method, wolfDTLSv1_server_method, "DTLSv1_0" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
#endif
    return EXPECT_RESULT();
}

#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT)
/* canned export of a session using older version 3 */
static unsigned char version_3[] = {
    0xA5, 0xA3, 0x01, 0x88, 0x00, 0x3c, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x80, 0x0C, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x30,
    0x05, 0x09, 0x0A, 0x01, 0x01, 0x00, 0x0D, 0x05,
    0xFE, 0xFD, 0x01, 0x25, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x06, 0x00, 0x05, 0x00, 0x06, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x06, 0x00, 0x01, 0x00, 0x07, 0x00, 0x00,
    0x00, 0x30, 0x00, 0x00, 0x00, 0x10, 0x01, 0x01,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x3F,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x05,
    0x12, 0xCF, 0x22, 0xA1, 0x9F, 0x1C, 0x39, 0x1D,
    0x31, 0x11, 0x12, 0x1D, 0x11, 0x18, 0x0D, 0x0B,
    0xF3, 0xE1, 0x4D, 0xDC, 0xB1, 0xF1, 0x39, 0x98,
    0x91, 0x6C, 0x48, 0xE5, 0xED, 0x11, 0x12, 0xA0,
    0x00, 0xF2, 0x25, 0x4C, 0x09, 0x26, 0xD1, 0x74,
    0xDF, 0x23, 0x40, 0x15, 0x6A, 0x42, 0x2A, 0x26,
    0xA5, 0xAC, 0x56, 0xD5, 0x4A, 0x20, 0xB7, 0xE9,
    0xEF, 0xEB, 0xAF, 0xA8, 0x1E, 0x23, 0x7C, 0x04,
    0xAA, 0xA1, 0x6D, 0x92, 0x79, 0x7B, 0xFA, 0x80,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x0C, 0x79, 0x7B, 0xFA, 0x80, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xA1, 0x6D,
    0x92, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x10, 0x00, 0x20, 0x00, 0x04, 0x00,
    0x10, 0x00, 0x10, 0x08, 0x02, 0x05, 0x08, 0x01,
    0x30, 0x28, 0x00, 0x00, 0x0F, 0x00, 0x02, 0x00,
    0x09, 0x31, 0x32, 0x37, 0x2E, 0x30, 0x2E, 0x30,
    0x2E, 0x31, 0xED, 0x4F
};
#endif /* defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT) */

static int test_wolfSSL_dtls_export(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT)
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions server_cbf;
    callback_functions client_cbf;
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    /* set using dtls */
    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    XMEMSET(&client_cbf, 0, sizeof(callback_functions));
    server_cbf.method = wolfDTLSv1_2_server_method;
    client_cbf.method = wolfDTLSv1_2_client_method;
    server_args.callbacks = &server_cbf;
    client_args.callbacks = &client_cbf;

    server_args.signal = &ready;
    client_args.signal = &ready;

    start_thread(run_wolfssl_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    run_wolfssl_client(&client_args);
    join_thread(serverThread);

    ExpectTrue(client_args.return_code);
    ExpectTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    if (EXPECT_SUCCESS()) {
        SOCKET_T sockfd = 0;
        WOLFSSL_CTX* ctx = NULL;
        WOLFSSL*     ssl = NULL;
        char msg[64] = "hello wolfssl!";
        char reply[1024];
        int  msgSz = (int)XSTRLEN(msg);
        byte *session, *window;
        unsigned int sessionSz = 0;
        unsigned int windowSz = 0;

#ifndef TEST_IPV6
        struct sockaddr_in peerAddr;
#else
        struct sockaddr_in6 peerAddr;
#endif /* TEST_IPV6 */

        int i;


        /* Set ctx to DTLS 1.2 */
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_server_method()));
        ExpectNotNull(ssl = wolfSSL_new(ctx));

        /* test importing version 3 */
        ExpectIntGE(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);

        /* test importing bad length and bad version */
        version_3[2] += 1;
        ExpectIntLT(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);
        version_3[2] -= 1; version_3[1] = 0XA0;
        ExpectIntLT(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);
        wolfSSL_free(ssl);
        wolfSSL_CTX_free(ctx);


    /* check storing client state after connection and storing window only */
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    /* set using dtls */
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    server_cbf.method = wolfDTLSv1_2_server_method;
    server_cbf.doUdp = 1;
    server_args.callbacks = &server_cbf;
    server_args.argc = 3; /* set loop_count to 3 */


    server_args.signal = &ready;
    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);

    /* create and connect with client */
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_client_method()));
    ExpectIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
          wolfSSL_CTX_use_certificate_file(ctx, cliCertFile, SSL_FILETYPE_PEM));
    ExpectIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile, SSL_FILETYPE_PEM));
    tcp_connect(&sockfd, wolfSSLIP, server_args.signal->port, 1, 0, NULL);
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    ExpectIntEQ(wolfSSL_set_fd(ssl, sockfd), WOLFSSL_SUCCESS);

    /* store server information connected too */
    XMEMSET(&peerAddr, 0, sizeof(peerAddr));
#ifndef TEST_IPV6
    peerAddr.sin_family = AF_INET;
    ExpectIntEQ(XINET_PTON(AF_INET, wolfSSLIP, &peerAddr.sin_addr),1);
    peerAddr.sin_port = XHTONS(server_args.signal->port);
#else
    peerAddr.sin6_family = AF_INET6;
    ExpectIntEQ(
        XINET_PTON(AF_INET6, wolfSSLIP, &peerAddr.sin6_addr),1);
    peerAddr.sin6_port = XHTONS(server_args.signal->port);
#endif

    ExpectIntEQ(wolfSSL_dtls_set_peer(ssl, &peerAddr, sizeof(peerAddr)),
                    WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_connect(ssl), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_dtls_export(ssl, NULL, &sessionSz), 0);
    session = (byte*)XMALLOC(sessionSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectIntGT(wolfSSL_dtls_export(ssl, session, &sessionSz), 0);
    ExpectIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
    ExpectIntGT(wolfSSL_read(ssl, reply, sizeof(reply)), 0);
    ExpectIntEQ(wolfSSL_dtls_export_state_only(ssl, NULL, &windowSz), 0);
    window = (byte*)XMALLOC(windowSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectIntGT(wolfSSL_dtls_export_state_only(ssl, window, &windowSz), 0);
    wolfSSL_free(ssl);

    for (i = 1; EXPECT_SUCCESS() && i < server_args.argc; i++) {
        /* restore state */
        ExpectNotNull(ssl = wolfSSL_new(ctx));
        ExpectIntGT(wolfSSL_dtls_import(ssl, session, sessionSz), 0);
        ExpectIntGT(wolfSSL_dtls_import(ssl, window, windowSz), 0);
        ExpectIntEQ(wolfSSL_set_fd(ssl, sockfd), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_dtls_set_peer(ssl, &peerAddr, sizeof(peerAddr)),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
        ExpectIntGE(wolfSSL_read(ssl, reply, sizeof(reply)), 0);
        ExpectIntGT(wolfSSL_dtls_export_state_only(ssl, window, &windowSz), 0);
        wolfSSL_free(ssl);
    }
    XFREE(session, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(window, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_CTX_free(ctx);

    fprintf(stderr, "done and waiting for server\n");
    join_thread(serverThread);
    ExpectIntEQ(server_args.return_code, TEST_SUCCESS);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
    }
#endif

    return EXPECT_RESULT();
}


#if defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_NO_TLS12)
#ifdef WOLFSSL_TLS13
static const byte canned_client_tls13_session[] = {
    0xA7, 0xA4, 0x01, 0x18, 0x00, 0x41, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x80, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x01, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x01, 0x0A, 0x0F, 0x10, 0x01, 0x02, 0x09, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x00,
    0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00,
    0x11, 0x01, 0x01, 0x00, 0x20, 0x84, 0x4F, 0x18,
    0xD8, 0xC1, 0x24, 0xD8, 0xBB, 0x17, 0x9E, 0x31,
    0xA3, 0xF8, 0xA7, 0x3C, 0xBA, 0xEC, 0xFA, 0xB4,
    0x7F, 0xC5, 0x78, 0xEB, 0x6D, 0xE3, 0x2B, 0x7B,
    0x94, 0xBE, 0x20, 0x11, 0x7E, 0x17, 0x10, 0xA7,
    0x10, 0x19, 0xEC, 0x62, 0xCC, 0xBE, 0xF5, 0x01,
    0x35, 0x3C, 0xEA, 0xEF, 0x44, 0x3C, 0x40, 0xA2,
    0xBC, 0x18, 0x43, 0xA1, 0xA1, 0x65, 0x5C, 0x48,
    0xE2, 0xF9, 0x38, 0xEB, 0x11, 0x10, 0x72, 0x7C,
    0x78, 0x22, 0x13, 0x3B, 0x19, 0x40, 0xF0, 0x73,
    0xBE, 0x96, 0x14, 0x78, 0x26, 0xB9, 0x6B, 0x2E,
    0x72, 0x22, 0x0D, 0x90, 0x94, 0xDD, 0x78, 0x77,
    0xFC, 0x0C, 0x2E, 0x63, 0x6E, 0xF0, 0x0C, 0x35,
    0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0, 0x6F,
    0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A, 0xA0,
    0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x35, 0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0,
    0x6F, 0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A,
    0xA0, 0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x0C, 0x00, 0x10,
    0x00, 0x10, 0x07, 0x02, 0x04, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03
};

static const byte canned_server_tls13_session[] = {
    0xA7, 0xA4, 0x01, 0x18, 0x00, 0x41, 0x01, 0x00,
    0x01, 0x00, 0x00, 0x80, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x01, 0x0A, 0x0F, 0x10, 0x01, 0x02, 0x00, 0x0F,
    0x05, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x00,
    0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00,
    0x11, 0x01, 0x01, 0x00, 0x20, 0x84, 0x4F, 0x18,
    0xD8, 0xC1, 0x24, 0xD8, 0xBB, 0x17, 0x9E, 0x31,
    0xA3, 0xF8, 0xA7, 0x3C, 0xBA, 0xEC, 0xFA, 0xB4,
    0x7F, 0xC5, 0x78, 0xEB, 0x6D, 0xE3, 0x2B, 0x7B,
    0x94, 0xBE, 0x20, 0x11, 0x7E, 0x17, 0x10, 0xA7,
    0x10, 0x19, 0xEC, 0x62, 0xCC, 0xBE, 0xF5, 0x01,
    0x35, 0x3C, 0xEA, 0xEF, 0x44, 0x3C, 0x40, 0xA2,
    0xBC, 0x18, 0x43, 0xA1, 0xA1, 0x65, 0x5C, 0x48,
    0xE2, 0xF9, 0x38, 0xEB, 0x11, 0x10, 0x72, 0x7C,
    0x78, 0x22, 0x13, 0x3B, 0x19, 0x40, 0xF0, 0x73,
    0xBE, 0x96, 0x14, 0x78, 0x26, 0xB9, 0x6B, 0x2E,
    0x72, 0x22, 0x0D, 0x90, 0x94, 0xDD, 0x78, 0x77,
    0xFC, 0x0C, 0x2E, 0x63, 0x6E, 0xF0, 0x0C, 0x35,
    0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0, 0x6F,
    0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A, 0xA0,
    0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0xD3, 0xB7, 0xEE, 0x3A, 0xA0, 0x8E, 0xA1, 0x4D,
    0xC3, 0x2E, 0x5E, 0x06, 0x35, 0x41, 0xCD, 0xF3,
    0x49, 0x31, 0x08, 0xD0, 0x6F, 0x02, 0x3D, 0xC1,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x0C, 0x00, 0x10,
    0x00, 0x10, 0x07, 0x02, 0x04, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04
};
#endif /* WOLFSSL_TLS13 */

static const byte canned_client_session[] = {
    0xA7, 0xA4, 0x01, 0x40, 0x00, 0x41, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0,
    0x27, 0x0A, 0x0D, 0x10, 0x01, 0x01, 0x0A, 0x00,
    0x05, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x00,
    0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x0A, 0x01, 0x01, 0x00, 0x20, 0x69, 0x11, 0x6D,
    0x97, 0x15, 0x6E, 0x52, 0x27, 0xD6, 0x1D, 0x1D,
    0xF5, 0x0D, 0x59, 0xA5, 0xAC, 0x2E, 0x8C, 0x0E,
    0xCB, 0x26, 0x1E, 0xE2, 0xCE, 0xBB, 0xCE, 0xE1,
    0x7D, 0xD7, 0xEF, 0xA5, 0x44, 0x80, 0x2A, 0xDE,
    0xBB, 0x75, 0xB0, 0x1D, 0x75, 0x17, 0x20, 0x4C,
    0x08, 0x05, 0x1B, 0xBA, 0x60, 0x1F, 0x6C, 0x91,
    0x8C, 0xAA, 0xBB, 0xE5, 0xA3, 0x0B, 0x12, 0x3E,
    0xC0, 0x35, 0x43, 0x1D, 0xE2, 0x10, 0xE2, 0x02,
    0x92, 0x4B, 0x8F, 0x05, 0xA9, 0x4B, 0xCC, 0x90,
    0xC3, 0x0E, 0xC2, 0x0F, 0xE9, 0x33, 0x85, 0x9B,
    0x3C, 0x19, 0x21, 0xD5, 0x62, 0xE5, 0xE1, 0x17,
    0x8F, 0x8C, 0x19, 0x52, 0xD8, 0x59, 0x10, 0x2D,
    0x20, 0x6F, 0xBA, 0xC1, 0x1C, 0xD1, 0x82, 0xC7,
    0x32, 0x1B, 0xBB, 0xCC, 0x30, 0x03, 0xD7, 0x3A,
    0xC8, 0x18, 0xED, 0x58, 0xC8, 0x11, 0xFE, 0x71,
    0x9C, 0x71, 0xD8, 0x6B, 0xE0, 0x25, 0x64, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10,
    0x00, 0x00, 0x06, 0x01, 0x04, 0x08, 0x01, 0x20,
    0x28, 0x00, 0x09, 0xE1, 0x50, 0x70, 0x02, 0x2F,
    0x7E, 0xDA, 0xBD, 0x40, 0xC5, 0x58, 0x87, 0xCE,
    0x43, 0xF3, 0xC5, 0x8F, 0xA1, 0x59, 0x93, 0xEF,
    0x7E, 0xD3, 0xD0, 0xB5, 0x87, 0x1D, 0x81, 0x54,
    0x14, 0x63, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03
};


static const byte canned_server_session[] = {
    0xA7, 0xA4, 0x01, 0x40, 0x00, 0x41, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0,
    0x27, 0x08, 0x0F, 0x10, 0x01, 0x01, 0x00, 0x11,
    0x05, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x00,
    0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x0A, 0x01, 0x01, 0x00, 0x20, 0x69, 0x11, 0x6D,
    0x97, 0x15, 0x6E, 0x52, 0x27, 0xD6, 0x1D, 0x1D,
    0xF5, 0x0D, 0x59, 0xA5, 0xAC, 0x2E, 0x8C, 0x0E,
    0xCB, 0x26, 0x1E, 0xE2, 0xCE, 0xBB, 0xCE, 0xE1,
    0x7D, 0xD7, 0xEF, 0xA5, 0x44, 0x80, 0x2A, 0xDE,
    0xBB, 0x75, 0xB0, 0x1D, 0x75, 0x17, 0x20, 0x4C,
    0x08, 0x05, 0x1B, 0xBA, 0x60, 0x1F, 0x6C, 0x91,
    0x8C, 0xAA, 0xBB, 0xE5, 0xA3, 0x0B, 0x12, 0x3E,
    0xC0, 0x35, 0x43, 0x1D, 0xE2, 0x10, 0xE2, 0x02,
    0x92, 0x4B, 0x8F, 0x05, 0xA9, 0x4B, 0xCC, 0x90,
    0xC3, 0x0E, 0xC2, 0x0F, 0xE9, 0x33, 0x85, 0x9B,
    0x3C, 0x19, 0x21, 0xD5, 0x62, 0xE5, 0xE1, 0x17,
    0x8F, 0x8C, 0x19, 0x52, 0xD8, 0x59, 0x10, 0x2D,
    0x20, 0x6F, 0xBA, 0xC1, 0x1C, 0xD1, 0x82, 0xC7,
    0x32, 0x1B, 0xBB, 0xCC, 0x30, 0x03, 0xD7, 0x3A,
    0xC8, 0x18, 0xED, 0x58, 0xC8, 0x11, 0xFE, 0x71,
    0x9C, 0x71, 0xD8, 0x6B, 0xE0, 0x25, 0x64, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10,
    0x00, 0x00, 0x06, 0x01, 0x04, 0x08, 0x01, 0x20,
    0x28, 0x00, 0xC5, 0x8F, 0xA1, 0x59, 0x93, 0xEF,
    0x7E, 0xD3, 0xD0, 0xB5, 0x87, 0x1D, 0x81, 0x54,
    0x14, 0x63, 0x09, 0xE1, 0x50, 0x70, 0x02, 0x2F,
    0x7E, 0xDA, 0xBD, 0x40, 0xC5, 0x58, 0x87, 0xCE,
    0x43, 0xF3, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04
};


static THREAD_RETURN WOLFSSL_THREAD tls_export_server(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = 0;
    WOLFSSL* ssl = 0;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(USE_WINDOWS_API)
    port = ((func_args*)args)->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    /* do it here to detect failure */
    tcp_accept(&sockfd, &clientfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0, 0);
    CloseSocket(sockfd);

    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfTLSv1_2_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }
    if (ctx == NULL) {
        goto done;
    }
    wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256");

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
    wolfSSL_set_fd(ssl, clientfd);

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }
    idx = wolfSSL_read(ssl, input, sizeof(input)-1);
    if (idx > 0) {
        input[idx] = '\0';
        fprintf(stderr, "Client message export/import: %s\n", input);
    }
    else {
        fprintf(stderr, "ret = %d error = %d\n", idx,
            wolfSSL_get_error(ssl, idx));
        goto done;
    }

    if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
        /*err_sys("SSL_write failed");*/
        WOLFSSL_RETURN_FROM_THREAD(0);
    }

#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AESGCM)
    OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
    TicketCleanup();
#endif
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}


static void load_tls12_canned_server(WOLFSSL* ssl)
{
    int clientfd = wolfSSL_get_fd(ssl);
    AssertIntEQ(wolfSSL_tls_import(ssl, canned_server_session,
                sizeof(canned_server_session)), sizeof(canned_server_session));
    wolfSSL_set_fd(ssl, clientfd);
}


#ifdef WOLFSSL_TLS13
static void load_tls13_canned_server(WOLFSSL* ssl)
{
    int clientfd = wolfSSL_get_fd(ssl);
    AssertIntEQ(wolfSSL_tls_import(ssl, canned_server_tls13_session,
            sizeof(canned_server_tls13_session)),
            sizeof(canned_server_tls13_session));
    wolfSSL_set_fd(ssl, clientfd);
}
#endif


/* v is for version WOLFSSL_TLSV1_2 or WOLFSSL_TLSV1_3 */
static int test_wolfSSL_tls_export_run(int v)
{
    EXPECT_DECLS;
    SOCKET_T sockfd = 0;
    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    char msg[64] = "hello wolfssl!";
    char reply[1024];
    word32 replySz;
    int  msgSz = (int)XSTRLEN(msg);
    const byte* clientSession = NULL;
    int   clientSessionSz = 0;

    tcp_ready ready;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions server_cbf;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    switch (v) {
        case WOLFSSL_TLSV1_2:
            server_cbf.method     = wolfTLSv1_2_server_method;
            server_cbf.ssl_ready  = load_tls12_canned_server;

            /* setup the client side */
            ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
            wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256");
            clientSession   = canned_client_session;
            clientSessionSz = sizeof(canned_client_session);
            break;
    #ifdef WOLFSSL_TLS13
        case WOLFSSL_TLSV1_3:
            server_cbf.method     = wolfTLSv1_3_server_method;
            server_cbf.ssl_ready  = load_tls13_canned_server;

            /* setup the client side */
            ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
            clientSession   = canned_client_tls13_session;
            clientSessionSz = sizeof(canned_client_tls13_session);
            break;
    #endif
    }
    server_args.callbacks = &server_cbf;
    server_args.signal    = &ready;

    start_thread(tls_export_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);


#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ExpectNotNull(ssl = wolfSSL_new(ctx));
    tcp_connect(&sockfd, wolfSSLIP, ready.port, 0, 0, ssl);
    ExpectIntEQ(wolfSSL_tls_import(ssl, clientSession, clientSessionSz),
                clientSessionSz);
    replySz = sizeof(reply);
    ExpectIntGT(wolfSSL_tls_export(ssl, (byte*)reply, &replySz), 0);
#if !defined(NO_PSK) && defined(HAVE_ANON)
    /* index 20 has is setting if PSK was on and 49 is if anon is allowed */
    ExpectIntEQ(XMEMCMP(reply, clientSession, replySz), 0);
#endif
    wolfSSL_set_fd(ssl, sockfd);

    ExpectIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
    ExpectIntGT(wolfSSL_read(ssl, reply, sizeof(reply)-1), 0);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    join_thread(serverThread);

    ExpectIntEQ(server_args.return_code, TEST_SUCCESS);
    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_tls_export(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_NO_TLS12)
    test_wolfSSL_tls_export_run(WOLFSSL_TLSV1_2);
    #ifdef WOLFSSL_TLS13
    test_wolfSSL_tls_export_run(WOLFSSL_TLSV1_3);
    #endif
    res = TEST_RES_CHECK(1);
#endif

    return res;
}

/*----------------------------------------------------------------------------*
 | TLS extensions tests
 *----------------------------------------------------------------------------*/

#ifdef ENABLE_TLS_CALLBACK_TEST
/* Connection test runner - generic */
static void test_wolfSSL_client_server(callback_functions* client_callbacks,
                                       callback_functions* server_callbacks)
{
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

    StartTCP();

    client_args.callbacks = client_callbacks;
    server_args.callbacks = server_callbacks;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    /* RUN Server side */
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;
    start_thread(run_wolfssl_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);

    /* RUN Client side */
    run_wolfssl_client(&client_args);
    join_thread(serverThread);

    FreeTcpReady(&ready);
#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    client_callbacks->return_code = client_args.return_code;
    server_callbacks->return_code = server_args.return_code;
}
#endif /* ENABLE_TLS_CALLBACK_TEST */


#ifdef HAVE_SNI
static int test_wolfSSL_UseSNI_params(void)
{
    EXPECT_DECLS;
#if !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* invalid [ctx|ssl] */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(NULL, 0, "ctx", 3));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    NULL, 0, "ssl", 3));
    /* invalid type */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx, -1, "ctx", 3));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl, -1, "ssl", 3));
    /* invalid data */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx,  0, NULL,  3));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl,  0, NULL,  3));
    /* success case */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx,  0, "ctx", 3));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl,  0, "ssl", 3));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT */

    return EXPECT_RESULT();
}

/* BEGIN of connection tests callbacks */
static void use_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_UseSNI(ctx, WOLFSSL_SNI_HOST_NAME, "www.wolfssl.com", 15));
}

static void use_SNI_at_ssl(WOLFSSL* ssl)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
             wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, "www.wolfssl.com", 15));
}

static void different_SNI_at_ssl(WOLFSSL* ssl)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
             wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, "ww2.wolfssl.com", 15));
}

static void use_SNI_WITH_CONTINUE_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                              WOLFSSL_SNI_CONTINUE_ON_MISMATCH);
}

static void use_SNI_WITH_FAKE_ANSWER_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                                WOLFSSL_SNI_ANSWER_ON_MISMATCH);
}

static void use_MANDATORY_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    use_SNI_at_ctx(ctx);
    wolfSSL_CTX_SNI_SetOptions(ctx, WOLFSSL_SNI_HOST_NAME,
                                                  WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void use_MANDATORY_SNI_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                                  WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void use_PSEUDO_MANDATORY_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    use_SNI_at_ctx(ctx);
    wolfSSL_CTX_SNI_SetOptions(ctx, WOLFSSL_SNI_HOST_NAME,
                 WOLFSSL_SNI_ANSWER_ON_MISMATCH | WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void verify_UNKNOWN_SNI_on_server(WOLFSSL* ssl)
{
    AssertIntEQ(UNKNOWN_SNI_HOST_NAME_E, wolfSSL_get_error(ssl, 0));
}

static void verify_SNI_ABSENT_on_server(WOLFSSL* ssl)
{
    AssertIntEQ(SNI_ABSENT_ERROR, wolfSSL_get_error(ssl, 0));
}

static void verify_SNI_no_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = (void*) &type; /* to be overwritten */

    AssertIntEQ(WOLFSSL_SNI_NO_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertNotNull(request);
    AssertIntEQ(0, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNull(request);
}

static void verify_SNI_real_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = NULL;

    AssertIntEQ(WOLFSSL_SNI_REAL_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertIntEQ(15, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNotNull(request);
    AssertStrEQ("www.wolfssl.com", (char*)request);
}

static void verify_SNI_fake_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = NULL;

    AssertIntEQ(WOLFSSL_SNI_FAKE_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertIntEQ(15, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNotNull(request);
    AssertStrEQ("ww2.wolfssl.com", (char*)request);
}

static void verify_FATAL_ERROR_on_client(WOLFSSL* ssl)
{
    AssertIntEQ(FATAL_ERROR, wolfSSL_get_error(ssl, 0));
}
/* END of connection tests callbacks */

static int test_wolfSSL_UseSNI_connection(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;
    size_t i;
#ifdef WOLFSSL_STATIC_MEMORY
    byte cliMem[TEST_TLS_STATIC_MEMSZ];
    byte svrMem[TEST_TLS_STATIC_MEMSZ];
#endif
    struct {
        method_provider client_meth;
        method_provider server_meth;
    #ifdef WOLFSSL_STATIC_MEMORY
        wolfSSL_method_func client_meth_ex;
        wolfSSL_method_func server_meth_ex;
    #endif
    } methods[] = {
#if defined(WOLFSSL_NO_TLS12) && !defined(WOLFSSL_TLS13)
        {wolfSSLv23_client_method, wolfSSLv23_server_method
        #ifdef WOLFSSL_STATIC_MEMORY
            ,wolfSSLv23_client_method_ex, wolfSSLv23_server_method_ex
        #endif
        },
#endif
#ifndef WOLFSSL_NO_TLS12
        {wolfTLSv1_2_client_method, wolfTLSv1_2_server_method
        #ifdef WOLFSSL_STATIC_MEMORY
            ,wolfTLSv1_2_client_method_ex, wolfTLSv1_2_server_method_ex
        #endif
        },
#endif
#ifdef WOLFSSL_TLS13
        {wolfTLSv1_3_client_method, wolfTLSv1_3_server_method
        #ifdef WOLFSSL_STATIC_MEMORY
            ,wolfTLSv1_3_client_method_ex, wolfTLSv1_3_server_method_ex
        #endif
        },
#endif
    };
    size_t methodsSz = sizeof(methods) / sizeof(*methods);

    for (i = 0; i < methodsSz; i++) {
        XMEMSET(&client_cb, 0, sizeof(callback_functions));
        XMEMSET(&server_cb, 0, sizeof(callback_functions));
        client_cb.method = methods[i].client_meth;
        server_cb.method = methods[i].server_meth;
        client_cb.devId = testDevId;
        server_cb.devId = testDevId;
    #ifdef WOLFSSL_STATIC_MEMORY
        client_cb.method_ex = methods[i].client_meth_ex;
        server_cb.method_ex = methods[i].server_meth_ex;
        client_cb.mem = cliMem;
        client_cb.memSz = (word32)sizeof(cliMem);
        server_cb.mem = svrMem;
        server_cb.memSz = (word32)sizeof(svrMem);;
    #endif

        /* success case at ctx */
        fprintf(stderr, "\n\tsuccess case at ctx\n");
        client_cb.ctx_ready = use_SNI_at_ctx; client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
        server_cb.ctx_ready = use_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_real_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* success case at ssl */
        fprintf(stderr, "\tsuccess case at ssl\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_SNI_at_ssl; client_cb.on_result = verify_SNI_real_matching;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_at_ssl; server_cb.on_result = verify_SNI_real_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* default mismatch behavior */
        fprintf(stderr, "\tdefault mismatch behavior\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl; client_cb.on_result = verify_FATAL_ERROR_on_client;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_at_ssl;       server_cb.on_result = verify_UNKNOWN_SNI_on_server;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* continue on mismatch */
        fprintf(stderr, "\tcontinue on mismatch\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl;         client_cb.on_result = NULL;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_WITH_CONTINUE_at_ssl; server_cb.on_result = verify_SNI_no_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* fake answer on mismatch */
        fprintf(stderr, "\tfake answer on mismatch\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl;            client_cb.on_result = NULL;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_WITH_FAKE_ANSWER_at_ssl; server_cb.on_result = verify_SNI_fake_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - success */
        fprintf(stderr, "\tsni abort - success\n");
        client_cb.ctx_ready = use_SNI_at_ctx;           client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
        server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_real_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - abort when absent (ctx) */
        fprintf(stderr, "\tsni abort - abort when absent (ctx)\n");
        client_cb.ctx_ready = NULL;                     client_cb.ssl_ready = NULL; client_cb.on_result = verify_FATAL_ERROR_on_client;
        server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_ABSENT_on_server;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - abort when absent (ssl) */
        fprintf(stderr, "\tsni abort - abort when absent (ssl)\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = NULL;                     client_cb.on_result = verify_FATAL_ERROR_on_client;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_MANDATORY_SNI_at_ssl; server_cb.on_result = verify_SNI_ABSENT_on_server;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - success when overwritten */
        fprintf(stderr, "\tsni abort - success when overwritten\n");
        client_cb.ctx_ready = NULL;                     client_cb.ssl_ready = NULL;           client_cb.on_result = NULL;
        server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = use_SNI_at_ssl; server_cb.on_result = verify_SNI_no_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - success when allowing mismatches */
        fprintf(stderr, "\tsni abort - success when allowing mismatches\n");
        client_cb.ctx_ready = NULL;                            client_cb.ssl_ready = different_SNI_at_ssl; client_cb.on_result = NULL;
        server_cb.ctx_ready = use_PSEUDO_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL;                 server_cb.on_result = verify_SNI_fake_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);
    }

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */

    return res;
}

static int test_wolfSSL_SNI_GetFromBuffer(void)
{
    EXPECT_DECLS;
    byte buff[] = { /* www.paypal.com */
        0x00, 0x00, 0x00, 0x00, 0xff, 0x01, 0x00, 0x00, 0x60, 0x03, 0x03, 0x5c,
        0xc4, 0xb3, 0x8c, 0x87, 0xef, 0xa4, 0x09, 0xe0, 0x02, 0xab, 0x86, 0xca,
        0x76, 0xf0, 0x9e, 0x01, 0x65, 0xf6, 0xa6, 0x06, 0x13, 0x1d, 0x0f, 0xa5,
        0x79, 0xb0, 0xd4, 0x77, 0x22, 0xeb, 0x1a, 0x00, 0x00, 0x16, 0x00, 0x6b,
        0x00, 0x67, 0x00, 0x39, 0x00, 0x33, 0x00, 0x3d, 0x00, 0x3c, 0x00, 0x35,
        0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x21,
        0x00, 0x00, 0x00, 0x13, 0x00, 0x11, 0x00, 0x00, 0x0e, 0x77, 0x77, 0x77,
        0x2e, 0x70, 0x61, 0x79, 0x70, 0x61, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x00,
        0x0d, 0x00, 0x06, 0x00, 0x04, 0x04, 0x01, 0x02, 0x01
    };

    byte buff2[] = { /* api.textmate.org */
        0x16, 0x03, 0x01, 0x00, 0xc6, 0x01, 0x00, 0x00, 0xc2, 0x03, 0x03, 0x52,
        0x8b, 0x7b, 0xca, 0x69, 0xec, 0x97, 0xd5, 0x08, 0x03, 0x50, 0xfe, 0x3b,
        0x99, 0xc3, 0x20, 0xce, 0xa5, 0xf6, 0x99, 0xa5, 0x71, 0xf9, 0x57, 0x7f,
        0x04, 0x38, 0xf6, 0x11, 0x0b, 0xb8, 0xd3, 0x00, 0x00, 0x5e, 0x00, 0xff,
        0xc0, 0x24, 0xc0, 0x23, 0xc0, 0x0a, 0xc0, 0x09, 0xc0, 0x07, 0xc0, 0x08,
        0xc0, 0x28, 0xc0, 0x27, 0xc0, 0x14, 0xc0, 0x13, 0xc0, 0x11, 0xc0, 0x12,
        0xc0, 0x26, 0xc0, 0x25, 0xc0, 0x2a, 0xc0, 0x29, 0xc0, 0x05, 0xc0, 0x04,
        0xc0, 0x02, 0xc0, 0x03, 0xc0, 0x0f, 0xc0, 0x0e, 0xc0, 0x0c, 0xc0, 0x0d,
        0x00, 0x3d, 0x00, 0x3c, 0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x35,
        0x00, 0x0a, 0x00, 0x67, 0x00, 0x6b, 0x00, 0x33, 0x00, 0x39, 0x00, 0x16,
        0x00, 0xaf, 0x00, 0xae, 0x00, 0x8d, 0x00, 0x8c, 0x00, 0x8a, 0x00, 0x8b,
        0x00, 0xb1, 0x00, 0xb0, 0x00, 0x2c, 0x00, 0x3b, 0x01, 0x00, 0x00, 0x3b,
        0x00, 0x00, 0x00, 0x15, 0x00, 0x13, 0x00, 0x00, 0x10, 0x61, 0x70, 0x69,
        0x2e, 0x74, 0x65, 0x78, 0x74, 0x6d, 0x61, 0x74, 0x65, 0x2e, 0x6f, 0x72,
        0x67, 0x00, 0x0a, 0x00, 0x08, 0x00, 0x06, 0x00, 0x17, 0x00, 0x18, 0x00,
        0x19, 0x00, 0x0b, 0x00, 0x02, 0x01, 0x00, 0x00, 0x0d, 0x00, 0x0c, 0x00,
        0x0a, 0x05, 0x01, 0x04, 0x01, 0x02, 0x01, 0x04, 0x03, 0x02, 0x03
    };

    byte buff3[] = { /* no sni extension */
        0x16, 0x03, 0x03, 0x00, 0x4d, 0x01, 0x00, 0x00, 0x49, 0x03, 0x03, 0xea,
        0xa1, 0x9f, 0x60, 0xdd, 0x52, 0x12, 0x13, 0xbd, 0x84, 0x34, 0xd5, 0x1c,
        0x38, 0x25, 0xa8, 0x97, 0xd2, 0xd5, 0xc6, 0x45, 0xaf, 0x1b, 0x08, 0xe4,
        0x1e, 0xbb, 0xdf, 0x9d, 0x39, 0xf0, 0x65, 0x00, 0x00, 0x16, 0x00, 0x6b,
        0x00, 0x67, 0x00, 0x39, 0x00, 0x33, 0x00, 0x3d, 0x00, 0x3c, 0x00, 0x35,
        0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x0a,
        0x00, 0x0d, 0x00, 0x06, 0x00, 0x04, 0x04, 0x01, 0x02, 0x01
    };

    byte buff4[] = { /* last extension has zero size */
        0x16, 0x03, 0x01, 0x00, 0xba, 0x01, 0x00, 0x00,
        0xb6, 0x03, 0x03, 0x83, 0xa3, 0xe6, 0xdc, 0x16, 0xa1, 0x43, 0xe9, 0x45,
        0x15, 0xbd, 0x64, 0xa9, 0xb6, 0x07, 0xb4, 0x50, 0xc6, 0xdd, 0xff, 0xc2,
        0xd3, 0x0d, 0x4f, 0x36, 0xb4, 0x41, 0x51, 0x61, 0xc1, 0xa5, 0x9e, 0x00,
        0x00, 0x28, 0xcc, 0x14, 0xcc, 0x13, 0xc0, 0x2b, 0xc0, 0x2f, 0x00, 0x9e,
        0xc0, 0x0a, 0xc0, 0x09, 0xc0, 0x13, 0xc0, 0x14, 0xc0, 0x07, 0xc0, 0x11,
        0x00, 0x33, 0x00, 0x32, 0x00, 0x39, 0x00, 0x9c, 0x00, 0x2f, 0x00, 0x35,
        0x00, 0x0a, 0x00, 0x05, 0x00, 0x04, 0x01, 0x00, 0x00, 0x65, 0xff, 0x01,
        0x00, 0x01, 0x00, 0x00, 0x0a, 0x00, 0x08, 0x00, 0x06, 0x00, 0x17, 0x00,
        0x18, 0x00, 0x19, 0x00, 0x0b, 0x00, 0x02, 0x01, 0x00, 0x00, 0x23, 0x00,
        0x00, 0x33, 0x74, 0x00, 0x00, 0x00, 0x10, 0x00, 0x1b, 0x00, 0x19, 0x06,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33, 0x08, 0x73, 0x70, 0x64, 0x79, 0x2f,
        0x33, 0x2e, 0x31, 0x08, 0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31,
        0x75, 0x50, 0x00, 0x00, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x0d, 0x00, 0x12, 0x00, 0x10, 0x04, 0x01, 0x05, 0x01, 0x02,
        0x01, 0x04, 0x03, 0x05, 0x03, 0x02, 0x03, 0x04, 0x02, 0x02, 0x02, 0x00,
        0x12, 0x00, 0x00
    };

    byte buff5[] = { /* SSL v2.0 client hello */
        0x00, 0x2b, 0x01, 0x03, 0x01, 0x00, 0x09, 0x00, 0x00,
        /* dummy bytes below, just to pass size check */
        0xb6, 0x03, 0x03, 0x83, 0xa3, 0xe6, 0xdc, 0x16, 0xa1, 0x43, 0xe9, 0x45,
        0x15, 0xbd, 0x64, 0xa9, 0xb6, 0x07, 0xb4, 0x50, 0xc6, 0xdd, 0xff, 0xc2,
        0xd3, 0x0d, 0x4f, 0x36, 0xb4, 0x41, 0x51, 0x61, 0xc1, 0xa5, 0x9e, 0x00,
    };

    byte result[32] = {0};
    word32 length   = 32;

    ExpectIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff4, sizeof(buff4),
                                                           0, result, &length));

    ExpectIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff3, sizeof(buff3),
                                                           0, result, &length));

    ExpectIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff2, sizeof(buff2),
                                                           1, result, &length));

    ExpectIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    buff[0] = 0x16;

    ExpectIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    buff[1] = 0x03;

    ExpectIntEQ(SNI_UNSUPPORTED, wolfSSL_SNI_GetFromBuffer(buff,
                                           sizeof(buff), 0, result, &length));
    buff[2] = 0x03;

    ExpectIntEQ(INCOMPLETE_DATA, wolfSSL_SNI_GetFromBuffer(buff,
                                           sizeof(buff), 0, result, &length));
    buff[4] = 0x64;

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    if (EXPECT_SUCCESS())
        result[length] = 0;
    ExpectStrEQ("www.paypal.com", (const char*) result);

    length = 32;

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SNI_GetFromBuffer(buff2, sizeof(buff2),
                                                           0, result, &length));
    if (EXPECT_SUCCESS())
        result[length] = 0;
    ExpectStrEQ("api.textmate.org", (const char*) result);

    /* SSL v2.0 tests */
    ExpectIntEQ(SNI_UNSUPPORTED, wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[2] = 0x02;
    ExpectIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[2] = 0x01; buff5[6] = 0x08;
    ExpectIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[6] = 0x09; buff5[8] = 0x01;
    ExpectIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    return EXPECT_RESULT();
}

#endif /* HAVE_SNI */

#endif /* HAVE_IO_TESTS_DEPENDENCIES */

static int test_wolfSSL_UseTrustedCA(void)
{
    EXPECT_DECLS;
#if defined(HAVE_TRUSTED_CA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_RSA)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL     *ssl = NULL;
    byte        id[20];

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull((ctx = wolfSSL_CTX_new(wolfSSLv23_server_method())));
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
#else
    ExpectNotNull((ctx = wolfSSL_CTX_new(wolfSSLv23_client_method())));
#endif
    ExpectNotNull((ssl = wolfSSL_new(ctx)));
    XMEMSET(id, 0, sizeof(id));

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(NULL, 0, NULL, 0));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1+1, NULL, 0));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1, NULL, 0));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1, id, 5));
#ifdef NO_SHA
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_KEY_SHA1, id, sizeof(id)));
#endif
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_X509_NAME, id, 0));

    /* success cases */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_PRE_AGREED, NULL, 0));
#ifndef NO_SHA
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_KEY_SHA1, id, sizeof(id)));
#endif
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_X509_NAME, id, 5));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* HAVE_TRUSTED_CA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseMaxFragment(void)
{
    EXPECT_DECLS;
#if defined(HAVE_MAX_FRAGMENT) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA)

#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
  #ifndef NO_WOLFSSL_SERVER
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #else
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #endif
    WOLFSSL     *ssl = NULL;
  #ifdef OPENSSL_EXTRA
    int (*UseMaxFragment)(SSL *s, uint8_t mode);
    int (*CTX_UseMaxFragment)(SSL_CTX *c, uint8_t mode);
  #else
    int (*UseMaxFragment)(WOLFSSL *s, unsigned char mode);
    int (*CTX_UseMaxFragment)(WOLFSSL_CTX *c, unsigned char mode);
  #endif

  #ifndef NO_WOLFSSL_SERVER
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
  #endif

    ExpectNotNull(ctx);

    ExpectNotNull(ssl = wolfSSL_new(ctx));

  #ifdef OPENSSL_EXTRA
    CTX_UseMaxFragment = SSL_CTX_set_tlsext_max_fragment_length;
    UseMaxFragment = SSL_set_tlsext_max_fragment_length;
  #else
    UseMaxFragment = wolfSSL_UseMaxFragment;
    CTX_UseMaxFragment = wolfSSL_CTX_UseMaxFragment;
  #endif

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(NULL, WOLFSSL_MFL_2_9));
    ExpectIntNE(WOLFSSL_SUCCESS, UseMaxFragment(    NULL, WOLFSSL_MFL_2_9));
    ExpectIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_MIN-1));
    ExpectIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_MAX+1));
    ExpectIntNE(WOLFSSL_SUCCESS, UseMaxFragment(ssl, WOLFSSL_MFL_MIN-1));
    ExpectIntNE(WOLFSSL_SUCCESS, UseMaxFragment(ssl, WOLFSSL_MFL_MAX+1));

    /* success case */
  #ifdef OPENSSL_EXTRA
    ExpectIntEQ(BAD_FUNC_ARG, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_8));
  #else
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_8));
  #endif
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_9));
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_10));
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_11));
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_12));
  #ifdef OPENSSL_EXTRA
    ExpectIntEQ(BAD_FUNC_ARG, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_2_13));

    ExpectIntEQ(BAD_FUNC_ARG, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_8));
  #else
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_13));

    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_8));
  #endif
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_9));
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_10));
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_11));
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_12));

  #ifdef OPENSSL_EXTRA
    ExpectIntEQ(BAD_FUNC_ARG, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_13));
  #else
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_13));
  #endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseTruncatedHMAC(void)
{
    EXPECT_DECLS;
#if defined(HAVE_TRUNCATED_HMAC) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
  #ifndef NO_WOLFSSL_SERVER
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #else
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #endif
    WOLFSSL     *ssl = NULL;

    ExpectNotNull(ctx);

  #ifndef NO_WOLFSSL_SERVER
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
  #endif

    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseTruncatedHMAC(NULL));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTruncatedHMAC(NULL));

    /* success case */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseTruncatedHMAC(ctx));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTruncatedHMAC(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseSupportedCurve(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SUPPORTED_CURVES) && !defined(NO_WOLFSSL_CLIENT) && \
    !defined(NO_TLS)
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS,
                    wolfSSL_CTX_UseSupportedCurve(NULL, WOLFSSL_ECC_SECP256R1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSupportedCurve(ctx,  0));

    ExpectIntNE(WOLFSSL_SUCCESS,
                        wolfSSL_UseSupportedCurve(NULL, WOLFSSL_ECC_SECP256R1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSupportedCurve(ssl,  0));

    /* success case */
    ExpectIntEQ(WOLFSSL_SUCCESS,
                     wolfSSL_CTX_UseSupportedCurve(ctx, WOLFSSL_ECC_SECP256R1));
    ExpectIntEQ(WOLFSSL_SUCCESS,
                         wolfSSL_UseSupportedCurve(ssl, WOLFSSL_ECC_SECP256R1));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

#if defined(HAVE_ALPN) && defined(HAVE_IO_TESTS_DEPENDENCIES)

static void verify_ALPN_FATAL_ERROR_on_client(WOLFSSL* ssl)
{
    AssertIntEQ(UNKNOWN_ALPN_PROTOCOL_NAME_E, wolfSSL_get_error(ssl, 0));
}

static void use_ALPN_all(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, alpn_list, sizeof(alpn_list),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_all_continue(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, alpn_list, sizeof(alpn_list),
                                             WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));
}

static void use_ALPN_one(WOLFSSL* ssl)
{
    /* spdy/2 */
    char proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_unknown(WOLFSSL* ssl)
{
    /* http/2.0 */
    char proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x32, 0x2e, 0x30};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_unknown_continue(WOLFSSL* ssl)
{
    /* http/2.0 */
    char proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x32, 0x2e, 0x30};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));
}

static void verify_ALPN_not_matching_spdy3(WOLFSSL* ssl)
{
    /* spdy/3 */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};

    char *proto = NULL;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntNE(1, sizeof(nego_proto) == protoSz);
    if (proto) {
        AssertIntNE(0, XMEMCMP(nego_proto, proto, sizeof(nego_proto)));
    }
}

static void verify_ALPN_not_matching_continue(WOLFSSL* ssl)
{
    char *proto = NULL;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_ALPN_NOT_FOUND,
                wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, (0 == protoSz));
    AssertIntEQ(1, (NULL == proto));
}

static void verify_ALPN_matching_http1(WOLFSSL* ssl)
{
    /* http/1.1 */
    char nego_proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    char *proto;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_ALPN_matching_spdy2(WOLFSSL* ssl)
{
    /* spdy/2 */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};
    char *proto;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_ALPN_client_list(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    char    *clist = NULL;
    word16  clistSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetPeerProtocol(ssl, &clist,
                                                          &clistSz));

    /* check value */
    AssertIntEQ(1, sizeof(alpn_list) == clistSz);
    AssertIntEQ(0, XMEMCMP(alpn_list, clist, clistSz));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_FreePeerProtocol(ssl, &clist));
}

static int test_wolfSSL_UseALPN_connection(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;

    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method = wolfSSLv23_client_method;
    server_cb.method = wolfSSLv23_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;

    /* success case same list */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_matching_http1;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case only one for server */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_one; server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case only one for client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_one; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case none for client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = NULL;         client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = NULL;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case mismatch behavior but option 'continue' set */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all_continue;     client_cb.on_result = verify_ALPN_not_matching_continue;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_unknown_continue; server_cb.on_result = NULL;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case read protocol send by client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_one; server_cb.on_result = verify_ALPN_client_list;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* mismatch behavior with same list
        * the first and only this one must be taken */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_not_matching_spdy3;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* default mismatch behavior */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all;     client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_unknown; server_cb.on_result = verify_ALPN_FATAL_ERROR_on_client;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */
    return res;
}

static int test_wolfSSL_UseALPN_params(void)
{
    EXPECT_DECLS;
#ifndef NO_WOLFSSL_CLIENT
    /* "http/1.1" */
    char http1[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    /* "spdy/1" */
    char spdy1[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x31};
    /* "spdy/2" */
    char spdy2[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};
    /* "spdy/3" */
    char spdy3[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    char buff[256];
    word32 idx;

    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS,
                wolfSSL_UseALPN(NULL, http1, sizeof(http1),
                                WOLFSSL_ALPN_FAILED_ON_MISMATCH));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, NULL, 0,
                                              WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* success case */
    /* http1 only */
    ExpectIntEQ(WOLFSSL_SUCCESS,
                wolfSSL_UseALPN(ssl, http1, sizeof(http1),
                                WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* http1, spdy1 */
    XMEMCPY(buff, http1, sizeof(http1));
    idx = sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                              WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* http1, spdy2, spdy1 */
    XMEMCPY(buff, http1, sizeof(http1));
    idx = sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy2, sizeof(spdy2));
    idx += sizeof(spdy2);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                              WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* spdy3, http1, spdy2, spdy1 */
    XMEMCPY(buff, spdy3, sizeof(spdy3));
    idx = sizeof(spdy3);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, http1, sizeof(http1));
    idx += sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy2, sizeof(spdy2));
    idx += sizeof(spdy2);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                            WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}
#endif /* HAVE_ALPN  */

#ifdef HAVE_ALPN_PROTOS_SUPPORT
static void CTX_set_alpn_protos(SSL_CTX *ctx)
{
    unsigned char p[] = {
       8, 'h', 't', 't', 'p', '/', '1', '.', '1',
       6, 's', 'p', 'd', 'y', '/', '2',
       6, 's', 'p', 'd', 'y', '/', '1',
    };

    unsigned char p_len = sizeof(p);
    int ret;

    ret = SSL_CTX_set_alpn_protos(ctx, p, p_len);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    AssertIntEQ(ret, 0);
#else
    AssertIntEQ(ret, SSL_SUCCESS);
#endif
}

static void set_alpn_protos(SSL* ssl)
{
    unsigned char p[] = {
       6, 's', 'p', 'd', 'y', '/', '3',
       8, 'h', 't', 't', 'p', '/', '1', '.', '1',
       6, 's', 'p', 'd', 'y', '/', '2',
       6, 's', 'p', 'd', 'y', '/', '1',
    };

    unsigned char p_len = sizeof(p);
    int ret;

    ret = SSL_set_alpn_protos(ssl, p, p_len);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    AssertIntEQ(ret, 0);
#else
    AssertIntEQ(ret, SSL_SUCCESS);
#endif

}

static void verify_alpn_matching_spdy3(WOLFSSL* ssl)
{
    /* "spdy/3" */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    const unsigned char *proto;
    unsigned int protoSz = 0;

    SSL_get0_alpn_selected(ssl, &proto, &protoSz);

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_alpn_matching_http1(WOLFSSL* ssl)
{
    /* "http/1.1" */
    char nego_proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    const unsigned char *proto;
    unsigned int protoSz = 0;

    SSL_get0_alpn_selected(ssl, &proto, &protoSz);

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static int test_wolfSSL_set_alpn_protos(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;

    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method = wolfSSLv23_client_method;
    server_cb.method = wolfSSLv23_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;

    /* use CTX_alpn_protos */
    client_cb.ctx_ready = CTX_set_alpn_protos; client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
    server_cb.ctx_ready = CTX_set_alpn_protos; server_cb.ssl_ready = NULL; server_cb.on_result = verify_alpn_matching_http1;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* use set_alpn_protos */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = set_alpn_protos; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = set_alpn_protos; server_cb.on_result = verify_alpn_matching_spdy3;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    res = TEST_SUCCESS;
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */
    return res;
}

#endif /* HAVE_ALPN_PROTOS_SUPPORT */

static int test_wolfSSL_DisableExtendedMasterSecret(void)
{
    EXPECT_DECLS;
#if defined(HAVE_EXTENDED_MASTER) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_DisableExtendedMasterSecret(NULL));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_DisableExtendedMasterSecret(NULL));

    /* success cases */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_DisableExtendedMasterSecret(ctx));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_DisableExtendedMasterSecret(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_wolfSSL_UseSecureRenegotiation(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SECURE_RENEGOTIATION) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(NULL));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(NULL));

    /* success cases */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

/* Test reconnecting with a different ciphersuite after a renegotiation. */
static int test_wolfSSL_SCR_Reconnect(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SECURE_RENEGOTIATION) && \
    defined(BUILD_TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384) && \
    defined(BUILD_TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256) && \
    defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES)
    struct test_memio_ctx test_ctx;
    WOLFSSL_CTX *ctx_c = NULL, *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL, *ssl_s = NULL;
    byte data;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    test_ctx.c_ciphers = "ECDHE-RSA-AES256-GCM-SHA384";
    test_ctx.s_ciphers =
        "ECDHE-RSA-AES256-GCM-SHA384:ECDHE-RSA-CHACHA20-POLY1305";
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx_c));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx_s));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl_c));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl_s));
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    /* WOLFSSL_FATAL_ERROR since it will block */
    ExpectIntEQ(wolfSSL_Rehandshake(ssl_s), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WOLFSSL_FATAL_ERROR),
                WOLFSSL_ERROR_WANT_READ);
    ExpectIntEQ(wolfSSL_read(ssl_c, &data, 1), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WOLFSSL_FATAL_ERROR),
                WOLFSSL_ERROR_WANT_READ);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    wolfSSL_free(ssl_c);
    ssl_c = NULL;
    wolfSSL_free(ssl_s);
    ssl_s = NULL;
    wolfSSL_CTX_free(ctx_c);
    ctx_c = NULL;
    test_ctx.c_ciphers = "ECDHE-RSA-CHACHA20-POLY1305";
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    wolfSSL_free(ssl_s);
    wolfSSL_free(ssl_c);
    wolfSSL_CTX_free(ctx_s);
    wolfSSL_CTX_free(ctx_c);
#endif
    return EXPECT_RESULT();
}

#if !defined(NO_FILESYSTEM) && !defined(NO_WOLFSSL_SERVER) && \
    (!defined(NO_RSA) || defined(HAVE_ECC))
/* Called when writing. */
static int DummySend(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    (void)ssl;
    (void)buf;
    (void)sz;
    (void)ctx;

    /* Force error return from wolfSSL_accept_TLSv13(). */
    return WANT_WRITE;
}
/* Called when reading. */
static int BufferInfoRecv(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    WOLFSSL_BUFFER_INFO* msg = (WOLFSSL_BUFFER_INFO*)ctx;
    int len = (int)msg->length;

    (void)ssl;
    (void)sz;

    /* Pass back as much of message as will fit in buffer. */
    if (len > sz)
        len = sz;
    XMEMCPY(buf, msg->buffer, len);
    /* Move over returned data. */
    msg->buffer += len;
    msg->length -= len;

    /* Amount actually copied. */
    return len;
}
#endif

/* Test the detection of duplicate known TLS extensions.
 * Specifically in a ClientHello.
 */
static int test_tls_ext_duplicate(void)
{
    EXPECT_DECLS;
#if !defined(NO_WOLFSSL_SERVER) && (!defined(NO_RSA) || defined(HAVE_ECC)) && \
    !defined(NO_FILESYSTEM)
    const unsigned char clientHelloDupTlsExt[] = {
        0x16, 0x03, 0x03, 0x00, 0x6a, 0x01, 0x00, 0x00,
        0x66, 0x03, 0x03, 0xf4, 0x65, 0xbd, 0x22, 0xfe,
        0x6e, 0xab, 0x66, 0xdd, 0xcf, 0xe9, 0x65, 0x55,
        0xe8, 0xdf, 0xc3, 0x8e, 0x4b, 0x00, 0xbc, 0xf8,
        0x23, 0x57, 0x1b, 0xa0, 0xc8, 0xa9, 0xe2, 0x8c,
        0x91, 0x6e, 0xf9, 0x20, 0xf7, 0x5c, 0xc5, 0x5b,
        0x75, 0x8c, 0x47, 0x0a, 0x0e, 0xc4, 0x1a, 0xda,
        0xef, 0x75, 0xe5, 0x21, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x13, 0x01,
        0x00, 0x9e, 0x01, 0x00,
        /* Extensions - duplicate signature algorithms. */
                                0x00, 0x19, 0x00, 0x0d,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01, 0x00, 0x0d,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01,
        /* Supported Versions extension for TLS 1.3. */
                                            0x00, 0x2b,
        0x00, 0x05, 0x04, 0x03, 0x04, 0x03, 0x03
    };
    WOLFSSL_BUFFER_INFO msg;
    const char* testCertFile;
    const char* testKeyFile;
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL     *ssl = NULL;

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
        WOLFSSL_FILETYPE_PEM));

    /* Read from 'msg'. */
    wolfSSL_SetIORecv(ctx, BufferInfoRecv);
    /* No where to send to - dummy sender. */
    wolfSSL_SetIOSend(ctx, DummySend);

    ssl = wolfSSL_new(ctx);
    ExpectNotNull(ssl);

    msg.buffer = (unsigned char*)clientHelloDupTlsExt;
    msg.length = (unsigned int)sizeof(clientHelloDupTlsExt);
    wolfSSL_SetIOReadCtx(ssl, &msg);

    ExpectIntNE(wolfSSL_accept(ssl), WOLFSSL_SUCCESS);
    /* can return duplicate ext error or socket error if the peer closed down
     * while sending alert */
    if (wolfSSL_get_error(ssl, 0) != SOCKET_ERROR_E) {
        ExpectIntEQ(wolfSSL_get_error(ssl, 0), DUPLICATE_TLS_EXT_E);
    }

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

/*----------------------------------------------------------------------------*
 | X509 Tests
 *----------------------------------------------------------------------------*/
static int test_wolfSSL_X509_NAME_get_entry(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA)
#if defined(OPENSSL_ALL) || \
        (defined(OPENSSL_EXTRA) && \
            (defined(KEEP_PEER_CERT) || defined(SESSION_CERTS)))
    /* use openssl like name to test mapping */
    X509_NAME_ENTRY* ne;
    X509_NAME* name;
    X509* x509 = NULL;
#ifndef NO_FILESYSTEM
    ASN1_STRING* asn;
    char* subCN = NULL;
#endif
    int idx;
    ASN1_OBJECT *object = NULL;
#if defined(WOLFSSL_APACHE_HTTPD) || defined(OPENSSL_ALL) || \
    defined(WOLFSSL_NGINX)
#ifndef NO_BIO
    BIO* bio = NULL;
#endif
#endif

#ifndef NO_FILESYSTEM
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = X509_get_subject_name(x509));
    ExpectIntGE(idx = X509_NAME_get_index_by_NID(name, NID_commonName, -1), 0);
    ExpectNotNull(ne = X509_NAME_get_entry(name, idx));
    ExpectNotNull(asn = X509_NAME_ENTRY_get_data(ne));
    ExpectNotNull(subCN = (char*)ASN1_STRING_data(asn));
    wolfSSL_FreeX509(x509);
    x509 = NULL;
#endif

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = X509_get_subject_name(x509));
    ExpectIntGE(idx = X509_NAME_get_index_by_NID(name, NID_commonName, -1), 0);

#if defined(WOLFSSL_APACHE_HTTPD) || defined(OPENSSL_ALL) || \
    defined(WOLFSSL_NGINX)
#ifndef NO_BIO
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(X509_NAME_print_ex(bio, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_NAME_print_ex_fp(stderr, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_SUCCESS);
    BIO_free(bio);
#endif
#endif

    ExpectNotNull(ne = X509_NAME_get_entry(name, idx));
    ExpectNotNull(object = X509_NAME_ENTRY_get_object(ne));
    wolfSSL_FreeX509(x509);
#endif /* OPENSSL_ALL || (OPENSSL_EXTRA && (KEEP_PEER_CERT || SESSION_CERTS) */
#endif /* !NO_CERTS && !NO_RSA */

    return EXPECT_RESULT();
}

/* Testing functions dealing with PKCS12 parsing out X509 certs */
static int test_wolfSSL_PKCS12(void)
{
    EXPECT_DECLS;
    /* .p12 file is encrypted with DES3 */
#ifndef HAVE_FIPS /* Password used in cert "wolfSSL test" is only 12-bytes
                   * (96-bit) FIPS mode requires Minimum of 14-byte (112-bit)
                   * Password Key
                   */
#if defined(OPENSSL_EXTRA) && !defined(NO_DES3) && !defined(NO_FILESYSTEM) && \
    !defined(NO_ASN) && !defined(NO_PWDBASED) && !defined(NO_RSA) && \
    !defined(NO_SHA) && defined(HAVE_PKCS12) && !defined(NO_BIO)
    byte buf[6000];
    char file[] = "./certs/test-servercert.p12";
    char order[] = "./certs/ecc-rsa-server.p12";
#ifdef WC_RC2
    char rc2p12[] = "./certs/test-servercert-rc2.p12";
#endif
    char pass[] = "a password";
    const char goodPsw[] = "wolfSSL test";
    const char badPsw[] = "bad";
#ifdef HAVE_ECC
    WOLFSSL_X509_NAME *subject = NULL;
    WOLFSSL_X509      *x509 = NULL;
#endif
    XFILE f = XBADFILE;
    int  bytes, ret, goodPswLen, badPswLen;
    WOLFSSL_BIO      *bio = NULL;
    WOLFSSL_EVP_PKEY *pkey = NULL;
    WC_PKCS12        *pkcs12 = NULL;
    WC_PKCS12        *pkcs12_2 = NULL;
    WOLFSSL_X509     *cert = NULL;
    WOLFSSL_X509     *tmp = NULL;
    WOLF_STACK_OF(WOLFSSL_X509) *ca = NULL;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)
    WOLFSSL_CTX      *ctx = NULL;
    WOLFSSL          *ssl = NULL;
    WOLF_STACK_OF(WOLFSSL_X509) *tmp_ca = NULL;
#endif

    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    goodPswLen = (int)XSTRLEN(goodPsw);
    badPswLen = (int)XSTRLEN(badPsw);

    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));

    ExpectIntEQ(BIO_write(bio, buf, bytes), bytes); /* d2i consumes BIO */
    ExpectNotNull(d2i_PKCS12_bio(bio, &pkcs12));
    ExpectNotNull(pkcs12);
    BIO_free(bio);
    bio = NULL;

    /* check verify MAC directly */
    ExpectIntEQ(ret = PKCS12_verify_mac(pkcs12, goodPsw, goodPswLen), 1);

    /* check verify MAC fail case directly */
    ExpectIntEQ(ret = PKCS12_verify_mac(pkcs12, badPsw, badPswLen), 0);

    /* check verify MAC fail case */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL), 0);
    ExpectNull(pkey);
    ExpectNull(cert);

    /* check parse with no extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;

    /* check parse with extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)

    /* Check that SSL_CTX_set0_chain correctly sets the certChain buffer */
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#if !defined(NO_WOLFSSL_CLIENT) && defined(SESSION_CERTS)
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
    /* Copy stack structure */
    ExpectNotNull(tmp_ca = X509_chain_up_ref(ca));
    ExpectIntEQ(SSL_CTX_set0_chain(ctx, tmp_ca), 1);
    /* CTX now owns the tmp_ca stack structure */
    tmp_ca = NULL;
    ExpectIntEQ(wolfSSL_CTX_get_extra_chain_certs(ctx, &tmp_ca), 1);
    ExpectNotNull(tmp_ca);
    ExpectIntEQ(sk_X509_num(tmp_ca), sk_X509_num(ca));
    /* Check that the main cert is also set */
    ExpectNotNull(SSL_CTX_get0_certificate(ctx));
    ExpectNotNull(ssl = SSL_new(ctx));
    ExpectNotNull(SSL_get_certificate(ssl));
    SSL_free(ssl);
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
    /* should be 2 other certs on stack */
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNull(sk_X509_pop(ca));

    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, X509_free);
    ca = NULL;

    /* check PKCS12_create */
    ExpectNull(PKCS12_create(pass, NULL, NULL, NULL, NULL, -1, -1, -1, -1,0));
    ExpectIntEQ(PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
            SSL_SUCCESS);
    ExpectNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, ca,
                    -1, -1, 100, -1, 0)));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    ExpectIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;
    ExpectNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, ca,
             NID_pbe_WithSHA1And3_Key_TripleDES_CBC,
             NID_pbe_WithSHA1And3_Key_TripleDES_CBC,
             2000, 1, 0)));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    /* convert to DER then back and parse */
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(i2d_PKCS12_bio(bio, pkcs12_2), SSL_SUCCESS);
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;

    ExpectNotNull(pkcs12_2 = d2i_PKCS12_bio(bio, NULL));
    BIO_free(bio);
    bio = NULL;
    ExpectIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);

    /* should be 2 other certs on stack */
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNull(sk_X509_pop(ca));


#ifndef NO_RC4
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;
    ExpectNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, NULL,
             NID_pbe_WithSHA1And128BitRC4,
             NID_pbe_WithSHA1And128BitRC4,
             2000, 1, 0)));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    ExpectIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);

#endif /* NO_RC4 */

    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    PKCS12_free(pkcs12);
    pkcs12 = NULL;
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

#ifdef HAVE_ECC
    /* test order of parsing */
    ExpectTrue((f = XFOPEN(order, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf((void*)buf, bytes));
    ExpectNotNull(pkcs12 = d2i_PKCS12_bio(bio, NULL));
    ExpectIntEQ((ret = PKCS12_parse(pkcs12, "", &pkey, &cert, &ca)),
            WOLFSSL_SUCCESS);

    /* check use of pkey after parse */
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#if !defined(NO_WOLFSSL_CLIENT) && defined(SESSION_CERTS)
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
    ExpectIntEQ(SSL_CTX_use_PrivateKey(ctx, pkey), WOLFSSL_SUCCESS);
    SSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif

    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

    /* compare subject lines of certificates */
    ExpectNotNull(subject = wolfSSL_X509_get_subject_name(cert));
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(eccRsaCertFile,
                SSL_FILETYPE_PEM));
    ExpectIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    X509_free(x509);
    x509 = NULL;

    /* test expected fail case */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(eccCertFile,
                SSL_FILETYPE_PEM));
    ExpectIntNE(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    X509_free(x509);
    x509 = NULL;
    X509_free(cert);
    cert = NULL;

    /* get subject line from ca stack */
    ExpectNotNull(cert = sk_X509_pop(ca));
    ExpectNotNull(subject = wolfSSL_X509_get_subject_name(cert));

    /* compare subject from certificate in ca to expected */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(eccCertFile,
                SSL_FILETYPE_PEM));
    ExpectIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(x509);
    x509 = NULL;
    X509_free(cert);
    cert = NULL;
    BIO_free(bio);
    bio = NULL;
    PKCS12_free(pkcs12);
    pkcs12 = NULL;
    sk_X509_pop_free(ca, NULL); /* TEST d2i_PKCS12_fp */
    ca = NULL;

    /* test order of parsing */
    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectNotNull(pkcs12 = d2i_PKCS12_fp(f, NULL));
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    /* check verify MAC fail case */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL), 0);
    ExpectNull(pkey);
    ExpectNull(cert);

    /* check parse with no extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;

    /* check parse with extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    PKCS12_free(pkcs12);
    pkcs12 = NULL;
#endif /* HAVE_ECC */

#ifdef WC_RC2
    /* test PKCS#12 with RC2 encryption */
    ExpectTrue((f = XFOPEN(rc2p12, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf((void*)buf, bytes));
    ExpectNotNull(pkcs12 = d2i_PKCS12_bio(bio, NULL));

    /* check verify MAC fail case */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL), 0);
    ExpectNull(pkey);
    ExpectNull(cert);

    /* check parse with not extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL),
        WOLFSSL_SUCCESS);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;

    /* check parse with extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
        WOLFSSL_SUCCESS);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

    wolfSSL_EVP_PKEY_free(pkey);
    wolfSSL_X509_free(cert);
    sk_X509_pop_free(ca, NULL);

    BIO_free(bio);
    bio = NULL;
    PKCS12_free(pkcs12);
    pkcs12 = NULL;
#endif /* WC_RC2 */

    /* Test i2d_PKCS12_bio */
    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectNotNull(pkcs12 = d2i_PKCS12_fp(f, NULL));
    if (f != XBADFILE)
        XFCLOSE(f);

    ExpectNotNull(bio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(ret = i2d_PKCS12_bio(bio, pkcs12), 1);

    ExpectIntEQ(ret = i2d_PKCS12_bio(NULL, pkcs12), 0);

    ExpectIntEQ(ret = i2d_PKCS12_bio(bio, NULL), 0);

    PKCS12_free(pkcs12);
    BIO_free(bio);

    (void)order;
#endif /* OPENSSL_EXTRA */
#endif /* HAVE_FIPS */
    return EXPECT_RESULT();
}


#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_DES3) && !defined(NO_PWDBASED) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_MD5)
    #define TEST_PKCS8_ENC
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) \
    && defined(HAVE_ECC) && defined(WOLFSSL_ENCRYPTED_KEYS)

/* used to keep track if FailTestCallback was called */
static int failTestCallbackCalled = 0;

static WC_INLINE int FailTestCallBack(char* passwd, int sz, int rw, void* userdata)
{
    (void)passwd;
    (void)sz;
    (void)rw;
    (void)userdata;

    /* mark called, test_wolfSSL_no_password_cb() will check and fail if set */
    failTestCallbackCalled = 1;

    return -1;
}
#endif

static int test_wolfSSL_no_password_cb(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) \
    && defined(HAVE_ECC) && defined(WOLFSSL_ENCRYPTED_KEYS)
    WOLFSSL_CTX* ctx = NULL;
    byte buff[FOURK_BUF];
    const char eccPkcs8PrivKeyDerFile[] = "./certs/ecc-privkeyPkcs8.der";
    const char eccPkcs8PrivKeyPemFile[] = "./certs/ecc-privkeyPkcs8.pem";
    XFILE f = XBADFILE;
    int bytes = 0;

#ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLS_client_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLS_server_method()));
#endif
    wolfSSL_CTX_set_default_passwd_cb(ctx, FailTestCallBack);

    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntLE(bytes, sizeof(buff));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    ExpectIntLE(bytes, sizeof(buff));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);

    /* Password callback should not be called by default */
    ExpectIntEQ(failTestCallbackCalled, 0);
#endif
    return EXPECT_RESULT();
}

#ifdef TEST_PKCS8_ENC
/* for PKCS8 test case */
static int PKCS8TestCallBack(char* passwd, int sz, int rw, void* userdata)
{
    int flag = 0;

    (void)rw;
    if (userdata != NULL) {
        flag = *((int*)userdata); /* user set data */
    }

    switch (flag) {
        case 1: /* flag set for specific WOLFSSL_CTX structure, note userdata
                 * can be anything the user wishes to be passed to the callback
                 * associated with the WOLFSSL_CTX */
            XSTRNCPY(passwd, "yassl123", sz);
            return 8;

        default:
            return BAD_FUNC_ARG;
    }
}
#endif /* TEST_PKCS8_ENC */

/* Testing functions dealing with PKCS8 */
static int test_wolfSSL_PKCS8(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    !defined(WOLFCRYPT_ONLY)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    byte buff[FOURK_BUF];
    byte der[FOURK_BUF];
    #ifndef NO_RSA
        const char serverKeyPkcs8PemFile[] = "./certs/server-keyPkcs8.pem";
        const char serverKeyPkcs8DerFile[] = "./certs/server-keyPkcs8.der";
    #endif
    const char eccPkcs8PrivKeyPemFile[] = "./certs/ecc-privkeyPkcs8.pem";
    #ifdef HAVE_ECC
        const char eccPkcs8PrivKeyDerFile[] = "./certs/ecc-privkeyPkcs8.der";
    #endif
    XFILE f = XBADFILE;
    int bytes = 0;
    WOLFSSL_CTX* ctx = NULL;
#if defined(HAVE_ECC) && !defined(NO_CODING)
    int ret;
    ecc_key key;
    word32 x = 0;
#endif
#ifdef TEST_PKCS8_ENC
    #if !defined(NO_RSA) && !defined(NO_SHA)
        const char serverKeyPkcs8EncPemFile[] = "./certs/server-keyPkcs8Enc.pem";
        const char serverKeyPkcs8EncDerFile[] = "./certs/server-keyPkcs8Enc.der";
    #endif
    #if defined(HAVE_ECC) && !defined(NO_SHA)
        const char eccPkcs8EncPrivKeyPemFile[] = "./certs/ecc-keyPkcs8Enc.pem";
        const char eccPkcs8EncPrivKeyDerFile[] = "./certs/ecc-keyPkcs8Enc.der";
    #endif
    int flag;
#endif

    (void)der;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif

#ifdef TEST_PKCS8_ENC
    wolfSSL_CTX_set_default_passwd_cb(ctx, PKCS8TestCallBack);
    wolfSSL_CTX_set_default_passwd_cb_userdata(ctx, (void*)&flag);
    flag = 1; /* used by password callback as return code */

    #if !defined(NO_RSA) && !defined(NO_SHA)
    /* test loading PEM PKCS8 encrypted file */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8EncPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* decrypt PKCS8 PEM to key in DER format with not using WOLFSSL_CTX */
    ExpectIntGT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "yassl123"), 0);

    /* test that error value is returned with a bad password */
    ExpectIntLT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "bad"), 0);

    /* test loading PEM PKCS8 encrypted file */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8EncDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    flag = 1; /* used by password callback as return code */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    #endif /* !NO_RSA && !NO_SHA */

    #if defined(HAVE_ECC) && !defined(NO_SHA)
    /* test loading PEM PKCS8 encrypted ECC Key file */
    ExpectTrue((f = XFOPEN(eccPkcs8EncPrivKeyPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    flag = 1; /* used by password callback as return code */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* decrypt PKCS8 PEM to key in DER format with not using WOLFSSL_CTX */
    ExpectIntGT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "yassl123"), 0);

    /* test that error value is returned with a bad password */
    ExpectIntLT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "bad"), 0);

    /* test loading DER PKCS8 encrypted ECC Key file */
    ExpectTrue((f = XFOPEN(eccPkcs8EncPrivKeyDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    flag = 1; /* used by password callback as return code */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* leave flag as "okay" */
    flag = 1;
    #endif /* HAVE_ECC && !NO_SHA */
#endif /* TEST_PKCS8_ENC */


#ifndef NO_RSA
    /* test loading ASN.1 (DER) PKCS8 private key file (not encrypted) */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8DerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* test loading PEM PKCS8 private key file (not encrypted) */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8PemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
#endif /* !NO_RSA */

    /* Test PKCS8 PEM ECC key no crypt */
    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
#ifdef HAVE_ECC
    /* Test PKCS8 PEM ECC key no crypt */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

#ifndef NO_CODING
    /* decrypt PKCS8 PEM to key in DER format */
    ExpectIntGT((bytes = wc_KeyPemToDer(buff, bytes, der,
        (word32)sizeof(der), NULL)), 0);
    ret = wc_ecc_init(&key);
    if (ret == 0) {
        ret = wc_EccPrivateKeyDecode(der, &x, &key, bytes);
        wc_ecc_free(&key);
    }
    ExpectIntEQ(ret, 0);
#endif

    /* Test PKCS8 DER ECC key no crypt */
    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    /* Test using a PKCS8 ECC PEM */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#else
    /* if HAVE_ECC is not defined then BEGIN EC PRIVATE KEY is not found */
    ExpectIntEQ((bytes = wc_KeyPemToDer(buff, bytes, der,
        (word32)sizeof(der), NULL)), ASN_NO_PEM_HEADER);
#endif /* HAVE_ECC */

    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* !NO_FILESYSTEM && !NO_ASN && HAVE_PKCS8 */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PKCS8_ED25519(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && defined(HAVE_PKCS8) && defined(HAVE_AES_CBC) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && defined(HAVE_ED25519) && \
    defined(HAVE_ED25519_KEY_IMPORT)
    const byte encPrivKey[] = \
    "-----BEGIN ENCRYPTED PRIVATE KEY-----\n"
    "MIGbMFcGCSqGSIb3DQEFDTBKMCkGCSqGSIb3DQEFDDAcBAheCGLmWGh7+AICCAAw\n"
    "DAYIKoZIhvcNAgkFADAdBglghkgBZQMEASoEEC4L5P6GappsTyhOOoQfvh8EQJMX\n"
    "OAdlsYKCOcFo4djg6AI1lRdeBRwVFWkha7gBdoCJOzS8wDvTbYcJMPvANu5ft3nl\n"
    "2L9W4v7swXkV+X+a1ww=\n"
    "-----END ENCRYPTED PRIVATE KEY-----\n";
    const char password[] = "abcdefghijklmnopqrstuvwxyz";
    byte der[FOURK_BUF];
    WOLFSSL_CTX* ctx = NULL;
    int bytes;

    XMEMSET(der, 0, sizeof(der));
    ExpectIntGT((bytes = wc_KeyPemToDer(encPrivKey, sizeof(encPrivKey), der,
        (word32)sizeof(der), password)), 0);
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, bytes,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PKCS8_ED448(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && defined(HAVE_PKCS8) && defined(HAVE_AES_CBC) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && defined(HAVE_ED448) && \
    defined(HAVE_ED448_KEY_IMPORT)
    const byte encPrivKey[] = \
    "-----BEGIN ENCRYPTED PRIVATE KEY-----\n"
    "MIGrMFcGCSqGSIb3DQEFDTBKMCkGCSqGSIb3DQEFDDAcBAjSbZKnG4EPggICCAAw\n"
    "DAYIKoZIhvcNAgkFADAdBglghkgBZQMEASoEEFvCFWBBHBlJBsYleBJlJWcEUNC7\n"
    "Tf5pZviT5Btar4D/MNg6BsQHSDf5KW4ix871EsgDY2Zz+euaoWspiMntz7gU+PQu\n"
    "T/JJcbD2Ly8BbE3l5WHMifAQqNLxJBfXrHkfYtAo\n"
    "-----END ENCRYPTED PRIVATE KEY-----\n";
    const char password[] = "abcdefghijklmnopqrstuvwxyz";
    byte der[FOURK_BUF];
    WOLFSSL_CTX* ctx = NULL;
    int bytes;

    XMEMSET(der, 0, sizeof(der));
    ExpectIntGT((bytes = wc_KeyPemToDer(encPrivKey, sizeof(encPrivKey), der,
        (word32)sizeof(der), password)), 0);
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, bytes,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return EXPECT_RESULT();
}

/* Testing functions dealing with PKCS5 */
static int test_wolfSSL_PKCS5(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SHA) && !defined(NO_PWDBASED)
#ifdef HAVE_FIPS /* Password minimum length is 14 (112-bit) in FIPS MODE */
    const char* passwd = "myfipsPa$$W0rd";
#else
    const char *passwd = "pass1234";
#endif
    const unsigned char *salt = (unsigned char *)"salt1234";
    unsigned char *out = (unsigned char *)XMALLOC(WC_SHA_DIGEST_SIZE, NULL,
                                                  DYNAMIC_TYPE_TMP_BUFFER);
    int ret = 0;

    ExpectNotNull(out);
    ExpectIntEQ(ret = PKCS5_PBKDF2_HMAC_SHA1(passwd,(int)XSTRLEN(passwd), salt,
        (int)XSTRLEN((const char *) salt), 10, WC_SHA_DIGEST_SIZE,out),
        WOLFSSL_SUCCESS);

#ifdef WOLFSSL_SHA512
    ExpectIntEQ(ret = PKCS5_PBKDF2_HMAC(passwd,(int)XSTRLEN(passwd), salt,
        (int)XSTRLEN((const char *) salt), 10, wolfSSL_EVP_sha512(),
        WC_SHA_DIGEST_SIZE, out), SSL_SUCCESS);
#endif

    XFREE(out, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SHA) */
    return EXPECT_RESULT();
}

/* test parsing URI from certificate */
static int test_wolfSSL_URI(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) \
    && (defined(KEEP_PEER_CERT) || defined(SESSION_CERTS) || \
    defined(OPENSSL_EXTRA))
    WOLFSSL_X509* x509 = NULL;
    const char uri[] = "./certs/client-uri-cert.pem";
    const char urn[] = "./certs/client-absolute-urn.pem";
    const char badUri[] = "./certs/client-relative-uri.pem";

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(uri,
        WOLFSSL_FILETYPE_PEM));
    wolfSSL_FreeX509(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(urn,
        WOLFSSL_FILETYPE_PEM));
    wolfSSL_FreeX509(x509);
    x509 = NULL;

#if !defined(IGNORE_NAME_CONSTRAINTS) && !defined(WOLFSSL_NO_ASN_STRICT) \
    && !defined(WOLFSSL_FPKI)
    ExpectNull(x509 = wolfSSL_X509_load_certificate_file(badUri,
        WOLFSSL_FILETYPE_PEM));
#else
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(badUri,
        WOLFSSL_FILETYPE_PEM));
#endif
    wolfSSL_FreeX509(x509);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_TBS(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) \
    && defined(OPENSSL_EXTRA)
    WOLFSSL_X509* x509 = NULL;
    const unsigned char* tbs;
    int tbsSz;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(caCertFile,
        WOLFSSL_FILETYPE_PEM));

    ExpectNull(tbs = wolfSSL_X509_get_tbs(NULL, &tbsSz));
    ExpectNull(tbs = wolfSSL_X509_get_tbs(x509, NULL));
    ExpectNotNull(tbs = wolfSSL_X509_get_tbs(x509, &tbsSz));
    ExpectIntEQ(tbsSz, 1003);

    wolfSSL_FreeX509(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_verify(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) && \
    defined(OPENSSL_EXTRA)
    WOLFSSL_X509* ca = NULL;
    WOLFSSL_X509* serv = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;
    unsigned char buf[2048];
    const unsigned char* pt = NULL;
    int bufSz;

    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(caCertFile,
        WOLFSSL_FILETYPE_PEM));

    ExpectIntNE(wolfSSL_X509_get_pubkey_buffer(NULL, buf, &bufSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_get_pubkey_buffer(ca, NULL, &bufSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(bufSz, 294);

    bufSz = 2048;
    ExpectIntEQ(wolfSSL_X509_get_pubkey_buffer(ca, buf, &bufSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_get_pubkey_type(NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(wolfSSL_X509_get_pubkey_type(ca), RSAk);


    ExpectNotNull(serv = wolfSSL_X509_load_certificate_file(svrCertFile,
        WOLFSSL_FILETYPE_PEM));

    /* success case */
    pt = buf;
    ExpectNotNull(pkey = wolfSSL_d2i_PUBKEY(NULL, &pt, bufSz));

    ExpectIntEQ(i2d_PUBKEY(pkey, NULL), bufSz);

    ExpectIntEQ(wolfSSL_X509_verify(serv, pkey), WOLFSSL_SUCCESS);
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;

    /* fail case */
    bufSz = 2048;
    ExpectIntEQ(wolfSSL_X509_get_pubkey_buffer(serv, buf, &bufSz),
        WOLFSSL_SUCCESS);
    pt = buf;
    ExpectNotNull(pkey = wolfSSL_d2i_PUBKEY(NULL, &pt, bufSz));
    ExpectIntEQ(wolfSSL_X509_verify(serv, pkey), WOLFSSL_FAILURE);

    ExpectIntEQ(wolfSSL_X509_verify(NULL, pkey), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_X509_verify(serv, NULL), WOLFSSL_FATAL_ERROR);
    wolfSSL_EVP_PKEY_free(pkey);

    wolfSSL_FreeX509(ca);
    wolfSSL_FreeX509(serv);
#endif
    return EXPECT_RESULT();
}


#if !defined(NO_DH) && !defined(NO_AES) && defined(WOLFSSL_CERT_GEN) && \
         defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
         defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
/* create certificate with version 2 */
static int test_set_x509_badversion(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    WOLFSSL_X509 *x509 = NULL, *x509v2 = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL, *pub = NULL;
    unsigned char *der = NULL, *key = NULL, *pt;
    char *header = NULL, *name = NULL;
    int derSz;
    long keySz;
    XFILE fp = XBADFILE;
    WOLFSSL_ASN1_TIME *notBefore = NULL, *notAfter = NULL;
    time_t t;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                WOLFSSL_FILETYPE_PEM));

    ExpectTrue((fp = XFOPEN(cliKeyFile, "rb")) != XBADFILE);
    ExpectIntEQ(wolfSSL_PEM_read(fp, &name, &header, &key, &keySz),
            WOLFSSL_SUCCESS);
    if (fp != XBADFILE)
        XFCLOSE(fp);
    pt = key;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, keySz));


    /* create the version 2 certificate */
    ExpectNotNull(x509v2 = X509_new());
    ExpectIntEQ(wolfSSL_X509_set_version(x509v2, 1), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509v2,
                wolfSSL_X509_get_subject_name(x509)), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509v2,
                wolfSSL_X509_get_issuer_name(x509)), WOLFSSL_SUCCESS);
    ExpectNotNull(pub = wolfSSL_X509_get_pubkey(x509));
    ExpectIntEQ(X509_set_pubkey(x509v2, pub), WOLFSSL_SUCCESS);

    t = time(NULL);
    ExpectNotNull(notBefore = wolfSSL_ASN1_TIME_adj(NULL, t, 0, 0));
    ExpectNotNull(notAfter = wolfSSL_ASN1_TIME_adj(NULL, t, 365, 0));
    ExpectTrue(wolfSSL_X509_set_notBefore(x509v2, notBefore));
    ExpectTrue(wolfSSL_X509_set_notAfter(x509v2, notAfter));

    ExpectIntGT(wolfSSL_X509_sign(x509v2, priv, EVP_sha256()), 0);
    derSz = wolfSSL_i2d_X509(x509v2, &der);
    ExpectIntGT(derSz, 0);
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx, der, derSz,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    /* TODO: Replace with API call */
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
    XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(name, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(header, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(x509v2);
    wolfSSL_EVP_PKEY_free(priv);
    wolfSSL_EVP_PKEY_free(pub);
    wolfSSL_ASN1_TIME_free(notBefore);
    wolfSSL_ASN1_TIME_free(notAfter);

    return EXPECT_RESULT();
}


/* override certificate version error */
static int test_override_x509(int preverify, WOLFSSL_X509_STORE_CTX* store)
{
    EXPECT_DECLS;
#ifndef OPENSSL_COMPATIBLE_DEFAULTS
    ExpectIntEQ(store->error, ASN_VERSION_E);
#else
    ExpectIntEQ(store->error, 0);
#endif
    ExpectIntEQ((int)wolfSSL_X509_get_version(store->current_cert), 1);
    (void)preverify;
    return EXPECT_RESULT() == TEST_SUCCESS;
}


/* set verify callback that will override bad certificate version */
static int test_set_override_x509(WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, test_override_x509);
    return TEST_SUCCESS;
}
#endif


static int test_wolfSSL_X509_TLS_version_test_1(void)
{
    EXPECT_DECLS;
#if !defined(NO_DH) && !defined(NO_AES) && defined(WOLFSSL_CERT_GEN) && \
         defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
         defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    test_ssl_cbf func_cb_client;
    test_ssl_cbf func_cb_server;

    /* test server rejects a client certificate that is not version 3 */
    XMEMSET(&func_cb_client, 0, sizeof(func_cb_client));
    XMEMSET(&func_cb_server, 0, sizeof(func_cb_server));

    func_cb_client.ctx_ready = &test_set_x509_badversion;
#ifndef WOLFSSL_NO_TLS12
    func_cb_client.method = wolfTLSv1_2_client_method;
#else
    func_cb_client.method = wolfTLSv1_3_client_method;
#endif

#ifndef WOLFSSL_NO_TLS12
    func_cb_server.method = wolfTLSv1_2_server_method;
#else
    func_cb_server.method = wolfTLSv1_3_server_method;
#endif

#ifndef OPENSSL_COMPATIBLE_DEFAULTS
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), TEST_FAIL);
#else
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), TEST_SUCCESS);
#endif
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_TLS_version_test_2(void)
{
    EXPECT_DECLS;
#if !defined(NO_DH) && !defined(NO_AES) && defined(WOLFSSL_CERT_GEN) && \
         defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
         defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    test_ssl_cbf func_cb_client;
    test_ssl_cbf func_cb_server;

    XMEMSET(&func_cb_client, 0, sizeof(func_cb_client));
    XMEMSET(&func_cb_server, 0, sizeof(func_cb_server));

    func_cb_client.ctx_ready = &test_set_x509_badversion;
    func_cb_server.ctx_ready = &test_set_override_x509;
#ifndef WOLFSSL_NO_TLS12
    func_cb_client.method = wolfTLSv1_2_client_method;
#else
    func_cb_client.method = wolfTLSv1_3_client_method;
#endif

#ifndef WOLFSSL_NO_TLS12
    func_cb_server.method = wolfTLSv1_2_server_method;
#else
    func_cb_server.method = wolfTLSv1_3_server_method;
#endif

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), TEST_SUCCESS);
#endif

    return EXPECT_RESULT();
}

/* Testing function  wolfSSL_CTX_SetMinVersion; sets the minimum downgrade
 * version allowed.
 * POST: 1 on success.
 */
static int test_wolfSSL_CTX_SetMinVersion(void)
{
    int                     res = TEST_SKIPPED;
#ifndef NO_WOLFSSL_CLIENT
    int                     failFlag = WOLFSSL_SUCCESS;
    WOLFSSL_CTX*            ctx;
    int                     itr;

    #ifndef NO_OLD_TLS
        const int versions[]  = {
                            #ifdef WOLFSSL_ALLOW_TLSV10
                                  WOLFSSL_TLSV1,
                            #endif
                                  WOLFSSL_TLSV1_1,
                                  WOLFSSL_TLSV1_2 };
    #elif !defined(WOLFSSL_NO_TLS12)
        const int versions[]  = { WOLFSSL_TLSV1_2 };
    #elif defined(WOLFSSL_TLS13)
        const int versions[]  = { WOLFSSL_TLSV1_3 };
    #else
        const int versions[0];
    #endif

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());

    for (itr = 0; itr < (int)(sizeof(versions)/sizeof(int)); itr++) {
        if (wolfSSL_CTX_SetMinVersion(ctx, *(versions + itr))
                != WOLFSSL_SUCCESS) {
            failFlag = WOLFSSL_FAILURE;
        }
    }

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(failFlag == WOLFSSL_SUCCESS);
#endif
    return res;

} /* END test_wolfSSL_CTX_SetMinVersion */


/*----------------------------------------------------------------------------*
 | OCSP Stapling
 *----------------------------------------------------------------------------*/


/* Testing wolfSSL_UseOCSPStapling function. OCSP stapling eliminates the need
 * need to contact the CA, lowering the cost of cert revocation checking.
 * PRE: HAVE_OCSP and HAVE_CERTIFICATE_STATUS_REQUEST
 * POST: 1 returned for success.
 */
static int test_wolfSSL_UseOCSPStapling(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) && defined(HAVE_OCSP) && \
        !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*    ctx = NULL;
    WOLFSSL*        ssl = NULL;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectIntEQ(wolfSSL_UseOCSPStapling(NULL, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), BAD_FUNC_ARG);
#ifndef NO_WOLFSSL_CLIENT
    ExpectIntEQ(wolfSSL_UseOCSPStapling(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), 1);
#else
    ExpectIntEQ(wolfSSL_UseOCSPStapling(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), BAD_FUNC_ARG);
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
} /* END test_wolfSSL_UseOCSPStapling */


/* Testing OCSP stapling version 2, wolfSSL_UseOCSPStaplingV2 function. OCSP
 * stapling eliminates the need to contact the CA and lowers cert revocation
 * check.
 * PRE: HAVE_CERTIFICATE_STATUS_REQUEST_V2 and HAVE_OCSP defined.
 */
static int test_wolfSSL_UseOCSPStaplingV2(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2) && defined(HAVE_OCSP) && \
        !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*        ctx = NULL;
    WOLFSSL*            ssl = NULL;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectIntEQ(wolfSSL_UseOCSPStaplingV2(NULL, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), BAD_FUNC_ARG);
#ifndef NO_WOLFSSL_CLIENT
    ExpectIntEQ(wolfSSL_UseOCSPStaplingV2(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), 1);
#else
    ExpectIntEQ(wolfSSL_UseOCSPStaplingV2(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), BAD_FUNC_ARG);
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();

} /* END test_wolfSSL_UseOCSPStaplingV2 */

/*----------------------------------------------------------------------------*
 | Multicast Tests
 *----------------------------------------------------------------------------*/
static int test_wolfSSL_mcast(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_MULTICAST) && \
    (defined(WOLFSSL_TLS13) || defined(WOLFSSL_SNIFFER))
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    byte preMasterSecret[512];
    byte clientRandom[32];
    byte serverRandom[32];
    byte suite[2] = {0, 0xfe};  /* WDM_WITH_NULL_SHA256 */
    byte buf[256];
    word16 newId;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_client_method()));

    ExpectIntEQ(wolfSSL_CTX_mcast_set_member_id(ctx, 0), WOLFSSL_SUCCESS);

    ExpectNotNull(ssl = wolfSSL_new(ctx));

    XMEMSET(preMasterSecret, 0x23, sizeof(preMasterSecret));
    XMEMSET(clientRandom, 0xA5, sizeof(clientRandom));
    XMEMSET(serverRandom, 0x5A, sizeof(serverRandom));
    ExpectIntEQ(wolfSSL_set_secret(ssl, 23, preMasterSecret,
        sizeof(preMasterSecret), clientRandom, serverRandom, suite),
        WOLFSSL_SUCCESS);

    ExpectIntLE(wolfSSL_mcast_read(ssl, &newId, buf, sizeof(buf)), 0);
    ExpectIntLE(newId, 100);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* WOLFSSL_DTLS && WOLFSSL_MULTICAST && (WOLFSSL_TLS13 ||
        * WOLFSSL_SNIFFER) */
    return EXPECT_RESULT();
}


/*----------------------------------------------------------------------------*
 |  Wolfcrypt
 *----------------------------------------------------------------------------*/

/*
 * Unit test for the wc_InitBlake2b()
 */
static int test_wc_InitBlake2b(void)
{
    EXPECT_DECLS;
#ifdef HAVE_BLAKE2
    Blake2b blake;

    /* Test good arg. */
    ExpectIntEQ(wc_InitBlake2b(&blake, 64), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitBlake2b(NULL, 64), BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitBlake2b(NULL, 128), BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitBlake2b(&blake, 128), BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitBlake2b(NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitBlake2b(&blake, 0), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
}     /* END test_wc_InitBlake2b*/

/*
 * Unit test for the wc_InitBlake2b_WithKey()
 */
static int test_wc_InitBlake2b_WithKey(void)
{
    EXPECT_DECLS;
#ifdef HAVE_BLAKE2
    Blake2b     blake;
    word32      digestSz = BLAKE2B_KEYBYTES;
    byte        key[BLAKE2B_KEYBYTES];
    word32      keylen = BLAKE2B_KEYBYTES;

    XMEMSET(key, 0, sizeof(key));

    /* Test good arg. */
    ExpectIntEQ(wc_InitBlake2b_WithKey(&blake, digestSz, key, keylen), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_InitBlake2b_WithKey(NULL, digestSz, key, keylen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitBlake2b_WithKey(&blake, digestSz, key, 256),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitBlake2b_WithKey(&blake, digestSz, NULL, keylen), 0);
#endif
    return EXPECT_RESULT();
}     /* END wc_InitBlake2b_WithKey*/

/*
 * Unit test for the wc_InitBlake2s_WithKey()
 */
static int test_wc_InitBlake2s_WithKey(void)
{
    EXPECT_DECLS;
#ifdef HAVE_BLAKE2S
    Blake2s     blake;
    word32      digestSz = BLAKE2S_KEYBYTES;
    byte        *key = (byte*)"01234567890123456789012345678901";
    word32      keylen = BLAKE2S_KEYBYTES;

    /* Test good arg. */
    ExpectIntEQ(wc_InitBlake2s_WithKey(&blake, digestSz, key, keylen), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_InitBlake2s_WithKey(NULL, digestSz, key, keylen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitBlake2s_WithKey(&blake, digestSz, key, 256),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitBlake2s_WithKey(&blake, digestSz, NULL, keylen), 0);
#endif
    return EXPECT_RESULT();
}     /* END wc_InitBlake2s_WithKey*/

/*
 * Unit test for the wc_InitMd5()
 */
static int test_wc_InitMd5(void)
{
    EXPECT_DECLS;
#ifndef NO_MD5
    wc_Md5 md5;

    /* Test good arg. */
    ExpectIntEQ(wc_InitMd5(&md5), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitMd5(NULL), BAD_FUNC_ARG);

    wc_Md5Free(&md5);
#endif
    return EXPECT_RESULT();
}     /* END test_wc_InitMd5 */


/*
 * Testing wc_UpdateMd5()
 */
static int test_wc_Md5Update(void)
{
    EXPECT_DECLS;
#ifndef NO_MD5
    wc_Md5 md5;
    byte hash[WC_MD5_DIGEST_SIZE];
    testVector a, b, c;

    ExpectIntEQ(wc_InitMd5(&md5), 0);

    /* Input */
    a.input = "a";
    a.inLen = XSTRLEN(a.input);
    ExpectIntEQ(wc_Md5Update(&md5, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_Md5Final(&md5, hash), 0);

    /* Update input. */
    a.input = "abc";
    a.output = "\x90\x01\x50\x98\x3c\xd2\x4f\xb0\xd6\x96\x3f\x7d\x28\xe1\x7f"
               "\x72";
    a.inLen = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);
    ExpectIntEQ(wc_Md5Update(&md5, (byte*) a.input, (word32) a.inLen), 0);
    ExpectIntEQ(wc_Md5Final(&md5, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_MD5_DIGEST_SIZE), 0);

    /* Pass in bad values. */
    b.input = NULL;
    b.inLen = 0;
    ExpectIntEQ(wc_Md5Update(&md5, (byte*)b.input, (word32)b.inLen), 0);
    c.input = NULL;
    c.inLen = WC_MD5_DIGEST_SIZE;
    ExpectIntEQ(wc_Md5Update(&md5, (byte*)c.input, (word32)c.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Md5Update(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);

    wc_Md5Free(&md5);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Md5Update()  */


/*
 *  Unit test on wc_Md5Final() in wolfcrypt/src/md5.c
 */
static int test_wc_Md5Final(void)
{
    EXPECT_DECLS;
#ifndef NO_MD5
    /* Instantiate */
    wc_Md5 md5;
    byte* hash_test[3];
    byte hash1[WC_MD5_DIGEST_SIZE];
    byte hash2[2*WC_MD5_DIGEST_SIZE];
    byte hash3[5*WC_MD5_DIGEST_SIZE];
    int times, i;

    /* Initialize */
    ExpectIntEQ(wc_InitMd5(&md5), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test)/sizeof(byte*);
    for (i = 0; i < times; i++) {
        ExpectIntEQ(wc_Md5Final(&md5, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_Md5Final(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Md5Final(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Md5Final(&md5, NULL), BAD_FUNC_ARG);

    wc_Md5Free(&md5);
#endif
    return EXPECT_RESULT();
}

/*
 * Unit test for the wc_InitSha()
 */
static int test_wc_InitSha(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA
    wc_Sha sha;

    /* Test good arg. */
    ExpectIntEQ(wc_InitSha(&sha), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitSha(NULL), BAD_FUNC_ARG);

    wc_ShaFree(&sha);
#endif
    return EXPECT_RESULT();
} /* END test_wc_InitSha */

/*
 *  Tesing wc_ShaUpdate()
 */
static int test_wc_ShaUpdate(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA
    wc_Sha sha;
    byte hash[WC_SHA_DIGEST_SIZE];
    testVector a, b, c;

    ExpectIntEQ(wc_InitSha(&sha), 0);

    /* Input. */
    a.input = "a";
    a.inLen = XSTRLEN(a.input);

    ExpectIntEQ(wc_ShaUpdate(&sha, NULL, 0), 0);
    ExpectIntEQ(wc_ShaUpdate(&sha, (byte*)a.input, 0), 0);
    ExpectIntEQ(wc_ShaUpdate(&sha, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_ShaFinal(&sha, hash), 0);

    /* Update input. */
    a.input = "abc";
    a.output = "\xA9\x99\x3E\x36\x47\x06\x81\x6A\xBA\x3E\x25\x71\x78\x50\xC2"
               "\x6C\x9C\xD0\xD8\x9D";
    a.inLen = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    ExpectIntEQ(wc_ShaUpdate(&sha, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_ShaFinal(&sha, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA_DIGEST_SIZE), 0);

    /* Try passing in bad values. */
    b.input = NULL;
    b.inLen = 0;
    ExpectIntEQ(wc_ShaUpdate(&sha, (byte*)b.input, (word32)b.inLen), 0);
    c.input = NULL;
    c.inLen = WC_SHA_DIGEST_SIZE;
    ExpectIntEQ(wc_ShaUpdate(&sha, (byte*)c.input, (word32)c.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ShaUpdate(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);

    wc_ShaFree(&sha);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ShaUpdate() */


/*
 * Unit test on wc_ShaFinal
 */
static int test_wc_ShaFinal(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA
    wc_Sha sha;
    byte* hash_test[3];
    byte hash1[WC_SHA_DIGEST_SIZE];
    byte hash2[2*WC_SHA_DIGEST_SIZE];
    byte hash3[5*WC_SHA_DIGEST_SIZE];
    int times, i;

    /* Initialize*/
    ExpectIntEQ(wc_InitSha(&sha), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test)/sizeof(byte*);
    for (i = 0; i < times; i++) {
        ExpectIntEQ(wc_ShaFinal(&sha, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_ShaFinal(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ShaFinal(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ShaFinal(&sha, NULL), BAD_FUNC_ARG);

    wc_ShaFree(&sha);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ShaFinal */


/*
 * Unit test for wc_InitSha256()
 */
static int test_wc_InitSha256(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA256
    wc_Sha256 sha256;

    /* Test good arg. */
    ExpectIntEQ(wc_InitSha256(&sha256), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitSha256(NULL), BAD_FUNC_ARG);

    wc_Sha256Free(&sha256);
#endif
    return EXPECT_RESULT();
} /* END test_wc_InitSha256 */


/*
 * Unit test for wc_Sha256Update()
 */
static int test_wc_Sha256Update(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA256
    wc_Sha256 sha256;
    byte hash[WC_SHA256_DIGEST_SIZE];
    byte hash_unaligned[WC_SHA256_DIGEST_SIZE+1];
    testVector a, b, c;

    ExpectIntEQ(wc_InitSha256(&sha256), 0);

    /*  Input. */
    a.input = "a";
    a.inLen = XSTRLEN(a.input);
    ExpectIntEQ(wc_Sha256Update(&sha256, NULL, 0), 0);
    ExpectIntEQ(wc_Sha256Update(&sha256, (byte*)a.input, 0), 0);
    ExpectIntEQ(wc_Sha256Update(&sha256, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_Sha256Final(&sha256, hash), 0);

    /* Update input. */
    a.input = "abc";
    a.output = "\xBA\x78\x16\xBF\x8F\x01\xCF\xEA\x41\x41\x40\xDE\x5D\xAE\x22"
               "\x23\xB0\x03\x61\xA3\x96\x17\x7A\x9C\xB4\x10\xFF\x61\xF2\x00"
               "\x15\xAD";
    a.inLen = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);
    ExpectIntEQ(wc_Sha256Update(&sha256, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_Sha256Final(&sha256, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA256_DIGEST_SIZE), 0);

    /* Unaligned check. */
    ExpectIntEQ(wc_Sha256Update(&sha256, (byte*)a.input+1, (word32)a.inLen-1),
        0);
    ExpectIntEQ(wc_Sha256Final(&sha256, hash_unaligned + 1), 0);

    /* Try passing in bad values */
    b.input = NULL;
    b.inLen = 0;
    ExpectIntEQ(wc_Sha256Update(&sha256, (byte*)b.input, (word32)b.inLen), 0);
    c.input = NULL;
    c.inLen = WC_SHA256_DIGEST_SIZE;
    ExpectIntEQ(wc_Sha256Update(&sha256, (byte*)c.input, (word32)c.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha256Update(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);

    wc_Sha256Free(&sha256);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha256Update */


/*
 * Unit test function for wc_Sha256Final()
 */
static int test_wc_Sha256Final(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA256
    wc_Sha256 sha256;
    byte* hash_test[3];
    byte hash1[WC_SHA256_DIGEST_SIZE];
    byte hash2[2*WC_SHA256_DIGEST_SIZE];
    byte hash3[5*WC_SHA256_DIGEST_SIZE];
    int times, i;

    /* Initialize */
    ExpectIntEQ(wc_InitSha256(&sha256), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test) / sizeof(byte*);
    for (i = 0; i < times; i++) {
        ExpectIntEQ(wc_Sha256Final(&sha256, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_Sha256Final(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha256Final(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha256Final(&sha256, NULL), BAD_FUNC_ARG);

    wc_Sha256Free(&sha256);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha256Final */
/*
 * Unit test function for wc_Sha256FinalRaw()
 */
static int test_wc_Sha256FinalRaw(void)
{
    EXPECT_DECLS;
#if !defined(NO_SHA256) && !defined(HAVE_SELFTEST) && !defined(WOLFSSL_DEVCRYPTO) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 3))) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    wc_Sha256 sha256;
    byte* hash_test[3];
    byte hash1[WC_SHA256_DIGEST_SIZE];
    byte hash2[2*WC_SHA256_DIGEST_SIZE];
    byte hash3[5*WC_SHA256_DIGEST_SIZE];
    int times, i;

    /* Initialize */
    ExpectIntEQ(wc_InitSha256(&sha256), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test) / sizeof(byte*);
    for (i = 0; i < times; i++) {
        ExpectIntEQ(wc_Sha256FinalRaw(&sha256, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_Sha256FinalRaw(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha256FinalRaw(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha256FinalRaw(&sha256, NULL), BAD_FUNC_ARG);

    wc_Sha256Free(&sha256);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Sha256FinalRaw */
/*
 * Unit test function for wc_Sha256GetFlags()
 */
static int test_wc_Sha256GetFlags(void)
{
    EXPECT_DECLS;
#if !defined(NO_SHA256) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha256 sha256;
    word32 flags = 0;

    /* Initialize */
    ExpectIntEQ(wc_InitSha256(&sha256), 0);

    ExpectIntEQ(wc_Sha256GetFlags(&sha256, &flags), 0);
    ExpectTrue((flags & WC_HASH_FLAG_ISCOPY) == 0);

    wc_Sha256Free(&sha256);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Sha256GetFlags */
/*
 * Unit test function for wc_Sha256Free()
 */
static int test_wc_Sha256Free(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA256
    wc_Sha256Free(NULL);
    /* Set result to SUCCESS. */
    ExpectTrue(1);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha256Free */
/*
 * Unit test function for wc_Sha256GetHash()
 */
static int test_wc_Sha256GetHash(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA256
    wc_Sha256 sha256;
    byte hash1[WC_SHA256_DIGEST_SIZE];

    /* Initialize */
    ExpectIntEQ(wc_InitSha256(&sha256), 0);

    ExpectIntEQ(wc_Sha256GetHash(&sha256, hash1), 0);

    /* test bad arguments*/
    ExpectIntEQ(wc_Sha256GetHash(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha256GetHash(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha256GetHash(&sha256, NULL), BAD_FUNC_ARG);

    wc_Sha256Free(&sha256);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha256GetHash */
/*
 * Unit test function for wc_Sha256Copy()
 */
static int test_wc_Sha256Copy(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA256
    wc_Sha256 sha256;
    wc_Sha256 temp;

    XMEMSET(&sha256, 0, sizeof(sha256));
    XMEMSET(&temp, 0, sizeof(temp));

    /* Initialize */
    ExpectIntEQ(wc_InitSha256(&sha256), 0);
    ExpectIntEQ(wc_InitSha256(&temp), 0);

    ExpectIntEQ(wc_Sha256Copy(&sha256, &temp), 0);

    /* test bad arguments*/
    ExpectIntEQ(wc_Sha256Copy(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha256Copy(NULL, &temp), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha256Copy(&sha256, NULL), BAD_FUNC_ARG);

    wc_Sha256Free(&sha256);
    wc_Sha256Free(&temp);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha256Copy */
/*
 * Testing wc_InitSha512()
 */
static int test_wc_InitSha512(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;

    /* Test good arg. */
    ExpectIntEQ(wc_InitSha512(&sha512), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitSha512(NULL), BAD_FUNC_ARG);

    wc_Sha512Free(&sha512);
#endif
    return EXPECT_RESULT();
} /* END test_wc_InitSha512 */


/*
 *  wc_Sha512Update() test.
 */
static int test_wc_Sha512Update(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;
    byte hash[WC_SHA512_DIGEST_SIZE];
    byte hash_unaligned[WC_SHA512_DIGEST_SIZE + 1];
    testVector a, b, c;

    ExpectIntEQ(wc_InitSha512(&sha512), 0);

    /* Input. */
    a.input = "a";
    a.inLen = XSTRLEN(a.input);
    ExpectIntEQ(wc_Sha512Update(&sha512, NULL, 0), 0);
    ExpectIntEQ(wc_Sha512Update(&sha512,(byte*)a.input, 0), 0);
    ExpectIntEQ(wc_Sha512Update(&sha512, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_Sha512Final(&sha512, hash), 0);

    /* Update input. */
    a.input = "abc";
    a.output = "\xdd\xaf\x35\xa1\x93\x61\x7a\xba\xcc\x41\x73\x49\xae\x20\x41"
               "\x31\x12\xe6\xfa\x4e\x89\xa9\x7e\xa2\x0a\x9e\xee\xe6\x4b"
               "\x55\xd3\x9a\x21\x92\x99\x2a\x27\x4f\xc1\xa8\x36\xba\x3c"
               "\x23\xa3\xfe\xeb\xbd\x45\x4d\x44\x23\x64\x3c\xe8\x0e\x2a"
               "\x9a\xc9\x4f\xa5\x4c\xa4\x9f";
    a.inLen = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);
    ExpectIntEQ(wc_Sha512Update(&sha512, (byte*) a.input, (word32) a.inLen), 0);
    ExpectIntEQ(wc_Sha512Final(&sha512, hash), 0);

    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA512_DIGEST_SIZE), 0);

    /* Unaligned check. */
    ExpectIntEQ(wc_Sha512Update(&sha512, (byte*)a.input+1, (word32)a.inLen-1),
        0);
    ExpectIntEQ(wc_Sha512Final(&sha512, hash_unaligned+1), 0);

    /* Try passing in bad values */
    b.input = NULL;
    b.inLen = 0;
    ExpectIntEQ(wc_Sha512Update(&sha512, (byte*)b.input, (word32)b.inLen), 0);
    c.input = NULL;
    c.inLen = WC_SHA512_DIGEST_SIZE;
    ExpectIntEQ(wc_Sha512Update(&sha512, (byte*)c.input, (word32)c.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512Update(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);

    wc_Sha512Free(&sha512);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Sha512Update  */

#ifdef WOLFSSL_SHA512
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
        (!defined(WOLFSSL_NOSHA512_224) || !defined(WOLFSSL_NOSHA512_256))
/* Performs test for
 * - wc_Sha512Final/wc_Sha512FinalRaw
 * - wc_Sha512_224Final/wc_Sha512_224Final
 * - wc_Sha512_256Final/wc_Sha512_256Final
 * parameter:
 * - type : must be one of WC_HASH_TYPE_SHA512, WC_HASH_TYPE_SHA512_224 or
 *          WC_HASH_TYPE_SHA512_256
 * - isRaw: if is non-zero, xxxFinalRaw function will be tested
 *return 0 on success
 */
static int test_Sha512_Family_Final(int type, int isRaw)
{
    EXPECT_DECLS;
    wc_Sha512 sha512;
    byte* hash_test[3];
    byte hash1[WC_SHA512_DIGEST_SIZE];
    byte hash2[2*WC_SHA512_DIGEST_SIZE];
    byte hash3[5*WC_SHA512_DIGEST_SIZE];
    int times, i;

    int(*initFp)(wc_Sha512*);
    int(*finalFp)(wc_Sha512*, byte*);
    void(*freeFp)(wc_Sha512*);

    if (type == WC_HASH_TYPE_SHA512) {
        initFp  = wc_InitSha512;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
    !defined(WOLFSSL_NO_HASH_RAW)
        finalFp = (isRaw)? wc_Sha512FinalRaw : wc_Sha512Final;
#else
        finalFp = (isRaw)? NULL : wc_Sha512Final;
#endif
        freeFp  = wc_Sha512Free;
    }
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if !defined(WOLFSSL_NOSHA512_224)
    else if (type == WC_HASH_TYPE_SHA512_224) {
        initFp  = wc_InitSha512_224;
    #if !defined(WOLFSSL_NO_HASH_RAW)
        finalFp = (isRaw)? wc_Sha512_224FinalRaw : wc_Sha512_224Final;
    #else
        finalFp = (isRaw)? NULL : wc_Sha512_224Final;
    #endif
        freeFp  = wc_Sha512_224Free;
    }
#endif
#if !defined(WOLFSSL_NOSHA512_256)
    else if (type == WC_HASH_TYPE_SHA512_256) {
        initFp  = wc_InitSha512_256;
    #if !defined(WOLFSSL_NO_HASH_RAW)
        finalFp = (isRaw)? wc_Sha512_256FinalRaw : wc_Sha512_256Final;
    #else
        finalFp = (isRaw)? NULL : wc_Sha512_256Final;
    #endif
        freeFp  = wc_Sha512_256Free;
    }
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    else
        return TEST_FAIL;

    /* Initialize  */
    ExpectIntEQ(initFp(&sha512), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test) / sizeof(byte *);

    /* Good test args. */
    for (i = 0; i < times; i++) {
        ExpectIntEQ(finalFp(&sha512, hash_test[i]), 0);
    }
    /* Test bad args. */
    ExpectIntEQ(finalFp(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(finalFp(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(finalFp(&sha512, NULL), BAD_FUNC_ARG);

    freeFp(&sha512);

    return EXPECT_RESULT();
}
#endif /* !HAVE_FIPS && !HAVE_SELFTEST &&
                        (!WOLFSSL_NOSHA512_224 || !WOLFSSL_NOSHA512_256) */
#endif /* WOLFSSL_SHA512 */
/*
 * Unit test function for wc_Sha512Final()
 */
static int test_wc_Sha512Final(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;
    byte* hash_test[3];
    byte hash1[WC_SHA512_DIGEST_SIZE];
    byte hash2[2*WC_SHA512_DIGEST_SIZE];
    byte hash3[5*WC_SHA512_DIGEST_SIZE];
    int times, i;

    /* Initialize  */
    ExpectIntEQ(wc_InitSha512(&sha512), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test) / sizeof(byte *);
    for (i = 0; i < times; i++) {
         ExpectIntEQ(wc_Sha512Final(&sha512, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_Sha512Final(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512Final(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512Final(&sha512, NULL), BAD_FUNC_ARG);

    wc_Sha512Free(&sha512);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha512Final */
/*
 * Unit test function for wc_Sha512GetFlags()
 */
static int test_wc_Sha512GetFlags(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA512) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha512 sha512;
    word32 flags = 0;

    /* Initialize */
    ExpectIntEQ(wc_InitSha512(&sha512), 0);

    ExpectIntEQ(wc_Sha512GetFlags(&sha512, &flags), 0);
    ExpectIntEQ((flags & WC_HASH_FLAG_ISCOPY), 0);

    wc_Sha512Free(&sha512);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha512GetFlags */
/*
 * Unit test function for wc_Sha512FinalRaw()
 */
static int test_wc_Sha512FinalRaw(void)
{
    EXPECT_DECLS;
#if (defined(WOLFSSL_SHA512) && !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 3)))) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    wc_Sha512 sha512;
    byte* hash_test[3];
    byte hash1[WC_SHA512_DIGEST_SIZE];
    byte hash2[2*WC_SHA512_DIGEST_SIZE];
    byte hash3[5*WC_SHA512_DIGEST_SIZE];
    int times, i;

    /* Initialize */
    ExpectIntEQ(wc_InitSha512(&sha512), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test) / sizeof(byte*);
    /* Good test args. */
    for (i = 0; i < times; i++) {
         ExpectIntEQ(wc_Sha512FinalRaw(&sha512, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_Sha512FinalRaw(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512FinalRaw(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512FinalRaw(&sha512, NULL), BAD_FUNC_ARG);

    wc_Sha512Free(&sha512);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha512FinalRaw */

/*
 * Unit test function for wc_Sha512Free()
 */
static int test_wc_Sha512Free(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA512
    wc_Sha512Free(NULL);
    /* Set result to SUCCESS. */
    ExpectTrue(1);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha512Free */
#ifdef WOLFSSL_SHA512

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
        (!defined(WOLFSSL_NOSHA512_224) || !defined(WOLFSSL_NOSHA512_256))
static int test_Sha512_Family_GetHash(int type )
{
    EXPECT_DECLS;
    int(*initFp)(wc_Sha512*);
    int(*ghashFp)(wc_Sha512*, byte*);
    wc_Sha512 sha512;
    byte hash1[WC_SHA512_DIGEST_SIZE];

    if (type == WC_HASH_TYPE_SHA512) {
        initFp  = wc_InitSha512;
        ghashFp = wc_Sha512GetHash;
    }
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if !defined(WOLFSSL_NOSHA512_224)
    else if (type == WC_HASH_TYPE_SHA512_224) {
        initFp  = wc_InitSha512_224;
        ghashFp = wc_Sha512_224GetHash;
    }
#endif
#if !defined(WOLFSSL_NOSHA512_256)
    else if (type == WC_HASH_TYPE_SHA512_256) {
        initFp  = wc_InitSha512_256;
        ghashFp = wc_Sha512_256GetHash;
    }
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    else {
        initFp  = NULL;
        ghashFp = NULL;
    }

    if (initFp == NULL || ghashFp == NULL)
        return TEST_FAIL;

    ExpectIntEQ(initFp(&sha512), 0);
    ExpectIntEQ(ghashFp(&sha512, hash1), 0);

    /* test bad arguments*/
    ExpectIntEQ(ghashFp(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(ghashFp(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(ghashFp(&sha512, NULL), BAD_FUNC_ARG);

    wc_Sha512Free(&sha512);
    return EXPECT_RESULT();
}
#endif /* !HAVE_FIPS && !HAVE_SELFTEST &&
                        (!WOLFSSL_NOSHA512_224 || !WOLFSSL_NOSHA512_256) */
#endif /* WOLFSSL_SHA512 */
/*
 * Unit test function for wc_Sha512GetHash()
 */
static int test_wc_Sha512GetHash(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;
    byte hash1[WC_SHA512_DIGEST_SIZE];

    /* Initialize */
    ExpectIntEQ(wc_InitSha512(&sha512), 0);

    ExpectIntEQ(wc_Sha512GetHash(&sha512, hash1), 0);

    /* test bad arguments*/
    ExpectIntEQ(wc_Sha512GetHash(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512GetHash(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512GetHash(&sha512, NULL), BAD_FUNC_ARG);

    wc_Sha512Free(&sha512);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha512GetHash */

/*
 * Unit test function for wc_Sha512Copy()
 */
static int test_wc_Sha512Copy(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;
    wc_Sha512 temp;

    XMEMSET(&sha512, 0, sizeof(wc_Sha512));
    XMEMSET(&temp, 0, sizeof(wc_Sha512));

    /* Initialize */
    ExpectIntEQ(wc_InitSha512(&sha512), 0);
    ExpectIntEQ(wc_InitSha512(&temp), 0);

    ExpectIntEQ(wc_Sha512Copy(&sha512, &temp), 0);

    /* test bad arguments*/
    ExpectIntEQ(wc_Sha512Copy(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512Copy(NULL, &temp), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512Copy(&sha512, NULL), BAD_FUNC_ARG);

    wc_Sha512Free(&sha512);
    wc_Sha512Free(&temp);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha512Copy */

static int test_wc_InitSha512_224(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    wc_Sha512 sha512;

    /* Test good arg. */
    ExpectIntEQ(wc_InitSha512_224(&sha512), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitSha512_224(NULL), BAD_FUNC_ARG);

    wc_Sha512_224Free(&sha512);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_224 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_224Update(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    wc_Sha512 sha512;
    byte hash[WC_SHA512_DIGEST_SIZE];
    testVector a, c;

    ExpectIntEQ(wc_InitSha512_224(&sha512), 0);

    /* Input. */
    a.input = "a";
    a.inLen = XSTRLEN(a.input);
    ExpectIntEQ(wc_Sha512_224Update(&sha512, NULL, 0), 0);
    ExpectIntEQ(wc_Sha512_224Update(&sha512,(byte*)a.input, 0), 0);
    ExpectIntEQ(wc_Sha512_224Update(&sha512, (byte*)a.input, (word32)a.inLen),
        0);
    ExpectIntEQ(wc_Sha512_224Final(&sha512, hash), 0);

    /* Update input. */
    a.input = "abc";
    a.output = "\x46\x34\x27\x0f\x70\x7b\x6a\x54\xda\xae\x75\x30\x46\x08"
               "\x42\xe2\x0e\x37\xed\x26\x5c\xee\xe9\xa4\x3e\x89\x24\xaa";
    a.inLen = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);
    ExpectIntEQ(wc_Sha512_224Update(&sha512, (byte*) a.input, (word32) a.inLen),
        0);
    ExpectIntEQ(wc_Sha512_224Final(&sha512, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA512_224_DIGEST_SIZE), 0);

    c.input = NULL;
    c.inLen = WC_SHA512_224_DIGEST_SIZE;
    ExpectIntEQ(wc_Sha512_224Update(&sha512, (byte*)c.input, (word32)c.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512_224Update(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);

    wc_Sha512_224Free(&sha512);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_224 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_224Final(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    ExpectIntEQ(test_Sha512_Family_Final(WC_HASH_TYPE_SHA512_224, 0),
        TEST_SUCCESS);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_224 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_224GetFlags(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha512 sha512;
    wc_Sha512 copy;
    word32 flags = 0;

    XMEMSET(&sha512, 0, sizeof(wc_Sha512));
    XMEMSET(&copy, 0, sizeof(wc_Sha512));

    /* Initialize */
    ExpectIntEQ(wc_InitSha512_224(&sha512), 0);
    ExpectIntEQ(wc_InitSha512_224(&copy), 0);

    ExpectIntEQ(wc_Sha512_224GetFlags(&sha512, &flags), 0);
    ExpectTrue((flags & WC_HASH_FLAG_ISCOPY) == 0);

    ExpectIntEQ(wc_Sha512_224Copy(&sha512, &copy), 0);
    ExpectIntEQ(wc_Sha512_224GetFlags(&copy, &flags), 0);
    ExpectTrue((flags & WC_HASH_FLAG_ISCOPY) == WC_HASH_FLAG_ISCOPY);

    wc_Sha512_224Free(&copy);
    wc_Sha512_224Free(&sha512);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_224FinalRaw(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
    defined(WOLFSSL_SHA512) &&  !defined(WOLFSSL_NOSHA512_224) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    ExpectIntEQ(test_Sha512_Family_Final(WC_HASH_TYPE_SHA512_224, 1),
        TEST_SUCCESS);
#endif
    return EXPECT_RESULT();
}

static int test_wc_Sha512_224Free(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    wc_Sha512_224Free(NULL);
    /* Set result to SUCCESS. */
    ExpectTrue(1);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_224GetHash(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    ExpectIntEQ(test_Sha512_Family_GetHash(WC_HASH_TYPE_SHA512_224),
        TEST_SUCCESS);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}
static int test_wc_Sha512_224Copy(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    wc_Sha512 sha512;
    wc_Sha512 temp;

    XMEMSET(&sha512, 0, sizeof(wc_Sha512));
    XMEMSET(&temp, 0, sizeof(wc_Sha512));

    /* Initialize */
    ExpectIntEQ(wc_InitSha512_224(&sha512), 0);
    ExpectIntEQ(wc_InitSha512_224(&temp), 0);

    ExpectIntEQ(wc_Sha512_224Copy(&sha512, &temp), 0);
    /* test bad arguments*/
    ExpectIntEQ(wc_Sha512_224Copy(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512_224Copy(NULL, &temp), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512_224Copy(&sha512, NULL), BAD_FUNC_ARG);

    wc_Sha512_224Free(&sha512);
    wc_Sha512_224Free(&temp);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_InitSha512_256(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    wc_Sha512 sha512;

    /* Test good arg. */
    ExpectIntEQ(wc_InitSha512_256(&sha512), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitSha512_256(NULL), BAD_FUNC_ARG);

    wc_Sha512_256Free(&sha512);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_256 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_256Update(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    wc_Sha512 sha512;
    byte hash[WC_SHA512_DIGEST_SIZE];
    testVector a, c;

    ExpectIntEQ(wc_InitSha512_256(&sha512), 0);

    /* Input. */
    a.input = "a";
    a.inLen = XSTRLEN(a.input);
    ExpectIntEQ(wc_Sha512_256Update(&sha512, NULL, 0), 0);
    ExpectIntEQ(wc_Sha512_256Update(&sha512,(byte*)a.input, 0), 0);
    ExpectIntEQ(wc_Sha512_256Update(&sha512, (byte*)a.input, (word32)a.inLen),
        0);
    ExpectIntEQ(wc_Sha512_256Final(&sha512, hash), 0);

    /* Update input. */
    a.input = "abc";
    a.output = "\x53\x04\x8e\x26\x81\x94\x1e\xf9\x9b\x2e\x29\xb7\x6b\x4c"
               "\x7d\xab\xe4\xc2\xd0\xc6\x34\xfc\x6d\x46\xe0\xe2\xf1\x31"
               "\x07\xe7\xaf\x23";
    a.inLen = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);
    ExpectIntEQ(wc_Sha512_256Update(&sha512, (byte*) a.input, (word32) a.inLen),
        0);
    ExpectIntEQ(wc_Sha512_256Final(&sha512, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA512_256_DIGEST_SIZE), 0);

    c.input = NULL;
    c.inLen = WC_SHA512_256_DIGEST_SIZE;
    ExpectIntEQ(wc_Sha512_256Update(&sha512, (byte*)c.input, (word32)c.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512_256Update(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);

    wc_Sha512_256Free(&sha512);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_256 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_256Final(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    ExpectIntEQ(test_Sha512_Family_Final(WC_HASH_TYPE_SHA512_256, 0),
        TEST_SUCCESS);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_256 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_256GetFlags(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha512 sha512, copy;
    word32 flags = 0;

    XMEMSET(&sha512, 0, sizeof(wc_Sha512));
    XMEMSET(&copy, 0, sizeof(wc_Sha512));

    /* Initialize */
    ExpectIntEQ(wc_InitSha512_256(&sha512), 0);
    ExpectIntEQ(wc_InitSha512_256(&copy), 0);

    ExpectIntEQ(wc_Sha512_256GetFlags(&sha512, &flags), 0);
    ExpectTrue((flags & WC_HASH_FLAG_ISCOPY) == 0);

    ExpectIntEQ(wc_Sha512_256Copy(&sha512, &copy), 0);
    ExpectIntEQ(wc_Sha512_256GetFlags(&copy, &flags), 0);
    ExpectTrue((flags & WC_HASH_FLAG_ISCOPY) == WC_HASH_FLAG_ISCOPY);

    wc_Sha512_256Free(&sha512);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_256FinalRaw(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
    defined(WOLFSSL_SHA512) &&  !defined(WOLFSSL_NOSHA512_256) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    ExpectIntEQ(test_Sha512_Family_Final(WC_HASH_TYPE_SHA512_256, 1),
        TEST_SUCCESS);
#endif
    return EXPECT_RESULT();
}

static int test_wc_Sha512_256Free(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    wc_Sha512_256Free(NULL);
    /* Set result to SUCCESS. */
    ExpectTrue(1);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wc_Sha512_256GetHash(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    ExpectIntEQ(test_Sha512_Family_GetHash(WC_HASH_TYPE_SHA512_256),
        TEST_SUCCESS);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();

}
static int test_wc_Sha512_256Copy(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    wc_Sha512 sha512;
    wc_Sha512 temp;

    XMEMSET(&sha512, 0, sizeof(wc_Sha512));
    XMEMSET(&temp, 0, sizeof(wc_Sha512));

    /* Initialize */
    ExpectIntEQ(wc_InitSha512_256(&sha512), 0);
    ExpectIntEQ(wc_InitSha512_256(&temp), 0);

    ExpectIntEQ(wc_Sha512_256Copy(&sha512, &temp), 0);
    /* test bad arguments*/
    ExpectIntEQ(wc_Sha512_256Copy(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512_256Copy(NULL, &temp), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha512_256Copy(&sha512, NULL), BAD_FUNC_ARG);

    wc_Sha512_256Free(&sha512);
    wc_Sha512_256Free(&temp);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}



/*
 * Testing wc_InitSha384()
 */
static int test_wc_InitSha384(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;

    /* Test good arg. */
    ExpectIntEQ(wc_InitSha384(&sha384), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitSha384(NULL), BAD_FUNC_ARG);

    wc_Sha384Free(&sha384);
#endif
    return EXPECT_RESULT();
} /* END test_wc_InitSha384 */


/*
 * test wc_Sha384Update()
 */
static int test_wc_Sha384Update(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;
    byte hash[WC_SHA384_DIGEST_SIZE];
    testVector a, b, c;

    ExpectIntEQ(wc_InitSha384(&sha384), 0);

    /* Input */
    a.input = "a";
    a.inLen = XSTRLEN(a.input);
    ExpectIntEQ(wc_Sha384Update(&sha384, NULL, 0), 0);
    ExpectIntEQ(wc_Sha384Update(&sha384, (byte*)a.input, 0), 0);
    ExpectIntEQ(wc_Sha384Update(&sha384, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_Sha384Final(&sha384, hash), 0);

    /* Update input. */
    a.input = "abc";
    a.output = "\xcb\x00\x75\x3f\x45\xa3\x5e\x8b\xb5\xa0\x3d\x69\x9a\xc6\x50"
               "\x07\x27\x2c\x32\xab\x0e\xde\xd1\x63\x1a\x8b\x60\x5a\x43\xff"
               "\x5b\xed\x80\x86\x07\x2b\xa1\xe7\xcc\x23\x58\xba\xec\xa1\x34"
               "\xc8\x25\xa7";
    a.inLen = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);
    ExpectIntEQ(wc_Sha384Update(&sha384, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_Sha384Final(&sha384, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA384_DIGEST_SIZE), 0);

    /* Pass in bad values. */
    b.input = NULL;
    b.inLen = 0;
    ExpectIntEQ(wc_Sha384Update(&sha384, (byte*)b.input, (word32)b.inLen), 0);
    c.input = NULL;
    c.inLen = WC_SHA384_DIGEST_SIZE;
    ExpectIntEQ( wc_Sha384Update(&sha384, (byte*)c.input, (word32)c.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha384Update(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);

    wc_Sha384Free(&sha384);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha384Update */

/*
 * Unit test function for wc_Sha384Final();
 */
static int test_wc_Sha384Final(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;
    byte* hash_test[3];
    byte hash1[WC_SHA384_DIGEST_SIZE];
    byte hash2[2*WC_SHA384_DIGEST_SIZE];
    byte hash3[5*WC_SHA384_DIGEST_SIZE];
    int times, i;

    /* Initialize */
    ExpectIntEQ(wc_InitSha384(&sha384), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test) / sizeof(byte*);
    /* Good test args. */
    for (i = 0; i < times; i++) {
         ExpectIntEQ(wc_Sha384Final(&sha384, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_Sha384Final(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha384Final(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha384Final(&sha384, NULL), BAD_FUNC_ARG);

    wc_Sha384Free(&sha384);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha384Final */
/*
 * Unit test function for wc_Sha384GetFlags()
 */
static int test_wc_Sha384GetFlags(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha384 sha384;
    word32 flags = 0;

    /* Initialize */
    ExpectIntEQ(wc_InitSha384(&sha384), 0);
    ExpectIntEQ(wc_Sha384GetFlags(&sha384, &flags), 0);
    ExpectTrue((flags & WC_HASH_FLAG_ISCOPY) == 0);

    wc_Sha384Free(&sha384);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Sha384GetFlags */
/*
 * Unit test function for wc_Sha384FinalRaw()
 */
static int test_wc_Sha384FinalRaw(void)
{
    EXPECT_DECLS;
#if (defined(WOLFSSL_SHA384) && !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 3)))) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    wc_Sha384 sha384;
    byte* hash_test[3];
    byte hash1[WC_SHA384_DIGEST_SIZE];
    byte hash2[2*WC_SHA384_DIGEST_SIZE];
    byte hash3[5*WC_SHA384_DIGEST_SIZE];
    int times, i;

    /* Initialize */
    ExpectIntEQ(wc_InitSha384(&sha384), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test) / sizeof(byte*);
    /* Good test args. */
    for (i = 0; i < times; i++) {
         ExpectIntEQ(wc_Sha384FinalRaw(&sha384, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_Sha384FinalRaw(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha384FinalRaw(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha384FinalRaw(&sha384, NULL), BAD_FUNC_ARG);

    wc_Sha384Free(&sha384);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha384FinalRaw */
/*
 * Unit test function for wc_Sha384Free()
 */
static int test_wc_Sha384Free(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA384
    wc_Sha384Free(NULL);
    /* Set result to SUCCESS. */
    ExpectTrue(1);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Sha384Free */
/*
 * Unit test function for wc_Sha384GetHash()
 */
static int test_wc_Sha384GetHash(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;
    byte hash1[WC_SHA384_DIGEST_SIZE];

    /* Initialize */
    ExpectIntEQ(wc_InitSha384(&sha384), 0);

    ExpectIntEQ(wc_Sha384GetHash(&sha384, hash1), 0);
    /* test bad arguments*/
    ExpectIntEQ(wc_Sha384GetHash(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha384GetHash(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha384GetHash(&sha384, NULL), BAD_FUNC_ARG);

    wc_Sha384Free(&sha384);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha384GetHash */
/*
 * Unit test function for wc_Sha384Copy()
 */
static int test_wc_Sha384Copy(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;
    wc_Sha384 temp;

    XMEMSET(&sha384, 0, sizeof(wc_Sha384));
    XMEMSET(&temp, 0, sizeof(wc_Sha384));

    /* Initialize */
    ExpectIntEQ(wc_InitSha384(&sha384), 0);
    ExpectIntEQ(wc_InitSha384(&temp), 0);

    ExpectIntEQ(wc_Sha384Copy(&sha384, &temp), 0);
    /* test bad arguments*/
    ExpectIntEQ(wc_Sha384Copy(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha384Copy(NULL, &temp), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha384Copy(&sha384, NULL), BAD_FUNC_ARG);

    wc_Sha384Free(&sha384);
    wc_Sha384Free(&temp);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha384Copy */

/*
 * Testing wc_InitSha224();
 */
static int test_wc_InitSha224(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;

    /* Test good arg. */
    ExpectIntEQ(wc_InitSha224(&sha224), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitSha224(NULL), BAD_FUNC_ARG);

    wc_Sha224Free(&sha224);
#endif
    return EXPECT_RESULT();
} /* END test_wc_InitSha224 */

/*
 * Unit test on wc_Sha224Update
 */
static int test_wc_Sha224Update(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;
    byte hash[WC_SHA224_DIGEST_SIZE];
    testVector a, b, c;

    ExpectIntEQ(wc_InitSha224(&sha224), 0);

    /* Input. */
    a.input = "a";
    a.inLen = XSTRLEN(a.input);
    ExpectIntEQ(wc_Sha224Update(&sha224, NULL, 0), 0);
    ExpectIntEQ(wc_Sha224Update(&sha224, (byte*)a.input, 0), 0);
    ExpectIntEQ(wc_Sha224Update(&sha224, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_Sha224Final(&sha224, hash), 0);

    /* Update input. */
    a.input = "abc";
    a.output = "\x23\x09\x7d\x22\x34\x05\xd8\x22\x86\x42\xa4\x77\xbd\xa2"
               "\x55\xb3\x2a\xad\xbc\xe4\xbd\xa0\xb3\xf7\xe3\x6c\x9d\xa7";
    a.inLen = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);
    ExpectIntEQ(wc_Sha224Update(&sha224, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_Sha224Final(&sha224, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA224_DIGEST_SIZE), 0);

    /* Pass in bad values. */
    b.input = NULL;
    b.inLen = 0;
    ExpectIntEQ(wc_Sha224Update(&sha224, (byte*)b.input, (word32)b.inLen), 0);
    c.input = NULL;
    c.inLen = WC_SHA224_DIGEST_SIZE;
    ExpectIntEQ(wc_Sha224Update(&sha224, (byte*)c.input, (word32)c.inLen),
       BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha224Update(NULL, (byte*)a.input, (word32)a.inLen),
       BAD_FUNC_ARG);

    wc_Sha224Free(&sha224);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha224Update */

/*
 * Unit test for wc_Sha224Final();
 */
static int test_wc_Sha224Final(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;
    byte* hash_test[3];
    byte hash1[WC_SHA224_DIGEST_SIZE];
    byte hash2[2*WC_SHA224_DIGEST_SIZE];
    byte hash3[5*WC_SHA224_DIGEST_SIZE];
    int times, i;

    /* Initialize */
    ExpectIntEQ(wc_InitSha224(&sha224), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test) / sizeof(byte*);
    /* Good test args. */
    /* Testing oversized buffers. */
    for (i = 0; i < times; i++) {
        ExpectIntEQ(wc_Sha224Final(&sha224, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_Sha224Final(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha224Final(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha224Final(&sha224, NULL), BAD_FUNC_ARG);

    wc_Sha224Free(&sha224);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha224Final */

/*
 * Unit test function for wc_Sha224SetFlags()
 */
static int test_wc_Sha224SetFlags(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA224) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha224 sha224;
    word32 flags = WC_HASH_FLAG_WILLCOPY;

    /* Initialize */
    ExpectIntEQ(wc_InitSha224(&sha224), 0);

    ExpectIntEQ(wc_Sha224SetFlags(&sha224, flags), 0);
    flags = 0;
    ExpectIntEQ(wc_Sha224GetFlags(&sha224, &flags), 0);
    ExpectTrue(flags == WC_HASH_FLAG_WILLCOPY);

    wc_Sha224Free(&sha224);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha224SetFlags */

/*
 * Unit test function for wc_Sha224GetFlags()
 */
static int test_wc_Sha224GetFlags(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA224) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha224 sha224;
    word32 flags = 0;

    /* Initialize */
    ExpectIntEQ(wc_InitSha224(&sha224), 0);

    ExpectIntEQ(wc_Sha224GetFlags(&sha224, &flags), 0);
    ExpectTrue((flags & WC_HASH_FLAG_ISCOPY) == 0);

    wc_Sha224Free(&sha224);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha224GetFlags */
/*
 * Unit test function for wc_Sha224Free()
 */
static int test_wc_Sha224Free(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA224
    wc_Sha224Free(NULL);
    /* Set result to SUCCESS. */
    ExpectTrue(1);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Sha224Free */

/*
 * Unit test function for wc_Sha224GetHash()
 */
static int test_wc_Sha224GetHash(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;
    byte hash1[WC_SHA224_DIGEST_SIZE];

    /* Initialize */
    ExpectIntEQ(wc_InitSha224(&sha224), 0);

    ExpectIntEQ(wc_Sha224GetHash(&sha224, hash1), 0);
    /* test bad arguments*/
    ExpectIntEQ(wc_Sha224GetHash(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha224GetHash(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha224GetHash(&sha224, NULL), BAD_FUNC_ARG);

    wc_Sha224Free(&sha224);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha224GetHash */

/*
 * Unit test function for wc_Sha224Copy()
 */
static int test_wc_Sha224Copy(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;
    wc_Sha224 temp;

    XMEMSET(&sha224, 0, sizeof(wc_Sha224));
    XMEMSET(&temp, 0, sizeof(wc_Sha224));

    /* Initialize */
    ExpectIntEQ(wc_InitSha224(&sha224), 0);
    ExpectIntEQ(wc_InitSha224(&temp), 0);

    ExpectIntEQ(wc_Sha224Copy(&sha224, &temp), 0);
    /* test bad arguments*/
    ExpectIntEQ(wc_Sha224Copy(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha224Copy(NULL, &temp), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha224Copy(&sha224, NULL), BAD_FUNC_ARG);

    wc_Sha224Free(&sha224);
    wc_Sha224Free(&temp);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha224Copy */


/*
 * Testing wc_InitRipeMd()
 */
static int test_wc_InitRipeMd(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_RIPEMD
    RipeMd ripemd;

    /* Test good arg. */
    ExpectIntEQ(wc_InitRipeMd(&ripemd), 0);
    /* Test bad arg. */
    ExpectIntEQ(wc_InitRipeMd(NULL), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();

} /* END test_wc_InitRipeMd */

/*
 * Testing wc_RipeMdUpdate()
 */
static int test_wc_RipeMdUpdate(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_RIPEMD
    RipeMd ripemd;
    byte hash[RIPEMD_DIGEST_SIZE];
    testVector a, b, c;

    ExpectIntEQ(wc_InitRipeMd(&ripemd), 0);

    /* Input */
    a.input = "a";
    a.inLen = XSTRLEN(a.input);
    ExpectIntEQ(wc_RipeMdUpdate(&ripemd, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_RipeMdFinal(&ripemd, hash), 0);

    /* Update input. */
    a.input = "abc";
    a.output = "\x8e\xb2\x08\xf7\xe0\x5d\x98\x7a\x9b\x04\x4a\x8e\x98\xc6"
               "\xb0\x87\xf1\x5a\x0b\xfc";
    a.inLen = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);
    ExpectIntEQ(wc_RipeMdUpdate(&ripemd, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_RipeMdFinal(&ripemd, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, RIPEMD_DIGEST_SIZE), 0);

    /* Pass in bad values. */
    b.input = NULL;
    b.inLen = 0;
    ExpectIntEQ(wc_RipeMdUpdate(&ripemd, (byte*)b.input, (word32)b.inLen), 0);
    c.input = NULL;
    c.inLen = RIPEMD_DIGEST_SIZE;
    ExpectIntEQ(wc_RipeMdUpdate(&ripemd, (byte*)c.input, (word32)c.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RipeMdUpdate(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_RipeMdUdpate */

/*
 * Unit test function for wc_RipeMdFinal()
 */
static int test_wc_RipeMdFinal(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_RIPEMD
    RipeMd ripemd;
    byte* hash_test[3];
    byte hash1[RIPEMD_DIGEST_SIZE];
    byte hash2[2*RIPEMD_DIGEST_SIZE];
    byte hash3[5*RIPEMD_DIGEST_SIZE];
    int times, i;

    /* Initialize */
    ExpectIntEQ(wc_InitRipeMd(&ripemd), 0);

    hash_test[0] = hash1;
    hash_test[1] = hash2;
    hash_test[2] = hash3;
    times = sizeof(hash_test) / sizeof(byte*);
    /* Testing oversized buffers. */
    for (i = 0; i < times; i++) {
         ExpectIntEQ(wc_RipeMdFinal(&ripemd, hash_test[i]), 0);
    }

    /* Test bad args. */
    ExpectIntEQ(wc_RipeMdFinal(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RipeMdFinal(NULL, hash1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RipeMdFinal(&ripemd, NULL), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_RipeMdFinal */


/*
 * Testing wc_InitSha3_224, wc_InitSha3_256, wc_InitSha3_384, and
 * wc_InitSha3_512
 */
static int test_wc_InitSha3(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3)
    wc_Sha3 sha3;

    (void)sha3;

#if !defined(WOLFSSL_NOSHA3_224)
    ExpectIntEQ(wc_InitSha3_224(&sha3, HEAP_HINT, testDevId), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_InitSha3_224(NULL, HEAP_HINT, testDevId), BAD_FUNC_ARG);
    wc_Sha3_224_Free(&sha3);
#endif /* NOSHA3_224 */
#if !defined(WOLFSSL_NOSHA3_256)
    ExpectIntEQ(wc_InitSha3_256(&sha3, HEAP_HINT, testDevId), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_InitSha3_256(NULL, HEAP_HINT, testDevId), BAD_FUNC_ARG);
    wc_Sha3_256_Free(&sha3);
#endif /* NOSHA3_256 */
#if !defined(WOLFSSL_NOSHA3_384)
    ExpectIntEQ(wc_InitSha3_384(&sha3, HEAP_HINT, testDevId), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_InitSha3_384(NULL, HEAP_HINT, testDevId),  BAD_FUNC_ARG);
    wc_Sha3_384_Free(&sha3);
#endif /* NOSHA3_384 */
#if !defined(WOLFSSL_NOSHA3_512)
    ExpectIntEQ(wc_InitSha3_512(&sha3, HEAP_HINT, testDevId), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_InitSha3_512(NULL, HEAP_HINT, testDevId), BAD_FUNC_ARG);
    wc_Sha3_512_Free(&sha3);
#endif /* NOSHA3_512 */
#endif
    return EXPECT_RESULT();
} /* END test_wc_InitSha3 */


/*
 * Testing wc_Sha3_Update()
 */
static int testing_wc_Sha3_Update(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_XILINX_CRYPT) && \
   !defined(WOLFSSL_AFALG_XILINX)
    wc_Sha3 sha3;
    byte    msg[] = "Everybody's working for the weekend.";
    byte    msg2[] = "Everybody gets Friday off.";
    byte    msgCmp[] = "\x45\x76\x65\x72\x79\x62\x6f\x64\x79\x27\x73\x20"
                    "\x77\x6f\x72\x6b\x69\x6e\x67\x20\x66\x6f\x72\x20\x74"
                    "\x68\x65\x20\x77\x65\x65\x6b\x65\x6e\x64\x2e\x45\x76"
                    "\x65\x72\x79\x62\x6f\x64\x79\x20\x67\x65\x74\x73\x20"
                    "\x46\x72\x69\x64\x61\x79\x20\x6f\x66\x66\x2e";
    word32  msglen = sizeof(msg) - 1;
    word32  msg2len = sizeof(msg2);
    word32  msgCmplen = sizeof(msgCmp);

    #if !defined(WOLFSSL_NOSHA3_224)
        ExpectIntEQ(wc_InitSha3_224(&sha3, HEAP_HINT, testDevId), 0);
        ExpectIntEQ(wc_Sha3_224_Update(&sha3, msg, msglen), 0);
        ExpectIntEQ(XMEMCMP(msg, sha3.t, msglen), 0);
        ExpectTrue(sha3.i == msglen);

        ExpectIntEQ(wc_Sha3_224_Update(&sha3, msg2, msg2len), 0);
        ExpectIntEQ(XMEMCMP(sha3.t, msgCmp, msgCmplen), 0);

        /* Pass bad args. */
        ExpectIntEQ(wc_Sha3_224_Update(NULL, msg2, msg2len), BAD_FUNC_ARG);
        ExpectIntEQ(wc_Sha3_224_Update(&sha3, NULL, 5), BAD_FUNC_ARG);
        wc_Sha3_224_Free(&sha3);

        ExpectIntEQ(wc_InitSha3_224(&sha3, HEAP_HINT, testDevId), 0);
        ExpectIntEQ(wc_Sha3_224_Update(&sha3, NULL, 0), 0);
        ExpectIntEQ(wc_Sha3_224_Update(&sha3, msg2, msg2len), 0);
        ExpectIntEQ(XMEMCMP(msg2, sha3.t, msg2len), 0);
        wc_Sha3_224_Free(&sha3);
    #endif /* SHA3_224 */

    #if !defined(WOLFSSL_NOSHA3_256)
        ExpectIntEQ(wc_InitSha3_256(&sha3, HEAP_HINT, testDevId), 0);
        ExpectIntEQ(wc_Sha3_256_Update(&sha3, msg, msglen), 0);
        ExpectIntEQ(XMEMCMP(msg, sha3.t, msglen), 0);
        ExpectTrue(sha3.i == msglen);

        ExpectIntEQ(wc_Sha3_256_Update(&sha3, msg2, msg2len), 0);
        ExpectIntEQ(XMEMCMP(sha3.t, msgCmp, msgCmplen), 0);

        /* Pass bad args. */
        ExpectIntEQ(wc_Sha3_256_Update(NULL, msg2, msg2len), BAD_FUNC_ARG);
        ExpectIntEQ(wc_Sha3_256_Update(&sha3, NULL, 5), BAD_FUNC_ARG);
        wc_Sha3_256_Free(&sha3);

        ExpectIntEQ(wc_InitSha3_256(&sha3, HEAP_HINT, testDevId), 0);
        ExpectIntEQ(wc_Sha3_256_Update(&sha3, NULL, 0), 0);
        ExpectIntEQ(wc_Sha3_256_Update(&sha3, msg2, msg2len), 0);
        ExpectIntEQ(XMEMCMP(msg2, sha3.t, msg2len), 0);
        wc_Sha3_256_Free(&sha3);
    #endif /* SHA3_256 */

    #if !defined(WOLFSSL_NOSHA3_384)
        ExpectIntEQ(wc_InitSha3_384(&sha3, HEAP_HINT, testDevId), 0);
        ExpectIntEQ(wc_Sha3_384_Update(&sha3, msg, msglen), 0);
        ExpectIntEQ(XMEMCMP(msg, sha3.t, msglen), 0);
        ExpectTrue(sha3.i == msglen);

        ExpectIntEQ(wc_Sha3_384_Update(&sha3, msg2, msg2len), 0);
        ExpectIntEQ(XMEMCMP(sha3.t, msgCmp, msgCmplen), 0);

        /* Pass bad args. */
        ExpectIntEQ(wc_Sha3_384_Update(NULL, msg2, msg2len), BAD_FUNC_ARG);
        ExpectIntEQ(wc_Sha3_384_Update(&sha3, NULL, 5), BAD_FUNC_ARG);
        wc_Sha3_384_Free(&sha3);

        ExpectIntEQ(wc_InitSha3_384(&sha3, HEAP_HINT, testDevId), 0);
        ExpectIntEQ(wc_Sha3_384_Update(&sha3, NULL, 0), 0);
        ExpectIntEQ(wc_Sha3_384_Update(&sha3, msg2, msg2len), 0);
        ExpectIntEQ(XMEMCMP(msg2, sha3.t, msg2len), 0);
        wc_Sha3_384_Free(&sha3);
    #endif /* SHA3_384 */

    #if !defined(WOLFSSL_NOSHA3_512)
        ExpectIntEQ(wc_InitSha3_512(&sha3, HEAP_HINT, testDevId), 0);
        ExpectIntEQ(wc_Sha3_512_Update(&sha3, msg, msglen), 0);
        ExpectIntEQ(XMEMCMP(msg, sha3.t, msglen), 0);
        ExpectTrue(sha3.i == msglen);

        ExpectIntEQ(wc_Sha3_512_Update(&sha3, msg2, msg2len), 0);
        ExpectIntEQ(XMEMCMP(sha3.t, msgCmp, msgCmplen), 0);

        /* Pass bad args. */
        ExpectIntEQ(wc_Sha3_512_Update(NULL, msg2, msg2len), BAD_FUNC_ARG);
        ExpectIntEQ(wc_Sha3_512_Update(&sha3, NULL, 5), BAD_FUNC_ARG);
        wc_Sha3_512_Free(&sha3);

        ExpectIntEQ(wc_InitSha3_512(&sha3, HEAP_HINT, testDevId), 0);
        ExpectIntEQ(wc_Sha3_512_Update(&sha3, NULL, 0), 0);
        ExpectIntEQ(wc_Sha3_512_Update(&sha3, msg2, msg2len), 0);
        ExpectIntEQ(XMEMCMP(msg2, sha3.t, msg2len), 0);
        wc_Sha3_512_Free(&sha3);
    #endif /* SHA3_512 */
#endif /* WOLFSSL_SHA3 */
    return EXPECT_RESULT();
} /* END testing_wc_Sha3_Update */

/*
 *  Testing wc_Sha3_224_Final()
 */
static int test_wc_Sha3_224_Final(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_224)
    wc_Sha3     sha3;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x8a\x24\x10\x8b\x15\x4a\xda\x21\xc9\xfd\x55"
                         "\x74\x49\x44\x79\xba\x5c\x7e\x7a\xb7\x6e\xf2"
                         "\x64\xea\xd0\xfc\xce\x33";
    byte        hash[WC_SHA3_224_DIGEST_SIZE];
    byte        hashRet[WC_SHA3_224_DIGEST_SIZE];

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ExpectIntEQ(wc_InitSha3_224(&sha3, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Sha3_224_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg)), 0);
    ExpectIntEQ(wc_Sha3_224_Final(&sha3, hash), 0);
    ExpectIntEQ(XMEMCMP(expOut, hash, WC_SHA3_224_DIGEST_SIZE), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_224_Final(NULL, hash), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_224_Final(&sha3, NULL), BAD_FUNC_ARG);
    wc_Sha3_224_Free(&sha3);

    ExpectIntEQ(wc_InitSha3_224(&sha3, HEAP_HINT, testDevId), 0);
    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashRet, 0, sizeof(hashRet));
    ExpectIntEQ(wc_Sha3_224_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg)), 0);
    ExpectIntEQ(wc_Sha3_224_GetHash(&sha3, hashRet), 0);
    ExpectIntEQ(wc_Sha3_224_Final(&sha3, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, hashRet, WC_SHA3_224_DIGEST_SIZE), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_224_GetHash(NULL, hashRet), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_224_GetHash(&sha3, NULL), BAD_FUNC_ARG);

    wc_Sha3_224_Free(&sha3);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha3_224_Final */


/*
 *  Testing wc_Sha3_256_Final()
 */
static int test_wc_Sha3_256_Final(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wc_Sha3     sha3;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x41\xc0\xdb\xa2\xa9\xd6\x24\x08\x49\x10\x03\x76\xa8"
                        "\x23\x5e\x2c\x82\xe1\xb9\x99\x8a\x99\x9e\x21\xdb\x32"
                        "\xdd\x97\x49\x6d\x33\x76";
    byte        hash[WC_SHA3_256_DIGEST_SIZE];
    byte        hashRet[WC_SHA3_256_DIGEST_SIZE];

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ExpectIntEQ(wc_InitSha3_256(&sha3, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Sha3_256_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg)), 0);
    ExpectIntEQ(wc_Sha3_256_Final(&sha3, hash), 0);
    ExpectIntEQ(XMEMCMP(expOut, hash, WC_SHA3_256_DIGEST_SIZE), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_256_Final(NULL, hash), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_256_Final(&sha3, NULL), BAD_FUNC_ARG);
    wc_Sha3_256_Free(&sha3);

    ExpectIntEQ(wc_InitSha3_256(&sha3, HEAP_HINT, testDevId), 0);
    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashRet, 0, sizeof(hashRet));
    ExpectIntEQ(wc_Sha3_256_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg)), 0);
    ExpectIntEQ(wc_Sha3_256_GetHash(&sha3, hashRet), 0);
    ExpectIntEQ(wc_Sha3_256_Final(&sha3, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, hashRet, WC_SHA3_256_DIGEST_SIZE), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_256_GetHash(NULL, hashRet), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_256_GetHash(&sha3, NULL), BAD_FUNC_ARG);

    wc_Sha3_256_Free(&sha3);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha3_256_Final */


/*
 *  Testing wc_Sha3_384_Final()
 */
static int test_wc_Sha3_384_Final(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_384)
    wc_Sha3        sha3;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x99\x1c\x66\x57\x55\xeb\x3a\x4b\x6b\xbd\xfb\x75\xc7"
                         "\x8a\x49\x2e\x8c\x56\xa2\x2c\x5c\x4d\x7e\x42\x9b\xfd"
                         "\xbc\x32\xb9\xd4\xad\x5a\xa0\x4a\x1f\x07\x6e\x62\xfe"
                         "\xa1\x9e\xef\x51\xac\xd0\x65\x7c\x22";
    byte        hash[WC_SHA3_384_DIGEST_SIZE];
    byte        hashRet[WC_SHA3_384_DIGEST_SIZE];

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ExpectIntEQ(wc_InitSha3_384(&sha3, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Sha3_384_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg)), 0);
    ExpectIntEQ(wc_Sha3_384_Final(&sha3, hash), 0);
    ExpectIntEQ(XMEMCMP(expOut, hash, WC_SHA3_384_DIGEST_SIZE), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_384_Final(NULL, hash), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_384_Final(&sha3, NULL), BAD_FUNC_ARG);
    wc_Sha3_384_Free(&sha3);

    ExpectIntEQ(wc_InitSha3_384(&sha3, HEAP_HINT, testDevId), 0);
    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashRet, 0, sizeof(hashRet));
    ExpectIntEQ(wc_Sha3_384_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg)), 0);
    ExpectIntEQ(wc_Sha3_384_GetHash(&sha3, hashRet), 0);
    ExpectIntEQ(wc_Sha3_384_Final(&sha3, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, hashRet, WC_SHA3_384_DIGEST_SIZE), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_384_GetHash(NULL, hashRet), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_384_GetHash(&sha3, NULL), BAD_FUNC_ARG);

    wc_Sha3_384_Free(&sha3);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha3_384_Final */



/*
 *  Testing wc_Sha3_512_Final()
 */
static int test_wc_Sha3_512_Final(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_512) && \
   !defined(WOLFSSL_NOSHA3_384)
    wc_Sha3     sha3;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x04\xa3\x71\xe8\x4e\xcf\xb5\xb8\xb7\x7c\xb4\x86\x10"
                         "\xfc\xa8\x18\x2d\xd4\x57\xce\x6f\x32\x6a\x0f\xd3\xd7"
                         "\xec\x2f\x1e\x91\x63\x6d\xee\x69\x1f\xbe\x0c\x98\x53"
                         "\x02\xba\x1b\x0d\x8d\xc7\x8c\x08\x63\x46\xb5\x33\xb4"
                         "\x9c\x03\x0d\x99\xa2\x7d\xaf\x11\x39\xd6\xe7\x5e";
    byte        hash[WC_SHA3_512_DIGEST_SIZE];
    byte        hashRet[WC_SHA3_512_DIGEST_SIZE];

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ExpectIntEQ(wc_InitSha3_512(&sha3, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Sha3_512_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg)), 0);
    ExpectIntEQ(wc_Sha3_512_Final(&sha3, hash), 0);
    ExpectIntEQ(XMEMCMP(expOut, hash, WC_SHA3_512_DIGEST_SIZE), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_512_Final(NULL, hash), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_512_Final(&sha3, NULL), BAD_FUNC_ARG);
    wc_Sha3_512_Free(&sha3);

    ExpectIntEQ(wc_InitSha3_512(&sha3, HEAP_HINT, testDevId), 0);
    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashRet, 0, sizeof(hashRet));
    ExpectIntEQ(wc_Sha3_512_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg)), 0);
    ExpectIntEQ(wc_Sha3_512_GetHash(&sha3, hashRet), 0);
    ExpectIntEQ(wc_Sha3_512_Final(&sha3, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, hashRet, WC_SHA3_512_DIGEST_SIZE), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_512_GetHash(NULL, hashRet), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_512_GetHash(&sha3, NULL), BAD_FUNC_ARG);

    wc_Sha3_512_Free(&sha3);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha3_512_Final */


/*
 *  Testing wc_Sha3_224_Copy()
 */
static int test_wc_Sha3_224_Copy(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_224)
    wc_Sha3     sha3, sha3Cpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[WC_SHA3_224_DIGEST_SIZE];
    byte        hashCpy[WC_SHA3_224_DIGEST_SIZE];

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));
    XMEMSET(&sha3, 0, sizeof(wc_Sha3));
    XMEMSET(&sha3Cpy, 0, sizeof(wc_Sha3));

    ExpectIntEQ(wc_InitSha3_224(&sha3, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_InitSha3_224(&sha3Cpy, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Sha3_224_Update(&sha3, (byte*)msg, msglen), 0);
    ExpectIntEQ(wc_Sha3_224_Copy(&sha3Cpy, &sha3), 0);
    ExpectIntEQ(wc_Sha3_224_Final(&sha3, hash), 0);
    ExpectIntEQ(wc_Sha3_224_Final(&sha3Cpy, hashCpy), 0);
    ExpectIntEQ(XMEMCMP(hash, hashCpy, sizeof(hash)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_224_Copy(NULL, &sha3), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_224_Copy(&sha3Cpy, NULL), BAD_FUNC_ARG);

    wc_Sha3_224_Free(&sha3);
    wc_Sha3_224_Free(&sha3Cpy);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha3_224_Copy */



/*
 *  Testing wc_Sha3_256_Copy()
 */
static int test_wc_Sha3_256_Copy(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wc_Sha3     sha3, sha3Cpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[WC_SHA3_256_DIGEST_SIZE];
    byte        hashCpy[WC_SHA3_256_DIGEST_SIZE];

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));
    XMEMSET(&sha3, 0, sizeof(wc_Sha3));
    XMEMSET(&sha3Cpy, 0, sizeof(wc_Sha3));

    ExpectIntEQ(wc_InitSha3_256(&sha3, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_InitSha3_256(&sha3Cpy, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Sha3_256_Update(&sha3, (byte*)msg, msglen), 0);
    ExpectIntEQ(wc_Sha3_256_Copy(&sha3Cpy, &sha3), 0);
    ExpectIntEQ(wc_Sha3_256_Final(&sha3, hash), 0);
    ExpectIntEQ(wc_Sha3_256_Final(&sha3Cpy, hashCpy), 0);
    ExpectIntEQ(XMEMCMP(hash, hashCpy, sizeof(hash)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_256_Copy(NULL, &sha3), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_256_Copy(&sha3Cpy, NULL), BAD_FUNC_ARG);

    wc_Sha3_256_Free(&sha3);
    wc_Sha3_256_Free(&sha3Cpy);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha3_256_Copy */



/*
 *  Testing wc_Sha3_384_Copy()
 */
static int test_wc_Sha3_384_Copy(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_384)
    wc_Sha3     sha3, sha3Cpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[WC_SHA3_384_DIGEST_SIZE];
    byte        hashCpy[WC_SHA3_384_DIGEST_SIZE];

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));
    XMEMSET(&sha3, 0, sizeof(wc_Sha3));
    XMEMSET(&sha3Cpy, 0, sizeof(wc_Sha3));

    ExpectIntEQ(wc_InitSha3_384(&sha3, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_InitSha3_384(&sha3Cpy, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Sha3_384_Update(&sha3, (byte*)msg, msglen), 0);
    ExpectIntEQ(wc_Sha3_384_Copy(&sha3Cpy, &sha3), 0);
    ExpectIntEQ(wc_Sha3_384_Final(&sha3, hash), 0);
    ExpectIntEQ(wc_Sha3_384_Final(&sha3Cpy, hashCpy), 0);
    ExpectIntEQ(XMEMCMP(hash, hashCpy, sizeof(hash)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_384_Copy(NULL, &sha3), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_384_Copy(&sha3Cpy, NULL), BAD_FUNC_ARG);

    wc_Sha3_384_Free(&sha3);
    wc_Sha3_384_Free(&sha3Cpy);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha3_384_Copy */


/*
 *  Testing wc_Sha3_512_Copy()
 */
static int test_wc_Sha3_512_Copy(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_512)
    wc_Sha3     sha3, sha3Cpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[WC_SHA3_512_DIGEST_SIZE];
    byte        hashCpy[WC_SHA3_512_DIGEST_SIZE];

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));
    XMEMSET(&sha3, 0, sizeof(wc_Sha3));
    XMEMSET(&sha3Cpy, 0, sizeof(wc_Sha3));

    ExpectIntEQ(wc_InitSha3_512(&sha3, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_InitSha3_512(&sha3Cpy, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Sha3_512_Update(&sha3, (byte*)msg, msglen), 0);
    ExpectIntEQ(wc_Sha3_512_Copy(&sha3Cpy, &sha3), 0);
    ExpectIntEQ(wc_Sha3_512_Final(&sha3, hash), 0);
    ExpectIntEQ(wc_Sha3_512_Final(&sha3Cpy, hashCpy), 0);
    ExpectIntEQ(XMEMCMP(hash, hashCpy, sizeof(hash)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Sha3_512_Copy(NULL, &sha3), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sha3_512_Copy(&sha3Cpy, NULL), BAD_FUNC_ARG);

    wc_Sha3_512_Free(&sha3);
    wc_Sha3_512_Free(&sha3Cpy);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha3_512_Copy */
/*
 * Unit test function for wc_Sha3_GetFlags()
 */
static int test_wc_Sha3_GetFlags(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SHA3) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha3 sha3;
    word32  flags = 0;

    /* Initialize */
    ExpectIntEQ(wc_InitSha3_224(&sha3, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Sha3_GetFlags(&sha3, &flags), 0);
    ExpectTrue((flags & WC_HASH_FLAG_ISCOPY) == 0);
    wc_Sha3_224_Free(&sha3);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha3_GetFlags */


static int test_wc_InitShake256(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHAKE256
    wc_Shake shake;

    ExpectIntEQ(wc_InitShake256(&shake, HEAP_HINT, testDevId), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_InitShake256(NULL, HEAP_HINT, testDevId), BAD_FUNC_ARG);

    wc_Shake256_Free(&shake);
#endif
    return EXPECT_RESULT();
}


static int testing_wc_Shake256_Update(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHAKE256
    wc_Shake shake;
    byte     msg[] = "Everybody's working for the weekend.";
    byte     msg2[] = "Everybody gets Friday off.";
    byte     msgCmp[] = "\x45\x76\x65\x72\x79\x62\x6f\x64\x79\x27\x73\x20"
                        "\x77\x6f\x72\x6b\x69\x6e\x67\x20\x66\x6f\x72\x20\x74"
                        "\x68\x65\x20\x77\x65\x65\x6b\x65\x6e\x64\x2e\x45\x76"
                        "\x65\x72\x79\x62\x6f\x64\x79\x20\x67\x65\x74\x73\x20"
                        "\x46\x72\x69\x64\x61\x79\x20\x6f\x66\x66\x2e";
    word32   msglen = sizeof(msg) - 1;
    word32   msg2len = sizeof(msg2);
    word32   msgCmplen = sizeof(msgCmp);

    ExpectIntEQ(wc_InitShake256(&shake, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Shake256_Update(&shake, msg, msglen), 0);
    ExpectIntEQ(XMEMCMP(msg, shake.t, msglen), 0);
    ExpectTrue(shake.i == msglen);

    ExpectIntEQ(wc_Shake256_Update(&shake, msg2, msg2len), 0);
    ExpectIntEQ(XMEMCMP(shake.t, msgCmp, msgCmplen), 0);

    /* Pass bad args. */
    ExpectIntEQ(wc_Shake256_Update(NULL, msg2, msg2len), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Shake256_Update(&shake, NULL, 5), BAD_FUNC_ARG);
    wc_Shake256_Free(&shake);

    ExpectIntEQ(wc_InitShake256(&shake, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Shake256_Update(&shake, NULL, 0), 0);
    ExpectIntEQ(wc_Shake256_Update(&shake, msg2, msg2len), 0);
    ExpectIntEQ(XMEMCMP(msg2, shake.t, msg2len), 0);
    wc_Shake256_Free(&shake);
#endif /* WOLFSSL_SHAKE256 */
    return EXPECT_RESULT();
}

static int test_wc_Shake256_Final(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHAKE256
    wc_Shake    shake;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x4d\x8c\x2d\xd2\x43\x5a\x01\x28\xee\xfb\xb8\xc3\x6f"
                         "\x6f\x87\x13\x3a\x79\x11\xe1\x8d\x97\x9e\xe1\xae\x6b"
                         "\xe5\xd4\xfd\x2e\x33\x29\x40\xd8\x68\x8a\x4e\x6a\x59"
                         "\xaa\x80\x60\xf1\xf9\xbc\x99\x6c\x05\xac\xa3\xc6\x96"
                         "\xa8\xb6\x62\x79\xdc\x67\x2c\x74\x0b\xb2\x24\xec\x37"
                         "\xa9\x2b\x65\xdb\x05\x39\xc0\x20\x34\x55\xf5\x1d\x97"
                         "\xcc\xe4\xcf\xc4\x91\x27\xd7\x26\x0a\xfc\x67\x3a\xf2"
                         "\x08\xba\xf1\x9b\xe2\x12\x33\xf3\xde\xbe\x78\xd0\x67"
                         "\x60\xcf\xa5\x51\xee\x1e\x07\x91\x41\xd4";
    byte        hash[114];

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ExpectIntEQ(wc_InitShake256(&shake, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_Shake256_Update(&shake, (byte*)msg, (word32)XSTRLEN(msg)),
        0);
    ExpectIntEQ(wc_Shake256_Final(&shake, hash, (word32)sizeof(hash)), 0);
    ExpectIntEQ(XMEMCMP(expOut, hash, (word32)sizeof(hash)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Shake256_Final(NULL, hash, (word32)sizeof(hash)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Shake256_Final(&shake, NULL, (word32)sizeof(hash)),
        BAD_FUNC_ARG);

    wc_Shake256_Free(&shake);
#endif
    return EXPECT_RESULT();
}
/*
 *  Testing wc_Shake256_Copy()
 */
static int test_wc_Shake256_Copy(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHAKE256
    wc_Shake    shake, shakeCpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[144];
    byte        hashCpy[144];
    word32      hashLen = sizeof(hash);
    word32      hashLenCpy = sizeof(hashCpy);

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));

    ExpectIntEQ(wc_InitShake256(&shake, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_InitShake256(&shakeCpy, HEAP_HINT, testDevId), 0);

    ExpectIntEQ(wc_Shake256_Update(&shake, (byte*)msg, msglen), 0);
    ExpectIntEQ(wc_Shake256_Copy(&shakeCpy, &shake), 0);
    ExpectIntEQ(wc_Shake256_Final(&shake, hash, hashLen), 0);
    ExpectIntEQ(wc_Shake256_Final(&shakeCpy, hashCpy, hashLenCpy), 0);
    ExpectIntEQ(XMEMCMP(hash, hashCpy, sizeof(hash)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Shake256_Copy(NULL, &shake), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Shake256_Copy(&shakeCpy, NULL), BAD_FUNC_ARG);

    wc_Shake256_Free(&shake);
    wc_Shake256_Free(&shakeCpy);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Shake256_Copy */
/*
 * Unit test function for wc_Shake256Hash()
 */
static int test_wc_Shake256Hash(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SHAKE256
    const byte data[] = { /* Hello World */
        0x48,0x65,0x6c,0x6c,0x6f,0x20,0x57,0x6f,
        0x72,0x6c,0x64
    };
    word32     len = sizeof(data);
    byte       hash[144];
    word32     hashLen = sizeof(hash);

    ExpectIntEQ(wc_Shake256Hash(data, len, hash, hashLen), 0);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_Shake256Hash */


/*
 *  Testing wc_InitSm3(), wc_Sm3Free()
 */
static int test_wc_InitSm3Free(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SM3
    wc_Sm3 sm3;

    /* Invalid Parameters */
    ExpectIntEQ(wc_InitSm3(NULL, NULL, INVALID_DEVID), BAD_FUNC_ARG);

    /* Valid Parameters */
    ExpectIntEQ(wc_InitSm3(&sm3, NULL, INVALID_DEVID), 0);

    wc_Sm3Free(NULL);
    wc_Sm3Free(&sm3);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_InitSm3 */

/*
 *  Testing wc_Sm3Update(), wc_Sm3Final()
 */
static int test_wc_Sm3UpdateFinal(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SM3
    wc_Sm3 sm3;
    byte data[WC_SM3_BLOCK_SIZE * 4];
    byte hash[WC_SM3_DIGEST_SIZE];
    byte calcHash[WC_SM3_DIGEST_SIZE];
    byte expHash[WC_SM3_DIGEST_SIZE] = {
        0x38, 0x48, 0x15, 0xa7, 0x0e, 0xae, 0x0b, 0x27,
        0x5c, 0xde, 0x9d, 0xa5, 0xd1, 0xa4, 0x30, 0xa1,
        0xca, 0xd4, 0x54, 0x58, 0x44, 0xa2, 0x96, 0x1b,
        0xd7, 0x14, 0x80, 0x3f, 0x80, 0x1a, 0x07, 0xb6
    };
    word32 chunk;
    word32 i;

    XMEMSET(data, 0, sizeof(data));

    ExpectIntEQ(wc_InitSm3(&sm3, NULL, INVALID_DEVID), 0);

    /* Invalid Parameters */
    ExpectIntEQ(wc_Sm3Update(NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3Update(&sm3, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3Update(NULL, data, 1), BAD_FUNC_ARG);

    /* Valid Parameters */
    ExpectIntEQ(wc_Sm3Update(&sm3, NULL, 0), 0);
    ExpectIntEQ(wc_Sm3Update(&sm3, data, 1), 0);
    ExpectIntEQ(wc_Sm3Update(&sm3, data, 1), 0);
    ExpectIntEQ(wc_Sm3Update(&sm3, data, WC_SM3_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_Sm3Update(&sm3, data, WC_SM3_BLOCK_SIZE - 2), 0);
    ExpectIntEQ(wc_Sm3Update(&sm3, data, WC_SM3_BLOCK_SIZE * 2), 0);
    /* Ensure too many bytes for lengths. */
    ExpectIntEQ(wc_Sm3Update(&sm3, data, WC_SM3_PAD_SIZE), 0);

    /* Invalid Parameters */
    ExpectIntEQ(wc_Sm3Final(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3Final(&sm3, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3Final(NULL, hash), BAD_FUNC_ARG);

    /* Valid Parameters */
    ExpectIntEQ(wc_Sm3Final(&sm3, hash), 0);
    ExpectBufEQ(hash, expHash, WC_SM3_DIGEST_SIZE);

    /* Chunk tests. */
    ExpectIntEQ(wc_Sm3Update(&sm3, data, sizeof(data)), 0);
    ExpectIntEQ(wc_Sm3Final(&sm3, calcHash), 0);
    for (chunk = 1; chunk <= WC_SM3_BLOCK_SIZE + 1; chunk++) {
        for (i = 0; i + chunk <= (word32)sizeof(data); i += chunk) {
            ExpectIntEQ(wc_Sm3Update(&sm3, data + i, chunk), 0);
        }
        if (i < (word32)sizeof(data)) {
            ExpectIntEQ(wc_Sm3Update(&sm3, data + i, (word32)sizeof(data) - i),
                0);
        }
        ExpectIntEQ(wc_Sm3Final(&sm3, hash), 0);
        ExpectBufEQ(hash, calcHash, WC_SM3_DIGEST_SIZE);
    }

    /* Not testing when the low 32-bit length overflows. */

    wc_Sm3Free(&sm3);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_Sm3Update */

/*
 *  Testing wc_Sm3GetHash()
 */
static int test_wc_Sm3GetHash(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_SM3
    wc_Sm3 sm3;
    byte hash[WC_SM3_DIGEST_SIZE];
    byte calcHash[WC_SM3_DIGEST_SIZE];
    byte data[WC_SM3_BLOCK_SIZE];

    XMEMSET(data, 0, sizeof(data));

    ExpectIntEQ(wc_InitSm3(&sm3, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_Sm3Final(&sm3, calcHash), 0);

    /* Invalid Parameters */
    ExpectIntEQ(wc_Sm3GetHash(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3GetHash(&sm3, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3GetHash(NULL, hash), BAD_FUNC_ARG);

    /* Valid Parameters */
    ExpectIntEQ(wc_Sm3GetHash(&sm3, hash), 0);
    ExpectBufEQ(hash, calcHash, WC_SM3_DIGEST_SIZE);

    /* With update. */
    ExpectIntEQ(wc_Sm3Update(&sm3, data, sizeof(data)), 0);
    ExpectIntEQ(wc_Sm3GetHash(&sm3, hash), 0);
    ExpectIntEQ(wc_Sm3Final(&sm3, calcHash), 0);
    ExpectBufEQ(hash, calcHash, WC_SM3_DIGEST_SIZE);

    wc_Sm3Free(&sm3);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_Sm3Update */

/*
 *  Testing wc_Sm3Copy()
 */
static int test_wc_Sm3Copy(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SM3) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sm3 sm3;
    wc_Sm3 sm3Copy;
    byte hash[WC_SM3_DIGEST_SIZE];
    byte hashCopy[WC_SM3_DIGEST_SIZE];
    byte data[WC_SM3_BLOCK_SIZE + 1];
    int i;

    ExpectIntEQ(wc_InitSm3(&sm3, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_InitSm3(&sm3Copy, NULL, INVALID_DEVID), 0);

    /* Invalid Parameters */
    ExpectIntEQ(wc_Sm3Copy(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3Copy(&sm3, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3Copy(NULL, &sm3Copy), BAD_FUNC_ARG);

    /* Valid Parameters */
    ExpectIntEQ(wc_Sm3Copy(&sm3, &sm3Copy), 0);

    /* Ensure all parts of data updated during hashing are copied. */
    for (i = 0; i < WC_SM3_BLOCK_SIZE + 1; i++) {
        ExpectIntEQ(wc_Sm3Update(&sm3, data, i), 0);
        ExpectIntEQ(wc_Sm3Copy(&sm3, &sm3Copy), 0);
        ExpectIntEQ(wc_Sm3Update(&sm3, data, 1), 0);
        ExpectIntEQ(wc_Sm3Update(&sm3Copy, data, 1), 0);
        ExpectIntEQ(wc_Sm3Final(&sm3, hash), 0);
        ExpectIntEQ(wc_Sm3Final(&sm3Copy, hashCopy), 0);
        ExpectBufEQ(hash, hashCopy, WC_SM3_DIGEST_SIZE);
    }

    wc_Sm3Free(&sm3Copy);
    wc_Sm3Free(&sm3);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_Sm3Copy */

/*
 * Testing wc_Sm3FinalRaw()
 */
static int test_wc_Sm3FinalRaw(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SM3) && !defined(HAVE_SELFTEST) && \
    !defined(WOLFSSL_DEVCRYPTO) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 3))) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    wc_Sm3 sm3;
    byte hash1[WC_SM3_DIGEST_SIZE];
    byte hash2[WC_SM3_DIGEST_SIZE];
    byte hash3[WC_SM3_DIGEST_SIZE];
    byte* hash_test[3] = { hash1, hash2, hash3 };
    int times;
    int i;

    XMEMSET(&sm3, 0, sizeof(sm3));

    /* Initialize */
    ExpectIntEQ(wc_InitSm3(&sm3, NULL, INVALID_DEVID), 0);

    /* Invalid Parameters */
    ExpectIntEQ(wc_Sm3FinalRaw(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3FinalRaw(&sm3, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3FinalRaw(NULL, hash1), BAD_FUNC_ARG);

    times = sizeof(hash_test) / sizeof(byte*);
    for (i = 0; i < times; i++) {
        ExpectIntEQ(wc_Sm3FinalRaw(&sm3, hash_test[i]), 0);
    }

    wc_Sm3Free(&sm3);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sm3FinalRaw */
/*
 *  Testing wc_Sm3GetFlags, wc_Sm3SetFlags()
 */
static int test_wc_Sm3GetSetFlags(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SM3) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sm3 sm3;
    wc_Sm3 sm3Copy;
    word32 flags = 0;

    ExpectIntEQ(wc_InitSm3(&sm3, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_InitSm3(&sm3Copy, NULL, INVALID_DEVID), 0);

    ExpectIntEQ(wc_Sm3GetFlags(NULL, &flags), 0);
    ExpectIntEQ(flags, 0);
    ExpectIntEQ(wc_Sm3SetFlags(NULL, WC_HASH_FLAG_WILLCOPY), 0);
    ExpectIntEQ(wc_Sm3GetFlags(NULL, &flags), 0);
    ExpectIntEQ(flags, 0);
    ExpectIntEQ(wc_Sm3GetFlags(&sm3, &flags), 0);
    ExpectIntEQ(flags, 0);
    ExpectIntEQ(wc_Sm3SetFlags(&sm3, WC_HASH_FLAG_WILLCOPY), 0);
    ExpectIntEQ(wc_Sm3GetFlags(&sm3, &flags), 0);
    ExpectIntEQ(flags, WC_HASH_FLAG_WILLCOPY);

    ExpectIntEQ(wc_Sm3Copy(&sm3, &sm3Copy), 0);
    ExpectIntEQ(wc_Sm3GetFlags(&sm3Copy, &flags), 0);
    ExpectIntEQ(flags, WC_HASH_FLAG_ISCOPY | WC_HASH_FLAG_WILLCOPY);

    wc_Sm3Free(&sm3Copy);
    wc_Sm3Free(&sm3);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_Sm3Update */

/*
 *  Testing wc_Sm3Hash()
 */
static int test_wc_Sm3Hash(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SM3) && defined(WOLFSSL_HASH_FLAGS)
    byte data[WC_SM3_BLOCK_SIZE];
    byte hash[WC_SM3_DIGEST_SIZE];

    /* Invalid parameters. */
    ExpectIntEQ(wc_Sm3Hash(NULL, sizeof(data), hash), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm3Hash(data, sizeof(data), NULL), BAD_FUNC_ARG);

    /* Valid parameters. */
    ExpectIntEQ(wc_Sm3Hash(data, sizeof(data), hash), 0);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_Sm3Hash */

/*
 * Test function for wc_HmacSetKey
 */
static int test_wc_Md5HmacSetKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && !defined(NO_MD5)
    Hmac hmac;
    int ret, times, itr;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
    };
    times = sizeof(keys) / sizeof(char*);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);

    for (itr = 0; itr < times; itr++) {
        ret = wc_HmacSetKey(&hmac, WC_MD5, (byte*)keys[itr],
            (word32)XSTRLEN(keys[itr]));
#if defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 5)
        wc_HmacFree(&hmac);
        ExpectIntEQ(ret, BAD_FUNC_ARG);
#else
        ExpectIntEQ(ret, 0);
#endif
    }

    /* Bad args. */
    ExpectIntEQ(wc_HmacSetKey(NULL, WC_MD5, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_MD5, NULL, (word32)XSTRLEN(keys[0])),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, 21, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);
    ret = wc_HmacSetKey(&hmac, WC_MD5, (byte*)keys[0], 0);
#if defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 5)
    ExpectIntEQ(ret, BAD_FUNC_ARG);
#elif defined(HAVE_FIPS)
    ExpectIntEQ(ret, HMAC_MIN_KEYLEN_E);
#else
    ExpectIntEQ(ret, 0);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Md5HmacSetKey */


/*
 * testing wc_HmacSetKey() on wc_Sha hash.
 */
static int test_wc_ShaHmacSetKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && !defined(NO_SHA)
    Hmac hmac;
    int ret, times, itr;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };

    times = sizeof(keys) / sizeof(char*);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);

    for (itr = 0; itr < times; itr++) {
        ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA, (byte*)keys[itr],
            (word32)XSTRLEN(keys[itr])), 0);
    }

    /* Bad args. */
    ExpectIntEQ(wc_HmacSetKey(NULL, WC_SHA, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA, NULL, (word32)XSTRLEN(keys[0])),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, 21, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);

    ret = wc_HmacSetKey(&hmac, WC_SHA, (byte*)keys[0], 0);
#ifdef HAVE_FIPS
    ExpectIntEQ(ret, HMAC_MIN_KEYLEN_E);
#else
    ExpectIntEQ(ret, 0);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ShaHmacSetKey() */

/*
 * testing wc_HmacSetKey() on Sha224 hash.
 */
static int test_wc_Sha224HmacSetKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA224)
    Hmac hmac;
    int ret, times, itr;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };
    times = sizeof(keys) / sizeof(char*);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);

    for (itr = 0; itr < times; itr++) {
        ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA224, (byte*)keys[itr],
            (word32)XSTRLEN(keys[itr])), 0);
    }

    /* Bad args. */
    ExpectIntEQ(wc_HmacSetKey(NULL, WC_SHA224, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA224, NULL, (word32)XSTRLEN(keys[0])),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, 21, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);
    ret = wc_HmacSetKey(&hmac, WC_SHA224, (byte*)keys[0], 0);
#ifdef HAVE_FIPS
    ExpectIntEQ(ret, HMAC_MIN_KEYLEN_E);
#else
    ExpectIntEQ(ret, 0);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha224HmacSetKey() */

 /*
  * testing wc_HmacSetKey() on Sha256 hash
  */
static int test_wc_Sha256HmacSetKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && !defined(NO_SHA256)
    Hmac hmac;
    int ret, times, itr;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };
    times = sizeof(keys) / sizeof(char*);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);

    for (itr = 0; itr < times; itr++) {
        ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA256, (byte*)keys[itr],
            (word32)XSTRLEN(keys[itr])), 0);
    }

    /* Bad args. */
    ExpectIntEQ(wc_HmacSetKey(NULL, WC_SHA256, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA256, NULL, (word32)XSTRLEN(keys[0])),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, 21, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);
    ret = wc_HmacSetKey(&hmac, WC_SHA256, (byte*)keys[0], 0);
#ifdef HAVE_FIPS
    ExpectIntEQ(ret, HMAC_MIN_KEYLEN_E);
#else
    ExpectIntEQ(ret, 0);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha256HmacSetKey() */


/*
 * testing wc_HmacSetKey on Sha384 hash.
 */
static int test_wc_Sha384HmacSetKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA384)
    Hmac hmac;
    int ret, times, itr;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };
    times = sizeof(keys) / sizeof(char*);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);

    for (itr = 0; itr < times; itr++) {
        ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA384, (byte*)keys[itr],
            (word32)XSTRLEN(keys[itr])), 0);
    }

    /* Bad args. */
    ExpectIntEQ(wc_HmacSetKey(NULL, WC_SHA384, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA384, NULL, (word32)XSTRLEN(keys[0])),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacSetKey(&hmac, 21, (byte*)keys[0],
        (word32)XSTRLEN(keys[0])), BAD_FUNC_ARG);
    ret = wc_HmacSetKey(&hmac, WC_SHA384, (byte*)keys[0], 0);
#ifdef HAVE_FIPS
    ExpectIntEQ(ret, HMAC_MIN_KEYLEN_E);
#else
    ExpectIntEQ(ret, 0);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha384HmacSetKey() */


/*
 * testing wc_HmacUpdate on wc_Md5 hash.
 */
static int test_wc_Md5HmacUpdate(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && !defined(NO_MD5) && !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 5))
    Hmac hmac;
    testVector a, b;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);
    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_MD5, (byte*)keys,
        (word32)XSTRLEN(keys)), 0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen), 0);
    /* Update Hmac. */
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacUpdate(&hmac, NULL, (word32)a.inLen), BAD_FUNC_ARG);

    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, 0), 0);

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Md5HmacUpdate */

/*
 * testing wc_HmacUpdate on SHA hash.
 */
static int test_wc_ShaHmacUpdate(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && !defined(NO_SHA)
    Hmac hmac;
    testVector a, b;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);
    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA, (byte*)keys,
        (word32)XSTRLEN(keys)), 0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen), 0);
    /* Update Hmac. */
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacUpdate(&hmac, NULL, (word32)a.inLen), BAD_FUNC_ARG);

    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, 0), 0);

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ShaHmacUpdate */

/*
 * testing wc_HmacUpdate on SHA224 hash.
 */
static int test_wc_Sha224HmacUpdate(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA224)
    Hmac hmac;
    testVector a, b;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);
    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA224, (byte*)keys,
        (word32)XSTRLEN(keys)), 0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen), 0);
    /* Update Hmac. */
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacUpdate(&hmac, NULL, (word32)a.inLen), BAD_FUNC_ARG);

    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, 0), 0);

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha224HmacUpdate */

/*
 * testing wc_HmacUpdate on SHA256 hash.
 */
static int test_wc_Sha256HmacUpdate(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && !defined(NO_SHA256)
    Hmac hmac;
    testVector a, b;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);
    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA256, (byte*)keys,
        (word32)XSTRLEN(keys)), 0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen), 0);
    /* Update Hmac. */
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacUpdate(&hmac, NULL, (word32)a.inLen), BAD_FUNC_ARG);

    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, 0), 0);

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha256HmacUpdate */

/*
 * testing wc_HmacUpdate on SHA384  hash.
 */
static int test_wc_Sha384HmacUpdate(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA384)
    Hmac hmac;
    testVector a, b;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);
    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA384, (byte*)keys,
        (word32)XSTRLEN(keys)), 0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen), 0);
    /* Update Hmac. */
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_HmacUpdate(&hmac, NULL, (word32)a.inLen), BAD_FUNC_ARG);

    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, 0), 0);

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha384HmacUpdate */

/*
 * Testing wc_HmacFinal() with MD5
 */

static int test_wc_Md5HmacFinal(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && !defined(NO_MD5) && !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 5))
    Hmac hmac;
    byte hash[WC_MD5_DIGEST_SIZE];
    testVector a;
    const char* key;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\x92\x94\x72\x7a\x36\x38\xbb\x1c\x13\xf4\x8e\xf8\x15\x8b\xfc"
               "\x9d";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_MD5, (byte*)key, (word32)XSTRLEN(key)),
        0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_HmacFinal(&hmac, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_MD5_DIGEST_SIZE), 0);

    /* Try bad parameters. */
    ExpectIntEQ(wc_HmacFinal(NULL, hash), BAD_FUNC_ARG);
#ifndef HAVE_FIPS
    ExpectIntEQ(wc_HmacFinal(&hmac, NULL), BAD_FUNC_ARG);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Md5HmacFinal */

/*
 * Testing wc_HmacFinal() with SHA
 */
static int test_wc_ShaHmacFinal(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && !defined(NO_SHA)
    Hmac hmac;
    byte hash[WC_SHA_DIGEST_SIZE];
    testVector a;
    const char* key;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\xb6\x17\x31\x86\x55\x05\x72\x64\xe2\x8b\xc0\xb6\xfb\x37\x8c"
               "\x8e\xf1\x46\xbe\x00";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA, (byte*)key, (word32)XSTRLEN(key)),
        0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_HmacFinal(&hmac, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA_DIGEST_SIZE), 0);

    /* Try bad parameters. */
    ExpectIntEQ(wc_HmacFinal(NULL, hash), BAD_FUNC_ARG);
#ifndef HAVE_FIPS
    ExpectIntEQ(wc_HmacFinal(&hmac, NULL), BAD_FUNC_ARG);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ShaHmacFinal */


/*
 * Testing wc_HmacFinal() with SHA224
 */
static int test_wc_Sha224HmacFinal(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA224)
    Hmac hmac;
    byte hash[WC_SHA224_DIGEST_SIZE];
    testVector a;
    const char* key;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\x89\x6f\xb1\x12\x8a\xbb\xdf\x19\x68\x32\x10\x7c\xd4\x9d\xf3"
               "\x3f\x47\xb4\xb1\x16\x99\x12\xba\x4f\x53\x68\x4b\x22";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA224, (byte*)key,
        (word32)XSTRLEN(key)), 0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_HmacFinal(&hmac, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA224_DIGEST_SIZE), 0);

    /* Try bad parameters. */
    ExpectIntEQ(wc_HmacFinal(NULL, hash), BAD_FUNC_ARG);
#ifndef HAVE_FIPS
    ExpectIntEQ(wc_HmacFinal(&hmac, NULL), BAD_FUNC_ARG);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha224HmacFinal */

/*
 * Testing wc_HmacFinal() with SHA256
 */
static int test_wc_Sha256HmacFinal(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && !defined(NO_SHA256)
    Hmac hmac;
    byte hash[WC_SHA256_DIGEST_SIZE];
    testVector a;
    const char* key;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\xb0\x34\x4c\x61\xd8\xdb\x38\x53\x5c\xa8\xaf\xce\xaf\x0b\xf1"
               "\x2b\x88\x1d\xc2\x00\xc9\x83\x3d\xa7\x26\xe9\x37\x6c\x2e\x32"
               "\xcf\xf7";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA256, (byte*)key,
        (word32)XSTRLEN(key)), 0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_HmacFinal(&hmac, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA256_DIGEST_SIZE), 0);

    /* Try bad parameters. */
    ExpectIntEQ(wc_HmacFinal(NULL, hash), BAD_FUNC_ARG);
#ifndef HAVE_FIPS
    ExpectIntEQ(wc_HmacFinal(&hmac, NULL), BAD_FUNC_ARG);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha256HmacFinal */

/*
 * Testing wc_HmacFinal() with SHA384
 */
static int test_wc_Sha384HmacFinal(void)
{
    EXPECT_DECLS;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA384)
    Hmac hmac;
    byte hash[WC_SHA384_DIGEST_SIZE];
    testVector a;
    const char* key;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\xaf\xd0\x39\x44\xd8\x48\x95\x62\x6b\x08\x25\xf4\xab\x46\x90"
               "\x7f\x15\xf9\xda\xdb\xe4\x10\x1e\xc6\x82\xaa\x03\x4c\x7c\xeb"
               "\xc5\x9c\xfa\xea\x9e\xa9\x07\x6e\xde\x7f\x4a\xf1\x52\xe8\xb2"
               "\xfa\x9c\xb6";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    ExpectIntEQ(wc_HmacInit(&hmac, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_HmacSetKey(&hmac, WC_SHA384, (byte*)key,
        (word32)XSTRLEN(key)), 0);
    ExpectIntEQ(wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen), 0);
    ExpectIntEQ(wc_HmacFinal(&hmac, hash), 0);
    ExpectIntEQ(XMEMCMP(hash, a.output, WC_SHA384_DIGEST_SIZE), 0);

    /* Try bad parameters. */
    ExpectIntEQ(wc_HmacFinal(NULL, hash), BAD_FUNC_ARG);
#ifndef HAVE_FIPS
    ExpectIntEQ(wc_HmacFinal(&hmac, NULL), BAD_FUNC_ARG);
#endif

    wc_HmacFree(&hmac);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Sha384HmacFinal */



/*
 * Testing wc_InitCmac()
 */
static int test_wc_InitCmac(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CMAC) && !defined(NO_AES)
    Cmac        cmac1;
    Cmac        cmac2;
    Cmac        cmac3;
    /* AES 128 key. */
    byte        key1[] = "\x01\x02\x03\x04\x05\x06\x07\x08"
                         "\x09\x10\x11\x12\x13\x14\x15\x16";
    /* AES 192 key. */
    byte        key2[] = "\x01\x02\x03\x04\x05\x06\x07\x08"
                         "\x09\x01\x11\x12\x13\x14\x15\x16"
                         "\x01\x02\x03\x04\x05\x06\x07\x08";
    /* AES 256 key. */
    byte        key3[] = "\x01\x02\x03\x04\x05\x06\x07\x08"
                         "\x09\x01\x11\x12\x13\x14\x15\x16"
                         "\x01\x02\x03\x04\x05\x06\x07\x08"
                         "\x09\x01\x11\x12\x13\x14\x15\x16";
    word32      key1Sz = (word32)sizeof(key1) - 1;
    word32      key2Sz = (word32)sizeof(key2) - 1;
    word32      key3Sz = (word32)sizeof(key3) - 1;
    int         type   = WC_CMAC_AES;

    (void)key1;
    (void)key1Sz;
    (void)key2;
    (void)key2Sz;

    XMEMSET(&cmac1, 0, sizeof(Cmac));
    XMEMSET(&cmac2, 0, sizeof(Cmac));
    XMEMSET(&cmac3, 0, sizeof(Cmac));

#ifdef WOLFSSL_AES_128
    ExpectIntEQ(wc_InitCmac(&cmac1, key1, key1Sz, type, NULL), 0);
#endif
#ifdef WOLFSSL_AES_192
    wc_AesFree(&cmac1.aes);
    ExpectIntEQ(wc_InitCmac(&cmac2, key2, key2Sz, type, NULL), 0);
#endif
#ifdef WOLFSSL_AES_256
    wc_AesFree(&cmac2.aes);
    ExpectIntEQ(wc_InitCmac(&cmac3, key3, key3Sz, type, NULL), 0);
#endif

    wc_AesFree(&cmac3.aes);
    /* Test bad args. */
    ExpectIntEQ(wc_InitCmac(NULL, key3, key3Sz, type, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitCmac(&cmac3, NULL, key3Sz, type, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitCmac(&cmac3, key3, 0, type, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_InitCmac(&cmac3, key3, key3Sz, 0, NULL), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_InitCmac */


/*
 * Testing wc_CmacUpdate()
 */
static int test_wc_CmacUpdate(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CMAC) && !defined(NO_AES) && defined(WOLFSSL_AES_128)
    Cmac        cmac;
    byte        key[] = {
        0x64, 0x4c, 0xbf, 0x12, 0x85, 0x9d, 0xf0, 0x55,
        0x7e, 0xa9, 0x1f, 0x08, 0xe0, 0x51, 0xff, 0x27
    };
    byte        in[] = "\xe2\xb4\xb6\xf9\x48\x44\x02\x64"
                       "\x5c\x47\x80\x9e\xd5\xa8\x3a\x17"
                       "\xb3\x78\xcf\x85\x22\x41\x74\xd9"
                       "\xa0\x97\x39\x71\x62\xf1\x8e\x8f"
                       "\xf4";
    word32      inSz  = (word32)sizeof(in) - 1;
    word32      keySz = (word32)sizeof(key);
    int         type  = WC_CMAC_AES;

    XMEMSET(&cmac, 0, sizeof(Cmac));

    ExpectIntEQ(wc_InitCmac(&cmac, key, keySz, type, NULL), 0);
    ExpectIntEQ(wc_CmacUpdate(&cmac, in, inSz), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_CmacUpdate(NULL, in, inSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_CmacUpdate(&cmac, NULL, 30), BAD_FUNC_ARG);
    wc_AesFree(&cmac.aes);
#endif
    return EXPECT_RESULT();
} /* END test_wc_CmacUpdate */


/*
 * Testing wc_CmacFinal()
 */
static int test_wc_CmacFinal(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CMAC) && !defined(NO_AES) && defined(WOLFSSL_AES_128)
    Cmac        cmac;
    byte        key[] = {
        0x64, 0x4c, 0xbf, 0x12, 0x85, 0x9d, 0xf0, 0x55,
        0x7e, 0xa9, 0x1f, 0x08, 0xe0, 0x51, 0xff, 0x27
    };
    byte        msg[] = {
        0xe2, 0xb4, 0xb6, 0xf9, 0x48, 0x44, 0x02, 0x64,
        0x5c, 0x47, 0x80, 0x9e, 0xd5, 0xa8, 0x3a, 0x17,
        0xb3, 0x78, 0xcf, 0x85, 0x22, 0x41, 0x74, 0xd9,
        0xa0, 0x97, 0x39, 0x71, 0x62, 0xf1, 0x8e, 0x8f,
        0xf4
    };
    /* Test vectors from CMACGenAES128.rsp from
     * http://csrc.nist.gov/groups/STM/cavp/block-cipher-modes.html#cmac
     * Per RFC4493 truncation of lsb is possible.
     */
    byte        expMac[] = {
        0x4e, 0x6e, 0xc5, 0x6f, 0xf9, 0x5d, 0x0e, 0xae,
        0x1c, 0xf8, 0x3e, 0xfc, 0xf4, 0x4b, 0xeb
    };
    byte        mac[AES_BLOCK_SIZE];
    word32      msgSz    = (word32)sizeof(msg);
    word32      keySz    = (word32)sizeof(key);
    word32      macSz    = sizeof(mac);
    word32      badMacSz = 17;
    int         expMacSz = sizeof(expMac);
    int         type     = WC_CMAC_AES;

    XMEMSET(&cmac, 0, sizeof(Cmac));
    XMEMSET(mac, 0, macSz);

    ExpectIntEQ(wc_InitCmac(&cmac, key, keySz, type, NULL), 0);
    ExpectIntEQ(wc_CmacUpdate(&cmac, msg, msgSz), 0);

    ExpectIntEQ(wc_CmacFinal(&cmac, mac, &macSz), 0);
    ExpectIntEQ(XMEMCMP(mac, expMac, expMacSz), 0);

    /* Pass in bad args. */
    ExpectIntEQ(wc_CmacFinal(NULL, mac, &macSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_CmacFinal(&cmac, NULL, &macSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_CmacFinal(&cmac, mac, &badMacSz), BUFFER_E);
#endif
    return EXPECT_RESULT();
} /* END test_wc_CmacFinal */


/*
 * Testing wc_AesCmacGenerate() && wc_AesCmacVerify()
 */
static int test_wc_AesCmacGenerate(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CMAC) && !defined(NO_AES) && defined(WOLFSSL_AES_128)
    byte        key[] = {
        0x26, 0xef, 0x8b, 0x40, 0x34, 0x11, 0x7d, 0x9e,
        0xbe, 0xc0, 0xc7, 0xfc, 0x31, 0x08, 0x54, 0x69
    };
    byte        msg[]    = "\x18\x90\x49\xef\xfd\x7c\xf9\xc8"
                           "\xf3\x59\x65\xbc\xb0\x97\x8f\xd4";
    byte        expMac[] = "\x29\x5f\x2f\x71\xfc\x58\xe6\xf6"
                           "\x3d\x32\x65\x4c\x66\x23\xc5";
    byte        mac[AES_BLOCK_SIZE];
    word32      keySz    = sizeof(key);
    word32      macSz    = sizeof(mac);
    word32      msgSz    = sizeof(msg) - 1;
    word32      expMacSz = sizeof(expMac) - 1;

    XMEMSET(mac, 0, macSz);

    ExpectIntEQ(wc_AesCmacGenerate(mac, &macSz, msg, msgSz, key, keySz), 0);
    ExpectIntEQ(XMEMCMP(mac, expMac, expMacSz), 0);

    /* Pass in bad args. */
    ExpectIntEQ(wc_AesCmacGenerate(NULL, &macSz, msg, msgSz, key, keySz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCmacGenerate(mac, &macSz, msg, msgSz, NULL, keySz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCmacGenerate(mac, &macSz, msg, msgSz, key, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCmacGenerate(mac, &macSz, NULL, msgSz, key, keySz),
        BAD_FUNC_ARG);

    ExpectIntEQ(wc_AesCmacVerify(mac, macSz, msg, msgSz, key, keySz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_AesCmacVerify(NULL, macSz, msg, msgSz, key, keySz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCmacVerify(mac, 0, msg, msgSz, key, keySz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCmacVerify(mac, macSz, msg, msgSz, NULL, keySz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCmacVerify(mac, macSz, msg, msgSz, key, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCmacVerify(mac, macSz, NULL, msgSz, key, keySz),
        BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();

} /* END test_wc_AesCmacGenerate */


/*
 * Testing streaming AES-GCM API.
 */
static int test_wc_AesGcmStream(void)
{
    EXPECT_DECLS;
#if !defined(NO_AES) && defined(WOLFSSL_AES_128) && defined(HAVE_AESGCM) && \
    defined(WOLFSSL_AESGCM_STREAM)
    int i;
    WC_RNG rng[1];
    Aes aesEnc[1];
    Aes aesDec[1];
    byte tag[AES_BLOCK_SIZE];
    byte in[AES_BLOCK_SIZE * 3 + 2] = { 0, };
    byte out[AES_BLOCK_SIZE * 3 + 2];
    byte plain[AES_BLOCK_SIZE * 3 + 2];
    byte aad[AES_BLOCK_SIZE * 3 + 2] = { 0, };
    byte key[AES_128_KEY_SIZE] = { 0, };
    byte iv[AES_IV_SIZE] = { 1, };
    byte ivOut[AES_IV_SIZE];
    static const byte expTagAAD1[AES_BLOCK_SIZE] = {
        0x6c, 0x35, 0xe6, 0x7f, 0x59, 0x9e, 0xa9, 0x2f,
        0x27, 0x2d, 0x5f, 0x8e, 0x7e, 0x42, 0xd3, 0x05
    };
    static const byte expTagPlain1[AES_BLOCK_SIZE] = {
        0x24, 0xba, 0x57, 0x95, 0xd0, 0x27, 0x9e, 0x78,
        0x3a, 0x88, 0x4c, 0x0a, 0x5d, 0x50, 0x23, 0xd1
    };
    static const byte expTag[AES_BLOCK_SIZE] = {
        0x22, 0x91, 0x70, 0xad, 0x42, 0xc3, 0xad, 0x96,
        0xe0, 0x31, 0x57, 0x60, 0xb7, 0x92, 0xa3, 0x6d
    };

    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&aesEnc, 0, sizeof(Aes));
    XMEMSET(&aesDec, 0, sizeof(Aes));

    /* Create a random for generating IV/nonce. */
    ExpectIntEQ(wc_InitRng(rng), 0);

    /* Initialize data structures. */
    ExpectIntEQ(wc_AesInit(aesEnc, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_AesInit(aesDec, NULL, INVALID_DEVID), 0);

    /* BadParameters to streaming init. */
    ExpectIntEQ(wc_AesGcmEncryptInit(NULL, NULL, 0, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptInit(NULL, NULL, 0, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptInit(aesEnc, NULL, AES_128_KEY_SIZE, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptInit(aesEnc, NULL, 0, NULL, GCM_NONCE_MID_SZ),
        BAD_FUNC_ARG);

    /* Bad parameters to encrypt update. */
    ExpectIntEQ(wc_AesGcmEncryptUpdate(NULL, NULL, NULL, 0, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 1, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, in, 1, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out, NULL, 1, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, NULL, 1),
        BAD_FUNC_ARG);
    /* Bad parameters to decrypt update. */
    ExpectIntEQ(wc_AesGcmDecryptUpdate(NULL, NULL, NULL, 0, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 1, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, in, 1, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, out, NULL, 1, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, NULL, 1),
        BAD_FUNC_ARG);

    /* Bad parameters to encrypt final. */
    ExpectIntEQ(wc_AesGcmEncryptFinal(NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncryptFinal(NULL, tag, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncryptFinal(NULL, NULL, AES_BLOCK_SIZE),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, NULL, AES_BLOCK_SIZE),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE + 1),
        BAD_FUNC_ARG);
    /* Bad parameters to decrypt final. */
    ExpectIntEQ(wc_AesGcmDecryptFinal(NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptFinal(NULL, tag, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptFinal(NULL, NULL, AES_BLOCK_SIZE),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptFinal(aesDec, NULL, AES_BLOCK_SIZE),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE + 1),
        BAD_FUNC_ARG);

    /* Check calling final before setting key fails. */
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, sizeof(tag)), MISSING_KEY);
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesDec, tag, sizeof(tag)), MISSING_KEY);
    /* Check calling update before setting key else fails. */
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, aad, 1),
        MISSING_KEY);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, aad, 1),
        MISSING_KEY);

    /* Set key but not IV. */
    ExpectIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), NULL, 0), 0);
    ExpectIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), NULL, 0), 0);
    /* Check calling final before setting IV fails. */
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, sizeof(tag)), MISSING_IV);
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesDec, tag, sizeof(tag)), MISSING_IV);
    /* Check calling update before setting IV else fails. */
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, aad, 1),
        MISSING_IV);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, aad, 1),
        MISSING_IV);

    /* Set IV using fixed part IV and external IV APIs. */
    ExpectIntEQ(wc_AesGcmSetIV(aesEnc, GCM_NONCE_MID_SZ, iv, AES_IV_FIXED_SZ,
        rng), 0);
    ExpectIntEQ(wc_AesGcmEncryptInit_ex(aesEnc, NULL, 0, ivOut,
        GCM_NONCE_MID_SZ), 0);
    ExpectIntEQ(wc_AesGcmSetExtIV(aesDec, ivOut, GCM_NONCE_MID_SZ), 0);
    ExpectIntEQ(wc_AesGcmInit(aesDec, NULL, 0, NULL, 0), 0);
    /* Encrypt and decrypt data. */
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out, in, 1, aad, 1), 0);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, plain, out, 1, aad, 1), 0);
    ExpectIntEQ(XMEMCMP(plain, in, 1), 0);
    /* Finalize and check tag matches. */
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Set key and IV through streaming init API. */
    ExpectIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), iv, AES_IV_SIZE), 0);
    ExpectIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), iv, AES_IV_SIZE), 0);
    /* Encrypt/decrypt one block and AAD of one block. */
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out, in, AES_BLOCK_SIZE, aad,
        AES_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, plain, out, AES_BLOCK_SIZE, aad,
        AES_BLOCK_SIZE), 0);
    ExpectIntEQ(XMEMCMP(plain, in, AES_BLOCK_SIZE), 0);
    /* Finalize and check tag matches. */
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Set key and IV through streaming init API. */
    ExpectIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), iv, AES_IV_SIZE), 0);
    ExpectIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), iv, AES_IV_SIZE), 0);
    /* No data to encrypt/decrypt one byte of AAD. */
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, aad, 1), 0);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, aad, 1), 0);
    /* Finalize and check tag matches. */
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    ExpectIntEQ(XMEMCMP(tag, expTagAAD1, AES_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Set key and IV through streaming init API. */
    ExpectIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), iv, AES_IV_SIZE), 0);
    ExpectIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), iv, AES_IV_SIZE), 0);
    /* Encrypt/decrypt one byte and no AAD. */
    ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out, in, 1, NULL, 0), 0);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, plain, out, 1, NULL, 0), 0);
    ExpectIntEQ(XMEMCMP(plain, in, 1), 0);
    /* Finalize and check tag matches. */
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    ExpectIntEQ(XMEMCMP(tag, expTagPlain1, AES_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Set key and IV through streaming init API. */
    ExpectIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), iv, AES_IV_SIZE), 0);
    ExpectIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), iv, AES_IV_SIZE), 0);
    /* Encryption AES is one byte at a time */
    for (i = 0; i < (int)sizeof(aad); i++) {
        ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, aad + i, 1),
            0);
    }
    for (i = 0; i < (int)sizeof(in); i++) {
        ExpectIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out + i, in + i, 1, NULL, 0),
            0);
    }
    /* Decryption AES is two bytes at a time */
    for (i = 0; i < (int)sizeof(aad); i += 2) {
        ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, aad + i, 2),
            0);
    }
    for (i = 0; i < (int)sizeof(aad); i += 2) {
        ExpectIntEQ(wc_AesGcmDecryptUpdate(aesDec, plain + i, out + i, 2, NULL,
            0), 0);
    }
    ExpectIntEQ(XMEMCMP(plain, in, sizeof(in)), 0);
    /* Finalize and check tag matches. */
    ExpectIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    ExpectIntEQ(XMEMCMP(tag, expTag, AES_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Check streaming encryption can be decrypted with one shot. */
    ExpectIntEQ(wc_AesGcmSetKey(aesDec, key, sizeof(key)), 0);
    ExpectIntEQ(wc_AesGcmDecrypt(aesDec, plain, out, sizeof(in), iv,
        AES_IV_SIZE, tag, AES_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(XMEMCMP(plain, in, sizeof(in)), 0);

    wc_AesFree(aesEnc);
    wc_AesFree(aesDec);
    wc_FreeRng(rng);
#endif
    return EXPECT_RESULT();
} /* END test_wc_AesGcmStream */


/*
 * Testing streaming SM4 API.
 */
static int test_wc_Sm4(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SM4
    EXPECT_DECLS;
    wc_Sm4 sm4;
#if defined(WOLFSSL_SM4_ECB) || defined(WOLFSSL_SM4_CBC) || \
    defined(WOLFSSL_SM4_CTR) || defined(WOLFSSL_SM4_CCM)
    unsigned char key[SM4_KEY_SIZE];
#endif
#if defined(WOLFSSL_SM4_CBC) || defined(WOLFSSL_SM4_CTR)
    unsigned char iv[SM4_IV_SIZE];
#endif

    /* Invalid parameters - wc_Sm4Init */
    ExpectIntEQ(wc_Sm4Init(NULL, NULL, INVALID_DEVID), BAD_FUNC_ARG);

    /* Valid cases - wc_Sm4Init */
    ExpectIntEQ(wc_Sm4Init(&sm4, NULL, INVALID_DEVID), 0);

#if defined(WOLFSSL_SM4_ECB) || defined(WOLFSSL_SM4_CBC) || \
    defined(WOLFSSL_SM4_CTR) || defined(WOLFSSL_SM4_CCM)
    XMEMSET(key, 0, sizeof(key));

    /* Invalid parameters - wc_Sm4SetKey. */
    ExpectIntEQ(wc_Sm4SetKey(NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetKey(&sm4, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetKey(NULL, key, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetKey(NULL, NULL, SM4_KEY_SIZE), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetKey(&sm4, key, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetKey(&sm4, NULL, SM4_KEY_SIZE), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetKey(NULL, key, SM4_KEY_SIZE), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetKey(&sm4, key, SM4_KEY_SIZE-1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetKey(&sm4, key, SM4_KEY_SIZE+1), BAD_FUNC_ARG);

    /* Valid cases - wc_Sm4SetKey. */
    ExpectIntEQ(wc_Sm4SetKey(&sm4, key, SM4_KEY_SIZE), 0);
#endif

#if defined(WOLFSSL_SM4_CBC) || defined(WOLFSSL_SM4_CTR)
    XMEMSET(iv, 0, sizeof(iv));

    /* Invalid parameters - wc_Sm4SetIV. */
    ExpectIntEQ(wc_Sm4SetIV(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetIV(&sm4, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4SetIV(NULL, iv), BAD_FUNC_ARG);

    /* Valid cases - wc_Sm4SetIV. */
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
#endif

    /* Valid cases - wc_Sm4Free */
    wc_Sm4Free(NULL);
    wc_Sm4Free(&sm4);

    res = EXPECT_RESULT();
#endif
    return res;
} /* END test_wc_Sm4 */

/*
 * Testing block based SM4-ECB API.
 */
static int test_wc_Sm4Ecb(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SM4_ECB
    EXPECT_DECLS;
    wc_Sm4 sm4;
    unsigned char key[SM4_KEY_SIZE];
    unsigned char in[SM4_BLOCK_SIZE * 2];
    unsigned char out[SM4_BLOCK_SIZE * 2];
    unsigned char out2[SM4_BLOCK_SIZE];

    XMEMSET(key, 0, sizeof(key));
    XMEMSET(in, 0, sizeof(in));

    ExpectIntEQ(wc_Sm4Init(&sm4, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_Sm4EcbEncrypt(&sm4, out, in, 0), MISSING_KEY);
    ExpectIntEQ(wc_Sm4EcbDecrypt(&sm4, out, in, 0), MISSING_KEY);

    /* Tested in test_wc_Sm4. */
    ExpectIntEQ(wc_Sm4SetKey(&sm4, key, SM4_KEY_SIZE), 0);

    /* Invalid parameters - wc_Sm4EcbEncrypt. */
    ExpectIntEQ(wc_Sm4EcbEncrypt(NULL, NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbEncrypt(&sm4, NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbEncrypt(NULL, out, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbEncrypt(NULL, NULL, in, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbEncrypt(NULL, NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbEncrypt(NULL, out, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbEncrypt(&sm4, NULL, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbEncrypt(&sm4, out, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbEncrypt(&sm4, out, in, 1), BAD_FUNC_ARG);

    /* Valid cases - wc_Sm4EcbEncrypt. */
    ExpectIntEQ(wc_Sm4EcbEncrypt(&sm4, out, in, 0), 0);
    ExpectIntEQ(wc_Sm4EcbEncrypt(&sm4, out2, in, SM4_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_Sm4EcbEncrypt(&sm4, out, in, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(out, out2, SM4_BLOCK_SIZE), 0);
    /*   In and out are same pointer. */
    ExpectIntEQ(wc_Sm4EcbEncrypt(&sm4, in, in, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(in, out, SM4_BLOCK_SIZE * 2), 0);

    /* Invalid parameters - wc_Sm4EcbDecrypt. */
    ExpectIntEQ(wc_Sm4EcbDecrypt(NULL, NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbDecrypt(&sm4, NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbDecrypt(NULL, out, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbDecrypt(NULL, NULL, in, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbDecrypt(NULL, NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbDecrypt(NULL, out, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbDecrypt(&sm4, NULL, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbDecrypt(&sm4, out, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4EcbDecrypt(&sm4, out, in, 1), BAD_FUNC_ARG);

    /* Valid cases - wc_Sm4EcbDecrypt. */
    ExpectIntEQ(wc_Sm4EcbDecrypt(&sm4, out, in, 0), 0);
    ExpectIntEQ(wc_Sm4EcbDecrypt(&sm4, out2, in, SM4_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_Sm4EcbDecrypt(&sm4, out, in, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(out, out2, SM4_BLOCK_SIZE), 0);
    /*   In and out are same pointer. */
    ExpectIntEQ(wc_Sm4EcbDecrypt(&sm4, in, in, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(in, out, SM4_BLOCK_SIZE * 2), 0);

    wc_Sm4Free(&sm4);

    res = EXPECT_RESULT();
#endif
    return res;
} /* END test_wc_Sm4Ecb */

/*
 * Testing block based SM4-CBC API.
 */
static int test_wc_Sm4Cbc(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SM4_CBC
    EXPECT_DECLS;
    wc_Sm4 sm4;
    unsigned char key[SM4_KEY_SIZE];
    unsigned char iv[SM4_IV_SIZE];
    unsigned char in[SM4_BLOCK_SIZE * 2];
    unsigned char out[SM4_BLOCK_SIZE * 2];
    unsigned char out2[SM4_BLOCK_SIZE];

    XMEMSET(key, 0, sizeof(key));
    XMEMSET(iv, 0, sizeof(iv));
    XMEMSET(in, 0, sizeof(in));

    ExpectIntEQ(wc_Sm4Init(&sm4, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, out, in, 0), MISSING_KEY);
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, out, in, 0), MISSING_KEY);
    /* Tested in test_wc_Sm4. */
    ExpectIntEQ(wc_Sm4SetKey(&sm4, key, SM4_KEY_SIZE), 0);
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, out, in, 0), MISSING_IV);
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, out, in, 0), MISSING_IV);
    /* Tested in test_wc_Sm4. */
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);

    /* Invalid parameters - wc_Sm4CbcEncrypt. */
    ExpectIntEQ(wc_Sm4CbcEncrypt(NULL, NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcEncrypt(NULL, out, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcEncrypt(NULL, NULL, in, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcEncrypt(NULL, NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcEncrypt(NULL, out, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, NULL, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, out, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, out, in, 1), BAD_FUNC_ARG);

    /* Valid cases - wc_Sm4CbcEncrypt. */
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, out, in, 0), 0);
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, out2, in, SM4_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, out, in, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(out, out2, SM4_BLOCK_SIZE), 0);
    /*   In and out are same pointer. */
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    ExpectIntEQ(wc_Sm4CbcEncrypt(&sm4, in, in, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(in, out, SM4_BLOCK_SIZE * 2), 0);

    /* Invalid parameters - wc_Sm4CbcDecrypt. */
    ExpectIntEQ(wc_Sm4CbcDecrypt(NULL, NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, NULL, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcDecrypt(NULL, out, NULL, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcDecrypt(NULL, NULL, in, 1), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcDecrypt(NULL, NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcDecrypt(NULL, out, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, NULL, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, out, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, out, in, 1), BAD_FUNC_ARG);

    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    /* Valid cases - wc_Sm4CbcDecrypt. */
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, out, in, 0), 0);
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, out2, in, SM4_BLOCK_SIZE), 0);
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, out, in, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(out, out2, SM4_BLOCK_SIZE), 0);
    /*   In and out are same pointer. */
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    ExpectIntEQ(wc_Sm4CbcDecrypt(&sm4, in, in, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(in, out, SM4_BLOCK_SIZE * 2), 0);

    wc_Sm4Free(&sm4);

    res = EXPECT_RESULT();
#endif
    return res;
} /* END test_wc_Sm4Cbc */

/*
 * Testing streaming SM4-CTR API.
 */
static int test_wc_Sm4Ctr(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SM4_CTR
    EXPECT_DECLS;
    wc_Sm4 sm4;
    unsigned char key[SM4_KEY_SIZE];
    unsigned char iv[SM4_IV_SIZE];
    unsigned char in[SM4_BLOCK_SIZE * 4];
    unsigned char out[SM4_BLOCK_SIZE * 4];
    unsigned char out2[SM4_BLOCK_SIZE * 4];
    word32 chunk;
    word32 i;

    XMEMSET(key, 0, sizeof(key));
    XMEMSET(iv, 0, sizeof(iv));
    XMEMSET(in, 0, sizeof(in));

    ExpectIntEQ(wc_Sm4Init(&sm4, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out, in, 0), MISSING_KEY);
    /* Tested in test_wc_Sm4. */
    ExpectIntEQ(wc_Sm4SetKey(&sm4, key, SM4_KEY_SIZE), 0);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out, in, 0), MISSING_IV);
    /* Tested in test_wc_Sm4. */
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);

    /* Invalid parameters - wc_Sm4CtrEncrypt. */
    ExpectIntEQ(wc_Sm4CtrEncrypt(NULL, NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CtrEncrypt(NULL, out, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CtrEncrypt(NULL, NULL, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, NULL, in, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CtrEncrypt(NULL, out, in, 0), BAD_FUNC_ARG);

    /* Valid cases - wc_Sm4CtrEncrypt. */
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out, in, 0), 0);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out2, in, 1), 0);
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out, in, 2), 0);
    ExpectIntEQ(XMEMCMP(out, out2, 1), 0);
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out2, in, SM4_BLOCK_SIZE), 0);
    ExpectIntEQ(XMEMCMP(out2, out, 2), 0);
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out, in, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(out, out2, SM4_BLOCK_SIZE), 0);
    /*   In and out are same pointer. Also check encrypt of cipher text produces
     *   plaintext.
     */
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out, out, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(XMEMCMP(in, out, SM4_BLOCK_SIZE * 2), 0);

    /* Chunking tests. */
    ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
    ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out2, in, (word32)sizeof(in)), 0);
    for (chunk = 1; chunk <= SM4_BLOCK_SIZE + 1; chunk++) {
        ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
        for (i = 0; i + chunk <= (word32)sizeof(in); i += chunk) {
             ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out + i, in + i, chunk), 0);
        }
        if (i < (word32)sizeof(in)) {
             ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out + i, in + i,
                 (word32)sizeof(in) - i), 0);
        }
        ExpectIntEQ(XMEMCMP(out, out2, (word32)sizeof(out)), 0);
    }

    for (i = 0; i < (word32)sizeof(iv); i++) {
        iv[i] = 0xff;
        ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
        ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out, in, SM4_BLOCK_SIZE * 2), 0);
        ExpectIntEQ(wc_Sm4SetIV(&sm4, iv), 0);
        ExpectIntEQ(wc_Sm4CtrEncrypt(&sm4, out2, out, SM4_BLOCK_SIZE * 2), 0);
        ExpectIntEQ(XMEMCMP(out2, in, SM4_BLOCK_SIZE * 2), 0);
    }

    wc_Sm4Free(&sm4);

    res = EXPECT_RESULT();
#endif
    return res;
} /* END test_wc_Sm4Ctr */

/*
 * Testing stream SM4-GCM API.
 */
static int test_wc_Sm4Gcm(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SM4_GCM
    EXPECT_DECLS;
    wc_Sm4 sm4;
    unsigned char key[SM4_KEY_SIZE];
    unsigned char nonce[GCM_NONCE_MAX_SZ];
    unsigned char in[SM4_BLOCK_SIZE * 2];
    unsigned char in2[SM4_BLOCK_SIZE * 2];
    unsigned char out[SM4_BLOCK_SIZE * 2];
    unsigned char out2[SM4_BLOCK_SIZE * 2];
    unsigned char dec[SM4_BLOCK_SIZE * 2];
    unsigned char tag[SM4_BLOCK_SIZE];
    unsigned char aad[SM4_BLOCK_SIZE * 2];
    word32 i;

    XMEMSET(key, 0, sizeof(key));
    XMEMSET(nonce, 0, sizeof(nonce));
    XMEMSET(in, 0, sizeof(in));
    XMEMSET(in2, 0, sizeof(in2));
    XMEMSET(aad, 0, sizeof(aad));

    ExpectIntEQ(wc_Sm4Init(&sm4, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, 0, nonce, GCM_NONCE_MID_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), MISSING_KEY);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, out, in, 0, nonce, GCM_NONCE_MID_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), MISSING_KEY);

    /* Invalid parameters - wc_Sm4GcmSetKey. */
    ExpectIntEQ(wc_Sm4GcmSetKey(NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmSetKey(&sm4, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmSetKey(NULL, key, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmSetKey(NULL, NULL, SM4_KEY_SIZE), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmSetKey(&sm4, key, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmSetKey(&sm4, NULL, SM4_KEY_SIZE), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmSetKey(NULL, key, SM4_KEY_SIZE), BAD_FUNC_ARG);

    /* Valid parameters - wc_Sm4GcmSetKey. */
    ExpectIntEQ(wc_Sm4GcmSetKey(&sm4, key, SM4_KEY_SIZE), 0);

    /* Invalid parameters - wc_Sm4GcmEncrypt. */
    ExpectIntEQ(wc_Sm4GcmEncrypt(NULL, NULL, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, NULL, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(NULL, out, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(NULL, NULL, in, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(NULL, NULL, NULL, 1, nonce, GCM_NONCE_MID_SZ,
        NULL, 0, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(NULL, NULL, NULL, 1, NULL, 0, tag,
        SM4_BLOCK_SIZE, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(NULL, out, in, 1, nonce, GCM_NONCE_MID_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, NULL, in, 1, nonce, GCM_NONCE_MID_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, NULL, 1, nonce, GCM_NONCE_MID_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, 1, NULL, GCM_NONCE_MID_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, 1, nonce, 0, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, 1, nonce, GCM_NONCE_MID_SZ,
        NULL, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, 1, nonce, GCM_NONCE_MID_SZ, tag,
        WOLFSSL_MIN_AUTH_TAG_SZ-1, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, 1, nonce, GCM_NONCE_MID_SZ, tag,
        SM4_BLOCK_SIZE+1, aad, sizeof(aad)), BAD_FUNC_ARG);

    /* Invalid parameters - wc_Sm4GcmDecrypt. */
    ExpectIntEQ(wc_Sm4GcmDecrypt(NULL, NULL, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, NULL, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(NULL, out, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(NULL, NULL, in, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(NULL, NULL, NULL, 1, nonce, GCM_NONCE_MID_SZ,
        NULL, 0, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(NULL, NULL, NULL, 1, NULL, 0, tag,
        SM4_BLOCK_SIZE, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(NULL, out, in, 1, nonce, GCM_NONCE_MID_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, NULL, in, 1, nonce, GCM_NONCE_MID_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, out, NULL, 1, nonce, GCM_NONCE_MID_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, out, in, 1, NULL, GCM_NONCE_MID_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, out, in, 1, nonce, 0, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, out, in, 1, nonce, GCM_NONCE_MID_SZ,
        NULL, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, out, in, 1, nonce, GCM_NONCE_MID_SZ, tag,
        WOLFSSL_MIN_AUTH_TAG_SZ-1, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, out, in, 1, nonce, GCM_NONCE_MID_SZ, tag,
        SM4_BLOCK_SIZE+1, aad, sizeof(aad)), BAD_FUNC_ARG);

    /* Valid cases - wc_Sm4GcmEncrypt/wc_Sm4GcmDecrypt. */
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, NULL, NULL, 0, nonce, GCM_NONCE_MID_SZ,
        tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, NULL, NULL, 0, nonce, GCM_NONCE_MID_SZ,
        tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, NULL, NULL, 0, nonce, GCM_NONCE_MID_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, NULL, NULL, 0, nonce, GCM_NONCE_MID_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, NULL, 1), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, NULL, 1), 0);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE * 2, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE * 2, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, in2, in2, SM4_BLOCK_SIZE * 2, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(XMEMCMP(in2, out, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, in2, in2, SM4_BLOCK_SIZE * 2, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(XMEMCMP(in2, in, SM4_BLOCK_SIZE * 2), 0);

    /* Check vald values of nonce - wc_Sm4GcmEncrypt/wc_Sm4GcmDecrypt. */
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
        GCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
        GCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE * 2, nonce,
        GCM_NONCE_MIN_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE * 2, nonce,
        GCM_NONCE_MIN_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE * 2, nonce,
        GCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)),
        SM4_GCM_AUTH_E);

    /* Check valid values of tag size - wc_Sm4GcmEncrypt/wc_Sm4GcmDecrypt. */
    for (i = WOLFSSL_MIN_AUTH_TAG_SZ; i < SM4_BLOCK_SIZE; i++) {
        ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
            GCM_NONCE_MID_SZ, tag, i, aad, sizeof(aad)), 0);
        ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
            GCM_NONCE_MID_SZ, tag, i, aad, sizeof(aad)), 0);
    }

    /* Check different in/out sizes. */
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, 0, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, out, in, 0, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, 1, nonce,
        GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    for (i = 2; i <= SM4_BLOCK_SIZE * 2; i++) {
        XMEMCPY(out2, out, i - 1);
        ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, out, in, i, nonce, GCM_NONCE_MID_SZ,
            tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
        ExpectIntEQ(XMEMCMP(out, out2, i - 1), 0);
        ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, dec, out, i, nonce, GCM_NONCE_MID_SZ,
            tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
        ExpectIntEQ(XMEMCMP(in, dec, i), 0);
    }

    /* Force the counter to roll over in first byte. */
    {
        static unsigned char largeIn[256 * SM4_BLOCK_SIZE];
        static unsigned char largeOut[256 * SM4_BLOCK_SIZE];

        ExpectIntEQ(wc_Sm4GcmEncrypt(&sm4, largeOut, largeIn, sizeof(largeIn),
            nonce, GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
        ExpectIntEQ(wc_Sm4GcmDecrypt(&sm4, largeOut, largeOut, sizeof(largeIn),
            nonce, GCM_NONCE_MID_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
        ExpectIntEQ(XMEMCMP(largeOut, largeIn, sizeof(largeIn)), 0);
    }

    wc_Sm4Free(&sm4);

    res = EXPECT_RESULT();
#endif
    return res;
} /* END test_wc_Sm4Gcm */

/*
 * Testing stream SM4-CCM API.
 */
static int test_wc_Sm4Ccm(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SM4_CCM
    EXPECT_DECLS;
    wc_Sm4 sm4;
    unsigned char key[SM4_KEY_SIZE];
    unsigned char nonce[CCM_NONCE_MAX_SZ];
    unsigned char in[SM4_BLOCK_SIZE * 2];
    unsigned char in2[SM4_BLOCK_SIZE * 2];
    unsigned char out[SM4_BLOCK_SIZE * 2];
    unsigned char out2[SM4_BLOCK_SIZE * 2];
    unsigned char dec[SM4_BLOCK_SIZE * 2];
    unsigned char tag[SM4_BLOCK_SIZE];
    unsigned char aad[SM4_BLOCK_SIZE * 2];
    word32 i;

    XMEMSET(key, 0, sizeof(key));
    XMEMSET(nonce, 0, sizeof(nonce));
    XMEMSET(in, 0, sizeof(in));
    XMEMSET(in2, 0, sizeof(in2));
    XMEMSET(aad, 0, sizeof(aad));

    ExpectIntEQ(wc_Sm4Init(&sm4, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, 0, nonce, CCM_NONCE_MAX_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), MISSING_KEY);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, out, in, 0, nonce, CCM_NONCE_MAX_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), MISSING_KEY);
    ExpectIntEQ(wc_Sm4SetKey(&sm4, key, SM4_KEY_SIZE), 0);

    /* Invalid parameters - wc_Sm4CcmEncrypt. */
    ExpectIntEQ(wc_Sm4CcmEncrypt(NULL, NULL, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, NULL, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(NULL, out, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(NULL, NULL, in, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(NULL, NULL, NULL, 1, nonce, CCM_NONCE_MAX_SZ,
        NULL, 0, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(NULL, NULL, NULL, 1, NULL, 0, tag,
        SM4_BLOCK_SIZE, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(NULL, out, in, 1, nonce, CCM_NONCE_MAX_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, NULL, in, 1, nonce, CCM_NONCE_MAX_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, NULL, 1, nonce, CCM_NONCE_MAX_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, 1, NULL, CCM_NONCE_MAX_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, 1, nonce, 0, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, 1, nonce, CCM_NONCE_MAX_SZ,
        NULL, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, 1, nonce, CCM_NONCE_MAX_SZ, tag,
        WOLFSSL_MIN_AUTH_TAG_SZ-1, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, 1, nonce, CCM_NONCE_MAX_SZ, tag,
        SM4_BLOCK_SIZE+1, aad, sizeof(aad)), BAD_FUNC_ARG);

    /* Invalid parameters - wc_Sm4CcmDecrypt. */
    ExpectIntEQ(wc_Sm4CcmDecrypt(NULL, NULL, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, NULL, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(NULL, out, NULL, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(NULL, NULL, in, 1, NULL, 0, NULL, 0, NULL,
        0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(NULL, NULL, NULL, 1, nonce, CCM_NONCE_MAX_SZ,
        NULL, 0, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(NULL, NULL, NULL, 1, NULL, 0, tag,
        SM4_BLOCK_SIZE, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(NULL, out, in, 1, nonce, CCM_NONCE_MAX_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, NULL, in, 1, nonce, CCM_NONCE_MAX_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, out, NULL, 1, nonce, CCM_NONCE_MAX_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, out, in, 1, NULL, CCM_NONCE_MAX_SZ, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, out, in, 1, nonce, 0, tag,
        SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, out, in, 1, nonce, CCM_NONCE_MAX_SZ,
        NULL, SM4_BLOCK_SIZE, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, out, in, 1, nonce, CCM_NONCE_MAX_SZ, tag,
        WOLFSSL_MIN_AUTH_TAG_SZ - 1, aad, sizeof(aad)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, out, in, 1, nonce, CCM_NONCE_MAX_SZ, tag,
        SM4_BLOCK_SIZE + 1, aad, sizeof(aad)), BAD_FUNC_ARG);

    /* Valid cases - wc_Sm4CcmEncrypt/wc_Sm4CcmDecrypt. */
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, NULL, NULL, 0, nonce, CCM_NONCE_MAX_SZ,
        tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, NULL, NULL, 0, nonce, CCM_NONCE_MAX_SZ,
        tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, NULL, NULL, 0, nonce, CCM_NONCE_MAX_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, NULL, NULL, 0, nonce, CCM_NONCE_MAX_SZ,
        tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, NULL, 1), 0);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, NULL, 1), 0);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE * 2, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE * 2, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, in2, in2, SM4_BLOCK_SIZE * 2, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(XMEMCMP(in2, out, SM4_BLOCK_SIZE * 2), 0);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, in2, in2, SM4_BLOCK_SIZE * 2, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    ExpectIntEQ(XMEMCMP(in2, in, SM4_BLOCK_SIZE * 2), 0);

    /* Check vald values of nonce - wc_Sm4CcmEncrypt/wc_Sm4CcmDecrypt. */
    for (i = CCM_NONCE_MIN_SZ; i <= CCM_NONCE_MAX_SZ; i++) {
        ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
            i, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
        ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
            i, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
    }
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
        CCM_NONCE_MIN_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)),
        SM4_CCM_AUTH_E);

    /* Check invalid values of tag size - wc_Sm4CcmEncrypt/wc_Sm4CcmDecrypt. */
    for (i = 0; i < 4; i++) {
        ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
            CCM_NONCE_MAX_SZ, tag, i * 2 + 1, aad, sizeof(aad)), BAD_FUNC_ARG);
        ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
            CCM_NONCE_MAX_SZ, tag, i * 2 + 1, aad, sizeof(aad)), BAD_FUNC_ARG);
    }
    /*   Odd values in range 4..SM4_BLOCK_SIZE. */
    for (i = 2; i < SM4_BLOCK_SIZE / 2; i++) {
        ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
            CCM_NONCE_MAX_SZ, tag, i * 2 + 1, aad, sizeof(aad)), BAD_FUNC_ARG);
        ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
            CCM_NONCE_MAX_SZ, tag, i * 2 + 1, aad, sizeof(aad)), BAD_FUNC_ARG);
    }
    /* Check valid values of tag size - wc_Sm4CcmEncrypt/wc_Sm4CcmDecrypt.
     * Even values in range 4..SM4_BLOCK_SIZE.
     */
    for (i = 2; i < SM4_BLOCK_SIZE / 2; i++) {
        ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, SM4_BLOCK_SIZE, nonce,
            CCM_NONCE_MAX_SZ, tag, i * 2, aad, sizeof(aad)), 0);
        ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, in, out, SM4_BLOCK_SIZE, nonce,
            CCM_NONCE_MAX_SZ, tag, i * 2, aad, sizeof(aad)), 0);
    }

    /* Check different in/out sizes. */
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, 0, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, out, in, 0, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, 1, nonce,
        CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, NULL, 0), 0);
    for (i = 2; i <= SM4_BLOCK_SIZE * 2; i++) {
        XMEMCPY(out2, out, i - 1);
        ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, out, in, i, nonce, CCM_NONCE_MAX_SZ,
            tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
        ExpectIntEQ(XMEMCMP(out, out2, i - 1), 0);
        ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, dec, out, i, nonce, CCM_NONCE_MAX_SZ,
            tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
        ExpectIntEQ(XMEMCMP(in, dec, i), 0);
    }

    /* Force the counter to roll over in first byte. */
    {
        static unsigned char largeIn[256 * SM4_BLOCK_SIZE];
        static unsigned char largeOut[256 * SM4_BLOCK_SIZE];

        ExpectIntEQ(wc_Sm4CcmEncrypt(&sm4, largeOut, largeIn, sizeof(largeIn),
            nonce, CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
        ExpectIntEQ(wc_Sm4CcmDecrypt(&sm4, largeOut, largeOut, sizeof(largeIn),
            nonce, CCM_NONCE_MAX_SZ, tag, SM4_BLOCK_SIZE, aad, sizeof(aad)), 0);
        ExpectIntEQ(XMEMCMP(largeOut, largeIn, sizeof(largeIn)), 0);
    }

    wc_Sm4Free(&sm4);

    res = EXPECT_RESULT();
#endif
    return res;
} /* END test_wc_Sm4Ccm */


/*
 * unit test for wc_Des3_SetIV()
 */
static int test_wc_Des3_SetIV(void)
{
    EXPECT_DECLS;
#ifndef NO_DES3
    Des3 des;
    const byte key[] = {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };
    const byte iv[] = {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };

    XMEMSET(&des, 0, sizeof(Des3));

    ExpectIntEQ(wc_Des3Init(&des, NULL, INVALID_DEVID), 0);

    /* DES_ENCRYPTION or DES_DECRYPTION */
    ExpectIntEQ(wc_Des3_SetKey(&des, key, iv, DES_ENCRYPTION), 0);
    ExpectIntEQ(XMEMCMP(iv, des.reg, DES_BLOCK_SIZE), 0);

#ifndef HAVE_FIPS /* no sanity checks with FIPS wrapper */
    /* Test explicitly wc_Des3_SetIV()  */
    ExpectIntEQ(wc_Des3_SetIV(NULL, iv), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_SetIV(&des, NULL), 0);
#endif
    wc_Des3Free(&des);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Des3_SetIV */

/*
 * unit test for wc_Des3_SetKey()
 */
static int test_wc_Des3_SetKey(void)
{
    EXPECT_DECLS;
#ifndef NO_DES3
    Des3 des;
    const byte key[] = {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };
    const byte iv[] = {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };

    XMEMSET(&des, 0, sizeof(Des3));

    ExpectIntEQ(wc_Des3Init(&des, NULL, INVALID_DEVID), 0);

    /* DES_ENCRYPTION or DES_DECRYPTION */
    ExpectIntEQ(wc_Des3_SetKey(&des, key, iv, DES_ENCRYPTION), 0);
    ExpectIntEQ(XMEMCMP(iv, des.reg, DES_BLOCK_SIZE), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Des3_SetKey(NULL, key, iv, DES_ENCRYPTION), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_SetKey(&des, NULL, iv, DES_ENCRYPTION), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_SetKey(&des, key, iv, -1), BAD_FUNC_ARG);
    /* Default case. Should return 0. */
    ExpectIntEQ(wc_Des3_SetKey(&des, key, NULL, DES_ENCRYPTION), 0);

    wc_Des3Free(&des);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Des3_SetKey */


/*
 * Test function for wc_Des3_CbcEncrypt and wc_Des3_CbcDecrypt
 */
static int test_wc_Des3_CbcEncryptDecrypt(void)
{
    EXPECT_DECLS;
#ifndef NO_DES3
    Des3 des;
    byte cipher[24];
    byte plain[24];
    const byte key[] = {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };
    const byte iv[] = {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };
    const byte vector[] = { /* "Now is the time for all " w/o trailing 0 */
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    XMEMSET(&des, 0, sizeof(Des3));

    ExpectIntEQ(wc_Des3Init(&des, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_Des3_SetKey(&des, key, iv, DES_ENCRYPTION), 0);

    ExpectIntEQ(wc_Des3_CbcEncrypt(&des, cipher, vector, 24), 0);
    ExpectIntEQ(wc_Des3_SetKey(&des, key, iv, DES_DECRYPTION), 0);
    ExpectIntEQ(wc_Des3_CbcDecrypt(&des, plain, cipher, 24), 0);
    ExpectIntEQ(XMEMCMP(plain, vector, 24), 0);

    /* Pass in bad args. */
    ExpectIntEQ(wc_Des3_CbcEncrypt(NULL, cipher, vector, 24), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcEncrypt(&des, NULL, vector, 24), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcEncrypt(&des, cipher, NULL, sizeof(vector)),
        BAD_FUNC_ARG);

    ExpectIntEQ(wc_Des3_CbcDecrypt(NULL, plain, cipher, 24), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcDecrypt(&des, NULL, cipher, 24), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcDecrypt(&des, plain, NULL, 24), BAD_FUNC_ARG);

    wc_Des3Free(&des);
#endif
    return EXPECT_RESULT();

} /* END wc_Des3_CbcEncrypt */

/*
 *  Unit test for wc_Des3_CbcEncryptWithKey and wc_Des3_CbcDecryptWithKey
 */
static int test_wc_Des3_CbcEncryptDecryptWithKey(void)
{
    EXPECT_DECLS;
#ifndef NO_DES3
    word32 vectorSz, cipherSz;
    byte cipher[24];
    byte plain[24];
    byte vector[] = { /* Now is the time for all w/o trailing 0 */
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    byte key[] = {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };
    byte iv[] = {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };

    vectorSz = sizeof(byte) * 24;
    cipherSz = sizeof(byte) * 24;

    ExpectIntEQ(wc_Des3_CbcEncryptWithKey(cipher, vector, vectorSz, key, iv),
        0);
    ExpectIntEQ(wc_Des3_CbcDecryptWithKey(plain, cipher, cipherSz, key, iv), 0);
    ExpectIntEQ(XMEMCMP(plain, vector, 24), 0);

    /* pass in bad args. */
    ExpectIntEQ(wc_Des3_CbcEncryptWithKey(NULL, vector, vectorSz, key, iv),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcEncryptWithKey(cipher, NULL, vectorSz, key, iv),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcEncryptWithKey(cipher, vector, vectorSz, NULL, iv),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcEncryptWithKey(cipher, vector, vectorSz, key, NULL),
        0);

    ExpectIntEQ(wc_Des3_CbcDecryptWithKey(NULL, cipher, cipherSz, key, iv),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcDecryptWithKey(plain, NULL, cipherSz, key, iv),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcDecryptWithKey(plain, cipher, cipherSz, NULL, iv),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_CbcDecryptWithKey(plain, cipher, cipherSz, key, NULL),
        0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Des3_CbcEncryptDecryptWithKey */
/*
 *  Unit test for wc_Des3_EcbEncrypt
 */
static int test_wc_Des3_EcbEncrypt(void)
{
    EXPECT_DECLS;
#if !defined(NO_DES3) && defined(WOLFSSL_DES_ECB)
    Des3    des;
    byte    cipher[24];
    word32  cipherSz = sizeof(cipher);
    const byte key[] = {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };
    const byte iv[] = {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };
    const byte vector[] = { /* "Now is the time for all " w/o trailing 0 */
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    XMEMSET(&des, 0, sizeof(Des3));

    ExpectIntEQ(wc_Des3Init(&des, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_Des3_SetKey(&des, key, iv, DES_ENCRYPTION), 0);

    /* Bad Cases */
    ExpectIntEQ(wc_Des3_EcbEncrypt(NULL, 0, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_EcbEncrypt(NULL, cipher, vector, cipherSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_EcbEncrypt(&des, 0, vector, cipherSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_EcbEncrypt(&des, cipher, NULL, cipherSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Des3_EcbEncrypt(&des, cipher, vector, 0), 0);

    /* Good Cases */
    ExpectIntEQ(wc_Des3_EcbEncrypt(&des, cipher, vector, cipherSz), 0);

    wc_Des3Free(&des);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Des3_EcbEncrypt */

/*
 * Testing wc_Chacha_SetKey() and wc_Chacha_SetIV()
 */
static int test_wc_Chacha_SetKey(void)
{
    EXPECT_DECLS;
#ifdef HAVE_CHACHA
    ChaCha     ctx;
    const byte key[] = {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };
    word32 keySz = (word32)(sizeof(key)/sizeof(byte));
    byte       cipher[128];

    ExpectIntEQ(wc_Chacha_SetKey(&ctx, key, keySz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_Chacha_SetKey(NULL, key, keySz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Chacha_SetKey(&ctx, key, 18), BAD_FUNC_ARG);

    ExpectIntEQ(wc_Chacha_SetIV(&ctx, cipher, 0), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_Chacha_SetIV(NULL, cipher, 0), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Chacha_SetKey */

/*
 * unit test for wc_Poly1305SetKey()
 */
static int test_wc_Poly1305SetKey(void)
{
    EXPECT_DECLS;
#ifdef HAVE_POLY1305
    Poly1305    ctx;
    const byte  key[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };
    word32 keySz = (word32)(sizeof(key)/sizeof(byte));

    ExpectIntEQ(wc_Poly1305SetKey(&ctx, key, keySz), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_Poly1305SetKey(NULL, key,keySz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Poly1305SetKey(&ctx, NULL, keySz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Poly1305SetKey(&ctx, key, 18), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Poly1305_SetKey() */

/*
 * Testing wc_Chacha_Process()
 */
static int test_wc_Chacha_Process(void)
{
    EXPECT_DECLS;
#ifdef HAVE_CHACHA
    ChaCha      enc, dec;
    byte        cipher[128];
    byte        plain[128];
    const byte  key[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };
    const char* input = "Everybody gets Friday off.";
    word32      keySz = sizeof(key)/sizeof(byte);
    unsigned long int inlen = XSTRLEN(input);

    /* Initialize stack variables. */
    XMEMSET(cipher, 0, 128);
    XMEMSET(plain, 0, 128);

    ExpectIntEQ(wc_Chacha_SetKey(&enc, key, keySz), 0);
    ExpectIntEQ(wc_Chacha_SetKey(&dec, key, keySz), 0);
    ExpectIntEQ(wc_Chacha_SetIV(&enc, cipher, 0), 0);
    ExpectIntEQ(wc_Chacha_SetIV(&dec, cipher, 0), 0);

    ExpectIntEQ(wc_Chacha_Process(&enc, cipher, (byte*)input, (word32)inlen),
        0);
    ExpectIntEQ(wc_Chacha_Process(&dec, plain, cipher, (word32)inlen), 0);
    ExpectIntEQ(XMEMCMP(input, plain, (int)inlen), 0);

#if !defined(USE_INTEL_CHACHA_SPEEDUP) && !defined(WOLFSSL_ARMASM)
    /* test checking and using leftovers, currently just in C code */
    ExpectIntEQ(wc_Chacha_SetIV(&enc, cipher, 0), 0);
    ExpectIntEQ(wc_Chacha_SetIV(&dec, cipher, 0), 0);

    ExpectIntEQ(wc_Chacha_Process(&enc, cipher, (byte*)input,
        (word32)inlen - 2), 0);
    ExpectIntEQ(wc_Chacha_Process(&enc, cipher + (inlen - 2),
        (byte*)input + (inlen - 2), 2), 0);
    ExpectIntEQ(wc_Chacha_Process(&dec, plain, (byte*)cipher,
        (word32)inlen - 2), 0);
    ExpectIntEQ(wc_Chacha_Process(&dec, cipher + (inlen - 2),
        (byte*)input + (inlen - 2), 2), 0);
    ExpectIntEQ(XMEMCMP(input, plain, (int)inlen), 0);

    /* check edge cases with counter increment */
    {
        /* expected results collected from wolfSSL 4.3.0 encrypted in one call*/
        const byte expected[] = {
            0x54,0xB1,0xE2,0xD4,0xA2,0x4D,0x52,0x5F,
            0x42,0x04,0x89,0x7C,0x6E,0x2D,0xFC,0x2D,
            0x10,0x25,0xB6,0x92,0x71,0xD5,0xC3,0x20,
            0xE3,0x0E,0xEC,0xF4,0xD8,0x10,0x70,0x29,
            0x2D,0x4C,0x2A,0x56,0x21,0xE1,0xC7,0x37,
            0x0B,0x86,0xF5,0x02,0x8C,0xB8,0xB8,0x38,
            0x41,0xFD,0xDF,0xD9,0xC3,0xE6,0xC8,0x88,
            0x06,0x82,0xD4,0x80,0x6A,0x50,0x69,0xD5,
            0xB9,0xB0,0x2F,0x44,0x36,0x5D,0xDA,0x5E,
            0xDE,0xF6,0xF5,0xFC,0x44,0xDC,0x07,0x51,
            0xA7,0x32,0x42,0xDB,0xCC,0xBD,0xE2,0xE5,
            0x0B,0xB1,0x14,0xFF,0x12,0x80,0x16,0x43,
            0xE7,0x40,0xD5,0xEA,0xC7,0x3F,0x69,0x07,
            0x64,0xD4,0x86,0x6C,0xE2,0x1F,0x8F,0x6E,
            0x35,0x41,0xE7,0xD3,0xB5,0x5D,0xD6,0xD4,
            0x9F,0x00,0xA9,0xAE,0x3D,0x28,0xA5,0x37,
            0x80,0x3D,0x11,0x25,0xE2,0xB6,0x99,0xD9,
            0x9B,0x98,0xE9,0x37,0xB9,0xF8,0xA0,0x04,
            0xDF,0x13,0x49,0x3F,0x19,0x6A,0x45,0x06,
            0x21,0xB4,0xC7,0x3B,0x49,0x45,0xB4,0xC8,
            0x03,0x5B,0x43,0x89,0xBD,0xB3,0x96,0x4B,
            0x17,0x6F,0x85,0xC6,0xCF,0xA6,0x05,0x35,
            0x1E,0x25,0x03,0xBB,0x55,0x0A,0xD5,0x54,
            0x41,0xEA,0xEB,0x50,0x40,0x1B,0x43,0x19,
            0x59,0x1B,0x0E,0x12,0x3E,0xA2,0x71,0xC3,
            0x1A,0xA7,0x11,0x50,0x43,0x9D,0x56,0x3B,
            0x63,0x2F,0x63,0xF1,0x8D,0xAE,0xF3,0x23,
            0xFA,0x1E,0xD8,0x6A,0xE1,0xB2,0x4B,0xF3,
            0xB9,0x13,0x7A,0x72,0x2B,0x6D,0xCC,0x41,
            0x1C,0x69,0x7C,0xCD,0x43,0x6F,0xE4,0xE2,
            0x38,0x99,0xFB,0xC3,0x38,0x92,0x62,0x35,
            0xC0,0x1D,0x60,0xE4,0x4B,0xDD,0x0C,0x14
        };
        const byte iv2[] = {
            0x9D,0xED,0xE7,0x0F,0xEC,0x81,0x51,0xD9,
            0x77,0x39,0x71,0xA6,0x21,0xDF,0xB8,0x93
        };
        byte input2[256];
        int i;

        for (i = 0; i < 256; i++)
            input2[i] = i;

        ExpectIntEQ(wc_Chacha_SetIV(&enc, iv2, 0), 0);

        ExpectIntEQ(wc_Chacha_Process(&enc, cipher, input2, 64), 0);
        ExpectIntEQ(XMEMCMP(expected, cipher, 64), 0);

        ExpectIntEQ(wc_Chacha_Process(&enc, cipher, input2 + 64, 128), 0);
        ExpectIntEQ(XMEMCMP(expected + 64, cipher, 128), 0);

        /* partial */
        ExpectIntEQ(wc_Chacha_Process(&enc, cipher, input2 + 192, 32), 0);
        ExpectIntEQ(XMEMCMP(expected + 192, cipher, 32), 0);

        ExpectIntEQ(wc_Chacha_Process(&enc, cipher, input2 + 224, 32), 0);
        ExpectIntEQ(XMEMCMP(expected + 224, cipher, 32), 0);
    }
#endif

    /* Test bad args. */
    ExpectIntEQ(wc_Chacha_Process(NULL, cipher, (byte*)input, (word32)inlen),
        BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Chacha_Process */

/*
 * Testing wc_ChaCha20Poly1305_Encrypt() and wc_ChaCha20Poly1305_Decrypt()
 */
static int test_wc_ChaCha20Poly1305_aead(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    const byte  key[] = {
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
        0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
        0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
    };
    const byte  plaintext[] = {
        0x4c, 0x61, 0x64, 0x69, 0x65, 0x73, 0x20, 0x61,
        0x6e, 0x64, 0x20, 0x47, 0x65, 0x6e, 0x74, 0x6c,
        0x65, 0x6d, 0x65, 0x6e, 0x20, 0x6f, 0x66, 0x20,
        0x74, 0x68, 0x65, 0x20, 0x63, 0x6c, 0x61, 0x73,
        0x73, 0x20, 0x6f, 0x66, 0x20, 0x27, 0x39, 0x39,
        0x3a, 0x20, 0x49, 0x66, 0x20, 0x49, 0x20, 0x63,
        0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6f, 0x66, 0x66,
        0x65, 0x72, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6f,
        0x6e, 0x6c, 0x79, 0x20, 0x6f, 0x6e, 0x65, 0x20,
        0x74, 0x69, 0x70, 0x20, 0x66, 0x6f, 0x72, 0x20,
        0x74, 0x68, 0x65, 0x20, 0x66, 0x75, 0x74, 0x75,
        0x72, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x6e, 0x73,
        0x63, 0x72, 0x65, 0x65, 0x6e, 0x20, 0x77, 0x6f,
        0x75, 0x6c, 0x64, 0x20, 0x62, 0x65, 0x20, 0x69,
        0x74, 0x2e
    };
    const byte  iv[] = {
        0x07, 0x00, 0x00, 0x00, 0x40, 0x41, 0x42, 0x43,
        0x44, 0x45, 0x46, 0x47
    };
    const byte  aad[] = { /* additional data */
        0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3,
        0xc4, 0xc5, 0xc6, 0xc7
    };
    const byte  cipher[] = { /* expected output from operation */
        0xd3, 0x1a, 0x8d, 0x34, 0x64, 0x8e, 0x60, 0xdb,
        0x7b, 0x86, 0xaf, 0xbc, 0x53, 0xef, 0x7e, 0xc2,
        0xa4, 0xad, 0xed, 0x51, 0x29, 0x6e, 0x08, 0xfe,
        0xa9, 0xe2, 0xb5, 0xa7, 0x36, 0xee, 0x62, 0xd6,
        0x3d, 0xbe, 0xa4, 0x5e, 0x8c, 0xa9, 0x67, 0x12,
        0x82, 0xfa, 0xfb, 0x69, 0xda, 0x92, 0x72, 0x8b,
        0x1a, 0x71, 0xde, 0x0a, 0x9e, 0x06, 0x0b, 0x29,
        0x05, 0xd6, 0xa5, 0xb6, 0x7e, 0xcd, 0x3b, 0x36,
        0x92, 0xdd, 0xbd, 0x7f, 0x2d, 0x77, 0x8b, 0x8c,
        0x98, 0x03, 0xae, 0xe3, 0x28, 0x09, 0x1b, 0x58,
        0xfa, 0xb3, 0x24, 0xe4, 0xfa, 0xd6, 0x75, 0x94,
        0x55, 0x85, 0x80, 0x8b, 0x48, 0x31, 0xd7, 0xbc,
        0x3f, 0xf4, 0xde, 0xf0, 0x8e, 0x4b, 0x7a, 0x9d,
        0xe5, 0x76, 0xd2, 0x65, 0x86, 0xce, 0xc6, 0x4b,
        0x61, 0x16
    };
    const byte  authTag[] = { /* expected output from operation */
        0x1a, 0xe1, 0x0b, 0x59, 0x4f, 0x09, 0xe2, 0x6a,
        0x7e, 0x90, 0x2e, 0xcb, 0xd0, 0x60, 0x06, 0x91
    };
    byte        generatedCiphertext[272];
    byte        generatedPlaintext[272];
    byte        generatedAuthTag[CHACHA20_POLY1305_AEAD_AUTHTAG_SIZE];

    /* Initialize stack variables. */
    XMEMSET(generatedCiphertext, 0, 272);
    XMEMSET(generatedPlaintext, 0, 272);

    /* Test Encrypt */
    ExpectIntEQ(wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad),
        plaintext, sizeof(plaintext), generatedCiphertext, generatedAuthTag),
        0);
    ExpectIntEQ(XMEMCMP(generatedCiphertext, cipher,
        sizeof(cipher)/sizeof(byte)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_ChaCha20Poly1305_Encrypt(NULL, iv, aad, sizeof(aad),
        plaintext, sizeof(plaintext), generatedCiphertext, generatedAuthTag),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Encrypt(key, NULL, aad, sizeof(aad),
        plaintext, sizeof(plaintext), generatedCiphertext, generatedAuthTag),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad), NULL,
        sizeof(plaintext), generatedCiphertext, generatedAuthTag),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad),
        NULL, sizeof(plaintext), generatedCiphertext, generatedAuthTag),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad),
        plaintext, sizeof(plaintext), NULL, generatedAuthTag), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad),
        plaintext, sizeof(plaintext), generatedCiphertext, NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), cipher,
        sizeof(cipher), authTag, generatedPlaintext), 0);
    ExpectIntEQ(XMEMCMP(generatedPlaintext, plaintext,
        sizeof(plaintext)/sizeof(byte)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_ChaCha20Poly1305_Decrypt(NULL, iv, aad, sizeof(aad), cipher,
        sizeof(cipher), authTag, generatedPlaintext),  BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Decrypt(key, NULL, aad, sizeof(aad),
        cipher, sizeof(cipher), authTag, generatedPlaintext), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), NULL,
        sizeof(cipher), authTag, generatedPlaintext), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), cipher,
        sizeof(cipher), NULL, generatedPlaintext), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), cipher,
        sizeof(cipher), authTag, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), NULL,
        sizeof(cipher), authTag, generatedPlaintext), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ChaCha20Poly1305_aead */


/*
 * Testing function for wc_Rc2SetKey().
 */
static int test_wc_Rc2SetKey(void)
{
    EXPECT_DECLS;
#ifdef WC_RC2
    Rc2  rc2;
    byte key40[] = { 0x01, 0x02, 0x03, 0x04, 0x05 };
    byte iv[]    = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };

    /* valid key and IV */
    ExpectIntEQ(wc_Rc2SetKey(&rc2, key40, (word32) sizeof(key40) / sizeof(byte),
        iv, 40), 0);
    /* valid key, no IV */
    ExpectIntEQ(wc_Rc2SetKey(&rc2, key40, (word32) sizeof(key40) / sizeof(byte),
        NULL, 40), 0);

    /* bad arguments  */
    /* null Rc2 struct */
    ExpectIntEQ(wc_Rc2SetKey(NULL, key40, (word32) sizeof(key40) / sizeof(byte),
        iv, 40), BAD_FUNC_ARG);
    /* null key */
    ExpectIntEQ(wc_Rc2SetKey(&rc2, NULL, (word32) sizeof(key40) / sizeof(byte),
        iv, 40), BAD_FUNC_ARG);
    /* key size == 0 */
    ExpectIntEQ(wc_Rc2SetKey(&rc2, key40, 0, iv, 40), WC_KEY_SIZE_E);
    /* key size > 128 */
    ExpectIntEQ(wc_Rc2SetKey(&rc2, key40, 129, iv, 40), WC_KEY_SIZE_E);
    /* effective bits == 0 */
    ExpectIntEQ(wc_Rc2SetKey(&rc2, key40, (word32)sizeof(key40) / sizeof(byte),
        iv, 0), WC_KEY_SIZE_E);
    /* effective bits > 1024 */
    ExpectIntEQ(wc_Rc2SetKey(&rc2, key40, (word32)sizeof(key40) / sizeof(byte),
        iv, 1025), WC_KEY_SIZE_E);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Rc2SetKey */

/*
 * Testing function for wc_Rc2SetIV().
 */
static int test_wc_Rc2SetIV(void)
{
    EXPECT_DECLS;
#ifdef WC_RC2
    Rc2  rc2;
    byte iv[] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };

    /* valid IV */
    ExpectIntEQ(wc_Rc2SetIV(&rc2, iv), 0);
    /* valid NULL IV */
    ExpectIntEQ(wc_Rc2SetIV(&rc2, NULL), 0);

    /* bad arguments */
    ExpectIntEQ(wc_Rc2SetIV(NULL, iv), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Rc2SetIV(NULL, NULL), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Rc2SetIV */

/*
 * Testing function for wc_Rc2EcbEncrypt() and wc_Rc2EcbDecrypt().
 */
static int test_wc_Rc2EcbEncryptDecrypt(void)
{
    EXPECT_DECLS;
#ifdef WC_RC2
    Rc2 rc2;
    int effectiveKeyBits = 63;
    byte cipher[RC2_BLOCK_SIZE];
    byte plain[RC2_BLOCK_SIZE];
    byte key[]    = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    byte input[]  = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    byte output[] = { 0xeb, 0xb7, 0x73, 0xf9, 0x93, 0x27, 0x8e, 0xff };

    XMEMSET(cipher, 0, sizeof(cipher));
    XMEMSET(plain, 0, sizeof(plain));

    ExpectIntEQ(wc_Rc2SetKey(&rc2, key, (word32) sizeof(key) / sizeof(byte),
        NULL, effectiveKeyBits), 0);
    ExpectIntEQ(wc_Rc2EcbEncrypt(&rc2, cipher, input, RC2_BLOCK_SIZE), 0);
    ExpectIntEQ(XMEMCMP(cipher, output, RC2_BLOCK_SIZE), 0);

    ExpectIntEQ(wc_Rc2EcbDecrypt(&rc2, plain, cipher, RC2_BLOCK_SIZE), 0);
    ExpectIntEQ(XMEMCMP(plain, input, RC2_BLOCK_SIZE), 0);

    /* Rc2EcbEncrypt bad arguments */
    /* null Rc2 struct */
    ExpectIntEQ(wc_Rc2EcbEncrypt(NULL, cipher, input, RC2_BLOCK_SIZE),
        BAD_FUNC_ARG);
    /* null out buffer */
    ExpectIntEQ(wc_Rc2EcbEncrypt(&rc2, NULL, input, RC2_BLOCK_SIZE),
        BAD_FUNC_ARG);
    /* null input buffer */
    ExpectIntEQ(wc_Rc2EcbEncrypt(&rc2, cipher, NULL, RC2_BLOCK_SIZE),
        BAD_FUNC_ARG);
    /* output buffer sz != RC2_BLOCK_SIZE (8) */
    ExpectIntEQ(wc_Rc2EcbEncrypt(&rc2, cipher, input, 7), BUFFER_E);

    /* Rc2EcbDecrypt bad arguments */
    /* null Rc2 struct */
    ExpectIntEQ(wc_Rc2EcbDecrypt(NULL, plain, output, RC2_BLOCK_SIZE),
        BAD_FUNC_ARG);
    /* null out buffer */
    ExpectIntEQ(wc_Rc2EcbDecrypt(&rc2, NULL, output, RC2_BLOCK_SIZE),
        BAD_FUNC_ARG);
    /* null input buffer */
    ExpectIntEQ(wc_Rc2EcbDecrypt(&rc2, plain, NULL, RC2_BLOCK_SIZE),
        BAD_FUNC_ARG);
    /* output buffer sz != RC2_BLOCK_SIZE (8) */
    ExpectIntEQ(wc_Rc2EcbDecrypt(&rc2, plain, output, 7), BUFFER_E);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Rc2EcbEncryptDecrypt */

/*
 * Testing function for wc_Rc2CbcEncrypt() and wc_Rc2CbcDecrypt().
 */
static int test_wc_Rc2CbcEncryptDecrypt(void)
{
    EXPECT_DECLS;
#ifdef WC_RC2
    Rc2 rc2;
    int effectiveKeyBits = 63;
    byte cipher[RC2_BLOCK_SIZE*2];
    byte plain[RC2_BLOCK_SIZE*2];
    /* vector taken from test.c */
    byte key[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    byte iv[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    byte input[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    byte output[] = {
        0xeb, 0xb7, 0x73, 0xf9, 0x93, 0x27, 0x8e, 0xff,
        0xf0, 0x51, 0x77, 0x8b, 0x65, 0xdb, 0x13, 0x57
    };

    XMEMSET(cipher, 0, sizeof(cipher));
    XMEMSET(plain, 0, sizeof(plain));

    ExpectIntEQ(wc_Rc2SetKey(&rc2, key, (word32) sizeof(key) / sizeof(byte),
        iv, effectiveKeyBits), 0);
    ExpectIntEQ(wc_Rc2CbcEncrypt(&rc2, cipher, input, sizeof(input)), 0);
    ExpectIntEQ(XMEMCMP(cipher, output, sizeof(output)), 0);

    /* reset IV for decrypt */
    ExpectIntEQ(wc_Rc2SetIV(&rc2, iv), 0);
    ExpectIntEQ(wc_Rc2CbcDecrypt(&rc2, plain, cipher, sizeof(cipher)), 0);
    ExpectIntEQ(XMEMCMP(plain, input, sizeof(input)), 0);

    /* Rc2CbcEncrypt bad arguments */
    /* null Rc2 struct */
    ExpectIntEQ(wc_Rc2CbcEncrypt(NULL, cipher, input, sizeof(input)),
        BAD_FUNC_ARG);
    /* null out buffer */
    ExpectIntEQ(wc_Rc2CbcEncrypt(&rc2, NULL, input, sizeof(input)),
        BAD_FUNC_ARG);
    /* null input buffer */
    ExpectIntEQ(wc_Rc2CbcEncrypt(&rc2, cipher, NULL, sizeof(input)),
        BAD_FUNC_ARG);

    /* Rc2CbcDecrypt bad arguments */
    /* in size is 0 */
    ExpectIntEQ(wc_Rc2CbcDecrypt(&rc2, plain, output, 0), 0);
    /* null Rc2 struct */
    ExpectIntEQ(wc_Rc2CbcDecrypt(NULL, plain, output, sizeof(output)),
        BAD_FUNC_ARG);
    /* null out buffer */
    ExpectIntEQ(wc_Rc2CbcDecrypt(&rc2, NULL, output, sizeof(output)),
        BAD_FUNC_ARG);
    /* null input buffer */
    ExpectIntEQ(wc_Rc2CbcDecrypt(&rc2, plain, NULL, sizeof(output)),
        BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_Rc2CbcEncryptDecrypt */


/*
 * Testing function for wc_AesSetIV
 */
static int test_wc_AesSetIV(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_AES) && defined(WOLFSSL_AES_128)
    Aes     aes;
    int     ret = 0;
    byte    key16[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    byte    iv1[]    = "1234567890abcdef";
    byte    iv2[]    = "0987654321fedcba";

    ret = wc_AesInit(&aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_AesSetKey(&aes, key16, (word32) sizeof(key16) / sizeof(byte),
                                                     iv1, AES_ENCRYPTION);
    if (ret == 0) {
        ret = wc_AesSetIV(&aes, iv2);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_AesSetIV(NULL, iv1);
        if (ret == BAD_FUNC_ARG) {
            /* NULL iv should return 0. */
            ret = wc_AesSetIV(&aes, NULL);
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_AesFree(&aes);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* test_wc_AesSetIV */


/*
 * Testing function for wc_AesSetKey().
 */
static int test_wc_AesSetKey(void)
{
    EXPECT_DECLS;
#ifndef NO_AES
    Aes  aes;
    byte key16[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#ifdef WOLFSSL_AES_192
    byte key24[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
#endif
#ifdef WOLFSSL_AES_256
    byte key32[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#endif
    byte badKey16[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65
    };
    byte iv[] = "1234567890abcdef";

    XMEMSET(&aes, 0, sizeof(Aes));

    ExpectIntEQ(wc_AesInit(&aes, NULL, INVALID_DEVID), 0);

#ifdef WOLFSSL_AES_128
    ExpectIntEQ(wc_AesSetKey(&aes, key16, (word32)sizeof(key16) / sizeof(byte),
        iv, AES_ENCRYPTION), 0);
#endif
#ifdef WOLFSSL_AES_192
    ExpectIntEQ(wc_AesSetKey(&aes, key24, (word32)sizeof(key24) / sizeof(byte),
        iv, AES_ENCRYPTION), 0);
#endif
#ifdef WOLFSSL_AES_256
    ExpectIntEQ(wc_AesSetKey(&aes, key32, (word32)sizeof(key32) / sizeof(byte),
        iv, AES_ENCRYPTION), 0);
#endif

    /* Pass in bad args. */
    ExpectIntEQ(wc_AesSetKey(NULL, key16, (word32)sizeof(key16) / sizeof(byte),
        iv, AES_ENCRYPTION), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesSetKey(&aes, badKey16,
        (word32)sizeof(badKey16) / sizeof(byte), iv, AES_ENCRYPTION),
        BAD_FUNC_ARG);

    wc_AesFree(&aes);
#endif
    return EXPECT_RESULT();
} /* END test_wc_AesSetKey */



/*
 * test function for wc_AesCbcEncrypt(), wc_AesCbcDecrypt(),
 * and wc_AesCbcDecryptWithKey()
 */
static int test_wc_AesCbcEncryptDecrypt(void)
{
    EXPECT_DECLS;
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(HAVE_AES_DECRYPT)&& \
    defined(WOLFSSL_AES_256)
    Aes  aes;
    byte key32[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    byte vector[] = { /* Now is the time for all good men w/o trailing 0 */
        0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
        0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20,
        0x66, 0x6f, 0x72, 0x20, 0x61, 0x6c, 0x6c, 0x20,
        0x67, 0x6f, 0x6f, 0x64, 0x20, 0x6d, 0x65, 0x6e
    };
    byte    iv[]   = "1234567890abcdef";
    byte    enc[sizeof(vector)];
    byte    dec[sizeof(vector)];
    byte    dec2[sizeof(vector)];

    /* Init stack variables. */
    XMEMSET(&aes, 0, sizeof(Aes));
    XMEMSET(enc, 0, sizeof(enc));
    XMEMSET(dec, 0, sizeof(vector));
    XMEMSET(dec2, 0, sizeof(vector));

    ExpectIntEQ(wc_AesInit(&aes, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_AesSetKey(&aes, key32, AES_BLOCK_SIZE * 2, iv,
        AES_ENCRYPTION), 0);
    ExpectIntEQ(wc_AesCbcEncrypt(&aes, enc, vector, sizeof(vector)), 0);
    wc_AesFree(&aes);

    /* Re init for decrypt and set flag. */
    ExpectIntEQ(wc_AesSetKey(&aes, key32, AES_BLOCK_SIZE * 2, iv,
        AES_DECRYPTION), 0);
    ExpectIntEQ(wc_AesCbcDecrypt(&aes, dec, enc, sizeof(vector)), 0);
    ExpectIntEQ(XMEMCMP(vector, dec, sizeof(vector)), 0);

    ExpectIntEQ(wc_AesCbcDecryptWithKey(dec2, enc, AES_BLOCK_SIZE, key32,
        sizeof(key32)/sizeof(byte), iv), 0);
    ExpectIntEQ(XMEMCMP(vector, dec2, AES_BLOCK_SIZE), 0);

    /* Pass in bad args */
    ExpectIntEQ(wc_AesCbcEncrypt(NULL, enc, vector, sizeof(vector)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCbcEncrypt(&aes, NULL, vector, sizeof(vector)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCbcEncrypt(&aes, enc, NULL, sizeof(vector)),
        BAD_FUNC_ARG);
#ifdef WOLFSSL_AES_CBC_LENGTH_CHECKS
    ExpectIntEQ(wc_AesCbcEncrypt(&aes, enc, vector, sizeof(vector) - 1),
        BAD_LENGTH_E);
#endif
#if defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && \
    (HAVE_FIPS_VERSION == 2) && defined(WOLFSSL_AESNI)
    fprintf(stderr, "Zero length inputs not supported with AESNI in FIPS "
                    "mode (v2), skip test");
#else
    /* Test passing in size of 0  */
    XMEMSET(enc, 0, sizeof(enc));
    ExpectIntEQ(wc_AesCbcEncrypt(&aes, enc, vector, 0), 0);
    /* Check enc was not modified */
    {
        int i;
        for (i = 0; i < (int)sizeof(enc); i++)
            ExpectIntEQ(enc[i], 0);
    }
#endif

    ExpectIntEQ(wc_AesCbcDecrypt(NULL, dec, enc, AES_BLOCK_SIZE), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCbcDecrypt(&aes, NULL, enc, AES_BLOCK_SIZE),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCbcDecrypt(&aes, dec, NULL, AES_BLOCK_SIZE),
        BAD_FUNC_ARG);
#ifdef WOLFSSL_AES_CBC_LENGTH_CHECKS
    ExpectIntEQ(wc_AesCbcDecrypt(&aes, dec, enc, AES_BLOCK_SIZE * 2 - 1),
        BAD_LENGTH_E);
#else
    ExpectIntEQ(wc_AesCbcDecrypt(&aes, dec, enc, AES_BLOCK_SIZE * 2 - 1),
        BAD_FUNC_ARG);
#endif

    /* Test passing in size of 0  */
    XMEMSET(dec, 0, sizeof(dec));
    ExpectIntEQ(wc_AesCbcDecrypt(&aes, dec, enc, 0), 0);
    /* Check dec was not modified */
    {
        int i;
        for (i = 0; i < (int)sizeof(dec); i++)
            ExpectIntEQ(dec[i], 0);
    }

    ExpectIntEQ(wc_AesCbcDecryptWithKey(NULL, enc, AES_BLOCK_SIZE,
        key32, sizeof(key32)/sizeof(byte), iv), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCbcDecryptWithKey(dec2, NULL, AES_BLOCK_SIZE,
        key32, sizeof(key32)/sizeof(byte), iv), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCbcDecryptWithKey(dec2, enc, AES_BLOCK_SIZE,
        NULL, sizeof(key32)/sizeof(byte), iv), BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCbcDecryptWithKey(dec2, enc, AES_BLOCK_SIZE,
        key32, sizeof(key32)/sizeof(byte), NULL), BAD_FUNC_ARG);

    wc_AesFree(&aes);
#endif
    return EXPECT_RESULT();
} /* END test_wc_AesCbcEncryptDecrypt */

/*
 * Testing wc_AesCtrEncrypt and wc_AesCtrDecrypt
 */
static int test_wc_AesCtrEncryptDecrypt(void)
{
    EXPECT_DECLS;
#if !defined(NO_AES) && defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_256)
    Aes aesEnc;
    Aes aesDec;
    byte key32[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    byte vector[] = { /* Now is the time for all w/o trailing 0 */
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    byte iv[] = "1234567890abcdef";
    byte enc[AES_BLOCK_SIZE * 2];
    byte dec[AES_BLOCK_SIZE * 2];

    /* Init stack variables. */
    XMEMSET(&aesEnc, 0, sizeof(Aes));
    XMEMSET(&aesDec, 0, sizeof(Aes));
    XMEMSET(enc, 0, AES_BLOCK_SIZE * 2);
    XMEMSET(dec, 0, AES_BLOCK_SIZE * 2);

    ExpectIntEQ(wc_AesInit(&aesEnc, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_AesInit(&aesDec, NULL, INVALID_DEVID), 0);

    ExpectIntEQ(wc_AesSetKey(&aesEnc, key32, AES_BLOCK_SIZE * 2, iv,
        AES_ENCRYPTION), 0);
    ExpectIntEQ(wc_AesCtrEncrypt(&aesEnc, enc, vector,
        sizeof(vector)/sizeof(byte)), 0);
    /* Decrypt with wc_AesCtrEncrypt() */
    ExpectIntEQ(wc_AesSetKey(&aesDec, key32, AES_BLOCK_SIZE * 2, iv,
        AES_ENCRYPTION), 0);
    ExpectIntEQ(wc_AesCtrEncrypt(&aesDec, dec, enc, sizeof(enc)/sizeof(byte)),
        0);
    ExpectIntEQ(XMEMCMP(vector, dec, sizeof(vector)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_AesCtrEncrypt(NULL, dec, enc, sizeof(enc)/sizeof(byte)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCtrEncrypt(&aesDec, NULL, enc, sizeof(enc)/sizeof(byte)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCtrEncrypt(&aesDec, dec, NULL, sizeof(enc)/sizeof(byte)),
        BAD_FUNC_ARG);

    wc_AesFree(&aesEnc);
    wc_AesFree(&aesDec);
#endif
    return EXPECT_RESULT();
} /* END test_wc_AesCtrEncryptDecrypt */

/*
 * test function for wc_AesGcmSetKey()
 */
static int test_wc_AesGcmSetKey(void)
{
    EXPECT_DECLS;
#if  !defined(NO_AES) && defined(HAVE_AESGCM)
    Aes aes;
#ifdef WOLFSSL_AES_128
    byte key16[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#endif
#ifdef WOLFSSL_AES_192
    byte key24[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
#endif
#ifdef WOLFSSL_AES_256
    byte key32[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#endif
    byte badKey16[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65
    };
    byte badKey24[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36
    };
    byte badKey32[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x37, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65
    };

    ExpectIntEQ(wc_AesInit(&aes, NULL, INVALID_DEVID), 0);

#ifdef WOLFSSL_AES_128
    ExpectIntEQ(wc_AesGcmSetKey(&aes, key16, sizeof(key16)/sizeof(byte)), 0);
#endif
#ifdef WOLFSSL_AES_192
    ExpectIntEQ(wc_AesGcmSetKey(&aes, key24, sizeof(key24)/sizeof(byte)), 0);
#endif
#ifdef WOLFSSL_AES_256
    ExpectIntEQ(wc_AesGcmSetKey(&aes, key32, sizeof(key32)/sizeof(byte)), 0);
#endif

    /* Pass in bad args. */
    ExpectIntEQ(wc_AesGcmSetKey(&aes, badKey16, sizeof(badKey16)/sizeof(byte)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmSetKey(&aes, badKey24, sizeof(badKey24)/sizeof(byte)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmSetKey(&aes, badKey32, sizeof(badKey32)/sizeof(byte)),
        BAD_FUNC_ARG);

    wc_AesFree(&aes);
#endif
    return EXPECT_RESULT();
} /* END test_wc_AesGcmSetKey */

/*
 * test function for wc_AesGcmEncrypt and wc_AesGcmDecrypt
 */
static int test_wc_AesGcmEncryptDecrypt(void)
{
    EXPECT_DECLS;
    /* WOLFSSL_AFALG requires 12 byte IV */
#if !defined(NO_AES) && defined(HAVE_AESGCM) && defined(WOLFSSL_AES_256) && \
    !defined(WOLFSSL_AFALG) && !defined(WOLFSSL_DEVCRYPTO_AES)
    Aes  aes;
    byte key32[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    byte vector[] = { /* Now is the time for all w/o trailing 0 */
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    const byte a[] = {
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xab, 0xad, 0xda, 0xd2
    };
    byte iv[] = "1234567890a";
    byte longIV[] = "1234567890abcdefghij";
    byte enc[sizeof(vector)];
    byte resultT[AES_BLOCK_SIZE];
    byte dec[sizeof(vector)];

    /* Init stack variables. */
    XMEMSET(&aes, 0, sizeof(Aes));
    XMEMSET(enc, 0, sizeof(vector));
    XMEMSET(dec, 0, sizeof(vector));
    XMEMSET(resultT, 0, AES_BLOCK_SIZE);

    ExpectIntEQ(wc_AesInit(&aes, NULL, INVALID_DEVID), 0);

    ExpectIntEQ(wc_AesGcmSetKey(&aes, key32, sizeof(key32)/sizeof(byte)), 0);
    ExpectIntEQ(wc_AesGcmEncrypt(&aes, enc, vector, sizeof(vector), iv,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT), a, sizeof(a)), 0);
    ExpectIntEQ(wc_AesGcmDecrypt(&aes, dec, enc, sizeof(vector), iv,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT), a, sizeof(a)), 0);
    ExpectIntEQ(XMEMCMP(vector, dec, sizeof(vector)), 0);

    /* Test bad args for wc_AesGcmEncrypt and wc_AesGcmDecrypt */
    ExpectIntEQ(wc_AesGcmEncrypt(NULL, enc, vector, sizeof(vector), iv,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT), a, sizeof(a)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncrypt(&aes, enc, vector, sizeof(vector), iv,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT) + 1, a, sizeof(a)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmEncrypt(&aes, enc, vector, sizeof(vector), iv,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT) - 5, a, sizeof(a)),
        BAD_FUNC_ARG);

#if (defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && \
        (HAVE_FIPS_VERSION == 2)) || defined(HAVE_SELFTEST) || \
        defined(WOLFSSL_AES_GCM_FIXED_IV_AAD)
        /* FIPS does not check the lower bound of ivSz */
#else
        ExpectIntEQ(wc_AesGcmEncrypt(&aes, enc, vector, sizeof(vector), iv, 0,
            resultT, sizeof(resultT), a, sizeof(a)), BAD_FUNC_ARG);
#endif

    /* This case is now considered good. Long IVs are now allowed.
     * Except for the original FIPS release, it still has an upper
     * bound on the IV length. */
#if (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))) && \
    !defined(WOLFSSL_AES_GCM_FIXED_IV_AAD)
    ExpectIntEQ(wc_AesGcmEncrypt(&aes, enc, vector, sizeof(vector), longIV,
        sizeof(longIV)/sizeof(byte), resultT, sizeof(resultT), a, sizeof(a)),
        0);
#else
    (void)longIV;
#endif /* Old FIPS */
    /* END wc_AesGcmEncrypt */

#ifdef HAVE_AES_DECRYPT
    ExpectIntEQ(wc_AesGcmDecrypt(NULL, dec, enc, sizeof(enc)/sizeof(byte), iv,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT), a, sizeof(a)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecrypt(&aes, NULL, enc, sizeof(enc)/sizeof(byte), iv,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT), a, sizeof(a)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecrypt(&aes, dec, NULL, sizeof(enc)/sizeof(byte), iv,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT), a, sizeof(a)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecrypt(&aes, dec, enc, sizeof(enc)/sizeof(byte), NULL,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT), a, sizeof(a)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecrypt(&aes, dec, enc, sizeof(enc)/sizeof(byte), iv,
        sizeof(iv)/sizeof(byte), NULL, sizeof(resultT), a, sizeof(a)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesGcmDecrypt(&aes, dec, enc, sizeof(enc)/sizeof(byte), iv,
        sizeof(iv)/sizeof(byte), resultT, sizeof(resultT) + 1, a, sizeof(a)),
        BAD_FUNC_ARG);
    #if ((defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && \
            (HAVE_FIPS_VERSION == 2)) || defined(HAVE_SELFTEST)) && \
            !defined(WOLFSSL_AES_GCM_FIXED_IV_AAD)
            /* FIPS does not check the lower bound of ivSz */
    #else
        ExpectIntEQ(wc_AesGcmDecrypt(&aes, dec, enc, sizeof(enc)/sizeof(byte),
            iv, 0, resultT, sizeof(resultT), a, sizeof(a)), BAD_FUNC_ARG);
    #endif
#endif /* HAVE_AES_DECRYPT */

    wc_AesFree(&aes);
#endif
    return EXPECT_RESULT();

} /* END test_wc_AesGcmEncryptDecrypt */

/*
 * test function for mixed (one-shot encrpytion + stream decryption) AES GCM
 * using a long IV (older FIPS does NOT support long IVs).  Relates to zd15423
 */
static int test_wc_AesGcmMixedEncDecLongIV(void)
{
    EXPECT_DECLS;
#if  (!defined(HAVE_FIPS) || \
      (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))) && \
     !defined(NO_AES) && defined(HAVE_AESGCM) && defined(WOLFSSL_AESGCM_STREAM)
    const byte key[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    const byte in[] = {
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    const byte aad[] = {
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xab, 0xad, 0xda, 0xd2
    };
    Aes aesEnc;
    Aes aesDec;
    byte iv[] = "1234567890abcdefghij";
    byte out[sizeof(in)];
    byte plain[sizeof(in)];
    byte tag[AES_BLOCK_SIZE];

    XMEMSET(&aesEnc, 0, sizeof(Aes));
    XMEMSET(&aesDec, 0, sizeof(Aes));
    XMEMSET(out, 0, sizeof(out));
    XMEMSET(plain, 0, sizeof(plain));
    XMEMSET(tag, 0, sizeof(tag));

    /* Perform one-shot encryption using long IV */
    ExpectIntEQ(wc_AesInit(&aesEnc, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_AesGcmSetKey(&aesEnc, key, sizeof(key)), 0);
    ExpectIntEQ(wc_AesGcmEncrypt(&aesEnc, out, in, sizeof(in), iv, sizeof(iv),
        tag, sizeof(tag), aad, sizeof(aad)), 0);

    /* Perform streaming decryption using long IV */
    ExpectIntEQ(wc_AesInit(&aesDec, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_AesGcmInit(&aesDec, key, sizeof(key), iv, sizeof(iv)), 0);
    ExpectIntEQ(wc_AesGcmDecryptUpdate(&aesDec, plain, out, sizeof(out), aad,
        sizeof(aad)), 0);
    ExpectIntEQ(wc_AesGcmDecryptFinal(&aesDec, tag, sizeof(tag)), 0);
    ExpectIntEQ(XMEMCMP(plain, in, sizeof(in)), 0);

    /* Free resources */
    wc_AesFree(&aesEnc);
    wc_AesFree(&aesDec);
#endif
    return EXPECT_RESULT();

} /* END wc_AesGcmMixedEncDecLongIV */

/*
 * unit test for wc_GmacSetKey()
 */
static int test_wc_GmacSetKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_AES) && defined(HAVE_AESGCM)
    Gmac gmac;
    byte key16[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#ifdef WOLFSSL_AES_192
    byte key24[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
#endif
#ifdef WOLFSSL_AES_256
    byte key32[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#endif
    byte badKey16[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x66
    };
    byte badKey24[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
    byte badKey32[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };

    XMEMSET(&gmac, 0, sizeof(Gmac));

    ExpectIntEQ(wc_AesInit(&gmac.aes, NULL, INVALID_DEVID), 0);

#ifdef WOLFSSL_AES_128
    ExpectIntEQ(wc_GmacSetKey(&gmac, key16, sizeof(key16)/sizeof(byte)), 0);
#endif
#ifdef WOLFSSL_AES_192
    ExpectIntEQ(wc_GmacSetKey(&gmac, key24, sizeof(key24)/sizeof(byte)), 0);
#endif
#ifdef WOLFSSL_AES_256
    ExpectIntEQ(wc_GmacSetKey(&gmac, key32, sizeof(key32)/sizeof(byte)), 0);
#endif

    /* Pass in bad args. */
    ExpectIntEQ(wc_GmacSetKey(NULL, key16, sizeof(key16)/sizeof(byte)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_GmacSetKey(&gmac, NULL, sizeof(key16)/sizeof(byte)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_GmacSetKey(&gmac, badKey16, sizeof(badKey16)/sizeof(byte)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_GmacSetKey(&gmac, badKey24, sizeof(badKey24)/sizeof(byte)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_GmacSetKey(&gmac, badKey32, sizeof(badKey32)/sizeof(byte)),
        BAD_FUNC_ARG);

    wc_AesFree(&gmac.aes);
#endif
    return EXPECT_RESULT();
} /* END test_wc_GmacSetKey */

/*
 * unit test for wc_GmacUpdate
 */
static int test_wc_GmacUpdate(void)
{
    EXPECT_DECLS;
#if !defined(NO_AES) && defined(HAVE_AESGCM)
    Gmac gmac;
#ifdef WOLFSSL_AES_128
    const byte key16[] = {
        0x89, 0xc9, 0x49, 0xe9, 0xc8, 0x04, 0xaf, 0x01,
        0x4d, 0x56, 0x04, 0xb3, 0x94, 0x59, 0xf2, 0xc8
    };
#endif
#ifdef WOLFSSL_AES_192
    byte key24[] = {
        0x41, 0xc5, 0xda, 0x86, 0x67, 0xef, 0x72, 0x52,
        0x20, 0xff, 0xe3, 0x9a, 0xe0, 0xac, 0x59, 0x0a,
        0xc9, 0xfc, 0xa7, 0x29, 0xab, 0x60, 0xad, 0xa0
    };
#endif
#ifdef WOLFSSL_AES_256
   byte key32[] = {
        0x78, 0xdc, 0x4e, 0x0a, 0xaf, 0x52, 0xd9, 0x35,
        0xc3, 0xc0, 0x1e, 0xea, 0x57, 0x42, 0x8f, 0x00,
        0xca, 0x1f, 0xd4, 0x75, 0xf5, 0xda, 0x86, 0xa4,
        0x9c, 0x8d, 0xd7, 0x3d, 0x68, 0xc8, 0xe2, 0x23
    };
#endif
#ifdef WOLFSSL_AES_128
    const byte authIn[] = {
        0x82, 0xad, 0xcd, 0x63, 0x8d, 0x3f, 0xa9, 0xd9,
        0xf3, 0xe8, 0x41, 0x00, 0xd6, 0x1e, 0x07, 0x77
    };
#endif
#ifdef WOLFSSL_AES_192
    const byte authIn2[] = {
       0x8b, 0x5c, 0x12, 0x4b, 0xef, 0x6e, 0x2f, 0x0f,
       0xe4, 0xd8, 0xc9, 0x5c, 0xd5, 0xfa, 0x4c, 0xf1
    };
#endif
    const byte authIn3[] = {
        0xb9, 0x6b, 0xaa, 0x8c, 0x1c, 0x75, 0xa6, 0x71,
        0xbf, 0xb2, 0xd0, 0x8d, 0x06, 0xbe, 0x5f, 0x36
    };
#ifdef WOLFSSL_AES_128
    const byte tag1[] = { /* Known. */
        0x88, 0xdb, 0x9d, 0x62, 0x17, 0x2e, 0xd0, 0x43,
        0xaa, 0x10, 0xf1, 0x6d, 0x22, 0x7d, 0xc4, 0x1b
    };
#endif
#ifdef WOLFSSL_AES_192
    const byte tag2[] = { /* Known */
        0x20, 0x4b, 0xdb, 0x1b, 0xd6, 0x21, 0x54, 0xbf,
        0x08, 0x92, 0x2a, 0xaa, 0x54, 0xee, 0xd7, 0x05
    };
#endif
    const byte tag3[] = { /* Known */
        0x3e, 0x5d, 0x48, 0x6a, 0xa2, 0xe3, 0x0b, 0x22,
        0xe0, 0x40, 0xb8, 0x57, 0x23, 0xa0, 0x6e, 0x76
    };
#ifdef WOLFSSL_AES_128
    const byte iv[] = {
        0xd1, 0xb1, 0x04, 0xc8, 0x15, 0xbf, 0x1e, 0x94,
        0xe2, 0x8c, 0x8f, 0x16
    };
#endif
#ifdef WOLFSSL_AES_192
    const byte iv2[] = {
        0x05, 0xad, 0x13, 0xa5, 0xe2, 0xc2, 0xab, 0x66,
        0x7e, 0x1a, 0x6f, 0xbc
    };
#endif
    const byte iv3[] = {
        0xd7, 0x9c, 0xf2, 0x2d, 0x50, 0x4c, 0xc7, 0x93,
        0xc3, 0xfb, 0x6c, 0x8a
    };
    byte tagOut[16];
    byte tagOut2[24];
    byte tagOut3[32];

    /* Init stack variables. */
    XMEMSET(&gmac, 0, sizeof(Gmac));
    XMEMSET(tagOut, 0, sizeof(tagOut));
    XMEMSET(tagOut2, 0, sizeof(tagOut2));
    XMEMSET(tagOut3, 0, sizeof(tagOut3));

    ExpectIntEQ(wc_AesInit(&gmac.aes, NULL, INVALID_DEVID), 0);

#ifdef WOLFSSL_AES_128
    ExpectIntEQ(wc_GmacSetKey(&gmac, key16, sizeof(key16)), 0);
    ExpectIntEQ(wc_GmacUpdate(&gmac, iv, sizeof(iv), authIn, sizeof(authIn),
        tagOut, sizeof(tag1)), 0);
    ExpectIntEQ(XMEMCMP(tag1, tagOut, sizeof(tag1)), 0);
#endif

#ifdef WOLFSSL_AES_192
    ExpectNotNull(XMEMSET(&gmac, 0, sizeof(Gmac)));
    ExpectIntEQ(wc_GmacSetKey(&gmac, key24, sizeof(key24)/sizeof(byte)), 0);
    ExpectIntEQ(wc_GmacUpdate(&gmac, iv2, sizeof(iv2), authIn2, sizeof(authIn2),
        tagOut2, sizeof(tag2)), 0);
    ExpectIntEQ(XMEMCMP(tagOut2, tag2, sizeof(tag2)), 0);
#endif

#ifdef WOLFSSL_AES_256
    ExpectNotNull(XMEMSET(&gmac, 0, sizeof(Gmac)));
    ExpectIntEQ(wc_GmacSetKey(&gmac, key32, sizeof(key32)/sizeof(byte)), 0);
    ExpectIntEQ(wc_GmacUpdate(&gmac, iv3, sizeof(iv3), authIn3, sizeof(authIn3),
        tagOut3, sizeof(tag3)), 0);
    ExpectIntEQ(XMEMCMP(tag3, tagOut3, sizeof(tag3)), 0);
#endif

    /* Pass bad args. */
    ExpectIntEQ(wc_GmacUpdate(NULL, iv3, sizeof(iv3), authIn3, sizeof(authIn3),
        tagOut3, sizeof(tag3)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_GmacUpdate(&gmac, iv3, sizeof(iv3), authIn3, sizeof(authIn3),
        tagOut3, sizeof(tag3) - 5), BAD_FUNC_ARG);
    ExpectIntEQ(wc_GmacUpdate(&gmac, iv3, sizeof(iv3), authIn3, sizeof(authIn3),
        tagOut3, sizeof(tag3) + 1),  BAD_FUNC_ARG);

    wc_AesFree(&gmac.aes);
#endif
    return EXPECT_RESULT();
} /* END test_wc_GmacUpdate */


/*
 * testing wc_CamelliaSetKey
 */
static int test_wc_CamelliaSetKey(void)
{
    EXPECT_DECLS;
#ifdef HAVE_CAMELLIA
    Camellia camellia;
    /*128-bit key*/
    static const byte key16[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    /* 192-bit key */
    static const byte key24[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77
    };
    /* 256-bit key */
    static const byte key32[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
    };
    static const byte iv[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };

    ExpectIntEQ(wc_CamelliaSetKey(&camellia, key16, (word32)sizeof(key16), iv),
        0);
    ExpectIntEQ(wc_CamelliaSetKey(&camellia, key16, (word32)sizeof(key16),
        NULL), 0);
    ExpectIntEQ(wc_CamelliaSetKey(&camellia, key24, (word32)sizeof(key24), iv),
        0);
    ExpectIntEQ(wc_CamelliaSetKey(&camellia, key24, (word32)sizeof(key24),
        NULL), 0);
    ExpectIntEQ(wc_CamelliaSetKey(&camellia, key32, (word32)sizeof(key32), iv),
        0);
    ExpectIntEQ(wc_CamelliaSetKey(&camellia, key32, (word32)sizeof(key32),
        NULL), 0);

    /* Bad args. */
    ExpectIntEQ(wc_CamelliaSetKey(NULL, key32, (word32)sizeof(key32), iv),
        BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_CammeliaSetKey */

/*
 * Testing wc_CamelliaSetIV()
 */
static int test_wc_CamelliaSetIV(void)
{
    EXPECT_DECLS;
#ifdef HAVE_CAMELLIA
    Camellia    camellia;
    static const byte iv[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };

    ExpectIntEQ(wc_CamelliaSetIV(&camellia, iv), 0);
    ExpectIntEQ(wc_CamelliaSetIV(&camellia, NULL), 0);

    /* Bad args. */
    ExpectIntEQ(wc_CamelliaSetIV(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_CamelliaSetIV(NULL, iv), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_CamelliaSetIV*/

/*
 * Test wc_CamelliaEncryptDirect and wc_CamelliaDecryptDirect
 */
static int test_wc_CamelliaEncryptDecryptDirect(void)
{
    EXPECT_DECLS;
#ifdef HAVE_CAMELLIA
    Camellia camellia;
    static const byte key24[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77
    };
    static const byte iv[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    static const byte plainT[] = {
        0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
        0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A
    };
    byte    enc[sizeof(plainT)];
    byte    dec[sizeof(enc)];

    /* Init stack variables.*/
    XMEMSET(enc, 0, 16);
    XMEMSET(enc, 0, 16);

    ExpectIntEQ(wc_CamelliaSetKey(&camellia, key24, (word32)sizeof(key24), iv),
        0);
    ExpectIntEQ(wc_CamelliaEncryptDirect(&camellia, enc, plainT), 0);
    ExpectIntEQ(wc_CamelliaDecryptDirect(&camellia, dec, enc), 0);
    ExpectIntEQ(XMEMCMP(plainT, dec, CAMELLIA_BLOCK_SIZE), 0);

    /* Pass bad args. */
    ExpectIntEQ(wc_CamelliaEncryptDirect(NULL, enc, plainT), BAD_FUNC_ARG);
    ExpectIntEQ(wc_CamelliaEncryptDirect(&camellia, NULL, plainT),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_CamelliaEncryptDirect(&camellia, enc, NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_CamelliaDecryptDirect(NULL, dec, enc), BAD_FUNC_ARG);
    ExpectIntEQ(wc_CamelliaDecryptDirect(&camellia, NULL, enc), BAD_FUNC_ARG);
    ExpectIntEQ(wc_CamelliaDecryptDirect(&camellia, dec, NULL), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test-wc_CamelliaEncryptDecryptDirect */

/*
 * Testing wc_CamelliaCbcEncrypt and wc_CamelliaCbcDecrypt
 */
static int test_wc_CamelliaCbcEncryptDecrypt(void)
{
    EXPECT_DECLS;
#ifdef HAVE_CAMELLIA
    Camellia camellia;
    static const byte key24[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77
    };
    static const byte plainT[] = {
        0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
        0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A
    };
    byte    enc[CAMELLIA_BLOCK_SIZE];
    byte    dec[CAMELLIA_BLOCK_SIZE];

    /* Init stack variables. */
    XMEMSET(enc, 0, CAMELLIA_BLOCK_SIZE);
    XMEMSET(enc, 0, CAMELLIA_BLOCK_SIZE);

    ExpectIntEQ(wc_CamelliaSetKey(&camellia, key24, (word32)sizeof(key24),
        NULL), 0);
    ExpectIntEQ(wc_CamelliaCbcEncrypt(&camellia, enc, plainT,
        CAMELLIA_BLOCK_SIZE), 0);

    ExpectIntEQ(wc_CamelliaSetKey(&camellia, key24, (word32)sizeof(key24),
        NULL), 0);
    ExpectIntEQ(wc_CamelliaCbcDecrypt(&camellia, dec, enc, CAMELLIA_BLOCK_SIZE),
        0);
    ExpectIntEQ(XMEMCMP(plainT, dec, CAMELLIA_BLOCK_SIZE), 0);

    /* Pass in bad args. */
    ExpectIntEQ(wc_CamelliaCbcEncrypt(NULL, enc, plainT, CAMELLIA_BLOCK_SIZE),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_CamelliaCbcEncrypt(&camellia, NULL, plainT,
        CAMELLIA_BLOCK_SIZE), BAD_FUNC_ARG);
    ExpectIntEQ(wc_CamelliaCbcEncrypt(&camellia, enc, NULL,
        CAMELLIA_BLOCK_SIZE), BAD_FUNC_ARG);

    ExpectIntEQ(wc_CamelliaCbcDecrypt(NULL, dec, enc, CAMELLIA_BLOCK_SIZE),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_CamelliaCbcDecrypt(&camellia, NULL, enc,
        CAMELLIA_BLOCK_SIZE), BAD_FUNC_ARG);
    ExpectIntEQ(wc_CamelliaCbcDecrypt(&camellia, dec, NULL,
        CAMELLIA_BLOCK_SIZE), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();
} /* END test_wc_CamelliaCbcEncryptDecrypt */


/*
 * Testing wc_Arc4SetKey()
 */
static int test_wc_Arc4SetKey(void)
{
    EXPECT_DECLS;
#ifndef NO_RC4
    Arc4 arc;
    const char* key = "\x01\x23\x45\x67\x89\xab\xcd\xef";
    int keyLen = 8;

    ExpectIntEQ(wc_Arc4SetKey(&arc, (byte*)key, keyLen), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_Arc4SetKey(NULL, (byte*)key, keyLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Arc4SetKey(&arc, NULL      , keyLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Arc4SetKey(&arc, (byte*)key, 0     ), BAD_FUNC_ARG);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Arc4SetKey */

/*
 * Testing wc_Arc4Process for ENC/DEC.
 */
static int test_wc_Arc4Process(void)
{
    EXPECT_DECLS;
#ifndef NO_RC4
    Arc4 enc;
    Arc4 dec;
    const char* key = "\x01\x23\x45\x67\x89\xab\xcd\xef";
    int keyLen = 8;
    const char* input = "\x01\x23\x45\x67\x89\xab\xcd\xef";
    byte cipher[8];
    byte plain[8];

    /* Init stack variables */
    XMEMSET(&enc, 0, sizeof(Arc4));
    XMEMSET(&dec, 0, sizeof(Arc4));
    XMEMSET(cipher, 0, sizeof(cipher));
    XMEMSET(plain, 0, sizeof(plain));

    /* Use for async. */
    ExpectIntEQ(wc_Arc4Init(&enc, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_Arc4Init(&dec, NULL, INVALID_DEVID), 0);

    ExpectIntEQ(wc_Arc4SetKey(&enc, (byte*)key, keyLen), 0);
    ExpectIntEQ(wc_Arc4SetKey(&dec, (byte*)key, keyLen), 0);

    ExpectIntEQ(wc_Arc4Process(&enc, cipher, (byte*)input, keyLen), 0);
    ExpectIntEQ(wc_Arc4Process(&dec, plain, cipher, keyLen), 0);
    ExpectIntEQ(XMEMCMP(plain, input, keyLen), 0);

    /* Bad args. */
    ExpectIntEQ(wc_Arc4Process(NULL, plain, cipher, keyLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Arc4Process(&dec, NULL, cipher, keyLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Arc4Process(&dec, plain, NULL, keyLen), BAD_FUNC_ARG);

    wc_Arc4Free(&enc);
    wc_Arc4Free(&dec);
#endif
    return EXPECT_RESULT();

} /* END test_wc_Arc4Process */


/*
 * Testing wc_Init RsaKey()
 */
static int test_wc_InitRsaKey(void)
{
    EXPECT_DECLS;
#ifndef NO_RSA
    RsaKey key;

    XMEMSET(&key, 0, sizeof(RsaKey));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);

    /* Test bad args. */
#ifndef HAVE_USER_RSA
    ExpectIntEQ(wc_InitRsaKey(NULL, HEAP_HINT), BAD_FUNC_ARG);
#else
    ExpectIntEQ(wc_InitRsaKey(NULL, HEAP_HINT), USER_CRYPTO_ERROR);
#endif

    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_InitRsaKey */


/*
 * Testing wc_RsaPrivateKeyDecode()
 */
static int test_wc_RsaPrivateKeyDecode(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && (defined(USE_CERT_BUFFERS_1024)\
        || defined(USE_CERT_BUFFERS_2048)) && !defined(HAVE_FIPS)
    RsaKey key;
    byte*  tmp = NULL;
    word32 idx = 0;
    int    bytes = 0;

    XMEMSET(&key, 0, sizeof(RsaKey));

    ExpectNotNull(tmp = (byte*)XMALLOC(FOURK_BUF, NULL,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    if (tmp != NULL) {
    #ifdef USE_CERT_BUFFERS_1024
        XMEMCPY(tmp, client_key_der_1024, sizeof_client_key_der_1024);
        bytes = sizeof_client_key_der_1024;
    #else
        XMEMCPY(tmp, client_key_der_2048, sizeof_client_key_der_2048);
        bytes = sizeof_client_key_der_2048;
    #endif /* Use cert buffers. */
    }

    ExpectIntEQ(wc_RsaPrivateKeyDecode(tmp, &idx, &key, (word32)bytes), 0);
#ifndef HAVE_USER_RSA
    /* Test bad args. */
    ExpectIntEQ(wc_RsaPrivateKeyDecode(NULL, &idx, &key, (word32)bytes),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPrivateKeyDecode(tmp, NULL, &key, (word32)bytes),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPrivateKeyDecode(tmp, &idx, NULL, (word32)bytes),
        BAD_FUNC_ARG);
#else
    /* Test bad args. User RSA. */
    ExpectIntEQ(wc_RsaPrivateKeyDecode(NULL, &idx, &key, (word32)bytes),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaPrivateKeyDecode(tmp, NULL, &key, (word32)bytes),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaPrivateKeyDecode(tmp, &idx, NULL, (word32)bytes),
        USER_CRYPTO_ERROR);
#endif

    XFREE(tmp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
#endif
    return EXPECT_RESULT();

} /* END test_wc_RsaPrivateKeyDecode */

/*
 * Testing wc_RsaPublicKeyDecode()
 */
static int test_wc_RsaPublicKeyDecode(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && (defined(USE_CERT_BUFFERS_1024)\
        || defined(USE_CERT_BUFFERS_2048)) && !defined(HAVE_FIPS)
    RsaKey keyPub;
    byte*  tmp = NULL;
    word32 idx = 0;
    int    bytes = 0;
    word32 keySz = 0;
    word32 tstKeySz = 0;
#if defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM)
    XFILE f = XBADFILE;
    const char* rsaPssPubKey = "./certs/rsapss/ca-rsapss-key.der";
    const char* rsaPssPubKeyNoParams = "./certs/rsapss/ca-3072-rsapss-key.der";
    byte buf[4096];
#endif

    XMEMSET(&keyPub, 0, sizeof(RsaKey));

    ExpectNotNull(tmp = (byte*)XMALLOC(GEN_BUF, NULL, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRsaKey(&keyPub, HEAP_HINT), 0);
    if (tmp != NULL) {
    #ifdef USE_CERT_BUFFERS_1024
        XMEMCPY(tmp, client_keypub_der_1024, sizeof_client_keypub_der_1024);
        bytes = sizeof_client_keypub_der_1024;
        keySz = 1024;
    #else
        XMEMCPY(tmp, client_keypub_der_2048, sizeof_client_keypub_der_2048);
        bytes = sizeof_client_keypub_der_2048;
        keySz = 2048;
    #endif
    }

    ExpectIntEQ(wc_RsaPublicKeyDecode(tmp, &idx, &keyPub, (word32)bytes), 0);
#ifndef HAVE_USER_RSA
    /* Pass in bad args. */
    ExpectIntEQ(wc_RsaPublicKeyDecode(NULL, &idx, &keyPub, (word32)bytes),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPublicKeyDecode(tmp, NULL, &keyPub, (word32)bytes),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPublicKeyDecode(tmp, &idx, NULL, (word32)bytes),
        BAD_FUNC_ARG);
#else
    /* Pass in bad args. */
    ExpectIntEQ(wc_RsaPublicKeyDecode(NULL, &idx, &keyPub, (word32)bytes),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaPublicKeyDecode(tmp, NULL, &keyPub, (word32)bytes),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaPublicKeyDecode(tmp, &idx, NULL, (word32)bytes),
        USER_CRYPTO_ERROR);
#endif

    DoExpectIntEQ(wc_FreeRsaKey(&keyPub), 0);

    /* Test for getting modulus key size */
    idx = 0;
    ExpectIntEQ(wc_RsaPublicKeyDecode_ex(tmp, &idx, (word32)bytes, NULL,
        &tstKeySz, NULL, NULL), 0);
    ExpectIntEQ(tstKeySz, keySz/8);

#if defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM)
    ExpectTrue((f = XFOPEN(rsaPssPubKey, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    idx = 0;
    ExpectIntEQ(wc_RsaPublicKeyDecode_ex(buf, &idx, bytes, NULL, NULL, NULL,
        NULL), 0);
    ExpectTrue((f = XFOPEN(rsaPssPubKeyNoParams, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    idx = 0;
    ExpectIntEQ(wc_RsaPublicKeyDecode_ex(buf, &idx, bytes, NULL, NULL, NULL,
        NULL), 0);
#endif

    XFREE(tmp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_RsaPublicKeyDecode */

/*
 * Testing wc_RsaPublicKeyDecodeRaw()
 */
static int test_wc_RsaPublicKeyDecodeRaw(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA)
    RsaKey     key;
    const byte n = 0x23;
    const byte e = 0x03;
    int        nSz = sizeof(n);
    int        eSz = sizeof(e);

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_RsaPublicKeyDecodeRaw(&n, nSz, &e, eSz, &key), 0);
#ifndef HAVE_USER_RSA
    /* Pass in bad args. */
    ExpectIntEQ(wc_RsaPublicKeyDecodeRaw(NULL, nSz, &e, eSz, &key),
       BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPublicKeyDecodeRaw(&n, nSz, NULL, eSz, &key),
       BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPublicKeyDecodeRaw(&n, nSz, &e, eSz, NULL),
       BAD_FUNC_ARG);
#else
    /* Pass in bad args. User RSA. */
    ExpectIntEQ(wc_RsaPublicKeyDecodeRaw(NULL, nSz, &e, eSz, &key),
       USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaPublicKeyDecodeRaw(&n, nSz, NULL, eSz, &key),
       USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaPublicKeyDecodeRaw(&n, nSz, &e, eSz, NULL),
       USER_CRYPTO_ERROR);
#endif

    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
#endif
    return EXPECT_RESULT();

} /* END test_wc_RsaPublicKeyDecodeRaw */


#if (!defined(NO_RSA) || !defined(HAVE_FAST_RSA)) && defined(WOLFSSL_KEY_GEN)
    /* In FIPS builds, wc_MakeRsaKey() will return an error if it cannot find
     * a probable prime in 5*(modLen/2) attempts. In non-FIPS builds, it keeps
     * trying until it gets a probable prime. */
    #ifdef HAVE_FIPS
        static int MakeRsaKeyRetry(RsaKey* key, int size, long e, WC_RNG* rng)
        {
            int ret;

            for (;;) {
                ret = wc_MakeRsaKey(key, size, e, rng);
                if (ret != PRIME_GEN_E) break;
                fprintf(stderr, "MakeRsaKey couldn't find prime; "
                                "trying again.\n");
            }

            return ret;
        }
        #define MAKE_RSA_KEY(a, b, c, d) MakeRsaKeyRetry(a, b, c, d)
    #else
        #define MAKE_RSA_KEY(a, b, c, d) wc_MakeRsaKey(a, b, c, d)
    #endif
#endif


/*
 * Testing wc_MakeRsaKey()
 */
static int test_wc_MakeRsaKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)

    RsaKey genKey;
    WC_RNG rng;
#if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    int bits = 1024;
#else
    int bits = 2048;
#endif

    XMEMSET(&genKey, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRsaKey(&genKey, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&genKey, bits, WC_RSA_EXPONENT, &rng), 0);
    DoExpectIntEQ(wc_FreeRsaKey(&genKey), 0);

#ifndef HAVE_USER_RSA
    /* Test bad args. */
    ExpectIntEQ(MAKE_RSA_KEY(NULL, bits, WC_RSA_EXPONENT, &rng), BAD_FUNC_ARG);
    ExpectIntEQ(MAKE_RSA_KEY(&genKey, bits, WC_RSA_EXPONENT, NULL),
        BAD_FUNC_ARG);
    /* e < 3 */
    ExpectIntEQ(MAKE_RSA_KEY(&genKey, bits, 2, &rng), BAD_FUNC_ARG);
    /* e & 1 == 0 */
    ExpectIntEQ(MAKE_RSA_KEY(&genKey, bits, 6, &rng), BAD_FUNC_ARG);
#else
    /* Test bad args. */
    ExpectIntEQ(MAKE_RSA_KEY(NULL, bits, WC_RSA_EXPONENT, &rng),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(MAKE_RSA_KEY(&genKey, bits, WC_RSA_EXPONENT, NULL),
        USER_CRYPTO_ERROR);
    /* e < 3 */
    ExpectIntEQ(MAKE_RSA_KEY(&genKey, bits, 2, &rng), USER_CRYPTO_ERROR);
    /* e & 1 == 0 */
    ExpectIntEQ(MAKE_RSA_KEY(&genKey, bits, 6, &rng), USER_CRYPTO_ERROR);
#endif /* HAVE_USER_RSA */

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_MakeRsaKey */

/*
 * Test the bounds checking on the cipher text versus the key modulus.
 * 1. Make a new RSA key.
 * 2. Set c to 1.
 * 3. Decrypt c into k. (error)
 * 4. Copy the key modulus to c and sub 1 from the copy.
 * 5. Decrypt c into k. (error)
 * Valid bounds test cases are covered by all the other RSA tests.
 */
static int test_RsaDecryptBoundsCheck(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WC_RSA_NO_PADDING) && \
    (defined(USE_CERT_BUFFERS_1024) || defined(USE_CERT_BUFFERS_2048)) && \
    defined(WOLFSSL_PUBLIC_MP) && !defined(NO_RSA_BOUNDS_CHECK)
    WC_RNG rng;
    RsaKey key;
    byte flatC[256];
    word32 flatCSz;
    byte out[256];
    word32 outSz = sizeof(out);

    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    if (EXPECT_SUCCESS()) {
        const byte* derKey;
        word32 derKeySz;
        word32 idx = 0;

        #ifdef USE_CERT_BUFFERS_1024
            derKey = server_key_der_1024;
            derKeySz = (word32)sizeof_server_key_der_1024;
            flatCSz = 128;
        #else
            derKey = server_key_der_2048;
            derKeySz = (word32)sizeof_server_key_der_2048;
            flatCSz = 256;
        #endif

        ExpectIntEQ(wc_RsaPrivateKeyDecode(derKey, &idx, &key, derKeySz), 0);
    }

    if (EXPECT_SUCCESS()) {
        XMEMSET(flatC, 0, flatCSz);
        flatC[flatCSz-1] = 1;

        ExpectIntEQ(wc_RsaDirect(flatC, flatCSz, out, &outSz, &key,
            RSA_PRIVATE_DECRYPT, &rng), RSA_OUT_OF_RANGE_E);
        if (EXPECT_SUCCESS()) {
            mp_int c;
            ExpectIntEQ(mp_init_copy(&c, &key.n), 0);
            ExpectIntEQ(mp_sub_d(&c, 1, &c), 0);
            ExpectIntEQ(mp_to_unsigned_bin(&c, flatC), 0);
            ExpectIntEQ(wc_RsaDirect(flatC, flatCSz, out, &outSz, &key,
                RSA_PRIVATE_DECRYPT, NULL), RSA_OUT_OF_RANGE_E);
            mp_clear(&c);
        }
    }

    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_RsaDecryptBoundsCheck */

/*
 * Testing wc_SetKeyUsage()
 */
static int test_wc_SetKeyUsage(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN) && !defined(HAVE_FIPS)
    Cert myCert;

    ExpectIntEQ(wc_InitCert(&myCert), 0);

    ExpectIntEQ(wc_SetKeyUsage(&myCert, "keyEncipherment,keyAgreement"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "digitalSignature,nonRepudiation"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "contentCommitment,encipherOnly"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "decipherOnly"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "cRLSign,keyCertSign"), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_SetKeyUsage(NULL, "decipherOnly"), BAD_FUNC_ARG);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, ""), KEYUSAGE_E);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, ","), KEYUSAGE_E);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "digitalSignature, cRLSign"),
        KEYUSAGE_E);
#endif
    return EXPECT_RESULT();
} /* END  test_wc_SetKeyUsage */

/*
 * Testing wc_CheckProbablePrime()
 */
static int test_wc_CheckProbablePrime(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_SELFTEST) && \
 !defined(HAVE_FIPS) && defined(WC_RSA_BLINDING)
#define CHECK_PROBABLE_PRIME_KEY_BITS 2048
    RsaKey key;
    WC_RNG rng;
    byte   e[3];
    word32 eSz = (word32)sizeof(e);
    byte   n[CHECK_PROBABLE_PRIME_KEY_BITS / 8];
    word32 nSz = (word32)sizeof(n);
    byte   d[CHECK_PROBABLE_PRIME_KEY_BITS / 8];
    word32 dSz = (word32)sizeof(d);
    byte   p[CHECK_PROBABLE_PRIME_KEY_BITS / 8 / 2];
    word32 pSz = (word32)sizeof(p);
    byte   q[CHECK_PROBABLE_PRIME_KEY_BITS / 8 / 2];
    word32 qSz = (word32)sizeof(q);
    int    nlen = CHECK_PROBABLE_PRIME_KEY_BITS;
    int*   isPrime;
    int    test[5];
    isPrime = test;

    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_RsaSetRNG(&key, &rng), 0);
    ExpectIntEQ(wc_MakeRsaKey(&key, CHECK_PROBABLE_PRIME_KEY_BITS,
        WC_RSA_EXPONENT, &rng), 0);
    PRIVATE_KEY_UNLOCK();
    ExpectIntEQ(wc_RsaExportKey(&key, e, &eSz, n, &nSz, d, &dSz, p, &pSz, q,
        &qSz), 0);
    PRIVATE_KEY_LOCK();

    /* Bad cases */
    ExpectIntEQ(wc_CheckProbablePrime(NULL, pSz, q, qSz, e, eSz, nlen, isPrime),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_CheckProbablePrime(p, 0, q, qSz, e, eSz, nlen, isPrime),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_CheckProbablePrime(p, pSz, NULL, qSz, e, eSz, nlen, isPrime),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_CheckProbablePrime(p, pSz, q, 0, e, eSz, nlen, isPrime),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_CheckProbablePrime(p, pSz, q, qSz, NULL, eSz, nlen, isPrime),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_CheckProbablePrime(p, pSz, q, qSz, e, 0, nlen, isPrime),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_CheckProbablePrime(NULL, 0, NULL, 0, NULL, 0, nlen, isPrime),
        BAD_FUNC_ARG);

    /* Good case */
    ExpectIntEQ(wc_CheckProbablePrime(p, pSz, q, qSz, e, eSz, nlen, isPrime),
        0);

    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    wc_FreeRng(&rng);
#undef CHECK_PROBABLE_PRIME_KEY_BITS
#endif
    return EXPECT_RESULT();
} /* END  test_wc_CheckProbablePrime */
/*
 * Testing wc_RsaPSS_Verify()
 */
static int test_wc_RsaPSS_Verify(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_SELFTEST) && \
 !defined(HAVE_FIPS) && defined(WC_RSA_BLINDING) && defined(WC_RSA_PSS)
    RsaKey        key;
    WC_RNG        rng;
    int           sz = 256;
    const char*   szMessage = "This is the string to be signed";
    unsigned char pSignature[2048/8]; /* 2048 is RSA_KEY_SIZE */
    unsigned char pDecrypted[2048/8];
    byte*         pt = pDecrypted;
    word32        outLen = sizeof(pDecrypted);

    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_RsaSetRNG(&key, &rng), 0);
    ExpectIntEQ(wc_MakeRsaKey(&key, 2048, WC_RSA_EXPONENT, &rng), 0);

    ExpectIntGT(sz = wc_RsaPSS_Sign((byte*)szMessage,
        (word32)XSTRLEN(szMessage)+1, pSignature, sizeof(pSignature),
        WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key, &rng), 0);

    /* Bad cases */
    ExpectIntEQ(wc_RsaPSS_Verify(NULL, sz, pt, outLen,
        WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPSS_Verify(pSignature, 0, pt, outLen,
        WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPSS_Verify(pSignature, sz, NULL, outLen,
        WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPSS_Verify(NULL, 0, NULL, outLen,
        WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);

    /* Good case */
    ExpectIntGT(wc_RsaPSS_Verify(pSignature, sz, pt, outLen,
        WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), 0);

    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    wc_FreeRng(&rng);
#endif
    return EXPECT_RESULT();
} /* END  test_wc_RsaPSS_Verify */
/*
 * Testing wc_RsaPSS_VerifyCheck()
 */
static int test_wc_RsaPSS_VerifyCheck(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_SELFTEST) && \
 !defined(HAVE_FIPS) && defined(WC_RSA_BLINDING) && defined(WC_RSA_PSS)
    RsaKey        key;
    WC_RNG        rng;
    int           sz = 256; /* 2048/8 */
    byte          digest[32];
    word32        digestSz = sizeof(digest);
    unsigned char pSignature[2048/8]; /* 2048 is RSA_KEY_SIZE */
    word32        pSignatureSz = sizeof(pSignature);
    unsigned char pDecrypted[2048/8];
    byte*         pt = pDecrypted;
    word32        outLen = sizeof(pDecrypted);

    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    XMEMSET(digest, 0, sizeof(digest));
    XMEMSET(pSignature, 0, sizeof(pSignature));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_RsaSetRNG(&key, &rng), 0);
    ExpectIntEQ(wc_MakeRsaKey(&key, 2048, WC_RSA_EXPONENT, &rng), 0);
    ExpectTrue((digestSz = wc_HashGetDigestSize(WC_HASH_TYPE_SHA256)) > 0);
    ExpectIntEQ(wc_Hash(WC_HASH_TYPE_SHA256, pSignature, sz, digest, digestSz),
        0);

    ExpectIntGT(sz = wc_RsaPSS_Sign(digest, digestSz, pSignature, pSignatureSz,
        WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key, &rng), 0);

    /* Bad cases */
    ExpectIntEQ(wc_RsaPSS_VerifyCheck(NULL, sz, pt, outLen, digest,
        digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPSS_VerifyCheck(pSignature, 0, pt, outLen, digest,
        digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPSS_VerifyCheck(pSignature, sz, NULL, outLen, digest,
        digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPSS_VerifyCheck(NULL, 0, NULL, outLen, digest,
        digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);

    /* Good case */
    ExpectIntGT(wc_RsaPSS_VerifyCheck(pSignature, sz, pt, outLen, digest,
        digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), 0);

    ExpectIntEQ(wc_FreeRsaKey(&key), 0);
    wc_FreeRng(&rng);
#endif
    return EXPECT_RESULT();
} /* END  test_wc_RsaPSS_VerifyCheck */
/*
 * Testing wc_RsaPSS_VerifyCheckInline()
 */
static int test_wc_RsaPSS_VerifyCheckInline(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_SELFTEST) && \
 !defined(HAVE_FIPS) && defined(WC_RSA_BLINDING) && defined(WC_RSA_PSS)
    RsaKey        key;
    WC_RNG        rng;
    int           sz = 256;
    byte          digest[32];
    word32        digestSz = sizeof(digest);
    unsigned char pSignature[2048/8]; /* 2048 is RSA_KEY_SIZE */
    unsigned char pDecrypted[2048/8];
    byte*         pt = pDecrypted;

    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    XMEMSET(digest, 0, sizeof(digest));
    XMEMSET(pSignature, 0, sizeof(pSignature));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_RsaSetRNG(&key, &rng), 0);
    ExpectIntEQ(wc_MakeRsaKey(&key, 2048, WC_RSA_EXPONENT, &rng), 0);
    ExpectTrue((digestSz = wc_HashGetDigestSize(WC_HASH_TYPE_SHA256)) > 0);
    ExpectIntEQ(wc_Hash(WC_HASH_TYPE_SHA256, pSignature, sz, digest, digestSz),
        0);

    ExpectIntGT(sz = wc_RsaPSS_Sign(digest, digestSz, pSignature,
        sizeof(pSignature), WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key, &rng), 0);

    /* Bad Cases */
    ExpectIntEQ(wc_RsaPSS_VerifyCheckInline(NULL, sz, &pt, digest,
        digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPSS_VerifyCheckInline(pSignature, 0, NULL, digest,
        digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPSS_VerifyCheckInline(NULL, 0, &pt, digest,
        digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaPSS_VerifyCheckInline(pSignature, sz, &pt, digest,
        digestSz, WC_HASH_TYPE_SHA, WC_MGF1SHA256, &key), BAD_FUNC_ARG);

    /* Good case */
    ExpectIntGT(wc_RsaPSS_VerifyCheckInline(pSignature, sz, &pt, digest,
        digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key), 0);

    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    wc_FreeRng(&rng);
#endif
    return EXPECT_RESULT();
} /* END  test_wc_RsaPSS_VerifyCheckInline */

#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
static void sample_mutex_cb (int flag, int type, const char* file, int line)
{
    (void)flag;
    (void)type;
    (void)file;
    (void)line;
}
#endif
/*
 * Testing wc_LockMutex_ex
 */
static int test_wc_LockMutex_ex(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    int flag = CRYPTO_LOCK;
    int type = 0;
    const char* file = "./test-LockMutex_ex.txt";
    int line = 0;

    /* without SetMutexCb */
    ExpectIntEQ(wc_LockMutex_ex(flag, type, file, line), BAD_STATE_E);
    /* with SetMutexCb */
    ExpectIntEQ(wc_SetMutexCb(sample_mutex_cb), 0);
    ExpectIntEQ(wc_LockMutex_ex(flag, type, file, line), 0);
    ExpectIntEQ(wc_SetMutexCb(NULL), 0);
#endif
    return EXPECT_RESULT();
} /* End test_wc_LockMutex_ex*/
/*
 * Testing wc_SetMutexCb
 */
static int test_wc_SetMutexCb(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    ExpectIntEQ(wc_SetMutexCb(sample_mutex_cb), 0);
    ExpectIntEQ(wc_SetMutexCb(NULL), 0);
#endif
    return EXPECT_RESULT();
} /* End test_wc_SetMutexCb*/

/*
 * Testing wc_RsaKeyToDer()
 */
static int test_wc_RsaKeyToDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey genKey;
    WC_RNG rng;
    byte*  der = NULL;
#if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    int     bits = 1024;
    word32  derSz = 611;
    /* (2 x 128) + 2 (possible leading 00) + (5 x 64) + 5 (possible leading 00)
       + 3 (e) + 8 (ASN tag) + 10 (ASN length) + 4 seqSz + 3 version */
#else
    int     bits = 2048;
    word32  derSz = 1196;
    /* (2 x 256) + 2 (possible leading 00) + (5 x 128) + 5 (possible leading 00)
       + 3 (e) + 8 (ASN tag) + 17 (ASN length) + 4 seqSz + 3 version */
#endif

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&genKey, 0, sizeof(genKey));

    ExpectNotNull(der = (byte*)XMALLOC(derSz, NULL, DYNAMIC_TYPE_TMP_BUFFER));
    /* Init structures. */
    ExpectIntEQ(wc_InitRsaKey(&genKey, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    /* Make key. */
    ExpectIntEQ(MAKE_RSA_KEY(&genKey, bits, WC_RSA_EXPONENT, &rng), 0);

    ExpectIntGT(wc_RsaKeyToDer(&genKey, der, derSz), 0);
#ifndef HAVE_USER_RSA
    /* Pass good/bad args. */
    ExpectIntEQ(wc_RsaKeyToDer(NULL, der, FOURK_BUF), BAD_FUNC_ARG);
    /* Get just the output length */
    ExpectIntGT(wc_RsaKeyToDer(&genKey, NULL, 0), 0);
    /* Try Public Key. */
    genKey.type = 0;
    ExpectIntEQ(wc_RsaKeyToDer(&genKey, der, FOURK_BUF), BAD_FUNC_ARG);
    #ifdef WOLFSSL_CHECK_MEM_ZERO
        /* Put back to Private Key */
        genKey.type = 1;
    #endif
#else
    /* Pass good/bad args. */
    ExpectIntEQ(wc_RsaKeyToDer(NULL, der, FOURK_BUF), USER_CRYPTO_ERROR);
    /* Get just the output length */
    ExpectIntGT(wc_RsaKeyToDer(&genKey, NULL, 0), 0);
    /* Try Public Key. */
    genKey.type = 0;
    ExpectIntEQ(wc_RsaKeyToDer(&genKey, der, FOURK_BUF), USER_CRYPTO_ERROR);
    #ifdef WOLFSSL_CHECK_MEM_ZERO
        /* Put back to Private Key */
        genKey.type = 1;
    #endif
#endif

    XFREE(der, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    DoExpectIntEQ(wc_FreeRsaKey(&genKey), 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_RsaKeyToDer */

/*
 *  Testing wc_RsaKeyToPublicDer()
 */
static int test_wc_RsaKeyToPublicDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey key;
    WC_RNG rng;
    byte*  der = NULL;
#if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    int    bits = 1024;
    word32 derLen = 162;
#else
    int    bits = 2048;
    word32 derLen = 294;
#endif
#ifndef HAVE_USER_RSA
    int    ret;
#endif

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ExpectNotNull(der = (byte*)XMALLOC(derLen, NULL, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng), 0);

    /* test getting size only */
    ExpectIntGT(wc_RsaKeyToPublicDer(&key, NULL, derLen), 0);
    ExpectIntGT(wc_RsaKeyToPublicDer(&key, der, derLen), 0);

    /* test getting size only */
    ExpectIntGT(wc_RsaKeyToPublicDer_ex(&key, NULL, derLen, 0), 0);
    ExpectIntGT(wc_RsaKeyToPublicDer_ex(&key, der, derLen, 0), 0);

#ifndef HAVE_USER_RSA
    /* Pass in bad args. */
    ExpectIntEQ(wc_RsaKeyToPublicDer(NULL, der, derLen), BAD_FUNC_ARG);
    ExpectIntLT(ret = wc_RsaKeyToPublicDer(&key, der, -1), 0);
    ExpectTrue((ret == BUFFER_E) || (ret == BAD_FUNC_ARG));
#else
    /* Pass in bad args. */
    ExpectIntEQ(wc_RsaKeyToPublicDer(NULL, der, derLen), USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaKeyToPublicDer(&key, der, -1), USER_CRYPTO_ERROR);
#endif

    XFREE(der, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();

} /* END test_wc_RsaKeyToPublicDer */

/*
 *  Testing wc_RsaPublicEncrypt() and wc_RsaPrivateDecrypt()
 */
static int test_wc_RsaPublicEncryptDecrypt(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey key;
    WC_RNG rng;
    const char inStr[] = TEST_STRING;
    const word32 plainLen = (word32)TEST_STRING_SZ;
    const word32 inLen = (word32)TEST_STRING_SZ;
    int          bits = TEST_RSA_BITS;
    const word32 cipherLen = TEST_RSA_BYTES;
    word32 cipherLenResult = cipherLen;
    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(plain, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(cipher, byte, TEST_RSA_BYTES, NULL);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    ExpectNotNull(in);
    ExpectNotNull(plain);
    ExpectNotNull(cipher);
#endif
    ExpectNotNull(XMEMCPY(in, inStr, inLen));

    /* Initialize stack structures. */
    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng), 0);

    /* Encrypt. */
    ExpectIntGT(cipherLenResult = wc_RsaPublicEncrypt(in, inLen, cipher,
        cipherLen, &key, &rng), 0);
    /* Pass bad args - tested in another testing function.*/

    /* Decrypt */
#if defined(WC_RSA_BLINDING) && !defined(HAVE_FIPS)
    /* Bind rng */
    ExpectIntEQ(wc_RsaSetRNG(&key, &rng), 0);
#endif
    ExpectIntGE(wc_RsaPrivateDecrypt(cipher, cipherLenResult, plain, plainLen,
        &key), 0);
    ExpectIntEQ(XMEMCMP(plain, inStr, plainLen), 0);
    /* Pass bad args - tested in another testing function.*/

    WC_FREE_VAR(in, NULL);
    WC_FREE_VAR(plain, NULL);
    WC_FREE_VAR(cipher, NULL);
    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();

} /* END test_wc_RsaPublicEncryptDecrypt */

/*
 * Testing wc_RsaPrivateDecrypt_ex() and wc_RsaPrivateDecryptInline_ex()
 */
static int test_wc_RsaPublicEncryptDecrypt_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_FIPS)\
        && !defined(WC_NO_RSA_OAEP) && !defined(HAVE_USER_RSA)\
        && !defined(NO_SHA256)
    RsaKey  key;
    WC_RNG  rng;
    const char inStr[] = TEST_STRING;
    const word32 inLen = (word32)TEST_STRING_SZ;
    const word32 plainSz = (word32)TEST_STRING_SZ;
    byte*   res = NULL;
    int     idx = 0;
    int          bits = TEST_RSA_BITS;
    const word32 cipherSz = TEST_RSA_BYTES;

    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(plain, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(cipher, byte, TEST_RSA_BYTES, NULL);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    ExpectNotNull(in);
    ExpectNotNull(plain);
    ExpectNotNull(cipher);
#endif
    ExpectNotNull(XMEMCPY(in, inStr, inLen));

    /* Initialize stack structures. */
    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRsaKey_ex(&key, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng), 0);

    /* Encrypt */
    ExpectIntGE(idx = wc_RsaPublicEncrypt_ex(in, inLen, cipher, cipherSz, &key,
        &rng, WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0), 0);
    /* Pass bad args - tested in another testing function.*/

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    /* Decrypt */
    #if defined(WC_RSA_BLINDING) && !defined(HAVE_FIPS)
        ExpectIntEQ(wc_RsaSetRNG(&key, &rng), 0);
    #endif
    ExpectIntGE(wc_RsaPrivateDecrypt_ex(cipher, (word32)idx, plain, plainSz,
        &key, WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0), 0);
    ExpectIntEQ(XMEMCMP(plain, inStr, plainSz), 0);
    /* Pass bad args - tested in another testing function.*/

    ExpectIntGE(wc_RsaPrivateDecryptInline_ex(cipher, (word32)idx, &res, &key,
        WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0), 0);
    ExpectIntEQ(XMEMCMP(inStr, res, plainSz), 0);
#endif

    WC_FREE_VAR(in, NULL);
    WC_FREE_VAR(plain, NULL);
    WC_FREE_VAR(cipher, NULL);
    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();

} /* END test_wc_RsaPublicEncryptDecrypt_ex */

/*
 * Tesing wc_RsaSSL_Sign() and wc_RsaSSL_Verify()
 */
static int test_wc_RsaSSL_SignVerify(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey key;
    WC_RNG rng;
    const char inStr[] = TEST_STRING;
    const word32 plainSz = (word32)TEST_STRING_SZ;
    const word32 inLen = (word32)TEST_STRING_SZ;
    word32 idx = 0;
    int    bits = TEST_RSA_BITS;
    const word32 outSz = TEST_RSA_BYTES;

    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(out, byte, TEST_RSA_BYTES, NULL);
    WC_DECLARE_VAR(plain, byte, TEST_STRING_SZ, NULL);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    ExpectNotNull(in);
    ExpectNotNull(out);
    ExpectNotNull(plain);
#endif
    ExpectNotNull(XMEMCPY(in, inStr, inLen));

    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng), 0);

    /* Sign. */
    ExpectIntEQ(wc_RsaSSL_Sign(in, inLen, out, outSz, &key, &rng), (int)outSz);
    idx = (int)outSz;
#ifndef HAVE_USER_RSA
    /* Test bad args. */
    ExpectIntEQ(wc_RsaSSL_Sign(NULL, inLen, out, outSz, &key, &rng),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaSSL_Sign(in, 0, out, outSz, &key, &rng),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaSSL_Sign(in, inLen, NULL, outSz, &key, &rng),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaSSL_Sign(in, inLen, out, outSz, NULL, &rng),
        BAD_FUNC_ARG);
#else
    /* Test bad args. */
    ExpectIntEQ(wc_RsaSSL_Sign(NULL, inLen, out, outSz, &key, &rng),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaSSL_Sign(in, 0, out, outSz, &key, &rng),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaSSL_Sign(in, inLen, NULL, outSz, &key, &rng),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaSSL_Sign(in, inLen, out, outSz, NULL, &rng),
        USER_CRYPTO_ERROR);
#endif

    /* Verify. */
    ExpectIntEQ(wc_RsaSSL_Verify(out, idx, plain, plainSz, &key), (int)inLen);
#ifndef HAVE_USER_RSA
    /* Pass bad args. */
    ExpectIntEQ(wc_RsaSSL_Verify(NULL, idx, plain, plainSz, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaSSL_Verify(out, 0, plain, plainSz, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaSSL_Verify(out, idx, NULL, plainSz, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaSSL_Verify(out, idx, plain, plainSz, NULL),
        BAD_FUNC_ARG);
#else
    /* Pass bad args. */
    ExpectIntEQ(wc_RsaSSL_Verify(NULL, idx, plain, plainSz, &key),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaSSL_Verify(out, 0, plain, plainSz, &key),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaSSL_Verify(out, idx, NULL, plainSz, &key),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaSSL_Verify(out, idx, plain, plainSz, NULL),
        USER_CRYPTO_ERROR);
#endif

    WC_FREE_VAR(in, NULL);
    WC_FREE_VAR(out, NULL);
    WC_FREE_VAR(plain, NULL);
    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();

} /* END test_wc_RsaSSL_SignVerify */

/*
 * Testing wc_RsaEncryptSize()
 */
static int test_wc_RsaEncryptSize(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey key;
    WC_RNG rng;

    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

#if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    ExpectIntEQ(MAKE_RSA_KEY(&key, 1024, WC_RSA_EXPONENT, &rng), 0);

    ExpectIntEQ(wc_RsaEncryptSize(&key), 128);
    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
#endif

    ExpectIntEQ(MAKE_RSA_KEY(&key, 2048, WC_RSA_EXPONENT, &rng), 0);
    ExpectIntEQ(wc_RsaEncryptSize(&key), 256);

    /* Pass in bad arg. */
#ifndef HAVE_USER_RSA
    ExpectIntEQ(wc_RsaEncryptSize(NULL), BAD_FUNC_ARG);
#else
    ExpectIntEQ(wc_RsaEncryptSize(NULL), 0);
#endif

    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();

} /* END test_wc_RsaEncryptSize*/

/*
 * Testing wc_RsaFlattenPublicKey()
 */
static int test_wc_RsaFlattenPublicKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey key;
    WC_RNG rng;
    byte   e[256];
    byte   n[256];
    word32 eSz = sizeof(e);
    word32 nSz = sizeof(n);
    #if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
        (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    int    bits = 1024;
    #else
    int    bits = 2048;
    #endif

    XMEMSET(&key, 0, sizeof(RsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRsaKey(&key, HEAP_HINT), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng), 0);

    ExpectIntEQ(wc_RsaFlattenPublicKey(&key, e, &eSz, n, &nSz), 0);
#ifndef HAVE_USER_RSA
    /* Pass bad args. */
    ExpectIntEQ(wc_RsaFlattenPublicKey(NULL, e, &eSz, n, &nSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaFlattenPublicKey(&key, NULL, &eSz, n, &nSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaFlattenPublicKey(&key, e, NULL, n, &nSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaFlattenPublicKey(&key, e, &eSz, NULL, &nSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_RsaFlattenPublicKey(&key, e, &eSz, n, NULL),
        BAD_FUNC_ARG);
#else
    /* Pass bad args. */
    ExpectIntEQ(wc_RsaFlattenPublicKey(NULL, e, &eSz, n, &nSz),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaFlattenPublicKey(&key, NULL, &eSz, n, &nSz),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaFlattenPublicKey(&key, e, NULL, n, &nSz),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaFlattenPublicKey(&key, e, &eSz, NULL, &nSz),
        USER_CRYPTO_ERROR);
    ExpectIntEQ(wc_RsaFlattenPublicKey(&key, e, &eSz, n, NULL),
        USER_CRYPTO_ERROR);
#endif

    DoExpectIntEQ(wc_FreeRsaKey(&key), 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();

} /* END test_wc_RsaFlattenPublicKey */



/*
 * unit test for wc_AesCcmSetKey
 */
static int test_wc_AesCcmSetKey(void)
{
    EXPECT_DECLS;
#ifdef HAVE_AESCCM
    Aes aes;
    const byte key16[] = {
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
        0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf
    };
    const byte key24[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
    const byte key32[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };

    XMEMSET(&aes, 0, sizeof(Aes));

    ExpectIntEQ(wc_AesInit(&aes, NULL, INVALID_DEVID), 0);

#ifdef WOLFSSL_AES_128
    ExpectIntEQ(wc_AesCcmSetKey(&aes, key16, sizeof(key16)), 0);
#endif
#ifdef WOLFSSL_AES_192
    ExpectIntEQ(wc_AesCcmSetKey(&aes, key24, sizeof(key24)), 0);
#endif
#ifdef WOLFSSL_AES_256
    ExpectIntEQ(wc_AesCcmSetKey(&aes, key32, sizeof(key32)), 0);
#endif

    /* Test bad args. */
   ExpectIntEQ(wc_AesCcmSetKey(&aes, key16, sizeof(key16) - 1), BAD_FUNC_ARG);
   ExpectIntEQ(wc_AesCcmSetKey(&aes, key24, sizeof(key24) - 1), BAD_FUNC_ARG);
   ExpectIntEQ(wc_AesCcmSetKey(&aes, key32, sizeof(key32) - 1), BAD_FUNC_ARG);

    wc_AesFree(&aes);
#endif
    return EXPECT_RESULT();

} /* END test_wc_AesCcmSetKey */

/*
 * Unit test function for wc_AesCcmEncrypt and wc_AesCcmDecrypt
 */
static int test_wc_AesCcmEncryptDecrypt(void)
{
    EXPECT_DECLS;
#if defined(HAVE_AESCCM) && defined(WOLFSSL_AES_128)
    Aes aes;
    const byte key16[] = {
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
        0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf
    };
    /* plaintext */
    const byte plainT[] = {
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e
    };
    /* nonce */
    const byte iv[] = {
        0x00, 0x00, 0x00, 0x03, 0x02, 0x01, 0x00, 0xa0,
        0xa1, 0xa2, 0xa3, 0xa4, 0xa5
    };
    const byte c[] = { /* cipher text. */
        0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
        0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
        0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84
    };
    const byte t[] = { /* Auth tag */
        0x17, 0xe8, 0xd1, 0x2c, 0xfd, 0xf9, 0x26, 0xe0
    };
    const byte authIn[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };
    byte cipherOut[sizeof(plainT)];
    byte authTag[sizeof(t)];
#ifdef HAVE_AES_DECRYPT
    byte plainOut[sizeof(cipherOut)];
#endif

    XMEMSET(&aes, 0, sizeof(Aes));

    ExpectIntEQ(wc_AesInit(&aes, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_AesCcmSetKey(&aes, key16, sizeof(key16)), 0);

    ExpectIntEQ(wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
        iv, sizeof(iv), authTag, sizeof(authTag), authIn , sizeof(authIn)), 0);
    ExpectIntEQ(XMEMCMP(cipherOut, c, sizeof(c)), 0);
    ExpectIntEQ(XMEMCMP(t, authTag, sizeof(t)), 0);
#ifdef HAVE_AES_DECRYPT
    ExpectIntEQ(wc_AesCcmDecrypt(&aes, plainOut, cipherOut, sizeof(plainOut),
        iv, sizeof(iv), authTag, sizeof(authTag), authIn, sizeof(authIn)), 0);
    ExpectIntEQ(XMEMCMP(plainOut, plainT, sizeof(plainT)), 0);
#endif

    /* Pass in bad args. Encrypt*/
    ExpectIntEQ(wc_AesCcmEncrypt(NULL, cipherOut, plainT, sizeof(cipherOut),
        iv, sizeof(iv), authTag, sizeof(authTag), authIn , sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmEncrypt(&aes, NULL, plainT, sizeof(cipherOut),
        iv, sizeof(iv), authTag, sizeof(authTag), authIn , sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmEncrypt(&aes, cipherOut, NULL, sizeof(cipherOut),
        iv, sizeof(iv), authTag, sizeof(authTag), authIn , sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
        NULL, sizeof(iv), authTag, sizeof(authTag), authIn , sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
        iv, sizeof(iv), NULL, sizeof(authTag), authIn , sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
        iv, sizeof(iv) + 1, authTag, sizeof(authTag), authIn , sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
        iv, sizeof(iv) - 7, authTag, sizeof(authTag), authIn , sizeof(authIn)),
        BAD_FUNC_ARG);

#ifdef HAVE_AES_DECRYPT
    /* Pass in bad args. Decrypt*/
    ExpectIntEQ(wc_AesCcmDecrypt(NULL, plainOut, cipherOut, sizeof(plainOut),
        iv, sizeof(iv), authTag, sizeof(authTag), authIn, sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmDecrypt(&aes, NULL, cipherOut, sizeof(plainOut),
        iv, sizeof(iv), authTag, sizeof(authTag), authIn, sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmDecrypt(&aes, plainOut, NULL, sizeof(plainOut),
        iv, sizeof(iv), authTag, sizeof(authTag), authIn, sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmDecrypt(&aes, plainOut, cipherOut, sizeof(plainOut),
        NULL, sizeof(iv), authTag, sizeof(authTag), authIn, sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmDecrypt(&aes, plainOut, cipherOut, sizeof(plainOut),
        iv, sizeof(iv), NULL, sizeof(authTag), authIn, sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmDecrypt(&aes, plainOut, cipherOut, sizeof(plainOut),
        iv, sizeof(iv) + 1, authTag, sizeof(authTag), authIn, sizeof(authIn)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesCcmDecrypt(&aes, plainOut, cipherOut, sizeof(plainOut),
        iv, sizeof(iv) - 7, authTag, sizeof(authTag), authIn, sizeof(authIn)),
        BAD_FUNC_ARG);
    #endif

    wc_AesFree(&aes);
#endif  /* HAVE_AESCCM */
    return EXPECT_RESULT();
} /* END test_wc_AesCcmEncryptDecrypt */


#if defined(WOLFSSL_AES_EAX) && \
    (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)

/*
 * Testing test_wc_AesEaxVectors()
 */
static int test_wc_AesEaxVectors(void)
{
    EXPECT_DECLS;

    typedef struct {
        byte key[AES_256_KEY_SIZE];
        int key_length;
        byte iv[AES_BLOCK_SIZE];
        int iv_length;
        byte aad[AES_BLOCK_SIZE * 2];
        int aad_length;
        byte msg[AES_BLOCK_SIZE * 5];
        int msg_length;
        byte ct[AES_BLOCK_SIZE * 5];
        int ct_length;
        byte tag[AES_BLOCK_SIZE];
        int tag_length;
        int valid;
    } AadVector;

    /*  Test vectors obtained from Google wycheproof project
     *  https://github.com/google/wycheproof
     *  from testvectors/aes_eax_test.json
     */
    const AadVector vectors[] = {
        {
            /* key, key length  */
            {0x23, 0x39, 0x52, 0xde, 0xe4, 0xd5, 0xed, 0x5f,
             0x9b, 0x9c, 0x6d, 0x6f, 0xf8, 0x0f, 0xf4, 0x78}, 16,
            /* iv, iv length  */
            {0x62, 0xec, 0x67, 0xf9, 0xc3, 0xa4, 0xa4, 0x07,
             0xfc, 0xb2, 0xa8, 0xc4, 0x90, 0x31, 0xa8, 0xb3}, 16,
            /* aad, aad length  */
            {0x6b, 0xfb, 0x91, 0x4f, 0xd0, 0x7e, 0xae, 0x6b}, 8,
            /* msg, msg length  */
            {0x00}, 0,
            /* ct, ct length  */
            {0x00}, 0,
            /* tag, tag length  */
            {0xe0, 0x37, 0x83, 0x0e, 0x83, 0x89, 0xf2, 0x7b,
             0x02, 0x5a, 0x2d, 0x65, 0x27, 0xe7, 0x9d, 0x01}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x91, 0x94, 0x5d, 0x3f, 0x4d, 0xcb, 0xee, 0x0b,
             0xf4, 0x5e, 0xf5, 0x22, 0x55, 0xf0, 0x95, 0xa4}, 16,
            /* iv, iv length  */
            {0xbe, 0xca, 0xf0, 0x43, 0xb0, 0xa2, 0x3d, 0x84,
             0x31, 0x94, 0xba, 0x97, 0x2c, 0x66, 0xde, 0xbd}, 16,
            /* aad, aad length  */
            {0xfa, 0x3b, 0xfd, 0x48, 0x06, 0xeb, 0x53, 0xfa}, 8,
            /* msg, msg length  */
            {0xf7, 0xfb}, 2,
            /* ct, ct length  */
            {0x19, 0xdd}, 2,
            /* tag, tag length  */
            {0x5c, 0x4c, 0x93, 0x31, 0x04, 0x9d, 0x0b, 0xda,
             0xb0, 0x27, 0x74, 0x08, 0xf6, 0x79, 0x67, 0xe5}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x01, 0xf7, 0x4a, 0xd6, 0x40, 0x77, 0xf2, 0xe7,
             0x04, 0xc0, 0xf6, 0x0a, 0xda, 0x3d, 0xd5, 0x23}, 16,
            /* iv, iv length  */
            {0x70, 0xc3, 0xdb, 0x4f, 0x0d, 0x26, 0x36, 0x84,
             0x00, 0xa1, 0x0e, 0xd0, 0x5d, 0x2b, 0xff, 0x5e}, 16,
            /* aad, aad length  */
            {0x23, 0x4a, 0x34, 0x63, 0xc1, 0x26, 0x4a, 0xc6}, 8,
            /* msg, msg length  */
            {0x1a, 0x47, 0xcb, 0x49, 0x33}, 5,
            /* ct, ct length  */
            {0xd8, 0x51, 0xd5, 0xba, 0xe0}, 5,
            /* tag, tag length  */
            {0x3a, 0x59, 0xf2, 0x38, 0xa2, 0x3e, 0x39, 0x19,
             0x9d, 0xc9, 0x26, 0x66, 0x26, 0xc4, 0x0f, 0x80}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0xd0, 0x7c, 0xf6, 0xcb, 0xb7, 0xf3, 0x13, 0xbd,
             0xde, 0x66, 0xb7, 0x27, 0xaf, 0xd3, 0xc5, 0xe8}, 16,
            /* iv, iv length  */
            {0x84, 0x08, 0xdf, 0xff, 0x3c, 0x1a, 0x2b, 0x12,
             0x92, 0xdc, 0x19, 0x9e, 0x46, 0xb7, 0xd6, 0x17}, 16,
            /* aad, aad length  */
            {0x33, 0xcc, 0xe2, 0xea, 0xbf, 0xf5, 0xa7, 0x9d}, 8,
            /* msg, msg length  */
            {0x48, 0x1c, 0x9e, 0x39, 0xb1}, 5,
            /* ct, ct length  */
            {0x63, 0x2a, 0x9d, 0x13, 0x1a}, 5,
            /* tag, tag length  */
            {0xd4, 0xc1, 0x68, 0xa4, 0x22, 0x5d, 0x8e, 0x1f,
             0xf7, 0x55, 0x93, 0x99, 0x74, 0xa7, 0xbe, 0xde}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x35, 0xb6, 0xd0, 0x58, 0x00, 0x05, 0xbb, 0xc1,
             0x2b, 0x05, 0x87, 0x12, 0x45, 0x57, 0xd2, 0xc2}, 16,
            /* iv, iv length  */
            {0xfd, 0xb6, 0xb0, 0x66, 0x76, 0xee, 0xdc, 0x5c,
             0x61, 0xd7, 0x42, 0x76, 0xe1, 0xf8, 0xe8, 0x16}, 16,
            /* aad, aad length  */
            {0xae, 0xb9, 0x6e, 0xae, 0xbe, 0x29, 0x70, 0xe9}, 8,
            /* msg, msg length  */
            {0x40, 0xd0, 0xc0, 0x7d, 0xa5, 0xe4}, 6,
            /* ct, ct length  */
            {0x07, 0x1d, 0xfe, 0x16, 0xc6, 0x75}, 6,
            /* tag, tag length  */
            {0xcb, 0x06, 0x77, 0xe5, 0x36, 0xf7, 0x3a, 0xfe,
             0x6a, 0x14, 0xb7, 0x4e, 0xe4, 0x98, 0x44, 0xdd}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0xbd, 0x8e, 0x6e, 0x11, 0x47, 0x5e, 0x60, 0xb2,
             0x68, 0x78, 0x4c, 0x38, 0xc6, 0x2f, 0xeb, 0x22}, 16,
            /* iv, iv length  */
            {0x6e, 0xac, 0x5c, 0x93, 0x07, 0x2d, 0x8e, 0x85,
             0x13, 0xf7, 0x50, 0x93, 0x5e, 0x46, 0xda, 0x1b}, 16,
            /* aad, aad length  */
            {0xd4, 0x48, 0x2d, 0x1c, 0xa7, 0x8d, 0xce, 0x0f}, 8,
            /* msg, msg length  */
            {0x4d, 0xe3, 0xb3, 0x5c, 0x3f, 0xc0, 0x39, 0x24,
             0x5b, 0xd1, 0xfb, 0x7d}, 12,
            /* ct, ct length  */
            {0x83, 0x5b, 0xb4, 0xf1, 0x5d, 0x74, 0x3e, 0x35,
             0x0e, 0x72, 0x84, 0x14}, 12,
            /* tag, tag length  */
            {0xab, 0xb8, 0x64, 0x4f, 0xd6, 0xcc, 0xb8, 0x69,
             0x47, 0xc5, 0xe1, 0x05, 0x90, 0x21, 0x0a, 0x4f}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x7c, 0x77, 0xd6, 0xe8, 0x13, 0xbe, 0xd5, 0xac,
             0x98, 0xba, 0xa4, 0x17, 0x47, 0x7a, 0x2e, 0x7d}, 16,
            /* iv, iv length  */
            {0x1a, 0x8c, 0x98, 0xdc, 0xd7, 0x3d, 0x38, 0x39,
             0x3b, 0x2b, 0xf1, 0x56, 0x9d, 0xee, 0xfc, 0x19}, 16,
            /* aad, aad length  */
            {0x65, 0xd2, 0x01, 0x79, 0x90, 0xd6, 0x25, 0x28}, 8,
            /* msg, msg length  */
            {0x8b, 0x0a, 0x79, 0x30, 0x6c, 0x9c, 0xe7, 0xed,
             0x99, 0xda, 0xe4, 0xf8, 0x7f, 0x8d, 0xd6, 0x16,
             0x36}, 17,
            /* ct, ct length  */
            {0x02, 0x08, 0x3e, 0x39, 0x79, 0xda, 0x01, 0x48,
             0x12, 0xf5, 0x9f, 0x11, 0xd5, 0x26, 0x30, 0xda,
             0x30}, 17,
            /* tag, tag length  */
            {0x13, 0x73, 0x27, 0xd1, 0x06, 0x49, 0xb0, 0xaa,
             0x6e, 0x1c, 0x18, 0x1d, 0xb6, 0x17, 0xd7, 0xf2}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x5f, 0xff, 0x20, 0xca, 0xfa, 0xb1, 0x19, 0xca,
             0x2f, 0xc7, 0x35, 0x49, 0xe2, 0x0f, 0x5b, 0x0d}, 16,
            /* iv, iv length  */
            {0xdd, 0xe5, 0x9b, 0x97, 0xd7, 0x22, 0x15, 0x6d,
             0x4d, 0x9a, 0xff, 0x2b, 0xc7, 0x55, 0x98, 0x26}, 16,
            /* aad, aad length  */
            {0x54, 0xb9, 0xf0, 0x4e, 0x6a, 0x09, 0x18, 0x9a}, 8,
            /* msg, msg length  */
            {0x1b, 0xda, 0x12, 0x2b, 0xce, 0x8a, 0x8d, 0xba,
             0xf1, 0x87, 0x7d, 0x96, 0x2b, 0x85, 0x92, 0xdd,
             0x2d, 0x56}, 18,
            /* ct, ct length  */
            {0x2e, 0xc4, 0x7b, 0x2c, 0x49, 0x54, 0xa4, 0x89,
             0xaf, 0xc7, 0xba, 0x48, 0x97, 0xed, 0xcd, 0xae,
             0x8c, 0xc3}, 18,
            /* tag, tag length  */
            {0x3b, 0x60, 0x45, 0x05, 0x99, 0xbd, 0x02, 0xc9,
             0x63, 0x82, 0x90, 0x2a, 0xef, 0x7f, 0x83, 0x2a}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0xa4, 0xa4, 0x78, 0x2b, 0xcf, 0xfd, 0x3e, 0xc5,
             0xe7, 0xef, 0x6d, 0x8c, 0x34, 0xa5, 0x61, 0x23}, 16,
            /* iv, iv length  */
            {0xb7, 0x81, 0xfc, 0xf2, 0xf7, 0x5f, 0xa5, 0xa8,
             0xde, 0x97, 0xa9, 0xca, 0x48, 0xe5, 0x22, 0xec}, 16,
            /* aad, aad length  */
            {0x89, 0x9a, 0x17, 0x58, 0x97, 0x56, 0x1d, 0x7e}, 8,
            /* msg, msg length  */
            {0x6c, 0xf3, 0x67, 0x20, 0x87, 0x2b, 0x85, 0x13,
             0xf6, 0xea, 0xb1, 0xa8, 0xa4, 0x44, 0x38, 0xd5,
             0xef, 0x11}, 18,
            /* ct, ct length  */
            {0x0d, 0xe1, 0x8f, 0xd0, 0xfd, 0xd9, 0x1e, 0x7a,
             0xf1, 0x9f, 0x1d, 0x8e, 0xe8, 0x73, 0x39, 0x38,
             0xb1, 0xe8}, 18,
            /* tag, tag length  */
            {0xe7, 0xf6, 0xd2, 0x23, 0x16, 0x18, 0x10, 0x2f,
             0xdb, 0x7f, 0xe5, 0x5f, 0xf1, 0x99, 0x17, 0x00}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x83, 0x95, 0xfc, 0xf1, 0xe9, 0x5b, 0xeb, 0xd6,
             0x97, 0xbd, 0x01, 0x0b, 0xc7, 0x66, 0xaa, 0xc3}, 16,
            /* iv, iv length  */
            {0x22, 0xe7, 0xad, 0xd9, 0x3c, 0xfc, 0x63, 0x93,
             0xc5, 0x7e, 0xc0, 0xb3, 0xc1, 0x7d, 0x6b, 0x44}, 16,
            /* aad, aad length  */
            {0x12, 0x67, 0x35, 0xfc, 0xc3, 0x20, 0xd2, 0x5a}, 8,
            /* msg, msg length  */
            {0xca, 0x40, 0xd7, 0x44, 0x6e, 0x54, 0x5f, 0xfa,
             0xed, 0x3b, 0xd1, 0x2a, 0x74, 0x0a, 0x65, 0x9f,
             0xfb, 0xbb, 0x3c, 0xea, 0xb7}, 21,
            /* ct, ct length  */
            {0xcb, 0x89, 0x20, 0xf8, 0x7a, 0x6c, 0x75, 0xcf,
             0xf3, 0x96, 0x27, 0xb5, 0x6e, 0x3e, 0xd1, 0x97,
             0xc5, 0x52, 0xd2, 0x95, 0xa7}, 21,
            /* tag, tag length  */
            {0xcf, 0xc4, 0x6a, 0xfc, 0x25, 0x3b, 0x46, 0x52,
             0xb1, 0xaf, 0x37, 0x95, 0xb1, 0x24, 0xab, 0x6e}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x3c, 0x8c, 0xc2, 0x97, 0x0a, 0x00, 0x8f, 0x75,
             0xcc, 0x5b, 0xea, 0xe2, 0x84, 0x72, 0x58, 0xc2}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11}, 32,
            /* ct, ct length  */
            {0x3c, 0x44, 0x1f, 0x32, 0xce, 0x07, 0x82, 0x23,
             0x64, 0xd7, 0xa2, 0x99, 0x0e, 0x50, 0xbb, 0x13,
             0xd7, 0xb0, 0x2a, 0x26, 0x96, 0x9e, 0x4a, 0x93,
             0x7e, 0x5e, 0x90, 0x73, 0xb0, 0xd9, 0xc9, 0x68}, 32,
            /* tag, tag length  */
            {0xdb, 0x90, 0xbd, 0xb3, 0xda, 0x3d, 0x00, 0xaf,
             0xd0, 0xfc, 0x6a, 0x83, 0x55, 0x1d, 0xa9, 0x5e}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0xae, 0xf0, 0x3d, 0x00, 0x59, 0x84, 0x94, 0xe9,
             0xfb, 0x03, 0xcd, 0x7d, 0x8b, 0x59, 0x08, 0x66}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11}, 32,
            /* ct, ct length  */
            {0xd1, 0x9a, 0xc5, 0x98, 0x49, 0x02, 0x6a, 0x91,
             0xaa, 0x1b, 0x9a, 0xec, 0x29, 0xb1, 0x1a, 0x20,
             0x2a, 0x4d, 0x73, 0x9f, 0xd8, 0x6c, 0x28, 0xe3,
             0xae, 0x3d, 0x58, 0x8e, 0xa2, 0x1d, 0x70, 0xc6}, 32,
            /* tag, tag length  */
            {0xc3, 0x0f, 0x6c, 0xd9, 0x20, 0x20, 0x74, 0xed,
             0x6e, 0x2a, 0x2a, 0x36, 0x0e, 0xac, 0x8c, 0x47}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x55, 0xd1, 0x25, 0x11, 0xc6, 0x96, 0xa8, 0x0d,
             0x05, 0x14, 0xd1, 0xff, 0xba, 0x49, 0xca, 0xda}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11}, 32,
            /* ct, ct length  */
            {0x21, 0x08, 0x55, 0x8a, 0xc4, 0xb2, 0xc2, 0xd5,
             0xcc, 0x66, 0xce, 0xa5, 0x1d, 0x62, 0x10, 0xe0,
             0x46, 0x17, 0x7a, 0x67, 0x63, 0x1c, 0xd2, 0xdd,
             0x8f, 0x09, 0x46, 0x97, 0x33, 0xac, 0xb5, 0x17}, 32,
            /* tag, tag length  */
            {0xfc, 0x35, 0x5e, 0x87, 0xa2, 0x67, 0xbe, 0x3a,
             0xe3, 0xe4, 0x4c, 0x0b, 0xf3, 0xf9, 0x9b, 0x2b}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x79, 0x42, 0x2d, 0xdd, 0x91, 0xc4, 0xee, 0xe2,
             0xde, 0xae, 0xf1, 0xf9, 0x68, 0x30, 0x53, 0x04}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11}, 32,
            /* ct, ct length  */
            {0x4d, 0x2c, 0x15, 0x24, 0xca, 0x4b, 0xaa, 0x4e,
             0xef, 0xcc, 0xe6, 0xb9, 0x1b, 0x22, 0x7e, 0xe8,
             0x3a, 0xba, 0xff, 0x81, 0x05, 0xdc, 0xaf, 0xa2,
             0xab, 0x19, 0x1f, 0x5d, 0xf2, 0x57, 0x50, 0x35}, 32,
            /* tag, tag length  */
            {0xe2, 0xc8, 0x65, 0xce, 0x2d, 0x7a, 0xbd, 0xac,
             0x02, 0x4c, 0x6f, 0x99, 0x1a, 0x84, 0x83, 0x90}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x0a, 0xf5, 0xaa, 0x7a, 0x76, 0x76, 0xe2, 0x83,
             0x06, 0x30, 0x6b, 0xcd, 0x9b, 0xf2, 0x00, 0x3a}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11}, 32,
            /* ct, ct length  */
            {0x8e, 0xb0, 0x1e, 0x62, 0x18, 0x5d, 0x78, 0x2e,
             0xb9, 0x28, 0x7a, 0x34, 0x1a, 0x68, 0x62, 0xac,
             0x52, 0x57, 0xd6, 0xf9, 0xad, 0xc9, 0x9e, 0xe0,
             0xa2, 0x4d, 0x9c, 0x22, 0xb3, 0xe9, 0xb3, 0x8a}, 32,
            /* tag, tag length  */
            {0x39, 0xc3, 0x39, 0xbc, 0x8a, 0x74, 0xc7, 0x5e,
             0x2c, 0x65, 0xc6, 0x11, 0x95, 0x44, 0xd6, 0x1e}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0xaf, 0x5a, 0x03, 0xae, 0x7e, 0xdd, 0x73, 0x47,
             0x1b, 0xdc, 0xdf, 0xac, 0x5e, 0x19, 0x4a, 0x60}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11}, 32,
            /* ct, ct length  */
            {0x94, 0xc5, 0xd2, 0xac, 0xa6, 0xdb, 0xbc, 0xe8,
             0xc2, 0x45, 0x13, 0xa2, 0x5e, 0x09, 0x5c, 0x0e,
             0x54, 0xa9, 0x42, 0x86, 0x0d, 0x32, 0x7a, 0x22,
             0x2a, 0x81, 0x5c, 0xc7, 0x13, 0xb1, 0x63, 0xb4}, 32,
            /* tag, tag length  */
            {0xf5, 0x0b, 0x30, 0x30, 0x4e, 0x45, 0xc9, 0xd4,
             0x11, 0xe8, 0xdf, 0x45, 0x08, 0xa9, 0x86, 0x12}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0xb3, 0x70, 0x87, 0x68, 0x0f, 0x0e, 0xdd, 0x5a,
             0x52, 0x22, 0x8b, 0x8c, 0x7a, 0xae, 0xa6, 0x64}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
             0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
             0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33,
             0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33}, 64,
            /* ct, ct length  */
            {0x3b, 0xb6, 0x17, 0x3e, 0x37, 0x72, 0xd4, 0xb6,
             0x2e, 0xef, 0x37, 0xf9, 0xef, 0x07, 0x81, 0xf3,
             0x60, 0xb6, 0xc7, 0x4b, 0xe3, 0xbf, 0x6b, 0x37,
             0x10, 0x67, 0xbc, 0x1b, 0x09, 0x0d, 0x9d, 0x66,
             0x22, 0xa1, 0xfb, 0xec, 0x6a, 0xc4, 0x71, 0xb3,
             0x34, 0x9c, 0xd4, 0x27, 0x7a, 0x10, 0x1d, 0x40,
             0x89, 0x0f, 0xbf, 0x27, 0xdf, 0xdc, 0xd0, 0xb4,
             0xe3, 0x78, 0x1f, 0x98, 0x06, 0xda, 0xab, 0xb6}, 64,
            /* tag, tag length  */
            {0xa0, 0x49, 0x87, 0x45, 0xe5, 0x99, 0x99, 0xdd,
             0xc3, 0x2d, 0x5b, 0x14, 0x02, 0x41, 0x12, 0x4e}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x4f, 0x80, 0x2d, 0xa6, 0x2a, 0x38, 0x45, 0x55,
             0xa1, 0x9b, 0xc2, 0xb3, 0x82, 0xeb, 0x25, 0xaf}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
             0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
             0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33,
             0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33,
             0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
             0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44}, 80,
            /* ct, ct length  */
            {0xe9, 0xb0, 0xbb, 0x88, 0x57, 0x81, 0x8c, 0xe3,
             0x20, 0x1c, 0x36, 0x90, 0xd2, 0x1d, 0xaa, 0x7f,
             0x26, 0x4f, 0xb8, 0xee, 0x93, 0xcc, 0x7a, 0x46,
             0x74, 0xea, 0x2f, 0xc3, 0x2b, 0xf1, 0x82, 0xfb,
             0x2a, 0x7e, 0x8a, 0xd5, 0x15, 0x07, 0xad, 0x4f,
             0x31, 0xce, 0xfc, 0x23, 0x56, 0xfe, 0x79, 0x36,
             0xa7, 0xf6, 0xe1, 0x9f, 0x95, 0xe8, 0x8f, 0xdb,
             0xf1, 0x76, 0x20, 0x91, 0x6d, 0x3a, 0x6f, 0x3d,
             0x01, 0xfc, 0x17, 0xd3, 0x58, 0x67, 0x2f, 0x77,
             0x7f, 0xd4, 0x09, 0x92, 0x46, 0xe4, 0x36, 0xe1}, 80,
            /* tag, tag length  */
            {0x67, 0x91, 0x0b, 0xe7, 0x44, 0xb8, 0x31, 0x5a,
             0xe0, 0xeb, 0x61, 0x24, 0x59, 0x0c, 0x5d, 0x8b}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0xb6, 0x7b, 0x1a, 0x6e, 0xfd, 0xd4, 0x0d, 0x37,
             0x08, 0x0f, 0xbe, 0x8f, 0x80, 0x47, 0xae, 0xb9}, 16,
            /* iv, iv length  */
            {0xfa, 0x29, 0x4b, 0x12, 0x99, 0x72, 0xf7, 0xfc,
             0x5b, 0xbd, 0x5b, 0x96, 0xbb, 0xa8, 0x37, 0xc9}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x00}, 0,
            /* ct, ct length  */
            {0x00}, 0,
            /* tag, tag length  */
            {0xb1, 0x4b, 0x64, 0xfb, 0x58, 0x98, 0x99, 0x69,
             0x95, 0x70, 0xcc, 0x91, 0x60, 0xe3, 0x98, 0x96}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x20, 0x9e, 0x6d, 0xbf, 0x2a, 0xd2, 0x6a, 0x10,
             0x54, 0x45, 0xfc, 0x02, 0x07, 0xcd, 0x9e, 0x9a}, 16,
            /* iv, iv length  */
            {0x94, 0x77, 0x84, 0x9d, 0x6c, 0xcd, 0xfc, 0xa1,
             0x12, 0xd9, 0x2e, 0x53, 0xfa, 0xe4, 0xa7, 0xca}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x01}, 1,
            /* ct, ct length  */
            {0x1d}, 1,
            /* tag, tag length  */
            {0x52, 0xa5, 0xf6, 0x00, 0xfe, 0x53, 0x38, 0x02,
             0x6a, 0x7c, 0xb0, 0x9c, 0x11, 0x64, 0x00, 0x82}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0xa5, 0x49, 0x44, 0x2e, 0x35, 0x15, 0x40, 0x32,
             0xd0, 0x7c, 0x86, 0x66, 0x00, 0x6a, 0xa6, 0xa2}, 16,
            /* iv, iv length  */
            {0x51, 0x71, 0x52, 0x45, 0x68, 0xe8, 0x1d, 0x97,
             0xe8, 0xc4, 0xde, 0x4b, 0xa5, 0x6c, 0x10, 0xa0}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x11, 0x82, 0xe9, 0x35, 0x96, 0xca, 0xc5, 0x60,
             0x89, 0x46, 0x40, 0x0b, 0xc7, 0x3f, 0x3a}, 15,
            /* ct, ct length  */
            {0xd7, 0xb8, 0xa6, 0xb4, 0x3d, 0x2e, 0x9f, 0x98,
             0xc2, 0xb4, 0x4c, 0xe5, 0xe3, 0xcf, 0xdb}, 15,
            /* tag, tag length  */
            {0x1b, 0xdd, 0x52, 0xfc, 0x98, 0x7d, 0xaf, 0x0e,
             0xe1, 0x92, 0x34, 0xc9, 0x05, 0xea, 0x64, 0x5f}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x95, 0x8b, 0xcd, 0xb6, 0x6a, 0x39, 0x52, 0xb5,
             0x37, 0x01, 0x58, 0x2a, 0x68, 0xa0, 0xe4, 0x74}, 16,
            /* iv, iv length  */
            {0x0e, 0x6e, 0xc8, 0x79, 0xb0, 0x2c, 0x6f, 0x51,
             0x69, 0x76, 0xe3, 0x58, 0x98, 0x42, 0x8d, 0xa7}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x14, 0x04, 0x15, 0x82, 0x3e, 0xcc, 0x89, 0x32,
             0xa0, 0x58, 0x38, 0x4b, 0x73, 0x8e, 0xa6, 0xea,
             0x6d, 0x4d, 0xfe, 0x3b, 0xbe, 0xee}, 22,
            /* ct, ct length  */
            {0x73, 0xe5, 0xc6, 0xf0, 0xe7, 0x03, 0xa5, 0x2d,
             0x02, 0xf7, 0xf7, 0xfa, 0xeb, 0x1b, 0x77, 0xfd,
             0x4f, 0xd0, 0xcb, 0x42, 0x1e, 0xaf}, 22,
            /* tag, tag length  */
            {0x6c, 0x15, 0x4a, 0x85, 0x96, 0x8e, 0xdd, 0x74,
             0x77, 0x65, 0x75, 0xa4, 0x45, 0x0b, 0xd8, 0x97}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x96, 0x5b, 0x75, 0x7b, 0xa5, 0x01, 0x8a, 0x8d,
             0x66, 0xed, 0xc7, 0x8e, 0x0c, 0xee, 0xe8, 0x6b}, 16,
            /* iv, iv length  */
            {0x2e, 0x35, 0x90, 0x1a, 0xe7, 0xd4, 0x91, 0xee,
             0xcc, 0x88, 0x38, 0xfe, 0xdd, 0x63, 0x14, 0x05}, 16,
            /* aad, aad length  */
            {0xdf, 0x10, 0xd0, 0xd2, 0x12, 0x24, 0x24, 0x50}, 8,
            /* msg, msg length  */
            {0x36, 0xe5, 0x7a, 0x76, 0x39, 0x58, 0xb0, 0x2c,
             0xea, 0x9d, 0x6a, 0x67, 0x6e, 0xbc, 0xe8, 0x1f}, 16,
            /* ct, ct length  */
            {0x93, 0x6b, 0x69, 0xb6, 0xc9, 0x55, 0xad, 0xfd,
             0x15, 0x53, 0x9b, 0x9b, 0xe4, 0x98, 0x9c, 0xb6}, 16,
            /* tag, tag length  */
            {0xee, 0x15, 0xa1, 0x45, 0x4e, 0x88, 0xfa, 0xad,
             0x8e, 0x48, 0xa8, 0xdf, 0x29, 0x83, 0xb4, 0x25}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x88, 0xd0, 0x20, 0x33, 0x78, 0x1c, 0x7b, 0x41,
             0x64, 0x71, 0x1a, 0x05, 0x42, 0x0f, 0x25, 0x6e}, 16,
            /* iv, iv length  */
            {0x7f, 0x29, 0x85, 0x29, 0x63, 0x15, 0x50, 0x7a,
             0xa4, 0xc0, 0xa9, 0x3d, 0x5c, 0x12, 0xbd, 0x77}, 16,
            /* aad, aad length  */
            {0x7c, 0x57, 0x1d, 0x2f, 0xbb, 0x5f, 0x62, 0x52,
             0x3c, 0x0e, 0xb3, 0x38, 0xbe, 0xf9, 0xa9}, 15,
            /* msg, msg length  */
            {0xd9, 0x8a, 0xdc, 0x03, 0xd9, 0xd5, 0x82, 0x73,
             0x2e, 0xb0, 0x7d, 0xf2, 0x3d, 0x7b, 0x9f, 0x74}, 16,
            /* ct, ct length  */
            {0x67, 0xca, 0xac, 0x35, 0x44, 0x3a, 0x31, 0x38,
             0xd2, 0xcb, 0x81, 0x1f, 0x0c, 0xe0, 0x4d, 0xd2}, 16,
            /* tag, tag length  */
            {0xb7, 0x96, 0x8e, 0x0b, 0x56, 0x40, 0xe3, 0xb2,
             0x36, 0x56, 0x96, 0x53, 0x20, 0x8b, 0x9d, 0xeb}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x51, 0x58, 0x40, 0xcf, 0x67, 0xd2, 0xe4, 0x0e,
             0xb6, 0x5e, 0x54, 0xa2, 0x4c, 0x72, 0xcb, 0xf2}, 16,
            /* iv, iv length  */
            {0xbf, 0x47, 0xaf, 0xdf, 0xd4, 0x92, 0x13, 0x7a,
             0x24, 0x23, 0x6b, 0xc3, 0x67, 0x97, 0xa8, 0x8e}, 16,
            /* aad, aad length  */
            {0x16, 0x84, 0x3c, 0x09, 0x1d, 0x43, 0xb0, 0xa1,
             0x91, 0xd0, 0xc7, 0x3d, 0x15, 0x60, 0x1b, 0xe9}, 16,
            /* msg, msg length  */
            {0xc8, 0x34, 0x58, 0x8c, 0xb6, 0xda, 0xf9, 0xf0,
             0x6d, 0xd2, 0x35, 0x19, 0xf4, 0xbe, 0x9f, 0x56}, 16,
            /* ct, ct length  */
            {0x20, 0x0a, 0xc4, 0x51, 0xfb, 0xeb, 0x0f, 0x61,
             0x51, 0xd6, 0x15, 0x83, 0xa4, 0x3b, 0x73, 0x43}, 16,
            /* tag, tag length  */
            {0x2a, 0xd4, 0x3e, 0x4c, 0xaa, 0x51, 0x98, 0x3a,
             0x9d, 0x4d, 0x24, 0x48, 0x1b, 0xf4, 0xc8, 0x39}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x2e, 0x44, 0x92, 0xd4, 0x44, 0xe5, 0xb6, 0xf4,
             0xce, 0xc8, 0xc2, 0xd3, 0x61, 0x5a, 0xc8, 0x58}, 16,
            /* iv, iv length  */
            {0xd0, 0x2b, 0xf0, 0x76, 0x3a, 0x9f, 0xef, 0xbf,
             0x70, 0xc3, 0x3a, 0xee, 0x1e, 0x9d, 0xa1, 0xd6}, 16,
            /* aad, aad length  */
            {0x90, 0x4d, 0x86, 0xf1, 0x33, 0xce, 0xc1, 0x5a,
             0x0c, 0x3c, 0xaf, 0x14, 0xd7, 0xe0, 0x29, 0xc8,
             0x2a, 0x07, 0x70, 0x5a, 0x23, 0xf0, 0xd0, 0x80}, 24,
            /* msg, msg length  */
            {0x9e, 0x62, 0xd6, 0x51, 0x1b, 0x0b, 0xda, 0x7d,
             0xd7, 0x74, 0x0b, 0x61, 0x4d, 0x97, 0xba, 0xe0}, 16,
            /* ct, ct length  */
            {0x27, 0xc6, 0xe9, 0xa6, 0x53, 0xc5, 0x25, 0x3c,
             0xa1, 0xc5, 0x67, 0x3f, 0x97, 0xb9, 0xb3, 0x3e}, 16,
            /* tag, tag length  */
            {0x2d, 0x58, 0x12, 0x71, 0xe1, 0xfa, 0x9e, 0x36,
             0x86, 0x13, 0x6c, 0xaa, 0x8f, 0x4d, 0x6c, 0x8e}, 16,
            /* valid */
            1,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe7, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe4, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0x66, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0f, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0xd0, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x12, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x11, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0x72,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x53, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0xd2, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0xb8, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb0, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9a, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x99, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x1b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa6}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa5}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xe7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0x27}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe7, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x53, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0xd0, 0x13, 0xa6, 0xdb, 0x72,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe6, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0x72,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0x27}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0x19, 0xf1, 0x83, 0xaf, 0xec, 0x59, 0x24, 0x0d,
             0xad, 0x67, 0x4e, 0x6d, 0x64, 0x3c, 0xa9, 0x58}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0x66, 0x8e, 0xfc, 0xd0, 0x93, 0x26, 0x5b, 0x72,
             0xd2, 0x18, 0x31, 0x12, 0x1b, 0x43, 0xd6, 0x27}, 16,
            /* valid */
            0,
        },
        {
            /* key, key length  */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length  */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length  */
            {0x00}, 0,
            /* msg, msg length  */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct, ct length  */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length  */
            {0xe7, 0x0f, 0x7d, 0x51, 0x12, 0xa7, 0xda, 0xf3,
             0x53, 0x99, 0xb0, 0x93, 0x9a, 0xc2, 0x57, 0xa6}, 16,
            /* valid */
            0,
        },
    };

    byte ciphertext[sizeof(vectors[0].ct)];
    byte authtag[sizeof(vectors[0].tag)];
    int i;
    int len;
    int ret;


    for (i = 0; i < (int)(sizeof(vectors)/sizeof(vectors[0])); i++) {

        XMEMSET(ciphertext, 0, sizeof(ciphertext));

        len = sizeof(authtag);
        ExpectIntEQ(wc_AesEaxEncryptAuth(vectors[i].key, vectors[i].key_length,
                                         ciphertext,
                                         vectors[i].msg, vectors[i].msg_length,
                                         vectors[i].iv, vectors[i].iv_length,
                                         authtag, len,
                                         vectors[i].aad, vectors[i].aad_length),
                                         0);

        /* check ciphertext matches vector */
        ExpectIntEQ(XMEMCMP(ciphertext, vectors[i].ct, vectors[i].ct_length),
                    0);

        /* check that computed tag matches vector only for vectors marked as valid */
        ret = XMEMCMP(authtag, vectors[i].tag, len);
        if (vectors[i].valid) {
            ExpectIntEQ(ret, 0);
        }
        else {
            ExpectIntNE(ret, 0);
        }

        XMEMSET(ciphertext, 0, sizeof(ciphertext));

        /* Decrypt, checking that the computed auth tags match */
        ExpectIntEQ(wc_AesEaxDecryptAuth(vectors[i].key, vectors[i].key_length,
                                         ciphertext,
                                         vectors[i].ct, vectors[i].ct_length,
                                         vectors[i].iv, vectors[i].iv_length,
                                         authtag, len,
                                         vectors[i].aad, vectors[i].aad_length),
                                         0);

        /* check decrypted ciphertext matches vector plaintext */
        ExpectIntEQ(XMEMCMP(ciphertext, vectors[i].msg, vectors[i].msg_length),
                    0);
    }
    return EXPECT_RESULT();
} /* END test_wc_AesEaxVectors */


/*
 * Testing test_wc_AesEaxEncryptAuth()
 */
static int test_wc_AesEaxEncryptAuth(void)
{
    EXPECT_DECLS;

    const byte key[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F};
    const byte iv[]  = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F};
    const byte aad[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07};
    const byte msg[] = {0x00, 0x01, 0x02, 0x03, 0x04};

    byte ciphertext[sizeof(msg)];
    byte authtag[AES_BLOCK_SIZE];
    int i;
    int len;

    len = sizeof(authtag);
    ExpectIntEQ(wc_AesEaxEncryptAuth(key, sizeof(key),
                                     ciphertext,
                                     msg, sizeof(msg),
                                     iv, sizeof(iv),
                                     authtag, len,
                                     aad, sizeof(aad)),
                                     0);

    /* Test null checking */
    ExpectIntEQ(wc_AesEaxEncryptAuth(NULL, sizeof(key),
                                     ciphertext,
                                     msg, sizeof(msg),
                                     iv, sizeof(iv),
                                     authtag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxEncryptAuth(key, sizeof(key),
                                     NULL,
                                     msg, sizeof(msg),
                                     iv, sizeof(iv),
                                     authtag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxEncryptAuth(key, sizeof(key),
                                     ciphertext,
                                     NULL, sizeof(msg),
                                     iv, sizeof(iv),
                                     authtag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxEncryptAuth(key, sizeof(key),
                                     ciphertext,
                                     msg, sizeof(msg),
                                     NULL, sizeof(iv),
                                     authtag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxEncryptAuth(key, sizeof(key),
                                     ciphertext,
                                     msg, sizeof(msg),
                                     iv, sizeof(iv),
                                     NULL, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxEncryptAuth(key, sizeof(key),
                                     ciphertext,
                                     msg, sizeof(msg),
                                     iv, sizeof(iv),
                                     authtag, len,
                                     NULL, sizeof(aad)),
                                     BAD_FUNC_ARG);

    /* Test bad key lengths */
    for (i = 0; i <= 32; i++) {
        int exp_ret;
        if (i == AES_128_KEY_SIZE || i == AES_192_KEY_SIZE
                                  || i == AES_256_KEY_SIZE) {
            exp_ret = 0;
        }
        else {
            exp_ret = BAD_FUNC_ARG;
        }

        ExpectIntEQ(wc_AesEaxEncryptAuth(key, i,
                                         ciphertext,
                                         msg, sizeof(msg),
                                         iv, sizeof(iv),
                                         authtag, len,
                                         aad, sizeof(aad)),
                                         exp_ret);
    }


    /* Test auth tag size out of range */
    len = AES_BLOCK_SIZE + 1;
    ExpectIntEQ(wc_AesEaxEncryptAuth(key, sizeof(key),
                                     ciphertext,
                                     msg, sizeof(msg),
                                     iv, sizeof(iv),
                                     authtag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);

    return EXPECT_RESULT();
} /* END test_wc_AesEaxEncryptAuth() */


/*
 * Testing test_wc_AesEaxDecryptAuth()
 */
static int test_wc_AesEaxDecryptAuth(void)
{
    EXPECT_DECLS;

    const byte key[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F};
    const byte iv[]  = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F};
    const byte aad[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07};
    const byte ct[] =  {0x00, 0x01, 0x02, 0x03, 0x04};
    /* Garbage tag that should always fail for above aad */
    const byte tag[] = {0xFE, 0xED, 0xBE, 0xEF, 0xDE, 0xAD, 0xC0, 0xDE,
                        0xCA, 0xFE, 0xBE, 0xEF, 0xDE, 0xAF, 0xBE, 0xEF};

    byte plaintext[sizeof(ct)];
    int i;
    int len;

    len = sizeof(tag);
    ExpectIntEQ(wc_AesEaxDecryptAuth(key, sizeof(key),
                                     plaintext,
                                     ct, sizeof(ct),
                                     iv, sizeof(iv),
                                     tag, len,
                                     aad, sizeof(aad)),
                                     AES_EAX_AUTH_E);

    /* Test null checking */
    ExpectIntEQ(wc_AesEaxDecryptAuth(NULL, sizeof(key),
                                     plaintext,
                                     ct, sizeof(ct),
                                     iv, sizeof(iv),
                                     tag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxDecryptAuth(key, sizeof(key),
                                     NULL,
                                     ct, sizeof(ct),
                                     iv, sizeof(iv),
                                     tag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxDecryptAuth(key, sizeof(key),
                                     plaintext,
                                     NULL, sizeof(ct),
                                     iv, sizeof(iv),
                                     tag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxDecryptAuth(key, sizeof(key),
                                     plaintext,
                                     ct, sizeof(ct),
                                     NULL, sizeof(iv),
                                     tag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxDecryptAuth(key, sizeof(key),
                                     plaintext,
                                     ct, sizeof(ct),
                                     iv, sizeof(iv),
                                     NULL, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);
    ExpectIntEQ(wc_AesEaxDecryptAuth(key, sizeof(key),
                                     plaintext,
                                     ct, sizeof(ct),
                                     iv, sizeof(iv),
                                     tag, len,
                                     NULL, sizeof(aad)),
                                     BAD_FUNC_ARG);

    /* Test bad key lengths */
    for (i = 0; i <= 32; i++) {
        int exp_ret;
        if (i == AES_128_KEY_SIZE || i == AES_192_KEY_SIZE
                                  || i == AES_256_KEY_SIZE) {
            exp_ret = AES_EAX_AUTH_E;
        }
        else {
            exp_ret = BAD_FUNC_ARG;
        }

        ExpectIntEQ(wc_AesEaxDecryptAuth(key, i,
                                         plaintext,
                                         ct, sizeof(ct),
                                         iv, sizeof(iv),
                                         tag, len,
                                         aad, sizeof(aad)),
                                         exp_ret);
    }


    /* Test auth tag size out of range */
    len = AES_BLOCK_SIZE + 1;
    ExpectIntEQ(wc_AesEaxDecryptAuth(key, sizeof(key),
                                     plaintext,
                                     ct, sizeof(ct),
                                     iv, sizeof(iv),
                                     tag, len,
                                     aad, sizeof(aad)),
                                     BAD_FUNC_ARG);

    return EXPECT_RESULT();
} /* END test_wc_AesEaxDecryptAuth() */

#endif /* WOLFSSL_AES_EAX &&
        * (!HAVE_FIPS || FIPS_VERSION_GE(5, 3)) && !HAVE_SELFTEST
        */

/*
 * Testing wc_InitDsaKey()
 */
static int test_wc_InitDsaKey(void)
{
    EXPECT_DECLS;
#ifndef NO_DSA
    DsaKey key;

    XMEMSET(&key, 0, sizeof(DsaKey));

    ExpectIntEQ(wc_InitDsaKey(&key), 0);

    /* Pass in bad args. */
    ExpectIntEQ(wc_InitDsaKey(NULL), BAD_FUNC_ARG);

    wc_FreeDsaKey(&key);
#endif
    return EXPECT_RESULT();

} /* END test_wc_InitDsaKey */

/*
 * Testing wc_DsaSign() and wc_DsaVerify()
 */
static int test_wc_DsaSignVerify(void)
{
    EXPECT_DECLS;
#if !defined(NO_DSA)
    DsaKey key;
    WC_RNG rng;
    wc_Sha sha;
    byte   signature[DSA_SIG_SIZE];
    byte   hash[WC_SHA_DIGEST_SIZE];
    word32 idx = 0;
    word32 bytes;
    int    answer;
#ifdef USE_CERT_BUFFERS_1024
    byte   tmp[ONEK_BUF];

    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    byte   tmp[TWOK_BUF];

    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    byte   tmp[TWOK_BUF];
    XFILE  fp = XBADFILE;

    XMEMSET(tmp, 0, sizeof(tmp));
    ExpectTrue((fp = XFOPEN("./certs/dsa2048.der", "rb")) != XBADFILE);
    ExpectTrue((bytes = (word32)XFREAD(tmp, 1, sizeof(tmp), fp)) > 0);
    if (fp != XBADFILE)
        XFCLOSE(fp);
#endif /* END USE_CERT_BUFFERS_1024 */

    ExpectIntEQ(wc_InitSha(&sha), 0);
    ExpectIntEQ(wc_ShaUpdate(&sha, tmp, bytes), 0);
    ExpectIntEQ(wc_ShaFinal(&sha, hash), 0);
    ExpectIntEQ(wc_InitDsaKey(&key), 0);
    ExpectIntEQ(wc_DsaPrivateKeyDecode(tmp, &idx, &key, bytes), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    /* Sign. */
    ExpectIntEQ(wc_DsaSign(hash, signature, &key, &rng), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_DsaSign(NULL, signature, &key, &rng), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaSign(hash, NULL, &key, &rng), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaSign(hash, signature, NULL, &rng), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaSign(hash, signature, &key, NULL), BAD_FUNC_ARG);

    /* Verify. */
    ExpectIntEQ(wc_DsaVerify(hash, signature, &key, &answer), 0);
    ExpectIntEQ(answer, 1);
    /* Pass in bad args. */
    ExpectIntEQ(wc_DsaVerify(NULL, signature, &key, &answer), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaVerify(hash, NULL, &key, &answer), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaVerify(hash, signature, NULL, &answer), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaVerify(hash, signature, &key, NULL), BAD_FUNC_ARG);

#if !defined(HAVE_FIPS) && defined(WOLFSSL_PUBLIC_MP)
    /* hard set q to 0 and test fail case */
    mp_free(&key.q);
    mp_init(&key.q);
    ExpectIntEQ(wc_DsaSign(hash, signature, &key, &rng), BAD_FUNC_ARG);

    mp_set(&key.q, 1);
    ExpectIntEQ(wc_DsaSign(hash, signature, &key, &rng), BAD_FUNC_ARG);
#endif

    DoExpectIntEQ(wc_FreeRng(&rng),0);
    wc_FreeDsaKey(&key);
    wc_ShaFree(&sha);
#endif
    return EXPECT_RESULT();

} /* END test_wc_DsaSign */

/*
 * Testing wc_DsaPrivateKeyDecode() and wc_DsaPublicKeyDecode()
 */
static int test_wc_DsaPublicPrivateKeyDecode(void)
{
    EXPECT_DECLS;
#if !defined(NO_DSA)
    DsaKey key;
    word32 bytes;
    word32 idx  = 0;
    int    ret;
#ifdef USE_CERT_BUFFERS_1024
    byte   tmp[ONEK_BUF];

    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    byte   tmp[TWOK_BUF];

    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    byte   tmp[TWOK_BUF];
    XFILE  fp = XBADFILE;

    XMEMSET(tmp, 0, sizeof(tmp));
    ExpectTrue((fp = XFOPEN("./certs/dsa2048.der", "rb")) != XBADFILE);
    ExpectTrue((bytes = (word32) XFREAD(tmp, 1, sizeof(tmp), fp)) > 0);
    if (fp != XBADFILE)
        XFCLOSE(fp);
#endif /* END USE_CERT_BUFFERS_1024 */

    ExpectIntEQ(wc_InitDsaKey(&key), 0);
    ExpectIntEQ(wc_DsaPrivateKeyDecode(tmp, &idx, &key, bytes), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_DsaPrivateKeyDecode(NULL, &idx, &key, bytes), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaPrivateKeyDecode(tmp, NULL, &key, bytes), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaPrivateKeyDecode(tmp, &idx, NULL, bytes), BAD_FUNC_ARG);
    ExpectIntLT(ret = wc_DsaPrivateKeyDecode(tmp, &idx, &key, bytes), 0);
    ExpectTrue((ret == ASN_PARSE_E) || (ret == BUFFER_E));
    wc_FreeDsaKey(&key);

    ExpectIntEQ(wc_InitDsaKey(&key), 0);
    idx = 0; /* Reset */
    ExpectIntEQ(wc_DsaPublicKeyDecode(tmp, &idx, &key, bytes), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_DsaPublicKeyDecode(NULL, &idx, &key, bytes), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaPublicKeyDecode(tmp, NULL, &key, bytes), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaPublicKeyDecode(tmp, &idx, NULL, bytes), BAD_FUNC_ARG);
    ExpectIntLT(ret = wc_DsaPublicKeyDecode(tmp, &idx, &key, bytes), 0);
    ExpectTrue((ret == ASN_PARSE_E) || (ret == BUFFER_E));
    wc_FreeDsaKey(&key);
#endif /* !NO_DSA */
    return EXPECT_RESULT();

} /* END test_wc_DsaPublicPrivateKeyDecode */


/*
 * Testing wc_MakeDsaKey() and wc_MakeDsaParameters()
 */
static int test_wc_MakeDsaKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_DSA) && defined(WOLFSSL_KEY_GEN)
    DsaKey genKey;
    WC_RNG rng;

    XMEMSET(&genKey, 0, sizeof(genKey));
    XMEMSET(&rng, 0, sizeof(rng));

    ExpectIntEQ(wc_InitDsaKey(&genKey), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectIntEQ(wc_MakeDsaParameters(&rng, ONEK_BUF, &genKey), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_MakeDsaParameters(NULL, ONEK_BUF, &genKey), BAD_FUNC_ARG);
    ExpectIntEQ(wc_MakeDsaParameters(&rng, ONEK_BUF, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_MakeDsaParameters(&rng, ONEK_BUF + 1, &genKey),
        BAD_FUNC_ARG);

    ExpectIntEQ(wc_MakeDsaKey(&rng, &genKey), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_MakeDsaKey(NULL, &genKey), BAD_FUNC_ARG);
    ExpectIntEQ(wc_MakeDsaKey(&rng, NULL), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_FreeDsaKey(&genKey);
#endif
    return EXPECT_RESULT();
} /* END test_wc_MakeDsaKey */

/*
 * Testing wc_DsaKeyToDer()
 */
static int test_wc_DsaKeyToDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_DSA) && defined(WOLFSSL_KEY_GEN)
    DsaKey key;
    word32 bytes;
    word32 idx = 0;
#ifdef USE_CERT_BUFFERS_1024
    byte   tmp[ONEK_BUF];
    byte   der[ONEK_BUF];

    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMSET(der, 0, sizeof(der));
    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    byte   tmp[TWOK_BUF];
    byte   der[TWOK_BUF];

    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMSET(der, 0, sizeof(der));
    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    byte   tmp[TWOK_BUF];
    byte   der[TWOK_BUF];
    XFILE fp = XBADFILE;

    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMSET(der, 0, sizeof(der));
    ExpectTrue((fp = XFOPEN("./certs/dsa2048.der", "rb")) != XBADFILE);
    ExpectTrue((bytes = (word32) XFREAD(tmp, 1, sizeof(tmp), fp)) > 0);
    if (fp != XBADFILE)
        XFCLOSE(fp);
#endif /* END USE_CERT_BUFFERS_1024 */

    XMEMSET(&key, 0, sizeof(DsaKey));

    ExpectIntEQ(wc_InitDsaKey(&key), 0);
    ExpectIntEQ(wc_DsaPrivateKeyDecode(tmp, &idx, &key, bytes), 0);
    ExpectIntGE(wc_DsaKeyToDer(&key, der, bytes), 0);
    ExpectIntEQ(XMEMCMP(der, tmp, bytes), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_DsaKeyToDer(NULL, der, FOURK_BUF), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaKeyToDer(&key, NULL, FOURK_BUF), BAD_FUNC_ARG);

    wc_FreeDsaKey(&key);
#endif /* !NO_DSA && WOLFSSL_KEY_GEN */
    return EXPECT_RESULT();

} /* END test_wc_DsaKeyToDer */

/*
 *  Testing wc_DsaKeyToPublicDer()
 *  (indirectly testing setDsaPublicKey())
 */
static int test_wc_DsaKeyToPublicDer(void)
{
    EXPECT_DECLS;
#ifndef HAVE_SELFTEST
#if !defined(NO_DSA) && defined(WOLFSSL_KEY_GEN)
    DsaKey key;
    WC_RNG rng;
    byte*  der = NULL;
    word32 sz = 0;
    word32 idx = 0;

    XMEMSET(&key, 0, sizeof(DsaKey));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectNotNull(der = (byte*)XMALLOC(ONEK_BUF, NULL,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitDsaKey(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_MakeDsaParameters(&rng, ONEK_BUF, &key), 0);
    ExpectIntEQ(wc_MakeDsaKey(&rng, &key), 0);

    ExpectIntGE(sz = wc_DsaKeyToPublicDer(&key, der, ONEK_BUF), 0);
    wc_FreeDsaKey(&key);

    idx = 0;
    ExpectIntEQ(wc_DsaPublicKeyDecode(der, &idx, &key, sz), 0);

    /* Test without the SubjectPublicKeyInfo header */
    ExpectIntGE(sz = wc_SetDsaPublicKey(der, &key, ONEK_BUF, 0), 0);
    wc_FreeDsaKey(&key);
    idx = 0;
    ExpectIntEQ(wc_DsaPublicKeyDecode(der, &idx, &key, sz), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_DsaKeyToPublicDer(NULL, der, FOURK_BUF), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DsaKeyToPublicDer(&key, NULL, FOURK_BUF), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_FreeDsaKey(&key);
    XFREE(der, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* !NO_DSA && WOLFSSL_KEY_GEN */
#endif /* !HAVE_SELFTEST */
    return EXPECT_RESULT();

} /* END test_wc_DsaKeyToPublicDer */

/*
 * Testing wc_DsaImportParamsRaw()
 */
static int test_wc_DsaImportParamsRaw(void)
{
    EXPECT_DECLS;
#if !defined(NO_DSA)
    DsaKey key;
    /* [mod = L=1024, N=160], from CAVP KeyPair */
    const char* p = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d"
                    "4b725ef341eabb47cf8a7a8a41e792a156b7ce97206c4f9c"
                    "5ce6fc5ae7912102b6b502e59050b5b21ce263dddb2044b6"
                    "52236f4d42ab4b5d6aa73189cef1ace778d7845a5c1c1c71"
                    "47123188f8dc551054ee162b634d60f097f719076640e209"
                    "80a0093113a8bd73";
    const char* q = "96c5390a8b612c0e422bb2b0ea194a3ec935a281";
    const char* g = "06b7861abbd35cc89e79c52f68d20875389b127361ca66822"
                    "138ce4991d2b862259d6b4548a6495b195aa0e0b6137ca37e"
                    "b23b94074d3c3d300042bdf15762812b6333ef7b07ceba786"
                    "07610fcc9ee68491dbc1e34cd12615474e52b18bc934fb00c"
                    "61d39e7da8902291c4434a4e2224c3f4fd9f93cd6f4f17fc0"
                    "76341a7e7d9";
    /* invalid p and q parameters */
    const char* invalidP = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d";
    const char* invalidQ = "96c5390a";

    XMEMSET(&key, 0, sizeof(DsaKey));

    ExpectIntEQ(wc_InitDsaKey(&key), 0);
    ExpectIntEQ(wc_DsaImportParamsRaw(&key, p, q, g), 0);

    /* test bad args */
    /* null key struct */
    ExpectIntEQ(wc_DsaImportParamsRaw(NULL, p, q, g), BAD_FUNC_ARG);
    /* null param pointers */
    ExpectIntEQ(wc_DsaImportParamsRaw(&key, NULL, NULL, NULL), BAD_FUNC_ARG);
    /* illegal p length */
    ExpectIntEQ(wc_DsaImportParamsRaw(&key, invalidP, q, g), BAD_FUNC_ARG);
    /* illegal q length */
    ExpectIntEQ(wc_DsaImportParamsRaw(&key, p, invalidQ, g), BAD_FUNC_ARG);

    wc_FreeDsaKey(&key);
#endif
    return EXPECT_RESULT();

} /* END test_wc_DsaImportParamsRaw */

/*
 * Testing wc_DsaImportParamsRawCheck()
 */
static int test_wc_DsaImportParamsRawCheck(void)
{
    EXPECT_DECLS;
#if !defined(NO_DSA) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    DsaKey key;
    int    trusted = 0;
    /* [mod = L=1024, N=160], from CAVP KeyPair */
    const char* p = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d"
                    "4b725ef341eabb47cf8a7a8a41e792a156b7ce97206c4f9c"
                    "5ce6fc5ae7912102b6b502e59050b5b21ce263dddb2044b6"
                    "52236f4d42ab4b5d6aa73189cef1ace778d7845a5c1c1c71"
                    "47123188f8dc551054ee162b634d60f097f719076640e209"
                    "80a0093113a8bd73";
    const char* q = "96c5390a8b612c0e422bb2b0ea194a3ec935a281";
    const char* g = "06b7861abbd35cc89e79c52f68d20875389b127361ca66822"
                    "138ce4991d2b862259d6b4548a6495b195aa0e0b6137ca37e"
                    "b23b94074d3c3d300042bdf15762812b6333ef7b07ceba786"
                    "07610fcc9ee68491dbc1e34cd12615474e52b18bc934fb00c"
                    "61d39e7da8902291c4434a4e2224c3f4fd9f93cd6f4f17fc0"
                    "76341a7e7d9";
    /* invalid p and q parameters */
    const char* invalidP = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d";
    const char* invalidQ = "96c5390a";

    ExpectIntEQ(wc_InitDsaKey(&key), 0);
    ExpectIntEQ(wc_DsaImportParamsRawCheck(&key, p, q, g, trusted, NULL), 0);

    /* test bad args */
    /* null key struct */
    ExpectIntEQ(wc_DsaImportParamsRawCheck(NULL, p, q, g, trusted, NULL),
        BAD_FUNC_ARG);
    /* null param pointers */
    ExpectIntEQ(wc_DsaImportParamsRawCheck(&key, NULL, NULL, NULL, trusted,
        NULL), BAD_FUNC_ARG);
    /* illegal p length */
    ExpectIntEQ(wc_DsaImportParamsRawCheck(&key, invalidP, q, g, trusted, NULL),
        BAD_FUNC_ARG);
    /* illegal q length */
    ExpectIntEQ(wc_DsaImportParamsRawCheck(&key, p, invalidQ, g, trusted, NULL),
        BAD_FUNC_ARG);

    wc_FreeDsaKey(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_DsaImportParamsRawCheck */

/*
 * Testing wc_DsaExportParamsRaw()
 */
static int test_wc_DsaExportParamsRaw(void)
{
    EXPECT_DECLS;
#if !defined(NO_DSA)
    DsaKey key;
    /* [mod = L=1024, N=160], from CAVP KeyPair */
    const char* p = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d"
                    "4b725ef341eabb47cf8a7a8a41e792a156b7ce97206c4f9c"
                    "5ce6fc5ae7912102b6b502e59050b5b21ce263dddb2044b6"
                    "52236f4d42ab4b5d6aa73189cef1ace778d7845a5c1c1c71"
                    "47123188f8dc551054ee162b634d60f097f719076640e209"
                    "80a0093113a8bd73";
    const char* q = "96c5390a8b612c0e422bb2b0ea194a3ec935a281";
    const char* g = "06b7861abbd35cc89e79c52f68d20875389b127361ca66822"
                    "138ce4991d2b862259d6b4548a6495b195aa0e0b6137ca37e"
                    "b23b94074d3c3d300042bdf15762812b6333ef7b07ceba786"
                    "07610fcc9ee68491dbc1e34cd12615474e52b18bc934fb00c"
                    "61d39e7da8902291c4434a4e2224c3f4fd9f93cd6f4f17fc0"
                    "76341a7e7d9";
    const char* pCompare = "\xd3\x83\x11\xe2\xcd\x38\x8c\x3e\xd6\x98\xe8\x2f"
                           "\xdf\x88\xeb\x92\xb5\xa9\xa4\x83\xdc\x88\x00\x5d"
                           "\x4b\x72\x5e\xf3\x41\xea\xbb\x47\xcf\x8a\x7a\x8a"
                           "\x41\xe7\x92\xa1\x56\xb7\xce\x97\x20\x6c\x4f\x9c"
                           "\x5c\xe6\xfc\x5a\xe7\x91\x21\x02\xb6\xb5\x02\xe5"
                           "\x90\x50\xb5\xb2\x1c\xe2\x63\xdd\xdb\x20\x44\xb6"
                           "\x52\x23\x6f\x4d\x42\xab\x4b\x5d\x6a\xa7\x31\x89"
                           "\xce\xf1\xac\xe7\x78\xd7\x84\x5a\x5c\x1c\x1c\x71"
                           "\x47\x12\x31\x88\xf8\xdc\x55\x10\x54\xee\x16\x2b"
                           "\x63\x4d\x60\xf0\x97\xf7\x19\x07\x66\x40\xe2\x09"
                           "\x80\xa0\x09\x31\x13\xa8\xbd\x73";
    const char* qCompare = "\x96\xc5\x39\x0a\x8b\x61\x2c\x0e\x42\x2b\xb2\xb0"
                           "\xea\x19\x4a\x3e\xc9\x35\xa2\x81";
    const char* gCompare = "\x06\xb7\x86\x1a\xbb\xd3\x5c\xc8\x9e\x79\xc5\x2f"
                           "\x68\xd2\x08\x75\x38\x9b\x12\x73\x61\xca\x66\x82"
                           "\x21\x38\xce\x49\x91\xd2\xb8\x62\x25\x9d\x6b\x45"
                           "\x48\xa6\x49\x5b\x19\x5a\xa0\xe0\xb6\x13\x7c\xa3"
                           "\x7e\xb2\x3b\x94\x07\x4d\x3c\x3d\x30\x00\x42\xbd"
                           "\xf1\x57\x62\x81\x2b\x63\x33\xef\x7b\x07\xce\xba"
                           "\x78\x60\x76\x10\xfc\xc9\xee\x68\x49\x1d\xbc\x1e"
                           "\x34\xcd\x12\x61\x54\x74\xe5\x2b\x18\xbc\x93\x4f"
                           "\xb0\x0c\x61\xd3\x9e\x7d\xa8\x90\x22\x91\xc4\x43"
                           "\x4a\x4e\x22\x24\xc3\xf4\xfd\x9f\x93\xcd\x6f\x4f"
                           "\x17\xfc\x07\x63\x41\xa7\xe7\xd9";
    byte pOut[MAX_DSA_PARAM_SIZE];
    byte qOut[MAX_DSA_PARAM_SIZE];
    byte gOut[MAX_DSA_PARAM_SIZE];
    word32 pOutSz;
    word32 qOutSz;
    word32 gOutSz;

    XMEMSET(&key, 0, sizeof(DsaKey));

    ExpectIntEQ(wc_InitDsaKey(&key), 0);
    /* first test using imported raw parameters, for expected */
    ExpectIntEQ(wc_DsaImportParamsRaw(&key, p, q, g), 0);
    pOutSz = sizeof(pOut);
    qOutSz = sizeof(qOut);
    gOutSz = sizeof(gOut);
    ExpectIntEQ(wc_DsaExportParamsRaw(&key, pOut, &pOutSz, qOut, &qOutSz, gOut,
        &gOutSz), 0);
    /* validate exported parameters are correct */
    ExpectIntEQ(XMEMCMP(pOut, pCompare, pOutSz), 0);
    ExpectIntEQ(XMEMCMP(qOut, qCompare, qOutSz), 0);
    ExpectIntEQ(XMEMCMP(gOut, gCompare, gOutSz), 0);

    /* test bad args */
    /* null key struct */
    ExpectIntEQ(wc_DsaExportParamsRaw(NULL, pOut, &pOutSz, qOut, &qOutSz, gOut,
        &gOutSz), BAD_FUNC_ARG);
    /* null output pointers */
    ExpectIntEQ(wc_DsaExportParamsRaw(&key, NULL, &pOutSz, NULL, &qOutSz, NULL,
        &gOutSz), LENGTH_ONLY_E);
    /* null output size pointers */
    ExpectIntEQ( wc_DsaExportParamsRaw(&key, pOut, NULL, qOut, NULL, gOut,
        NULL), BAD_FUNC_ARG);
    /* p output buffer size too small */
    pOutSz = 1;
    ExpectIntEQ(wc_DsaExportParamsRaw(&key, pOut, &pOutSz, qOut, &qOutSz, gOut,
        &gOutSz), BUFFER_E);
    pOutSz = sizeof(pOut);
    /* q output buffer size too small */
    qOutSz = 1;
    ExpectIntEQ(wc_DsaExportParamsRaw(&key, pOut, &pOutSz, qOut, &qOutSz, gOut,
        &gOutSz), BUFFER_E);
    qOutSz = sizeof(qOut);
    /* g output buffer size too small */
    gOutSz = 1;
    ExpectIntEQ(wc_DsaExportParamsRaw(&key, pOut, &pOutSz, qOut, &qOutSz, gOut,
        &gOutSz), BUFFER_E);

    wc_FreeDsaKey(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_DsaExportParamsRaw */

/*
 * Testing wc_DsaExportKeyRaw()
 */
static int test_wc_DsaExportKeyRaw(void)
{
    EXPECT_DECLS;
#if !defined(NO_DSA) && defined(WOLFSSL_KEY_GEN)
    DsaKey key;
    WC_RNG rng;
    byte xOut[MAX_DSA_PARAM_SIZE];
    byte yOut[MAX_DSA_PARAM_SIZE];
    word32 xOutSz, yOutSz;

    XMEMSET(&key, 0, sizeof(key));
    XMEMSET(&rng, 0, sizeof(rng));

    ExpectIntEQ(wc_InitDsaKey(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_MakeDsaParameters(&rng, 1024, &key), 0);
    ExpectIntEQ(wc_MakeDsaKey(&rng, &key), 0);

    /* try successful export */
    xOutSz = sizeof(xOut);
    yOutSz = sizeof(yOut);
    ExpectIntEQ(wc_DsaExportKeyRaw(&key, xOut, &xOutSz, yOut, &yOutSz), 0);

    /* test bad args */
    /* null key struct */
    ExpectIntEQ(wc_DsaExportKeyRaw(NULL, xOut, &xOutSz, yOut, &yOutSz),
        BAD_FUNC_ARG);
    /* null output pointers */
    ExpectIntEQ(wc_DsaExportKeyRaw(&key, NULL, &xOutSz, NULL, &yOutSz),
        LENGTH_ONLY_E);
    /* null output size pointers */
    ExpectIntEQ(wc_DsaExportKeyRaw(&key, xOut, NULL, yOut, NULL),
        BAD_FUNC_ARG);
    /* x output buffer size too small */
    xOutSz = 1;
    ExpectIntEQ(wc_DsaExportKeyRaw(&key, xOut, &xOutSz, yOut, &yOutSz),
        BUFFER_E);
    xOutSz = sizeof(xOut);
    /* y output buffer size too small */
    yOutSz = 1;
    ExpectIntEQ(wc_DsaExportKeyRaw(&key, xOut, &xOutSz, yOut, &yOutSz),
        BUFFER_E);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_FreeDsaKey(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_DsaExportParamsRaw */


/*
 * Testing wc_ed25519_make_key().
 */
static int test_wc_ed25519_make_key(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_MAKE_KEY)
    ed25519_key   key;
    WC_RNG        rng;
    unsigned char pubkey[ED25519_PUB_KEY_SIZE+1];
    int           pubkey_sz = ED25519_PUB_KEY_SIZE;

    XMEMSET(&key, 0, sizeof(ed25519_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectIntEQ(wc_ed25519_make_public(&key, pubkey, pubkey_sz),
        ECC_PRIV_KEY_E);
    ExpectIntEQ(wc_ed25519_make_public(&key, pubkey+1, pubkey_sz),
        ECC_PRIV_KEY_E);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_make_key(NULL, ED25519_KEY_SIZE, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, NULL),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE - 1, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE + 1, &key),
        BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed25519_make_key */

/*
 * Testing wc_ed25519_init()
 */
static int test_wc_ed25519_init(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519)
    ed25519_key key;

    XMEMSET(&key, 0, sizeof(ed25519_key));

    ExpectIntEQ(wc_ed25519_init(&key), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_init(NULL), BAD_FUNC_ARG);

    wc_ed25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed25519_init */

/*
 * Test wc_ed25519_sign_msg() and wc_ed25519_verify_msg()
 */
static int test_wc_ed25519_sign_msg(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_SIGN)
    WC_RNG      rng;
    ed25519_key key;
    byte        msg[] = "Everybody gets Friday off.\n";
    byte        sig[ED25519_SIG_SIZE+1];
    word32      msglen = sizeof(msg);
    word32      siglen = ED25519_SIG_SIZE;
    word32      badSigLen = ED25519_SIG_SIZE - 1;
#ifdef HAVE_ED25519_VERIFY
    int         verify_ok = 0; /*1 = Verify success.*/
#endif

    /* Initialize stack variables. */
    XMEMSET(&key, 0, sizeof(ed25519_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(sig, 0, sizeof(sig));

    /* Initialize key. */
    ExpectIntEQ(wc_ed25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_ed25519_sign_msg(msg, msglen, sig, &siglen, &key), 0);
    ExpectIntEQ(siglen, ED25519_SIG_SIZE);
    ExpectIntEQ(wc_ed25519_sign_msg(msg, msglen, sig+1, &siglen, &key), 0);
    ExpectIntEQ(siglen, ED25519_SIG_SIZE);

    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_sign_msg(NULL, msglen, sig, &siglen, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_sign_msg(msg, msglen, NULL, &siglen, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_sign_msg(msg, msglen, sig, NULL, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_sign_msg(msg, msglen, sig, &siglen, NULL),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_sign_msg(msg, msglen, sig, &badSigLen, &key),
        BUFFER_E);
    ExpectIntEQ(badSigLen, ED25519_SIG_SIZE);
    badSigLen -= 1;

#ifdef HAVE_ED25519_VERIFY
    ExpectIntEQ(wc_ed25519_verify_msg(sig+1, siglen, msg, msglen, &verify_ok,
        &key), 0);
    ExpectIntEQ(verify_ok, 1);

    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_verify_msg(sig+1, siglen - 1, msg, msglen,
        &verify_ok, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_verify_msg(sig+1, siglen + 1, msg, msglen,
        &verify_ok, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_verify_msg(NULL, siglen, msg, msglen, &verify_ok,
        &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_verify_msg(sig+1, siglen, NULL, msglen, &verify_ok,
        &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_verify_msg(sig+1, siglen, msg, msglen, NULL, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_verify_msg(sig+1, siglen, msg, msglen, &verify_ok,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_verify_msg(sig+1, badSigLen, msg, msglen, &verify_ok,
        &key), BAD_FUNC_ARG);
#endif /* Verify. */

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed25519_free(&key);
#endif
    return EXPECT_RESULT();

} /* END test_wc_ed25519_sign_msg */

/*
 * Testing wc_ed25519_import_public()
 */
static int test_wc_ed25519_import_public(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT)
    ed25519_key pubKey;
    WC_RNG      rng;
    const byte  in[] = "Ed25519PublicKeyUnitTest......\n";
    word32      inlen = sizeof(in);

    XMEMSET(&pubKey, 0, sizeof(ed25519_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed25519_init(&pubKey), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
#ifdef HAVE_ED25519_MAKE_KEY
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &pubKey), 0);
#endif

    ExpectIntEQ(wc_ed25519_import_public_ex(in, inlen, &pubKey, 1), 0);
    ExpectIntEQ(XMEMCMP(in, pubKey.p, inlen), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_import_public(NULL, inlen, &pubKey), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_import_public(in, inlen, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_import_public(in, inlen - 1, &pubKey), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed25519_free(&pubKey);
#endif
    return EXPECT_RESULT();
} /* END wc_ed25519_import_public */

/*
 * Testing wc_ed25519_import_private_key()
 */
static int test_wc_ed25519_import_private_key(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT)
    ed25519_key key;
    WC_RNG      rng;
    const byte  privKey[] = "Ed25519PrivateKeyUnitTest.....\n";
    const byte  pubKey[] = "Ed25519PublicKeyUnitTest......\n";
    word32      privKeySz = sizeof(privKey);
    word32      pubKeySz = sizeof(pubKey);
#ifdef HAVE_ED25519_KEY_EXPORT
    byte        bothKeys[sizeof(privKey) + sizeof(pubKey)];
    word32      bothKeysSz = sizeof(bothKeys);
#endif

    XMEMSET(&key, 0, sizeof(ed25519_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
#ifdef HAVE_ED25519_MAKE_KEY
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key), 0);
#endif

    ExpectIntEQ(wc_ed25519_import_private_key_ex(privKey, privKeySz, pubKey,
        pubKeySz, &key, 1), 0);
    ExpectIntEQ(XMEMCMP(pubKey, key.p, privKeySz), 0);
    ExpectIntEQ(XMEMCMP(privKey, key.k, pubKeySz), 0);

#ifdef HAVE_ED25519_KEY_EXPORT
    ExpectIntEQ(wc_ed25519_export_private(&key, bothKeys, &bothKeysSz), 0);
    ExpectIntEQ(wc_ed25519_import_private_key_ex(bothKeys, bothKeysSz, NULL, 0,
        &key, 1), 0);
    ExpectIntEQ(XMEMCMP(pubKey, key.p, privKeySz), 0);
    ExpectIntEQ(XMEMCMP(privKey, key.k, pubKeySz), 0);
#endif

    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_import_private_key(NULL, privKeySz, pubKey, pubKeySz,
        &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_import_private_key(privKey, privKeySz, NULL,
        pubKeySz, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_import_private_key(privKey, privKeySz, pubKey,
        pubKeySz, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_import_private_key(privKey, privKeySz - 1, pubKey,
        pubKeySz, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_import_private_key(privKey, privKeySz, pubKey,
        pubKeySz - 1, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_import_private_key(privKey, privKeySz, NULL, 0,
        &key), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed25519_import_private_key */

/*
 * Testing wc_ed25519_export_public() and wc_ed25519_export_private_only()
 */
static int test_wc_ed25519_export(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    ed25519_key key;
    WC_RNG      rng;
    byte        priv[ED25519_PRV_KEY_SIZE];
    byte        pub[ED25519_PUB_KEY_SIZE];
    word32      privSz = sizeof(priv);
    word32      pubSz = sizeof(pub);
#ifndef HAVE_ED25519_MAKE_KEY
    const byte  privKey[] = {
        0xf8, 0x55, 0xb7, 0xb6, 0x49, 0x3f, 0x99, 0x9c,
        0x88, 0xe3, 0xc5, 0x42, 0x6a, 0xa4, 0x47, 0x4a,
        0xe4, 0x95, 0xda, 0xdb, 0xbf, 0xf8, 0xa7, 0x42,
        0x9d, 0x0e, 0xe7, 0xd0, 0x57, 0x8f, 0x16, 0x69
    };
    const byte  pubKey[] = {
        0x42, 0x3b, 0x7a, 0xf9, 0x82, 0xcf, 0xf9, 0xdf,
        0x19, 0xdd, 0xf3, 0xf0, 0x32, 0x29, 0x6d, 0xfa,
        0xfd, 0x76, 0x4f, 0x68, 0xc2, 0xc2, 0xe0, 0x6c,
        0x47, 0xae, 0xc2, 0x55, 0x68, 0xac, 0x0d, 0x4d
    };
#endif

    XMEMSET(&key, 0, sizeof(ed25519_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
#ifdef HAVE_ED25519_MAKE_KEY
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key), 0);
#else
    ExpectIntEQ(wc_ed25519_import_private_key_ex(privKey, sizeof(privKey),
        pubKey, sizeof(pubKey), &key, 1), 0);
#endif

    ExpectIntEQ(wc_ed25519_export_public(&key, pub, &pubSz), 0);
    ExpectIntEQ(pubSz, ED25519_KEY_SIZE);
    ExpectIntEQ(XMEMCMP(key.p, pub, pubSz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_export_public(NULL, pub, &pubSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_public(&key, NULL, &pubSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_public(&key, pub, NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed25519_export_private_only(&key, priv, &privSz), 0);
    ExpectIntEQ(privSz, ED25519_KEY_SIZE);
    ExpectIntEQ(XMEMCMP(key.k, priv, privSz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_export_private_only(NULL, priv, &privSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_private_only(&key, NULL, &privSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_private_only(&key, priv, NULL),
        BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed25519_export */

/*
 *  Testing wc_ed25519_size()
 */
static int test_wc_ed25519_size(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519)
    ed25519_key key;
    WC_RNG      rng;
#ifndef HAVE_ED25519_MAKE_KEY
    const byte  privKey[] = {
        0xf8, 0x55, 0xb7, 0xb6, 0x49, 0x3f, 0x99, 0x9c,
        0x88, 0xe3, 0xc5, 0x42, 0x6a, 0xa4, 0x47, 0x4a,
        0xe4, 0x95, 0xda, 0xdb, 0xbf, 0xf8, 0xa7, 0x42,
        0x9d, 0x0e, 0xe7, 0xd0, 0x57, 0x8f, 0x16, 0x69
    };
    const byte  pubKey[] = {
        0x42, 0x3b, 0x7a, 0xf9, 0x82, 0xcf, 0xf9, 0xdf,
        0x19, 0xdd, 0xf3, 0xf0, 0x32, 0x29, 0x6d, 0xfa,
        0xfd, 0x76, 0x4f, 0x68, 0xc2, 0xc2, 0xe0, 0x6c,
        0x47, 0xae, 0xc2, 0x55, 0x68, 0xac, 0x0d, 0x4d
    };
#endif

    XMEMSET(&key, 0, sizeof(ed25519_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
#ifdef HAVE_ED25519_MAKE_KEY
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key), 0);
#else
    ExpectIntEQ(wc_ed25519_import_private_key_ex(privKey, sizeof(privKey),
        pubKey, sizeof(pubKey), &key, 1), 0);
#endif

    ExpectIntEQ(wc_ed25519_size(&key), ED25519_KEY_SIZE);
    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_size(NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed25519_sig_size(&key), ED25519_SIG_SIZE);
    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_sig_size(NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed25519_pub_size(&key), ED25519_PUB_KEY_SIZE);
    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_pub_size(NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed25519_priv_size(&key), ED25519_PRV_KEY_SIZE);
    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_priv_size(NULL), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed25519_size */

/*
 * Testing wc_ed25519_export_private() and wc_ed25519_export_key()
 */
static int test_wc_ed25519_exportKey(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    WC_RNG      rng;
    ed25519_key key;
    byte        priv[ED25519_PRV_KEY_SIZE];
    byte        pub[ED25519_PUB_KEY_SIZE];
    byte        privOnly[ED25519_PRV_KEY_SIZE];
    word32      privSz      = sizeof(priv);
    word32      pubSz       = sizeof(pub);
    word32      privOnlySz  = sizeof(privOnly);
#ifndef HAVE_ED25519_MAKE_KEY
    const byte  privKey[] = {
        0xf8, 0x55, 0xb7, 0xb6, 0x49, 0x3f, 0x99, 0x9c,
        0x88, 0xe3, 0xc5, 0x42, 0x6a, 0xa4, 0x47, 0x4a,
        0xe4, 0x95, 0xda, 0xdb, 0xbf, 0xf8, 0xa7, 0x42,
        0x9d, 0x0e, 0xe7, 0xd0, 0x57, 0x8f, 0x16, 0x69
    };
    const byte  pubKey[] = {
        0x42, 0x3b, 0x7a, 0xf9, 0x82, 0xcf, 0xf9, 0xdf,
        0x19, 0xdd, 0xf3, 0xf0, 0x32, 0x29, 0x6d, 0xfa,
        0xfd, 0x76, 0x4f, 0x68, 0xc2, 0xc2, 0xe0, 0x6c,
        0x47, 0xae, 0xc2, 0x55, 0x68, 0xac, 0x0d, 0x4d
    };
#endif

    XMEMSET(&key, 0, sizeof(ed25519_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
#ifdef HAVE_ED25519_MAKE_KEY
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key), 0);
#else
    ExpectIntEQ(wc_ed25519_import_private_key_ex(privKey, sizeof(privKey),
        pubKey, sizeof(pubKey), &key, 1), 0);
#endif

    ExpectIntEQ(wc_ed25519_export_private(&key, privOnly, &privOnlySz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_export_private(NULL, privOnly, &privOnlySz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_private(&key, NULL, &privOnlySz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_private(&key, privOnly, NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed25519_export_key(&key, priv, &privSz, pub, &pubSz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed25519_export_key(NULL, priv, &privSz, pub, &pubSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_key(&key, NULL, &privSz, pub, &pubSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_key(&key, priv, NULL, pub, &pubSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_key(&key, priv, &privSz, NULL, &pubSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_export_key(&key, priv, &privSz, pub, NULL),
        BAD_FUNC_ARG);

    /* Cross check output. */
    ExpectIntEQ(XMEMCMP(priv, privOnly, privSz), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed25519_exportKey */

/*
 * Testing wc_Ed25519PublicKeyToDer
 */
static int test_wc_Ed25519PublicKeyToDer(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    ed25519_key key;
    byte        derBuf[1024];

    XMEMSET(&key, 0, sizeof(ed25519_key));

    /* Test bad args */
    ExpectIntEQ(wc_Ed25519PublicKeyToDer(NULL, NULL, 0, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed25519_init(&key), 0);
    ExpectIntEQ(wc_Ed25519PublicKeyToDer(&key, derBuf, 0, 0), BUFFER_E);
    wc_ed25519_free(&key);

    /*  Test good args */
    if (EXPECT_SUCCESS()) {
        WC_RNG rng;

        XMEMSET(&rng, 0, sizeof(WC_RNG));

        ExpectIntEQ(wc_ed25519_init(&key), 0);
        ExpectIntEQ(wc_InitRng(&rng), 0);
        ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key), 0);
        ExpectIntGT(wc_Ed25519PublicKeyToDer(&key, derBuf, 1024, 1), 0);

        DoExpectIntEQ(wc_FreeRng(&rng), 0);
        wc_ed25519_free(&key);
    }
#endif
    return EXPECT_RESULT();
} /* END testing wc_Ed25519PublicKeyToDer */

/*
 * Testing wc_curve25519_init and wc_curve25519_free.
 */
static int test_wc_curve25519_init(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519)
    curve25519_key key;

    ExpectIntEQ(wc_curve25519_init(&key), 0);
    /* Test bad args for wc_curve25519_init */
    ExpectIntEQ(wc_curve25519_init(NULL), BAD_FUNC_ARG);

    /* Test good args for wc_curve_25519_free */
    wc_curve25519_free(&key);
    /* Test bad args for wc_curve25519 free. */
    wc_curve25519_free(NULL);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve25519_init and wc_curve_25519_free*/
/*
 * Testing test_wc_curve25519_size.
 */
static int test_wc_curve25519_size(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519)
    curve25519_key key;

    ExpectIntEQ(wc_curve25519_init(&key), 0);

    /* Test good args for wc_curve25519_size */
    ExpectIntEQ(wc_curve25519_size(&key), CURVE25519_KEYSIZE);
    /* Test bad args for wc_curve25519_size */
    ExpectIntEQ(wc_curve25519_size(NULL), 0);

    wc_curve25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve25519_size*/

/*
 * Testing test_wc_curve25519_export_key_raw().
 */
static int test_wc_curve25519_export_key_raw(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519) && defined(HAVE_CURVE25519_KEY_EXPORT)
    curve25519_key key;
    WC_RNG         rng;
    byte           privateKey[CURVE25519_KEYSIZE];
    byte           publicKey[CURVE25519_KEYSIZE];
    word32         prvkSz;
    word32         pubkSz;
    byte           prik[CURVE25519_KEYSIZE];
    byte           pubk[CURVE25519_KEYSIZE];
    word32         prksz;
    word32         pbksz;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key), 0);

    /* bad-argument-test cases - target function should return BAD_FUNC_ARG */
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw(NULL, privateKey, &prvkSz,
        publicKey, &pubkSz), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw(&key, NULL, &prvkSz, publicKey,
        &pubkSz), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw(&key, privateKey, NULL,
        publicKey, &pubkSz), BAD_FUNC_ARG);
    /* prvkSz = CURVE25519_KEYSIZE; */
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw(&key, privateKey, &prvkSz,
        NULL, &pubkSz), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw(&key, privateKey, &prvkSz,
        publicKey, NULL), BAD_FUNC_ARG);

    /* cross-testing */
    prksz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_private_raw(&key, prik, &prksz), 0);
    pbksz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_public(&key, pubk, &pbksz), 0);
    prvkSz = CURVE25519_KEYSIZE;
    /* pubkSz = CURVE25519_KEYSIZE; */
    ExpectIntEQ(wc_curve25519_export_key_raw(&key, privateKey, &prvkSz,
        publicKey,  &pubkSz), 0);
    ExpectIntEQ(prksz, CURVE25519_KEYSIZE);
    ExpectIntEQ(pbksz, CURVE25519_KEYSIZE);
    ExpectIntEQ(prvkSz, CURVE25519_KEYSIZE);
    ExpectIntEQ(pubkSz, CURVE25519_KEYSIZE);
    ExpectIntEQ(XMEMCMP(privateKey, prik, CURVE25519_KEYSIZE), 0);
    ExpectIntEQ(XMEMCMP(publicKey,  pubk, CURVE25519_KEYSIZE), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* end of test_wc_curve25519_export_key_raw */

/*
 * Testing test_wc_curve25519_export_key_raw_ex().
 */
static int test_wc_curve25519_export_key_raw_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519) && defined(HAVE_CURVE25519_KEY_EXPORT)
    curve25519_key key;
    WC_RNG         rng;
    byte           privateKey[CURVE25519_KEYSIZE];
    byte           publicKey[CURVE25519_KEYSIZE];
    word32         prvkSz;
    word32         pubkSz;
    byte           prik[CURVE25519_KEYSIZE];
    byte           pubk[CURVE25519_KEYSIZE];
    word32         prksz;
    word32         pbksz;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key), 0);

    /* bad-argument-test cases - target function should return BAD_FUNC_ARG */
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(NULL, privateKey,
        &prvkSz, publicKey, &pubkSz, EC25519_LITTLE_ENDIAN), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, NULL,
        &prvkSz, publicKey, &pubkSz, EC25519_LITTLE_ENDIAN), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, privateKey,
        NULL, publicKey, &pubkSz, EC25519_LITTLE_ENDIAN), BAD_FUNC_ARG);
    /* prvkSz = CURVE25519_KEYSIZE; */
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, privateKey,
        &prvkSz, NULL, &pubkSz, EC25519_LITTLE_ENDIAN), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, privateKey,
        &prvkSz, publicKey, NULL, EC25519_LITTLE_ENDIAN), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    /* pubkSz = CURVE25519_KEYSIZE; */
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(NULL, privateKey,
        &prvkSz, publicKey, &pubkSz, EC25519_BIG_ENDIAN), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, NULL,
        &prvkSz, publicKey, &pubkSz, EC25519_BIG_ENDIAN), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, privateKey,
        NULL, publicKey, &pubkSz, EC25519_BIG_ENDIAN), BAD_FUNC_ARG);
    /* prvkSz = CURVE25519_KEYSIZE; */
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, privateKey,
        &prvkSz, NULL, &pubkSz, EC25519_BIG_ENDIAN), BAD_FUNC_ARG);
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, privateKey,
        &prvkSz, publicKey, NULL, EC25519_BIG_ENDIAN), BAD_FUNC_ARG);

    /* illegal value for endian */
    prvkSz = CURVE25519_KEYSIZE;
    /* pubkSz = CURVE25519_KEYSIZE; */
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, privateKey, &prvkSz,
        publicKey, NULL, EC25519_BIG_ENDIAN + 10), BAD_FUNC_ARG);

    /* cross-testing */
    prksz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_private_raw( &key, prik, &prksz), 0);
    pbksz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_public( &key, pubk, &pbksz), 0);
    prvkSz = CURVE25519_KEYSIZE;
    /* pubkSz = CURVE25519_KEYSIZE; */
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, privateKey, &prvkSz,
        publicKey, &pubkSz, EC25519_BIG_ENDIAN), 0);
    ExpectIntEQ(prksz, CURVE25519_KEYSIZE);
    ExpectIntEQ(pbksz, CURVE25519_KEYSIZE);
    ExpectIntEQ(prvkSz, CURVE25519_KEYSIZE);
    ExpectIntEQ(pubkSz, CURVE25519_KEYSIZE);
    ExpectIntEQ(XMEMCMP(privateKey, prik, CURVE25519_KEYSIZE), 0);
    ExpectIntEQ(XMEMCMP(publicKey,  pubk, CURVE25519_KEYSIZE), 0);
    ExpectIntEQ(wc_curve25519_export_key_raw_ex(&key, privateKey, &prvkSz,
        publicKey, &pubkSz, EC25519_LITTLE_ENDIAN), 0);
    ExpectIntEQ(prvkSz, CURVE25519_KEYSIZE);
    ExpectIntEQ(pubkSz, CURVE25519_KEYSIZE);

    /* try once with another endian */
    prvkSz = CURVE25519_KEYSIZE;
    pubkSz = CURVE25519_KEYSIZE;
    ExpectIntEQ(wc_curve25519_export_key_raw_ex( &key, privateKey, &prvkSz,
        publicKey, &pubkSz, EC25519_BIG_ENDIAN), 0);
    ExpectIntEQ(prvkSz, CURVE25519_KEYSIZE);
    ExpectIntEQ(pubkSz, CURVE25519_KEYSIZE);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* end of test_wc_curve25519_export_key_raw_ex */

/*
 * Testing wc_curve25519_make_key
 */
static int test_wc_curve25519_make_key(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519)
    curve25519_key key;
    WC_RNG         rng;
    int            keysize;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectIntEQ(wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key), 0);
    ExpectIntEQ(keysize = wc_curve25519_size(&key), CURVE25519_KEYSIZE);
    ExpectIntEQ(wc_curve25519_make_key(&rng, keysize, &key), 0);
    /* test bad cases*/
    ExpectIntEQ(wc_curve25519_make_key(NULL, 0, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_make_key(&rng, keysize, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_make_key(NULL, keysize, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_make_key(&rng, 0, &key), ECC_BAD_ARG_E);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve25519_make_key*/

/*
 * Testing wc_curve25519_shared_secret_ex
 */
static int test_wc_curve25519_shared_secret_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519)
    curve25519_key private_key;
    curve25519_key public_key;
    WC_RNG         rng;
    byte           out[CURVE25519_KEYSIZE];
    word32         outLen = sizeof(out);
    int            endian = EC25519_BIG_ENDIAN;

    ExpectIntEQ(wc_curve25519_init(&private_key), 0);
    ExpectIntEQ(wc_curve25519_init(&public_key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectIntEQ(wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &private_key),
        0);
    ExpectIntEQ(wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &public_key),
        0);

    ExpectIntEQ(wc_curve25519_shared_secret_ex(&private_key, &public_key, out,
        &outLen, endian), 0);

    /* test bad cases*/
    ExpectIntEQ(wc_curve25519_shared_secret_ex(NULL, NULL, NULL, 0, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_shared_secret_ex(NULL, &public_key, out, &outLen,
        endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_shared_secret_ex(&private_key, NULL, out, &outLen,
        endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_shared_secret_ex(&private_key, &public_key, NULL,
        &outLen, endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_shared_secret_ex(&private_key, &public_key, out,
        NULL, endian), BAD_FUNC_ARG);

    /* curve25519.c is checking for public_key size less than or equal to 0x7f,
     * increasing to 0x8f checks for error being returned*/
    public_key.p.point[CURVE25519_KEYSIZE-1] = 0x8F;
    ExpectIntEQ(wc_curve25519_shared_secret_ex(&private_key, &public_key, out,
        &outLen, endian), ECC_BAD_ARG_E);

    outLen = outLen - 2;
    ExpectIntEQ(wc_curve25519_shared_secret_ex(&private_key, &public_key, out,
        &outLen, endian), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve25519_free(&private_key);
    wc_curve25519_free(&public_key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve25519_shared_secret_ex*/

/*
 * Testing wc_curve25519_make_pub
 */
static int test_wc_curve25519_make_pub(void)
{
    EXPECT_DECLS;
#ifdef HAVE_CURVE25519
    curve25519_key key;
    WC_RNG         rng;
    byte           out[CURVE25519_KEYSIZE];

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key), 0);

    ExpectIntEQ(wc_curve25519_make_pub((int)sizeof(out), out,
        (int)sizeof(key.k), key.k), 0);
    /* test bad cases*/
    ExpectIntEQ(wc_curve25519_make_pub((int)sizeof(key.k) - 1, key.k,
        (int)sizeof out, out), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_curve25519_make_pub((int)sizeof out, out, (int)sizeof(key.k),
        NULL), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_curve25519_make_pub((int)sizeof out - 1, out,
        (int)sizeof(key.k), key.k), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_curve25519_make_pub((int)sizeof out, NULL,
        (int)sizeof(key.k), key.k), ECC_BAD_ARG_E);
    /* verify clamping test */
    key.k[0] |= ~248;
    ExpectIntEQ(wc_curve25519_make_pub((int)sizeof out, out, (int)sizeof(key.k),
        key.k), ECC_BAD_ARG_E);
    key.k[0] &= 248;
    /* repeat the expected-to-succeed test. */
    ExpectIntEQ(wc_curve25519_make_pub((int)sizeof out, out, (int)sizeof(key.k),
        key.k), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve25519_make_pub */

/*
 * Testing test_wc_curve25519_export_public_ex
 */
static int test_wc_curve25519_export_public_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519)
    curve25519_key key;
    WC_RNG         rng;
    byte           out[CURVE25519_KEYSIZE];
    word32         outLen = sizeof(out);
    int            endian = EC25519_BIG_ENDIAN;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key), 0);

    ExpectIntEQ(wc_curve25519_export_public(&key, out, &outLen), 0);
    ExpectIntEQ(wc_curve25519_export_public_ex(&key, out, &outLen, endian), 0);
    /* test bad cases*/
    ExpectIntEQ(wc_curve25519_export_public_ex(NULL, NULL, NULL, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_export_public_ex(NULL, out, &outLen, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_export_public_ex(&key, NULL, &outLen, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_export_public_ex(&key, out, NULL, endian),
        BAD_FUNC_ARG);
    outLen = outLen - 2;
    ExpectIntEQ(wc_curve25519_export_public_ex(&key, out, &outLen, endian),
        ECC_BAD_ARG_E);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve25519_export_public_ex*/
/*
 * Testing test_wc_curve25519_import_private_raw_ex
 */
static int test_wc_curve25519_import_private_raw_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519)
    curve25519_key key;
    WC_RNG         rng;
    byte           priv[CURVE25519_KEYSIZE];
    byte           pub[CURVE25519_KEYSIZE];
    word32         privSz = sizeof(priv);
    word32         pubSz = sizeof(pub);
    int            endian = EC25519_BIG_ENDIAN;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key), 0);

    ExpectIntEQ(wc_curve25519_export_private_raw_ex(&key, priv, &privSz,
        endian), 0);
    ExpectIntEQ(wc_curve25519_export_public(&key, pub, &pubSz), 0);
    ExpectIntEQ(wc_curve25519_import_private_raw_ex(priv, privSz, pub, pubSz,
        &key, endian), 0);
    /* test bad cases*/
    ExpectIntEQ(wc_curve25519_import_private_raw_ex(NULL, 0, NULL, 0, NULL,
        endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_import_private_raw_ex(NULL, privSz, pub, pubSz,
        &key, endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_import_private_raw_ex(priv, privSz, NULL, pubSz,
        &key, endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_import_private_raw_ex(priv, privSz, pub, pubSz,
        NULL, endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_import_private_raw_ex(priv, 0, pub, pubSz,
        &key, endian), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_curve25519_import_private_raw_ex(priv, privSz, pub, 0,
        &key, endian), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_curve25519_import_private_raw_ex(priv, privSz, pub, pubSz,
        &key, EC25519_LITTLE_ENDIAN), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve25519_import_private_raw_ex*/

/*
 * Testing test_wc_curve25519_import_private
 */
static int test_wc_curve25519_import_private(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519)
    curve25519_key key;
    WC_RNG         rng;
    byte           priv[CURVE25519_KEYSIZE];
    word32         privSz = sizeof(priv);

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve25519_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key), 0);

    ExpectIntEQ(wc_curve25519_export_private_raw(&key, priv, &privSz), 0);
    ExpectIntEQ(wc_curve25519_import_private(priv, privSz, &key), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve25519_import*/

/*
 * Testing test_wc_curve25519_export_private_raw_ex
 */
static int test_wc_curve25519_export_private_raw_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519)
    curve25519_key key;
    byte           out[CURVE25519_KEYSIZE];
    word32         outLen = sizeof(out);
    int            endian = EC25519_BIG_ENDIAN;

    ExpectIntEQ(wc_curve25519_init(&key), 0);

    ExpectIntEQ(wc_curve25519_export_private_raw_ex(&key, out, &outLen, endian),
        0);
    /* test bad cases*/
    ExpectIntEQ(wc_curve25519_export_private_raw_ex(NULL, NULL, NULL, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_export_private_raw_ex(NULL, out, &outLen, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_export_private_raw_ex(&key, NULL, &outLen,
        endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_export_private_raw_ex(&key, out, NULL, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve25519_export_private_raw_ex(&key, out, &outLen,
        EC25519_LITTLE_ENDIAN), 0);
    outLen = outLen - 2;
    ExpectIntEQ(wc_curve25519_export_private_raw_ex(&key, out, &outLen, endian),
        ECC_BAD_ARG_E);

    wc_curve25519_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve25519_export_private_raw_ex*/

/*
 * Testing wc_ed448_make_key().
 */
static int test_wc_ed448_make_key(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448)
    ed448_key     key;
    WC_RNG        rng;
    unsigned char pubkey[ED448_PUB_KEY_SIZE];

    XMEMSET(&key, 0, sizeof(ed448_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectIntEQ(wc_ed448_make_public(&key, pubkey, sizeof(pubkey)),
        ECC_PRIV_KEY_E);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_make_key(NULL, ED448_KEY_SIZE, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE - 1, &key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE + 1, &key),
        BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed448_make_key */


/*
 * Testing wc_ed448_init()
 */
static int test_wc_ed448_init(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448)
    ed448_key key;

    XMEMSET(&key, 0, sizeof(ed448_key));

    ExpectIntEQ(wc_ed448_init(&key), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_init(NULL), BAD_FUNC_ARG);

    wc_ed448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed448_init */

/*
 * Test wc_ed448_sign_msg() and wc_ed448_verify_msg()
 */
static int test_wc_ed448_sign_msg(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_SIGN)
    ed448_key key;
    WC_RNG    rng;
    byte      msg[] = "Everybody gets Friday off.\n";
    byte      sig[ED448_SIG_SIZE];
    word32    msglen = sizeof(msg);
    word32    siglen = sizeof(sig);
    word32    badSigLen = sizeof(sig) - 1;
#ifdef HAVE_ED448_VERIFY
    int       verify_ok = 0; /*1 = Verify success.*/
#endif

    /* Initialize stack variables. */
    XMEMSET(&key, 0, sizeof(ed448_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(sig, 0, siglen);

    /* Initialize key. */
    ExpectIntEQ(wc_ed448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_ed448_sign_msg(msg, msglen, sig, &siglen, &key, NULL, 0), 0);
    ExpectIntEQ(siglen, ED448_SIG_SIZE);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_sign_msg(NULL, msglen, sig, &siglen, &key, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_sign_msg(msg, msglen, NULL, &siglen, &key, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_sign_msg(msg, msglen, sig, NULL, &key, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_sign_msg(msg, msglen, sig, &siglen, NULL, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_sign_msg(msg, msglen, sig, &badSigLen, &key, NULL, 0),
        BUFFER_E);
    ExpectIntEQ(badSigLen, ED448_SIG_SIZE);
    badSigLen -= 1;

#ifdef HAVE_ED448_VERIFY
    ExpectIntEQ(wc_ed448_verify_msg(sig, siglen, msg, msglen, &verify_ok, &key,
        NULL, 0), 0);
    ExpectIntEQ(verify_ok, 1);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_verify_msg(sig, siglen - 1, msg, msglen, &verify_ok,
        &key, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_verify_msg(sig, siglen + 1, msg, msglen, &verify_ok,
        &key, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_verify_msg(NULL, siglen, msg, msglen, &verify_ok,
        &key, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_verify_msg(sig, siglen, NULL, msglen, &verify_ok,
        &key, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_verify_msg(sig, siglen, msg, msglen, NULL,
        &key, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_verify_msg(sig, siglen, msg, msglen, &verify_ok,
        NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_verify_msg(sig, badSigLen, msg, msglen, &verify_ok,
        &key, NULL, 0), BAD_FUNC_ARG);
#endif /* Verify. */

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed448_sign_msg */

/*
 * Testing wc_ed448_import_public()
 */
static int test_wc_ed448_import_public(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT)
    ed448_key  pubKey;
    WC_RNG     rng;
    const byte in[] =
                    "Ed448PublicKeyUnitTest.................................\n";
    word32     inlen = sizeof(in);

    XMEMSET(&pubKey, 0, sizeof(ed448_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed448_init(&pubKey), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &pubKey), 0);

    ExpectIntEQ(wc_ed448_import_public_ex(in, inlen, &pubKey, 1), 0);
    ExpectIntEQ(XMEMCMP(in, pubKey.p, inlen), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_import_public(NULL, inlen, &pubKey), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_import_public(in, inlen, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_import_public(in, inlen - 1, &pubKey), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed448_free(&pubKey);
#endif
    return EXPECT_RESULT();
} /* END wc_ed448_import_public */

/*
 * Testing wc_ed448_import_private_key()
 */
static int test_wc_ed448_import_private_key(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT)
    ed448_key  key;
    WC_RNG     rng;
    const byte privKey[] =
        "Ed448PrivateKeyUnitTest................................\n";
    const byte pubKey[] =
        "Ed448PublicKeyUnitTest.................................\n";
    word32     privKeySz = sizeof(privKey);
    word32     pubKeySz = sizeof(pubKey);
#ifdef HAVE_ED448_KEY_EXPORT
    byte       bothKeys[sizeof(privKey) + sizeof(pubKey)];
    word32     bothKeysSz = sizeof(bothKeys);
#endif

    XMEMSET(&key, 0, sizeof(ed448_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_ed448_import_private_key_ex(privKey, privKeySz, pubKey,
        pubKeySz, &key, 1), 0);
    ExpectIntEQ(XMEMCMP(pubKey, key.p, privKeySz), 0);
    ExpectIntEQ(XMEMCMP(privKey, key.k, pubKeySz), 0);

#ifdef HAVE_ED448_KEY_EXPORT
    ExpectIntEQ(wc_ed448_export_private(&key, bothKeys, &bothKeysSz), 0);
    ExpectIntEQ(wc_ed448_import_private_key_ex(bothKeys, bothKeysSz, NULL, 0,
        &key, 1), 0);
    ExpectIntEQ(XMEMCMP(pubKey, key.p, privKeySz), 0);
    ExpectIntEQ(XMEMCMP(privKey, key.k, pubKeySz), 0);
#endif

    /* Test bad args. */
    ExpectIntEQ(wc_ed448_import_private_key(NULL, privKeySz, pubKey, pubKeySz,
        &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_import_private_key(privKey, privKeySz, NULL, pubKeySz,
        &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_import_private_key(privKey, privKeySz, pubKey,
        pubKeySz, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_import_private_key(privKey, privKeySz - 1, pubKey,
        pubKeySz, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_import_private_key(privKey, privKeySz, pubKey,
        pubKeySz - 1, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_import_private_key(privKey, privKeySz, NULL, 0, &key),
        BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed448_import_private_key */

/*
 * Testing wc_ed448_export_public() and wc_ed448_export_private_only()
 */
static int test_wc_ed448_export(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key key;
    WC_RNG    rng;
    byte      priv[ED448_PRV_KEY_SIZE];
    byte      pub[ED448_PUB_KEY_SIZE];
    word32    privSz = sizeof(priv);
    word32    pubSz = sizeof(pub);

    XMEMSET(&key, 0, sizeof(ed448_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_ed448_export_public(&key, pub, &pubSz), 0);
    ExpectIntEQ(pubSz, ED448_KEY_SIZE);
    ExpectIntEQ(XMEMCMP(key.p, pub, pubSz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_export_public(NULL, pub, &pubSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_public(&key, NULL, &pubSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_public(&key, pub, NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed448_export_private_only(&key, priv, &privSz), 0);
    ExpectIntEQ(privSz, ED448_KEY_SIZE);
    ExpectIntEQ(XMEMCMP(key.k, priv, privSz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_export_private_only(NULL, priv, &privSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_private_only(&key, NULL, &privSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_private_only(&key, priv, NULL), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed448_export */

/*
 *  Testing wc_ed448_size()
 */
static int test_wc_ed448_size(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448)
    ed448_key key;
    WC_RNG    rng;

    XMEMSET(&key, 0, sizeof(ed448_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_ed448_size(&key), ED448_KEY_SIZE);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_size(NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed448_sig_size(&key), ED448_SIG_SIZE);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_sig_size(NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed448_pub_size(&key), ED448_PUB_KEY_SIZE);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_pub_size(NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed448_priv_size(&key), ED448_PRV_KEY_SIZE);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_priv_size(NULL), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed448_size */

/*
 * Testing wc_ed448_export_private() and wc_ed448_export_key()
 */
static int test_wc_ed448_exportKey(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key key;
    WC_RNG    rng;
    byte      priv[ED448_PRV_KEY_SIZE];
    byte      pub[ED448_PUB_KEY_SIZE];
    byte      privOnly[ED448_PRV_KEY_SIZE];
    word32    privSz      = sizeof(priv);
    word32    pubSz       = sizeof(pub);
    word32    privOnlySz  = sizeof(privOnly);

    XMEMSET(&key, 0, sizeof(ed448_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_ed448_export_private(&key, privOnly, &privOnlySz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_export_private(NULL, privOnly, &privOnlySz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_private(&key, NULL, &privOnlySz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_private(&key, privOnly, NULL), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed448_export_key(&key, priv, &privSz, pub, &pubSz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ed448_export_key(NULL, priv, &privSz, pub, &pubSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_key(&key, NULL, &privSz, pub, &pubSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_key(&key, priv, NULL, pub, &pubSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_key(&key, priv, &privSz, NULL, &pubSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ed448_export_key(&key, priv, &privSz, pub, NULL),
        BAD_FUNC_ARG);

    /* Cross check output. */
    ExpectIntEQ(XMEMCMP(priv, privOnly, privSz), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ed448_exportKey */

/*
 * Testing wc_Ed448PublicKeyToDer
 */
static int test_wc_Ed448PublicKeyToDer(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    ed448_key key;
    byte      derBuf[1024];

    XMEMSET(&key, 0, sizeof(ed448_key));

    /* Test bad args */
    ExpectIntEQ(wc_Ed448PublicKeyToDer(NULL, NULL, 0, 0), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ed448_init(&key), 0);
    ExpectIntEQ(wc_Ed448PublicKeyToDer(&key, derBuf, 0, 0), BUFFER_E);
    wc_ed448_free(&key);

    /*  Test good args */
    if (EXPECT_SUCCESS()) {
        WC_RNG rng;

        XMEMSET(&rng, 0, sizeof(WC_RNG));

        ExpectIntEQ(wc_ed448_init(&key), 0);
        ExpectIntEQ(wc_InitRng(&rng), 0);
        ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key), 0);

        ExpectIntGT(wc_Ed448PublicKeyToDer(&key, derBuf, 1024, 1), 0);

        DoExpectIntEQ(wc_FreeRng(&rng), 0);
        wc_ed448_free(&key);
    }
#endif
    return EXPECT_RESULT();
} /* END testing wc_Ed448PublicKeyToDer */

/*
 * Testing wc_curve448_init and wc_curve448_free.
 */
static int test_wc_curve448_init(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448)
    curve448_key key;

    /* Test bad args for wc_curve448_init */
    ExpectIntEQ(wc_curve448_init(&key), 0);
    /* Test bad args for wc_curve448_init */
    ExpectIntEQ(wc_curve448_init(NULL), BAD_FUNC_ARG);

    /* Test good args for wc_curve_448_free */
    wc_curve448_free(&key);
    /* Test bad args for wc_curve448_free */
    wc_curve448_free(NULL);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve448_init and wc_curve_448_free*/

/*
 * Testing wc_curve448_make_key
 */
static int test_wc_curve448_make_key(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448)
    curve448_key key;
    WC_RNG       rng;
    int          keysize;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectIntEQ(wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key), 0);
    ExpectIntEQ(keysize = wc_curve448_size(&key), CURVE448_KEY_SIZE);
    ExpectIntEQ(wc_curve448_make_key(&rng, keysize, &key), 0);

    /* test bad cases */
    ExpectIntEQ(wc_curve448_make_key(NULL, 0, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_make_key(&rng, keysize, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_make_key(NULL, keysize, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_make_key(&rng, 0, &key), ECC_BAD_ARG_E);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve448_make_key*/
/*
 * Testing test_wc_curve448_shared_secret_ex
 */
static int test_wc_curve448_shared_secret_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448)
    curve448_key private_key;
    curve448_key public_key;
    WC_RNG       rng;
    byte         out[CURVE448_KEY_SIZE];
    word32       outLen = sizeof(out);
    int          endian = EC448_BIG_ENDIAN;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve448_init(&private_key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &private_key), 0);

    ExpectIntEQ(wc_curve448_init(&public_key), 0);
    ExpectIntEQ(wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &public_key), 0);
    ExpectIntEQ(wc_curve448_shared_secret_ex(&private_key, &public_key, out,
        &outLen, endian), 0);

    /* test bad cases */
    ExpectIntEQ(wc_curve448_shared_secret_ex(NULL, NULL, NULL, 0, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_shared_secret_ex(NULL, &public_key, out, &outLen,
        endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_shared_secret_ex(&private_key, NULL, out, &outLen,
        endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_shared_secret_ex(&private_key, &public_key, NULL,
        &outLen, endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_shared_secret_ex(&private_key, &public_key, out,
        NULL, endian), BAD_FUNC_ARG);
    outLen = outLen - 2;
    ExpectIntEQ(wc_curve448_shared_secret_ex(&private_key, &public_key, out,
        &outLen, endian), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve448_free(&private_key);
    wc_curve448_free(&public_key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve448_shared_secret_ex*/

/*
 * Testing test_wc_curve448_export_public_ex
 */
static int test_wc_curve448_export_public_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448)
    WC_RNG        rng;
    curve448_key  key;
    byte          out[CURVE448_KEY_SIZE];
    word32        outLen = sizeof(out);
    int           endian = EC448_BIG_ENDIAN;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_curve448_export_public(&key, out, &outLen), 0);
    ExpectIntEQ(wc_curve448_export_public_ex(&key, out, &outLen, endian), 0);
    /* test bad cases*/
    ExpectIntEQ(wc_curve448_export_public_ex(NULL, NULL, NULL, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_export_public_ex(NULL, out, &outLen, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_export_public_ex(&key, NULL, &outLen, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_export_public_ex(&key, out, NULL, endian),
        BAD_FUNC_ARG);
    outLen = outLen - 2;
    ExpectIntEQ(wc_curve448_export_public_ex(&key, out, &outLen, endian),
        ECC_BAD_ARG_E);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve448_export_public_ex*/

/*
 * Testing test_wc_curve448_export_private_raw_ex
 */
static int test_wc_curve448_export_private_raw_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448)
    curve448_key key;
    byte         out[CURVE448_KEY_SIZE];
    word32       outLen = sizeof(out);
    int          endian = EC448_BIG_ENDIAN;

    ExpectIntEQ(wc_curve448_init(&key), 0);
    ExpectIntEQ(wc_curve448_export_private_raw_ex(&key, out, &outLen, endian),
        0);
    /* test bad cases*/
    ExpectIntEQ(wc_curve448_export_private_raw_ex(NULL, NULL, NULL, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_export_private_raw_ex(NULL, out, &outLen, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_export_private_raw_ex(&key, NULL, &outLen, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_export_private_raw_ex(&key, out, NULL, endian),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_export_private_raw_ex(&key, out, &outLen,
        EC448_LITTLE_ENDIAN), 0);
    outLen = outLen - 2;
    ExpectIntEQ(wc_curve448_export_private_raw_ex(&key, out, &outLen, endian),
        ECC_BAD_ARG_E);

    wc_curve448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve448_export_private_raw_ex*/

/*
 * Testing test_wc_curve448_import_private_raw_ex
 */
static int test_wc_curve448_import_private_raw_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448)
    curve448_key key;
    WC_RNG       rng;
    byte         priv[CURVE448_KEY_SIZE];
    byte         pub[CURVE448_KEY_SIZE];
    word32       privSz = sizeof(priv);
    word32       pubSz = sizeof(pub);
    int          endian = EC448_BIG_ENDIAN;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_curve448_export_private_raw(&key, priv, &privSz), 0);
    ExpectIntEQ(wc_curve448_export_public(&key, pub, &pubSz), 0);
    ExpectIntEQ(wc_curve448_import_private_raw_ex(priv, privSz, pub, pubSz,
        &key, endian), 0);
    /* test bad cases */
    ExpectIntEQ(wc_curve448_import_private_raw_ex(NULL, 0, NULL, 0, NULL, 0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_import_private_raw_ex(NULL, privSz, pub, pubSz,
        &key, endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_import_private_raw_ex(priv, privSz, NULL, pubSz,
        &key, endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_import_private_raw_ex(priv, privSz, pub, pubSz,
        NULL, endian), BAD_FUNC_ARG);
    ExpectIntEQ(wc_curve448_import_private_raw_ex(priv, 0, pub, pubSz,
        &key, endian), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_curve448_import_private_raw_ex(priv, privSz, pub, 0,
        &key, endian), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_curve448_import_private_raw_ex(priv, privSz, pub, pubSz,
        &key, EC448_LITTLE_ENDIAN), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve448_import_private_raw_ex*/
/*
 * Testing test_curve448_export_key_raw
 */
static int test_wc_curve448_export_key_raw(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448)
    curve448_key key;
    WC_RNG       rng;
    byte         priv[CURVE448_KEY_SIZE];
    byte         pub[CURVE448_KEY_SIZE];
    word32       privSz = sizeof(priv);
    word32       pubSz = sizeof(pub);

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_curve448_export_private_raw(&key, priv, &privSz), 0);
    ExpectIntEQ(wc_curve448_export_public(&key, pub, &pubSz), 0);
    ExpectIntEQ(wc_curve448_export_key_raw(&key, priv, &privSz, pub, &pubSz),
        0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve448_import_private_raw_ex*/

/*
 * Testing test_wc_curve448_import_private
 */
static int test_wc_curve448_import_private(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448)
    curve448_key key;
    WC_RNG       rng;
    byte         priv[CURVE448_KEY_SIZE];
    word32       privSz = sizeof(priv);

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_curve448_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key), 0);

    ExpectIntEQ(wc_curve448_export_private_raw(&key, priv, &privSz), 0);
    ExpectIntEQ(wc_curve448_import_private(priv, privSz, &key), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_curve448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve448_import*/
/*
 * Testing test_wc_curve448_size.
 */
static int test_wc_curve448_size(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448)
    curve448_key key;

    ExpectIntEQ(wc_curve448_init(&key), 0);

    /*  Test good args for wc_curve448_size */
    ExpectIntEQ(wc_curve448_size(&key), CURVE448_KEY_SIZE);
    /* Test bad args for wc_curve448_size */
    ExpectIntEQ(wc_curve448_size(NULL), 0);

    wc_curve448_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_curve448_size*/

/*
 * Testing wc_ecc_make_key.
 */
static int test_wc_ecc_make_key(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    int     ret;

    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    /* Pass in bad args. */
    ExpectIntEQ(wc_ecc_make_key(NULL, KEY14, &key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_make_key(&rng, KEY14, NULL), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_make_key */


/*
 * Testing wc_ecc_init()
 */
static int test_wc_ecc_init(void)
{
    EXPECT_DECLS;
#ifdef HAVE_ECC
    ecc_key key;

    XMEMSET(&key, 0, sizeof(ecc_key));

    ExpectIntEQ(wc_ecc_init(&key), 0);
    /* Pass in bad args. */
    ExpectIntEQ(wc_ecc_init(NULL), BAD_FUNC_ARG);

    wc_ecc_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_init */

/*
 * Testing wc_ecc_check_key()
 */
static int test_wc_ecc_check_key(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    int     ret;

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_ecc_check_key(&key), 0);

    /* Pass in bad args. */
    ExpectIntEQ(wc_ecc_check_key(NULL), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_check_key */

/*
 * Testing wc_ecc_get_generator()
 */
static int test_wc_ecc_get_generator(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG) && !defined(HAVE_SELFTEST) && \
    !defined(HAVE_FIPS) && defined(OPENSSL_EXTRA)
    ecc_point* pt = NULL;

    ExpectNotNull(pt = wc_ecc_new_point());

    ExpectIntEQ(wc_ecc_get_generator(pt, wc_ecc_get_curve_idx(ECC_SECP256R1)),
        MP_OKAY);

    /* Test bad args. */
    /* Returns Zero for bad arg. */
    ExpectIntNE(wc_ecc_get_generator(pt, -1), MP_OKAY);
    ExpectIntNE(wc_ecc_get_generator(NULL, wc_ecc_get_curve_idx(ECC_SECP256R1)),
        MP_OKAY);
    /* If we ever get to 1000 curves increase this number */
    ExpectIntNE(wc_ecc_get_generator(pt, 1000), MP_OKAY);
    ExpectIntNE(wc_ecc_get_generator(NULL, -1), MP_OKAY);

    wc_ecc_del_point(pt);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_get_generator */

/*
 * Testing wc_ecc_size()
 */
static int test_wc_ecc_size(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    WC_RNG      rng;
    ecc_key     key;
    int         ret;

    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_ecc_size(&key), KEY14);
    /* Test bad args. */
    /* Returns Zero for bad arg. */
    ExpectIntEQ(wc_ecc_size(NULL), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_size */

static int test_wc_ecc_params(void)
{
    EXPECT_DECLS;
    /* FIPS/CAVP self-test modules do not have `wc_ecc_get_curve_params`.
        It was added after certifications */
#if defined(HAVE_ECC) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    const ecc_set_type* ecc_set;
#if !defined(NO_ECC256) && !defined(NO_ECC_SECP)
    /* Test for SECP256R1 curve */
    int curve_id = ECC_SECP256R1;
    int curve_idx;

    ExpectIntNE(curve_idx = wc_ecc_get_curve_idx(curve_id), ECC_CURVE_INVALID);
    ExpectNotNull(ecc_set = wc_ecc_get_curve_params(curve_idx));
    ExpectIntEQ(ecc_set->id, curve_id);
#endif
    /* Test case when SECP256R1 is not enabled */
    /* Test that we get curve params for index 0 */
    ExpectNotNull(ecc_set = wc_ecc_get_curve_params(0));
#endif /* HAVE_ECC && !HAVE_FIPS && !HAVE_SELFTEST */
    return EXPECT_RESULT();
}

/*
 * Testing wc_ecc_sign_hash() and wc_ecc_verify_hash()
 */
static int test_wc_ecc_signVerify_hash(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_SIGN) && !defined(NO_ASN) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    int     ret;
#ifdef HAVE_ECC_VERIFY
    int     verify = 0;
#endif
    word32  siglen = ECC_BUFSIZE;
    byte    sig[ECC_BUFSIZE];
    byte    adjustedSig[ECC_BUFSIZE+1];
    byte    digest[] = TEST_STRING;
    word32  digestlen = (word32)TEST_STRING_SZ;

    /* Init stack var */
    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(sig, 0, siglen);
    XMEMSET(adjustedSig, 0, ECC_BUFSIZE+1);

    /* Init structs. */
    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_ecc_sign_hash(digest, digestlen, sig, &siglen, &rng, &key),
        0);

    /* Check bad args. */
    ExpectIntEQ(wc_ecc_sign_hash(NULL, digestlen, sig, &siglen, &rng, &key),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sign_hash(digest, digestlen, NULL, &siglen, &rng, &key),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sign_hash(digest, digestlen, sig, NULL, &rng, &key),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sign_hash(digest, digestlen, sig, &siglen, NULL, &key),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sign_hash(digest, digestlen, sig, &siglen, &rng, NULL),
        ECC_BAD_ARG_E);

#ifdef HAVE_ECC_VERIFY
    ExpectIntEQ(wc_ecc_verify_hash(sig, siglen, digest, digestlen, &verify,
        &key), 0);
    ExpectIntEQ(verify, 1);

    /* test check on length of signature passed in */
    XMEMCPY(adjustedSig, sig, siglen);
    adjustedSig[1] = adjustedSig[1] + 1; /* add 1 to length for extra byte*/
#ifndef NO_STRICT_ECDSA_LEN
    ExpectIntNE(wc_ecc_verify_hash(adjustedSig, siglen+1, digest, digestlen,
        &verify, &key), 0);
#else
    /* if NO_STRICT_ECDSA_LEN is set then extra bytes after the signature
     * is allowed */
    ExpectIntEQ(wc_ecc_verify_hash(adjustedSig, siglen+1, digest, digestlen,
        &verify, &key), 0);
#endif

    /* Test bad args. */
    ExpectIntEQ(wc_ecc_verify_hash(NULL, siglen, digest, digestlen, &verify,
        &key), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_verify_hash(sig, siglen, NULL, digestlen, &verify, &key),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_verify_hash(sig, siglen, digest, digestlen, NULL, &key),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_verify_hash(sig, siglen, digest, digestlen, &verify,
        NULL), ECC_BAD_ARG_E);
#endif /* HAVE_ECC_VERIFY */

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif
    return EXPECT_RESULT();
} /*  END test_wc_ecc_sign_hash */


/*
 * Testing wc_ecc_shared_secret()
 */
static int test_wc_ecc_shared_secret(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_DHE) && !defined(WC_NO_RNG)
    ecc_key     key;
    ecc_key     pubKey;
    WC_RNG      rng;
#if defined(NO_ECC256)
    int         ret;
#endif
    byte        out[KEY32];
    int         keySz = sizeof(out);
    word32      outlen = (word32)sizeof(out);

#if defined(HAVE_ECC) && !defined(NO_ECC256)
    const char* qx =
        "bb33ac4c27504ac64aa504c33cde9f36db722dce94ea2bfacb2009392c16e861";
    const char* qy =
        "02e9af4dd302939a315b9792217ff0cf18da9111023486e82058330b803489d8";
    const char* d  =
        "45b66902739c6c85a1385b72e8e8c7acc4038d533504fa6c28dc348de1a8098c";
    const char* curveName = "SECP256R1";
    const byte expected_shared_secret[] =
        {
            0x65, 0xc0, 0xd4, 0x61, 0x17, 0xe6, 0x09, 0x75,
            0xf0, 0x12, 0xa0, 0x4d, 0x0b, 0x41, 0x30, 0x7a,
            0x51, 0xf0, 0xb3, 0xaf, 0x23, 0x8f, 0x0f, 0xdf,
            0xf1, 0xff, 0x23, 0x64, 0x28, 0xca, 0xf8, 0x06
        };
#endif

    PRIVATE_KEY_UNLOCK();

    /* Initialize variables. */
    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&pubKey, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(out, 0, keySz);

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_ecc_init(&pubKey), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);

#if !defined(NO_ECC256)
    ExpectIntEQ(wc_ecc_import_raw(&key, qx, qy, d, curveName), 0);
    ExpectIntEQ(wc_ecc_import_raw(&pubKey, qx, qy, NULL, curveName), 0);
#else
    ret = wc_ecc_make_key(&rng, keySz, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    ret = wc_ecc_make_key(&rng, keySz, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
#endif

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ExpectIntEQ(wc_ecc_set_rng(&key, &rng), 0);
#endif

    ExpectIntEQ(wc_ecc_shared_secret(&key, &pubKey, out, &outlen), 0);

#if !defined(NO_ECC256)
    ExpectIntEQ(XMEMCMP(out, expected_shared_secret, outlen), 0);
#endif

    /* Test bad args. */
    ExpectIntEQ(wc_ecc_shared_secret(NULL, &pubKey, out, &outlen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_shared_secret(&key, NULL, out, &outlen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_shared_secret(&key, &pubKey, NULL, &outlen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_shared_secret(&key, &pubKey, out, NULL),
        BAD_FUNC_ARG);
    /* Invalid length */
    outlen = 1;
    ExpectIntEQ(wc_ecc_shared_secret(&key, &pubKey, out, &outlen),
        BUFFER_E);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&pubKey);
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    PRIVATE_KEY_LOCK();
#endif
    return EXPECT_RESULT();
} /* END tests_wc_ecc_shared_secret */

/*
 * testint wc_ecc_export_x963()
 */
static int test_wc_ecc_export_x963(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    byte    out[ECC_ASN963_MAX_BUF_SZ];
    word32  outlen = sizeof(out);
    int     ret;

    PRIVATE_KEY_UNLOCK();

    /* Initialize variables. */
    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(out, 0, outlen);

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY20, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_ecc_export_x963(&key, out, &outlen), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_ecc_export_x963(NULL, out, &outlen), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_export_x963(&key, NULL, &outlen), LENGTH_ONLY_E);
    ExpectIntEQ(wc_ecc_export_x963(&key, out, NULL), ECC_BAD_ARG_E);
    key.idx = -4;
    ExpectIntEQ(wc_ecc_export_x963(&key, out, &outlen), ECC_BAD_ARG_E);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    PRIVATE_KEY_LOCK();
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_export_x963 */

/*
 * Testing wc_ecc_export_x963_ex()
 * compile with --enable-compkey will use compression.
 */
static int test_wc_ecc_export_x963_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    int     ret;
    byte    out[ECC_ASN963_MAX_BUF_SZ];
    word32  outlen = sizeof(out);
    #ifdef HAVE_COMP_KEY
        word32  badOutLen = 5;
    #endif

    /* Init stack variables. */
    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(out, 0, outlen);

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY64, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

#ifdef HAVE_COMP_KEY
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, out, &outlen, COMP), 0);
#else
    ExpectIntEQ(ret = wc_ecc_export_x963_ex(&key, out, &outlen, NOCOMP), 0);
#endif

    /* Test bad args. */
#ifdef HAVE_COMP_KEY
    ExpectIntEQ(wc_ecc_export_x963_ex(NULL, out, &outlen, COMP), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, NULL, &outlen, COMP), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, out, NULL, COMP), BAD_FUNC_ARG);
#if defined(HAVE_FIPS) && (!defined(FIPS_VERSION_LT) || FIPS_VERSION_LT(5,3))
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, out, &badOutLen, COMP), BUFFER_E);
#else
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, out, &badOutLen, COMP),
        LENGTH_ONLY_E);
#endif
    key.idx = -4;
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, out, &outlen, COMP), ECC_BAD_ARG_E);
#else
    ExpectIntEQ(wc_ecc_export_x963_ex(NULL, out, &outlen, NOCOMP),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, NULL, &outlen, NOCOMP),
        LENGTH_ONLY_E);
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, out, &outlen, 1), NOT_COMPILED_IN);
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, out, NULL, NOCOMP),
        ECC_BAD_ARG_E);
    key.idx = -4;
    ExpectIntEQ(wc_ecc_export_x963_ex(&key, out, &outlen, NOCOMP),
        ECC_BAD_ARG_E);
#endif

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_export_x963_ex */

/*
 * testing wc_ecc_import_x963()
 */
static int test_wc_ecc_import_x963(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_IMPORT) && \
    defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key pubKey;
    ecc_key key;
    WC_RNG  rng;
    byte    x963[ECC_ASN963_MAX_BUF_SZ];
    word32  x963Len = (word32)sizeof(x963);
    int     ret;

    /* Init stack variables. */
    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&pubKey, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(x963, 0, x963Len);

    ExpectIntEQ(wc_ecc_init(&pubKey), 0);
    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY24, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    PRIVATE_KEY_UNLOCK();
    ExpectIntEQ(wc_ecc_export_x963(&key, x963, &x963Len), 0);
    PRIVATE_KEY_LOCK();

    ExpectIntEQ(wc_ecc_import_x963(x963, x963Len, &pubKey), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_ecc_import_x963(NULL, x963Len, &pubKey), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_import_x963(x963, x963Len, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_import_x963(x963, x963Len + 1, &pubKey), ECC_BAD_ARG_E);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);
    wc_ecc_free(&pubKey);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif
    return EXPECT_RESULT();
} /* END wc_ecc_import_x963 */

/*
 * testing wc_ecc_import_private_key()
 */
static int test_wc_ecc_import_private_key(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_IMPORT) && \
    defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key key;
    ecc_key keyImp;
    WC_RNG  rng;
    byte    privKey[ECC_PRIV_KEY_BUF]; /* Raw private key.*/
    byte    x963Key[ECC_ASN963_MAX_BUF_SZ];
    word32  privKeySz = (word32)sizeof(privKey);
    word32  x963KeySz = (word32)sizeof(x963Key);
    int     ret;

    /* Init stack variables. */
    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&keyImp, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(privKey, 0, privKeySz);
    XMEMSET(x963Key, 0, x963KeySz);

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_ecc_init(&keyImp), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY48, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    PRIVATE_KEY_UNLOCK();
    ExpectIntEQ(wc_ecc_export_x963(&key, x963Key, &x963KeySz), 0);
    PRIVATE_KEY_LOCK();
    ExpectIntEQ(wc_ecc_export_private_only(&key, privKey, &privKeySz), 0);

    ExpectIntEQ(wc_ecc_import_private_key(privKey, privKeySz, x963Key,
        x963KeySz, &keyImp), 0);
    /* Pass in bad args. */
    ExpectIntEQ(wc_ecc_import_private_key(privKey, privKeySz, x963Key,
        x963KeySz, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_import_private_key(NULL, privKeySz, x963Key, x963KeySz,
        &keyImp), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&keyImp);
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_import_private_key */


/*
 * Testing wc_ecc_export_private_only()
 */
static int test_wc_ecc_export_private_only(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    byte    out[ECC_PRIV_KEY_BUF];
    word32  outlen = sizeof(out);
    int     ret;

    /* Init stack variables. */
    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(out, 0, outlen);

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY32, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_ecc_export_private_only(&key, out, &outlen), 0);
    /* Pass in bad args. */
    ExpectIntEQ(wc_ecc_export_private_only(NULL, out, &outlen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_export_private_only(&key, NULL, &outlen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_export_private_only(&key, out, NULL), BAD_FUNC_ARG);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_export_private_only */


/*
 * Testing wc_ecc_rs_to_sig()
 */
static int test_wc_ecc_rs_to_sig(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(NO_ASN)
    /* first [P-192,SHA-1] vector from FIPS 186-3 NIST vectors */
    const char* R = "6994d962bdd0d793ffddf855ec5bf2f91a9698b46258a63e";
    const char* S = "02ba6465a234903744ab02bc8521405b73cf5fc00e1a9f41";
    const char* zeroStr = "0";
    byte        sig[ECC_MAX_SIG_SIZE];
    word32      siglen = (word32)sizeof(sig);
    /* R and S max size is the order of curve. 2^192.*/
    int         keySz = KEY24;
    byte        r[KEY24];
    byte        s[KEY24];
    word32      rlen = (word32)sizeof(r);
    word32      slen = (word32)sizeof(s);

    /* Init stack variables. */
    XMEMSET(sig, 0, ECC_MAX_SIG_SIZE);
    XMEMSET(r, 0, keySz);
    XMEMSET(s, 0, keySz);

    ExpectIntEQ(wc_ecc_rs_to_sig(R, S, sig, &siglen), 0);
    ExpectIntEQ(wc_ecc_sig_to_rs(sig, siglen, r, &rlen, s, &slen), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_rs_to_sig(NULL, S, sig, &siglen), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_rs_to_sig(R, NULL, sig, &siglen), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_rs_to_sig(R, S, sig, NULL), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_rs_to_sig(R, S, NULL, &siglen), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_rs_to_sig(R, zeroStr, sig, &siglen), MP_ZERO_E);
    ExpectIntEQ(wc_ecc_rs_to_sig(zeroStr, S, sig, &siglen), MP_ZERO_E);
    ExpectIntEQ(wc_ecc_sig_to_rs(NULL, siglen, r, &rlen, s, &slen),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sig_to_rs(sig, siglen, NULL, &rlen, s, &slen),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sig_to_rs(sig, siglen, r, NULL, s, &slen),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sig_to_rs(sig, siglen, r, &rlen, NULL, &slen),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sig_to_rs(sig, siglen, r, &rlen, s, NULL),
        ECC_BAD_ARG_E);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_rs_to_sig */

static int test_wc_ecc_import_raw(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(NO_ECC256)
    ecc_key     key;
    const char* qx =
        "bb33ac4c27504ac64aa504c33cde9f36db722dce94ea2bfacb2009392c16e861";
    const char* qy =
        "02e9af4dd302939a315b9792217ff0cf18da9111023486e82058330b803489d8";
    const char* d  =
        "45b66902739c6c85a1385b72e8e8c7acc4038d533504fa6c28dc348de1a8098c";
    const char* curveName = "SECP256R1";
#ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    const char* kNullStr = "";
    int ret;
#endif

    XMEMSET(&key, 0, sizeof(ecc_key));

    ExpectIntEQ(wc_ecc_init(&key), 0);

    /* Test good import */
    ExpectIntEQ(wc_ecc_import_raw(&key, qx, qy, d, curveName), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_ecc_import_raw(NULL, qx, qy, d, curveName), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_import_raw(&key, NULL, qy, d, curveName), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_import_raw(&key, qx, NULL, d, curveName), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_import_raw(&key, qx, qy, d, NULL), BAD_FUNC_ARG);
#ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    #if !defined(USE_FAST_MATH) && !defined(WOLFSSL_SP_MATH)
        wc_ecc_free(&key);
    #endif
    ExpectIntLT(ret = wc_ecc_import_raw(&key, kNullStr, kNullStr, kNullStr,
        curveName), 0);
    ExpectTrue((ret == ECC_INF_E) || (ret == BAD_FUNC_ARG));
#endif
#if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    #if !defined(USE_FAST_MATH) && !defined(WOLFSSL_SP_MATH)
        wc_ecc_free(&key);
    #endif
#ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    ExpectIntLT(ret = wc_ecc_import_raw(&key, "0", qy, d, curveName), 0);
    ExpectTrue((ret == BAD_FUNC_ARG) || (ret == MP_VAL));
#else
    ExpectIntEQ(wc_ecc_import_raw(&key, "0", qy, d, curveName), 0);
#endif
    #if !defined(USE_FAST_MATH) && !defined(WOLFSSL_SP_MATH)
        wc_ecc_free(&key);
    #endif
#ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    ExpectIntLT(ret = wc_ecc_import_raw(&key, qx, "0", d, curveName), 0);
    ExpectTrue((ret == BAD_FUNC_ARG) || (ret == MP_VAL));
#else
    ExpectIntEQ(wc_ecc_import_raw(&key, qx, "0", d, curveName), 0);
#endif
    #if !defined(USE_FAST_MATH) && !defined(WOLFSSL_SP_MATH)
        wc_ecc_free(&key);
    #endif
    ExpectIntEQ(wc_ecc_import_raw(&key, "0", "0", d, curveName), ECC_INF_E);
#endif

    wc_ecc_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_import_raw */

static int test_wc_ecc_import_unsigned(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(NO_ECC256) && !defined(HAVE_SELFTEST) && \
    (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
     HAVE_FIPS_VERSION >= 2))
    ecc_key    key;
    const byte qx[] = {
        0xbb, 0x33, 0xac, 0x4c, 0x27, 0x50, 0x4a, 0xc6,
        0x4a, 0xa5, 0x04, 0xc3, 0x3c, 0xde, 0x9f, 0x36,
        0xdb, 0x72, 0x2d, 0xce, 0x94, 0xea, 0x2b, 0xfa,
        0xcb, 0x20, 0x09, 0x39, 0x2c, 0x16, 0xe8, 0x61
    };
    const byte qy[] = {
        0x02, 0xe9, 0xaf, 0x4d, 0xd3, 0x02, 0x93, 0x9a,
        0x31, 0x5b, 0x97, 0x92, 0x21, 0x7f, 0xf0, 0xcf,
        0x18, 0xda, 0x91, 0x11, 0x02, 0x34, 0x86, 0xe8,
        0x20, 0x58, 0x33, 0x0b, 0x80, 0x34, 0x89, 0xd8
    };
    const byte d[] = {
        0x45, 0xb6, 0x69, 0x02, 0x73, 0x9c, 0x6c, 0x85,
        0xa1, 0x38, 0x5b, 0x72, 0xe8, 0xe8, 0xc7, 0xac,
        0xc4, 0x03, 0x8d, 0x53, 0x35, 0x04, 0xfa, 0x6c,
        0x28, 0xdc, 0x34, 0x8d, 0xe1, 0xa8, 0x09, 0x8c
    };
#ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    const byte nullBytes[32] = {0};
    int ret;
#endif
    int        curveId = ECC_SECP256R1;

    XMEMSET(&key, 0, sizeof(ecc_key));

    ExpectIntEQ(wc_ecc_init(&key), 0);

    ExpectIntEQ(wc_ecc_import_unsigned(&key, (byte*)qx, (byte*)qy, (byte*)d,
        curveId), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_import_unsigned(NULL, (byte*)qx, (byte*)qy, (byte*)d,
        curveId), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_import_unsigned(&key, NULL, (byte*)qy, (byte*)d,
        curveId), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_import_unsigned(&key, (byte*)qx, NULL, (byte*)d,
        curveId), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_import_unsigned(&key, (byte*)qx, (byte*)qy, (byte*)d,
        ECC_CURVE_INVALID), BAD_FUNC_ARG);
#ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    ExpectIntLT(ret = wc_ecc_import_unsigned(&key, (byte*)nullBytes,
        (byte*)nullBytes, (byte*)nullBytes, curveId), 0);
    ExpectTrue((ret == ECC_INF_E) || (ret == BAD_FUNC_ARG));
#endif

    wc_ecc_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_import_unsigned */


/*
 * Testing wc_ecc_sig_size()
 */
static int test_wc_ecc_sig_size(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    int     keySz = KEY16;
    int     ret;

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, keySz, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntLE(wc_ecc_sig_size(&key),
         (2 * keySz + SIG_HEADER_SZ + ECC_MAX_PAD_SZ));

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_sig_size */

/*
 * Testing wc_ecc_ctx_new()
 */
static int test_wc_ecc_ctx_new(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG)
    WC_RNG    rng;
    ecEncCtx* cli = NULL;
    ecEncCtx* srv = NULL;

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectNotNull(cli = wc_ecc_ctx_new(REQ_RESP_CLIENT, &rng));
    ExpectNotNull(srv = wc_ecc_ctx_new(REQ_RESP_SERVER, &rng));
    wc_ecc_ctx_free(cli);
    cli = NULL;
    wc_ecc_ctx_free(srv);

    /* Test bad args. */
    /* wc_ecc_ctx_new_ex() will free if returned NULL. */
    ExpectNull(cli = wc_ecc_ctx_new(0, &rng));
    ExpectNull(cli = wc_ecc_ctx_new(REQ_RESP_CLIENT, NULL));

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_ctx_free(cli);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_ctx_new */

/*
 * Tesing wc_ecc_reset()
 */
static int test_wc_ecc_ctx_reset(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG)
    ecEncCtx* ctx = NULL;
    WC_RNG    rng;

    XMEMSET(&rng, 0, sizeof(rng));

    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectNotNull(ctx = wc_ecc_ctx_new(REQ_RESP_CLIENT, &rng));

    ExpectIntEQ(wc_ecc_ctx_reset(ctx, &rng), 0);

    /* Pass in bad args. */
    ExpectIntEQ(wc_ecc_ctx_reset(NULL, &rng), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_ctx_reset(ctx, NULL), BAD_FUNC_ARG);

    wc_ecc_ctx_free(ctx);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_ctx_reset */

/*
 * Testing wc_ecc_ctx_set_peer_salt() and wc_ecc_ctx_get_own_salt()
 */
static int test_wc_ecc_ctx_set_peer_salt(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG)
    WC_RNG      rng;
    ecEncCtx*   cliCtx      = NULL;
    ecEncCtx*   servCtx     = NULL;
    const byte* cliSalt     = NULL;
    const byte* servSalt    = NULL;

    XMEMSET(&rng, 0, sizeof(rng));

    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectNotNull(cliCtx = wc_ecc_ctx_new(REQ_RESP_CLIENT, &rng));
    ExpectNotNull(servCtx = wc_ecc_ctx_new(REQ_RESP_SERVER, &rng));

    /* Test bad args. */
    ExpectNull(cliSalt = wc_ecc_ctx_get_own_salt(NULL));

    ExpectNotNull(cliSalt = wc_ecc_ctx_get_own_salt(cliCtx));
    ExpectNotNull(servSalt = wc_ecc_ctx_get_own_salt(servCtx));

    ExpectIntEQ(wc_ecc_ctx_set_peer_salt(cliCtx, servSalt), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_ctx_set_peer_salt(NULL, servSalt), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_ctx_set_peer_salt(cliCtx, NULL), BAD_FUNC_ARG);

    wc_ecc_ctx_free(cliCtx);
    wc_ecc_ctx_free(servCtx);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();

} /* END test_wc_ecc_ctx_set_peer_salt */

/*
 * Testing wc_ecc_ctx_set_info()
 */
static int test_wc_ecc_ctx_set_info(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG)
    ecEncCtx*   ctx = NULL;
    WC_RNG      rng;
    const char* optInfo = "Optional Test Info.";
    int         optInfoSz = (int)XSTRLEN(optInfo);
    const char* badOptInfo = NULL;

    XMEMSET(&rng, 0, sizeof(rng));

    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectNotNull(ctx = wc_ecc_ctx_new(REQ_RESP_CLIENT, &rng));

    ExpectIntEQ(wc_ecc_ctx_set_info(ctx, (byte*)optInfo, optInfoSz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_ctx_set_info(NULL, (byte*)optInfo, optInfoSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_ctx_set_info(ctx, (byte*)badOptInfo, optInfoSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_ctx_set_info(ctx, (byte*)optInfo, -1), BAD_FUNC_ARG);

    wc_ecc_ctx_free(ctx);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_ctx_set_info */

/*
 * Testing wc_ecc_encrypt() and wc_ecc_decrypt()
 */
static int test_wc_ecc_encryptDecrypt(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG) && \
    defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    ecc_key     srvKey;
    ecc_key     cliKey;
    ecc_key     tmpKey;
    WC_RNG      rng;
    int         ret;
    const char* msg   = "EccBlock Size 16";
    word32      msgSz = (word32)XSTRLEN("EccBlock Size 16");
#ifdef WOLFSSL_ECIES_OLD
    byte        out[(sizeof("EccBlock Size 16") - 1) + WC_SHA256_DIGEST_SIZE];
#elif defined(WOLFSSL_ECIES_GEN_IV)
    byte        out[KEY20 * 2 + 1 + AES_BLOCK_SIZE +
                    (sizeof("EccBlock Size 16") - 1) + WC_SHA256_DIGEST_SIZE];
#else
    byte        out[KEY20 * 2 + 1 + (sizeof("EccBlock Size 16") - 1) +
                    WC_SHA256_DIGEST_SIZE];
#endif
    word32      outSz = (word32)sizeof(out);
    byte        plain[sizeof("EccBlock Size 16")];
    word32      plainSz = (word32)sizeof(plain);
    int         keySz = KEY20;

    /* Init stack variables. */
    XMEMSET(out, 0, outSz);
    XMEMSET(plain, 0, plainSz);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&srvKey, 0, sizeof(ecc_key));
    XMEMSET(&cliKey, 0, sizeof(ecc_key));
    XMEMSET(&tmpKey, 0, sizeof(ecc_key));

    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ecc_init(&cliKey), 0);
    ret = wc_ecc_make_key(&rng, keySz, &cliKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &cliKey.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_ecc_init(&srvKey), 0);
    ret = wc_ecc_make_key(&rng, keySz, &srvKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &srvKey.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_ecc_init(&tmpKey), 0);

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ExpectIntEQ(wc_ecc_set_rng(&srvKey, &rng), 0);
    ExpectIntEQ(wc_ecc_set_rng(&cliKey, &rng), 0);
#endif

    ExpectIntEQ(wc_ecc_encrypt(&cliKey, &srvKey, (byte*)msg, msgSz, out,
        &outSz, NULL), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_encrypt(NULL, &srvKey, (byte*)msg, msgSz, out, &outSz,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_encrypt(&cliKey, NULL, (byte*)msg, msgSz, out, &outSz,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_encrypt(&cliKey, &srvKey, NULL, msgSz, out, &outSz,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_encrypt(&cliKey, &srvKey, (byte*)msg, msgSz, NULL,
        &outSz, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_encrypt(&cliKey, &srvKey, (byte*)msg, msgSz, out, NULL,
        NULL), BAD_FUNC_ARG);

#ifdef WOLFSSL_ECIES_OLD
    tmpKey.dp = cliKey.dp;
    ExpectIntEQ(wc_ecc_copy_point(&cliKey.pubkey, &tmpKey.pubkey), 0);
#endif

    ExpectIntEQ(wc_ecc_decrypt(&srvKey, &tmpKey, out, outSz, plain, &plainSz,
         NULL), 0);
    ExpectIntEQ(wc_ecc_decrypt(NULL, &tmpKey, out, outSz, plain, &plainSz,
         NULL), BAD_FUNC_ARG);
#ifdef WOLFSSL_ECIES_OLD
    /* NULL parameter allowed in new implementations - public key comes from
     * the message. */
    ExpectIntEQ(wc_ecc_decrypt(&srvKey, NULL, out, outSz, plain, &plainSz,
        NULL), BAD_FUNC_ARG);
#endif
    ExpectIntEQ(wc_ecc_decrypt(&srvKey, &tmpKey, NULL, outSz, plain, &plainSz,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_decrypt(&srvKey, &tmpKey, out, outSz, NULL, &plainSz,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_decrypt(&srvKey, &tmpKey, out, outSz, plain, NULL, NULL),
        BAD_FUNC_ARG);

    ExpectIntEQ(XMEMCMP(msg, plain, msgSz), 0);

    wc_ecc_free(&tmpKey);
    wc_ecc_free(&srvKey);
    wc_ecc_free(&cliKey);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_encryptDecrypt */

/*
 * Testing wc_ecc_del_point() and wc_ecc_new_point()
 */
static int test_wc_ecc_del_point(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC)
    ecc_point* pt = NULL;

    ExpectNotNull(pt = wc_ecc_new_point());
    wc_ecc_del_point(pt);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_del_point */

/*
 * Testing wc_ecc_point_is_at_infinity(), wc_ecc_export_point_der(),
 * wc_ecc_import_point_der(), wc_ecc_copy_point(), wc_ecc_point_is_on_curve(),
 * and wc_ecc_cmp_point()
 */
static int test_wc_ecc_pointFns(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && \
    !defined(WC_NO_RNG) && !defined(WOLFSSL_ATECC508A) && \
    !defined(WOLFSSL_ATECC608A)
    ecc_key    key;
    WC_RNG     rng;
    int        ret;
    ecc_point* point = NULL;
    ecc_point* cpypt = NULL;
    int        idx = 0;
    int        keySz = KEY32;
    byte       der[DER_SZ(KEY32)];
    word32     derlenChk = 0;
    word32     derSz = DER_SZ(KEY32);

    /* Init stack variables. */
    XMEMSET(der, 0, derSz);
    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ecc_init(&key), 0);
    ret = wc_ecc_make_key(&rng, keySz, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectNotNull(point = wc_ecc_new_point());
    ExpectNotNull(cpypt = wc_ecc_new_point());

    /* Export */
    ExpectIntEQ(wc_ecc_export_point_der((idx = key.idx), &key.pubkey, NULL,
        &derlenChk), LENGTH_ONLY_E);
    /* Check length value. */
    ExpectIntEQ(derSz, derlenChk);
    ExpectIntEQ(wc_ecc_export_point_der((idx = key.idx), &key.pubkey, der,
        &derSz), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_export_point_der(-2, &key.pubkey, der, &derSz),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_export_point_der((idx = key.idx), NULL, der, &derSz),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_export_point_der((idx = key.idx), &key.pubkey, der,
        NULL), ECC_BAD_ARG_E);

    /* Import */
    ExpectIntEQ(wc_ecc_import_point_der(der, derSz, idx, point), 0);
    ExpectIntEQ(wc_ecc_cmp_point(&key.pubkey, point), 0);
    /* Test bad args. */
    ExpectIntEQ( wc_ecc_import_point_der(NULL, derSz, idx, point),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_import_point_der(der, derSz, idx, NULL), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_import_point_der(der, derSz, -1, point), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_import_point_der(der, derSz + 1, idx, point),
        ECC_BAD_ARG_E);

    /* Copy */
    ExpectIntEQ(wc_ecc_copy_point(point, cpypt), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_copy_point(NULL, cpypt), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_copy_point(point, NULL), ECC_BAD_ARG_E);

    /* Compare point */
    ExpectIntEQ(wc_ecc_cmp_point(point, cpypt), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_cmp_point(NULL, cpypt), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_cmp_point(point, NULL), BAD_FUNC_ARG);

    /* At infinity if return == 1, otherwise return == 0. */
    ExpectIntEQ(wc_ecc_point_is_at_infinity(point), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_point_is_at_infinity(NULL), BAD_FUNC_ARG);

#if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2)))
#ifdef USE_ECC_B_PARAM
    /* On curve if ret == 0 */
    ExpectIntEQ(wc_ecc_point_is_on_curve(point, idx), 0);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_point_is_on_curve(NULL, idx), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_point_is_on_curve(point, 1000), ECC_BAD_ARG_E);
#endif /* USE_ECC_B_PARAM */
#endif /* !HAVE_SELFTEST && (!HAVE_FIPS || HAVE_FIPS_VERSION > 2) */

    /* Free */
    wc_ecc_del_point(point);
    wc_ecc_del_point(cpypt);
    wc_ecc_free(&key);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_pointFns */


/*
 * Testing wc_ecc_sahred_secret_ssh()
 */
static int test_wc_ecc_shared_secret_ssh(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_DHE) && \
    !defined(WC_NO_RNG) && !defined(WOLFSSL_ATECC508A) && \
    !defined(WOLFSSL_ATECC608A)
    ecc_key key;
    ecc_key key2;
    WC_RNG  rng;
    int     ret;
    int     keySz = KEY32;
    int     key2Sz = KEY24;
    byte    secret[KEY32];
    word32  secretLen = keySz;

    /* Init stack variables. */
    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&key2, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(secret, 0, secretLen);

    /* Make keys */
    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, keySz, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);

    ExpectIntEQ(wc_ecc_init(&key2), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, key2Sz, &key2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key2.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ExpectIntEQ(wc_ecc_set_rng(&key, &rng), 0);
#endif

    ExpectIntEQ(wc_ecc_shared_secret_ssh(&key, &key2.pubkey, secret,
        &secretLen), 0);
    /* Pass in bad args. */
    ExpectIntEQ(wc_ecc_shared_secret_ssh(NULL, &key2.pubkey, secret,
        &secretLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_shared_secret_ssh(&key, NULL, secret, &secretLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_shared_secret_ssh(&key, &key2.pubkey, NULL, &secretLen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_shared_secret_ssh(&key, &key2.pubkey, secret, NULL),
        BAD_FUNC_ARG);
    key.type = ECC_PUBLICKEY;
    ExpectIntEQ(wc_ecc_shared_secret_ssh(&key, &key2.pubkey, secret,
        &secretLen), ECC_BAD_ARG_E);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);
    wc_ecc_free(&key2);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_shared_secret_ssh */

/*
 * Testing wc_ecc_verify_hash_ex() and wc_ecc_verify_hash_ex()
 */
static int test_wc_ecc_verify_hash_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_SIGN) && defined(WOLFSSL_PUBLIC_MP) \
    && !defined(WC_NO_RNG) && !defined(WOLFSSL_ATECC508A) && \
       !defined(WOLFSSL_ATECC608A) && !defined(WOLFSSL_KCAPI_ECC)
    ecc_key       key;
    WC_RNG        rng;
    int           ret;
    mp_int        r;
    mp_int        s;
    mp_int        z;
    unsigned char hash[] = "Everyone gets Friday off.EccSig";
    unsigned char iHash[] = "Everyone gets Friday off.......";
    unsigned char shortHash[] = TEST_STRING;
    word32        hashlen = sizeof(hash);
    word32        iHashLen = sizeof(iHash);
    word32        shortHashLen = sizeof(shortHash);
    int           keySz = KEY32;
    int           verify_ok = 0;

    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&r, 0, sizeof(mp_int));
    XMEMSET(&s, 0, sizeof(mp_int));
    XMEMSET(&z, 0, sizeof(mp_int));

    /* Initialize r, s and z. */
    ExpectIntEQ(mp_init_multi(&r, &s, &z, NULL, NULL, NULL), MP_OKAY);

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, keySz, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_ecc_sign_hash_ex(hash, hashlen, &rng, &key, &r, &s), 0);
    /* verify_ok should be 1. */
    ExpectIntEQ(wc_ecc_verify_hash_ex(&r, &s, hash, hashlen, &verify_ok, &key),
        0);
    ExpectIntEQ(verify_ok, 1);

    /* verify_ok should be 0 */
    ExpectIntEQ(wc_ecc_verify_hash_ex(&r, &s, iHash, iHashLen, &verify_ok,
        &key), 0);
    ExpectIntEQ(verify_ok, 0);

    /* verify_ok should be 0. */
    ExpectIntEQ(wc_ecc_verify_hash_ex(&r, &s, shortHash, shortHashLen,
        &verify_ok, &key), 0);
    ExpectIntEQ(verify_ok, 0);

    /* Test bad args. */
    ExpectIntEQ(wc_ecc_sign_hash_ex(NULL, hashlen, &rng, &key, &r, &s),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sign_hash_ex(hash, hashlen, NULL, &key, &r, &s),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sign_hash_ex(hash, hashlen, &rng, NULL, &r, &s),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sign_hash_ex(hash, hashlen, &rng, &key, NULL, &s),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_sign_hash_ex(hash, hashlen, &rng, &key, &r, NULL),
        ECC_BAD_ARG_E);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_verify_hash_ex(NULL, &s, shortHash, shortHashLen,
        &verify_ok, &key), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_verify_hash_ex(&r, NULL, shortHash, shortHashLen,
        &verify_ok, &key), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_verify_hash_ex(&z, &s, shortHash, shortHashLen,
        &verify_ok, &key), MP_ZERO_E);
    ExpectIntEQ(wc_ecc_verify_hash_ex(&r, &z, shortHash, shortHashLen,
        &verify_ok, &key), MP_ZERO_E);
    ExpectIntEQ(wc_ecc_verify_hash_ex(&z, &z, shortHash, shortHashLen,
        &verify_ok, &key), MP_ZERO_E);
    ExpectIntEQ(wc_ecc_verify_hash_ex(&r, &s, NULL, shortHashLen, &verify_ok,
        &key), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_verify_hash_ex(&r, &s, shortHash, shortHashLen, NULL,
        &key), ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_verify_hash_ex(&r, &s, shortHash, shortHashLen,
        &verify_ok, NULL), ECC_BAD_ARG_E);

    wc_ecc_free(&key);
    mp_free(&r);
    mp_free(&s);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_verify_hash_ex */

/*
 * Testing wc_ecc_mulmod()
 */

static int test_wc_ecc_mulmod(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG) && \
    !(defined(WOLFSSL_ATECC508A) || defined(WOLFSSL_ATECC608A) || \
      defined(WOLFSSL_VALIDATE_ECC_IMPORT))
    ecc_key     key1;
    ecc_key     key2;
    ecc_key     key3;
    WC_RNG      rng;
    int         ret;

    XMEMSET(&key1, 0, sizeof(ecc_key));
    XMEMSET(&key2, 0, sizeof(ecc_key));
    XMEMSET(&key3, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ecc_init(&key1), 0);
    ExpectIntEQ(wc_ecc_init(&key2), 0);
    ExpectIntEQ(wc_ecc_init(&key3), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, KEY32, &key1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key1.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);

    ExpectIntEQ(wc_ecc_import_raw_ex(&key2, key1.dp->Gx, key1.dp->Gy,
        key1.dp->Af, ECC_SECP256R1), 0);
    ExpectIntEQ(wc_ecc_import_raw_ex(&key3, key1.dp->Gx, key1.dp->Gy,
        key1.dp->prime, ECC_SECP256R1), 0);

    ExpectIntEQ(wc_ecc_mulmod(wc_ecc_key_get_priv(&key1), &key2.pubkey,
        &key3.pubkey, wc_ecc_key_get_priv(&key2), wc_ecc_key_get_priv(&key3),
        1), 0);

    /* Test bad args. */
    ExpectIntEQ(ret = wc_ecc_mulmod(NULL, &key2.pubkey, &key3.pubkey,
        wc_ecc_key_get_priv(&key2), wc_ecc_key_get_priv(&key3), 1),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_mulmod(wc_ecc_key_get_priv(&key1), NULL, &key3.pubkey,
        wc_ecc_key_get_priv(&key2), wc_ecc_key_get_priv(&key3), 1),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_mulmod(wc_ecc_key_get_priv(&key1), &key2.pubkey, NULL,
        wc_ecc_key_get_priv(&key2), wc_ecc_key_get_priv(&key3), 1),
        ECC_BAD_ARG_E);
    ExpectIntEQ(wc_ecc_mulmod(wc_ecc_key_get_priv(&key1), &key2.pubkey,
        &key3.pubkey, wc_ecc_key_get_priv(&key2), NULL, 1), ECC_BAD_ARG_E);

    wc_ecc_free(&key1);
    wc_ecc_free(&key2);
    wc_ecc_free(&key3);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif /* HAVE_ECC && !WOLFSSL_ATECC508A */
    return EXPECT_RESULT();
} /* END test_wc_ecc_mulmod */

/*
 * Testing wc_ecc_is_valid_idx()
 */
static int test_wc_ecc_is_valid_idx(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    int     ret;
    int     iVal = -2;
    int     iVal2 = 3000;

    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, 32, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_ecc_is_valid_idx(key.idx), 1);
    /* Test bad args. */
    ExpectIntEQ(wc_ecc_is_valid_idx(iVal), 0);
    ExpectIntEQ(wc_ecc_is_valid_idx(iVal2), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_is_valid_idx */

/*
 * Testing wc_ecc_get_curve_id_from_oid()
 */
static int test_wc_ecc_get_curve_id_from_oid(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(NO_ECC256) && !defined(HAVE_SELFTEST) && \
    !defined(HAVE_FIPS)
    const byte oid[] = {0x2A,0x86,0x48,0xCE,0x3D,0x03,0x01,0x07};
    word32 len = sizeof(oid);

    /* Bad Cases */
    ExpectIntEQ(wc_ecc_get_curve_id_from_oid(NULL, len), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_get_curve_id_from_oid(oid, 0), ECC_CURVE_INVALID);
    /* Good Case */
    ExpectIntEQ(wc_ecc_get_curve_id_from_oid(oid, len), ECC_SECP256R1);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_get_curve_id_from_oid */

/*
 * Testing wc_ecc_sig_size_calc()
 */
static int test_wc_ecc_sig_size_calc(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG) && !defined(HAVE_SELFTEST)
    ecc_key key;
    WC_RNG  rng;
    int     sz = 0;
    int     ret;

    XMEMSET(&key, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ecc_init(&key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ret = wc_ecc_make_key(&rng, 16, &key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    sz = key.dp->size;
    ExpectIntGT(wc_ecc_sig_size_calc(sz), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ecc_free(&key);
#endif
    return EXPECT_RESULT();
} /* END test_wc_ecc_sig_size_calc */

/*
 * Testing wc_ecc_sm2_make_key()
 */
static int test_wc_ecc_sm2_make_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(WOLFSSL_SM2)
    EXPECT_DECLS;
    WC_RNG  rng[1];
    ecc_key key[1];

    XMEMSET(rng, 0, sizeof(*rng));
    XMEMSET(key, 0, sizeof(*key));

    ExpectIntEQ(wc_InitRng(rng), 0);
    ExpectIntEQ(wc_ecc_init(key), 0);

    /* Test invalid parameters. */
    ExpectIntEQ(wc_ecc_sm2_make_key(NULL, NULL, WC_ECC_FLAG_NONE),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_make_key(rng, NULL, WC_ECC_FLAG_NONE),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_make_key(NULL, key, WC_ECC_FLAG_NONE),
        BAD_FUNC_ARG);

    /* Test valid parameters. */
    ExpectIntEQ(wc_ecc_sm2_make_key(rng, key, WC_ECC_FLAG_NONE), 0);
    ExpectIntEQ(key->dp->id, ECC_SM2P256V1);

    wc_ecc_free(key);
    wc_FreeRng(rng);
#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = EXPECT_RESULT();
#endif
    return res;
}

/*
 * Testing wc_ecc_sm2_shared_secret()
 */
static int test_wc_ecc_sm2_shared_secret(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(WOLFSSL_SM2)
    EXPECT_DECLS;
    WC_RNG  rng[1];
    ecc_key keyA[1];
    ecc_key keyB[1];
    byte outA[32];
    byte outB[32];
    word32 outALen = 32;
    word32 outBLen = 32;

    XMEMSET(rng, 0, sizeof(*rng));
    XMEMSET(keyA, 0, sizeof(*keyA));
    XMEMSET(keyB, 0, sizeof(*keyB));

    ExpectIntEQ(wc_InitRng(rng), 0);
    ExpectIntEQ(wc_ecc_init(keyA), 0);
    ExpectIntEQ(wc_ecc_init(keyB), 0);
    ExpectIntEQ(wc_ecc_sm2_make_key(rng, keyA, WC_ECC_FLAG_NONE), 0);
    ExpectIntEQ(wc_ecc_sm2_make_key(rng, keyB, WC_ECC_FLAG_NONE), 0);

#ifdef ECC_TIMING_RESISTANT
    ExpectIntEQ(wc_ecc_set_rng(keyA, rng), 0);
    ExpectIntEQ(wc_ecc_set_rng(keyB, rng), 0);
#endif

    /* Test invalid parameters. */
    ExpectIntEQ(wc_ecc_sm2_shared_secret(NULL, NULL, NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_shared_secret(keyA, NULL, NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_shared_secret(NULL, keyB, NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_shared_secret(NULL, NULL, outA, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_shared_secret(NULL, NULL, NULL, &outALen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_shared_secret(NULL, keyB, outA, &outALen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_shared_secret(keyA, NULL, outA, &outALen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_shared_secret(keyA, keyB, NULL, &outALen),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_shared_secret(keyA, keyB, outA, NULL), BAD_FUNC_ARG);

    /* Test valid parameters. */
    ExpectIntEQ(wc_ecc_sm2_shared_secret(keyA, keyB, outA, &outALen), 0);
    ExpectIntLE(outALen, 32);
    ExpectIntEQ(wc_ecc_sm2_shared_secret(keyB, keyA, outB, &outBLen), 0);
    ExpectIntLE(outBLen, 32);
    ExpectIntEQ(outALen, outBLen);
    ExpectBufEQ(outA, outB, outALen);

    wc_ecc_free(keyB);
    wc_ecc_free(keyA);
    wc_FreeRng(rng);
#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = EXPECT_RESULT();
#endif
    return res;
}

/*
 * Testing wc_ecc_sm2_create_digest()
 */
static int test_wc_ecc_sm2_create_digest(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(WOLFSSL_SM2) && !defined(NO_HASH_WRAPPER) && \
    (defined(WOLFSSL_SM3) || !defined(NO_SHA256))
    EXPECT_DECLS;
    ecc_key key[1];
    enum wc_HashType hashType;
    unsigned char pub[] = {
        0x04,
        0x63, 0x7F, 0x1B, 0x13, 0x50, 0x36, 0xC9, 0x33,
        0xDC, 0x3F, 0x7A, 0x8E, 0xBB, 0x1B, 0x7B, 0x2F,
        0xD1, 0xDF, 0xBD, 0x26, 0x8D, 0x4F, 0x89, 0x4B,
        0x5A, 0xD4, 0x7D, 0xBD, 0xBE, 0xCD, 0x55, 0x8F,
        0xE8, 0x81, 0x01, 0xD0, 0x80, 0x48, 0xE3, 0x6C,
        0xCB, 0xF6, 0x1C, 0xA3, 0x8D, 0xDF, 0x7A, 0xBA,
        0x54, 0x2B, 0x44, 0x86, 0xE9, 0x9E, 0x49, 0xF3,
        0xA7, 0x47, 0x0A, 0x85, 0x7A, 0x09, 0x64, 0x33
    };
    unsigned char id[] = {
        0x01, 0x02, 0x03,
    };
    unsigned char msg[] = {
        0x01, 0x02, 0x03,
    };
    unsigned char hash[32];
#ifdef WOLFSSL_SM3
    unsigned char expHash[32] = {
        0xc1, 0xdd, 0x92, 0xc5, 0x60, 0xd3, 0x94, 0x28,
        0xeb, 0x0f, 0x57, 0x79, 0x3f, 0xc9, 0x96, 0xc5,
        0xfa, 0xf5, 0x90, 0xb2, 0x64, 0x2f, 0xaf, 0x9c,
        0xc8, 0x57, 0x21, 0x6a, 0x52, 0x7e, 0xf1, 0x95
    };
#else
    unsigned char expHash[32] = {
        0xea, 0x41, 0x55, 0x21, 0x61, 0x00, 0x5c, 0x9a,
        0x57, 0x35, 0x6b, 0x49, 0xca, 0x8f, 0x65, 0xc2,
        0x0e, 0x29, 0x0c, 0xa0, 0x1d, 0xa7, 0xc4, 0xed,
        0xdd, 0x51, 0x12, 0xf6, 0xe7, 0x55, 0xc5, 0xf4
    };
#endif

#ifdef WOLFSSL_SM3
    hashType = WC_HASH_TYPE_SM3;
#else
    hashType = WC_HASH_TYPE_SHA256;
#endif

    XMEMSET(key, 0, sizeof(*key));

    ExpectIntEQ(wc_ecc_init(key), 0);

    /* Test with no curve set. */
    ExpectIntEQ(wc_ecc_sm2_create_digest(id, sizeof(id), msg, sizeof(msg),
        hashType, hash, sizeof(hash), key), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ecc_import_x963_ex(pub, sizeof(pub), key, ECC_SM2P256V1), 0);

    /* Test invalid parameters. */
    ExpectIntEQ(wc_ecc_sm2_create_digest(NULL, sizeof(id), NULL, sizeof(msg),
        hashType, NULL, sizeof(hash), NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_create_digest(id, sizeof(id), NULL, sizeof(msg),
        hashType, NULL, sizeof(hash), NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_create_digest(NULL, sizeof(id), msg, sizeof(msg),
        hashType, NULL, sizeof(hash), NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_create_digest(NULL, sizeof(id), NULL, sizeof(msg),
        hashType, hash, sizeof(hash), NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_create_digest(NULL, sizeof(id), NULL, sizeof(msg),
        hashType, NULL, sizeof(hash), key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_create_digest(NULL, sizeof(id), msg, sizeof(msg),
        hashType, hash, sizeof(hash), key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_create_digest(id, sizeof(id), NULL, sizeof(msg),
        hashType, hash, sizeof(hash), key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_create_digest(id, sizeof(id), msg, sizeof(msg),
        hashType, NULL, sizeof(hash), key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_create_digest(id, sizeof(id), msg, sizeof(msg),
        hashType, hash, sizeof(hash), NULL), BAD_FUNC_ARG);

    /* Bad hash type. */
    ExpectIntEQ(wc_ecc_sm2_create_digest(id, sizeof(id), msg, sizeof(msg),
        -1, hash, 0, key), BAD_FUNC_ARG);
    /* Bad hash size. */
    ExpectIntEQ(wc_ecc_sm2_create_digest(id, sizeof(id), msg, sizeof(msg),
        hashType, hash, 0, key), BUFFER_E);

    /* Test valid parameters. */
    ExpectIntEQ(wc_ecc_sm2_create_digest(id, sizeof(id), msg, sizeof(msg),
        hashType, hash, sizeof(hash), key), 0);
    ExpectBufEQ(hash, expHash, sizeof(expHash));

    wc_ecc_free(key);

    res = EXPECT_RESULT();
#endif
    return res;
}
/*
 * Testing wc_ecc_sm2_verify_hash_ex()
 */
static int test_wc_ecc_sm2_verify_hash_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(WOLFSSL_SM2) && defined(HAVE_ECC_VERIFY) && \
    defined(WOLFSSL_PUBLIC_MP)
    EXPECT_DECLS;
    ecc_key key[1];
    mp_int r[1];
    mp_int s[1];
    int verified;
    unsigned char pub[] = {
        0x04,
        0x63, 0x7F, 0x1B, 0x13, 0x50, 0x36, 0xC9, 0x33,
        0xDC, 0x3F, 0x7A, 0x8E, 0xBB, 0x1B, 0x7B, 0x2F,
        0xD1, 0xDF, 0xBD, 0x26, 0x8D, 0x4F, 0x89, 0x4B,
        0x5A, 0xD4, 0x7D, 0xBD, 0xBE, 0xCD, 0x55, 0x8F,
        0xE8, 0x81, 0x01, 0xD0, 0x80, 0x48, 0xE3, 0x6C,
        0xCB, 0xF6, 0x1C, 0xA3, 0x8D, 0xDF, 0x7A, 0xBA,
        0x54, 0x2B, 0x44, 0x86, 0xE9, 0x9E, 0x49, 0xF3,
        0xA7, 0x47, 0x0A, 0x85, 0x7A, 0x09, 0x64, 0x33
    };
    unsigned char hash[] = {
        0x3B, 0xFA, 0x5F, 0xFB, 0xC4, 0x27, 0x8C, 0x9D,
        0x02, 0x3A, 0x19, 0xCB, 0x1E, 0xAA, 0xD2, 0xF1,
        0x50, 0x69, 0x5B, 0x20
    };
    unsigned char rData[] = {
        0xD2, 0xFC, 0xA3, 0x88, 0xE3, 0xDF, 0xA3, 0x00,
        0x73, 0x9B, 0x3C, 0x2A, 0x0D, 0xAD, 0x44, 0xA2,
        0xFC, 0x62, 0xD5, 0x6B, 0x84, 0x54, 0xD8, 0x40,
        0x22, 0x62, 0x3D, 0x5C, 0xA6, 0x61, 0x9B, 0xE7,
    };
    unsigned char sData[] = {
        0x1D,
        0xB5, 0xB5, 0xD9, 0xD8, 0xF1, 0x20, 0xDD, 0x97,
        0x92, 0xBF, 0x7E, 0x9B, 0x3F, 0xE6, 0x3C, 0x4B,
        0x03, 0xD8, 0x80, 0xBD, 0xB7, 0x27, 0x7E, 0x6A,
        0x84, 0x23, 0xDE, 0x61, 0x7C, 0x8D, 0xDC
    };
    unsigned char rBadData[] = {
        0xD2, 0xFC, 0xA3, 0x88, 0xE3, 0xDF, 0xA3, 0x00,
        0x73, 0x9B, 0x3C, 0x2A, 0x0D, 0xAD, 0x44, 0xA2,
        0xFC, 0x62, 0xD5, 0x6B, 0x84, 0x54, 0xD8, 0x40,
        0x22, 0x62, 0x3D, 0x5C, 0xA6, 0x61, 0x9B, 0xE8,
    };

    XMEMSET(key, 0, sizeof(*key));
    XMEMSET(r, 0, sizeof(*r));
    XMEMSET(s, 0, sizeof(*s));

    ExpectIntEQ(mp_init(r), 0);
    ExpectIntEQ(mp_init(s), 0);
    ExpectIntEQ(mp_read_unsigned_bin(r, rData, sizeof(rData)), 0);
    ExpectIntEQ(mp_read_unsigned_bin(s, sData, sizeof(sData)), 0);

    ExpectIntEQ(wc_ecc_init(key), 0);

    /* Test with no curve set. */
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, s, hash, sizeof(hash),
        &verified, key), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ecc_import_x963_ex(pub, sizeof(pub), key, ECC_SM2P256V1), 0);

    /* Test invalid parameters. */
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(NULL, NULL, NULL, sizeof(hash),
        NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, NULL, NULL, sizeof(hash),
        NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(NULL, s, NULL, sizeof(hash),
        NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(NULL, NULL, hash, sizeof(hash),
        NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(NULL, NULL, NULL, sizeof(hash),
        &verified, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(NULL, NULL, NULL, sizeof(hash),
        NULL, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(NULL, s, hash, sizeof(hash),
        &verified, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, NULL, hash, sizeof(hash),
        &verified, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, s, NULL, sizeof(hash),
        &verified, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, s, hash, sizeof(hash),
        NULL, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, s, hash, sizeof(hash),
        &verified, NULL), BAD_FUNC_ARG);

    /* Make key not on the SM2 curve. */
    ExpectIntEQ(wc_ecc_set_curve(key, 32, ECC_SECP256R1), 0);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, s, hash, sizeof(hash),
        &verified, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_set_curve(key, 32, ECC_SM2P256V1), 0);

    /* Test valid parameters. */
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, s, hash, sizeof(hash),
        &verified, key), 0);
    ExpectIntEQ(verified, 1);

    ExpectIntEQ(mp_read_unsigned_bin(r, rBadData, sizeof(rBadData)), 0);
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, s, hash, sizeof(hash),
        &verified, key), 0);
    ExpectIntEQ(verified, 0);

    mp_free(s);
    mp_free(r);
    wc_ecc_free(key);
#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = EXPECT_RESULT();
#endif
    return res;
}

/*
 * Testing wc_ecc_sm2_verify_hash()
 */
static int test_wc_ecc_sm2_verify_hash(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(WOLFSSL_SM2) && defined(HAVE_ECC_VERIFY)
    EXPECT_DECLS;
    ecc_key key[1];
    int verified;
    unsigned char pub[] = {
        0x04,
        0x63, 0x7F, 0x1B, 0x13, 0x50, 0x36, 0xC9, 0x33,
        0xDC, 0x3F, 0x7A, 0x8E, 0xBB, 0x1B, 0x7B, 0x2F,
        0xD1, 0xDF, 0xBD, 0x26, 0x8D, 0x4F, 0x89, 0x4B,
        0x5A, 0xD4, 0x7D, 0xBD, 0xBE, 0xCD, 0x55, 0x8F,
        0xE8, 0x81, 0x01, 0xD0, 0x80, 0x48, 0xE3, 0x6C,
        0xCB, 0xF6, 0x1C, 0xA3, 0x8D, 0xDF, 0x7A, 0xBA,
        0x54, 0x2B, 0x44, 0x86, 0xE9, 0x9E, 0x49, 0xF3,
        0xA7, 0x47, 0x0A, 0x85, 0x7A, 0x09, 0x64, 0x33
    };
    unsigned char hash[] = {
        0x3B, 0xFA, 0x5F, 0xFB, 0xC4, 0x27, 0x8C, 0x9D,
        0x02, 0x3A, 0x19, 0xCB, 0x1E, 0xAA, 0xD2, 0xF1,
        0x50, 0x69, 0x5B, 0x20
    };
    unsigned char sig[] = {
        0x30, 0x45, 0x02, 0x21, 0x00, 0xD2, 0xFC, 0xA3,
        0x88, 0xE3, 0xDF, 0xA3, 0x00, 0x73, 0x9B, 0x3C,
        0x2A, 0x0D, 0xAD, 0x44, 0xA2, 0xFC, 0x62, 0xD5,
        0x6B, 0x84, 0x54, 0xD8, 0x40, 0x22, 0x62, 0x3D,
        0x5C, 0xA6, 0x61, 0x9B, 0xE7, 0x02, 0x20, 0x1D,
        0xB5, 0xB5, 0xD9, 0xD8, 0xF1, 0x20, 0xDD, 0x97,
        0x92, 0xBF, 0x7E, 0x9B, 0x3F, 0xE6, 0x3C, 0x4B,
        0x03, 0xD8, 0x80, 0xBD, 0xB7, 0x27, 0x7E, 0x6A,
        0x84, 0x23, 0xDE, 0x61, 0x7C, 0x8D, 0xDC
    };
    unsigned char sigBad[] = {
        0x30, 0x45, 0x02, 0x21, 0x00, 0xD2, 0xFC, 0xA3,
        0x88, 0xE3, 0xDF, 0xA3, 0x00, 0x73, 0x9B, 0x3C,
        0x2A, 0x0D, 0xAD, 0x44, 0xA2, 0xFC, 0x62, 0xD5,
        0x6B, 0x84, 0x54, 0xD8, 0x40, 0x22, 0x62, 0x3D,
        0x5C, 0xA6, 0x61, 0x9B, 0xE7, 0x02, 0x20, 0x1D,
        0xB5, 0xB5, 0xD9, 0xD8, 0xF1, 0x20, 0xDD, 0x97,
        0x92, 0xBF, 0x7E, 0x9B, 0x3F, 0xE6, 0x3C, 0x4B,
        0x03, 0xD8, 0x80, 0xBD, 0xB7, 0x27, 0x7E, 0x6A,
        0x84, 0x23, 0xDE, 0x61, 0x7C, 0x8D, 0xDD
    };


    XMEMSET(key, 0, sizeof(*key));
    ExpectIntEQ(wc_ecc_init(key), 0);

    /* Test with no curve set. */
    ExpectIntEQ(wc_ecc_sm2_verify_hash(sig, sizeof(sig), hash, sizeof(hash),
        &verified, key), BAD_FUNC_ARG);

    ExpectIntEQ(wc_ecc_import_x963_ex(pub, sizeof(pub), key, ECC_SM2P256V1), 0);

    /* Test invalid parameters. */
    ExpectIntEQ(wc_ecc_sm2_verify_hash(NULL, sizeof(sig), NULL, sizeof(hash),
        NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash(sig, sizeof(sig), NULL, sizeof(hash),
        NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash(NULL, sizeof(sig), hash, sizeof(hash),
        NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash(NULL, sizeof(sig), NULL, sizeof(hash),
        &verified, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash(NULL, sizeof(sig), NULL, sizeof(hash),
        NULL, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash(NULL, sizeof(sig), hash, sizeof(hash),
        &verified, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash(sig, sizeof(sig), NULL, sizeof(hash),
        &verified, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash(sig, sizeof(sig), hash, sizeof(hash),
        NULL, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_verify_hash(sig, sizeof(sig), hash, sizeof(hash),
        &verified, NULL), BAD_FUNC_ARG);

    /* Make key not on the SM2 curve. */
    ExpectIntEQ(wc_ecc_set_curve(key, 32, ECC_SECP256R1), 0);
    ExpectIntEQ(wc_ecc_sm2_verify_hash(sig, sizeof(sig), hash, sizeof(hash),
        &verified, key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_set_curve(key, 32, ECC_SM2P256V1), 0);

    /* Test valid parameters. */
    ExpectIntEQ(wc_ecc_sm2_verify_hash(sig, sizeof(sig), hash, sizeof(hash),
        &verified, key), 0);
    ExpectIntEQ(verified, 1);

    ExpectIntEQ(wc_ecc_sm2_verify_hash(sigBad, sizeof(sigBad), hash,
        sizeof(hash), &verified, key), 0);
    ExpectIntEQ(verified, 0);

    wc_ecc_free(key);
#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = EXPECT_RESULT();
#endif
    return res;
}

/*
 * Testing wc_ecc_sm2_verify_hash_ex()
 */
static int test_wc_ecc_sm2_sign_hash_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(WOLFSSL_SM2) && defined(HAVE_ECC_SIGN) && \
    defined(WOLFSSL_PUBLIC_MP)
    EXPECT_DECLS;
    WC_RNG  rng[1];
    ecc_key key[1];
    mp_int r[1];
    mp_int s[1];
    unsigned char hash[32];
#ifdef HAVE_ECC_VERIFY
    int verified;
#endif

    XMEMSET(rng, 0, sizeof(*rng));
    XMEMSET(key, 0, sizeof(*key));
    XMEMSET(r, 0, sizeof(*r));
    XMEMSET(s, 0, sizeof(*s));

    ExpectIntEQ(wc_InitRng(rng), 0);
    ExpectIntEQ(mp_init(r), 0);
    ExpectIntEQ(mp_init(s), 0);
    ExpectIntEQ(wc_RNG_GenerateBlock(rng, hash, sizeof(hash)), 0);

    ExpectIntEQ(wc_ecc_init(key), 0);

    /* Test with no curve set. */
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(hash, sizeof(hash), rng, key, r, s),
        BAD_FUNC_ARG);

    ExpectIntEQ(wc_ecc_sm2_make_key(rng, key, WC_ECC_FLAG_NONE), 0);

    /* Test invalid parameters. */
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(NULL, sizeof(hash), NULL, NULL, NULL,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(hash, sizeof(hash), NULL, NULL, NULL,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(NULL, sizeof(hash), rng, NULL, NULL,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(NULL, sizeof(hash), NULL, key, NULL,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(NULL, sizeof(hash), NULL, NULL, r,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(NULL, sizeof(hash), NULL, NULL, NULL,
        s), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(NULL, sizeof(hash), rng, key, r, s),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(hash, sizeof(hash), NULL, key, r, s),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(hash, sizeof(hash), rng, NULL, r, s),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(hash, sizeof(hash), rng, key, NULL, s),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(hash, sizeof(hash), rng, key, r, NULL),
        BAD_FUNC_ARG);

    /* Make key not on the SM2 curve. */
    ExpectIntEQ(wc_ecc_set_curve(key, 32, ECC_SECP256R1), 0);
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(hash, sizeof(hash), rng, key, r, s),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_set_curve(key, 32, ECC_SM2P256V1), 0);

#ifdef WOLFSSL_SP_MATH_ALL
    {
        mp_int smallR[1];
        sp_init_size(smallR, 1);
        /* Force failure in _ecc_sm2_calc_r_s by r being too small. */
        ExpectIntLT(wc_ecc_sm2_sign_hash_ex(hash, sizeof(hash), rng, key,
            smallR, s), 0);
    }
#endif

    /* Test valid parameters. */
    ExpectIntEQ(wc_ecc_sm2_sign_hash_ex(hash, sizeof(hash), rng, key, r, s),
        0);
#ifdef HAVE_ECC_VERIFY
    ExpectIntEQ(wc_ecc_sm2_verify_hash_ex(r, s, hash, sizeof(hash), &verified,
        key), 0);
    ExpectIntEQ(verified, 1);
#endif

    mp_free(s);
    mp_free(r);
    wc_ecc_free(key);
    wc_FreeRng(rng);
#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = EXPECT_RESULT();
#endif
    return res;
}


/*
 * Testing wc_ecc_sm2_verify_hash()
 */
static int test_wc_ecc_sm2_sign_hash(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(WOLFSSL_SM2) && defined(HAVE_ECC_SIGN)
    EXPECT_DECLS;
    WC_RNG  rng[1];
    ecc_key key[1];
    unsigned char hash[32];
    unsigned char sig[72];
    word32 sigSz = sizeof(sig);
#ifdef HAVE_ECC_VERIFY
    int verified;
#endif

    XMEMSET(rng, 0, sizeof(*rng));
    XMEMSET(key, 0, sizeof(*key));

    ExpectIntEQ(wc_InitRng(rng), 0);
    ExpectIntEQ(wc_RNG_GenerateBlock(rng, hash, sizeof(hash)), 0);

    ExpectIntEQ(wc_ecc_init(key), 0);

    /* Test with no curve set. */
    ExpectIntEQ(wc_ecc_sm2_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, key),
        BAD_FUNC_ARG);

    ExpectIntEQ(wc_ecc_sm2_make_key(rng, key, WC_ECC_FLAG_NONE), 0);

    /* Test invalid parameters. */
    ExpectIntEQ(wc_ecc_sm2_sign_hash(NULL, sizeof(hash), NULL, NULL, NULL,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(hash, sizeof(hash), NULL, NULL, NULL,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(NULL, sizeof(hash), sig, NULL, NULL,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(NULL, sizeof(hash), NULL, &sigSz, NULL,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(NULL, sizeof(hash), NULL, NULL, rng,
        NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(NULL, sizeof(hash), NULL, NULL, NULL,
        key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(NULL, sizeof(hash), sig, &sigSz, rng,
        key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(hash, sizeof(hash), NULL, &sigSz, rng,
        key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(hash, sizeof(hash), sig, NULL, rng,
        key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(hash, sizeof(hash), sig, &sigSz, NULL,
        key), BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(hash, sizeof(hash), sig, &sigSz, rng,
        NULL), BAD_FUNC_ARG);

    /* Make key not on the SM2 curve. */
    ExpectIntEQ(wc_ecc_set_curve(key, 32, ECC_SECP256R1), 0);
    ExpectIntEQ(wc_ecc_sm2_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, key),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_ecc_set_curve(key, 32, ECC_SM2P256V1), 0);

    /* Test valid parameters. */
    ExpectIntEQ(wc_ecc_sm2_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, key),
        0);
#ifdef HAVE_ECC_VERIFY
    ExpectIntEQ(wc_ecc_sm2_verify_hash(sig, sigSz, hash, sizeof(hash),
        &verified, key), 0);
    ExpectIntEQ(verified, 1);
#endif

    wc_ecc_free(key);
    wc_FreeRng(rng);
#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = EXPECT_RESULT();
#endif
    return res;
}


/*
 * Testing ToTraditional
 */
static int test_ToTraditional(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && (defined(HAVE_PKCS8) || defined(HAVE_PKCS12)) && \
    (defined(WOLFSSL_TEST_CERT) || defined(OPENSSL_EXTRA) || \
     defined(OPENSSL_EXTRA_X509_SMALL)) && !defined(NO_FILESYSTEM)
    XFILE  f = XBADFILE;
    byte   input[TWOK_BUF];
    word32 sz;

    ExpectTrue((f = XFOPEN("./certs/server-keyPkcs8.der", "rb")) != XBADFILE);
    ExpectTrue((sz = (word32)XFREAD(input, 1, sizeof(input), f)) > 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    /* Good case */
    ExpectIntGT(ToTraditional(input, sz), 0);
    /* Bad cases */
    ExpectIntEQ(ToTraditional(NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(ToTraditional(NULL, sz), BAD_FUNC_ARG);
#ifdef WOLFSSL_ASN_TEMPLATE
    ExpectIntEQ(ToTraditional(input, 0), BUFFER_E);
#else
    ExpectIntEQ(ToTraditional(input, 0), ASN_PARSE_E);
#endif
#endif
    return EXPECT_RESULT();
} /* End test_ToTraditional*/

/*
 * Testing wc_EccPrivateKeyToDer
 */
static int test_wc_EccPrivateKeyToDer(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    byte    output[ONEK_BUF];
    ecc_key eccKey;
    WC_RNG  rng;
    word32  inLen;
    word32  outLen = 0;
    int     ret;

    XMEMSET(&eccKey, 0, sizeof(ecc_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ecc_init(&eccKey), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);

    inLen = (word32)sizeof(output);
    /* Bad Cases */
    ExpectIntEQ(wc_EccPrivateKeyToDer(NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_EccPrivateKeyToDer(NULL, output, inLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_EccPrivateKeyToDer(&eccKey, NULL, inLen), LENGTH_ONLY_E);
    ExpectIntEQ(wc_EccPrivateKeyToDer(&eccKey, output, 0), BAD_FUNC_ARG);
    /* Good Case */
    ExpectIntGT(outLen = wc_EccPrivateKeyToDer(&eccKey, output, inLen), 0);

    wc_ecc_free(&eccKey);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);

#if defined(OPENSSL_EXTRA) && defined(HAVE_ALL_CURVES)
    {
        /* test importing private only into a PKEY struct */
        EC_KEY*   ec = NULL;
        EVP_PKEY* pkey = NULL;
        const unsigned char* der;

        der = output;
        ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_EC, NULL, &der, outLen));

        der = output;
        ExpectNotNull(ec = d2i_ECPrivateKey(NULL, &der, outLen));
        ExpectIntEQ(EVP_PKEY_assign_EC_KEY(pkey, ec), SSL_SUCCESS);
        if (EXPECT_FAIL()) {
            EC_KEY_free(ec);
        }
        EVP_PKEY_free(pkey); /* EC_KEY should be free'd by free'ing pkey */
    }
#endif
#endif
    return EXPECT_RESULT();
} /* End test_wc_EccPrivateKeyToDer*/

/*
 * Testing wc_DhPublicKeyDecode
 */
static int test_wc_DhPublicKeyDecode(void)
{
    EXPECT_DECLS;
#ifndef NO_DH
#if defined(WOLFSSL_DH_EXTRA) && defined(USE_CERT_BUFFERS_2048)
    DhKey  key;
    word32 inOutIdx;

    XMEMSET(&key, 0, sizeof(DhKey));

    ExpectIntEQ(wc_InitDhKey(&key), 0);

    ExpectIntEQ(wc_DhPublicKeyDecode(NULL,NULL,NULL,0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,NULL,NULL,0),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,NULL,NULL,0),
        BAD_FUNC_ARG);
    inOutIdx = 0;
    ExpectIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,&inOutIdx,NULL, 0),
        BAD_FUNC_ARG);
    inOutIdx = 0;
    ExpectIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,&inOutIdx,&key, 0),
        BAD_FUNC_ARG);
    inOutIdx = 0;
    ExpectIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,&inOutIdx,&key,
        sizeof_dh_pub_key_der_2048), 0);
    ExpectIntNE(key.p.used, 0);
    ExpectIntNE(key.g.used, 0);
    ExpectIntEQ(key.q.used, 0);
    ExpectIntNE(key.pub.used, 0);
    ExpectIntEQ(key.priv.used, 0);

    DoExpectIntEQ(wc_FreeDhKey(&key), 0);
#endif
#endif /* !NO_DH */
    return EXPECT_RESULT();
}

/*
 * Testing wc_Ed25519KeyToDer
 */
static int test_wc_Ed25519KeyToDer(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    byte        output[ONEK_BUF];
    ed25519_key ed25519Key;
    WC_RNG      rng;
    word32      inLen;

    XMEMSET(&ed25519Key, 0, sizeof(ed25519_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed25519_init(&ed25519Key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key), 0);
    inLen = (word32)sizeof(output);

    /* Bad Cases */
    ExpectIntEQ(wc_Ed25519KeyToDer(NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Ed25519KeyToDer(NULL, output, inLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Ed25519KeyToDer(&ed25519Key, output, 0), BAD_FUNC_ARG);
    /* Good Cases */
    /* length only */
    ExpectIntGT(wc_Ed25519KeyToDer(&ed25519Key, NULL, inLen), 0);
    ExpectIntGT(wc_Ed25519KeyToDer(&ed25519Key, output, inLen), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed25519_free(&ed25519Key);
#endif
    return EXPECT_RESULT();
} /* End test_wc_Ed25519KeyToDer*/

/*
 * Testing wc_Ed25519PrivateKeyToDer
 */
static int test_wc_Ed25519PrivateKeyToDer(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    byte        output[ONEK_BUF];
    ed25519_key ed25519PrivKey;
    WC_RNG      rng;
    word32      inLen;

    XMEMSET(&ed25519PrivKey, 0, sizeof(ed25519_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed25519_init(&ed25519PrivKey), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519PrivKey),
        0);
    inLen = (word32)sizeof(output);

    /* Bad Cases */
    ExpectIntEQ(wc_Ed25519PrivateKeyToDer(NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Ed25519PrivateKeyToDer(NULL, output, inLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Ed25519PrivateKeyToDer(&ed25519PrivKey, output, 0),
        BAD_FUNC_ARG);
    /* Good Cases */
    /* length only */
    ExpectIntGT(wc_Ed25519PrivateKeyToDer(&ed25519PrivKey, NULL, inLen), 0);
    ExpectIntGT(wc_Ed25519PrivateKeyToDer(&ed25519PrivKey, output, inLen), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed25519_free(&ed25519PrivKey);
#endif
    return EXPECT_RESULT();
} /* End test_wc_Ed25519PrivateKeyToDer*/

/*
 * Testing wc_Ed448KeyToDer
 */
static int test_wc_Ed448KeyToDer(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    byte      output[ONEK_BUF];
    ed448_key ed448Key;
    WC_RNG    rng;
    word32    inLen;

    XMEMSET(&ed448Key, 0, sizeof(ed448_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed448_init(&ed448Key), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key), 0);
    inLen = (word32)sizeof(output);

    /* Bad Cases */
    ExpectIntEQ(wc_Ed448KeyToDer(NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Ed448KeyToDer(NULL, output, inLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Ed448KeyToDer(&ed448Key, output, 0), BAD_FUNC_ARG);
    /* Good Cases */
    /* length only */
    ExpectIntGT(wc_Ed448KeyToDer(&ed448Key, NULL, inLen), 0);
    ExpectIntGT(wc_Ed448KeyToDer(&ed448Key, output, inLen), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed448_free(&ed448Key);
#endif
    return EXPECT_RESULT();
} /* End test_wc_Ed448KeyToDer*/

/*
 * Testing wc_Ed448PrivateKeyToDer
 */
static int test_wc_Ed448PrivateKeyToDer(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    byte      output[ONEK_BUF];
    ed448_key ed448PrivKey;
    WC_RNG    rng;
    word32    inLen;

    XMEMSET(&ed448PrivKey, 0, sizeof(ed448_key));
    XMEMSET(&rng, 0, sizeof(WC_RNG));

    ExpectIntEQ(wc_ed448_init(&ed448PrivKey), 0);
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448PrivKey),
        0);
    inLen = (word32)sizeof(output);

    /* Bad Cases */
    ExpectIntEQ(wc_Ed448PrivateKeyToDer(NULL, NULL, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Ed448PrivateKeyToDer(NULL, output, inLen), BAD_FUNC_ARG);
    ExpectIntEQ(wc_Ed448PrivateKeyToDer(&ed448PrivKey, output, 0),
        BAD_FUNC_ARG);
    /* Good cases */
    /* length only */
    ExpectIntGT(wc_Ed448PrivateKeyToDer(&ed448PrivKey, NULL, inLen), 0);
    ExpectIntGT(wc_Ed448PrivateKeyToDer(&ed448PrivKey, output, inLen), 0);

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
    wc_ed448_free(&ed448PrivKey);
#endif
    return EXPECT_RESULT();
} /* End test_wc_Ed448PrivateKeyToDer*/

/*
 * Testing wc_SetSubjectBuffer
 */
static int test_wc_SetSubjectBuffer(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CERT_GEN) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    Cert   cert;
    XFILE  file = XBADFILE;
    byte*  der = NULL;
    word32 derSz;

    derSz = FOURK_BUF;
    ExpectNotNull(der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectTrue((file = XFOPEN("./certs/ca-cert.der", "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, FOURK_BUF, file)) > 0);
    if (file != XBADFILE)
        XFCLOSE(file);

    ExpectIntEQ(wc_InitCert(&cert), 0);
    ExpectIntEQ(wc_SetSubjectBuffer(&cert, der, derSz), 0);
    ExpectIntEQ(wc_SetSubjectBuffer(NULL, der, derSz), BAD_FUNC_ARG);

    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    return EXPECT_RESULT();
} /* End test_wc_SetSubjectBuffer*/

/*
 * Testing wc_SetSubjectKeyIdFromPublicKey_ex
 */
static int test_wc_SetSubjectKeyIdFromPublicKey_ex(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    WC_RNG      rng;
    Cert        cert;
#if !defined(NO_RSA) && defined(HAVE_RSA)
    RsaKey      rsaKey;
    int         bits = 2048;
#endif
#if defined(HAVE_ECC)
    ecc_key     eccKey;
    int         ret;
#endif
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    ed25519_key ed25519Key;
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key   ed448Key;
#endif

#ifndef HAVE_FIPS
    ExpectIntEQ(wc_InitRng_ex(&rng, HEAP_HINT, testDevId), 0);
#else
    ExpectIntEQ(wc_InitRng(&rng), 0);
#endif

    ExpectIntEQ(wc_InitCert(&cert), 0);

#if !defined(NO_RSA) && defined(HAVE_RSA) && defined(WOLFSSL_KEY_GEN)
    /* RSA */
    XMEMSET(&rsaKey, 0, sizeof(RsaKey));
    ExpectIntEQ(wc_InitRsaKey(&rsaKey, HEAP_HINT), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&rsaKey, bits, WC_RSA_EXPONENT, &rng), 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, RSA_TYPE, &rsaKey),
        0);
    DoExpectIntEQ(wc_FreeRsaKey(&rsaKey), 0);
#endif

#if defined(HAVE_ECC)
    /* ECC */
    XMEMSET(&eccKey, 0, sizeof(ecc_key));
    ExpectIntEQ(wc_ecc_init(&eccKey), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ECC_TYPE, &eccKey),
        0);
    DoExpectIntEQ(wc_ecc_free(&eccKey), 0);
#endif

#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    /* ED25519 */
    XMEMSET(&ed25519Key, 0, sizeof(ed25519_key));
    ExpectIntEQ(wc_ed25519_init(&ed25519Key), 0);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key), 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED25519_TYPE,
        &ed25519Key), 0);
    wc_ed25519_free(&ed25519Key);
#endif

#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    /* ED448 */
    XMEMSET(&ed448Key, 0, sizeof(ed448_key));
    ExpectIntEQ(wc_ed448_init(&ed448Key), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key), 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED448_TYPE,
        &ed448Key), 0);
    wc_ed448_free(&ed448Key);
#endif

    wc_FreeRng(&rng);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif /* WOLFSSL_CERT_EXT && WOLFSSL_CERT_GEN */
    return EXPECT_RESULT();
} /* End test_wc_SetSubjectKeyIdFromPublicKey_ex*/

/*
 * Testing wc_SetAuthKeyIdFromPublicKey_ex
 */
static int test_wc_SetAuthKeyIdFromPublicKey_ex(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    WC_RNG      rng;
    Cert        cert;
#if !defined(NO_RSA) && defined(HAVE_RSA)
    RsaKey      rsaKey;
    int         bits = 2048;
#endif
#if defined(HAVE_ECC)
    ecc_key     eccKey;
    int         ret;
#endif
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    ed25519_key ed25519Key;
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key   ed448Key;
#endif

#ifndef HAVE_FIPS
    ExpectIntEQ(wc_InitRng_ex(&rng, HEAP_HINT, testDevId), 0);
#else
    ExpectIntEQ(wc_InitRng(&rng), 0);
#endif

    ExpectIntEQ(wc_InitCert(&cert), 0);

#if !defined(NO_RSA) && defined(HAVE_RSA) && defined(WOLFSSL_KEY_GEN)
    /* RSA */
    XMEMSET(&rsaKey, 0, sizeof(RsaKey));
    ExpectIntEQ(wc_InitRsaKey(&rsaKey, HEAP_HINT), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&rsaKey, bits, WC_RSA_EXPONENT, &rng), 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, RSA_TYPE, &rsaKey), 0);
    DoExpectIntEQ(wc_FreeRsaKey(&rsaKey), 0);
#endif

#if defined(HAVE_ECC)
    /* ECC */
    XMEMSET(&eccKey, 0, sizeof(ecc_key));
    ExpectIntEQ(wc_ecc_init(&eccKey), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, ECC_TYPE, &eccKey), 0);
    DoExpectIntEQ(wc_ecc_free(&eccKey), 0);
#endif

#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    /* ED25519 */
    XMEMSET(&ed25519Key, 0, sizeof(ed25519_key));
    ExpectIntEQ(wc_ed25519_init(&ed25519Key), 0);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key), 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED25519_TYPE,
        &ed25519Key), 0);
    wc_ed25519_free(&ed25519Key);
#endif

#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    /* ED448 */
    XMEMSET(&ed448Key, 0, sizeof(ed448_key));
    ExpectIntEQ(wc_ed448_init(&ed448Key), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key), 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED448_TYPE, &ed448Key),
        0);
    wc_ed448_free(&ed448Key);
#endif

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif /* defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)*/
    return EXPECT_RESULT();
} /* End test_wc_SetAuthKeyIdFromPublicKey_ex*/

/*
 * Testing wc_PKCS7_New()
 */
static int test_wc_PKCS7_New(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, testDevId));
    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test-wc_PKCS7_New */

/*
 * Testing wc_PKCS7_Init()
 */
static int test_wc_PKCS7_Init(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;
    void*  heap = NULL;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(heap, testDevId));

    ExpectIntEQ(wc_PKCS7_Init(pkcs7, heap, testDevId), 0);
    /* Pass in bad args. */
    ExpectIntEQ(wc_PKCS7_Init(NULL, heap, testDevId), BAD_FUNC_ARG);

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test-wc_PKCS7_Init */


/*
 * Testing wc_PKCS7_InitWithCert()
 */
static int test_wc_PKCS7_InitWithCert(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        unsigned char    cert[sizeof(client_cert_der_2048)];
        int              certSz = (int)sizeof(cert);

        XMEMSET(cert, 0, certSz);
        XMEMCPY(cert, client_cert_der_2048, sizeof(client_cert_der_2048));
    #elif defined(USE_CERT_BUFFERS_1024)
        unsigned char    cert[sizeof(client_cert_der_1024)];
        int              certSz = (int)sizeof(cert);

        XMEMSET(cert, 0, certSz);
        XMEMCPY(cert, client_cert_der_1024, sizeof_client_cert_der_1024);
    #else
        unsigned char   cert[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz;

        ExpectTrue((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024,
            fp), 0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        int              certSz = (int)sizeof(cert);

        XMEMSET(cert, 0, certSz);
        XMEMCPY(cert, cliecc_cert_der_256, sizeof(cliecc_cert_der_256));
    #else
        unsigned char   cert[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz;

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof(cliecc_cert_der_256),
            fp), 0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#else
        #error PKCS7 requires ECC or RSA
#endif

#ifdef HAVE_ECC
    {
    /* bad test case from ZD 11011, malformed cert gives bad ECC key */
        static unsigned char certWithInvalidEccKey[] = {
        0x30, 0x82, 0x03, 0x5F, 0x30, 0x82, 0x03, 0x04, 0xA0, 0x03, 0x02, 0x01,
        0x02, 0x02, 0x14, 0x61, 0xB3, 0x1E, 0x59, 0xF3, 0x68, 0x6C, 0xA4, 0x79,
        0x42, 0x83, 0x2F, 0x1A, 0x50, 0x71, 0x03, 0xBE, 0x31, 0xAA, 0x2C, 0x30,
        0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30,
        0x81, 0x8D, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x04, 0x08,
        0x0C, 0x06, 0x4F, 0x72, 0x65, 0x67, 0x6F, 0x6E, 0x31, 0x0E, 0x30, 0x0C,
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x05, 0x53, 0x61, 0x6C, 0x65, 0x6D,
        0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0A, 0x43,
        0x6C, 0x69, 0x65, 0x6E, 0x74, 0x20, 0x45, 0x43, 0x43, 0x31, 0x0D, 0x30,
        0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x04, 0x46, 0x61, 0x73, 0x74,
        0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77,
        0x77, 0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
        0x6F, 0x6D, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
        0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40,
        0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30,
        0x1E, 0x17, 0x0D, 0x32, 0x30, 0x30, 0x36, 0x31, 0x39, 0x31, 0x33, 0x32,
        0x33, 0x34, 0x31, 0x5A, 0x17, 0x0D, 0x32, 0x33, 0x30, 0x33, 0x31, 0x36,
        0x31, 0x33, 0x32, 0x33, 0x34, 0x31, 0x5A, 0x30, 0x81, 0x8D, 0x31, 0x0B,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31,
        0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x06, 0x4F, 0x72,
        0x65, 0x67, 0x6F, 0x6E, 0x31, 0x0E, 0x30, 0x0C, 0x06, 0x03, 0x55, 0x04,
        0x07, 0x0C, 0x05, 0x53, 0x61, 0x6C, 0x65, 0x6D, 0x31, 0x13, 0x30, 0x11,
        0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0A, 0x43, 0x6C, 0x69, 0x65, 0x6E,
        0x74, 0x20, 0x45, 0x43, 0x43, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55,
        0x04, 0x0B, 0x0C, 0x04, 0x46, 0x61, 0x73, 0x74, 0x31, 0x18, 0x30, 0x26,
        0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77, 0x77, 0x77, 0x2E, 0x77,
        0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x31, 0x1F,
        0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
        0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40, 0x77, 0x6F, 0x6C, 0x66,
        0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30, 0x59, 0x30, 0x13, 0x06,
        0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86,
        0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x02, 0x00, 0x04, 0x55, 0xBF,
        0xF4, 0x0F, 0x44, 0x50, 0x9A, 0x3D, 0xCE, 0x9B, 0xB7, 0xF0, 0xC5, 0x4D,
        0xF5, 0x70, 0x7B, 0xD4, 0xEC, 0x24, 0x8E, 0x19, 0x80, 0xEC, 0x5A, 0x4C,
        0xA2, 0x24, 0x03, 0x62, 0x2C, 0x9B, 0xDA, 0xEF, 0xA2, 0x35, 0x12, 0x43,
        0x84, 0x76, 0x16, 0xC6, 0x56, 0x95, 0x06, 0xCC, 0x01, 0xA9, 0xBD, 0xF6,
        0x75, 0x1A, 0x42, 0xF7, 0xBD, 0xA9, 0xB2, 0x36, 0x22, 0x5F, 0xC7, 0x5D,
        0x7F, 0xB4, 0xA3, 0x82, 0x01, 0x3E, 0x30, 0x82, 0x01, 0x3A, 0x30, 0x1D,
        0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0xEB, 0xD4, 0x4B,
        0x59, 0x6B, 0x95, 0x61, 0x3F, 0x51, 0x57, 0xB6, 0x04, 0x4D, 0x89, 0x41,
        0x88, 0x44, 0x5C, 0xAB, 0xF2, 0x30, 0x81, 0xCD, 0x06, 0x03, 0x55, 0x1D,
        0x23, 0x04, 0x81, 0xC5, 0x30, 0x81, 0xC2, 0x80, 0x14, 0xEB, 0xD4, 0x4B,
        0x59, 0x72, 0x95, 0x61, 0x3F, 0x51, 0x57, 0xB6, 0x04, 0x4D, 0x89, 0x41,
        0x88, 0x44, 0x5C, 0xAB, 0xF2, 0xA1, 0x81, 0x93, 0xA4, 0x81, 0x90, 0x30,
        0x81, 0x8D, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x08, 0x08,
        0x0C, 0x06, 0x4F, 0x72, 0x65, 0x67, 0x6F, 0x6E, 0x31, 0x0E, 0x30, 0x0C,
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x05, 0x53, 0x61, 0x6C, 0x65, 0x6D,
        0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0A, 0x43,
        0x6C, 0x69, 0x65, 0x6E, 0x74, 0x20, 0x45, 0x43, 0x43, 0x31, 0x0D, 0x30,
        0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x04, 0x46, 0x61, 0x73, 0x74,
        0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77,
        0x77, 0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
        0x6F, 0x6D, 0x30, 0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
        0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40,
        0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x82,
        0x14, 0x61, 0xB3, 0x1E, 0x59, 0xF3, 0x68, 0x6C, 0xA4, 0x79, 0x42, 0x83,
        0x2F, 0x1A, 0x50, 0x71, 0x03, 0xBE, 0x32, 0xAA, 0x2C, 0x30, 0x0C, 0x06,
        0x03, 0x55, 0x1D, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30,
        0x1C, 0x06, 0x03, 0x55, 0x1D, 0x11, 0x04, 0x15, 0x30, 0x13, 0x82, 0x0B,
        0x65, 0x78, 0x61, 0x6D, 0x70, 0x6C, 0x65, 0x2E, 0x63, 0x6F, 0x6D, 0x87,
        0x04, 0x23, 0x00, 0x00, 0x01, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x25,
        0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07,
        0x03, 0x01, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02,
        0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02,
        0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xE4, 0xA0, 0x23, 0x26,
        0x2B, 0x0B, 0x42, 0x0F, 0x97, 0x37, 0x6D, 0xCB, 0x14, 0x23, 0xC3, 0xC3,
        0xE6, 0x44, 0xCF, 0x5F, 0x4C, 0x26, 0xA3, 0x72, 0x64, 0x7A, 0x9C, 0xCB,
        0x64, 0xAB, 0xA6, 0xBE, 0x02, 0x21, 0x00, 0xAA, 0xC5, 0xA3, 0x50, 0xF6,
        0xF1, 0xA5, 0xDB, 0x05, 0xE0, 0x75, 0xD2, 0xF7, 0xBA, 0x49, 0x5F, 0x8F,
        0x7D, 0x1C, 0x44, 0xB1, 0x6E, 0xDF, 0xC8, 0xDA, 0x10, 0x48, 0x2D, 0x53,
        0x08, 0xA8, 0xB4
        };
#endif
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        /* If initialization is not successful, it's free'd in init func. */
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)cert, (word32)certSz),
            0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        /* Valid initialization usage. */
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

        /* Pass in bad args. No need free for null checks, free at end.*/
        ExpectIntEQ(wc_PKCS7_InitWithCert(NULL, (byte*)cert, (word32)certSz),
            BAD_FUNC_ARG);
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, (word32)certSz),
            BAD_FUNC_ARG);

#ifdef HAVE_ECC
        ExpectIntLT(wc_PKCS7_InitWithCert(pkcs7, certWithInvalidEccKey,
            sizeof(certWithInvalidEccKey)), 0);
    }
#endif

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_InitWithCert */


/*
 * Testing wc_PKCS7_EncodeData()
 */
static int test_wc_PKCS7_EncodeData(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;
    byte   output[FOURK_BUF];
    byte   data[] = "My encoded DER cert.";

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        unsigned char cert[sizeof(client_cert_der_2048)];
        unsigned char key[sizeof(client_key_der_2048)];
        int certSz = (int)sizeof(cert);
        int keySz = (int)sizeof(key);

        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, client_cert_der_2048, certSz);
        XMEMCPY(key, client_key_der_2048, keySz);
    #elif defined(USE_CERT_BUFFERS_1024)
        unsigned char cert[sizeof(sizeof_client_cert_der_1024)];
        unsigned char key[sizeof_client_key_der_1024];
        int certSz = (int)sizeof(cert);
        int keySz = (int)sizeof(key);

        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, client_cert_der_1024, certSz);
        XMEMCPY(key, client_key_der_1024, keySz);
    #else
        unsigned char cert[ONEK_BUF];
        unsigned char key[ONEK_BUF];
        XFILE         fp = XBADFILE;
        int           certSz;
        int           keySz;

        ExpectTrue((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/1024/client-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_client_key_der_1024, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        unsigned char    key[sizeof(ecc_clikey_der_256)];
        int              certSz = (int)sizeof(cert);
        int              keySz = (int)sizeof(key);
        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, cliecc_cert_der_256, sizeof_cliecc_cert_der_256);
        XMEMCPY(key, ecc_clikey_der_256, sizeof_ecc_clikey_der_256);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz, keySz;

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_cliecc_cert_der_256,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_ecc_clikey_der_256, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#endif

    XMEMSET(output, 0, sizeof(output));

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)cert, certSz), 0);

    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = sizeof(data);
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = keySz;
    }
    ExpectIntGT(wc_PKCS7_EncodeData(pkcs7, output, (word32)sizeof(output)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_PKCS7_EncodeData(NULL, output, (word32)sizeof(output)),
                                                            BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeData(pkcs7, NULL, (word32)sizeof(output)),
                                                            BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeData(pkcs7, output, 5), BUFFER_E);

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_PKCS7_EncodeData */


#if defined(HAVE_PKCS7) && defined(HAVE_PKCS7_RSA_RAW_SIGN_CALLBACK) && \
    !defined(NO_RSA) && !defined(NO_SHA256)
/* RSA sign raw digest callback */
static int rsaSignRawDigestCb(PKCS7* pkcs7, byte* digest, word32 digestSz,
                              byte* out, word32 outSz, byte* privateKey,
                              word32 privateKeySz, int devid, int hashOID)
{
    /* specific DigestInfo ASN.1 encoding prefix for a SHA2565 digest */
    byte digInfoEncoding[] = {
        0x30, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86,
        0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05,
        0x00, 0x04, 0x20
    };

    int ret;
    byte digestInfo[ONEK_BUF];
    byte sig[FOURK_BUF];
    word32 digestInfoSz = 0;
    word32 idx = 0;
    RsaKey rsa;

    /* SHA-256 required only for this example callback due to above
     * digInfoEncoding[] */
    if (pkcs7 == NULL || digest == NULL || out == NULL ||
        (sizeof(digestInfo) < sizeof(digInfoEncoding) + digestSz) ||
        (hashOID != SHA256h)) {
        return -1;
    }

    /* build DigestInfo */
    XMEMCPY(digestInfo, digInfoEncoding, sizeof(digInfoEncoding));
    digestInfoSz += sizeof(digInfoEncoding);
    XMEMCPY(digestInfo + digestInfoSz, digest, digestSz);
    digestInfoSz += digestSz;

    /* set up RSA key */
    ret = wc_InitRsaKey_ex(&rsa, pkcs7->heap, devid);
    if (ret != 0) {
        return ret;
    }

    ret = wc_RsaPrivateKeyDecode(privateKey, &idx, &rsa, privateKeySz);

    /* sign DigestInfo */
    if (ret == 0) {
        ret = wc_RsaSSL_Sign(digestInfo, digestInfoSz, sig, sizeof(sig),
                             &rsa, pkcs7->rng);
        if (ret > 0) {
            if (ret > (int)outSz) {
                /* output buffer too small */
                ret = -1;
            }
            else {
                /* success, ret holds sig size */
                XMEMCPY(out, sig, ret);
            }
        }
    }

    wc_FreeRsaKey(&rsa);

    return ret;
}
#endif


/*
 * Testing wc_PKCS7_EncodeSignedData()
 */
static int test_wc_PKCS7_EncodeSignedData(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;
    WC_RNG rng;
    byte   output[FOURK_BUF];
    byte   badOut[1];
    word32 outputSz = (word32)sizeof(output);
    word32 badOutSz = 0;
    byte   data[] = "Test data to encode.";
#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        byte        key[sizeof(client_key_der_2048)];
        byte        cert[sizeof(client_cert_der_2048)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_2048, keySz);
        XMEMCPY(cert, client_cert_der_2048, certSz);
    #elif defined(USE_CERT_BUFFERS_1024)
        byte        key[sizeof_client_key_der_1024];
        byte        cert[sizeof(sizeof_client_cert_der_1024)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_1024, keySz);
        XMEMCPY(cert, client_cert_der_1024, certSz);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz;
        int             keySz;

        ExpectTrue((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/1024/client-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_client_key_der_1024, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        unsigned char    key[sizeof(ecc_clikey_der_256)];
        int              certSz = (int)sizeof(cert);
        int              keySz = (int)sizeof(key);
        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, cliecc_cert_der_256, certSz);
        XMEMCPY(key, ecc_clikey_der_256, keySz);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz;
        int             keySz;

        ExpectTrue((fp = XOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, ONEK_BUF, fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, ONEK_BUF, fp), 0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#endif

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    XMEMSET(output, 0, outputSz);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = (word32)sizeof(data);
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = (word32)sizeof(key);
        pkcs7->encryptOID = RSAk;
    #ifdef NO_SHA
        pkcs7->hashOID = SHA256h;
    #else
        pkcs7->hashOID = SHAh;
    #endif
        pkcs7->rng = &rng;
    }

    ExpectIntGT(wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz), 0);

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);

    /* Pass in bad args. */
    ExpectIntEQ(wc_PKCS7_EncodeSignedData(NULL, output, outputSz),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeSignedData(pkcs7, NULL, outputSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeSignedData(pkcs7, badOut,
                                badOutSz), BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->hashOID = 0; /* bad hashOID */
    }
    ExpectIntEQ(wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz),
        BAD_FUNC_ARG);

#if defined(HAVE_PKCS7) && defined(HAVE_PKCS7_RSA_RAW_SIGN_CALLBACK) && \
    !defined(NO_RSA) && !defined(NO_SHA256)
    /* test RSA sign raw digest callback, if using RSA and compiled in.
     * Example callback assumes SHA-256, so only run test if compiled in. */
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = (word32)sizeof(data);
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = (word32)sizeof(key);
        pkcs7->encryptOID = RSAk;
        pkcs7->hashOID = SHA256h;
        pkcs7->rng = &rng;
    }

    ExpectIntEQ(wc_PKCS7_SetRsaSignRawDigestCb(pkcs7, rsaSignRawDigestCb), 0);

    ExpectIntGT(wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz), 0);
#endif

    wc_PKCS7_Free(pkcs7);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_EncodeSignedData */


/*
 * Testing wc_PKCS7_EncodeSignedData_ex() and wc_PKCS7_VerifySignedData_ex()
 */
static int test_wc_PKCS7_EncodeSignedData_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    int        i;
    PKCS7*     pkcs7 = NULL;
    WC_RNG     rng;
    byte       outputHead[FOURK_BUF/2];
    byte       outputFoot[FOURK_BUF/2];
    word32     outputHeadSz = (word32)sizeof(outputHead);
    word32     outputFootSz = (word32)sizeof(outputFoot);
    byte       data[FOURK_BUF];
    wc_HashAlg hash;
#ifdef NO_SHA
    enum wc_HashType hashType = WC_HASH_TYPE_SHA256;
#else
    enum wc_HashType hashType = WC_HASH_TYPE_SHA;
#endif
    byte        hashBuf[WC_MAX_DIGEST_SIZE];
    word32      hashSz = wc_HashGetDigestSize(hashType);

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        byte        key[sizeof(client_key_der_2048)];
        byte        cert[sizeof(client_cert_der_2048)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_2048, keySz);
        XMEMCPY(cert, client_cert_der_2048, certSz);
    #elif defined(USE_CERT_BUFFERS_1024)
        byte        key[sizeof_client_key_der_1024];
        byte        cert[sizeof(sizeof_client_cert_der_1024)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_1024, keySz);
        XMEMCPY(cert, client_cert_der_1024, certSz);
    #else
        unsigned char  cert[ONEK_BUF];
        unsigned char  key[ONEK_BUF];
        XFILE          fp = XBADFILE;
        int            certSz;
        int            keySz;

        ExpectTure((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/1024/client-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_client_key_der_1024, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char   cert[sizeof(cliecc_cert_der_256)];
        unsigned char   key[sizeof(ecc_clikey_der_256)];
        int             certSz = (int)sizeof(cert);
        int             keySz = (int)sizeof(key);

        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, cliecc_cert_der_256, sizeof_cliecc_cert_der_256);
        XMEMCPY(key, ecc_clikey_der_256, sizeof_ecc_clikey_der_256);
    #else
        unsigned char cert[ONEK_BUF];
        unsigned char key[ONEK_BUF];
        XFILE         fp = XBADFILE;
        int           certSz;
        int           keySz;

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_cliecc_cert_der_256,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_ecc_clikey_der_256, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#endif

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    /* initialize large data with sequence */
    for (i=0; i<(int)sizeof(data); i++)
        data[i] = i & 0xff;

    XMEMSET(outputHead, 0, outputHeadSz);
    XMEMSET(outputFoot, 0, outputFootSz);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    if (pkcs7 != NULL) {
        pkcs7->content = NULL; /* not used for ex */
        pkcs7->contentSz = (word32)sizeof(data);
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = (word32)sizeof(key);
        pkcs7->encryptOID = RSAk;
    #ifdef NO_SHA
        pkcs7->hashOID = SHA256h;
    #else
        pkcs7->hashOID = SHAh;
    #endif
        pkcs7->rng = &rng;
    }

    /* calculate hash for content */
    XMEMSET(&hash, 0, sizeof(wc_HashAlg));
    ExpectIntEQ(wc_HashInit(&hash, hashType), 0);
    ExpectIntEQ(wc_HashUpdate(&hash, hashType, data, sizeof(data)), 0);
    ExpectIntEQ(wc_HashFinal(&hash, hashType, hashBuf), 0);
    DoExpectIntEQ(wc_HashFree(&hash, hashType), 0);

    /* Perform PKCS7 sign using hash directly */
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, outputFoot, &outputFootSz), 0);
    ExpectIntGT(outputHeadSz, 0);
    ExpectIntGT(outputFootSz, 0);

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

    /* required parameter even on verify when using _ex, if using outputHead
     * and outputFoot */
    if (pkcs7 != NULL) {
        pkcs7->contentSz = (word32)sizeof(data);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, outputFoot, outputFootSz), 0);

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* assembly complete PKCS7 sign and use normal verify */
    {
        byte* output = NULL;
        word32 outputSz = 0;

        ExpectNotNull(output = (byte*)XMALLOC(
            outputHeadSz + sizeof(data) + outputFootSz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        if (output != NULL) {
            XMEMCPY(&output[outputSz], outputHead, outputHeadSz);
            outputSz += outputHeadSz;
            XMEMCPY(&output[outputSz], data, sizeof(data));
            outputSz += sizeof(data);
            XMEMCPY(&output[outputSz], outputFoot, outputFootSz);
            outputSz += outputFootSz;
        }

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
        ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
        XFREE(output, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }

    /* Pass in bad args. */
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(NULL, hashBuf, hashSz, outputHead,
        &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, NULL, hashSz, outputHead,
        &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, 0, outputHead,
        &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz, NULL,
        &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, NULL, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, NULL, &outputFootSz), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, outputFoot, NULL), BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->hashOID = 0; /* bad hashOID */
    }
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);

    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(NULL, hashBuf, hashSz, outputHead,
        outputHeadSz, outputFoot, outputFootSz), BAD_FUNC_ARG);

    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, NULL, hashSz, outputHead,
        outputHeadSz, outputFoot, outputFootSz), BAD_FUNC_ARG);
#ifndef NO_PKCS7_STREAM
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, 0, outputHead,
        outputHeadSz, outputFoot, outputFootSz), WC_PKCS7_WANT_READ_E);
#else
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, 0, outputHead,
        outputHeadSz, outputFoot, outputFootSz), BUFFER_E);
#endif
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz, NULL,
        outputHeadSz, outputFoot, outputFootSz), BAD_FUNC_ARG);
#ifndef NO_PKCS7_STREAM
    /* can pass in 0 buffer length with streaming API */
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, 0, outputFoot, outputFootSz), WC_PKCS7_WANT_READ_E);
#else
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, 0, outputFoot, outputFootSz), BAD_FUNC_ARG);
#endif
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, NULL, outputFootSz), BAD_FUNC_ARG);
#ifndef NO_PKCS7_STREAM
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, outputFoot, 0), WC_PKCS7_WANT_READ_E);
#else
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, outputFoot, 0), BUFFER_E);
#endif

    wc_PKCS7_Free(pkcs7);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_EncodeSignedData_ex */


#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM)

/**
 * Loads certs/keys from files or buffers into the argument buffers,
 * helper function called by CreatePKCS7SignedData().
 *
 * Returns 0 on success, negative on error.
 */
static int LoadPKCS7SignedDataCerts(
        int useIntermediateCertChain, int pkAlgoType,
        byte* intCARoot, word32* intCARootSz,
        byte* intCA1, word32* intCA1Sz,
        byte* intCA2, word32* intCA2Sz,
        byte* cert, word32* certSz,
        byte* key, word32* keySz)
{
    EXPECT_DECLS;
    int ret = 0;
    XFILE fp = XBADFILE;

#ifndef NO_RSA
    const char* intCARootRSA   = "./certs/ca-cert.der";
    const char* intCA1RSA      = "./certs/intermediate/ca-int-cert.der";
    const char* intCA2RSA      = "./certs/intermediate/ca-int2-cert.der";
    const char* intServCertRSA = "./certs/intermediate/server-int-cert.der";
    const char* intServKeyRSA  = "./certs/server-key.der";

    #if !defined(USE_CERT_BUFFERS_2048) && !defined(USE_CERT_BUFFERS_1024)
        const char* cli1024Cert    = "./certs/1024/client-cert.der";
        const char* cli1024Key     = "./certs/1024/client-key.der";
    #endif
#endif
#ifdef HAVE_ECC
    const char* intCARootECC   = "./certs/ca-ecc-cert.der";
    const char* intCA1ECC      = "./certs/intermediate/ca-int-ecc-cert.der";
    const char* intCA2ECC      = "./certs/intermediate/ca-int2-ecc-cert.der";
    const char* intServCertECC = "./certs/intermediate/server-int-ecc-cert.der";
    const char* intServKeyECC  = "./certs/ecc-key.der";

    #ifndef USE_CERT_BUFFERS_256
        const char* cliEccCert     = "./certs/client-ecc-cert.der";
        const char* cliEccKey      = "./certs/client-ecc-key.der";
    #endif
#endif

    if (cert == NULL || certSz == NULL || key == NULL || keySz == NULL ||
        ((useIntermediateCertChain == 1) &&
        (intCARoot == NULL || intCARootSz == NULL || intCA1 == NULL ||
         intCA1Sz == NULL || intCA2 == NULL || intCA2Sz == NULL))) {
        return BAD_FUNC_ARG;
    }

    /* Read/load certs and keys to use for signing based on PK type and chain */
    switch (pkAlgoType) {
#ifndef NO_RSA
        case RSA_TYPE:
            if (useIntermediateCertChain == 1) {
                ExpectTrue((fp = XFOPEN(intCARootRSA, "rb")) != XBADFILE);
                *intCARootSz = (word32)XFREAD(intCARoot, 1, *intCARootSz, fp);
                if (fp != XBADFILE) {
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCARootSz, 0);

                ExpectTrue((fp = XFOPEN(intCA1RSA, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCA1Sz = (word32)XFREAD(intCA1, 1, *intCA1Sz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCA1Sz, 0);

                ExpectTrue((fp = XFOPEN(intCA2RSA, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCA2Sz = (word32)XFREAD(intCA2, 1, *intCA2Sz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCA2Sz, 0);

                ExpectTrue((fp = XFOPEN(intServCertRSA, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*certSz, 0);

                ExpectTrue((fp = XFOPEN(intServKeyRSA, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*keySz, 0);
            }
            else {
            #if defined(USE_CERT_BUFFERS_2048)
                *keySz  = sizeof_client_key_der_2048;
                *certSz = sizeof_client_cert_der_2048;
                XMEMCPY(key, client_key_der_2048, *keySz);
                XMEMCPY(cert, client_cert_der_2048, *certSz);
            #elif defined(USE_CERT_BUFFERS_1024)
                *keySz  = sizeof_client_key_der_1024;
                *certSz = sizeof_client_cert_der_1024;
                XMEMCPY(key, client_key_der_1024, *keySz);
                XMEMCPY(cert, client_cert_der_1024, *certSz);
            #else
                ExpectTrue((fp = XFOPEN(cli1024Key, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*keySz, 0);

                ExpectTrue((fp = XFOPEN(cli1024Cert, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*certSz, 0);
            #endif /* USE_CERT_BUFFERS_2048 */
            }
            break;
#endif /* !NO_RSA */
#ifdef HAVE_ECC
        case ECC_TYPE:
            if (useIntermediateCertChain == 1) {
                ExpectTrue((fp = XFOPEN(intCARootECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCARootSz = (word32)XFREAD(intCARoot, 1, *intCARootSz,
                                                  fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCARootSz, 0);

                ExpectTrue((fp = XFOPEN(intCA1ECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCA1Sz = (word32)XFREAD(intCA1, 1, *intCA1Sz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCA1Sz, 0);

                ExpectTrue((fp = XFOPEN(intCA2ECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCA2Sz = (word32)XFREAD(intCA2, 1, *intCA2Sz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCA2Sz, 0);

                ExpectTrue((fp = XFOPEN(intServCertECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*certSz, 0);

                ExpectTrue((fp = XFOPEN(intServKeyECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*keySz, 0);
            }
            else {
            #if defined(USE_CERT_BUFFERS_256)
                *keySz  = sizeof_ecc_clikey_der_256;
                *certSz = sizeof_cliecc_cert_der_256;
                XMEMCPY(key, ecc_clikey_der_256, *keySz);
                XMEMCPY(cert, cliecc_cert_der_256, *certSz);
            #else
                ExpectTrue((fp = XFOPEN(cliEccKey, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*keySz, 0);

                ExpectTrue((fp = XFOPEN(cliEccCert, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*certSz, 0);
            #endif /* USE_CERT_BUFFERS_256 */
            }
            break;
#endif /* HAVE_ECC */
        default:
            WOLFSSL_MSG("Unsupported SignedData PK type");
            ret = BAD_FUNC_ARG;
            break;
    }

    if (EXPECT_FAIL() && (ret == 0)) {
        ret = BAD_FUNC_ARG;
    }
    return ret;
}

/**
 * Creates a PKCS7/CMS SignedData bundle to use for testing.
 *
 * output          output buffer to place SignedData
 * outputSz        size of output buffer
 * data            data buffer to be signed
 * dataSz          size of data buffer
 * withAttribs     [1/0] include attributes in SignedData message
 * detachedSig     [1/0] create detached signature, no content
 * useIntCertChain [1/0] use certificate chain and include intermediate and
 *                 root CAs in bundle
 * pkAlgoType      RSA_TYPE or ECC_TYPE, choose what key/cert type to use
 *
 * Return size of bundle created on success, negative on error */
static int CreatePKCS7SignedData(unsigned char* output, int outputSz,
                                 byte* data, word32 dataSz,
                                 int withAttribs, int detachedSig,
                                 int useIntermediateCertChain,
                                 int pkAlgoType)
{
    EXPECT_DECLS;
    int ret = 0;
    WC_RNG rng;
    PKCS7* pkcs7 = NULL;

    static byte messageTypeOid[] =
               { 0x06, 0x0a, 0x60, 0x86, 0x48, 0x01, 0x86, 0xF8, 0x45, 0x01,
                 0x09, 0x02 };
    static byte messageType[] = { 0x13, 2, '1', '9' };

    PKCS7Attrib attribs[] =
    {
        { messageTypeOid, sizeof(messageTypeOid), messageType,
                                       sizeof(messageType) }
    };

    byte intCARoot[TWOK_BUF];
    byte intCA1[TWOK_BUF];
    byte intCA2[TWOK_BUF];
    byte cert[TWOK_BUF];
    byte key[TWOK_BUF];

    word32 intCARootSz = sizeof(intCARoot);
    word32 intCA1Sz    = sizeof(intCA1);
    word32 intCA2Sz    = sizeof(intCA2);
    word32 certSz      = sizeof(cert);
    word32 keySz       = sizeof(key);

    XMEMSET(intCARoot, 0, intCARootSz);
    XMEMSET(intCA1, 0, intCA1Sz);
    XMEMSET(intCA2, 0, intCA2Sz);
    XMEMSET(cert, 0, certSz);
    XMEMSET(key, 0, keySz);

    ret = LoadPKCS7SignedDataCerts(useIntermediateCertChain, pkAlgoType,
                intCARoot, &intCARootSz, intCA1, &intCA1Sz, intCA2, &intCA2Sz,
                cert, &certSz, key, &keySz);
    ExpectIntEQ(ret, 0);

    XMEMSET(output, 0, outputSz);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    if (useIntermediateCertChain == 1) {
        /* Add intermediate and root CA certs into SignedData Certs SET */
        ExpectIntEQ(wc_PKCS7_AddCertificate(pkcs7, intCA2, intCA2Sz), 0);
        ExpectIntEQ(wc_PKCS7_AddCertificate(pkcs7, intCA1, intCA1Sz), 0);
        ExpectIntEQ(wc_PKCS7_AddCertificate(pkcs7, intCARoot, intCARootSz), 0);
    }

    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = dataSz;
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = (word32)sizeof(key);
        if (pkAlgoType == RSA_TYPE) {
            pkcs7->encryptOID = RSAk;
        }
        else {
            pkcs7->encryptOID = ECDSAk;
        }
    #ifdef NO_SHA
        pkcs7->hashOID = SHA256h;
    #else
        pkcs7->hashOID = SHAh;
    #endif
        pkcs7->rng = &rng;
        if (withAttribs) {
            /* include a signed attribute */
            pkcs7->signedAttribs   = attribs;
            pkcs7->signedAttribsSz = (sizeof(attribs)/sizeof(PKCS7Attrib));
        }
    }

    if (detachedSig) {
        ExpectIntEQ(wc_PKCS7_SetDetached(pkcs7, 1), 0);
    }

    outputSz = wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz);
    ExpectIntGT(outputSz, 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (detachedSig && (pkcs7 != NULL)) {
        pkcs7->content = data;
        pkcs7->contentSz = dataSz;
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);

    wc_PKCS7_Free(pkcs7);
    wc_FreeRng(&rng);

    if (EXPECT_FAIL()) {
        outputSz = 0;
    }
    return outputSz;
}
#endif

/*
 * Testing wc_PKCS_VerifySignedData()
 */
static int test_wc_PKCS7_VerifySignedData_RSA(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    PKCS7* pkcs7 = NULL;
    byte   output[6000]; /* Large size needed for bundles with int CA certs */
    word32 outputSz = sizeof(output);
    byte   data[] = "Test data to encode.";
    byte   badOut[1];
    word32 badOutSz = 0;
    byte   badContent[] = "This is different content than was signed";
    wc_HashAlg hash;
#ifdef NO_SHA
    enum wc_HashType hashType = WC_HASH_TYPE_SHA256;
#else
    enum wc_HashType hashType = WC_HASH_TYPE_SHA;
#endif
    byte        hashBuf[WC_MAX_DIGEST_SIZE];
    word32      hashSz = wc_HashGetDigestSize(hashType);
#ifndef NO_RSA
    PKCS7DecodedAttrib* decodedAttrib = NULL;
    /* contentType OID (1.2.840.113549.1.9.3) */
    static const byte contentTypeOid[] =
        { 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xF7, 0x0d, 0x01, 0x09, 0x03 };

    /* PKCS#7 DATA content type (contentType defaults to DATA) */
    static const byte dataType[] =
        { 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01 };

    /* messageDigest OID (1.2.840.113549.1.9.4) */
    static const byte messageDigestOid[] =
        { 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x04 };
#ifndef NO_ASN_TIME
    /* signingTime OID () */
    static const byte signingTimeOid[] =
        { 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x05};
#endif
#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
    int dateLength = 0;
    byte dateFormat;
    const byte* datePart = NULL;
    struct tm timearg;
    time_t now;
    struct tm* nowTm = NULL;
#ifdef NEED_TMP_TIME
    struct tm tmpTimeStorage;
    struct tm* tmpTime = &tmpTimeStorage;
#endif
#endif /* !NO_ASN && !NO_ASN_TIME */

    XMEMSET(&hash, 0, sizeof(wc_HashAlg));

    /* Success test with RSA certs/key */
    ExpectIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
        (word32)sizeof(data), 0, 0, 0, RSA_TYPE)), 0);

    /* calculate hash for content, used later */
    ExpectIntEQ(wc_HashInit(&hash, hashType), 0);
    ExpectIntEQ(wc_HashUpdate(&hash, hashType, data, sizeof(data)), 0);
    ExpectIntEQ(wc_HashFinal(&hash, hashType, hashBuf), 0);
    DoExpectIntEQ(wc_HashFree(&hash, hashType), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);

    /* Check that decoded signed attributes are correct */

    /* messageDigest should be first */
    if (pkcs7 != NULL) {
        decodedAttrib = pkcs7->decodedAttrib;
    }
    ExpectNotNull(decodedAttrib);
    ExpectIntEQ(decodedAttrib->oidSz, (word32)sizeof(messageDigestOid));
    ExpectIntEQ(XMEMCMP(decodedAttrib->oid, messageDigestOid,
        decodedAttrib->oidSz), 0);
    /* + 2 for OCTET STRING and length bytes */
    ExpectIntEQ(decodedAttrib->valueSz, hashSz + 2);
    ExpectNotNull(decodedAttrib->value);
    ExpectIntEQ(XMEMCMP(decodedAttrib->value + 2, hashBuf, hashSz), 0);

#ifndef NO_ASN_TIME
    /* signingTime should be second */
    if (decodedAttrib != NULL) {
        decodedAttrib = decodedAttrib->next;
    }
    ExpectNotNull(decodedAttrib);
    ExpectIntEQ(decodedAttrib->oidSz, (word32)sizeof(signingTimeOid));
    ExpectIntEQ(XMEMCMP(decodedAttrib->oid, signingTimeOid,
        decodedAttrib->oidSz), 0);

    ExpectIntGT(decodedAttrib->valueSz, 0);
    ExpectNotNull(decodedAttrib->value);
#endif

    /* Verify signingTime if ASN and time are available */
#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
    ExpectIntEQ(wc_GetDateInfo(decodedAttrib->value, decodedAttrib->valueSz,
        &datePart, &dateFormat, &dateLength), 0);
    ExpectNotNull(datePart);
    ExpectIntGT(dateLength, 0);
    XMEMSET(&timearg, 0, sizeof(timearg));
    ExpectIntEQ(wc_GetDateAsCalendarTime(datePart, dateLength, dateFormat,
         &timearg), 0);

    /* Get current time and compare year/month/day against attribute value */
    ExpectIntEQ(wc_GetTime(&now, sizeof(now)), 0);
    nowTm = (struct tm*)XGMTIME((time_t*)&now, tmpTime);
    ExpectNotNull(nowTm);

    ExpectIntEQ(timearg.tm_year, nowTm->tm_year);
    ExpectIntEQ(timearg.tm_mon, nowTm->tm_mon);
    ExpectIntEQ(timearg.tm_mday, nowTm->tm_mday);
#endif /* !NO_ASN && !NO_ASN_TIME */

    /* contentType should be third */
    if (decodedAttrib != NULL) {
        decodedAttrib = decodedAttrib->next;
    }
    ExpectNotNull(decodedAttrib);
    ExpectIntEQ(decodedAttrib->oidSz, (word32)sizeof(contentTypeOid));
    ExpectIntEQ(XMEMCMP(decodedAttrib->oid, contentTypeOid,
        decodedAttrib->oidSz), 0);
    ExpectIntEQ(decodedAttrib->valueSz, (int)sizeof(dataType) + 2);
    ExpectNotNull(decodedAttrib->value);
    ExpectIntEQ(XMEMCMP(decodedAttrib->value + 2, dataType, sizeof(dataType)),
        0);
#endif /* !NO_RSA */

    /* Test bad args. */
    ExpectIntEQ(wc_PKCS7_VerifySignedData(NULL, output, outputSz),
                                          BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, NULL, outputSz),
                                          BAD_FUNC_ARG);
    #ifndef NO_PKCS7_STREAM
        /* can pass in 0 buffer length with streaming API */
        ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, badOut,
                                    badOutSz), WC_PKCS7_WANT_READ_E);
    #else
        ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, badOut,
                                    badOutSz), BAD_FUNC_ARG);
    #endif
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_RSA
    /* Try RSA certs/key/sig first */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
                                                  (word32)sizeof(data),
                                                  1, 1, 0, RSA_TYPE)), 0);
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = badContent;
        pkcs7->contentSz = sizeof(badContent);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz),
                SIG_VERIFY_E);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* Test success case with detached signature and valid content */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = sizeof(data);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* verify using pre-computed content digest only (no content) */
    {
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_Init(pkcs7, NULL, 0), 0);
        ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
            output, outputSz, NULL, 0), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
    }
#endif /* !NO_RSA */

    /* Test verify on signedData containing intermediate/root CA certs */
#ifndef NO_RSA
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
                                                  (word32)sizeof(data),
                                                  0, 0, 1, RSA_TYPE)), 0);
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif /* !NO_RSA */
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_VerifySignedData()_RSA */

/*
 * Testing wc_PKCS_VerifySignedData()
 */
static int test_wc_PKCS7_VerifySignedData_ECC(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && defined(HAVE_ECC)
    PKCS7* pkcs7 = NULL;
    byte   output[6000]; /* Large size needed for bundles with int CA certs */
    word32 outputSz = sizeof(output);
    byte   data[] = "Test data to encode.";
    byte   badContent[] = "This is different content than was signed";
    wc_HashAlg hash;
#ifdef NO_SHA
    enum wc_HashType hashType = WC_HASH_TYPE_SHA256;
#else
    enum wc_HashType hashType = WC_HASH_TYPE_SHA;
#endif
    byte        hashBuf[WC_MAX_DIGEST_SIZE];
    word32      hashSz = wc_HashGetDigestSize(hashType);

    XMEMSET(&hash, 0, sizeof(wc_HashAlg));

    /* Success test with ECC certs/key */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
        (word32)sizeof(data), 0, 0, 0, ECC_TYPE)), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* Invalid content should error, use detached signature so we can
     * easily change content */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
        (word32)sizeof(data), 1, 1, 0, ECC_TYPE)), 0);
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = badContent;
        pkcs7->contentSz = sizeof(badContent);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz),
        SIG_VERIFY_E);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* Test success case with detached signature and valid content */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = sizeof(data);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* verify using pre-computed content digest only (no content) */
    {
        /* calculate hash for content */
        ExpectIntEQ(wc_HashInit(&hash, hashType), 0);
        ExpectIntEQ(wc_HashUpdate(&hash, hashType, data, sizeof(data)), 0);
        ExpectIntEQ(wc_HashFinal(&hash, hashType, hashBuf), 0);
        ExpectIntEQ(wc_HashFree(&hash, hashType), 0);

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_Init(pkcs7, NULL, 0), 0);
        ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
            output, outputSz, NULL, 0), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
    }

    /* Test verify on signedData containing intermediate/root CA certs */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
        (word32)sizeof(data), 0, 0, 1, ECC_TYPE)), 0);
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_VerifySignedData_ECC() */


#if defined(HAVE_PKCS7) && !defined(NO_AES) && defined(HAVE_AES_CBC) && \
    !defined(NO_AES_256)
static const byte defKey[] = {
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
};
static byte aesHandle[32]; /* simulated hardware key handle */

/* return 0 on success */
static int myDecryptionFunc(PKCS7* pkcs7, int encryptOID, byte* iv, int ivSz,
        byte* aad, word32 aadSz, byte* authTag, word32 authTagSz,
        byte* in, int inSz, byte* out, void* usrCtx)
{
    int ret;
    Aes aes;

    if (usrCtx == NULL) {
        /* no simulated handle passed in */
        return -1;
    }

    switch (encryptOID) {
        case AES256CBCb:
            if (ivSz  != AES_BLOCK_SIZE)
                return BAD_FUNC_ARG;
            break;

        default:
            WOLFSSL_MSG("Unsupported content cipher type for test");
            return ALGO_ID_E;
    };

    /* simulate using handle to get key */
    ret = wc_AesInit(&aes, HEAP_HINT, INVALID_DEVID);
    if (ret == 0) {
        ret = wc_AesSetKey(&aes, (byte*)usrCtx, 32, iv, AES_DECRYPTION);
        if (ret == 0)
            ret = wc_AesCbcDecrypt(&aes, out, in, inSz);
        wc_AesFree(&aes);
    }

    (void)aad;
    (void)aadSz;
    (void)authTag;
    (void)authTagSz;
    (void)pkcs7;
    return ret;
}


/* returns key size on success */
static int myCEKwrapFunc(PKCS7* pkcs7, byte* cek, word32 cekSz, byte* keyId,
        word32 keyIdSz, byte* orginKey, word32 orginKeySz,
        byte* out, word32 outSz, int keyWrapAlgo, int type, int direction)
{
    int ret = -1;

    if (out == NULL)
        return BAD_FUNC_ARG;

    if (keyId[0] != 0x00) {
        return -1;
    }

    if (type != (int)PKCS7_KEKRI) {
        return -1;
    }

    switch (keyWrapAlgo) {
        case AES256_WRAP:
            /* simulate setting a handle for later decryption but use key
             * as handle in the test case here */
            ret = wc_AesKeyUnWrap(defKey, sizeof(defKey), cek, cekSz,
                                      aesHandle, sizeof(aesHandle), NULL);
            if (ret < 0)
                return ret;

            ret = wc_PKCS7_SetDecodeEncryptedCtx(pkcs7, (void*)aesHandle);
            if (ret < 0)
                return ret;

            /* return key size on success */
            return sizeof(defKey);

        default:
            WOLFSSL_MSG("Unsupported key wrap algorithm in example");
            return BAD_KEYWRAP_ALG_E;
    };

    (void)cekSz;
    (void)cek;
    (void)outSz;
    (void)keyIdSz;
    (void)direction;
    (void)orginKey; /* used with KAKRI */
    (void)orginKeySz;
    return ret;
}
#endif /* HAVE_PKCS7 && !NO_AES && HAVE_AES_CBC && !NO_AES_256 */


/*
 * Testing wc_PKCS7_EncodeEnvelopedData()
 */
static int test_wc_PKCS7_EncodeDecodeEnvelopedData(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7*      pkcs7 = NULL;
#ifdef ECC_TIMING_RESISTANT
    WC_RNG      rng;
#endif
    word32      tempWrd32   = 0;
    byte*       tmpBytePtr = NULL;
    const char  input[] = "Test data to encode.";
    int         i;
    int         testSz = 0;
    #if !defined(NO_RSA) && (!defined(NO_AES) || (!defined(NO_SHA) || \
        !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))
        byte*   rsaCert     = NULL;
        byte*   rsaPrivKey  = NULL;
        word32  rsaCertSz;
        word32  rsaPrivKeySz;
        #if !defined(NO_FILESYSTEM) && (!defined(USE_CERT_BUFFERS_1024) && \
                                           !defined(USE_CERT_BUFFERS_2048) )
            static const char* rsaClientCert = "./certs/client-cert.der";
            static const char* rsaClientKey = "./certs/client-key.der";
            rsaCertSz = (word32)sizeof(rsaClientCert);
            rsaPrivKeySz = (word32)sizeof(rsaClientKey);
        #endif
    #endif
    #if defined(HAVE_ECC) && (!defined(NO_AES) || (!defined(NO_SHA) ||\
        !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))
        byte*   eccCert     = NULL;
        byte*   eccPrivKey  = NULL;
        word32  eccCertSz;
        word32  eccPrivKeySz;
        #if !defined(NO_FILESYSTEM) && !defined(USE_CERT_BUFFERS_256)
            static const char* eccClientCert = "./certs/client-ecc-cert.der";
            static const char* eccClientKey = "./certs/ecc-client-key.der";
        #endif
    #endif
    /* Generic buffer size. */
    byte    output[ONEK_BUF];
    byte    decoded[sizeof(input)/sizeof(char)];
    int     decodedSz = 0;
#ifndef NO_FILESYSTEM
    XFILE certFile = XBADFILE;
    XFILE keyFile = XBADFILE;
#endif

#ifdef ECC_TIMING_RESISTANT
    XMEMSET(&rng, 0, sizeof(WC_RNG));
#endif

#if !defined(NO_RSA) && (!defined(NO_AES) || (!defined(NO_SHA) ||\
    !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))
    /* RSA certs and keys. */
    #if defined(USE_CERT_BUFFERS_1024)
        rsaCertSz = (word32)sizeof_client_cert_der_1024;
        /* Allocate buffer space. */
        ExpectNotNull(rsaCert = (byte*)XMALLOC(rsaCertSz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        /* Init buffer. */
        if (rsaCert != NULL) {
            XMEMCPY(rsaCert, client_cert_der_1024, rsaCertSz);
        }
        rsaPrivKeySz = (word32)sizeof_client_key_der_1024;
        ExpectNotNull(rsaPrivKey = (byte*)XMALLOC(rsaPrivKeySz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        if (rsaPrivKey != NULL) {
            XMEMCPY(rsaPrivKey, client_key_der_1024, rsaPrivKeySz);
        }
    #elif defined(USE_CERT_BUFFERS_2048)
        rsaCertSz = (word32)sizeof_client_cert_der_2048;
        /* Allocate buffer */
        ExpectNotNull(rsaCert = (byte*)XMALLOC(rsaCertSz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        /* Init buffer. */
        if (rsaCert != NULL) {
            XMEMCPY(rsaCert, client_cert_der_2048, rsaCertSz);
        }
        rsaPrivKeySz = (word32)sizeof_client_key_der_2048;
        ExpectNotNull(rsaPrivKey = (byte*)XMALLOC(rsaPrivKeySz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        if (rsaPrivKey != NULL) {
            XMEMCPY(rsaPrivKey, client_key_der_2048, rsaPrivKeySz);
        }
    #else
        /* File system. */
        ExpectTrue((certFile = XFOPEN(rsaClientCert, "rb")) != XBADFILE);
        rsaCertSz = (word32)FOURK_BUF;
        ExpectNotNull(rsaCert = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        ExpectTrue((rsaCertSz = (word32)XFREAD(rsaCert, 1, rsaCertSz,
            certFile)) > 0);
        if (certFile != XBADFILE)
            XFCLOSE(certFile);
        ExpectTrue((keyFile = XFOPEN(rsaClientKey, "rb")) != XBADFILE);
        ExpectNotNull(rsaPrivKey = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        rsaPrivKeySz = (word32)FOURK_BUF;
        ExpectTrue((rsaPrivKeySz = (word32)XFREAD(rsaPrivKey, 1, rsaPrivKeySz,
            keyFile)) > 0);
        if (keyFile != XBADFILE)
            XFCLOSE(keyFile);
    #endif /* USE_CERT_BUFFERS */
#endif /* NO_RSA */

/* ECC */
#if defined(HAVE_ECC) && (!defined(NO_AES) || (!defined(NO_SHA) ||\
    !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))

    #ifdef USE_CERT_BUFFERS_256
        ExpectNotNull(eccCert = (byte*)XMALLOC(TWOK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        /* Init buffer. */
        eccCertSz = (word32)sizeof_cliecc_cert_der_256;
        if (eccCert != NULL) {
            XMEMCPY(eccCert, cliecc_cert_der_256, eccCertSz);
        }
        ExpectNotNull(eccPrivKey = (byte*)XMALLOC(TWOK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        eccPrivKeySz = (word32)sizeof_ecc_clikey_der_256;
        if (eccPrivKey != NULL) {
            XMEMCPY(eccPrivKey, ecc_clikey_der_256, eccPrivKeySz);
        }
    #else /* File system. */
        ExpectTrue((certFile = XFOPEN(eccClientCert, "rb")) != XBADFILE);
        eccCertSz = (word32)FOURK_BUF;
        ExpectNotNull(eccCert = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        ExpectTrue((eccCertSz = (word32)XFREAD(eccCert, 1, eccCertSz,
            certFile)) > 0);
        if (certFile != XBADFILE) {
            XFCLOSE(certFile);
        }
        ExpectTrue((keyFile = XFOPEN(eccClientKey, "rb")) != XBADFILE);
        eccPrivKeySz = (word32)FOURK_BUF;
        ExpectNotNull(eccPrivKey = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        ExpectTrue((eccPrivKeySz = (word32)XFREAD(eccPrivKey, 1, eccPrivKeySz,
            keyFile)) > 0);
        if (keyFile != XBADFILE) {
            XFCLOSE(keyFile);
        }
    #endif /* USE_CERT_BUFFERS_256 */
#endif /* END HAVE_ECC */

#ifndef NO_FILESYSTEM
    /* Silence. */
    (void)keyFile;
    (void)certFile;
#endif

    {
    const pkcs7EnvelopedVector testVectors[] = {
    /* DATA is a global variable defined in the makefile. */
#if !defined(NO_RSA)
    #ifndef NO_DES3
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, DES3b, 0, 0,
            rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
    #endif /* NO_DES3 */
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #ifndef NO_AES_128
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES128CBCb,
            0, 0, rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
        #endif
        #ifndef NO_AES_192
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES192CBCb,
            0, 0, rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
        #endif
        #ifndef NO_AES_256
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES256CBCb,
            0, 0, rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
        #endif
    #endif /* NO_AES && HAVE_AES_CBC */

#endif /* NO_RSA */
#if defined(HAVE_ECC)
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #if !defined(NO_SHA) && !defined(NO_AES_128)
            {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA,
                AES128CBCb, AES128_WRAP, dhSinglePass_stdDH_sha1kdf_scheme,
                eccCert, eccCertSz, eccPrivKey, eccPrivKeySz},
        #endif
        #if !defined(NO_SHA256) && !defined(NO_AES_256)
            {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA,
                AES256CBCb, AES256_WRAP, dhSinglePass_stdDH_sha256kdf_scheme,
                eccCert, eccCertSz, eccPrivKey, eccPrivKeySz},
        #endif
        #if defined(WOLFSSL_SHA512) && !defined(NO_AES_256)
            {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA,
                AES256CBCb, AES256_WRAP, dhSinglePass_stdDH_sha512kdf_scheme,
                eccCert, eccCertSz, eccPrivKey, eccPrivKeySz},
        #endif
    #endif /* NO_AES && HAVE_AES_CBC*/
#endif /* END HAVE_ECC */
    }; /* END pkcs7EnvelopedVector */

#ifdef ECC_TIMING_RESISTANT
    ExpectIntEQ(wc_InitRng(&rng), 0);
#endif

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);

    testSz = (int)sizeof(testVectors)/(int)sizeof(pkcs7EnvelopedVector);
    for (i = 0; i < testSz; i++) {
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (testVectors + i)->cert,
                                    (word32)(testVectors + i)->certSz), 0);
        if (pkcs7 != NULL) {
#ifdef ECC_TIMING_RESISTANT
            pkcs7->rng = &rng;
#endif

            pkcs7->content       = (byte*)(testVectors + i)->content;
            pkcs7->contentSz     = (testVectors + i)->contentSz;
            pkcs7->contentOID    = (testVectors + i)->contentOID;
            pkcs7->encryptOID    = (testVectors + i)->encryptOID;
            pkcs7->keyWrapOID    = (testVectors + i)->keyWrapOID;
            pkcs7->keyAgreeOID   = (testVectors + i)->keyAgreeOID;
            pkcs7->privateKey    = (testVectors + i)->privateKey;
            pkcs7->privateKeySz  = (testVectors + i)->privateKeySz;
        }

        ExpectIntGE(wc_PKCS7_EncodeEnvelopedData(pkcs7, output,
            (word32)sizeof(output)), 0);

        decodedSz = wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
            (word32)sizeof(output), decoded, (word32)sizeof(decoded));
        ExpectIntGE(decodedSz, 0);
        /* Verify the size of each buffer. */
        ExpectIntEQ((word32)sizeof(input)/sizeof(char), decodedSz);
        /* Don't free the last time through the loop. */
        if (i < testSz - 1) {
            wc_PKCS7_Free(pkcs7);
            pkcs7 = NULL;
            ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        }
    }  /* END test loop. */
    }

    /* Test bad args. */
    ExpectIntEQ(wc_PKCS7_EncodeEnvelopedData(NULL, output,
                    (word32)sizeof(output)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeEnvelopedData(pkcs7, NULL,
                    (word32)sizeof(output)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeEnvelopedData(pkcs7, output, 0), BAD_FUNC_ARG);

    /* Decode.  */
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(NULL, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), NULL, (word32)sizeof(decoded)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, 0), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, NULL,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output, 0, decoded,
        (word32)sizeof(decoded)), BAD_FUNC_ARG);
    /* Should get a return of BAD_FUNC_ARG with structure data. Order matters.*/
#if defined(HAVE_ECC) && !defined(NO_AES) && defined(HAVE_AES_CBC)
    /* only a failure for KARI test cases */
    if (pkcs7 != NULL) {
        tempWrd32 = pkcs7->singleCertSz;
        pkcs7->singleCertSz = 0;
    }
    #if defined(WOLFSSL_ASN_TEMPLATE)
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        BUFFER_E);
    #else
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        ASN_PARSE_E);
    #endif
    if (pkcs7 != NULL) {
        pkcs7->singleCertSz = tempWrd32;

        tmpBytePtr = pkcs7->singleCert;
        pkcs7->singleCert = NULL;
    }
    #if defined(NO_PKCS7_STREAM)
    /* when none streaming mode is used and PKCS7 is in bad state buffer error
     * is returned from kari parse which gets set to bad func arg */
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        BAD_FUNC_ARG);
    #else
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        ASN_PARSE_E);
    #endif
    if (pkcs7 != NULL) {
        pkcs7->singleCert = tmpBytePtr;
    }
#endif
    if (pkcs7 != NULL) {
        tempWrd32 = pkcs7->privateKeySz;
        pkcs7->privateKeySz = 0;
    }

    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->privateKeySz = tempWrd32;

        tmpBytePtr = pkcs7->privateKey;
        pkcs7->privateKey = NULL;
    }
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->privateKey = tmpBytePtr;
    }

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#if !defined(NO_AES) && defined(HAVE_AES_CBC) && !defined(NO_AES_256)
    /* test of decrypt callback with KEKRI enveloped data */
    {
        int envelopedSz = 0;
        const byte keyId[] = { 0x00 };

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        if (pkcs7 != NULL) {
            pkcs7->content      = (byte*)input;
            pkcs7->contentSz    = (word32)(sizeof(input)/sizeof(char));
            pkcs7->contentOID   = DATA;
            pkcs7->encryptOID   = AES256CBCb;
        }
        ExpectIntGT(wc_PKCS7_AddRecipient_KEKRI(pkcs7, AES256_WRAP,
                    (byte*)defKey, sizeof(defKey), (byte*)keyId,
                    sizeof(keyId), NULL, NULL, 0, NULL, 0, 0), 0);
        ExpectIntEQ(wc_PKCS7_SetSignerIdentifierType(pkcs7, CMS_SKID), 0);
        ExpectIntGT((envelopedSz = wc_PKCS7_EncodeEnvelopedData(pkcs7, output,
                        (word32)sizeof(output))), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;

        /* decode envelopedData */
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_SetWrapCEKCb(pkcs7, myCEKwrapFunc), 0);
        ExpectIntEQ(wc_PKCS7_SetDecodeEncryptedCb(pkcs7, myDecryptionFunc), 0);
        ExpectIntGT((decodedSz = wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
                        envelopedSz, decoded, sizeof(decoded))), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
    }
#endif /* !NO_AES && !NO_AES_256 */

#ifndef NO_RSA
    XFREE(rsaCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(rsaPrivKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* NO_RSA */
#ifdef HAVE_ECC
    XFREE(eccCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(eccPrivKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* HAVE_ECC */

#ifdef ECC_TIMING_RESISTANT
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif

#if defined(USE_CERT_BUFFERS_2048) && !defined(NO_DES3) && \
    !defined(NO_RSA) && !defined(NO_SHA)
    {
        byte   out[7];
        byte   *cms = NULL;
        word32 cmsSz;
        XFILE  cmsFile = XBADFILE;

        XMEMSET(out, 0, sizeof(out));
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectTrue((cmsFile = XFOPEN("./certs/test/ktri-keyid-cms.msg", "rb"))
            != XBADFILE);
        cmsSz = (word32)FOURK_BUF;
        ExpectNotNull(cms = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        ExpectTrue((cmsSz = (word32)XFREAD(cms, 1, cmsSz, cmsFile)) > 0);
        if (cmsFile != XBADFILE)
            XFCLOSE(cmsFile);

        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)client_cert_der_2048,
            sizeof_client_cert_der_2048), 0);
        if (pkcs7 != NULL) {
            pkcs7->privateKey   = (byte*)client_key_der_2048;
            pkcs7->privateKeySz = sizeof_client_key_der_2048;
        }
        ExpectIntLT(wc_PKCS7_DecodeEnvelopedData(pkcs7, cms, cmsSz, out,
            2), 0);
        ExpectIntGT(wc_PKCS7_DecodeEnvelopedData(pkcs7, cms, cmsSz, out,
            sizeof(out)), 0);
        XFREE(cms, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        ExpectIntEQ(XMEMCMP(out, "test", 4), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
    }
#endif /* USE_CERT_BUFFERS_2048 && !NO_DES3 && !NO_RSA && !NO_SHA */
#endif /* HAVE_PKCS7 */
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_EncodeDecodeEnvelopedData() */


/*
 * Testing wc_PKCS7_EncodeEncryptedData()
 */
static int test_wc_PKCS7_EncodeEncryptedData(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_PKCS7_ENCRYPTED_DATA)
    PKCS7*      pkcs7 = NULL;
    byte*       tmpBytePtr = NULL;
    byte        encrypted[TWOK_BUF];
    byte        decoded[TWOK_BUF];
    word32      tmpWrd32 = 0;
    int         tmpInt = 0;
    int         decodedSz;
    int         encryptedSz = 0;
    int         testSz;
    int         i;
    const byte data[] = { /* Hello World */
        0x48,0x65,0x6c,0x6c,0x6f,0x20,0x57,0x6f,
        0x72,0x6c,0x64
    };
    #ifndef NO_DES3
        byte desKey[] = {
            0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef
        };
        byte des3Key[] = {
            0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
            0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
            0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
        };
    #endif
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #ifndef NO_AES_128
        byte aes128Key[] = {
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
        };
        #endif
        #ifndef NO_AES_192
        byte aes192Key[] = {
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
        };
        #endif
        #ifndef NO_AES_256
        byte aes256Key[] = {
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
        };
        #endif
    #endif /* !NO_AES && HAVE_AES_CBC */
    const pkcs7EncryptedVector testVectors[] =
    {
    #ifndef NO_DES3
        {data, (word32)sizeof(data), DATA, DES3b, des3Key, sizeof(des3Key)},

        {data, (word32)sizeof(data), DATA, DESb, desKey, sizeof(desKey)},
    #endif /* !NO_DES3 */
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #ifndef NO_AES_128
        {data, (word32)sizeof(data), DATA, AES128CBCb, aes128Key,
         sizeof(aes128Key)},
        #endif

        #ifndef NO_AES_192
        {data, (word32)sizeof(data), DATA, AES192CBCb, aes192Key,
         sizeof(aes192Key)},
        #endif

        #ifndef NO_AES_256
        {data, (word32)sizeof(data), DATA, AES256CBCb, aes256Key,
         sizeof(aes256Key)},
        #endif

    #endif /* !NO_AES && HAVE_AES_CBC */
    };

    testSz = sizeof(testVectors) / sizeof(pkcs7EncryptedVector);

    for (i = 0; i < testSz; i++) {
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);
        if (pkcs7 != NULL) {
            pkcs7->content              = (byte*)testVectors[i].content;
            pkcs7->contentSz            = testVectors[i].contentSz;
            pkcs7->contentOID           = testVectors[i].contentOID;
            pkcs7->encryptOID           = testVectors[i].encryptOID;
            pkcs7->encryptionKey        = testVectors[i].encryptionKey;
            pkcs7->encryptionKeySz      = testVectors[i].encryptionKeySz;
            pkcs7->heap                 = HEAP_HINT;
        }

        /* encode encryptedData */
        ExpectIntGT(encryptedSz = wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
            sizeof(encrypted)), 0);

        /* Decode encryptedData */
        ExpectIntGT(decodedSz = wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted,
            encryptedSz, decoded, sizeof(decoded)), 0);

        ExpectIntEQ(XMEMCMP(decoded, data, decodedSz), 0);
        /* Keep values for last itr. */
        if (i < testSz - 1) {
            wc_PKCS7_Free(pkcs7);
            pkcs7 = NULL;
        }
    }
    if (pkcs7 == NULL || testSz == 0) {
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);
    }

    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(NULL, encrypted,
        sizeof(encrypted)),BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, NULL,
        sizeof(encrypted)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        0), BAD_FUNC_ARG);
    /* Testing the struct. */
    if (pkcs7 != NULL) {
        tmpBytePtr = pkcs7->content;
        pkcs7->content = NULL;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->content = tmpBytePtr;
        tmpWrd32 = pkcs7->contentSz;
        pkcs7->contentSz = 0;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->contentSz = tmpWrd32;
        tmpInt = pkcs7->encryptOID;
        pkcs7->encryptOID = 0;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->encryptOID = tmpInt;
        tmpBytePtr = pkcs7->encryptionKey;
        pkcs7->encryptionKey = NULL;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->encryptionKey = tmpBytePtr;
        tmpWrd32 = pkcs7->encryptionKeySz;
        pkcs7->encryptionKeySz = 0;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->encryptionKeySz = tmpWrd32;
    }

    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(NULL, encrypted, encryptedSz,
        decoded, sizeof(decoded)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, NULL, encryptedSz,
        decoded, sizeof(decoded)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, 0,
        decoded, sizeof(decoded)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, encryptedSz,
        NULL, sizeof(decoded)), BAD_FUNC_ARG);
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, encryptedSz,
        decoded, 0), BAD_FUNC_ARG);
    /* Test struct fields */

    if (pkcs7 != NULL) {
        tmpBytePtr = pkcs7->encryptionKey;
        pkcs7->encryptionKey = NULL;
    }
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, encryptedSz,
        decoded, sizeof(decoded)), BAD_FUNC_ARG);
    if (pkcs7 != NULL) {
        pkcs7->encryptionKey = tmpBytePtr;
        pkcs7->encryptionKeySz = 0;
    }
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, encryptedSz,
        decoded, sizeof(decoded)), BAD_FUNC_ARG);

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_EncodeEncryptedData() */

/*
 * Testing wc_PKCS7_Degenerate()
 */
static int test_wc_PKCS7_Degenerate(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM)
    PKCS7* pkcs7 = NULL;
    char   fName[] = "./certs/test-degenerate.p7b";
    XFILE  f = XBADFILE;
    byte   der[4096];
    word32 derSz = 0;

    ExpectTrue((f = XFOPEN(fName, "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, sizeof(der), f)) > 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    /* test degenerate success */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
#ifndef NO_RSA
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#else
    ExpectIntNE(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#endif
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* test with turning off degenerate cases */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    wc_PKCS7_AllowDegenerate(pkcs7, 0); /* override allowing degenerate case */
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, der, derSz),
        PKCS7_NO_SIGNER_E);
    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_Degenerate() */

#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && \
    defined(ASN_BER_TO_DER) && !defined(NO_DES3) && !defined(NO_SHA)
static byte berContent[] = {
    0x30, 0x80, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
    0xF7, 0x0D, 0x01, 0x07, 0x03, 0xA0, 0x80, 0x30,
    0x80, 0x02, 0x01, 0x00, 0x31, 0x82, 0x01, 0x48,
    0x30, 0x82, 0x01, 0x44, 0x02, 0x01, 0x00, 0x30,
    0x81, 0xAC, 0x30, 0x81, 0x9E, 0x31, 0x0B, 0x30,
    0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
    0x55, 0x53, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03,
    0x55, 0x04, 0x08, 0x0C, 0x07, 0x4D, 0x6F, 0x6E,
    0x74, 0x61, 0x6E, 0x61, 0x31, 0x10, 0x30, 0x0E,
    0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x07, 0x42,
    0x6F, 0x7A, 0x65, 0x6D, 0x61, 0x6E, 0x31, 0x15,
    0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C,
    0x0C, 0x77, 0x6F, 0x6C, 0x66, 0x53, 0x53, 0x4C,
    0x5F, 0x31, 0x30, 0x32, 0x34, 0x31, 0x19, 0x30,
    0x17, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x10,
    0x50, 0x72, 0x6F, 0x67, 0x72, 0x61, 0x6D, 0x6D,
    0x69, 0x6E, 0x67, 0x2D, 0x31, 0x30, 0x32, 0x34,
    0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
    0x03, 0x0C, 0x0F, 0x77, 0x77, 0x77, 0x2E, 0x77,
    0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
    0x6F, 0x6D, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x09,
    0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
    0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40,
    0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E,
    0x63, 0x6F, 0x6D, 0x02, 0x09, 0x00, 0xBB, 0xD3,
    0x10, 0x03, 0xE6, 0x9D, 0x28, 0x03, 0x30, 0x0D,
    0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
    0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x81, 0x80,
    0x2F, 0xF9, 0x77, 0x4F, 0x04, 0x5C, 0x16, 0x62,
    0xF0, 0x77, 0x8D, 0x95, 0x4C, 0xB1, 0x44, 0x9A,
    0x8C, 0x3C, 0x8C, 0xE4, 0xD1, 0xC1, 0x14, 0x72,
    0xD0, 0x4A, 0x1A, 0x94, 0x27, 0x0F, 0xAA, 0xE8,
    0xD0, 0xA2, 0xE7, 0xED, 0x4C, 0x7F, 0x0F, 0xC7,
    0x1B, 0xFB, 0x81, 0x0E, 0x76, 0x8F, 0xDD, 0x32,
    0x11, 0x68, 0xA0, 0x13, 0xD2, 0x8D, 0x95, 0xEF,
    0x80, 0x53, 0x81, 0x0E, 0x1F, 0xC8, 0xD6, 0x76,
    0x5C, 0x31, 0xD3, 0x77, 0x33, 0x29, 0xA6, 0x1A,
    0xD3, 0xC6, 0x14, 0x36, 0xCA, 0x8E, 0x7D, 0x72,
    0xA0, 0x29, 0x4C, 0xC7, 0x3A, 0xAF, 0xFE, 0xF7,
    0xFC, 0xD7, 0xE2, 0x8F, 0x6A, 0x20, 0x46, 0x09,
    0x40, 0x22, 0x2D, 0x79, 0x38, 0x11, 0xB1, 0x4A,
    0xE3, 0x48, 0xE8, 0x10, 0x37, 0xA0, 0x22, 0xF7,
    0xB4, 0x79, 0xD1, 0xA9, 0x3D, 0xC2, 0xAB, 0x37,
    0xAE, 0x82, 0x68, 0x1A, 0x16, 0xEF, 0x33, 0x0C,
    0x30, 0x80, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
    0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x14, 0x06,
    0x08, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x03,
    0x07, 0x04, 0x08, 0xAD, 0xD0, 0x38, 0x9B, 0x16,
    0x4B, 0x7F, 0x99, 0xA0, 0x80, 0x04, 0x82, 0x03,
    0xE8, 0x6D, 0x48, 0xFB, 0x8A, 0xBD, 0xED, 0x6C,
    0xCD, 0xC6, 0x48, 0xFD, 0xB7, 0xB0, 0x7C, 0x86,
    0x2C, 0x8D, 0xF0, 0x23, 0x12, 0xD8, 0xA3, 0x2A,
    0x21, 0x6F, 0x8B, 0x75, 0xBB, 0x47, 0x7F, 0xC9,
    0xBA, 0xBA, 0xFF, 0x91, 0x09, 0x01, 0x7A, 0x5C,
    0x96, 0x02, 0xB8, 0x8E, 0xF8, 0x67, 0x7E, 0x8F,
    0xF9, 0x51, 0x0E, 0xFF, 0x8E, 0xE2, 0x61, 0xC0,
    0xDF, 0xFA, 0xE2, 0x4C, 0x50, 0x90, 0xAE, 0xA1,
    0x15, 0x38, 0x3D, 0xBE, 0x88, 0xD7, 0x57, 0xC0,
    0x11, 0x44, 0xA2, 0x61, 0x05, 0x49, 0x6A, 0x94,
    0x04, 0x10, 0xD9, 0xC2, 0x2D, 0x15, 0x20, 0x0D,
    0xBD, 0xA2, 0xEF, 0xE4, 0x68, 0xFA, 0x39, 0x75,
    0x7E, 0xD8, 0x64, 0x44, 0xCB, 0xE0, 0x00, 0x6D,
    0x57, 0x4E, 0x8A, 0x17, 0xA9, 0x83, 0x6C, 0x7F,
    0xFE, 0x01, 0xEE, 0xDE, 0x99, 0x3A, 0xB2, 0xFF,
    0xD3, 0x72, 0x78, 0xBA, 0xF1, 0x23, 0x54, 0x48,
    0x02, 0xD8, 0x38, 0xA9, 0x54, 0xE5, 0x4A, 0x81,
    0xB9, 0xC0, 0x67, 0xB2, 0x7D, 0x3C, 0x6F, 0xCE,
    0xA4, 0xDD, 0x34, 0x5F, 0x60, 0xB1, 0xA3, 0x7A,
    0xE4, 0x43, 0xF2, 0x89, 0x64, 0x35, 0x09, 0x32,
    0x51, 0xFB, 0x5C, 0x67, 0x0C, 0x3B, 0xFC, 0x36,
    0x6B, 0x37, 0x43, 0x6C, 0x03, 0xCD, 0x44, 0xC7,
    0x2B, 0x62, 0xD6, 0xD1, 0xF4, 0x07, 0x7B, 0x19,
    0x91, 0xF0, 0xD7, 0xF5, 0x54, 0xBC, 0x0F, 0x42,
    0x6B, 0x69, 0xF7, 0xA3, 0xC8, 0xEE, 0xB9, 0x7A,
    0x9E, 0x3D, 0xDF, 0x53, 0x47, 0xF7, 0x50, 0x67,
    0x00, 0xCF, 0x2B, 0x3B, 0xE9, 0x85, 0xEE, 0xBD,
    0x4C, 0x64, 0x66, 0x0B, 0x77, 0x80, 0x9D, 0xEF,
    0x11, 0x32, 0x77, 0xA8, 0xA4, 0x5F, 0xEE, 0x2D,
    0xE0, 0x43, 0x87, 0x76, 0x87, 0x53, 0x4E, 0xD7,
    0x1A, 0x04, 0x7B, 0xE1, 0xD1, 0xE1, 0xF5, 0x87,
    0x51, 0x13, 0xE0, 0xC2, 0xAA, 0xA3, 0x4B, 0xAA,
    0x9E, 0xB4, 0xA6, 0x1D, 0x4E, 0x28, 0x57, 0x0B,
    0x80, 0x90, 0x81, 0x4E, 0x04, 0xF5, 0x30, 0x8D,
    0x51, 0xCE, 0x57, 0x2F, 0x88, 0xC5, 0x70, 0xC4,
    0x06, 0x8F, 0xDD, 0x37, 0xC1, 0x34, 0x1E, 0x0E,
    0x15, 0x32, 0x23, 0x92, 0xAB, 0x40, 0xEA, 0xF7,
    0x43, 0xE2, 0x1D, 0xE2, 0x4B, 0xC9, 0x91, 0xF4,
    0x63, 0x21, 0x34, 0xDB, 0xE9, 0x86, 0x83, 0x1A,
    0xD2, 0x52, 0xEF, 0x7A, 0xA2, 0xEE, 0xA4, 0x11,
    0x56, 0xD3, 0x6C, 0xF5, 0x6D, 0xE4, 0xA5, 0x2D,
    0x99, 0x02, 0x10, 0xDF, 0x29, 0xC5, 0xE3, 0x0B,
    0xC4, 0xA1, 0xEE, 0x5F, 0x4A, 0x10, 0xEE, 0x85,
    0x73, 0x2A, 0x92, 0x15, 0x2C, 0xC8, 0xF4, 0x8C,
    0xD7, 0x3D, 0xBC, 0xAD, 0x18, 0xE0, 0x59, 0xD3,
    0xEE, 0x75, 0x90, 0x1C, 0xCC, 0x76, 0xC6, 0x64,
    0x17, 0xD2, 0xD0, 0x91, 0xA6, 0xD0, 0xC1, 0x4A,
    0xAA, 0x58, 0x22, 0xEC, 0x45, 0x98, 0xF2, 0xCC,
    0x4C, 0xE4, 0xBF, 0xED, 0xF6, 0x44, 0x72, 0x36,
    0x65, 0x3F, 0xE3, 0xB5, 0x8B, 0x3E, 0x54, 0x9C,
    0x82, 0x86, 0x5E, 0xB0, 0xF2, 0x12, 0xE5, 0x69,
    0xFA, 0x46, 0xA2, 0x54, 0xFC, 0xF5, 0x4B, 0xE0,
    0x24, 0x3B, 0x99, 0x04, 0x1A, 0x7A, 0xF7, 0xD1,
    0xFF, 0x68, 0x97, 0xB2, 0x85, 0x82, 0x95, 0x27,
    0x2B, 0xF4, 0xE7, 0x1A, 0x74, 0x19, 0xEC, 0x8C,
    0x4E, 0xA7, 0x0F, 0xAD, 0x4F, 0x5A, 0x02, 0x80,
    0xC1, 0x6A, 0x9E, 0x54, 0xE4, 0x8E, 0xA3, 0x41,
    0x3F, 0x6F, 0x9C, 0x82, 0x9F, 0x83, 0xB0, 0x44,
    0x01, 0x5F, 0x10, 0x9D, 0xD3, 0xB6, 0x33, 0x5B,
    0xAF, 0xAC, 0x6B, 0x57, 0x2A, 0x01, 0xED, 0x0E,
    0x17, 0xB9, 0x80, 0x76, 0x12, 0x1C, 0x51, 0x56,
    0xDD, 0x6D, 0x94, 0xAB, 0xD2, 0xE5, 0x15, 0x2D,
    0x3C, 0xC5, 0xE8, 0x62, 0x05, 0x8B, 0x40, 0xB1,
    0xC2, 0x83, 0xCA, 0xAC, 0x4B, 0x8B, 0x39, 0xF7,
    0xA0, 0x08, 0x43, 0x5C, 0xF7, 0xE8, 0xED, 0x40,
    0x72, 0x73, 0xE3, 0x6B, 0x18, 0x67, 0xA0, 0xB6,
    0x0F, 0xED, 0x8F, 0x9A, 0xE4, 0x27, 0x62, 0x23,
    0xAA, 0x6D, 0x6C, 0x31, 0xC9, 0x9D, 0x6B, 0xE0,
    0xBF, 0x9D, 0x7D, 0x2E, 0x76, 0x71, 0x06, 0x39,
    0xAC, 0x96, 0x1C, 0xAF, 0x30, 0xF2, 0x62, 0x9C,
    0x84, 0x3F, 0x43, 0x5E, 0x19, 0xA8, 0xE5, 0x3C,
    0x9D, 0x43, 0x3C, 0x43, 0x41, 0xE8, 0x82, 0xE7,
    0x5B, 0xF3, 0xE2, 0x15, 0xE3, 0x52, 0x20, 0xFD,
    0x0D, 0xB2, 0x4D, 0x48, 0xAD, 0x53, 0x7E, 0x0C,
    0xF0, 0xB9, 0xBE, 0xC9, 0x58, 0x4B, 0xC8, 0xA8,
    0xA3, 0x36, 0xF1, 0x2C, 0xD2, 0xE1, 0xC8, 0xC4,
    0x3C, 0x48, 0x70, 0xC2, 0x6D, 0x6C, 0x3D, 0x99,
    0xAC, 0x43, 0x19, 0x69, 0xCA, 0x67, 0x1A, 0xC9,
    0xE1, 0x47, 0xFA, 0x0A, 0xE6, 0x5B, 0x6F, 0x61,
    0xD0, 0x03, 0xE4, 0x03, 0x4B, 0xFD, 0xE2, 0xA5,
    0x8D, 0x83, 0x01, 0x7E, 0xC0, 0x7B, 0x2E, 0x0B,
    0x29, 0xDD, 0xD6, 0xDC, 0x71, 0x46, 0xBD, 0x9A,
    0x40, 0x46, 0x1E, 0x0A, 0xB1, 0x00, 0xE7, 0x71,
    0x29, 0x77, 0xFC, 0x9A, 0x76, 0x8A, 0x5F, 0x66,
    0x9B, 0x63, 0x91, 0x12, 0x78, 0xBF, 0x67, 0xAD,
    0xA1, 0x72, 0x9E, 0xC5, 0x3E, 0xE5, 0xCB, 0xAF,
    0xD6, 0x5A, 0x0D, 0xB6, 0x9B, 0xA3, 0x78, 0xE8,
    0xB0, 0x8F, 0x69, 0xED, 0xC1, 0x73, 0xD5, 0xE5,
    0x1C, 0x18, 0xA0, 0x58, 0x4C, 0x49, 0xBD, 0x91,
    0xCE, 0x15, 0x0D, 0xAA, 0x5A, 0x07, 0xEA, 0x1C,
    0xA7, 0x4B, 0x11, 0x31, 0x80, 0xAF, 0xA1, 0x0A,
    0xED, 0x6C, 0x70, 0xE4, 0xDB, 0x75, 0x86, 0xAE,
    0xBF, 0x4A, 0x05, 0x72, 0xDE, 0x84, 0x8C, 0x7B,
    0x59, 0x81, 0x58, 0xE0, 0xC0, 0x15, 0xB5, 0xF3,
    0xD5, 0x73, 0x78, 0x83, 0x53, 0xDA, 0x92, 0xC1,
    0xE6, 0x71, 0x74, 0xC7, 0x7E, 0xAA, 0x36, 0x06,
    0xF0, 0xDF, 0xBA, 0xFB, 0xEF, 0x54, 0xE8, 0x11,
    0xB2, 0x33, 0xA3, 0x0B, 0x9E, 0x0C, 0x59, 0x75,
    0x13, 0xFA, 0x7F, 0x88, 0xB9, 0x86, 0xBD, 0x1A,
    0xDB, 0x52, 0x12, 0xFB, 0x6D, 0x1A, 0xCB, 0x49,
    0x94, 0x94, 0xC4, 0xA9, 0x99, 0xC0, 0xA4, 0xB6,
    0x60, 0x36, 0x09, 0x94, 0x2A, 0xD5, 0xC4, 0x26,
    0xF4, 0xA3, 0x6A, 0x0E, 0x57, 0x8B, 0x7C, 0xA4,
    0x1D, 0x75, 0xE8, 0x2A, 0xF3, 0xC4, 0x3C, 0x7D,
    0x45, 0x6D, 0xD8, 0x24, 0xD1, 0x3B, 0xF7, 0xCF,
    0xE4, 0x45, 0x2A, 0x55, 0xE5, 0xA9, 0x1F, 0x1C,
    0x8F, 0x55, 0x8D, 0xC1, 0xF7, 0x74, 0xCC, 0x26,
    0xC7, 0xBA, 0x2E, 0x5C, 0xC1, 0x71, 0x0A, 0xAA,
    0xD9, 0x6D, 0x76, 0xA7, 0xF9, 0xD1, 0x18, 0xCB,
    0x5A, 0x52, 0x98, 0xA8, 0x0D, 0x3F, 0x06, 0xFC,
    0x49, 0x11, 0x21, 0x5F, 0x86, 0x19, 0x33, 0x81,
    0xB5, 0x7A, 0xDA, 0xA1, 0x47, 0xBF, 0x7C, 0xD7,
    0x05, 0x96, 0xC7, 0xF5, 0xC1, 0x61, 0xE5, 0x18,
    0xA5, 0x38, 0x68, 0xED, 0xB4, 0x17, 0x62, 0x0D,
    0x01, 0x5E, 0xC3, 0x04, 0xA6, 0xBA, 0xB1, 0x01,
    0x60, 0x5C, 0xC1, 0x3A, 0x34, 0x97, 0xD6, 0xDB,
    0x67, 0x73, 0x4D, 0x33, 0x96, 0x01, 0x67, 0x44,
    0xEA, 0x47, 0x5E, 0x44, 0xB5, 0xE5, 0xD1, 0x6C,
    0x20, 0xA9, 0x6D, 0x4D, 0xBC, 0x02, 0xF0, 0x70,
    0xE4, 0xDD, 0xE9, 0xD5, 0x5C, 0x28, 0x29, 0x0B,
    0xB4, 0x60, 0x2A, 0xF1, 0xF7, 0x1A, 0xF0, 0x36,
    0xAE, 0x51, 0x3A, 0xAE, 0x6E, 0x48, 0x7D, 0xC7,
    0x5C, 0xF3, 0xDC, 0xF6, 0xED, 0x27, 0x4E, 0x8E,
    0x48, 0x18, 0x3E, 0x08, 0xF1, 0xD8, 0x3D, 0x0D,
    0xE7, 0x2F, 0x65, 0x8A, 0x6F, 0xE2, 0x1E, 0x06,
    0xC1, 0x04, 0x58, 0x7B, 0x4A, 0x75, 0x60, 0x92,
    0x13, 0xC6, 0x40, 0x2D, 0x3A, 0x8A, 0xD1, 0x03,
    0x05, 0x1F, 0x28, 0x66, 0xC2, 0x57, 0x2A, 0x4C,
    0xE1, 0xA3, 0xCB, 0xA1, 0x95, 0x30, 0x10, 0xED,
    0xDF, 0xAE, 0x70, 0x49, 0x4E, 0xF6, 0xB4, 0x5A,
    0xB6, 0x22, 0x56, 0x37, 0x05, 0xE7, 0x3E, 0xB2,
    0xE3, 0x96, 0x62, 0xEC, 0x09, 0x53, 0xC0, 0x50,
    0x3D, 0xA7, 0xBC, 0x9B, 0x39, 0x02, 0x26, 0x16,
    0xB5, 0x34, 0x17, 0xD4, 0xCA, 0xFE, 0x1D, 0xE4,
    0x5A, 0xDA, 0x4C, 0xC2, 0xCA, 0x8E, 0x79, 0xBF,
    0xD8, 0x4C, 0xBB, 0xFA, 0x30, 0x7B, 0xA9, 0x3E,
    0x52, 0x19, 0xB1, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00
};
#endif /* HAVE_PKCS7 && !NO_FILESYSTEM && ASN_BER_TO_DER &&
        * !NO_DES3 && !NO_SHA
        */

/*
 * Testing wc_PKCS7_BER()
 */
static int test_wc_PKCS7_BER(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && \
    !defined(NO_SHA) && defined(ASN_BER_TO_DER)
    PKCS7* pkcs7 = NULL;
    char   fName[] = "./certs/test-ber-exp02-05-2022.p7b";
    XFILE  f = XBADFILE;
    byte   der[4096];
#ifndef NO_DES3
    byte   decoded[2048];
#endif
    word32 derSz = 0;

    ExpectTrue((f = XFOPEN(fName, "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, sizeof(der), f)) > 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
#ifndef NO_RSA
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#else
    ExpectIntNE(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#endif
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_DES3
    /* decode BER content */
    ExpectTrue((f = XFOPEN("./certs/1024/client-cert.der", "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, sizeof(der), f)) > 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
#ifndef NO_RSA
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, der, derSz), 0);
#else
    ExpectIntNE(wc_PKCS7_InitWithCert(pkcs7, der, derSz), 0);
#endif

    ExpectTrue((f = XFOPEN("./certs/1024/client-key.der", "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, sizeof(der), f)) > 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    if (pkcs7 != NULL) {
        pkcs7->privateKey   = der;
        pkcs7->privateKeySz = derSz;
    }
#ifndef NO_RSA
#ifdef WOLFSSL_SP_MATH
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, berContent,
        sizeof(berContent), decoded, sizeof(decoded)), WC_KEY_SIZE_E);
#else
    ExpectIntGT(wc_PKCS7_DecodeEnvelopedData(pkcs7, berContent,
        sizeof(berContent), decoded, sizeof(decoded)), 0);
#endif
#else
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, berContent,
        sizeof(berContent), decoded, sizeof(decoded)), NOT_COMPILED_IN);
#endif
    wc_PKCS7_Free(pkcs7);
#endif /* !NO_DES3 */
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_BER() */

static int test_wc_PKCS7_signed_enveloped(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_RSA) && !defined(NO_AES) && \
    !defined(NO_FILESYSTEM)
    XFILE  f = XBADFILE;
    PKCS7* pkcs7 = NULL;
#ifdef HAVE_AES_CBC
    PKCS7* inner = NULL;
#endif
    void*  pt = NULL;
    WC_RNG rng;
    unsigned char key[FOURK_BUF/2];
    unsigned char cert[FOURK_BUF/2];
    unsigned char env[FOURK_BUF];
    int envSz  = FOURK_BUF;
    int keySz = 0;
    int certSz = 0;
    unsigned char sig[FOURK_BUF * 2];
    int sigSz = FOURK_BUF * 2;
#ifdef HAVE_AES_CBC
    unsigned char decoded[FOURK_BUF];
    int decodedSz = FOURK_BUF;
#endif

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    /* load cert */
    ExpectTrue((f = XFOPEN(cliCertDerFile, "rb")) != XBADFILE);
    ExpectIntGT((certSz = (int)XFREAD(cert, 1, sizeof(cert), f)), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    /* load key */
    ExpectTrue((f = XFOPEN(cliKeyFile, "rb")) != XBADFILE);
    ExpectIntGT((keySz = (int)XFREAD(key, 1, sizeof(key), f)), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntGT(keySz = wolfSSL_KeyPemToDer(key, keySz, key, keySz, NULL), 0);

    /* sign cert for envelope */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);
    if (pkcs7 != NULL) {
        pkcs7->content    = cert;
        pkcs7->contentSz  = certSz;
        pkcs7->contentOID = DATA;
        pkcs7->privateKey   = key;
        pkcs7->privateKeySz = keySz;
        pkcs7->encryptOID   = RSAk;
        pkcs7->hashOID      = SHA256h;
        pkcs7->rng          = &rng;
    }
    ExpectIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig, sigSz)), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    DoExpectIntEQ(wc_FreeRng(&rng), 0);

#ifdef HAVE_AES_CBC
    /* create envelope */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);
    if (pkcs7 != NULL) {
        pkcs7->content   = sig;
        pkcs7->contentSz = sigSz;
        pkcs7->contentOID = DATA;
        pkcs7->encryptOID = AES256CBCb;
        pkcs7->privateKey   = key;
        pkcs7->privateKeySz = keySz;
    }
    ExpectIntGT((envSz = wc_PKCS7_EncodeEnvelopedData(pkcs7, env, envSz)), 0);
    ExpectIntLT(wc_PKCS7_EncodeEnvelopedData(pkcs7, env, 2), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif

    /* create bad signed enveloped data */
    sigSz = FOURK_BUF * 2;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);
    if (pkcs7 != NULL) {
        pkcs7->content    = env;
        pkcs7->contentSz  = envSz;
        pkcs7->contentOID = DATA;
        pkcs7->privateKey   = key;
        pkcs7->privateKeySz = keySz;
        pkcs7->encryptOID   = RSAk;
        pkcs7->hashOID      = SHA256h;
        pkcs7->rng = &rng;
    }

    /* Set no certs in bundle for this test. Hang on to the pointer though to
     * free it later. */
    if (pkcs7 != NULL) {
        pt = (void*)pkcs7->certList;
        pkcs7->certList = NULL; /* no certs in bundle */
    }
    ExpectIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig, sigSz)), 0);
    if (pkcs7 != NULL) {
        /* restore pointer for PKCS7 free call */
        pkcs7->certList = (Pkcs7Cert*)pt;
    }
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* check verify fails */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, sig, sigSz),
            PKCS7_SIGNEEDS_CHECK);

    /* try verifying the signature manually */
    {
        RsaKey rKey;
        word32 idx = 0;
        byte digest[MAX_SEQ_SZ + MAX_ALGO_SZ + MAX_OCTET_STR_SZ +
           