/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.bean.html5;

import java.util.*;

import org.apache.juneau.annotation.*;

/**
 * DTO for an HTML <a class="doclink" href="https://www.w3.org/TR/html5/grouping-content.html#the-dl-element">&lt;dl&gt;</a>
 * element.
 *
 * <p>
 * The dl element represents a description list (also known as a definition list or association list).
 * It is used to group terms (dt elements) with their descriptions (dd elements), creating a list
 * of term-description pairs. The dl element is commonly used for glossaries, metadata lists, or
 * any other content where you need to associate terms with their definitions or descriptions.
 *
 * <h5 class='section'>Examples:</h5>
 * <p class='bcode w800'>
 * 	<jk>import static</jk> org.apache.juneau.bean.html5.HtmlBuilder.*;
 *
 * 	<jc>// Simple description list</jc>
 * 	Dl <jv>simple</jv> = <jsm>dl</jsm>(
 * 		<jsm>dt</jsm>(<js>"HTML"</js>),
 * 		<jsm>dd</jsm>(<js>"HyperText Markup Language"</js>),
 * 		<jsm>dt</jsm>(<js>"CSS"</js>),
 * 		<jsm>dd</jsm>(<js>"Cascading Style Sheets"</js>)
 * 	);
 *
 * 	<jc>// Description list with styling</jc>
 * 	Dl <jv>styled</jv> = <jsm>dl</jsm>(
 * 		<jsm>dt</jsm>(<js>"API"</js>),
 * 		<jsm>dd</jsm>(<js>"Application Programming Interface"</js>),
 * 		<jsm>dt</jsm>(<js>"DOM"</js>),
 * 		<jsm>dd</jsm>(<js>"Document Object Model"</js>)
 * 	)._class(<js>"glossary"</js>);
 *
 * 	<jc>// Description list with multiple descriptions</jc>
 * 	Dl <jv>multiple</jv> = <jsm>dl</jsm>(
 * 		<jsm>dt</jsm>(<js>"JavaScript"</js>),
 * 		<jsm>dd</jsm>(<js>"A programming language for web development."</js>),
 * 		<jsm>dd</jsm>(<js>"It runs in web browsers and on servers."</js>),
 * 		<jsm>dt</jsm>(<js>"Python"</js>),
 * 		<jsm>dd</jsm>(<js>"A high-level programming language."</js>),
 * 		<jsm>dd</jsm>(<js>"Known for its simplicity and readability."</js>)
 * 	);
 *
 * 	<jc>// Description list with complex content</jc>
 * 	Dl <jv>complex</jv> = <jsm>dl</jsm>(
 * 		<jsm>dt</jsm>(<js>"Web Standards"</js>),
 * 		<jsm>dd</jsm>(
 * 			<js>"Standards developed by the "</js>,
 * 			<jsm>a</jsm>(<js>"https://w3.org"</js>, <js>"W3C"</js>),
 * 			<js>" to ensure web compatibility."</js>
 * 		),
 * 		<jsm>dt</jsm>(<js>"Responsive Design"</js>),
 * 		<jsm>dd</jsm>(
 * 			<js>"Design approach that adapts to different "</js>,
 * 			<jsm>strong</jsm>(<js>"screen sizes"</js>),
 * 			<js>" and devices."</js>
 * 		)
 * 	);
 *
 * 	<jc>// Description list with ID</jc>
 * 	Dl <jv>withId</jv> = <jsm>dl</jsm>(
 * 		<jsm>dt</jsm>(<js>"Framework"</js>),
 * 		<jsm>dd</jsm>(<js>"A collection of pre-written code for common tasks."</js>),
 * 		<jsm>dt</jsm>(<js>"Library"</js>),
 * 		<jsm>dd</jsm>(<js>"A collection of reusable code modules."</js>)
 * 	).id(<js>"tech-terms"</js>);
 *
 * 	<jc>// Description list with styling</jc>
 * 	Dl <jv>styled2</jv> = <jsm>dl</jsm>(
 * 		<jsm>dt</jsm>(<js>"Frontend"</js>),
 * 		<jsm>dd</jsm>(<js>"The client-side part of a web application."</js>),
 * 		<jsm>dt</jsm>(<js>"Backend"</js>),
 * 		<jsm>dd</jsm>(<js>"The server-side part of a web application."</js>)
 * 	).style(<js>"border: 1px solid #ccc; padding: 10px;"</js>);
 * </p>
 *
 * <p>
 * The following convenience methods are provided for constructing instances of this bean:
 * <ul class='javatree'>
 * 	<li class='jc'>{@link HtmlBuilder}
 * 	<ul class='javatree'>
 * 		<li class='jm'>{@link HtmlBuilder#dl() dl()}
 * 		<li class='jm'>{@link HtmlBuilder#dl(Object...) dl(Object...)}
 * 	</ul>
 * </ul>
 * </p>
 *
 * <h5 class='section'>See Also:</h5><ul>
 * 	<li class='link'><a class="doclink" href="https://juneau.apache.org/docs/topics/JuneauBeanHtml5">juneau-bean-html5</a>
 * </ul>
 */
@Bean(typeName = "dl")
public class Dl extends HtmlElementContainer {

	/**
	 * Creates an empty {@link Dl} element.
	 */
	public Dl() {}

	/**
	 * Creates a {@link Dl} element with the specified child nodes.
	 *
	 * @param children The child nodes.
	 */
	public Dl(Object...children) {
		children(children);
	}

	@Override /* Overridden from HtmlElement */
	public Dl _class(String value) { // NOSONAR - Intentional naming.
		super._class(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl accesskey(String value) {
		super.accesskey(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl attr(String key, Object val) {
		super.attr(key, val);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl attrUri(String key, Object val) {
		super.attrUri(key, val);
		return this;
	}

	@Override /* Overridden from HtmlElementContainer */
	public Dl child(Object value) {
		super.child(value);
		return this;
	}

	@Override /* Overridden from HtmlElementContainer */
	public Dl children(Object...value) {
		super.children(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl contenteditable(Object value) {
		super.contenteditable(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl dir(String value) {
		super.dir(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl hidden(Object value) {
		super.hidden(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl id(String value) {
		super.id(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl lang(String value) {
		super.lang(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onabort(String value) {
		super.onabort(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onblur(String value) {
		super.onblur(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl oncancel(String value) {
		super.oncancel(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl oncanplay(String value) {
		super.oncanplay(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl oncanplaythrough(String value) {
		super.oncanplaythrough(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onchange(String value) {
		super.onchange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onclick(String value) {
		super.onclick(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl oncuechange(String value) {
		super.oncuechange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl ondblclick(String value) {
		super.ondblclick(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl ondurationchange(String value) {
		super.ondurationchange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onemptied(String value) {
		super.onemptied(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onended(String value) {
		super.onended(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onerror(String value) {
		super.onerror(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onfocus(String value) {
		super.onfocus(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl oninput(String value) {
		super.oninput(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl oninvalid(String value) {
		super.oninvalid(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onkeydown(String value) {
		super.onkeydown(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onkeypress(String value) {
		super.onkeypress(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onkeyup(String value) {
		super.onkeyup(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onload(String value) {
		super.onload(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onloadeddata(String value) {
		super.onloadeddata(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onloadedmetadata(String value) {
		super.onloadedmetadata(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onloadstart(String value) {
		super.onloadstart(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onmousedown(String value) {
		super.onmousedown(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onmouseenter(String value) {
		super.onmouseenter(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onmouseleave(String value) {
		super.onmouseleave(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onmousemove(String value) {
		super.onmousemove(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onmouseout(String value) {
		super.onmouseout(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onmouseover(String value) {
		super.onmouseover(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onmouseup(String value) {
		super.onmouseup(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onmousewheel(String value) {
		super.onmousewheel(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onpause(String value) {
		super.onpause(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onplay(String value) {
		super.onplay(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onplaying(String value) {
		super.onplaying(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onprogress(String value) {
		super.onprogress(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onratechange(String value) {
		super.onratechange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onreset(String value) {
		super.onreset(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onresize(String value) {
		super.onresize(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onscroll(String value) {
		super.onscroll(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onseeked(String value) {
		super.onseeked(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onseeking(String value) {
		super.onseeking(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onselect(String value) {
		super.onselect(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onshow(String value) {
		super.onshow(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onstalled(String value) {
		super.onstalled(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onsubmit(String value) {
		super.onsubmit(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onsuspend(String value) {
		super.onsuspend(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl ontimeupdate(String value) {
		super.ontimeupdate(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl ontoggle(String value) {
		super.ontoggle(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onvolumechange(String value) {
		super.onvolumechange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl onwaiting(String value) {
		super.onwaiting(value);
		return this;
	}

	@Override /* Overridden from HtmlElementContainer */
	public Dl setChildren(List<Object> children) {
		super.setChildren(children);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl spellcheck(Object value) {
		super.spellcheck(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl style(String value) {
		super.style(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl tabindex(Object value) {
		super.tabindex(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl title(String value) {
		super.title(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Dl translate(Object value) {
		super.translate(value);
		return this;
	}
}