/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.conversation;

import org.aopalliance.aop.Advice;
import org.springframework.aop.Pointcut;
import org.springframework.aop.PointcutAdvisor;

/**
 * A Spring object that defines what Advice objects should be applied to which beans.
 * <p>
 * Various standard Spring BeanPostProcessor classes look for registered beans of type
 * Advisor, test whether they apply to the bean being created and if so then add the
 * advice to a proxy for that bean (creating the proxy if necessary).  
 * <p>
 * Various sub-interfaces can be implemented by an Advisor class in order to filter
 * which classes or methods it applies to. In Spring2.5, a class that simply implements
 * Advisor gets applied to every method of every class. But in Spring2.0, it will get
 * silently ignored - either PointcutAdvisor ot IntroductionAdvisor must be implemented.
 * <p>
 * This mock implementation is happy to apply the MockAdvice advice to any class
 * that is offered to it by a BeanPostProcessor.
 */
public class MockAdvisor implements PointcutAdvisor
{
    MockAdvice advice = new MockAdvice();
    public Advice getAdvice()
    {
        return advice;
    }

    public boolean isPerInstance()
    {
        return false;
    }

    public Pointcut getPointcut()
    {
        // return a pointcut that matches all methods
        return Pointcut.TRUE;
    }
    
    public void setMessage(String msg)
    {
        advice.setMessage(msg);
    }
}
