# Pilatus PC-12/45 Electrical System
# Bea Wolf (D-ECHO) 2024
# Reference(s): see pc-12.xml

# based on turboprop engine electrical system by Syd Adams    ####

# Power Consumption References:
#		No.		Instrument		URL
#		E1		ACU-6100		https://www.becker-avionics.com/wp-content/uploads/00_Literature/ACU6100_IO.pdf

#	Ref. PIM p. 7-14-1
#
# 	DC System
#		normal voltage: 28V
#		power supply:
#			* GEN 1 System 28V 300A optional 400A - also serves as starter motor
#			* GEN 2 System 28V 115A
#			* BAT 1 24V 40Ah
#			* optional BAT 2 24V 40Ah
#			* optionally both batteries 42 Ah
#			* optional Emergency Power System
#			* external power ( max. 28V, 1000A )
#		Buses:
#			* Battery Direct Bus - connected directly to battery
#			* Battery Bus - connected to battery via BAT 1 Switch
#			* GEN 1 Bus - connected to GEN 1 via GEN 1 Switch
#			* GEN 2 Bus - connected to GEN 2 via GEN 2 Switch
#			* Avionic 1 Bus - connected to Battery Bus via Avionic 1 Switch
#			* Avionic 2 Bus - connected to GEN 1 Bus via Avionic 2 Switch
#			* Non-Essential Bus - connected to Battery Bus via automatic load shedding device and switch
#			* Standby Bus - connected to Avionic 1 Bus or Battery Direct Bus if Standby Bus Switch is on and Standby Bus Voltage higher than Avionic 1 Bus Voltage
#		Bus Connections
#			* BUS TIE - connects GEN 1 Bus and Battery Bus 		- auto opens when load is above 220 Amps
#			* GEN 2 TIE - connects GEN 2 Bus and Battery Bus	- auto opens when load is above 145 Amps
#
#	AC System
#		normal voltage: 26VAC 400Hz
#		power supply:
#			* Inverter No. 1 powered from Battery Bus
#			* Inverter No. 2 powered from GEN 1 Bus

# Basic props.nas objects
var electrical = props.globals.getNode("systems/electrical");
var electrical_sw = electrical.initNode("internal-switches");
var output = electrical.getNode("outputs");
var breakers = props.globals.initNode("/controls/circuit-breakers");
var controls = props.globals.getNode("/controls/electric");
var light_ctrl = props.globals.getNode("/controls/lighting");
var deice = props.globals.initNode("/controls/de-ice");
var fuel = props.globals.initNode("/controls/fuel");
var engine = props.globals.getNode("/controls/engines/engine[0]");

# Helper functions
var check_or_create = func ( prop, value, type ) {
	var obj = props.globals.getNode(prop, 1);
	if( obj.getValue() == nil ){
		return props.globals.initNode(prop, value, type);
	} else {
		return obj;
	}
}

#	Switches
var switches = {
	master:		controls.initNode("master",		1,	"BOOL"),	# MASTER POWER EMERGENCY OFF Switch - normally ON and Guarded
	batt1:		controls.initNode("battery[0]",	0,	"BOOL"),
	#	Generator switch mapping: -1 = RESET; 0 = OFF; 1 = ON
	gen1:			controls.initNode("generator[0]",	0,	"INT"),
	gen2:			controls.initNode("generator[1]",	0,	"BOOL"),
	non_ess_bus:	controls.initNode("non-ess-bus",	1,	"BOOL"),	# 0 = AUTO, 1 = OVRD (always on)
	avionic1:		controls.initNode("avionic[0]",	0,	"BOOL"),
	avionic2:		controls.initNode("avionic[1]",	0,	"BOOL"),
	ext_pwr:		controls.initNode("ext-power",	0,	"BOOL"),
	stby_bus:		controls.initNode("standby-bus",	0,	"BOOL"),	# 0 = Avionic 1 Bus, 1 = Direct Battery Bus
	inverter:		controls.initNode("inverter",		0,	"BOOL"),	# 0 = Use Inverter 1, 1 = Use Inverter 2
	
	strobe_light:	light_ctrl.initNode("strobe-lights",	0,	"BOOL"),
	beacon_light:	light_ctrl.initNode("beacon",			0,	"BOOL"),
	
	# TODO Interior Lights
	
	#			De-Ice
	boots:		deice.initNode("boots",			0,	"BOOL"),
	boots_mode:		deice.initNode("boots-mode",		0,	"BOOL"),	# 0 = 1 MIN, 1 = 3 MIN
	probes:		deice.initNode("probes",		0,	"BOOL"),
	inert_sep:		deice.initNode("inert-sep",		0,	"BOOL"),
	prop:			deice.initNode("prop",			0,	"BOOL"),
	
	lh_wsh:		deice.initNode("windshield[0]",		0,	"BOOL"),
	lh_wsh_mode:	deice.initNode("windshield[0]-mode",	0,	"BOOL"), # 0 = LIGHT, 1 = HEAVY
	rh_wsh:		deice.initNode("windshield[1]",		0,	"BOOL"),
	rh_wsh_mode:	deice.initNode("windshield[1]-mode",	0,	"BOOL"), # 0 = LIGHT, 1 = HEAVY
	#			Fuel
	fp_lh:		fuel.initNode("fuel-pump-lh",	0,	"BOOL"), # 0 = AUTO, 1 = ON
	fp_rh:		fuel.initNode("fuel-pump-rh",	0,	"BOOL"), # 0 = AUTO, 1 = ON
	
	#			Engine Controls
	ignition:		engine.initNode("ignition",	0,	"BOOL"), # 0 = AUTO, 1 = ON
	starter_s:		engine.initNode("starter-sw",	0,	"BOOL"),
	starter_c:		engine.initNode("starter",	0,	"BOOL"),
	starter_interrupt:engine.initNode("starter-intrp", 0,"BOOL"),
};

var inverter = [
	output.getNode("inverter[0]"),
	output.getNode("inverter[1]"),
];

var strobeLight = aircraft.light.new("/sim/model/lights/strobe", [0.08, 2.0], switches.strobe_light); # TODO On/Off Time
var beaconLight = aircraft.light.new("/sim/model/lights/beacon", [0.06, 2.0], switches.beacon_light); # TODO On/Off Time

var int_switches = {
	strobe_light:	props.globals.getNode("/sim/model/lights/strobe/state", 1),
	beacon_light:	props.globals.getNode("/sim/model/lights/beacon/state", 1),
};

var bus_breakers = {
	bus_tie:	breakers.initNode("bus-tie",	1, "BOOL"),
	gen2_tie:	breakers.initNode("gen2-tie",	1, "BOOL"),,
};

var delta_sec	=	props.globals.getNode("sim/time/delta-sec");


#	TODO calculate battery temperature correctly
Battery = {
	new : func( switch, breaker, volt, amps, amp_hours, charge_percent, charge_amps, n){
		m = { parents : [Battery] };
		m.switch = props.globals.initNode(switch, 1, "BOOL");
		m.temp = electrical.initNode("battery-temperature["~n~"]", 15.0, "DOUBLE");
		m.breaker = breaker;
		m.ideal_volts = volt;
		m.ideal_amps = amps;
		m.volt_p = electrical.initNode("battery-volts["~n~"]", 0.0, "DOUBLE");
		m.amp_hours = amp_hours;
		m.charge_percent = charge_percent; 
		m.charge_amps = charge_amps;
		m.amps_p = electrical.initNode("battery-amps["~n~"]", 0.0, "DOUBLE");
		return m;
	},
	apply_load : func(load,dt) {
		if( me.switch.getBoolValue() and me.breaker.getBoolValue() ){
			me.amps_p.setDoubleValue( load );
			var amphrs_used = load * dt / 3600.0;
			var percent_used = amphrs_used / me.amp_hours;
			me.charge_percent -= percent_used;
			if ( me.charge_percent < 0.0 ) {
				me.charge_percent = 0.0;
			} elsif ( me.charge_percent > 1.0 ) {
				me.charge_percent = 1.0;
			}
			var output =me.amp_hours * me.charge_percent;
			return output;
		}else return 0;
	},
	
	get_output_volts : func {
		if( me.switch.getBoolValue() and me.breaker.getBoolValue() ){
			var x = 1.0 - me.charge_percent;
			var tmp = -(3.0 * x - 1.0);
			var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
			var output =me.ideal_volts * factor;
			me.volt_p.setDoubleValue( output );
			return output;
		}else return 0;
	},
	
	get_output_amps : func {
		if( me.switch.getBoolValue() and me.breaker.getBoolValue() ){
			var x = 1.0 - me.charge_percent;
			var tmp = -(3.0 * x - 1.0);
			var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
			var output =me.ideal_amps * factor;
			return output;
		}else return 0;
	}
};

# var alternator = Alternator.new(num,switch,rpm_source,rpm_threshold,volts,amps);
Alternator = {
	new : func (num,switch,src,thr,vlt,amp){
		m = { parents : [Alternator] };
		m.switch =  switch;
		m.meter =  props.globals.getNode("systems/electrical/gen-load["~num~"]",1);
		m.meter.setDoubleValue(0);
		m.gen_output =  props.globals.getNode("engines/engine["~num~"]/amp-v",1);
		m.gen_output.setDoubleValue(0);
		m.meter.setDoubleValue(0);
		m.rpm_source =  props.globals.getNode(src,1);
		m.rpm_threshold = thr;
		m.ideal_volts = vlt;
		m.ideal_amps = amp;
		return m;
	},
	
	apply_load : func(load) {
		var cur_volt=me.gen_output.getValue();
		var cur_amp=me.meter.getValue();
		if(cur_volt >1){
			var factor=1/cur_volt;
			gout = (load * factor);
			if(gout>1)gout=1;
		}else{
			gout=0;
		}
		if(cur_amp > gout)me.meter.setValue(cur_amp - 0.01);
		if(cur_amp < gout)me.meter.setValue(cur_amp + 0.01);
	},
	
	get_output_volts : func {
		var out = 0;
		if(me.switch.getIntValue() == 1){
			var factor = me.rpm_source.getDoubleValue() / me.rpm_threshold;
			if ( factor > 1.0 )factor = 1.0;
			var out = (me.ideal_volts * factor);
		}
		me.gen_output.setValue(out);
		if (out > 1) return out;
		return 0;
	},
	
	get_output_amps : func {
		var ampout =0;
		if(me.switch.getBoolValue()){
			var factor = me.rpm_source.getValue() / me.rpm_threshold;
			if ( factor > 1.0 ) {
				factor = 1.0;
			}
			ampout = me.ideal_amps * factor;
		}
		return ampout;
	}
};

var battery1 = Battery.new("/systems/electrical/battery/serviceable[0]", breakers.initNode("batt1", 1, "BOOL"), 24, 30, 40, 1.0, 7.0, 0);

var generator1 = Alternator.new(0, switches.gen1, "/engines/engine[0]/n1", 55.0, 28.0, 300.0);
var generator2 = Alternator.new(1, switches.gen2, "/engines/engine[0]/n1", 55.0, 28.0, 145.0);

var bus = {
	new: func( name, on_update ) {
		m = { parents : [bus] };
		m.name = name;
		m.volts = check_or_create("systems/electrical/bus/" ~ name ~ "-volts", 0.0, "DOUBLE");
		m.serviceable = check_or_create("systems/electrical/bus/" ~ name ~ "-serviceable", 1, "BOOL");
		m.on_update = on_update;
		m.bus_volts = 0.0;
		return m;
	}
};

# Bus interdependence
var gen1_bus_from_batt_bus = 0;
var gen2_bus_from_batt_bus = 0;

# "Reverse" Load not handled by the usual logic
var gen1_bus_load_from_batt_bus = 0.0;
var gen2_bus_load_from_batt_bus = 0.0;

var battery_direct_bus = bus.new( 
"battery-direct-bus", 
func() {
	if( me.serviceable.getBoolValue() ){
		if( switches.batt1.getBoolValue() and battery_bus.bus_volts > ( battery1.get_output_volts() + 1.0 ) ){
			me.bus_volts = battery_bus.bus_volts;
		} else {
			me.bus_volts = battery1.get_output_volts();
		}
	} else {
		me.bus_volts = 0.0;
	}
	
	var load = 0.1;	# estimate value for all direct bus consumers
	load += battery_bus.on_update( me.bus_volts );
	if( switches.stby_bus.getBoolValue() ) load += stby_bus.on_update( me.bus_volts );
	
	battery1.apply_load( load, delta_sec.getDoubleValue() );
	
	me.volts.setDoubleValue( me.bus_volts );   
},
);

var battery_bus = bus.new( 
"battery-bus", 
func( bv ) {
	if( me.serviceable.getBoolValue() ){
		if( bus_breakers.bus_tie.getBoolValue() and generator1_bus.bus_volts > ( battery1.get_output_volts()  + 1.0 ) and !gen1_bus_from_batt_bus ){
			me.bus_volts = generator1_bus.bus_volts;
		}elsif(  switches.batt1.getBoolValue() ){
			me.bus_volts = bv;
		} else {
			me.bus_volts = 0.0;
		}
	} else {
		me.bus_volts = 0.0;
	}
	
	var load = 1.0;	# estimate value for all direct bus consumers
	load += avionic1_bus.on_update( me.bus_volts );
	var genload = [
		generator1_bus.on_update( me.bus_volts ),
		generator2_bus.on_update( me.bus_volts ),
	];
	var charge_from_generator = 0.0;
	if( genload[0] < 0 ){
		# Charge the battery from Generator 1
		charge_from_generator = math.min( battery1.charge_amps + load, 220 );
		gen1_bus_load_from_batt_bus = charge_from_generator;
		gen2_bus_load_from_batt_bus = 0.0;
		load = -charge_from_generator;
	} elsif( genload[1] < 0 ){
		# Charge the battery from Generator 2 only if Generator 1 is unavailable
		charge_from_generator = math.min( battery1.charge_amps + load, 145 );
		gen1_bus_load_from_batt_bus = 0.0;
		gen2_bus_load_from_batt_bus = load;
		load = -charge_from_generator;
	}
	
	# Automatic Load-Shedding
	if( switches.non_ess_bus.getBoolValue() or load < 50.0 ){
		load += non_ess_bus.on_update( me.bus_volts );
	} else {
		non_ess_bus.on_update( 0.0 );
	}
	
	me.volts.setDoubleValue( me.bus_volts );  
	
	return load;
},
);

var generator1_bus = bus.new(
	"generator1-bus",
	func ( bv ) {
		var powered_by = "";
		me.bus_volts = 0.0;
		if( me.serviceable.getBoolValue() ){
			if( switches.gen1.getIntValue() == 1 and generator1.get_output_volts() != 0.0 ){
				me.bus_volts = generator1.get_output_volts();
				powered_by = "generator1";
				gen1_bus_from_batt_bus = 0;
			}
			if ( bus_breakers.bus_tie.getBoolValue() and bv > me.bus_volts ){
				me.bus_volts = bv;
				powered_by = "battery_bus";
				gen1_bus_from_batt_bus = 1;
			}
		}
		
		var load = 1.0;	# estimate value for all direct bus consumers
		load += avionic2_bus.on_update( me.bus_volts );
		
		if( switches.starter_s.getBoolValue() and me.bus_volts > 20 ){
			load += 150;
			switches.starter_c.setBoolValue( 1 );
		} else {
			switches.starter_c.setBoolValue( 0 );
		}
		
		me.volts.setDoubleValue( me.bus_volts );
		
		if( powered_by == "generator1" ){
			if( bus_breakers.bus_tie.getBoolValue() ){
				generator1.apply_load( load + gen1_bus_load_from_batt_bus );
				return -1.0;
			} else {
				generator1.apply_load( load );
				return 0.0;
			}
		} elsif ( powered_by == "battery_bus" ){
			return load;
		} else {
			return 0.0;
		}
	},
);

var generator2_bus = bus.new(
	"generator2-bus",
	func ( bv ) {
		var powered_by = "";
		me.bus_volts = 0.0;
		if( me.serviceable.getBoolValue() ){
			if( switches.gen2.getBoolValue() and generator2.get_output_volts() != 0.0 ){
				me.bus_volts = generator2.get_output_volts();
				powered_by = "generator2";
				gen2_bus_from_batt_bus = 0;
			}
			if ( bus_breakers.gen2_tie.getBoolValue() and bv > me.bus_volts ){
				me.bus_volts = bv;
				powered_by = "battery_bus";
				gen2_bus_from_batt_bus = 0;
			}
		}
		
		me.volts.setDoubleValue( me.bus_volts );
		
		var load = 1.0;	# estimate value for all direct bus consumers
		
		if( powered_by == "generator2" ){
			if( bus_breakers.gen2_tie.getBoolValue()){
				generator2.apply_load( load + gen2_bus_load_from_batt_bus );
				return -1.0;
			} else {
				generator2.apply_load( load );
				return 0.0;
			}
		} else if ( powered_by == "battery_bus" ){
			return load;
		} else {
			return 0.0;
		}
	},
);

var avionic1_bus = bus.new(
	"avionic1-bus",
	func( bv ) {
		if( me.serviceable.getBoolValue() and switches.avionic1.getBoolValue() ) {
			me.bus_volts = bv;
		} else {
			me.bus_volts = 0.0;
		}
		
		var load = 1.0;	# estimate value for all direct bus consumers
		if( !switches.stby_bus.getBoolValue() ) load += stby_bus.on_update( me.bus_volts );
		
		me.volts.setDoubleValue( me.bus_volts );
		return load;
	},
);

var avionic2_bus = bus.new(
	"avionic2-bus",
	func( bv ) {
		if( me.serviceable.getBoolValue() and switches.avionic2.getBoolValue() ) {
			me.bus_volts = bv;
		} else {
			me.bus_volts = 0.0;
		}
		
		var load = 1.0;	# estimate value for all direct bus consumers
		
		me.volts.setDoubleValue( me.bus_volts );
		return load;
	},
);

var non_ess_bus = bus.new(
	"non-ess-bus",
	func( bv ) {
		if( me.serviceable.getBoolValue() ) {
			me.bus_volts = bv;
		} else {
			me.bus_volts = 0.0;
		}
		
		var load = 1.0;	# estimate value for all direct bus consumers
		
		me.volts.setDoubleValue( me.bus_volts );
		return load;
	},
);

var stby_bus = bus.new(
	"stby-bus",
	func( bv ) {
		if( me.serviceable.getBoolValue() ) {
			me.bus_volts = bv;
		} else {
			me.bus_volts = 0.0;
		}
		
		var load = 1.0;	# estimate value for all direct bus consumers
		
		me.volts.setDoubleValue( me.bus_volts );
		return load;	
	},
);

var ac_bus = bus.new(
	"ac-bus",
	func{
		if( me.serviceable.getBoolValue() and 
			( inverter[0].getDoubleValue() > 18.0 and !switches.inverter.getBoolValue() ) or 
			( inverter[1].getDoubleValue() > 18.0 and  switches.inverter.getBoolValue() ) ){
			me.bus_volts = 26.0;
		} else {
			me.bus_volts = 0.0;
		}
		
		me.volts.setDoubleValue( me.bus_volts );
	},
);


update_electrical = func {
	
	battery_direct_bus.on_update();
	ac_bus.on_update();
	
	
	# Flaps only operate when electrically powered:
	#	var flaps_ctl_power = props.globals.getNode("systems/electrical/outputs/flaps-ctl");
	#	var cmd_flaps = props.globals.getNode("controls/flight/flaps");
	#	var int_flaps = props.globals.getNode("controls/flight/flaps-int");
	#	if( flaps_ctl_power.getDoubleValue() > 15 and cmd_flaps.getDoubleValue() != int_flaps.getDoubleValue() ){
	#		int_flaps.setDoubleValue( cmd_flaps.getDoubleValue() );
	#	}
}

var electrical_updater = maketimer( 0.0, update_electrical );
electrical_updater.simulatedTime = 1;
electrical_updater.start();
