/////////////////////////////////////////////////////////////////////////////
//
// ColourButton
// Control panel like colour picker
// Based on code taken from http://www.codeproject.com/
//
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "ColourButton.h"
#include <TmSchema.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// Implementation of CColourButtonDlg
/////////////////////////////////////////////////////////////////////////////

// The colour table, initialized to Windows' 20 static colours
COLORREF CColourButtonDlg::m_Colours[20] =
{
	RGB(0,0,0),
	RGB(128,0,0),
	RGB(0,128,0),
	RGB(128,128,0),
	RGB(0,0,128),
	RGB(128,0,128),
	RGB(0,128,128),
	RGB(192,192,192),
	RGB(192,220,192),
	RGB(166,202,240),
	RGB(255,251,240),
	RGB(160,160,164),
	RGB(128,128,128),
	RGB(255,0,0),
	RGB(0,255,0),
	RGB(255,255,0),
	RGB(0,0,255),
	RGB(255,0,255),
	RGB(0,255,255),
	RGB(255,255,255)
};

// MRU table. This "colourful" (no pun intended) order ensures that the
// colours at the center of the colour table will get replaced first.
// This preserves the white and black colours even if they're not used
// (as they'll get replaced last).
BYTE CColourButtonDlg::m_Used[20] =
{
	1,3,5,7,9,11,13,15,17,19,20,18,16,14,12,10,8,6,4,2
};

CColourButtonDlg::CColourButtonDlg(CWnd* pParent) : CDialog(CColourButtonDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(CColourButtonDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT

	m_ColourIndex = 0;
	m_pParent = NULL;
}

void CColourButtonDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CColourButtonDlg)
		// NOTE: the ClassWizard will add DDX and DDV calls here
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CColourButtonDlg, CDialog)
	//{{AFX_MSG_MAP(CColourButtonDlg)
	ON_BN_CLICKED(IDC_OTHER, OnOther)
	ON_WM_LBUTTONDOWN()	
	ON_WM_LBUTTONUP()
	ON_WM_DRAWITEM()	
	//}}AFX_MSG_MAP
	ON_COMMAND_RANGE(IDC_COLOUR1,IDC_COLOUR20,OnColor)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CColourButtonDlg message handlers
/////////////////////////////////////////////////////////////////////////////

BOOL CColourButtonDlg::OnInitDialog() 
{
	CDialog::OnInitDialog();

	CRect r1,r2;
	m_pParent->GetWindowRect(r1);

	// Move the dialog to be below the button
	SetWindowPos(NULL,r1.left,r1.bottom,0,0,SWP_NOSIZE|SWP_NOZORDER);

	// Check to see if the dialog has a portion outside the
	// screen, and if so, adjust.
	GetWindowRect(r2);
	if (r2.bottom > ::GetSystemMetrics(SM_CYSCREEN))
		r2.top = r1.top - r2.Height();
	if (r2.right > ::GetSystemMetrics(SM_CXSCREEN))
		r2.left = ::GetSystemMetrics(SM_CXSCREEN) - r2.Width();
	SetWindowPos(NULL,r2.left,r2.top,0,0,SWP_NOSIZE|SWP_NOZORDER);

	if (m_strOther.GetLength() > 0)
	{
		CWnd* pOther = GetDlgItem(IDC_OTHER);
		if (pOther != NULL)
			pOther->SetWindowText(m_strOther);
	}

	// Capture the mouse, this allows the dialog to close when
	// the user clicks outside (Remember that the dialog has no
	// "close" button.).
	SetCapture();
	return TRUE;
}

void CColourButtonDlg::EndDialog( int nResult )
{
	::ReleaseCapture();
	CDialog::EndDialog(nResult);
}

void CColourButtonDlg::OnLButtonDown(UINT nFlags, CPoint point) 
{
	POINT p;
	p.x = point.x;
	p.y = point.y;
	ClientToScreen(&p);

	CRect r;
	GetWindowRect(r);

	// The user clicked ...
	if (!PtInRect(r,p))
	{
		// ... outside the dialog, close.
		EndDialog(IDCANCEL);
	}
	else
	{
		// ... inside the dialog. Since this window has the mouse captured,
		// its children get no messages. So, check to see if the click was
		// in one of its children and pass the click on.
		// If the user clicks inside the dialog but not on any of the
		// controls, ChildWindowFromPoint() returns a pointer to the dialog.
		// In this case we do not resend the message as it would result in
		// a loop.
		CWnd *pChild = ChildWindowFromPoint(point);
		if ((pChild != NULL) && (pChild != this))
			pChild->SendMessage(WM_LBUTTONDOWN,0,0L);
	}

	CDialog::OnLButtonDown(nFlags, point);
}

void CColourButtonDlg::OnLButtonUp(UINT nFlags, CPoint point) 
{
	CWnd *pChild = ChildWindowFromPoint(point,CWP_ALL);
	if ((pChild != NULL) && (pChild != this))
		pChild->SendMessage(WM_LBUTTONDOWN,0,0L);
	CDialog::OnLButtonUp(nFlags,point);
}

void CColourButtonDlg::OnDrawItem(int nIDCtl, LPDRAWITEMSTRUCT lpd) 
{
	CDC DC;
	DC.Attach(lpd->hDC);

	CPen NullPen;
	NullPen.CreateStockObject(NULL_PEN);
	CPen *pOldPen = DC.SelectObject(&NullPen);

	// Draw the colour wells using the current colour table
	CBrush ColourBrush;
	ColourBrush.CreateSolidBrush(m_Colours[nIDCtl - IDC_COLOUR1]);
	CBrush *pOldBrush = DC.SelectObject(&ColourBrush);

	lpd->rcItem.right++;
	lpd->rcItem.bottom++;
	DC.Rectangle(&lpd->rcItem);

	DC.SelectObject(pOldPen);
	DC.SelectObject(pOldBrush);
	DC.Detach();
	CDialog::OnDrawItem(nIDCtl,lpd);
}

void CColourButtonDlg::OnColor(UINT id)
{
	// A well has been clicked, so set the colour index and close.
	m_ColourIndex = id - IDC_COLOUR1;

	// This colour is now the MRU
	for (int i = 0; i < 20; i++)
	{
		if (m_Used[m_ColourIndex] > m_Used[i])
			m_Used[i]++;
	}
	m_Used[m_ColourIndex] = 1;
	EndDialog(IDOK);
}

void CColourButtonDlg::OnOther() 
{
	::ReleaseCapture();

	// The "Other" button.
	CColorDialog Dialog;
	Dialog.m_cc.Flags |= CC_FULLOPEN;
	if (Dialog.DoModal() == IDOK)
	{
		// The user clicked OK, so set the colour and close
		COLORREF NewColour = Dialog.GetColor();

		// Check to see if the selected colour is already in the table.
		m_ColourIndex = -1;
		for (int i = 0; i < 20; i++)
		{
			if (m_Colours[i] == NewColour)
			{
				m_ColourIndex = i;
				break;
			}
		}

		// If the colour was not found, replace the LRU with this colour.
		if (m_ColourIndex == -1)
		{
			for (i = 0; i < 20; i++)
			{
				if (m_Used[i] == 20)
				{
					m_Colours[i] = NewColour;
					m_ColourIndex = i;
					break;
				}
			}
		}

		// This is the new MRU
		for (i = 0; i < 20; i++)
		{
			if (m_Used[m_ColourIndex] > m_Used[i])
				m_Used[i]++;
		}
		m_Used[m_ColourIndex] = 1;
		EndDialog(IDOK);
	}
	else
	{
		// If the user clicked "Cancel", reclaim the mouse capture.
		SetCapture();
	}
}

/////////////////////////////////////////////////////////////////////////////
// Implementation of CColourButton
/////////////////////////////////////////////////////////////////////////////

CColourButton::CColourButton(BOOL bPopup)
{
	m_hThemeDll = ::LoadLibrary("UxTheme.dll");
	m_bPopup = bPopup;
	m_CurrentColour = RGB(255,255,255);

	// This will allow the dialog to position itself
	m_Dialog.m_pParent = this;
}

CColourButton::~CColourButton()
{
	if (m_hThemeDll)
		::FreeLibrary(m_hThemeDll);
}

BEGIN_MESSAGE_MAP(CColourButton, CButton)
	//{{AFX_MSG_MAP(CColourButton)	
	ON_CONTROL_REFLECT(BN_CLICKED, OnClicked)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

BOOL CColourButton::IsAppThemed(void)
{
	if (m_hThemeDll)
	{
		typedef BOOL(__stdcall *PFNISAPPTHEMED)();

		PFNISAPPTHEMED pfnIsAppThemed =
			(PFNISAPPTHEMED)::GetProcAddress(m_hThemeDll,"IsAppThemed");
		if (pfnIsAppThemed)
			return (*pfnIsAppThemed)();
	}
	return FALSE;
}

HTHEME CColourButton::OpenThemeData(LPCWSTR pszClassList)
{
	if (m_hThemeDll)
	{
		typedef HTHEME(__stdcall *PFNOPENTHEMEDATA)(HWND hWnd, LPCWSTR pszClassList);

		PFNOPENTHEMEDATA pfnOpenThemeData =
			(PFNOPENTHEMEDATA)::GetProcAddress(m_hThemeDll,"OpenThemeData");
		if (pfnOpenThemeData)
			return (*pfnOpenThemeData)(GetSafeHwnd(),pszClassList);
	}
	return 0;
}

void CColourButton::CloseThemeData(HTHEME hTheme)
{
	if (m_hThemeDll)
	{
		typedef HRESULT(__stdcall *PFNCLOSETHEMEDATA)(HTHEME hTheme);

		PFNCLOSETHEMEDATA pfnCloseThemeData =
			(PFNCLOSETHEMEDATA)::GetProcAddress(m_hThemeDll,"CloseThemeData");
		if (pfnCloseThemeData)
			(*pfnCloseThemeData)(hTheme);
	}
}

void CColourButton::DrawThemeBackground(HTHEME hTheme, HDC hDC, 
	int iPartId, int iStateId, const RECT *pRect, const RECT *pClipRect)
{
	if (m_hThemeDll)
	{
		typedef HRESULT(__stdcall *PFNDRAWTHEMEBACKGROUND)(HTHEME hTheme, HDC hDC, 
			int iPartId, int iStateId, const RECT *pRect, OPTIONAL const RECT *pClipRect);

		PFNDRAWTHEMEBACKGROUND pfnDrawThemeBackground =
			(PFNDRAWTHEMEBACKGROUND)::GetProcAddress(m_hThemeDll,"DrawThemeBackground");
		if (pfnDrawThemeBackground)
			(*pfnDrawThemeBackground)(hTheme,hDC,iPartId,iStateId,pRect,pClipRect);
	}
}

/////////////////////////////////////////////////////////////////////////////
// CColourButton message handlers
/////////////////////////////////////////////////////////////////////////////

void CColourButton::DrawItem(LPDRAWITEMSTRUCT lpd)
{
	// Set up a device context
	CDC DC;
	DC.Attach(lpd->hDC);

	// Store for convinience
	int top    = lpd->rcItem.top;
	int left   = lpd->rcItem.left;
	int bottom = lpd->rcItem.bottom;
	int right  = lpd->rcItem.right;

	CPen NullPen, BlackPen, HighlightPen, ShadowPen;
	NullPen.CreateStockObject(NULL_PEN);
	BlackPen.CreateStockObject(BLACK_PEN);
	HighlightPen.CreatePen(PS_SOLID,1,::GetSysColor(COLOR_BTNHIGHLIGHT));
	ShadowPen.CreatePen(PS_SOLID,1,::GetSysColor(COLOR_BTNSHADOW));

	CBrush BackBrush, ColourBrush;
	BackBrush.CreateSolidBrush(GetSysColor(COLOR_3DFACE));
	ColourBrush.CreateSolidBrush(m_CurrentColour);

	CPen* pOldPen = DC.SelectObject(&NullPen);
	CBrush* pOldBrush = DC.SelectObject(&BackBrush);

	BOOL drawn = FALSE;
	if (IsAppThemed())
	{
		// Open the button theme
		HTHEME hTheme = OpenThemeData(L"Button");
		if (hTheme)
		{
			// Get the button state
			UINT nState = PBS_NORMAL;
			if (lpd->itemState & ODS_DISABLED)
				nState |= PBS_DISABLED;
			if (lpd->itemState & ODS_SELECTED)
				nState |= PBS_PRESSED;

			// Draw the themed background
			DrawThemeBackground(hTheme,DC,BP_PUSHBUTTON,nState,&lpd->rcItem,NULL);
			CloseThemeData(hTheme);
			drawn = TRUE;
		}
	}

	if (drawn == FALSE)
	{
		// Clear the background
		DC.Rectangle(&lpd->rcItem);

		// Get the button state
		UINT nState = DFCS_BUTTONPUSH|DFCS_ADJUSTRECT;
		if (lpd->itemState & ODS_DISABLED)
			nState |= DFCS_INACTIVE;
		if (lpd->itemState & ODS_SELECTED)
			nState |= DFCS_PUSHED;

		// Draw the border
		DC.DrawFrameControl(&(lpd->rcItem),DFC_BUTTON,nState);
		if (lpd->itemState & ODS_SELECTED)
		{
			// Get the contents of the button to draw one pixel down and one
			// to the right. This completes the "pushed" effect.
			left++;
			right++;
			bottom++;
			top++;
		}
	}

	// Draw the divider ...
	DC.SelectObject(&HighlightPen);
	DC.MoveTo(right-10,top+4);
	DC.LineTo(right-10,bottom-4);
	DC.SelectObject(&ShadowPen);
	DC.MoveTo(right-11,top+4);
	DC.LineTo(right-11,bottom-4);

	// ... and the triangle
	if (lpd->itemState & ODS_DISABLED)
		DC.SelectObject(&ShadowPen);
	else
		DC.SelectObject(&BlackPen);
	DC.MoveTo(right-4,(bottom/2)-1);
	DC.LineTo(right-9,(bottom/2)-1);
	DC.MoveTo(right-5,(bottom/2));
	DC.LineTo(right-8,(bottom/2));

	if (lpd->itemState & ODS_DISABLED)
	{
		COLORREF Highlight = ::GetSysColor(COLOR_BTNHIGHLIGHT);
		DC.SetPixel(right-4,(bottom/2)-1,Highlight);
		DC.SetPixel(right-5,(bottom/2),Highlight);
		DC.SetPixel(right-6,(bottom/2)+1,Highlight);
	}
	else
		DC.SetPixel(right-6,(bottom/2)+1,RGB(0,0,0));

	// Draw the colour rectangle only if enabled
	if (!(lpd->itemState & ODS_DISABLED))
	{
		DC.SelectObject(&ColourBrush);
		DC.Rectangle(left+5,top+4,right-15,bottom-4);
	}

	// Draw the focus
	if (lpd->itemState & ODS_FOCUS)
	{
		CRect r(left+3,top+3,right-3,bottom-3);
		DC.DrawFocusRect(&r);
	}

	DC.SelectObject(pOldPen);
	DC.SelectObject(pOldBrush);
	DC.Detach();
}

void CColourButton::OnClicked() 
{
	if (m_bPopup)
	{
		// When the button is clicked, show the dialog.
		if (m_Dialog.DoModal() == IDOK)
		{
			SetCurrentColour(CColourButtonDlg::m_Colours[m_Dialog.m_ColourIndex]);
			InvalidateRect(NULL);
		}
	}
	else
	{
		// Just use the system colour picker
		CColorDialog Dialog(GetCurrentColour());
		Dialog.m_cc.Flags |= CC_FULLOPEN;
		if (Dialog.DoModal() == IDOK)
		{
			SetCurrentColour(Dialog.GetColor());
			InvalidateRect(NULL);
		}
	}
}
